#########################################################################################################
# Simulation prior the current research, Siska Fitrianie, Merijn Bruijnes, Fengxiang Li, Amal Abdulrahman, 
# Willem-Paul Brinkman. 2022. The Artificial-Social-Agent Questionnaire: Establishing the long and short 
# questionnaire versions. In ACM International Conference on Intelligent Virtual Agents (IVA’22), September, 
# 2022, Faro, Portugal. ACM, New York, NY, USA. https://doi.org/10.1145/3514197.3549612
# 
# Reading the previous research:
# Siska Fitrianie, Merijn Bruijnes, Fengxiang Li, and Willem-Paul Brinkman. 2021. 
# Questionnaire Items for Evaluating Artificial Social Agents - Expert Generated, Content Validated and 
# Reliability Analysed. In Proceedings of the 21th ACM International Conference on Intelligent Virtual 
# Agents (IVA '21). Association for Computing Machinery, New York, NY, USA, 84–86. 
# DOI:https://doi.org/10.1145/3472306.3478341 
#
# Create two simulation datasets n=406 using a function in the package Lavaan
# Output:  sim_406p_14agents.csv (original simulated data) and sim_406p_14agents_std.csv (standardized)
#########################################################################################################

### Library ####
library(lavaan)

### Create data using Lavaan's function simulateData()
createData <- function(model, s){
  set.seed(1234)
  simData <- lavaan:::simulateData(model, model.type="cfa", sample.nobs=s)
  simData.int <- data.frame(simData)
  colnames(simData.int)<-items
  return(simData.int)
}

### Add column 'agent' 1:14
addAgent <- function(d_){
  d<-data.frame(ncol=1)
  colnames(d)=c("agent")
  id<- 1
  for(i in 1:nrow(d_)){
    d[i,1]<- id
    if(i%%(nrow(d_)/14) == 0) id = id + 1
  }
  return(cbind(d,d_))
}

### standardization each item for each agent
standardize <- function(d_){
  d_standard = data.frame()
  i = 1
  for(col in 2:ncol(d_)){
    newCol = col - 1
    for (row in 1:nrow(d_)){
      d<-d_[d_[,1]==i, col]
      if (length(d)==0) {
        print(col)
        print(i)
      }
      M <- mean(d)
      std <- sd(d)
      d_standard[row,newCol] <- (d_[row, col] - M)/std
      
      if (row%%14 == 0) {i = i + 1}
      if (i>14) {i=1}
    }
  }
  names(d_standard) <- colnames(d_[c(2:ncol(d_))])
  return(d_standard)
}

itemIDs <- c("
  C01D01Q3", "C01D01Q10", "C01D01Q13", "C01D01Q15", "C01D01Q16", "C01D02Q7", "C01D02Q10", "C01D02Q0", "C01D02Q1", "
  C01D02Q9", "C01D03Q7", "C01D03Q9", "C01D03Q10", "C01D03Q11", "C01D03Q12 +C01D04Q0", "C01D04Q13", "C01D04Q15", "
  C01D04Q8", "C01D04Q9", "C01D05Q2", "C01D05Q9", "C01D05Q11", "C01D05Q12", "C01D05Q0", "C02D00Q1", "C02D00Q2", "
  C02D00Q8", "R_C02D00Q11", "R_C02D00Q15", "C03D01Q1", "C03D01Q7", "C03D01Q5", "C03D01Q6", "C03D01Q12", "C03D02Q0", "
  C03D02Q5", "C03D02Q10", "R_C03D02Q13", "R_C03D02Q1", "C04D00Q1", "C04D00Q10", "R_C04D00Q11", "C04D00Q4", "
  C04D00Q12", "R_C05D00Q3", "C05D00Q6", "C05D00Q7", "C05D00Q1", "C05D00Q18", "C06D01Q1", "R_C06D01Q6", "
  R_C06D01Q13", "C06D01Q8", "C06D01Q11", "C07D00Q0", "C07D00Q11", "C07D00Q13", "R_C07D00Q15", "C07D00Q14", "
  R_C08D00Q3", "C08D00Q4", "R_C08D00Q10", "C08D00Q0", "R_C08D00Q9", "C09D00Q4", "C09D00Q5", "C09D00Q10", "C09D00Q3", "
  C09D00Q1", "R_C09D00Q9", "C10D00Q1", "C10D00Q6", "C10D00Q10", "C10D00Q16", "C10D00Q18", "C11D01Q6", "C11D01Q9", "
  C11D01Q10", "C11D01Q11", "C11D01Q14", "C11D02Q1", "C11D02Q3", "C11D02Q4", "C11D02Q7", "C11D02Q8", "C12D00Q0", "
  C12D00Q13", "C12D00Q1", "C12D00Q8", "R_C12D00Q6", "R_C13D00Q3", "C13D00Q7", "R_C13D00Q11", "R_C13D00Q6", "
  R_C13D00Q15", "C14D00Q0", "C14D00Q10", "R_C14D00Q13", "C14D00Q15", "C14D00Q16", "C15D00Q10", "C15D00Q11", "
  C15D00Q8", "C15D00Q12", "R_C15D00Q15", "C16D00Q11", "C16D00Q12", "C16D00Q13", "C16D00Q16", "C16D00Q17", "
  C17D00Q2", "C17D00Q3", "C17D00Q8", "C17D00Q9", "C17D00Q10", "C18D01Q2", "C18D01Q3", "R_C18D01Q5", "
  C18D01Q8", "R_C18D01Q14", "C18D03Q0", "C18D03Q1", "C18D03Q2", "C18D03Q9", "C18D03Q12", "C19D00Q3", "
  C19D00Q7", "C19D00Q14", "C19D00Q26", "C19D00Q28")

## Theoretical Model based on 131 items resulted from the reliability analysis
theoreticalModel <- '
#latent variable first order
C01D01 =~ C01D01Q3 + C01D01Q10 + C01D01Q13 + C01D01Q15 + C01D01Q16 
C01D02 =~ C01D02Q7 + C01D02Q10 + C01D02Q0 + C01D02Q1 + C01D02Q9 
C01D03 =~ C01D03Q7 + C01D03Q9 + C01D03Q10 + C01D03Q11 + C01D03Q12 
C01D04 =~ C01D04Q0 + C01D04Q13 + C01D04Q15 + C01D04Q8 + C01D04Q9
C01D05 =~ C01D05Q2 + C01D05Q9 + C01D05Q11 + C01D05Q12 + C01D05Q0
C02 =~ C02D00Q1 + C02D00Q2 + C02D00Q8 + R_C02D00Q11 + R_C02D00Q15
C03D01 =~ C03D01Q1 + C03D01Q7 + C03D01Q5 + C03D01Q6 + C03D01Q12
C03D02 =~ C03D02Q0 + C03D02Q5 + C03D02Q10 + R_C03D02Q13 + R_C03D02Q1
C04 =~ C04D00Q1 + C04D00Q10 + R_C04D00Q11 + C04D00Q4 + C04D00Q12 
C05 =~ R_C05D00Q3 + C05D00Q6 + C05D00Q7 + C05D00Q1 + C05D00Q18 
C06D01 =~ C06D01Q1 + R_C06D01Q6 + R_C06D01Q13 + C06D01Q8 + C06D01Q11
C07 =~ C07D00Q0 + C07D00Q11 + C07D00Q13 + R_C07D00Q15 + C07D00Q14  
C08 =~ R_C08D00Q3 + C08D00Q4 + R_C08D00Q10 + C08D00Q0 + R_C08D00Q9  
C09 =~ C09D00Q4 + C09D00Q5 + C09D00Q10 + C09D00Q3 + C09D00Q1 + R_C09D00Q9 
C10 =~ C10D00Q1 + C10D00Q6 + C10D00Q10 + C10D00Q16 + C10D00Q18 
C11D01 =~ C11D01Q6 + C11D01Q9 + C11D01Q10 + C11D01Q11 + C11D01Q14 
C11D02 =~ C11D02Q1 + C11D02Q3 + C11D02Q4 + C11D02Q7 + C11D02Q8 
C12 =~ C12D00Q0 + C12D00Q13 + C12D00Q1 + C12D00Q8 + R_C12D00Q6 
C13 =~ R_C13D00Q3 + C13D00Q7 + R_C13D00Q11 + R_C13D00Q6 + R_C13D00Q15
C14 =~ C14D00Q0 + C14D00Q10 + R_C14D00Q13 + C14D00Q15 + C14D00Q16 
C15 =~ C15D00Q10 + C15D00Q11 + C15D00Q8 + C15D00Q12 + R_C15D00Q15 
C16 =~ C16D00Q11 + C16D00Q12 + C16D00Q13 + C16D00Q16 + C16D00Q17 
C17 =~ C17D00Q2 + C17D00Q3 + C17D00Q8 + C17D00Q9 + C17D00Q10 
C18D01 =~ C18D01Q2 + C18D01Q3 + R_C18D01Q5 + C18D01Q8 + R_C18D01Q14
C18D03 =~ C18D03Q0 + C18D03Q1 + C18D03Q2 + C18D03Q9 + C18D03Q12  
C19 =~ C19D00Q3 + C19D00Q7 + C19D00Q14 + C19D00Q26 + C19D00Q28  

#latent variable second order
C01 =~ C01D01 + C01D02 + C01D03 + C01D04 + C01D05
C03 =~ C03D01 + C03D02
C11 =~ C11D01 + C11D02
C18 =~ C18D01 + C18D03
'

## 406 = 14 * 29 
size<-c(406)

## create simulation data, standardize it and save both original and standardized data to .cvs files
for(i in (1:length(size))){
  d<-createData(theoreticalModel, size[i])
  write.csv(d,paste("../r_results/sim_", size[i], "p_14agents.csv", sep=""), row.names = FALSE)
  d<-addAgent(d)
  d_std<-standardize(d)
  write.csv(d_std, paste("../r_results/sim_", size[i], "p_14agents_std.csv", sep="") , row.names = FALSE)
}