% Original script written by Tomas Lenssen at the Delft Technical
% University. Script makes use of data from the Subjects folder in which
% subject 1 to 9 are stored. S1 was a first trial and its data is
% incomplete. As such, S2 to S9 are used. Data is split in disassembly and
% reassembly of the instrument. Of the reassembly, time data is available
% of the timestamp when the end-effector is threaded in completely.
% However, this data is not available for all repetitions, of all subjects.
% As such, the time between the shaft coupling and collet coupling is
% taken directly.
%
% Full disassembly is done after tip extraction and shaft decoupling.
% Full reassembly is done after shaft coupling and collet tightening.
% Tip-exchange is done by tip extraction and collet tightening.
% Time requirements for each is done by adding the time for each segment.
%
% Final variable M shows the mean, STD and median (:,3) 
% for tip extr, Shaft dec, shaft coup, collet, disassembly, reassembly and
% tip-exchange (8,:) where (5:0) is kept blank as a separator. 

clc
clear
format
format compact
format long g

%% Load all data

for s = 2:9 % subjects
    clear FileRaw
    FileRaw(:) = fileread('Subjects\S' + string(s) + '.txt');
    
    for j = 1:7 % repetitions
        %Read time data
        for i = 1:5 %sub-tasks
            Data(i,j,s) = str2num(FileRaw((-2+i*3+(j-1)*17:-1+i*3+(j-1)*17)));
            if str2num(FileRaw((-2+i*3+(j-1)*17:-1+i*3+(j-1)*17))) == 0
                Data(i,j,s) = NaN;
            end
        end
        
        %Correct single 0, subject was very fast
        Data(2,5,8) = 15.5;
        
        %Make Time corrected Data
        DataC(1,j,s) = Data(1,j,s);
        DataC(2,j,s) = Data(2,j,s) - Data(1,j,s);
        DataC(3,j,s) = Data(3,j,s);
        DataC(4,j,s) = Data(4,j,s) - Data(3,j,s);
        DataC(5,j,s) = Data(5,j,s) - Data(3,j,s);
        
        
        
        %Read instructions
        for i = 1:2
            Data(i,j+7,s) = str2num(FileRaw(123+(i-1)*2+(j-1)*6));
        end
    end
end


%% Visualize Instructions
figure(1); clf(1);
for s = 2:9 % subjects
    for j = 1:7 % repetitions
        for i = 1:2
            subplot(2,1,i); hold on;
            plot([(s-1)*0.1-0.1+j,(s-1)*0.1-0.1+j], [0,Data(i,j+7,s)])
            ylim([0, 2.5]); xlim([0.5,7.5]);
        end
    end
end
subplot(2,1,1); title('Disassembly'); ylabel('Instructions'); xlabel('Repetitions'); yticks([0 1 2 3]); box on;
subplot(2,1,2); title('Reassembly');  ylabel('Instructions'); xlabel('Repetitions'); yticks([0 1 2 3]); box on;

%% Visualize Raw Data
clc

Title = ["Tip ext.","Shaft dec.","Shaft coup.","Tip Thread","Collet"];
SP = [1,2,4,5,6];
SM = [50,15,40,60,80];
figure(1); clf(1);
for s = 2:9
    for i = 1:5
        subplot(2,3,SP(i)); hold on;
        plot(DataC(i,1:7,s),'-b.')
        title(Title(i))
        xlabel('Repetitions'); ylabel('Seconds');
        ylim([0,SM(i)]); xlim([0.5,7.5]);
    end
end


%% visualize boxplot
clc

Title = ["Tip ext.","Shaft dec.","Shaft coup.","Tip Thread","Collet"];
SP = [1,2,4,5,6];
figure(1); clf(1);

for i = 1:5
    subplot(2,3,SP(i)); hold on;
    A = squeeze(DataC(i,1:7,2:9));
    boxplot(A')
    title(Title(i))
    xlabel('Repetitions'); ylabel('Seconds');
    ylim([0,SM(i)]); xlim([0.5,7.5]);
end

%% visualize corrected boxplot
clc

Title = ["Tip ext.","Shaft decoup.","Shaft coup.","Collet"];
figure(1); clf(1);

I = [1, 2, 3, 4, 4];

for i = [1, 2, 3, 5]
    subplot(2,2,I(i)); hold on;
    A = squeeze(DataC(i,1:7,2:9));
    boxplot(A')
    title(Title(I(i)))
    xlabel('Repetitions'); ylabel('Seconds');
    ylim([0,SM(i)]); xlim([0.5,7.5]);
    clear A;
end

%% Find Mean, STD and Median of last 3 repetitions
clc
for i = [1,2,3,5]
    A = 0; k = 1;
    for s = 2:9 
        for j = 5:7
            if ~isnan(DataC(i,j,s))
               A(k) = DataC(i,j,s);
               k = k + 1;
            end
        end
    end
    M(I(i),1) = mean(A);
    M(I(i),2) = std(A);
    M(I(i),3) = median(A);
end
M


%% DataO minus outliers

DataO = DataC;
DataO(3,1,2) = NaN;
DataO(3,6,9) = NaN;
DataO(3,7,2) = NaN;


%% Find Mean, STD and Median of last 3 repetitions minus outliers

for i = [1,2,3,5]
    A = 0; k = 1;
    for s = 2:9 
        for j = 5:7
            if ~isnan(DataO(i,j,s))
               A(k) = DataO(i,j,s);
               k = k + 1;
            end
        end
    end
    M(I(i),1) = mean(A);
    M(I(i),2) = std(A);
    M(I(i),3) = median(A);
end
M


%% Find Mean, STD and Median of last 3 repetitions for phases
clc
A = 0; k = 1;
    for s = 2:9 
        for j = 5:7
            if ~isnan(DataO(1,j,s)) && ~isnan(DataO(2,j,s))
               A(k) = DataO(1,j,s) + DataO(2,j,s); %Disassembly
               k = k + 1;
            end
        end
    end
    M(6,1) = mean(A);
    M(6,2) = std(A);
    M(6,3) = median(A);


A = 0; k = 1;

    for s = 2:9 
        for j = 5:7
            if ~isnan(DataO(3,j,s)) && ~isnan(DataO(5,j,s))
               A(k) = DataO(3,j,s) + DataO(5,j,s); %Reassembly
               k = k + 1;
            end
        end
    end
    M(7,1) = mean(A);
    M(7,2) = std(A);
    M(7,3) = median(A);


A = 0; k = 1;
    for s = 2:9 
        for j = 5:7
            if ~isnan(DataO(1,j,s)) && ~isnan(DataO(5,j,s)) 
               A(k) = DataO(1,j,s) + DataO(5,j,s); %Tip Exchange
               k = k + 1;
            end
        end
    end
    M(8,1) = mean(A);
    M(8,2) = std(A);
    M(8,3) = median(A);

M


%% Summary
clc


Scope = "Full disassembly takes  " + M(6,1) + " seconds (" + M(6,2) + ")"
Scope = "Full  reassembly takes  " + M(7,1) + " seconds (" + M(7,2) + ")"
Scope = "Full Tip exchange takes " + M(8,1) + " seconds (" + M(8,2) + ")"









