/* [Hidden] */
//variable for making very thin cuboids
infinitesimal = 0.001;
//Sets the resolution of circular features
$fn=50;

/* [Print config - match slicer settings] */
//Layer height, as selected in your slicer
Layer_height = 0.15;
//Extrusion width. If there are multiple extrusion widths in your slicer, set them all to the same figure.
Extrusion_width= 0.25;

//Variables for thin walls, as walls are not simply nWalls * Extrusion_width
//ref. https://manual.slic3r.org/advanced/flow-math 
extrusionSpacing=Extrusion_width-Layer_height*(1-(PI/4)); 

//generate wall thickness for thin walls of n walls:
//extrusionSpacing * (n-1)) + Extrusion_width
nWalls = [(extrusionSpacing*(1-1))+Extrusion_width,
(extrusionSpacing * (2-1)) + Extrusion_width,
(extrusionSpacing * (3-1)) + Extrusion_width,
(extrusionSpacing * (4-1)) + Extrusion_width,
(extrusionSpacing * (5-1)) + Extrusion_width,
(extrusionSpacing * (6-1)) + Extrusion_width];

echo("Below are the recommended thin wall thicknesses calculated for the chosen layer height and extrusion width:");
echo(nWalls);
echo("These should ideally be used for Spine_thickness and Rib_thickness.");

/* [Segment config - all units in mm or degrees] */

//Sets the segment OD
Outer_diameter = 8.0;

//Sets the overall segment length
Segment_length = 35.0; //mm

//Sets the thickness of the axial spine
Spine_thickness = nWalls[1];
//Sets the theoretical max bending angle
Max_bending_angle = 90.0; //degrees

//Sets the thickness of the solid ends
End_thickness = 3.0; //mm

//Sets the thickness of the helix ribs
Rib_thickness = nWalls[4]; //[nWalls[0],nWalls[1],nWalls[2],nWalls[3],nWalls[4],nWalls[5]]

//Sets how much is cut off to create a flat side
Flat_side = 0.5; //mm

//Sets whether the helix runs CW (-1) or CCW (1)
Helix_handedness = -1; //[-1,1]

//Sets the resolution of the helix
Helix_resolution = 720; //changes the resolution of the helix

//Sets the size of the square tendon channels
Tendon_channel_size = 1;//mm

//Sets the thinnest point between the tendon channel and the outer surface of the segment
tendonChannelThinWall = nWalls[1];

/* [Working channel sizes] */
//Render north working channel
Render_north_channel = true;
//North working channel diameter
northLumenDiameter = 2.234;

//Render south working channel
Render_south_channel = true;
//South working channel diameter
southLumenDiameter = 2.234;

//Render east working channel
Render_east_channel = true;
//East working channel diameter
eastLumenDiameter = 2.57;

//Render west working channel
Render_west_channel = true;
//West working channel diameter
westLumenDiameter = 2.57;

//Minimum separation between two channels
minSep=nWalls[1];
//calculate the XY position of the tendon channel for a given value of tendonChannelThinWall
omega = asin((tendonChannelThinWall*sin(45))/(Outer_diameter/2));
rho = ((Outer_diameter/2)* cos(omega)) - (sqrt(2*(1)^2))-(tendonChannelThinWall*cos(45));
tendonChannelPositionX = rho * cos(45);
tendonChannelPositionY = rho * sin(45);

//actual height of the helical section, as L includes the end caps
Lcorrected = Segment_length - ( 2 * End_thickness ) + Rib_thickness;

//number of helix turns
nTurns = (((PI * Outer_diameter/2 * Max_bending_angle ) / (180)) - Lcorrected) / (-Rib_thickness);


//what follows are some horrible bodges because OpenSCAD doesn't like helices.
//Z points for helix generation
helixStartPointZ = End_thickness-Rib_thickness;
helixEndPointZ = helixStartPointZ+Lcorrected;
helixStepsZ=[for(z=[helixStartPointZ:Lcorrected/(Helix_resolution-1):helixEndPointZ])z];

//angular spacing
totalRotation = 360*nTurns; 
helixStepsRotation = [for(ang=[0:totalRotation/(len(helixStepsZ)-1):totalRotation])ang*Helix_handedness];
helixAngle= atan(Lcorrected/(nTurns*PI*Outer_diameter));

module helixSegment(p1, r1, p2, r2){
    //function that draws two rectangles in arbitrary places with arbitrary rotations and hulls them together.
    hull(){
        translate(p1) //first rectangle location
        rotate(r1) //first rectangle rotation
        cube([Outer_diameter/2,infinitesimal,Rib_thickness]);
        translate(p2)//second rectangle location
        rotate(r2) //second rectangle rotation
        cube([Outer_diameter/2,infinitesimal,Rib_thickness]);
          };
}
//Now generate all the helix segments, one by one
module mainBody(){
for(thisPoint=[0:1:Helix_resolution]){
    //echo(thisPoint);
    if(thisPoint<Helix_resolution-1){
    helixSegment( [0,-infinitesimal/2,helixStepsZ[thisPoint]],
                  [helixAngle,0,helixStepsRotation[thisPoint]],
                  [0,-infinitesimal/2,helixStepsZ[thisPoint+1]],
                  [helixAngle,0,helixStepsRotation[thisPoint+1]]);
    }
}

//Create the spine and end pieces
//cylinder(h=Segment_length, r=Spine_thickness); //segment spine
cube([Spine_thickness,Spine_thickness,Segment_length]);
cylinder(h=End_thickness, r=Outer_diameter/2); //low Z-end
translate([0,0,Segment_length-End_thickness]) //go to the high z-end
    cylinder(h=End_thickness, r=Outer_diameter/2); //high Z-end
}

//working channel size and positioning
northLumenWallThickness = nWalls[1];
//northLumenDiameter=(tendonChannelPositionX*2)-(2*northLumenWallThickness);

northLumenY = (Outer_diameter/2)-northLumenDiameter/2-Layer_height*3;

southLumenWallThickness = nWalls[1];
//southLumenDiameter = (tendonChannelPositionX*2)-(2*southLumenWallThickness);

southLumenY = -1*((Outer_diameter/2)-southLumenDiameter/2-Flat_side-Layer_height*3);


westLumenWallThickness = Layer_height*3;
//westLumenDiameter = (tendonChannelPositionY*2)-(2*westLumenWallThickness);

westLumenX = ((Outer_diameter/2)-westLumenDiameter/2-nWalls[1]);

eastLumenWallThickness = Layer_height*3;
//eastLumenDiameter = (tendonChannelPositionY*2)-(2*eastLumenWallThickness);

eastLumenX = -((Outer_diameter/2)-eastLumenDiameter/2-nWalls[1]);


//function to detect when overly large working channels interfere
function lumenWallThickness(lumenAX,lumenAY,lumenADiameter,lumenBX,lumenBY,lumenBDiameter) =
    sqrt((lumenAX-lumenBX)^2+(lumenAY-lumenBY)^2)-(lumenADiameter/2+lumenBDiameter/2);
  
NWwall = lumenWallThickness(0,northLumenY,northLumenDiameter,westLumenX,0,westLumenDiameter);
//clamp lumen diameter 
finalNorthLumenDiameter = NWwall-minSep<=0 ? northLumenDiameter+((NWwall-minSep)) : northLumenDiameter;  

finalWestLumenDiameter = NWwall-minSep<=0 ? westLumenDiameter+((NWwall-minSep)) : westLumenDiameter;  
    
NEwall = lumenWallThickness(0,northLumenY,northLumenDiameter,eastLumenX,0,eastLumenDiameter);

finalEastLumenDiameter = NEwall-minSep<=0 ? eastLumenDiameter+(NEwall-minSep):eastLumenDiameter;

SWwall = lumenWallThickness(0,southLumenY,southLumenDiameter,westLumenX,0,westLumenDiameter);

finalSouthLumenDiameter = SWwall<=0 ? southLumenDiameter+(SWwall-minSep*2) : southLumenDiameter;

difference(){
    mainBody();
    //flat side
    translate([0,(-Outer_diameter/2)+Flat_side/2-0.1,Segment_length/2])cube([Outer_diameter,Flat_side+0.2,Segment_length+1],center=true);
    
    //tendon channels
    //North-east channel
    translate([tendonChannelPositionX,tendonChannelPositionY,-0.5])cube([Tendon_channel_size,Tendon_channel_size,Segment_length+1]);
    //South-west channel
    translate([-tendonChannelPositionX-Tendon_channel_size,-tendonChannelPositionY-Tendon_channel_size,-0.5])cube([Tendon_channel_size,Tendon_channel_size,Segment_length+1]);
    //North-west channel
    translate([-tendonChannelPositionX-Tendon_channel_size,tendonChannelPositionY,-0.5])cube([Tendon_channel_size,Tendon_channel_size,Segment_length+1]);
    //South-east channel
    translate([tendonChannelPositionX,-tendonChannelPositionY-Tendon_channel_size,-0.5])cube([Tendon_channel_size,Tendon_channel_size,Segment_length+1]); 
    
    //working channels
    //North working channel
    if(Render_north_channel){
    translate([0,northLumenY,-0.5])cylinder(h=Segment_length+1, r=finalNorthLumenDiameter/2);}
    //South working channel
    if(Render_south_channel){
    translate([0,southLumenY,-0.5])cylinder(h=Segment_length+1, r=finalSouthLumenDiameter/2);}
    //West working channel 
    if(Render_west_channel){
    translate([-westLumenX,0,-0.5])cylinder(h=Segment_length+1, r=finalWestLumenDiameter/2);} 
    //East working channel
    if(Render_east_channel){
    translate([-eastLumenX,0,-0.5])cylinder(h=Segment_length+1, r=finalEastLumenDiameter/2);}
    
    //tendon attachment groove
    grooveWidth=End_thickness - (nWalls[1]*2);
    
    grooveDepth=(Outer_diameter/2)-sqrt((tendonChannelPositionY-0)^2+(tendonChannelPositionX-0)^2)-sqrt(2*(Tendon_channel_size^2));
    
    translate([0,0,nWalls[3]])
    rotate_extrude()
    translate([Outer_diameter/2-grooveDepth, 0, 0])
    square([grooveDepth+1,grooveWidth]);
    
    translate([0,0,Segment_length-nWalls[3]-grooveWidth])
    rotate_extrude()
    translate([Outer_diameter/2-grooveDepth, 0, 0])
    square([grooveDepth+1,grooveWidth]);
    
    //tendon access notch
    notchSize=3;
    //north-west corner
    //proximal
    translate([tendonChannelPositionX,tendonChannelPositionY,nWalls[3]])
    cube([notchSize,notchSize,grooveWidth],center=false);
    //distal
    translate([tendonChannelPositionX,tendonChannelPositionY,Segment_length-nWalls[3]-grooveWidth])
    cube([notchSize,notchSize,grooveWidth],center=false);
    
    //north-east corner
    //proximal
    translate([-tendonChannelPositionX-notchSize,tendonChannelPositionY,nWalls[3]])
    cube([notchSize,notchSize,grooveWidth],center=false);
    //distal
    translate([-tendonChannelPositionX-notchSize,tendonChannelPositionY,Segment_length-nWalls[3]-grooveWidth])
    cube([notchSize,notchSize,grooveWidth],center=false);
    
    //south-west corner
    //proximal
    translate([tendonChannelPositionX,-tendonChannelPositionY-notchSize,nWalls[3]])
    cube([notchSize,notchSize,grooveWidth],center=false);
    //distal
    translate([tendonChannelPositionX,-tendonChannelPositionY-notchSize,Segment_length-nWalls[3]-grooveWidth])
    cube([notchSize,notchSize,grooveWidth],center=false);
    
    //south-east corner
    //proximal
    translate([-tendonChannelPositionX-notchSize,-tendonChannelPositionY-notchSize,nWalls[3]])
    cube([notchSize,notchSize,grooveWidth],center=false);
    //distal
    translate([-tendonChannelPositionX-notchSize,-tendonChannelPositionY-notchSize,Segment_length-nWalls[3]-grooveWidth])
    cube([notchSize,notchSize,grooveWidth],center=false);
    
    }