classdef (Abstract) Config
    % CONFIG  Contains settings.
    %
    % Subclasses should perform appropriate type-checking.


    methods
        function obj = Config(args)
            % CONFIG  Creates a new [Config] object, assigning properties using the
            % name-value pairs in [args].

            fn = fieldnames(args);
            for i = 1:numel(fn)
                obj.(fn{i}) = args.(fn{i});
            end
        end


        function new_obj = set(obj, args)
            % SET  Creates a copy of this [Config] instance, additionally assigning
            % properties using the name-value pairs in [args].

            new_obj = obj;

            fn = fieldnames(args);
            for i = 1:numel(fn)
                new_obj.(fn{i}) = args.(fn{i});
            end
        end

        function h = cache_id(obj)
            % CACHE_ID  Returns an ID that uniquely identifies the outputs of this config in
            % a cache.

            h = hash(obj);
        end
    end

    methods (Static)
        function partial_obj = partial(args)
            % PARTIAL  Creates a partial [Config] "object" that can be passed as a sequence
            % of name-value pairs to other `Config` methods as `partial_obj{:}`.
            %
            % For example, one can create a partial copy of an existing `Config` object as
            % follows:
            % ```matlab
            % old_config = Config();
            % partial = Config.partial(foo = "bar");
            %
            % new_config = config.set(partial{:});
            % ```

            fn = fieldnames(args);
            pairs = arrayfun(@(it) {fn{it}, args.(fn{it})}, 1:numel(fn), UniformOutput = false);
            partial_obj = [pairs{:}];
        end

        function combinations = combinations(varargin, args)
            % COMBINATIONS  Given multiple partial [Config]s, returns all possible
            % combinations.
            %
            % Each argument is a cell array containing multiple partial [Config]s. A
            % combination consists of the concatenation of one partial [Config] from each
            % argument.

            arguments (Repeating)% (Input, Repeating)
                varargin (:, 1) cell {mustBeNonempty};
            end
            arguments% (Input)
                args.constructor (1, 1) = 0;  % 0, or ((1, :) cell = PartialConfig) -> (1, 1) Config
            end
            % arguments (Output)
            %     combinations (1, :) cell;  % cell<cell> or cell<Config>
            % end

            argin = cellfun(@(arg) cellfun(@(it) mat2cell(it, 1, repelem(2, numel(it) / 2)), ...
                                           arg, ...
                                           UniformOutput=false), ...
                            varargin, ...
                            UniformOutput=false);

            cells = cell(numel(argin), 1);
            [cells{:}] = ndgrid(argin{:});

            combinations = cellfun(@(it) it(:), cells, UniformOutput = false)';
            combinations = [combinations{:}];
            combinations = arrayfun(@(it) horzcat(combinations{it, :}), 1:height(combinations), UniformOutput = false)';
            combinations = cellfun(@(it) horzcat(it{:}), combinations, UniformOutput = false)';

            if isa(args.constructor, "function_handle")
                combinations = cellfun(@(it) args.constructor(it{:}), combinations);
            end
        end
    end
end
