
library(vegan)
library(qiime2R)
library(tidyverse)


metadata2 <- read.table("metadata-KB-all-rsm-nalow-fmin1815.tsv", sep="\t", header=TRUE, stringsAsFactors=FALSE)
head(metadata2)
colnames(metadata2)

#--Uzip qiime2 artifact into R
#--Features are rows and samples are column
tableAllfiltered <- read_qza("table_2018-2019-rsm-fmin96-na-wp-noCM-low-fmin1815.qza")
counts <- t(tableAllfiltered$data)
dim(counts)

#--Split Taxon column
taxonomyAll <- read_qza("tax-132-v3v4-all-rep-seqs_2018-2019_rsm_fmin96.qza")
#str(taxonomyAll)
dim(taxonomyAll$data)


#--convert the table into a tabular split version
taxonomyAll <- taxonomyAll$data %>% as_tibble() %>% separate(Taxon, sep=";", c("Kingdom", "Phylum", "Class", "Order", "Family", "Genus", "Species"))

#--a check
#taxonomyAll[8, ]
#strsplit(as.character(taxonomyAll$data$Taxon)[8], split=";")[[1]]

#---read tree
treeAll <- read_qza("rooted_tree.qza")

abund <- as.data.frame(tableAllfiltered$data)

dim(abund)
dim(taxonomyAll)
dim(metadata2)

#tidy your data
metadata2<-metadata2[ order(metadata2$SampleID), ]
abund<-as.data.frame(t(abund))
abund<-abund[ order(row.names(abund)), ]



############################################################################

###########################################################################
#
# Only AB data: nmds
#

#Select data
#Remove E.coli treatments and manure
abund_AB<-subset(abund, subset = !metadata2$EcoliAB=="EC")
metadata_AB<-subset(metadata2, subset = !metadata2$EcoliAB=="EC") 

abund_AB<-subset(abund_AB, subset = !metadata_AB$SampleType3 =="start-manure")
metadata_AB<-subset(metadata_AB, subset = !metadata_AB$SampleType3 =="start-manure") 

#remove soil1 and soil2

abund_AB<-subset(abund_AB, metadata_AB$ShortDescr!="Soil1" & metadata_AB$ShortDescr!="Soil2")
metadata_AB<-subset(metadata_AB, metadata_AB$ShortDescr!="Soil1" & metadata_AB$ShortDescr!="Soil2")


#Normalize:
abund_norm <- decostand(abund_AB, method = "hellinger")


#NMDS
abund.mds<-metaMDS(abund_norm, distance="bray") 
rda <- rda(abund_norm)


#new figures

metadata_AB$plant2 <- paste(metadata_AB$Plant, metadata_AB$Year)

treat <- factor(metadata_AB$Treatment)
plant <- factor(metadata_AB$Plant)
plant2 <- factor(metadata_AB$plant2)
col.gr <- c("chartreuse2", "chartreuse4", "cornflowerblue", "tan3")
pchs <- c(15,16,17,18,3)

pdf("nmds_16S.pdf", w=7, h=7) #with colors
fig<-ordiplot(abund.mds, type="none", display="sites")
points(fig,"sites", pch = pchs[treat], col = col.gr[plant2])
title(main = "16S-Bacteria")
legend("bottomleft", legend = c("leek2018","leek2019", "lettuce", "soil"), 
       col=c("chartreuse2", "chartreuse4", "cornflowerblue", "tan3"), pch=c(16,16,16), 
       bty = "n", ncol=1)
legend("bottomright", legend = c("manure", "SF-E","SF-L","NPK","untreat.soil"), 
       pch=c(15,16,17,18,3), 
       bty = "n", ncol=1)
dev.off()


pdf("rda_16S.pdf", w=7, h=7) #with colors
fig<-ordiplot(rda, type="none", display="sites")
points(fig,"sites", pch = pchs[treat], col = col.gr[plant])
title(main = "16S-Bacteria")
legend("bottomleft", legend = c("leek", "lettuce", "soil"), 
       col=c("chartreuse4", "cornflowerblue", "tan3"), pch=c(16,16,16), 
       bty = "n", ncol=1)
legend("bottomright", legend = c("manure", "SF-E","SF-L","NPK","untreat.soil"), 
       pch=c(15,16,17,18,3), 
       bty = "n", ncol=1)
dev.off()

anosim(vegdist(abund_AB, method="bray"), metadata_AB$Treatment) 

  # ANOSIM statistic R: 0.0397 
  # Significance: 0.063 

anosim(vegdist(abund_AB, method="bray"), metadata_AB$Plant) 
  # ANOSIM statistic R: 0.8534 
  # Significance: 0.001 

anosim(vegdist(abund_AB, method="bray"), metadata_AB$SampleType2) 
  # ANOSIM statistic R: 0.2824 
  # Significance: 0.001 


anosim(vegdist(abund_AB, method="bray"), metadata_AB$plant2) 
  # ANOSIM statistic R: 0.8208 
  # Significance: 0.001


adonis2(abund_AB ~ plant2*Treatment, method="bray", data= metadata_AB)

adonis2(abund_AB ~ SampleType2*Treatment, method="bray", data= metadata_AB)




#Alpha Diversity 

library(ggplot2)

abund_AB <- t(abund_AB)
taxonomy_filtered <- taxonomyAll[taxonomyAll$Feature.ID %in% rownames(abund_AB) , ]
taxonomy_filtered[is.na(taxonomy_filtered)] <- "undetermined"

rownames(metadata_AB) <- metadata_AB$SampleID
rownames(taxonomy_filtered) <- taxonomy_filtered$Feature.ID

OTU = otu_table(abund_AB, taxa_are_rows = TRUE)
TAX = tax_table(as.matrix(taxonomy_filtered))
sampledata=sample_data(metadata_AB)

physeq = phyloseq(OTU, TAX, sampledata)

physeq

theme_set(theme_bw())
plot_richness(physeq, x="Treatment", measures=c("Shannon"), color="Plant")
ggsave("diversity-treatment.pdf")

plot_richness(physeq, x="Plant", measures=c("Shannon"), color="Treatment")
ggsave("diversity-plant.pdf")


results = estimate_richness(physeq, measures = 'Shannon')
d = sample_data(physeq)
d$Shannon <- results$Shannon
write.csv(d, file = "diversity.csv")

ggplot(d$Shannon)

lettuce = results[d[,'Plant'] == 'lettuce',]
leek = results[d[,'Plant'] == 'leek',]
t.test(lettuce, leek)
wilcox.test(lettuce, leek)

#split datasets (leek and lettuce)
ps.lk <- subset_samples(physeq, Plant=="leek")
ps.lt <- subset_samples(physeq, Plant=="lettuce")


library(viridis)
library(ggpubr)

shannon <- read.csv("diversity.csv", h=T, row.names = 1)

#remove untreat.soil
shannon <- subset(shannon, subset = !metadata_AB$SampleDescr1=="untreated soil") 

ggplot(shannon, aes(x=Treatment, y=Shannon, fill=Treatment)) + 
  geom_boxplot() + theme_bw() +
  theme(axis.line = element_line(colour = "black"),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank()) +
  labs(y= "H'index", x = "crop") + 
  scale_fill_brewer(palette="Set3") +
  facet_wrap(~plant2, scale="fixed")
ggsave("16S_diversity-boxplot-plant.pdf")


supp.labs <- c("S-M", "S-NPK","S-MSE", "S-MSL")
names(supp.labs) <- c("manure", "NPK", "SF-E", "SF-L")

ggplot(shannon, aes(x=plant2, y=Shannon, fill=plant2)) + 
  geom_boxplot(fill="lightgray") + theme_bw() +
  theme(axis.line = element_line(colour = "black"),
        axis.text.x = element_text(angle = 45, hjust = 1),
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.border = element_blank(),
        panel.background = element_blank()) +
  labs(y= "H'index", x = "Treatment") + 
  facet_wrap(~Treatment, scale="fixed", labeller = labeller(Treatment = supp.labs))
ggsave("16S_diversity-boxplot-treatment.pdf", w=5,h=5)


library(ggstatsplot)
library(car)
library(FSA)

shapiro.test(shannon$Shannon) # data not normal
leveneTest(Shannon ~ Treatment, data = shannon) # variance between treatments is the same

kruskal.test(Shannon ~ Treatment, data = shannon) # ns
kruskal.test(Shannon ~ plant2, data = shannon) # p-value=1.572e-09

dunnTest(Shannon ~plant2, data = shannon, method="bonferroni")


p <- ggbetweenstats( data = shannon, 
                x = plant2, 
                y = Shannon,
    type = "nonparametric", #Kruskal-Wallis
    plot.type = "box",
    pairwise.comparisons = TRUE,
    pairwise.display = "significant",
    centrality.plotting = FALSE,
     bf.message = FALSE
   )

ggsave("diversity_boxplot_rhizosphere_kruskall_plant.pdf", p, width = 8, height = 9)


p2 <- grouped_ggbetweenstats(
  data = shannon,
  x = plant2,
  y = Shannon,
  grouping.var = Treatment,
  type = "nonparametric", #Kruskal-Wallis
  plot.type = "box",
  pairwise.comparisons = TRUE,
  pairwise.display = "significant",
  centrality.plotting = FALSE,
  bf.message = FALSE
)

ggsave("diversity_boxplot_rhizosphere_kruskall_plant_facet.pdf", p2, width = 10, height = 10)
