
function theta = RunSim(Par)
%% Finite difference method 
% Common parameters
Dm = 0.1;               % micro m^2 per s
Dc = 10;                % micro m^2 per s
n = 4;
     
%% Parameters Cdc42
Ic = Par(1);            % [microM/s]
delta_C = Par(2);       % [1/s]
Ctot = 4.4;             % [microM]
Dca = Dm;               % [micrometer^2/s]
Dci = Dc;               % [micrometer^2/s]
alpha_C = Par(3);       % [1/s] 

%% Parameters Rac
Ir = Par(4);            % [microM/s]
delta_R = Par(5);       % [1/s]
Rtot = 2.4;             % [microM]
Dra = Dm;                % [micrometer^2/s]
Dri = Dc;              % [micrometer^2/s]
beta_R = Par(6);        % [microM]  

%% Parameters Rho
Irho = Par(7);          % [micro M/s]
delta_Rho = Par(8);     % [1/s]
Rhotot = 4;             % [microM]
Drhoa = Dm;             % [micrometer^2/s]
Drhoi = Dc;             % [micrometer^2/s]
beta_rho = Par(9);      % [microM]
alpha_rho=Par(10);

%% Parameters PAR
k_on_P = Par(11);       %[microM/s]
k_off_P = Par(12);      % [1/s]
Ptot = 1.6;             % [microM]
Dpc = 0.2;              % [micrometer^2/s]
Dpun = 2;               % [micrometer^2/s]
beta_PR = Par(13);      % [microM]
beta_PS = Par(14);      % [microM]

%% Parameters Scrib
k_on_S = Par(15);       % [microM]
k_off_S = Par(16);      % [1/s]
Stot = 1.6;             % [microM]
Dsc = 0.2;              % [micrometer^2/s]
Dsun = 2;               % [micrometer^2/s]
beta_SP = Par(17);      % [microM]

%% Parameters Integrin
I_I = Par(18);          % [microM/s]       
delta_I = Par(19);      % [1/s]
Itot = Par(20);         % [microM]
D_Ii = 10;              % [micrometer^2/s]
alpha_I = Par(21);      % [1/s]
l=1;     % If activation by integrin is on left side of the cell (l=1)
r=0;     % If activation by integrin is on right side of the cell (r=1)

%% Common parameters
x_max=10;               % Length, L, [micrometer]
Nx=30;                  % Number of divisions along length, [-]
delta_x=x_max/Nx;       % Length increment
x=0:delta_x:x_max;      % Position, x, [micrometer]

t_max=100;              % Simulation time, t, [s]           
delta_t=0.005;          % Time increment, [s]              
Nt=t_max/delta_t;       % Number of time steps, [-]
t=0:delta_t:t_max;      % Time, t, [s]

%% Allocation
Ca=zeros(length(x),1); Ci=zeros(length(x),1);
Rhoa=zeros(length(x),1); Rhoi=zeros(length(x),1);
Ra=zeros(length(x),1); Ri=zeros(length(x),1);
Pc=zeros(length(x),1); Pun=zeros(length(x),1);
Sc=zeros(length(x),1); Sun=zeros(length(x),1);
Ia=zeros(length(x),1); Ii=zeros(length(x),1);
A=zeros(12*length(x),12*length(x)); Vektor=zeros(12*length(x),length(t));
coupl=zeros(12*length(x),length(t)); RHS=zeros(12*length(x),length(t));

%% Definition of matrix A
%% ************* Cdc42 *********************
b_Ca=1/delta_t-delta_C-2*Dca/(delta_x^2);            % Cdc42 active
a_Ca=Dca/(delta_x^2);                             
c_Ca=Dca/(delta_x^2);                           
b_Ci=1/delta_t-2*Dci/(delta_x^2);                    % Cdc42 inactive
a_Ci=Dci/(delta_x^2);                           
c_Ci=Dci/(delta_x^2);                           
% Coupling of Ci to Ca
d_Ci_Ca=delta_C;

%% ************* Rac1 *********************
b_Ra=1/delta_t-delta_R-2*Dra/(delta_x^2);            % Rac active
a_Ra=Dra/(delta_x^2);                             
c_Ra=Dra/(delta_x^2);                             
b_Ri=1/delta_t-2*Dri/(delta_x^2);                    % Rac inactive
a_Ri=Dri/(delta_x^2);                           
c_Ri=Dri/(delta_x^2);                           
% Coupling of Ri to Ra
d_Ri_Ra=delta_R;

%% ************* Rho *********************
b_rhoa=1/delta_t-delta_Rho-2*Drhoa/(delta_x^2);      % Rho active
a_rhoa=Drhoa/(delta_x^2);                           
c_rhoa=Drhoa/(delta_x^2);                           
b_rhoi=1/delta_t-2*Drhoi/(delta_x^2);                % Rho inactive
a_rhoi=Drhoi/(delta_x^2);                         
c_rhoi=Drhoi/(delta_x^2);                         
% Coupling of Rhoi to Rhoa
d_Rhoi_Rhoa=delta_Rho;

%% ************* Par *********************
b_Pc=1/delta_t-k_off_P-2*Dpc/(delta_x^2);            % PAR complex
a_Pc=Dpc/(delta_x^2);                             
c_Pc=Dpc/(delta_x^2);                             
b_Pun=1/delta_t-2*Dpun/(delta_x^2);                  % PAR unbounded
a_Pun=Dpun/(delta_x^2);                           
c_Pun=Dpun/(delta_x^2);                           
% Coupling of Pun to Pc
d_Pun_Pc = k_off_P;

%% ************* Scribble *********************
b_Sc=1/delta_t-k_off_S-2*Dsc/(delta_x^2);            % Scribble complex
a_Sc=Dsc/(delta_x^2);                             
c_Sc=Dsc/(delta_x^2);                             
b_Sun=1/delta_t-2*Dsun/(delta_x^2);                  % Scribble unbounded
a_Sun=Dsun/(delta_x^2);                           
c_Sun=Dsun/(delta_x^2);                           
% Coupling of Sun to Sc
d_Sun_Sc=k_off_S;

%% *************** Integrin ******************
% along the cell
b_Ia=0;                                             % I active
a_Ia=0;                            
c_Ia=0;                           
b_Ii=1/delta_t-2*D_Ii/(delta_x^2);                  % I inactive
a_Ii=D_Ii/(delta_x^2);                           
c_Ii=D_Ii/(delta_x^2);                          
% at the membrane
d_I_Ii=I_I/Itot;
b_Ia_lr=1/delta_t-delta_I;
b_Ii_lr=b_Ii-I_I/Itot;
d_Ii_I=delta_I;

%% Creating matrix A
for i=2:length(x)-1
    A(i,i)=b_Ca;                                        % 1) Ca
    A(i,i-1)=a_Ca;
    A(i,i+1)=c_Ca;
    A(length(x)+i,length(x)+i)=b_Ra;                    % 2) Ra
    A(length(x)+i,length(x)+i-1)=a_Ra;
    A(length(x)+i,length(x)+i+1)=c_Ra;
    A(2*length(x)+i,2*length(x)+i)=b_rhoa;              % 3) Rhoa
    A(2*length(x)+i,2*length(x)+i-1)=a_rhoa;
    A(2*length(x)+i,2*length(x)+i+1)=c_rhoa;
    A(3*length(x)+i,3*length(x)+i)=b_Ci;                % 4) Ci
    A(3*length(x)+i,3*length(x)+i-1)=a_Ci;
    A(3*length(x)+i,3*length(x)+i+1)=c_Ci;
    A(4*length(x)+i,4*length(x)+i)=b_Ri;                % 5) Ri
    A(4*length(x)+i,4*length(x)+i-1)=a_Ri;
    A(4*length(x)+i,4*length(x)+i+1)=c_Ri;
    A(5*length(x)+i,5*length(x)+i)=b_rhoi;              % 6) Rhoi
    A(5*length(x)+i,5*length(x)+i-1)=a_rhoi;
    A(5*length(x)+i,5*length(x)+i+1)=c_rhoi;
    A(6*length(x)+i,6*length(x)+i)=b_Pc;                % 7) PAR, Pc
    A(6*length(x)+i,6*length(x)+i-1)=a_Pc;
    A(6*length(x)+i,6*length(x)+i+1)=c_Pc;
    A(7*length(x)+i,7*length(x)+i)=b_Pun;               % 8) Pun
    A(7*length(x)+i,7*length(x)+i-1)=a_Pun;
    A(7*length(x)+i,7*length(x)+i+1)=c_Pun;
    A(8*length(x)+i,8*length(x)+i)=b_Sc;                % 9) Scribble, Sc
    A(8*length(x)+i,8*length(x)+i-1)=a_Sc;
    A(8*length(x)+i,8*length(x)+i+1)=c_Sc;
    A(9*length(x)+i,9*length(x)+i)=b_Sun;               % 10) Sun
    A(9*length(x)+i,9*length(x)+i-1)=a_Sun;
    A(9*length(x)+i,9*length(x)+i+1)=c_Sun;   
    A(10*length(x)+i,10*length(x)+i)=b_Ia;              % 11) Ia
    A(10*length(x)+i,10*length(x)+i-1)=a_Ia;
    A(10*length(x)+i,10*length(x)+i+1)=c_Ia;
    A(11*length(x)+i,11*length(x)+i)=b_Ii;              % 12) Ii
    A(11*length(x)+i,11*length(x)+i-1)=a_Ii;
    A(11*length(x)+i,11*length(x)+i+1)=c_Ii;

    % Coupling members (linear) between variables
    A(3*length(x)+i,i)=d_Ci_Ca;
    A(4*length(x)+i,length(x)+i)=d_Ri_Ra;
    A(5*length(x)+i,2*length(x)+i)=d_Rhoi_Rhoa;
    A(7*length(x)+i,6*length(x)+i)=d_Pun_Pc;
    A(9*length(x)+i,8*length(x)+i)=d_Sun_Sc;
end

%% Boundary conditions, no flux exchange at the borders, no protein passage through cell membrane
    A(1,1)=a_Ca+b_Ca;                               % Ca 
    A(1,2)=c_Ca;
    A(length(x),length(x))=b_Ca+c_Ca; 
    A(length(x),length(x)-1)=a_Ca;
    A(length(x)+1,length(x)+1)=a_Ra+b_Ra;           % Ra 
    A(length(x)+1,length(x)+2)=c_Ra;                                                                                                                                                                                                                                                                                                                                                                                                                                      
    A(2*length(x),2*length(x))=b_Ra+c_Ra; 
    A(2*length(x),2*length(x)-1)=a_Ra;
    A(2*length(x)+1,2*length(x)+1)=a_rhoa+b_rhoa;   % Rhoa 
    A(2*length(x)+1,2*length(x)+2)=c_rhoa;                                                                                                                                                                                                                                                                                                                                                                                                                                        
    A(3*length(x),3*length(x))=b_rhoa+c_rhoa; 
    A(3*length(x),3*length(x)-1)=a_rhoa;
    A(3*length(x)+1,3*length(x)+1)=a_Ci+b_Ci;       % Ci 
    A(3*length(x)+1,3*length(x)+2)=c_Ci;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(4*length(x),4*length(x))=b_Ci+c_Ci; 
    A(4*length(x),4*length(x)-1)=a_Ci;
    A(4*length(x)+1,4*length(x)+1)=a_Ri+b_Ri;       % Ri 
    A(4*length(x)+1,4*length(x)+2)=c_Ri;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(5*length(x),5*length(x))=b_Ri+c_Ri; 
    A(5*length(x),5*length(x)-1)=a_Ri;    
    A(5*length(x)+1,5*length(x)+1)=a_rhoi+b_rhoi;   % Rhoi 
    A(5*length(x)+1,5*length(x)+2)=c_rhoi;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(6*length(x),6*length(x))=b_rhoi+c_rhoi; 
    A(6*length(x),6*length(x)-1)=a_rhoi;
    A(6*length(x)+1,6*length(x)+1)=a_Pc+b_Pc;       % Pc 
    A(6*length(x)+1,6*length(x)+2)=c_Pc;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(7*length(x),7*length(x))=b_Pc+c_Pc; 
    A(7*length(x),7*length(x)-1)=a_Pc; 
    A(7*length(x)+1,7*length(x)+1)=a_Pun+b_Pun;     % Pun 
    A(7*length(x)+1,7*length(x)+2)=c_Pun;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(8*length(x),8*length(x))=b_Pun+c_Pun; 
    A(8*length(x),8*length(x)-1)=a_Pun;    
    A(8*length(x)+1,8*length(x)+1)=a_Sc+b_Sc;       % Sc 
    A(8*length(x)+1,8*length(x)+2)=c_Sc;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(9*length(x),9*length(x))=b_Sc+c_Sc; 
    A(9*length(x),9*length(x)-1)=a_Sc;     
    A(9*length(x)+1,9*length(x)+1)=a_Sun+b_Sun;     % Sun
    A(9*length(x)+1,9*length(x)+2)=c_Sun;                                                                                                                                                                                                                                                                                                                                                                                                                                            
    A(10*length(x),10*length(x))=b_Sun+c_Sun;
    A(10*length(x),10*length(x)-1)=a_Sun;  
    
    % Activated integrin on left or right side of the cell
    if l==1
        for m=1:5
    % left border - coupling
    A(10*length(x)+m,10*length(x)+m)=b_Ia_lr;           % Ia
    A(10*length(x)+m,10*length(x)+m+1)=c_Ia;
        if m==1
        A(11*length(x)+m,11*length(x)+m)=b_Ii_lr+a_Ii;  % Ii
        A(11*length(x)+m,11*length(x)+m+1)=c_Ii;
        else
        A(11*length(x)+m,11*length(x)+m)=b_Ii_lr;       % Ii
        end
    A(10*length(x)+m,11*length(x)+m)=d_I_Ii;
    A(11*length(x)+m,10*length(x)+m)=d_Ii_I;
        end
    % right border - no coupling
    A(11*length(x),11*length(x))=b_Ia+c_Ia;             % Ia
    A(11*length(x),11*length(x)-1)=a_Ia;
    A(12*length(x),12*length(x))=b_Ii+c_Ii;             % Ii
    A(12*length(x),12*length(x)-1)=a_Ii;
    end
	
    if r==1      
    % left border - no coupling
    A(10*length(x)+1,10*length(x)+1)=b_Ia+a_Ia;         % Ia
    A(10*length(x)+1,10*length(x)+2)=a_Ia;
    A(11*length(x)+1,11*length(x)+1)=b_Ii+a_Ii;         % Ii
    A(11*length(x)+1,11*length(x)+2)=c_Ii;
      for m=1:5
    % right border - coupling      
    A(11*length(x)+1-m,11*length(x)+1-m)=b_Ia_lr;               % Ia
    A(11*length(x)+1-m,11*length(x)-1+1-m)=a_Ia;
        if m==1
        A(12*length(x)+1-m,12*length(x)+1-m)=b_Ii_lr+c_Ii;      % Ii
        A(12*length(x)+1-m,12*length(x)-1+1-m)=a_Ii;
        else
        A(12*length(x)+1-m,12*length(x)+1-m)=b_Ii_lr;           % Ii
        end
    A(11*length(x)+1-m,12*length(x)+1-m)=d_I_Ii;
    A(12*length(x)+1-m,11*length(x)+1-m)=d_Ii_I;
      end
    end
    
    % Linear coupling members on boarders (left and right side)
    A(3*length(x)+1,1)=d_Ci_Ca;
    A(4*length(x),length(x))=d_Ci_Ca;
    A(4*length(x)+1,length(x)+1)=d_Ri_Ra;
    A(5*length(x),2*length(x))=d_Ri_Ra;
    A(5*length(x)+1,2*length(x)+1)=d_Rhoi_Rhoa;
    A(6*length(x),3*length(x))=d_Rhoi_Rhoa;   
    A(7*length(x)+1,6*length(x)+1)=d_Pun_Pc;
    A(8*length(x),7*length(x))=d_Pun_Pc;
    A(9*length(x)+1,8*length(x)+1)=d_Sun_Sc;
    A(10*length(x),9*length(x))=d_Sun_Sc;
    
%% Set initial values
Ca(:,1)=1.75;         % [micrometer]
Ci(:,1)=2.65;         % [micrometer]
Ra(:,1)=1;            % [micrometer]
Ri(:,1)=1.4;          % [micrometer]
Rhoa(:,1)=0.8;        % [micrometer]
Rhoi(:,1)=3.2;        % [micrometer]
Pc(:,1)=0.6;          % [micrometer]
Pun(:,1)=1;           % [micrometer]
Sc(:,1)=0.6;          % [micrometer]
Sun(:,1)=1;           % [micrometer]

 for i=1:length(x)
     % Activation at left side
     if l==1
        if i<=5 
            Ia(i,1)=(Itot/i^2)*exp(1/i^2-1);
        else
            Ia(i,1)=0;
        end
     end
     % Activation at right side
     if r==1
     if i<=length(x)-5
        Ia(i,1)=0;     
     else
        Ia(i,1)=(Itot/(length(x)+1-i)^2)*exp(1/((length(x)+1-i))^2-1); 
     end
     end
 end
 Ii(:,1)=Itot-Ia(:,1);

Vector(1:length(x),1)=Ca(:,1);
Vector(length(x)+1:2*length(x),1)=Ra(:,1);
Vector(2*length(x)+1:3*length(x),1)=Rhoa(:,1);
Vector(3*length(x)+1:4*length(x),1)=Ci(:,1);
Vector(4*length(x)+1:5*length(x),1)=Ri(:,1);
Vector(5*length(x)+1:6*length(x),1)=Rhoi(:,1);
Vector(6*length(x)+1:7*length(x),1)=Pc(:,1);
Vector(7*length(x)+1:8*length(x),1)=Pun(:,1);
Vector(8*length(x)+1:9*length(x),1)=Sc(:,1);
Vector(9*length(x)+1:10*length(x),1)=Sun(:,1);
Vector(10*length(x)+1:11*length(x),1)=Ia(:,1);
Vector(11*length(x)+1:12*length(x),1)=Ii(:,1);

%% Nonlinear coupling members from equations and solving equations
k1=0;
for j=1:length(t)-1      
        for i=1:length(x)
            coupl(i,j)=(Ic+alpha_rho*Vector(2*length(x)+i,j))*(Vector(3*length(x)+i,j)/Ctot); 
            coupl(length(x)+i,j)=((Ir+alpha_I*Vector(10*length(x)+i,j))/(1+(Vector(2*length(x)+i,j)^n)/(beta_rho^n)+(Vector(6*length(x)+i,j)^n)/(beta_PR^n)))*(Vector(4*length(x)+i,j)/Rtot);
            coupl(2*length(x)+i,j)=(Irho/(1+((Vector(length(x)+i,j)^n)/(beta_R^n))))*(Vector(5*length(x)+i,j)/Rhotot); 
            coupl(3*length(x)+i,j)=-coupl(i,j);
            coupl(4*length(x)+i,j)=-coupl(length(x)+i,j);
            coupl(5*length(x)+i,j)=-coupl(2*length(x)+i,j);
            coupl(6*length(x)+i,j)=((k_on_P+alpha_C*Vector(i,j))/(1+((Vector(8*length(x)+i,j)^n)/(beta_SP^n))))*(Vector(7*length(x)+i,j)/Ptot);
            coupl(7*length(x)+i,j)=-coupl(6*length(x)+i,j);
            coupl(8*length(x)+i,j)=(k_on_S/(1+((Vector(6*length(x)+i,j)^n)/(beta_PS^n))))*(Vector(9*length(x)+i,j)/Stot);
            coupl(9*length(x)+i,j)=-coupl(8*length(x)+i,j); 
        end
        
    RHS(:,j)=A(:,:)*Vector(:,j)+coupl(:,j);
    Vector(:,j+1)=delta_t*RHS(:,j);

    % Check if values go below zero
    for i=1:12*length(x)
        if Vector(i,j+1)<0
            Vector(i,j+1)=0;
            k1=k1+1;
        end           
    end
end

%% Plotting the results
y1=Vektor(1:length(x),length(t));                   % Ca
y2=Vektor(length(x)+1:2*length(x),length(t));       % Ra
y3=Vektor(2*length(x)+1:3*length(x),length(t));     % Rhoa
y4=Vektor(3*length(x)+1:4*length(x),length(t));     % Ci
y5=Vektor(4*length(x)+1:5*length(x),length(t));     % Ri
y6=Vektor(5*length(x)+1:6*length(x),length(t));     % Rhoi
y7=Vektor(6*length(x)+1:7*length(x),length(t));     % Pc
y8=Vektor(7*length(x)+1:8*length(x),length(t));     % Pun
y9=Vektor(8*length(x)+1:9*length(x),length(t));     % Sc
y10=Vektor(9*length(x)+1:10*length(x),length(t));   % Sun
y11=Vektor(10*length(x)+1:11*length(x),length(t));  % Ia
y12=Vektor(11*length(x)+1:12*length(x),length(t));  % Ii

theta = [y1,y2,y3,y7,y9,y11];

end