%% Get ambiguity-resolved detector (ARD) critical value
% This function computes the ARD critical value based on the lookup table
% and observation model parameters 
%
% -------------------------------------------------------------------------
%_INPUTS            
%  lookup_table  Struct array, contains the polynomial coefficients
%  f             Number of frequency of the model
%  ra            Ambiguity-known redundancy of the model
%  alpha         User specified significance level
%  IB_SR         Success rate of the integer-bootstrapping estimator
%
%_OUTPUTS           
%  k             ARD critical value / NaN if critical value is not computed
%
% -------------------------------------------------------------------------
% Copyright (c) - 2024 - Geoscience & Remote Sensing department @ TUDelft
%
% Authors: 
%   - Chengyu Yin       (C.yin@tudelft.nl)
%
% Changelog:
%   [2024-11-22] 
%       First version of function
%
% Last Modified: 2024-11-22
% -------------------------------------------------------------------------

function k = get_ARD_ILS_critical_value(lookup_table, f, ra, alpha, IB_SR)

% Check success rate
if IB_SR < 0.8 
    warning('IB success rate is lower than 80%');
    k      = nan;
    return
end

table   = lookup_table(f); % Take table accroding to number of frequencies

% Get location of coefficients according to indexes
i_alpha = find(table.alpha_list == alpha);
if isempty(i_alpha)
    warning('Input significance level is not included in the ARD lookup table');
    k      = nan;
    return
end

i_ra    = find(table.ra_list==ra);
if isempty(i_ra)
    warning(['Model with redundancy ' num2str(ra) ' is not covered by the ARD lookup table']);
    k      = nan;
    return
end

% Take coefficients from lookup table
a0 = table.a0(i_ra, i_alpha);
a1 = table.a1(i_ra, i_alpha);
a2 = table.a2(i_ra, i_alpha);

% Compute the ARD critical value
x  = 1 - IB_SR;
k  = a0 + a1*x + a2*x^2;

end