function [Gusts]= ReadFolder_tail(Wing,Angle,Freq)
    % Function that reads the the dynamic case for "Wing" at AoA = "Angle"
    % and gust frequency = Freq
    % Input: 
    %   - Wing: String with wing folder name
    %   - Angle: String with AoA folder name
    %   - Freq: String with gust frequency folder name
    % Output:
    %   - Gusts: Structure with the dynamic results of a wing for both 
    %             locked and free conditions
    % Note: Different than "ReadFolder.m" because of naming convention of wings.
    % however, the procedure is the same

    % Important constants
    Delta = 1750;   %Number of samples per measurement
    Epsilon = 30;   %Outliers filter input: radius to consider cluster
    Npoints = 1;    %Outliers filter input: min points to consider cluster
    filter = 440;   %SR for SR filter
    % Go into desired directory
    cd(Wing);
    cd('Dynamic');
    cd(Angle);
    cd(Freq);

    % List files in directory
    S=dir;
    L=length(dir);
    
    % Go through files
    k=1;    %Index for saving in final structure
    
    for i=3:L
        % Go into 'i'th file in the directory
        % Note i=1 and i=2 are not files.
        file = S(i).name;
        fid = fopen(file, 'rt');
        tline = fgetl(fid);
        headers = strsplit(tline, ',');     %a cell array of strings
        headers = headers(1:13);    %remove extra titles (error when saving file)
        headers(13) = {'Trigger'};
        datacell = textscan(fid, '%f%f%f%f%f%f%f%f%f%f%f%f%f', 'Delimiter',',', 'CollectOutput', 1);
        fclose(fid);
        datavalues = datacell{1};    %Save data as numeric array
        
        %Extract information from file name
        ix=strfind(file,'_');        %Find positions of '_' in file name
        release = string(file(ix(3)+1:1:ix(4)-1));      %Release condition 
    
        Trigger = datavalues(:,13);
        TT = size(Trigger,1);
        Shift = zeros(TT,1);
        
        %Find the pulse from the gust generator within the recorded signal
        Trigger(:,1) = datavalues(:,13)>2.5;
        
        %Find in which sample where trigger changed from 0 to 1
        for j=2:1:TT
            if Trigger(j-1,1)==0 && Trigger(j,1)==1
                Shift(j,1)=1;
            end
        end
        
        %Find index where trigger is detected
        Shifter = find(Shift==1);
        
        %Divide Measurement in N runs of same gust
        N = size(Shifter,1);
        
        %Lowpass filter for accelerometers
         lpFilt = designfilt('lowpassiir','FilterOrder',8, ...
         'PassbandFrequency',10,'PassbandRipple',0.2, ...
         'SampleRate',465);
     
        %Filter whole accelerometer signal
        %If filtered after splitting signal, results do not match the
        %signal
        Filt_AF = filtfilt(lpFilt,datavalues(:,4));
        Filt_AR = filtfilt(lpFilt,datavalues(:,6));
        
        %Matrix with 'Delta' samples for each run (N-1)
        Results = zeros(Delta,13,N-1);
        for j=1:N-1
            ref0 = Shifter(j);
            ref1 = ref0 + Delta-1;
            Results(:,:,j) = datavalues(ref0:ref1,:);
            Results(:,1,j) = Results(:,1,j)- Results(1,1,j);
            
            %Save accelerometer filtered data
            Results(:,5,j) = Filt_AF(ref0:ref1);
            Results(:,6,j) = Filt_AR(ref0:ref1);
            
            %Convert servo position into binary (1 for locked, 0 for free) 
            Results(:,12,j) = abs(Results(:,12,j))>3;   
            
            %Convert trigger signal into binary (1 for trigger, 0 for none) 
            Results(:,13,j) = Results(:,13,j)>2.5;
        end
               
        %Compute Sampling rate of each run
        SR=zeros(N-1,1);
        %SR= Samples/time between first and last samples
        SR(:) = Delta/(Results(Delta,1,:)-Results(1,1,:));
        remove = find(SR<filter);  %Find where SR is lower than required
        
        %Remove data with SR below desired
        SR(remove)=[];          
        Results(:,:,remove)=[];
        
        %Filter outliers
        N = size(SR,1);
        Max_M = zeros(N,1);
        for j=1:1:N
            %Find peak value for each run
            Max_M(j) = max(Results(:,11,j));
        end
        
        %Find clusters within Max moment array
        idx = dbscan(Max_M,Epsilon,Npoints);
        Oclust=unique(idx); %Find unique values of cluster identificator
        %Oclust is an array with the values that the idx array can have
        Ncount=histc(idx,Oclust);%Find repetitions of Oclust values
        %In other words, find number of samples per cluster
        %Find the cluster with more repetitions(theoretically the most
        %consistent) Only interested in the index
        [~,index]=max(Ncount);
        remove=idx~=Oclust(index);
        Results(:,:,remove)=[];
        SR(remove)=[];
        
        if k==1 %If it is the first file of the list
            Gusts(k).release = release;
            Gusts(k).headers = headers;
            Gusts(k).SR=SR;
            Gusts(k).results = Results;
            k=k+1;
        else
            %If not the first file of the list
            j = 1;
            Found = false;
            %Check if the test case was already in the list
            while j<=k-1 && Found==false
                if(release==Gusts(j).release)
                    Found = true;
                end
                j=j+1;
            end
            if Found == true
                %If test case was in the list, add the data at the end of
                %that line of the struct
                A = Gusts(j-1).results;
                SRa = Gusts(j-1).SR;
                B = Results;
                SRb=SR;
                m = size(A,3);
                n = size(B,3);
                C = zeros(Delta,13,m+n);
                C(:,:,1:1:m)=A;
                C(:,:,m+1:1:m+n)=B;
                Gusts(j-1).results = C;
                Gusts(j-1).SR= [SRa; SRb];
                clearvars A B m n C SRa SRb
            else
                %If the test case wasn't in the list, add it to the struct
                Gusts(k).headers = headers;
                Gusts(k).release = release;
                Gusts(k).results = Results;
                Gusts(k).SR=SR;
                k=k+1;
            end
        end
    end

    cd ..
    cd ..
    cd ..
    cd ..
end
