% MATLAB script belonging to the paper:
% De Winter, J. C. F., Dodou, D., & Eisma, Y. B. (2023). Responses to Raven Matrices: Governed by Visual Complexity and Centrality? Perception.
% Department of Cognitive Robotics, Mekelweg 2, 2628 CD Delft, Delft University of Technology, Delft, The Netherlands
% j.c.f.dewinter@tudelft.nl

clear all;close all;clc %#ok<CLALL>

itemnames={'Item1.png','Item4.png','Item8.png','Item11.png','Item15.png','Item18.png','Item21.png','Item23.png','Item25.png','Item30.png','Item31.png','Item35.png'}; % file names of images
II=[1 4 8 11 15 18 21 23 25 30 31 35]; % item numbers of items used in the eye-tracking study
CorrectAnswers=[5 4 1 5 2 7 8 6 7 5 4 3]; % Correct answers of the 12 Raven items
load('filtered_eye_tracking_data_raven.mat') % Load blinking info, response times (DT), fixation and saccade data, gaze x/y data, and pupil diameter data
load('edge_density_and_response_times_table_s3_figure_s1_s2_s3.mat') % Load edge density of all 36 items, and response times of other studies, retrieved from the literature

% Contains participant x item x time sample matrices of
% Blink (BB)
% Pupil diameter (PP)
% X point in pixels (XXr)
% Y gaze point (YYr)
% As well as matrices of fixations (per trial: FFr; all: FFra) and saccade statistics (per trial: SSr; all: SSra)
% For this paper, only the X and Y gaze point data and fixation data is used.

ppvalideye=1:165;ppvalideye([30 44 56 93 109 153])=[];  % In the previous study, the following participants were excluded: 30, 44, 56, 93, 109, 153

% Previous papers using the same participants:
% Eisma, Y. B., & De Winter, J. C. F. (2020). How do people perform an inspection time task? An examination of visual illusions, task experience, and blinking. Journal of Cognition, 3, 34. https://doi.org/10.5334/joc.123
% Eisma, Y. B., Reiff, A., Kooijman, L., Dodou, D., & De Winter, J. C. F. (2021). External human-machine interfaces: Effects of message perspective. Transportation Research Part F: Traffic Psychology and Behaviour, 78, 30–41. https://doi.org/10.1016/j.trf.2021.01.013
%% Read participant response data
read_response_data=0;
if read_response_data==1
    for i=1:length(ppvalideye)
        pp=ppvalideye(i);
        if pp<10;folder(pp)={['p0' num2str(pp)]};else;folder(pp)={['p' num2str(pp)]};end %#ok<SAGROW>
        filename1=[folder{pp} '\RESULTS_RAVENS_MEAN_TIME_DATA_LOGGING.txt'];
        filename2=[folder{pp} '\RESULTS_RAVENS_QUESTIONS.txt'];
        filename3=[folder{pp} '\RESULTS_RAVENS_RESPONSE.txt'];
        filename4=[folder{pp} '\RESULTS_QUESTIONS.txt'];
        PPtime(pp)=tdfread(filename1); %#ok<SAGROW>
        PPquestions(pp)=tdfread(filename2); %#ok<SAGROW>
        PPresponse(pp)=tdfread(filename3); %#ok<SAGROW>
        PPquestions_age_gender{pp}=tdfread(filename4); %#ok<SAGROW>
    end
    save('participant_response_data.mat','PPtime','PPquestions','PPresponse','PPquestions_age_gender')
else
    load('participant_response_data.mat')
end
% Question 1 = I experienced the visual-spatial problems as difficult (0 = completely agree, 10 = completely agree)
% Question 2 = I think I performed better than the average participant in this experiment (0 = completely agree, 10 = completely agree)
% Question 3 = I experienced the visual-spatial problems as stressful (0 = completely agree, 10 = completely agree)

%% Process age and gender
Questions_age_gender=NaN(165,3);
for i=1:length(ppvalideye) % loop for Raven
    pp=ppvalideye(i);
    Questions_age_gender(pp,1)=PPquestions_age_gender{pp}.AGE(1);
    Questions_age_gender(pp,2)=1*strcmp(PPquestions_age_gender{pp}.Gender(2,1),'m')+2*strcmp(PPquestions_age_gender{pp}.Gender(2,1),'f');
    Questions_age_gender(pp,3)=1*strcmp(PPquestions_age_gender{pp}.SeeingAids(end,1:end),'No')+2*strcmp(PPquestions_age_gender{pp}.SeeingAids(end,1:end),'"Yes, contacts"')+3*strcmp(PPquestions_age_gender{pp}.SeeingAids(end,1:end),'"Yes, glasses"');
    Questions_age_gender(pp,4)=1*strcmp(PPquestions_age_gender{pp}.Refraction(end,1:end),'no')+2*strcmp(PPquestions_age_gender{pp}.Refraction(end,1:end),'yes');
    Questions_age_gender(pp,5)=str2double(PPquestions_age_gender{pp}.Refr_left(end,1:end));
    Questions_age_gender(pp,6)=str2double(PPquestions_age_gender{pp}.Refr_right(end,1:end));
end

Questions_age_gender(Questions_age_gender(:,5)>100,5)=NaN; % invalid self-reported visual correction
Questions_age_gender(Questions_age_gender(:,6)>100,6)=NaN; % invalid self-reported visual correction

%%
disp(['Mean age = ' num2str(mean(Questions_age_gender(ppvalideye,1)))])
disp(['SD age = ' num2str(std(Questions_age_gender(ppvalideye,1)))])
disp(['Number of males = ' num2str(sum(Questions_age_gender(ppvalideye,2)==1))])
disp(['Number of females = ' num2str(sum(Questions_age_gender(ppvalideye,2)==2))])
disp(['Number of participants without seeing aids = ' num2str(sum(Questions_age_gender(ppvalideye,3)==1))])
disp(['Number of participants with contact lenses = ' num2str(sum(Questions_age_gender(ppvalideye,3)==2))])
disp(['Number of participants with glasses = ' num2str(sum(Questions_age_gender(ppvalideye,3)==3))])

%% Raven processing
[Answer,Correct,AnswerTime]=deal(NaN(165,12));
Questions=NaN(165,3); % three post-experiment questions
for i=1:length(ppvalideye) % loop for Raven
    pp=ppvalideye(i);
    for item=1:12
        try
            AnswerTime(pp,item)=str2double(PPresponse(pp).RAVENS_ANSWER_ENTER_RT(item+1,:));
            if AnswerTime(pp,item)<=0
                AnswerTime(pp,item)=NaN;
            end
            Answer(pp,item)=str2double(PPresponse(pp).RAVENS_ANSWER(item+1,:)); % Answer given
            Correct(pp,item)=Answer(pp,item)==CorrectAnswers(item); % Whether this answer is correct
            if isnan(Answer(pp,item)) % if no answer was provided
                Correct(pp,item)=NaN; % then neither correct nor incorrect
            end
        catch error
        end
    end

    Questions(pp,1)=PPquestions(pp).Question1(1); % post-experiment question 1
    Questions(pp,2)=str2double(PPquestions(pp).Question2(2,:)); % post-experiment question 2
    Questions(pp,3)=str2double(PPquestions(pp).Question3(3,:)); % post-experiment question 3

end
%% Table S1 Descriptive statistics of the 12 Raven items used in the experiment
DT(isnan(Correct))=NaN;
disp('Mean and SD of Number of items correct, number of items incorrect, and number of items unanswered')
disp(round([nanmean(nansum(Correct(ppvalideye,:)==1,2)) nanstd(nansum(Correct(ppvalideye,:)==1,2))],2)) %#ok<*NANSTD> % correct
disp(round([nanmean(nansum(Correct(ppvalideye,:)==0,2)) nanstd(nansum(Correct(ppvalideye,:)==0,2))],2)) %#ok<*NANSUM,*NANMEAN> % incorrect
disp(round([nanmean(nansum(isnan(Correct(ppvalideye,:)),2)) nanstd(nansum(isnan(Correct(ppvalideye,:)),2))],2)) % no answer
disp('Min and Max of Number of items correct, Number of items incorrect, and Number of items unanswered')
disp([min(sum(Correct(ppvalideye,:)==1,2)) max(sum(Correct(ppvalideye,:)==1,2))])
disp([min(sum(Correct(ppvalideye,:)==0,2)) max(sum(Correct(ppvalideye,:)==0,2))])
disp([min(sum(isnan(Correct(ppvalideye,:)),2)) max(sum(isnan(Correct(ppvalideye,:)),2))])
disp('Table S1. Descriptive statistics of the 12 Raven items used in the experiment')
disp([nansum(Correct(ppvalideye,:))' nansum(Correct(ppvalideye,:)==0)' round(100*nansum(Correct(ppvalideye,:)==1)'./nansum(Correct(ppvalideye,:)>=0)') nansum(isnan(Correct(ppvalideye,:)))' round(nanmean(DT(ppvalideye,:))'/1000,1) round(nanstd(DT(ppvalideye,:))'/1000,1)])
disp('Mean and SD of time taken (in minutes)')
disp(round([mean(nansum(DT(ppvalideye,:),2)/1000)/60 nanstd(nansum(DT(ppvalideye,:),2)/1000)/60],2))
%% Determine AOI numbers for each fixation (add to Column 7 of FFr matrix, and create in AOI matrix for each gaze sample)
FFr(:,:,:,7)=NaN;
AOI=NaN(size(XXr));
for i=1:size(FFr,1) % loop over participants
    for i2=1:size(FFr,2) % loop over trials
        fixations_for_trial=squeeze(FFr(i,i2,:,:));

        FFr(i,i2,fixations_for_trial(:,5)>=-100,7)=-1; % no AOI

        FFr(i,i2,fixations_for_trial(:,5)>=458 &  fixations_for_trial(:,5)<=715 & fixations_for_trial(:,6)>=603 &  fixations_for_trial(:,6)<=827,7)=1; %  1 Response alternatives
        FFr(i,i2,fixations_for_trial(:,5)>=716 &  fixations_for_trial(:,5)<=972 & fixations_for_trial(:,6)>=603 &  fixations_for_trial(:,6)<=827,7)=2; %  2
        FFr(i,i2,fixations_for_trial(:,5)>=973 &  fixations_for_trial(:,5)<=1230 & fixations_for_trial(:,6)>=603 &  fixations_for_trial(:,6)<=827,7)=3; %  3
        FFr(i,i2,fixations_for_trial(:,5)>=1231 &  fixations_for_trial(:,5)<=1488 & fixations_for_trial(:,6)>=603 &  fixations_for_trial(:,6)<=827,7)=4; %  4
        FFr(i,i2,fixations_for_trial(:,5)>=458 &  fixations_for_trial(:,5)<=715 & fixations_for_trial(:,6)>=833 &  fixations_for_trial(:,6)<=1057,7)=5; % 5
        FFr(i,i2,fixations_for_trial(:,5)>=716 &  fixations_for_trial(:,5)<=972 & fixations_for_trial(:,6)>=833 &  fixations_for_trial(:,6)<=1057,7)=6; % 6
        FFr(i,i2,fixations_for_trial(:,5)>=973 &  fixations_for_trial(:,5)<=1230 & fixations_for_trial(:,6)>=833 &  fixations_for_trial(:,6)<=1057,7)=7; % 7
        FFr(i,i2,fixations_for_trial(:,5)>=1231 &  fixations_for_trial(:,5)<=1488 & fixations_for_trial(:,6)>=833 &  fixations_for_trial(:,6)<=1057,7)=8; % 8

        FFr(i,i2,fixations_for_trial(:,6)>=51 &  fixations_for_trial(:,6)<=213 & fixations_for_trial(:,5)>=599 & fixations_for_trial(:,5)<=846,7)=91; %  1 Problem
        FFr(i,i2,fixations_for_trial(:,6)>=51 &  fixations_for_trial(:,6)<=213 & fixations_for_trial(:,5)>=847 & fixations_for_trial(:,5)<=1094,7)=92; %  2
        FFr(i,i2,fixations_for_trial(:,6)>=51 &  fixations_for_trial(:,6)<=213 & fixations_for_trial(:,5)>=1095 & fixations_for_trial(:,5)<=1342,7)=93; %  3
        FFr(i,i2,fixations_for_trial(:,6)>=214 &  fixations_for_trial(:,6)<=376 & fixations_for_trial(:,5)>=599 & fixations_for_trial(:,5)<=846,7)=94; %  4
        FFr(i,i2,fixations_for_trial(:,6)>=214 &  fixations_for_trial(:,6)<=376 & fixations_for_trial(:,5)>=847 & fixations_for_trial(:,5)<=1094,7)=95; % 5
        FFr(i,i2,fixations_for_trial(:,6)>=214 &  fixations_for_trial(:,6)<=376 & fixations_for_trial(:,5)>=1095 & fixations_for_trial(:,5)<=1342,7)=96; % 6
        FFr(i,i2,fixations_for_trial(:,6)>=377 &  fixations_for_trial(:,6)<=539 & fixations_for_trial(:,5)>=599 & fixations_for_trial(:,5)<=846,7)=97; % 7
        FFr(i,i2,fixations_for_trial(:,6)>=377 &  fixations_for_trial(:,6)<=539 & fixations_for_trial(:,5)>=847 & fixations_for_trial(:,5)<=1094,7)=98; % 8
        FFr(i,i2,fixations_for_trial(:,6)>=377 &  fixations_for_trial(:,6)<=539 & fixations_for_trial(:,5)>=1095 & fixations_for_trial(:,5)<=1342,7)=99; % 9

        X_for_trial=squeeze(XXr(i,i2,:));
        Y_for_trial=squeeze(YYr(i,i2,:));

        AOI(i,i2,X_for_trial>-100)=-1; % if there is eye-tracking data, then code AOI as -1
        AOI(i,i2,X_for_trial>=458 &  X_for_trial<=715 & Y_for_trial>=603 &  Y_for_trial<=827)=1; %  1  % Response alternatives
        AOI(i,i2,X_for_trial>=716 &  X_for_trial<=972 & Y_for_trial>=603 &  Y_for_trial<=827)=2; %  2
        AOI(i,i2,X_for_trial>=973 &  X_for_trial<=1230 & Y_for_trial>=603 &  Y_for_trial<=827)=3; %  3
        AOI(i,i2,X_for_trial>=1231 &  X_for_trial<=1488 & Y_for_trial>=603 &  Y_for_trial<=827)=4; %  4
        AOI(i,i2,X_for_trial>=458 &  X_for_trial<=715 & Y_for_trial>=833 &  Y_for_trial<=1057)=5; % 5
        AOI(i,i2,X_for_trial>=716 &  X_for_trial<=972 & Y_for_trial>=833 &  Y_for_trial<=1057)=6; % 6
        AOI(i,i2,X_for_trial>=973 &  X_for_trial<=1230 & Y_for_trial>=833 &  Y_for_trial<=1057)=7; % 7
        AOI(i,i2,X_for_trial>=1231 &  X_for_trial<=1488 & Y_for_trial>=833 &  Y_for_trial<=1057)=8; % 8

        AOI(i,i2,Y_for_trial>=51 &  Y_for_trial<=213 & X_for_trial>=599 & X_for_trial<=846)=91; %  1 % Problem matrix
        AOI(i,i2,Y_for_trial>=51 &  Y_for_trial<=213 & X_for_trial>=847 & X_for_trial<=1094)=92; %  2
        AOI(i,i2,Y_for_trial>=51 &  Y_for_trial<=213 & X_for_trial>=1095 & X_for_trial<=1342)=93; %  3
        AOI(i,i2,Y_for_trial>=214 &  Y_for_trial<=376 & X_for_trial>=599 & X_for_trial<=846)=94; %  4
        AOI(i,i2,Y_for_trial>=214 &  Y_for_trial<=376 & X_for_trial>=847 & X_for_trial<=1094)=95; % 5
        AOI(i,i2,Y_for_trial>=214 &  Y_for_trial<=376 & X_for_trial>=1095 & X_for_trial<=1342)=96; % 6
        AOI(i,i2,Y_for_trial>=377 &  Y_for_trial<=539 & X_for_trial>=599 & X_for_trial<=846)=97; % 7
        AOI(i,i2,Y_for_trial>=377 &  Y_for_trial<=539 & X_for_trial>=847 & X_for_trial<=1094)=98; % 8
        AOI(i,i2,Y_for_trial>=377 &  Y_for_trial<=539 & X_for_trial>=1095 & X_for_trial<=1342)=99; % 9
    end
end

%%
DT(isnan(Correct))=NaN; % set response time to 0 is there is no response
Questions(Questions(:,1)>10,1)=NaN; % set strange response to questions to NaN
for i=1:size(Correct,1)
    for i2=1:size(Correct,2) % if no data on Correctness, then do not use eye-tracking data
        if isnan(Correct(i,i2)) % if there is no response, set data to NaN
            AOI(i,i2,:)=NaN;
            FFr(i,i2,:,:)=NaN;
            DT(i,i2,:)=NaN;
        end
    end
end
%% Descriptive statistics and intercorrelations between measures (n = 159)
Xcorr=[nansum(Correct==1,2) ... %  number of correct answers (1)
    nansum(Correct==0,2) ... % number of incorrect answers (2)
    nansum(Correct==1,2)./nansum(Correct>=0,2) ... % proportion of answers that are correct (3)
    nanmean(DT,2)/1000 ... % answering time (4)
    nanmean(100*nansum(AOI>10,3)./nansum(AOI>=-10,3),2) ... %  percentage of gaze time on problem  (5)
    nanmean(100*nansum(AOI<10 & AOI>0,3)./nansum(AOI>=-10,3),2)]; % percentage of gaze time on response alternatives (6)

Xcorr(~ismember(1:165,ppvalideye),:)=NaN;

disp('Table S2. Descriptive statistics and intercorrelations between measures (n = 159). M, SD')
disp([round(nanmean(Xcorr(ppvalideye,[1 3:end]))',2) round(nanstd(Xcorr(ppvalideye,[1 3:end]))',2)])

[c,p]=corr(Xcorr(ppvalideye,:),'rows','pairwise');
disp('Table S2. Correlation matrix')
disp(round(c([1 3:end],[1 3:end]),2));
disp('Statistical significance (0 or 1)')
disp(p([1 3:end],[1 3:end])<0.05);

%% Figure 3. Heatmap of all collected eye-gaze data. The heatmap was created by dividing the screen into 5 x 5-pixel squares and counting the number of samples (after downsampling to 100 Hz) the squares were gazed at. All counts were then divided by the number of participants (159).
create_heatmaps_raven(XXr(ppvalideye,:,:),YYr(ppvalideye,:,:));
%%
h=NaN(165,12,18);
for p=ppvalideye % loop over participants
    for t=1:12 % loop over items 1 to 12
        a=squeeze(FFr(p,t,:,7));a(isnan(a))=[];
        if ~isnan(Correct(p,t))
            h(p,t,:)=histc(a,[-1 1:8 91:99]); %#ok<HISTC> % count fixations on all areas of interest for participant p and item (trial) t
        end
    end
end

%% Table S4. Participants. mean number of fixations per matrix cell and total number of fixations per item. The bottom row lists the mean of the 12 items.
disp('Table S4. Participants’ mean number of fixations per matrix cell and total number of fixations per Raven item. The bottom row lists the mean of the 12 Raven items.')
M=squeeze(nanmean(h,1)); % compute mean of participants
disp(M(:,10:18)) % display 12 items x 9 AOI matrix with means of participants

%% Table S5. Participants’ mean number of fixations per response alternative. Also shown is the percentage of fixations on the response alternatives that fell on the correct response.
disp('Table S5. Table S5. Participants’ mean number of fixations per response alternative. Also shown is the percentage of fixations on the response alternatives that fell on the correct response.')
M=squeeze(nanmean(h,1)); % compute mean of participants
disp(M(:,2:9)) % display 12 items x 8 AOI matrix with means of participants

%% Read Qualtrics responses and compute complexity score based on the pairwise comparisons
read_qualtrics_responses % read survey responses and compute complexity score (96 x 1 vector) using the PW scale function (Pérez-Ortiz, M., & Mantiuk, R. K. (2017). A practical guide and software for analysing pairwise comparison experiments. arXiv. https://arxiv.org/abs/1712.03686)
%% % Figure 1. Example item of Raven matrices. The areas of interest (AOIs) are depicted as green outlines (not visible during the experiment).
draw_aois % also compute edge density values for all cells, and store in VC matrix
%%
disp('Correlation between edge density and perceived complexity (n = 96)')
disp(round(corr(complexity,reshape(VC',96,1)),2)) % complexity is a  96x1 vector of perceived complexity scores, VC is a 12 x 9 matrix of edge density scores
% "The two measures of complexity were strongly associated (r = 0.72, n = 96)"

VC2=reshape(complexity,8,12)'; % create 12 x 8 matrix of perceived complexity scores

for NI=[9 8] % number of items. If 9, it produces Tables 2 and 3; if 8, it produces Tables 4 and 5

    %% Columns 1-3 belong to Table 3: Results of 12 regression analyses for predicting the number of fixations on matrix cells from edge density and centrality of the matrix cells, not using the right bottom matrix cell (n = 8).
    %% Columns 4-6 belong to Table 4: Results of 12 regression analyses for predicting the number of fixations on matrix cells from perceived complexity and centrality of the matrix cells, not using the right bottom matrix cell (n = 8).
    w=[3 5 3 5 8 5 3 5 3]'; % Cell centrality was defined using numeric weights, where the four edge cells (top left, top right, bottom left, bottom right) received a weight of 3, the middle edge cells (top middle, middle left, middle right, bottom middle) received a weight of 5, and the central cell received a weight of 8

    if NI==9
        VC(:,9)=VC(9,1); % empty cell
        VC2(:,9)=VC2(9,1); % empty cell
    elseif NI==8
        if size(VC,2)==9
            VC(:,9)=[];
        end
        if size(VC2,2)==9
            VC2(:,9)=[];
        end
    end

    cc=NaN(12,12);

    for i=1:12 % loop over 12 items

        x=[VC(i,1:NI)' w(1:NI)];
        y=M(i,10:10+NI-1)';

        x=zscore(x);
        y=zscore(y);

        st=regstats(y, x);

        cc(i,1)=st.beta(2);
        cc(i,2)=st.beta(3);
        [cc(i,3),cc(i,6)]=corr(st.yhat,y);
        cc(i,4:5)=st.tstat.pval(2:3);
        x=[VC2(i,1:NI)' w(1:NI)];
        y=M(i,10:10+NI-1)';

        x=zscore(x);
        y=zscore(y);

        st=regstats(y, x);

        cc(i,7)=st.beta(2);
        cc(i,8)=st.beta(3);
        [cc(i,9),cc(i,12)]=corr(st.yhat,y);
        cc(i,10:11)=st.tstat.pval(2:3);
    end
    if NI==9
        disp('Table 2 (columns 1-3) and Table 3 (colums 4-6)')
    elseif NI==8
        disp('Table 4 (columns 1-3) and Table 5 (colums 4-6)')
    end
    disp(round(cc(:,[1:3 7:9]),2))
    disp('Means of all 12 items:')
    disp(round(mean(cc(:,[1:3 7:9])),2))

    disp('Significance (0 or 1):')
    disp(cc(:,[4:6 10:12])<0.05)
    %% Figure 5. Mean number of fixations per Raven item versus edge density per Raven item (left) and versus perceived complexity per Raven item (right).
    if NI==9
        v1=mean(VC,2); % Edge density of matrix of item
        v2=mean(VC2,2); % Perceived complexity of matrix of item
        v3=sum(M(:,10:10+NI-1),2); % % Mean number of fixations of item (sum of NI items)

        figure;
        colors=colormap("jet");
        colors=colors(round(linspace(1,256,length(v1)))',:);
        subplot(1,2,1)
        hold on;grid on;box on
        clear scatter1
        for i=1:12
            scatter1(i)=scatter(v1(i),v3(i),800,'markerfacecolor',colors(i,:),'markeredgecolor',colors(i,:));hold on;grid on %#ok<SAGROW>
            scatter1(i).MarkerFaceAlpha = .3; %#ok<SAGROW>
            text(v1(i),v3(i),num2str(i),'horizontalalignment','center')
        end

        set(gca,'pos',[0.055 0.082 0.42 0.90])
        xlabel('Edge density of matrix of Raven item')
        ylabel('Mean number of fixations of Raven item')
        h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')

        subplot(1,2,2)
        hold on;grid on;box on
        clear scatter1
        for i=1:12
            scatter1(i)=scatter(v2(i),v3(i),700,'markerfacecolor',colors(i,:),'markeredgecolor',colors(i,:));hold on;grid on %#ok<SAGROW>
            scatter1(i).MarkerFaceAlpha = .3; %#ok<SAGROW>
            text(v2(i),v3(i),num2str(i),'horizontalalignment','center')
        end
        set(gca,'pos',[0.565 0.082 0.42 0.90])
        xlabel('Perceived complexity of matrix of Raven item')
        ylabel('Mean number of fixations of Raven item')
        h=findobj('FontName','Helvetica');set(h,'FontSize',24,'Fontname','Arial')
    end
    %% Figure 4. Matrix cells (96 in total, i.e., 12 items × 8 cells per item) sorted by edge density. Below each image, the edge density (in green) and perceived complexity (in yellow) are shown. Visual complexity ranks are shown in parenthesis, where 1 refers to the lowest value and 96 refers to the highest value.
    if NI==8

        VCv=[VC(1,1:NI)';VC(2,1:NI)';VC(3,1:NI)';VC(4,1:NI)';VC(5,1:NI)';VC(6,1:NI)';VC(7,1:NI)';VC(8,1:NI)';VC(9,1:NI)';VC(10,1:NI)';VC(11,1:NI)';VC(12,1:NI)']; % edge density vector (96 x 1)
        VC2v=[VC2(1,1:NI)';VC2(2,1:NI)';VC2(3,1:NI)';VC2(4,1:NI)';VC2(5,1:NI)';VC2(6,1:NI)';VC2(7,1:NI)';VC2(8,1:NI)';VC2(9,1:NI)';VC2(10,1:NI)';VC2(11,1:NI)';VC2(12,1:NI)']; % perceived complexity vector (96 x 1)
        iv=reshape(repmat(1:12,NI,1),1,NI*12)'; % item number vector (1 to 12) (96 x 1)
        cn=repmat(1:NI,1,12)'; % cell number vector (1 to 9) (96 x 1)
        VC2vr=tiedrank(VC2v); % determine the rank-order of the perceived complexity

        [VCvs,VCvso]=sort(VCv); % edge density vector, sorted
        figure
        t=tiledlayout(8,12);
        for i=1:length(VCv) % loop over 96 cells
            nexttile
            imshow(squeeze(IM(iv(VCvso(i)),cn(VCvso(i)),:,:,:))/255)
            text(120,165,[sprintf('%0.4f',VCvs(i)+10^-9) ' (' num2str(i) ')'],'color','g','horizontalalignment','center') % add edge density in numeric form
            text(120,200,[sprintf('%0.3f',complexity(VCvso(i))+10^-9) ' (' num2str(VC2vr(VCvso(i),1)) ')'],'color','y','horizontalalignment','center') % add corresponding perceived complexity, in yellow
        end
        t.TileSpacing = 'tight';
        t.Padding = 'tight';
        h=findobj('FontName','Helvetica');set(h,'FontSize',14,'Fontname','Arial')
    end
    set(gcf,'color',[.5 .5 .5])
    %%
    if NI==8
        disp('SD of edge density (Table 4) and SD of perceived complexity (Table 5)')
        disp([round(std(VC,[],2),4) round(std(VC2,[],2),3)])
    end

    if NI==9
        %% Table 1. Correlation coefficients between complexity measures and difficulty measures and item response time measures among Raven items (n = 12)
        disp('Table 1. Correlation coefficients between complexity measures and difficulty measures and item response time measures among Raven items (n = 12)')
        Xi1=[nanmean(VC,2) ... % edge density
            nanmean(VC2,2)];   % perceived complexity

        Xi2=[sum(M(:,10:18),2) ...                % number of fixations on matrix area
            nanmean(Correct(ppvalideye,:),1)' ... % percentage of correct responses (%)
            (1:12)' ...                           % Raven item number (coded as 1 to 12)
            nanmean(DT(ppvalideye,:),1)'/1000];   % mean response time (s)

        [c,p]=corr(Xi1,Xi2);
        disp('Table 1. Correlation coefficients:')
        disp(round(c,2))
        disp('Table 1. p-values:')
        disp(round(p,3))
        %%
        disp('"We also computed the partial correlation coefficient between edge density and mean response time, while controlling for Raven item number (coded as 1 to 12). The partial correlation was found to be 0.82 (p = 0.002)."')
        [c,p]=partialcorr(Xi1(:,1),Xi2(:,4),Xi2(:,3));
        disp([round(c,2) round(p,3)])

        disp('"Similarly, the partial correlation coefficient between perceived complexity and mean response time, while controlling for Raven item number was 0.53 (p = 0.090)."')
        [c,p]=partialcorr(Xi1(:,2),Xi2(:,4),Xi2(:,3));
        disp([round(c,2) round(p,3)])
    end
    if NI==9
        %% %% Figure S2. Edge density for the Raven item means of the redrawn cells used in the analysis versus edge density from screenshots of the scans from the RAPM booklet (Raven, 1976).
        v1=mean(VC,2); % edge density computed from redrawn cells, averaged across the cells
        v2=VCall(II); % edge density computed from scan of booklet

        figure;
        colors=colormap("jet");
        colors=colors(round(linspace(1,256,length(v1)))',:);
        subplot(1,2,1)
        hold on;grid on;box on
        clear scatter1
        for i=1:12
            scatter1(i)=scatter(v1(i),v2(i),800,'markerfacecolor',colors(i,:),'markeredgecolor',colors(i,:));hold on;grid on %#ok<SAGROW>
            scatter1(i).MarkerFaceAlpha = .3; %#ok<SAGROW>
            text(v1(i),v2(i),num2str(i),'horizontalalignment','center')
        end
        set(gca,'pos',[0.08 0.09 0.42 0.89])
        xlabel('Edge density (redrawn)')
        ylabel('Edge density (scan)')
        h=findobj('FontName','Helvetica');set(h,'FontSize',26,'Fontname','Arial')
    end
end % end of NI loop

%% Figure S1. Edge density for all 36 items of the Raven Advanced Progressive Matrices.
opengl hardware
figure
plot(VCall,'k-o','Linewidth',3,'markersize',20)
hold on
grid on
plot(II,VCall(II),'ro','markerfacecolor','r','Linewidth',3,'markersize',20)
legend('Not included in current study','Included in current study','location','northwest')
set(gca,'xlim',[0.5 36.5])
xlabel('Item number')
ylabel('Edge density of item')
h=findobj('FontName','Helvetica');set(h,'FontSize',26,'Fontname','Arial')
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
%% Figure S3. Mean response times for items of Raven's Advanced Progressive Matrices for the current study and for lab experiments from the literature. It can be seen that more difficult items (i.e., higher item number) generally took more time for participants to solve. It can also be seen that mean response times differed substantially between studies.
figure;
colors=[0 0 1 % 16 different colors
 1 0 0
 0 1 0
 0 0 0.172
 1 0.103 0.724
 1 0.828 0
 0 0.345 0
 0.517 0.517 1
 0.621 0.31 0.276
 0 1 0.759
 0 0.517 0.586
 0 0 0.483
 0.586 0.828 0.31
 0.966 0.621 0.862
 0.828 0.069 1
 0.483 0.103 0.414]; 
for i=1:size(colors,1)
    plot(AM(:,i),'o','markeredgecolor',colors(i,:),'markerfacecolor',colors(i,:),'markersize',20);hold on
end
set(gca,'xlim',[0.5 36.5])
grid on
xlabel('Item number')
ylabel('Mean response time (s)')

legend('1. Current study',...
         '2. Adam & Vogel (2018)', ...
         '3. Ellis et al. (2021)', ...
         '4. Frischkorn & Von Bastian (2021)', ...
         '5. Sense et al. (2019)', ...
         '6. Goldhammer et al. (2015)', ...
         '7. Gonthier & Roulin (2020)', ...
         '8. Liu et al. (2022)', ...
         '9. Loesche et al. (2015), Study 1–4', ...
         '10. Poulton et al. (2022), Study 1', ...
         '11. Poulton et al. (2022), Study 2', ...
         '12. Rivollier et al. (2021), Study 1', ...
         '13. Robison & Campbell (2023)',...
         '14. Tsukahara & Engle (2021), Study 1', ...
         '15. Tsukahara & Engle (2023)', ...
         '16. Vigneau et al. (2006)', ...
         'location','eastoutside')

h=findobj('FontName','Helvetica');set(h,'FontSize',26,'Fontname','Arial')
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
%% Compute and store results for Table S3
cc=NaN(11,6);
for i=1:size(AM,2) % loop over studies
    x=[ (1:36)' VCall(:,1)]; % item number, edge density
    y=AM(:,i); % mean response times of study i
    x(isnan(y),:)=[]; % remove NaNs before conducting regression analysis
    y(isnan(y),:)=[]; % remove NaNs before conducting regression analysis
    [cc(i,1:2),cc(i,4:5)]=corr(x,y,'rows','pairwise'); % correlation between item number and edge density vs. mean response time
    [cc(i,3),cc(i,6)]=partialcorr(x(:,2),y,x(:,1)); % partial correlation
end
% cc(:,1:3) contain correlation coefficients
% cc(:,4:6) contain p-values

%% Table S3. Correlation coefficients between Raven item number and mean response time, correlation coefficient between edge density and mean response time, and partial correlation between edge density and mean response time while controlling for item number.
disp('Table S3. Correlation coefficients between Raven item number and mean response time, correlation coefficient between edge density and mean response time, and partial correlation between edge density and mean response time while controlling for item number.')
disp([ (1:size(cc,1))' round(cc(:,1:3),2)] )
disp('Significance for Table S3')
disp([ (1:size(cc,1))' cc(:,4:6)<0.05])

disp('Correlation between item number and scanned item edge density:')
disp(round(corr((1:36)',VCall,'rows','pairwise'),2))
disp('Corresponding p-value:')
[c,p]=corr((1:36)',VCall,'rows','pairwise');
disp(round(p,3))
disp('Correlation between edge density from scans (VCall) vs. redrawn (VC) (Figure S2):')
disp(corr(mean(VC,2,'omitnan'),VCall(II)))