__includes ["plots.nls" "goverment_policies.nls" "breeds.nls" "init.nls" "households.nls" "houses.nls" "utility.nls" "tests.nls"]

globals [
  houses                         ;; Group of all houses
  households                     ;; Group of all active households (this should not include households that are simulating a relationship being formed)

  supply_area                    ;; Patches for unowned houses
  household_area                 ;; Patches for households

  new_houses_progress            ;; The progress towards the next house being build.
  migration_progress             ;; The progress towards the next household being forced to release their home

  death_chance                   ;; A list holding the monthly chance of a person dying seperated by their age and the composition of their household
  child_chance                   ;; A list holding the monthly chance of a child being born in a household seperated by the age and the composition of a household
  cohabitant_chance              ;; A list holding the monthly chance of a household getting into a relationship, seperated by age
  marriage_split_chance          ;; A list holding the monthly chance of a married pair divorcing seperated by age
  cohabitant_split_chance        ;; A list holding the monthly chance of a unmarried pair divorcing seperated by age
  age_diff_cohabitants           ;; A list holding the cumulative probabibilty of a certaing age difference when a relationship forms

  waiting_time_at_transfer_year
  transfers_year
  waiting_time_at_transfer_all
  transfers_all

  run-seed
  months
]

to setup
  ;; Reset everything
  clear-all

  set months 0

  set run-seed new-seed
  if seed != 0 [
    set run-seed seed
  ]
  random-seed run-seed

  ;; Set default shapes
  set-default-shape home_ownership_households "person"
  set-default-shape private_rental_households "person"
  set-default-shape social_rental_households "person"
  set-default-shape searching_households "person"
  set-default-shape pr_houses "house"
  set-default-shape s_houses "house"
  set-default-shape ho_houses "house"


  ;; Close any file that might be accidantally still opened
  file-close

  ;; Load data on death
  file-open "data/MonthlyDeaths_dist.data"
  set death_chance file-read
  file-close

  ;; Load data on child-birth
  file-open "data/MonthlyBirths_dist.data"
  set child_chance file-read
  file-close

  ;; Load data on relationship forming and breaking
  file-open "data/MarriageBreak.data"
  set marriage_split_chance file-read
  file-close
  file-open "data/CohabitantBreak.data"
  set cohabitant_split_chance file-read
  file-close
  file-open "data/RelationForm.data"
  set cohabitant_chance file-read
  file-close

  ;; Load data on the age difference in relationships that form
  file-open "data/RelationAgediff.data"
  set age_diff_cohabitants file-read
  file-close

  ;; Declare seperate areas for houses and households
  set supply_area patches with [pycor > world-height / 2 and pxcor > 1]
  set household_area patches with [pycor <= world-height / 2 and pxcor > 1]

  ;; Reserve space to place relationships, as they do not require their tile to function properly
  ask patches with [pxcor <= 0][set pcolor grey]

  ;; Initialize Houses
  generate_houses housing_supply

  ;; Initialize Households
  generate_households

  ;; Match Houses and Households
  assign_houses

  ;; Set progress for new houses and migration simulation
  set new_houses_progress 0
  set migration_progress 0

  ;; Set Measurement stats
  set transfers_year 0
  set waiting_time_at_transfer_year 0
  set transfers_all 0
  set waiting_time_at_transfer_all 0

  ;; Reset ticks
  reset-ticks

  ;; Initialize plots
  generate_plots
end

to experiment-go
  repeat 12[
    go
  ]
end

to go
  if months mod 12 = 0 [
    set transfers_year 0
    set waiting_time_at_transfer_year 0
  ]

  ;; Add new houses
  set new_houses_progress (ifelse-value
    (new_houses_type = "flat") [new_houses_progress + new_houses_monthly]
    (new_houses_type = "percentage of households") [new_houses_progress + new_houses_monthly * count households * 0.01]
    (new_houses_type = "percentage of houses") [new_houses_progress + new_houses_monthly * count houses * 0.01]
    (new_houses_type = "percentage of shortgage") [new_houses_progress + new_houses_monthly * ( count households - count houses) * 0.01]
  )

  set migration_progress (ifelse-value
    (migration_type = "flat") [migration_progress + migration]
    (migration_type = "percentage of households") [migration_progress + migration * count households * 0.01]
    (migration_type = "percentage of houses") [migration_progress + migration * count houses * 0.01]
  )

  if migration_progress > 1 [
    let selection up-to-n-of floor migration_progress households with [current_market != "Parents" and current_market != "None"]
    ask selection
    [
      empty_house_of_turtle
      set breed searching_households
      set current_market "None"
      set searching_reason "migration"
    ]
    set migration_progress migration_progress - count selection
  ]

  if new_houses_progress > 1 [
    generate_new_houses floor new_houses_progress
    set new_houses_progress new_houses_progress - floor new_houses_progress
  ]

  ;; Handle housing with a contract that ends
  ;  social_expire_contracts
  ;  free_expire_contracts

  ;; Disable display to avoid links flashing on screen and to increase performance
  no-display

  ;; Have households evaluate wether they want to look for a new home.
  ask households with [breed != searching_households] [
    let current_house one-of other turtles-here

    ;; If the household lives in a too small house start searching
    if household_size > 2 * ([house_size] of current_house - 1) and [house_size] of current_house != 5[
      set breed searching_households
      set searching_reason "Small_House"
    ]
    ;; If the household lives in a too expensive house start searching
    if income% > 50 [
      set breed searching_households
      set searching_reason "High_Rent"
    ]

    ;; If the household lives in social housing but started earning above the threshold
    if social_leave = true and income > maximum_income_social_rental_market and current_market = "social_rental"[
      set breed searching_households
      set searching_reason "Social_Leave"
    ]

    if old_large_house_leavers = true and age > 65 and length adults <= 2 and [house_size] of current_house > 2 [
      set breed searching_households
      set searching_reason "Large_House_And_Old"
    ]
  ]

  ask searching_households [
    ifelse searching_reason = "Small_House" [
      let current_house one-of other turtles-here
      if not (household_size > 2 * ([house_size] of current_house - 1)) [
        unsearch_household
      ]
    ][
      ifelse searching_reason = "High_Rent" and not (income% > 50) [
        unsearch_household
      ][
        if searching_reason = "Social_Leave" and not (social_leave = true and income > maximum_income_social_rental_market and current_market = "social_rental") [
          unsearch_household
        ]
      ]
    ]
  ]

  ;; Advertise houses
  ask houses with [status = "empty" or status = "for_sale" or status = "preparing_sale"][
    advertise
  ]

  ;; React to advertisments
  react_to_advertisements

  ;; Re-enable rendering
  display

  ;; Select Winners
  ask ho_houses with [status = "empty" or status = "for_sale"][
    select_winner
  ]
  ask s_houses with [status = "empty" or status = "for_sale"][
    select_winner
  ]
  foreach sort-on [rent] (pr_houses with [status = "empty" or status = "for_sale"])[
    the-turtle -> ask the-turtle [select_winner]
  ]
  ;; Update relationships
  split_relationships
  check
  start_relationships
  check

  if months mod 12 = 0[
  ;; Income Shocks
    let shock_agents_size count households with [household_type != "forming_relationship"] * 0.01 * shocked% / 2
    let shock_up n-of shock_agents_size households with [household_type != "forming_relationship"]
    let shock_down n-of shock_agents_size households with [household_type != "forming_relationship" and not member? self shock_up]
    ask shock_up [
      set income income * (1 + 0.01 * shock_size)
    ]
    ask shock_down [
      set income income * (1 - 0.01 * shock_size)
    ]
  ]


  ;; Updates at the end of the month
  monthly_updates
  check

  set months months + 1
  if months mod 12 = 0[
    generate_plots
    tick
  ]
end

to unsearch_household
  set breed (ifelse-value
    (current_market = "social_rental")[social_rental_households]
    (current_market = "home_ownership")[home_ownership_households]
    (current_market = "private_rental" )[private_rental_households]
  )
end
@#$#@#$#@
GRAPHICS-WINDOW
97
14
5115
333
-1
-1
10.0
1
10
1
1
1
0
0
0
1
0
500
0
30
0
0
1
Years
30.0

BUTTON
4
10
67
43
setup
setup
NIL
1
T
OBSERVER
NIL
S
NIL
NIL
1

BUTTON
3
46
66
79
go
go
T
1
T
OBSERVER
NIL
G
NIL
NIL
0

BUTTON
3
83
80
116
1 month
go
NIL
1
T
OBSERVER
NIL
M
NIL
NIL
0

BUTTON
2
121
69
154
1 year
repeat 12[\n go\n]
NIL
1
T
OBSERVER
NIL
Y
NIL
NIL
0

SLIDER
365
696
593
729
population_size
population_size
0
1000
1000.0
10
1
NIL
HORIZONTAL

SLIDER
365
760
593
793
free_housing_supply%
free_housing_supply%
0
100
18.0
0.01
1
NIL
HORIZONTAL

SLIDER
365
792
593
825
social_housing_supply%
social_housing_supply%
0
100
51.0
0.01
1
NIL
HORIZONTAL

SLIDER
365
824
593
857
homeownership_housing_supply%
homeownership_housing_supply%
0
100
31.0
0.01
1
NIL
HORIZONTAL

SLIDER
0
341
293
374
maximum_simultaneous_reactions_social_market
maximum_simultaneous_reactions_social_market
0
50
4.0
1
1
NIL
HORIZONTAL

PLOT
1904
25
2806
449
income_distributions
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"All" 1.0 0 -16777216 true "" ""
"Social Rental" 1.0 1 -13840069 true "" ""
"Private Rental" 1.0 0 -2674135 true "" ""
"Home-Ownership" 1.0 0 -13791810 true "" ""

PLOT
1196
10
1856
160
House Counter
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"All" 1.0 0 -16777216 true "" ""
"Social Rental Houses" 1.0 0 -13840069 true "" ""
"Private Rental Houses" 1.0 0 -2674135 true "" ""
"Home Ownership Houses" 1.0 0 -13345367 true "" ""

PLOT
1196
160
1856
328
Household Counter
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"All" 1.0 0 -16777216 true "" ""
"Social Renting Households" 1.0 0 -13840069 true "" "\n"
"Private Renting Households" 1.0 0 -2674135 true "" ""
"Home Owning Households" 1.0 0 -13345367 true "" ""
"Rest" 1.0 0 -2064490 true "" ""
"Households on Waiting List" 1.0 0 -7500403 true "" ""
"Households in social market with high income" 1.0 0 -955883 true "" ""
"Households Searching" 1.0 0 -8630108 true "" ""

PLOT
1196
331
1856
481
length_waiting_list
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Longest Waiting" 1.0 0 -2674135 true "" ""
"Avg Waiting" 1.0 0 -955883 true "" ""
"Longest Waiting income < maximum_income" 1.0 0 -8053223 true "" ""
"Avg Waiting income < maximum_income" 1.0 0 -6995700 true "" ""
"Avg Waiting All" 1.0 0 -8275240 true "" ""
"Max Waiting Lim" 1.0 2 -5825686 true "" ""

SLIDER
0
438
280
471
maximum_income_social_rental_market
maximum_income_social_rental_market
0
50000
40024.0
1
1
NIL
HORIZONTAL

PLOT
804
180
1195
330
age_distributions
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"All" 1.0 0 -16777216 true "" ""
"Social Rental" 1.0 0 -13840069 true "" ""
"Private Rental" 1.0 0 -2674135 true "" ""
"Home-Ownership" 1.0 0 -13791810 true "" ""

PLOT
804
331
1195
481
monthly_costs
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Social Rental" 1.0 0 -13840069 true "" ""
"Private Rental" 1.0 0 -2674135 true "" ""
"Mortgage Payments" 1.0 0 -13791810 true "" ""

SLIDER
0
373
293
406
maximum_simultaneous_reactions_lottery_social_market
maximum_simultaneous_reactions_lottery_social_market
0
50
4.0
1
1
NIL
HORIZONTAL

SLIDER
0
405
293
438
supply_lottery
supply_lottery
0
100
10.0
1
1
%
HORIZONTAL

SLIDER
0
471
280
504
maximum_income_supply_middle_incomes
maximum_income_supply_middle_incomes
0
50000
44655.0
1
1
NIL
HORIZONTAL

PLOT
1196
482
1855
604
Average Rents
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Private Rent" 1.0 0 -2674135 true "" ""

TEXTBOX
1921
724
2508
780
In; Wachtlijsten sociale huursector\nIncome is checked on the moment a house is given out
11
0.0
1

PLOT
1196
604
1854
724
Average House Price
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Home Ownership" 1.0 0 -13345367 true "" ""

CHOOSER
191
504
387
549
mortgage_type
mortgage_type
"linear" "annuity"
1

SLIDER
0
505
181
538
mortgage_length
mortgage_length
0
360
360.0
12
1
months
HORIZONTAL

SLIDER
192
551
371
584
mortgage_interest
mortgage_interest
0
0.05
0.021
0.0001
1
NIL
HORIZONTAL

PLOT
804
483
1195
633
latest_sale_prices_distributions
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Home-Ownership" 1.0 0 -13791810 true "" ""

TEXTBOX
233
775
383
915
Aantal woningen amsterdam: 447.210\n15,36% + 3,1% vrije sector\n30,85% koop\n13,11% particulier gereguleerd\n37,58% Sociale huur\n\nhttps://www.nul20.nl/dashboard-amsterdam
11
0.0
1

TEXTBOX
1918
519
2455
701
https://www.amsterdam.nl/veelgevraagd/?productid=%7b5362EE4B-6834-4D1A-B8F8-D4622EF25437%7d#case_%7BE4F1B211-3407-4BC0-9662-308096A48DFE%7D\n\nto 1.041,76 max 63.237 (But not if you leave social housing)\nPriority families\nPriority those that leave social housing\nto 911,95 priority until 51.997
11
0.0
1

PLOT
804
10
1195
179
montly_costs_as_%_of_income
% of Income
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Social Rental" 1.0 0 -13840069 true "" ""
"Private Rental" 1.0 0 -2674135 true "" ""
"Mortgage Payments" 1.0 0 -13791810 true "" ""

TEXTBOX
1918
635
2592
733
https://opendata.cbs.nl/statline/#/CBS/nl/dataset/84487NED/table?ts=1615804736704\n\nPrivate: 13.6 15.4 \nSocial: 16.3 cheap 10.2 expensive
11
0.0
1

SLIDER
365
728
593
761
housing_supply
housing_supply
0
1000
951.0
1
1
NIL
HORIZONTAL

SLIDER
594
764
766
797
Size1/2%
Size1/2%
0
100
31.29
0.01
1
NIL
HORIZONTAL

SLIDER
594
813
766
846
Size3%
Size3%
0
100
34.35
0.01
1
NIL
HORIZONTAL

SLIDER
594
847
766
880
Size4%
Size4%
0
100
23.69
0.01
1
NIL
HORIZONTAL

SLIDER
594
881
766
914
Size5%
Size5%
0
100
10.67
0.01
1
NIL
HORIZONTAL

SLIDER
1264
739
1515
772
social_rent_432.51%
social_rent_432.51%
0
100
21.22
.01
1
NIL
HORIZONTAL

SLIDER
1264
774
1515
807
social_rent_432.51_to_619.01%
social_rent_432.51_to_619.01%
0
100
53.37
.01
1
NIL
HORIZONTAL

SLIDER
1264
809
1515
842
social_rent_619.01_to_663.40%
social_rent_619.01_to_663.40%
0
100
11.16
0.01
1
NIL
HORIZONTAL

SLIDER
1264
844
1515
877
social_rent_663.40_to_737.14%
social_rent_663.40_to_737.14%
0
100
14.25
.01
1
NIL
HORIZONTAL

SLIDER
1556
739
1794
772
house_worth_to_221000
house_worth_to_221000
0
100
12.21
0.01
1
NIL
HORIZONTAL

SLIDER
1556
772
1794
805
house_worth_221000_to_310000
house_worth_221000_to_310000
0
100
22.18
0.01
1
NIL
HORIZONTAL

SLIDER
1557
806
1795
839
house_worth_310000_to_385000
house_worth_310000_to_385000
0
100
22.48
1
1
NIL
HORIZONTAL

SLIDER
1557
840
1795
873
house_worth_385000_to_511000
house_worth_385000_to_511000
0
100
22.48
0.01
1
NIL
HORIZONTAL

SLIDER
1557
874
1795
907
house_worth_511000_plus
house_worth_511000_plus
0
100
20.65
0.01
1
NIL
HORIZONTAL

BUTTON
2
161
96
194
90 years
repeat 1080[\n go\n]
NIL
1
T
OBSERVER
NIL
X
NIL
NIL
0

TEXTBOX
1268
725
1418
743
21.22 53.37 11.16 14.25
11
0.0
1

TEXTBOX
1828
857
1978
885
12.21 22.18 22.48 22.48 20.65
11
0.0
1

SLIDER
0
717
227
750
new_houses_monthly
new_houses_monthly
0
100
0.84
0.01
1
NIL
HORIZONTAL

SLIDER
0
658
251
691
%_of_new_cohabitants_that_marry
%_of_new_cohabitants_that_marry
0
1
0.15
0.01
1
NIL
HORIZONTAL

SLIDER
0
797
227
830
migration
migration
0
500
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
324
341
496
374
shocked%
shocked%
0
100
20.0
0.01
1
%
HORIZONTAL

SLIDER
324
377
496
410
shock_size
shock_size
0
20
20.0
1
1
%
HORIZONTAL

CHOOSER
589
370
729
415
spouse_waiting_time
spouse_waiting_time
"always" "divorce" "never"
0

CHOOSER
0
752
193
797
new_houses_type
new_houses_type
"flat" "percentage of households" "percentage of houses" "percentage of shortgage"
0

CHOOSER
591
416
730
461
building_waiting_time
building_waiting_time
"while_searching" "always"
1

PLOT
805
633
1169
911
motivations
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"high rent" 1.0 0 -955883 true "" "plot count searching_households with [searching_reason = \"High_Rent\" and any? my-out-relationships = false]"
"moving out" 1.0 0 -13840069 true "" "plot count searching_households with [searching_reason = \"Moving_Out\" and any? my-out-relationships = false]"
"small house" 1.0 0 -1184463 true "" "plot count searching_households with [searching_reason = \"Small_House\" and any? my-out-relationships = false]"
"split " 1.0 0 -5825686 true "" "plot count searching_households with [searching_reason = \"Split\" and any? my-out-relationships = false]"
"relationship" 1.0 0 -13345367 true "" "plot count searching_households with [searching_reason = \"Relationship\" and any? my-out-relationships = false]"
"\"Social_Leave\"" 1.0 0 -7500403 true "" "plot count searching_households with [searching_reason = \"Social_Leave\" and any? my-out-relationships = false]"
"Migration" 1.0 0 -6459832 true "" "plot count searching_households with [searching_reason = \"migration\"]"
"\"Old and Large\"" 1.0 0 -2064490 true "" "plot count searching_households with [searching_reason = \"Large_House_And_Old\" and any? my-out-relationships = false]"

SWITCH
324
412
516
445
social_leave
social_leave
1
1
-1000

PLOT
509
521
709
671
Time_waited_at_transfer
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"Time Year" 1.0 0 -16777216 true "" "ifelse transfers_year = 0[plot 0][\nplot waiting_time_at_transfer_year / transfers_year]"
"Time All" 1.0 0 -11085214 false "" "ifelse transfers_all = 0 [plot 0][\nplot waiting_time_at_transfer_all / transfers_all\n]"

CHOOSER
0
831
193
876
migration_type
migration_type
"flat" "percentage of households" "percentage of houses"
1

INPUTBOX
3
547
158
607
seed
0.0
1
0
Number

SWITCH
323
447
517
480
old_large_house_leavers
old_large_house_leavers
1
1
-1000

SWITCH
400
563
503
596
plot?
plot?
1
1
-1000

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

house bungalow
false
0
Rectangle -7500403 true true 210 75 225 255
Rectangle -7500403 true true 90 135 210 255
Rectangle -16777216 true false 165 195 195 255
Line -16777216 false 210 135 210 255
Rectangle -16777216 true false 105 202 135 240
Polygon -7500403 true true 225 150 75 150 150 75
Line -16777216 false 75 150 225 150
Line -16777216 false 195 120 225 150
Polygon -16777216 false false 165 195 150 195 180 165 210 195
Rectangle -16777216 true false 135 105 165 135

house efficiency
false
0
Rectangle -7500403 true true 180 90 195 195
Rectangle -7500403 true true 90 165 210 255
Rectangle -16777216 true false 165 195 195 255
Rectangle -16777216 true false 105 202 135 240
Polygon -7500403 true true 225 165 75 165 150 90
Line -16777216 false 75 165 225 165

house ranch
false
0
Rectangle -7500403 true true 270 120 285 255
Rectangle -7500403 true true 15 180 270 255
Polygon -7500403 true true 0 180 300 180 240 135 60 135 0 180
Rectangle -16777216 true false 120 195 180 255
Line -7500403 true 150 195 150 255
Rectangle -16777216 true false 45 195 105 240
Rectangle -16777216 true false 195 195 255 240
Line -7500403 true 75 195 75 240
Line -7500403 true 225 195 225 240
Line -16777216 false 270 180 270 255
Line -16777216 false 0 180 300 180

house two story
false
0
Polygon -7500403 true true 2 180 227 180 152 150 32 150
Rectangle -7500403 true true 270 75 285 255
Rectangle -7500403 true true 75 135 270 255
Rectangle -16777216 true false 124 195 187 256
Rectangle -16777216 true false 210 195 255 240
Rectangle -16777216 true false 90 150 135 180
Rectangle -16777216 true false 210 150 255 180
Line -16777216 false 270 135 270 255
Rectangle -7500403 true true 15 180 75 255
Polygon -7500403 true true 60 135 285 135 240 90 105 90
Line -16777216 false 75 135 75 180
Rectangle -16777216 true false 30 195 93 240
Line -16777216 false 60 135 285 135
Line -16777216 false 255 105 285 135
Line -16777216 false 0 180 75 180
Line -7500403 true 60 195 60 240
Line -7500403 true 154 195 154 255

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="migration" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="migration" first="0" step="0.01" last="0.1"/>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="spousewaiting2" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
      <value value="&quot;divorce&quot;"/>
      <value value="&quot;never&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="reactions4" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="reactions2" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="reactions12" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="reactions16" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="SocialLeave" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="maximumincome" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="50000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="lottery" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <steppedValueSet variable="supply_lottery" first="0" step="10" last="100"/>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_Baseline" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_Even" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_proto" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_1333" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_base_size" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_1234" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="building_%" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>experiment-go</go>
    <timeLimit steps="75"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="building_%_houses" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>experiment-go</go>
    <timeLimit steps="75"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;percentage of houses&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.09"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="building_%_households" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>experiment-go</go>
    <timeLimit steps="75"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.09"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="building_%_shortage" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>experiment-go</go>
    <timeLimit steps="75"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;percentage of shortgage&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="1.77"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="waiting_time_building" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;while_searching&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_more_free" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="28"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="41"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_more_owned" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="41"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="41"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_more_social_less_owned" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="61"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_more_owned_less_private" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="51"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="41"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="compositions_more_social_less_private" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="900"/>
    <metric>max [max time_waitinglist] of searching_households</metric>
    <metric>mean [max time_waitinglist] of searching_households</metric>
    <metric>max [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of searching_households with [income &lt; maximum_income_social_rental_market]</metric>
    <metric>mean [max time_waitinglist] of households</metric>
    <metric>max [max time_waitinglist] of searching_households with [(current_market = "none" or current_market = "parents") and income &lt; maximum_income_social_rental_market]</metric>
    <metric>count households</metric>
    <metric>count households with [current_market = "social_rental"]</metric>
    <metric>count households with [current_market = "private_rental"]</metric>
    <metric>count households with [current_market = "home_ownership"]</metric>
    <metric>count searching_households with [current_market = "none"]</metric>
    <metric>count searching_households</metric>
    <metric>count searching_households with [max time_waitinglist &gt; 0]</metric>
    <metric>count households with [current_market = "social_rental" and income &gt; maximum_income_social_rental_market]</metric>
    <metric>count searching_households</metric>
    <metric>count houses</metric>
    <metric>count s_houses</metric>
    <metric>count pr_houses</metric>
    <metric>count ho_houses</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [first sale_prices] of ho_houses with [first sale_prices != 0]</metric>
    <metric>count searching_households with [searching_reason = "Divorce" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "High_Rent" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Moving_Out" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Small_House" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Split" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Relationship" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "Social_Leave" and any? my-out-relationships = false]</metric>
    <metric>count searching_households with [searching_reason = "migration"]</metric>
    <metric>count searching_households with [searching_reason = "Large_House_And_Old" and any? my-out-relationships = false]</metric>
    <metric>waiting_time_at_transfer_year</metric>
    <metric>transfers_year</metric>
    <metric>waiting_time_at_transfer_all</metric>
    <metric>transfers_all</metric>
    <metric>min [income%] of households with [current_market = "social_rental"]</metric>
    <metric>min [income%] of households with [current_market = "private_rental"]</metric>
    <metric>min [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>max [income%] of households with [current_market = "social_rental"]</metric>
    <metric>max [income%] of households with [current_market = "private_rental"]</metric>
    <metric>max [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [income%] of households with [current_market = "social_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "private_rental"]</metric>
    <metric>mean [income%] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "social_rental"]</metric>
    <metric>mean [age] of households with [current_market = "private_rental"]</metric>
    <metric>mean [age] of households with [current_market = "home_ownership"]</metric>
    <metric>mean [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>min [age] of households with [current_market = "social_rental"]</metric>
    <metric>min [age] of households with [current_market = "private_rental"]</metric>
    <metric>min [age] of households with [current_market = "home_ownership"]</metric>
    <metric>min [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>max [age] of households with [current_market = "social_rental"]</metric>
    <metric>max [age] of households with [current_market = "private_rental"]</metric>
    <metric>max [age] of households with [current_market = "home_ownership"]</metric>
    <metric>max [age] of households with [current_market = "none" or current_market = "parents"]</metric>
    <metric>mean [rent] of s_houses with [status = "occupied"]</metric>
    <metric>mean [rent] of pr_houses with [status = "occupied"]</metric>
    <metric>mean [monthly_mortgage_payment] of home_owning_households</metric>
    <metric>run-seed</metric>
    <enumeratedValueSet variable="free_housing_supply%">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_housing_supply%">
      <value value="61"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="homeownership_housing_supply%">
      <value value="31"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size1/2%">
      <value value="31.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size3%">
      <value value="34.35"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size4%">
      <value value="23.69"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Size5%">
      <value value="10.67"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_to_221000">
      <value value="12.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_221000_to_310000">
      <value value="22.18"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_310000_to_385000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_385000_to_511000">
      <value value="22.48"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="house_worth_511000_plus">
      <value value="20.65"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51%">
      <value value="21.22"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_432.51_to_619.01%">
      <value value="53.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_619.01_to_663.40%">
      <value value="11.16"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_rent_663.40_to_737.14%">
      <value value="14.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%_of_new_cohabitants_that_marry">
      <value value="0.15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_interest">
      <value value="0.021"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_length">
      <value value="360"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mortgage_type">
      <value value="&quot;annuity&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_type">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new_houses_monthly">
      <value value="0.84"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_social_rental_market">
      <value value="40024"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_income_supply_middle_incomes">
      <value value="44655"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_lottery_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="maximum_simultaneous_reactions_social_market">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="supply_lottery">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="spouse_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="building_waiting_time">
      <value value="&quot;always&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="migration_type">
      <value value="&quot;percentage of households&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social_leave">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="old_large_house_leavers">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="population_size">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="housing_supply">
      <value value="951"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shock_size">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="shocked%">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plot?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seed">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
