%% MODEL ChannelFlow2D

% Loreta Cornacchia & Johan van de Koppel
% NIOZ Royal Netherlands Institute for Sea Research, Yerseke
% Updated May 2020
 
clear all;
close all;
clc;

modname = 'ChannelFlowV2';   % Fill in model name, to save output
RUN     = 0;                        % Choose new RUN-name

%% 1. Prepare model

on=1; off=0;                        % Define on/off switches
Periodic    = off       ;  % If on, a periodic boundary condition is assumed   
LoadData    = off       ;  % Whether data is loaded from an existing .mat file
PlantGrowth = on        ;  % If off, plant dynamics are not active

SAVE    = off;                       % Whether output should be saved
savedir = [modname,'_OUTPUT/'];     % Where output files should be saved


%% 2. Biophysical parameters

% 2a. Hydrodynamics
g           =   9.8     ;  % [m/s^2] Gravitational acceleration
Slope       =   7.0e-04 ;  % [m/m] Slope of the underlying channel bed
DifU        =   0.02    ;  % [m^2/s] Turbulent eddy viscosity (Set to obtain numerically stable results)
Uin         =   0.18    ;  % [m/s] The inflowing water speed

% 2b. Friction: Gauckler-Manning or Baptist et al. (J. Hydr. Res. 2007)
n           =  0.025    ;  % [s m^(-1/3)] Gauckler-Manning friction coefficient
Dc          =  0.04     ;  % [-] Drag coefficient (Naden et al., 2004)
L           =  0.5      ;  % [m] Shoot length
kv          =  0.41     ;  % [-] Von Karman constant

% 2c. Vegetation parameters
r           =  1.0      ;  % [1/day] Growth rate of plants
k           =  200      ;  % [g/m^2] Vegetation carrying capacity
m_W         =  3.8      ;  % [1/m] Plant mortality coefficient due to flow stress
DifP        =  0.00085  ;  % [m^2/day] Diffusion rate of plants

%% 3. Simulation parameters
NX          =   600      ;  % [-] Number of cells in x-direction (downslope)
NY          =   250      ;  % [-] Number of cells in y-direction (perpendicular to downslope)

Length_x    =   50       ;  % [m] Length of the simulated area
Length_y    =   15       ;  % [m] Width of the simulated area

dT          =   0.01     ;  % [days] Timestep size
EndTime     =   500     ;  % [days] Model simulation duration

% Graphics parameters
NumPlots    =   100      ;  % [-] Number of screen refreshments
ArrowNr     =   10       ;  % [-] Number of arrows plotted in flow field plot

% Output name for this model run
namestr     =   [modname,'_RUN',num2str(RUN)];

%% 5. Initialisation of variables

dx=Length_x/NX;
dy=Length_y/NY;

S = zeros(NX,NY);       % [m] Sediment elevation (stays 0)             
P = zeros(NX,NY);       % [g/m^2] Vegetation biomass 
Aw = zeros(NX,NY);      % [m^2/m^2] Wetted plant surface area
W = zeros(NX,NY);       % [m] Water depth
u = zeros(NX,NY);       % [m/s] Water velocity in the x dimension
v = zeros(NX,NY);       % [m/s] Water velocity in the y dimension

du  = zeros(NX,NY);     % Change in water velocity u
dv  = zeros(NX,NY);     % Change in water velocity v
dW  = zeros(NX,NY);     % Change due to water fluxes

% The advection operators in X and Y direction (Central differences in space, periodic at all boundaries)
d_dy = @(w)(w(1:end,[2:end,1])-w(1:end,[end,1:end-1]))/2/dy; 
d_dx = @(w)(w([2:end,1],1:end)-w([end,1:end-1],1:end))/2/dx;

% The diffusion (Laplacian) operator (Central differences in space, periodic at all boundaries)
d2_dxy2 = @(w)(w(1:end,[2:end,1])+w(1:end,[end,1:end-1])-2*w(1:end,1:end))/dy/dy + ...
              (w([2:end,1],1:end)+w([end,1:end-1],1:end)-2*w(1:end,1:end))/dx/dx;

%% 6. Initial conditions

% Subsediment gradient
Su=repmat(1:-1/(NX-1):0,NY,1)'*Slope*Length_x;

% Initial conditions for vegetation (two sides)
P(1:NX, 1:floor(NY*1/3)) = 200; 
P(1:NX,floor(NY*2/3):NY) = 200; 


W(:) = 0.3; 

count = 0;
count2 = 1;


u(:) = Uin;


if LoadData == on,
    load LoadData;
end;

Ux=u(1,:);  % Ux is used to calculate boundary conditions later
 

%% 7. Setting up the graphs  

% Arrays used for the plotting
[Y, X] = meshgrid(1:NX, 1:NY);

Xl=(1:NX)/NX*Length_x;           % Plot dimensions in X for the image plots
Yl=(1:NY)/NY*Length_y;           % Plot dimensions in Y for the image plots
Xs=X/NY*Length_x;                % X coordinates for each cell
Ys=Y/NX*Length_y;                % Y coordinates for each cell
Ri_x=1:floor(NX/ArrowNr):NX;     % Arrow positions along the X dimension
Ri_y=1:floor(NY/ArrowNr):NY;     % Arrow positions along the Y dimension

TitleStr= 'Channel simulation';

x = get(0,'ScreenSize'); ScreenDim=x(3:4);
MainWindowDim=floor(ScreenDim.*[1 0.83]);
          
%% 8. Set up the figures
Handle=figure('Position',[(ScreenDim-MainWindowDim)/2 MainWindowDim], 'Color','w');

fs = 17; % Plot fontsize
uv_range    = [0.0 0.6]; % Range of flow velocities for plotting

subplot('Position',[ 0.04 0.07 0.3 0.85]) % Biomass
Im1=imagesc(Yl, Xl, P); axis image;
colorbar;
colormap('jet');
caxis([0 150]);
title('Plant biomass (g m^{-2})', 'FontSize',fs); 
ylabel('Downslope distance (m)','FontSize',fs);
xlabel('Cross-slope distance (m)', 'FontSize',fs);
set(gca,'FontSize',fs)

subplot('Position',[ 0.68 0.57 0.27 0.35])
Plot1 = plot((1:NY)/NY*Length_y,(P(NX/2,:)./k),'k-','LineWidth',0.5);hold on;
Plot2 = plot((1:NY)/NY*Length_y,u(NX/2,:),'r-','LineWidth',0.5);
axis([0 Length_y 0 1]); 
ylabel('Plant biomass/Carrying capacity {\it k}','FontSize',fs)
xlabel('Cross-slope distance (m)','FontSize',fs);
set(gca,'FontSize',fs)
axesPosition = get(gca, 'Position');           % Get the current axes position
hNewAxes = axes('Position', axesPosition, ...  % Place a new axes on top...
                'Color', 'none', ...           %   ... with no background color
                'YLim', [0 1], ...            %   ... and a different scale
                'YAxisLocation', 'right', ...  %   ... located on the right
                'XTick', [], ...               %   ... with no x tick marks
                'Box', 'off');                 %   ... and no surrounding box
ylabel(hNewAxes, 'Flow speed (m s^{-1})','FontSize',fs,'Color','r');  % Add a label to the right y axis
title('Biomass and flow velocities along cross-section','FontSize',fs)
set(gca,'FontSize',fs)


subplot('Position',[ 0.68 0.07 0.27 0.35])
Plot3=plot(((1:NX)/NX*Length_x)',S(:,NY/2)+W(:,NY/2),'b-','LineWidth',0.5); hold on;
Plot4=plot((1:NX)/NX*Length_x,Su(:,NY/2),'r-','LineWidth',0.5); 
title('River bed and water depth','FontSize',fs);
ylabel('Elevation (m)','FontSize',fs)
xlabel('Downslope distance (m)','FontSize',fs)
set(gca,'FontSize',fs)


%% 9. Setting up simulation variables

jj=0; ii=1e10; Time=0; counter = 1;
tic;
%% 10. The simulation loop
    


while (Time<=(EndTime+1)),
    
  % Shallow water equations
      
  NetSpeed=sqrt(u.^2+v.^2);
  
  % Calculate bending angle as a function of incoming flow velocity (Bal et al., 2011):
  alpha = 15.5.*(NetSpeed.^(-0.38));
  
  % Calculate deflected vegetation height:
  Hv = L*sin(alpha/180*pi);
  
  
  % Relate biomass P to plant surface area, and convert from cm^2 to m^2
  % (Gregg & Rose, Aquat. Bot., 1982)
  Aw2 = (814.8.*(P)-25.05)*0.0001;
  
  Aw = max(Aw2,0);
  
  Vh = max(W./Hv,1);
  
  % Calculate vegetative Chezy roughness (Baptist et al, 2007):
  THS = (P < 0.03);
  Cd0 = (1/n).*(W.^(1/6));
  Cf1 = (g)./(Cd0.^2);
  Cd1 = (sqrt(1./((Cd0).^(-2)+(Dc*Aw/(2*g))))) + ((sqrt(g)/kv)*log(Vh));
  Cf2 = (g)./(Cd1.^2);
  Cf  = (THS.*Cf1 + not(THS).*Cf2);
  
  % Determining the elevation of the water surface
  H = S+W+Su;  
  
  % Determining the net speed of the water in both u and v direction
  
  du = - g.*(d_dx(W) - Slope) ...
       - u.*d_dx(u) ...
       - v.*d_dy(u) ...
       + DifU*d2_dxy2(u) ...
       - Cf.*(NetSpeed.*u)./W;
        
  dv = - g.*d_dy(W) ...
       - u.*d_dx(v) ...
       - v.*d_dy(v) ...
       + DifU*d2_dxy2(v) ...
       - Cf.*(NetSpeed.*v)./W;

  u = u + dT*du;
  v = v + dT*dv;               
        
  %% 11. Boundary conditions for water velocity, periodic on all sides
  
  if Periodic==off, % Periodic is default  
     %Constant flux in one dimension
     u(1,:)=Ux;    u(NX,:)=u(NX-1,:); %u(:,1)=u(:,NY-1); u(:,NY)=u(:,2);    

v(:,1)  = 0;  
v(:,NY) = 0; 

Ux = ((u(NX,:))./(sum(u(NX,:))).*(Uin*NY));
    

    
  end

  %% 12. Depth-averaged continuity equation
  
  dH  = - d_dx(u.*W) - d_dy(v.*W);   %OK!
  H = H + dT*(dH);                                %OK!
  W = H - S - Su;                                 %OK!

%      
  %% 13. Plant equations
  if PlantGrowth==on,
  dP1dt = r.*(k - P)./k.*P -  ...
         m_W.*P.*(NetSpeed) + DifP*d2_dxy2(P);
  P = P + dT*dP1dt;
  
  else
      dP1dt = 0;
      P = P + dT*dP1dt;
  end
  
   %% 14. Check conditions for numerical method convergence
    Pm = max([dx*u(:); dy*v(:)])/DifU;   % Mesh-Peclet number should not exceed 2, to prevent wiggles ("shark teeth") associated with central differences discretisation
    if(Pm>2) 
        % beep; warning('Peclet number should not exceed 2')
        disp('Peclet number should not exceed 2')
    end

    CFL = dT*max([u(:)/dx; v(:)/dy]); % Courant-Friedrichs-Lewy condition to be satisfied to ensure numerical convergence.
    if(CFL>1) 
        % beep; warning('CFL number should not exceed 1')
        disp('CFL number should not exceed 1')
    end

  %% 15. Boundary conditions
  if Periodic==off, % Periodic is default, rest is added
      W(1,:)=W(2,:)*2-W(3,:); W(NX,:)=W(NX-1,:)*2-W(NX-2,:);
  end; 

  %% 16. Graphic representation of the model in every timestep
 
  ii=ii+1;
  if (ii>EndTime/NumPlots/dT)
      
        set(Im1,'CData',P);
      
        set(Plot1,'Ydata', (P(NX/2,:)./k)); 
        set(Plot2,'Ydata', u(NX/2,:));
        set(Plot3,'Ydata', W(:,NY/2)+S(:,NY/2)+Su(:,NY/2));      
        set(Plot4,'Ydata', S(:,NY/2)+Su(:,NY/2));   

%        Plotting the flow field
        subplot('Position',[ 0.34 0.07 0.3 0.85])
        ImU = imagesc(Yl,Xl,NetSpeed); axis image; 
        hold on;
        colorbar; caxis(uv_range);
        Plot4=quiver(Ys(Ri_y,Ri_x), Xs(Ri_y,Ri_x), v(Ri_x,Ri_y),u(Ri_x,Ri_y),'w','LineWidth',1.5); 
        axis ij; axis image; %axis([0 Length_y, 0 Length_x ]);
        title('Absolute flow speed (m s^{-1}) and direction','FontSize',fs);      
        xlabel('Cross-slope distance [m]','FontSize',fs)
        set(gca,'FontSize',fs)
        hold off;  


        drawnow;  
        jj=jj+1;
        ii=0;
        
% Getting a movie frame
% if Movie==on 
% writerObj = VideoWriter('out.avi'); % Name it.
% open(writerObj); 
% MM(:,jj)=getframe(gcf); %end;
% writeVideo(writerObj,MM(:,jj));
% close(writerObj); % Saves the movie.
% 
% mov(:,jj)= struct('cdata',[],'colormap',[]);
% set(gca,'nextplot','replacechildren')
% mov(:,jj)=getframe(gcf);
% movie2avi(mov, '1moviename.avi', 'compression', 'None');
%         
          set(Handle,'Name', [TitleStr '; frame ' num2str(Time,'%1.0f') ' of ' num2str(EndTime)]);  
%         
   end

%% 17. Prepare next timestep

Time=Time+dT;

    % Stop simulation if results become unstable 
    if (sum(isnan(S(:))) ~= 0) | (sum(isnan(P(:))) ~= 0) | (sum(isnan(W(:))) ~= 0) | (sum(isnan(u(:))) ~= 0) | (sum(isnan(v(:))) ~= 0)
        disp('NaNs detected')
        break   % Break to continue with next simulation (when doing a bifcurcation analysis)
    end

end     % End of simulation loop

if SAVE == on
    mkdir(savedir);                     % Create new output folder
    EndImage = getframe(gcf); % Save image exactly as displayed on screen
    imwrite(EndImage.cdata, [savedir, namestr, '.png']);
    save([savedir, namestr, '.mat']) % Save entire workspace as .mat file
end


disp('Finished!');beep;toc;
