# Import necessary libraries
import numpy as np  # For numerical computations
from scipy.interpolate import interp1d  # For interpolation operations
import matplotlib.pyplot as plt  # For data visualization
from scipy.optimize import curve_fit  # For non-linear curve fitting

# Define the dispersion relation function
def disper(w, h, g=9.81):
    # Calculate the dimensionless parameter w2
    w2 = (w ** 2) * h / g
    
    # Initial guess for q
    # q is approximated based on the dimensionless parameter w2
    q = w2 / ((1 - np.exp(-w2 ** (5 / 4))) ** (2 / 5))

    # Perform two iterations using the Newton-Raphson method to refine q
    for _ in range(2):
        # Compute tanh(q) and its related derivatives
        thq = np.tanh(q)
        thq2 = 1 - thq ** 2  # Derivative of tanh(q): 1 - tanh^2(q)
        
        # Compute the coefficients of the equation
        a = (1 - q * thq) * thq2  # Coefficient of the quadratic term
        b = thq + q * thq2  # Coefficient of the linear term
        c = q * thq - w2  # Constant term

        # Intermediate variable arg to store the update
        arg = np.zeros_like(q)
        
        # Calculate the root of the equation where a != 0
        iq = a != 0
        arg[iq] = (b[iq] ** 2) - 4 * a[iq] * c[iq]  # Discriminant
        arg[iq] = (-b[iq] + np.sqrt(arg[iq])) / (2 * a[iq])  # Compute the update

        # Use an alternative method to calculate the update when the equation flattens
        iq2 = np.abs(a * c) < 1.0e-8 * (b ** 2)
        arg[iq2] = -c[iq2] / b[iq2]
        
        # Update the value of q
        q = q + arg
        
    # Compute the final wave number k
    # k = sign(w) * q / h, where sign(w) preserves the direction of the wave number
    k = np.sign(w) * q / h
    
    # Handle NaN values: Replace NaN with 0
    kk = np.copy(k)
    kk[np.isnan(kk)] = 0

    # Return the wave number array
    return kk
