;; Agent-Based Simulation of West Asian Urban Dynamics: Impact of Refugees
;; JASSS
;; (c) 2014-2016 S. Picascia
;; (c) 2015-2020 N. Yorke-Smith
;; licensed under CC BY-NC-ND 4.0
;; contact: n.yorke-smith@tudelft.nl
;;
;; This model is derived from S. Picascia's NetLogo model gentaxelling, which describes the basic operation: https://github.com/harrykipper/gentaxelling
;;
;; This models runs with NetLogo 6.0.4.
;;
;; Due to copyright reasons, the Beirut GIS data cannot be released.
;; It should be placed in subfolder GIS/Beirut.

extensions [gis table vid profiler]

globals [
  version save-directory save-directory-pictures file-name-results
  rnetlogo-param-refugees  ; if nonzero, overwrite refugee-start/end-tick according to this parameter
  rnetlogo-param-regen     ; if nonzero, overwrite regeneration?according to this parameter
  city       ; city = those patches with instantiated 'city-zone'; for them their 'area' = their 'city-zone'
  areas      ; list of areas of the city
  divisions  ; raw GIS input
  districts  ; = areas in this version of model (used to be different)
  allured-districts firstallure declining  ; metrics on district
  neighbourhoods-table  ; compiles a list of neighbourhoods at start for quick lookup when running
  occupancys-table      ; compiles a list of citizens in each neighbourhood, need to keep it updated
  disp-freq  ; how many ticks between updates of entropy graph
  disp?      ; whether entropy graph should be displayed this tick
  ownersize  ; size of owner turtle
  city-allure av-income sd-income sd-income+ sd-income- occupation-rate gentrified-districts downfiltered-districts  ; reporters for analysis
  recolonisation degentrification recreation regentrification  ; more reporters
  housing-waiting-list  ; social housing waitlist
  immigrated-count emigrated-count refugee-immigrated-count refugee-emigrated-count  ; immigration and emigration counters
  median-Income medianPrices highestMedianPrice mostRefugee have-refugees? howmanySlums howmanyWellMaintained howmanyOvercrowded ; data lists for csv output
  lebanese refugees others  ; data lists for population-by-ethnicity-metagroup counting, for csv output
  extra-agent-creation-multiplier extra-citizen-creation-multiplier
]

breed [citizens citizen]  ; citizens are people in the city
breed [people person]     ; people may or may not be citizens
links-own [time-lo]	      ; used to be called "time"; renamed for NetLogo 6
turtles-own [mobility-propensity months-here birthday culture income credit-mine dissonance place-changes time-in-a-slum owner?]
patches-own [city-zone    ; identifier from GIS
  condition price centre? cbd? dist local-dist premium price-gap months-empty
  area          ; = city-zone in this version of model (used to be different)
  neighbourhood ; = area in this version of model (used to be different)
  ;ward          ; not used for Beirut
  allure social? last-renovated]


to setup
  ; clear-all will clear any globals that don't have an interface widget
  ; we use some such globals to drive netlogo from R
  ; so we save them to temp local variables (which clear-all doesn't clear)
  let my-rnetlogo-param-refugees rnetlogo-param-refugees
  let my-rnetlogo-param-regen rnetlogo-param-regen
  clear-all
  set rnetlogo-param-refugees my-rnetlogo-param-refugees
  set rnetlogo-param-regen my-rnetlogo-param-regen

  set version "0.6.5_BEY"
  set save-directory "results/"
  set save-directory-pictures (word save-directory "pics/")
  let gis-data-dir "GIS/Beirut/"
  set disp-freq 5
  set disp? false
  if record? [vid:reset-recorder]

  set firstallure 0
  set allured-districts table:make
  set gentrified-districts table:make
  set downfiltered-districts table:make
  set housing-waiting-list table:make
  set declining []
  set recolonisation []
  set degentrification []
  set regentrification []
  set recreation []

  set howmanyWellMaintained []
  set howmanyOvercrowded []
  set howmanySlums []
  set median-Income []
  set medianPrices []
  set highestMedianPrice []
  set mostRefugee  []
  set lebanese []
  set refugees []
  set others []

  set have-refugees? true
  if refugee-start-tick >= min (list refugee-end-tick end-tick) [set have-refugees? false]

  ; parse refugee input parameter
  if rnetlogo-param-refugees != 0 [
    ;print word "rnetlogo-param-refugees " rnetlogo-param-refugees
    set rnetlogo-param-refugees (closest-integer-to rnetlogo-param-refugees)
    ifelse rnetlogo-param-refugees = 1 [ ; none
      set have-refugees? false
      print "PARAMETER: no refugees"
    ] [ ifelse rnetlogo-param-refugees = 2 [ ; window
      set refugee-start-tick 90
      set refugee-end-tick 180
      set have-refugees? true
      print "PARAMETER: refugees"
    ] [ ifelse rnetlogo-param-refugees = 3 [ ; long window
      set refugee-start-tick 90
      set refugee-end-tick 270
      set have-refugees? true
      print "PARAMETER: refugees many refugees"
    ] [ if rnetlogo-param-refugees = 4 [ ; flood
      set refugee-start-tick 90
      set refugee-end-tick end-tick
      set have-refugees? true
      print "PARAMETER: refugees flood"
    ]]]]
  ]
  if rnetlogo-param-regen != 0 [
    ;print word "rnetlogo-param-regen " rnetlogo-param-regen
      set rnetlogo-param-regen (closest-integer-to rnetlogo-param-regen)
    ifelse rnetlogo-param-regen = 1 [ ; none regen
      set regeneration? "none"
      print "PARAMETER: no regen"
    ] [ ifelse rnetlogo-param-regen = 2 [ ; regular regen
      set regeneration? "regular"
      print "PARAMETER: regen"
    ] [ if rnetlogo-param-regen = 3 [ ; EU-slums regen
      set regeneration? "EU-slums"
      print "PARAMETER: EU-slums regen"
    ]]]
  ]

  set extra-agent-creation-multiplier ifelse-value have-refugees? [6] [4]         ; we create population * max-nonlum-occupancy agents, multiplied by this number
  set extra-citizen-creation-multiplier ifelse-value have-refugees? [1] [0.666]   ; we allocate population * max-nonlum-occupancy citizens, multiplied by this number

  set city-allure n-values traits ["x"]
  ifelse enable-culture? [
    set mixing? true
    set pull? true
    set push? true
    set strong-neighbourhood? true
  ][ ; no culture
    set mixing? false
    set pull? false
    set push? false
    set strong-neighbourhood? false
  ]

  if non-slum-population >= slum-population [
    set slum-population non-slum-population + 1
    output-print word "Slum population limit must be greater than non-slum population limit, setting it to " non-slum-population
  ]
  if refugee-end-tick < refugee-start-tick [
    ;set refugee-end-tick refugee-start-tick
    output-print "Warning: refugee-end-tick < refugee-start-tick"
  ]

  ;; initialize patches
  set ownersize 0.5
  ask patches
    [ set pcolor blue ;white
      set social? false		  ;; TODO replace with GIS social -- except no social in Beirut
      set months-empty 0
      set allure []
      set cbd? false
      ;set centreness 0.00001
      set condition random-float 1	  ; overwritten by GIS condition later
      ;set-neighbourhood
      ;create-price        ; this might update condition also  ; overwritten by GIS price later
    ]

  ;; Beirut GIS load
  set divisions gis:load-dataset (word gis-data-dir "beirut_div.shp")
  gis:set-world-envelope gis:envelope-of divisions
  draw-city
  ;apply-prices  ; GIS prices and conditions
  gis:set-coverage-minimum-threshold 0.75
  gis:apply-coverage divisions "CAD_NAME" city-zone
  ask patches [ifelse is-string? city-zone [
    ;DEBUG output-print (word "set-area for " city-zone)
    set-area
  ][;output-print (word self " has no city-zone " city-zone)
  ]]
  apply-prices-by-area  ; GIS prices and conditions

  ;; city is those patches with instantiated 'city-zone', for them their 'area' = their 'city-zone'
  set city patches with [
    is-string? city-zone
  ]
  ;set districts remove-duplicates [neighbourhood] of patches  ; now done from areas
  compile-lists
  check-area-sanity  ; also sets districts = areas
  set neighbourhoods-table table:from-list map [? -> list ? (patches with [neighbourhood = ?]) ] districts  ; (needs NetLogo 6.0.2+)
  ;DEBUG print neighbourhoods-table

  ;;set-city-condition
  ask city [
    set-last-renovated
    colour-patches
  ]
  set-centres-and-distances

  generate-population
  populate-city-skewed-population
  ask citizens [
    colour-agent
    shape-agent
  ]
  ; set-social
  occupancy-table-update
  ;DEBUG print occupancys-table

  ;output-print "Ready!"
  print "Ready!"
  reset-ticks
  if record? [vid:start-recorder]
  if profile? [profiler:start]
end



to compile-lists
  ;set districts remove-duplicates [ward] of city
  set areas remove-duplicates [area] of city
  output-print word (length areas) " areas in city"
end

to set-area
  set area city-zone
  set-neighbourhood
end

to check-area-sanity
  foreach areas [ ? ->
    let these city with [area = ?]
    if count these < 5 [
      set areas remove ? areas
      let where adjacentarea ?
      ask these [
        set area where
        set-neighbourhood
      ]
    ]
  ]
  set districts areas  ; districts = areas
end


to generate-population
  ;DEBUG output-print "Creating agents..."
  create-people N-Agents * extra-agent-creation-multiplier * slum-population [render-human]  ; multiplied because we might have high immigration rate and we need enough refugees in the population
  set-default-shape turtles "circle"
  ;DEBUG output-print "Setting ethnicities..."
  create-ethnic-division
  ;DEBUG output-print "Setting economic statuses..."
  create-economic-status
end

to render-human
  set birthday 0
  ;if ticks = 0 [create-culture]
  create-culture
  reset-mobility-propensity
  set hidden? true
  set size 0.4
end

to render-human-after-time-zero
  set birthday ticks
  ;create-culture
  reset-mobility-propensity
  ;set hidden? true
  set size 0.4
  colour-agent
  shape-agent
end

to reset-mobility-propensity
  set mobility-propensity (random-float prob-move) + 0.01
end

;; culture item 0: ethnicity
;;   0 => lebanese
;;   1 => low cast arab (e.g. syrian/refugee)
;;   2 => high cast arab (e.g. UAE)
;;   3 => high cast non-arab (e.g. USA)
;;   4 => low cast non-arab (e.g. maid)
;; culture item 1: religion
;;   0 => christian
;;   1 => sunni
;;   2 => shi'a
;;   3 => druze
;;   4 => other
to create-culture
  set culture n-values traits [random values]
  set culture replace-item 0 culture 5  ; this means: ethnicity unknown/not allocated
  set culture replace-item 1 culture 5  ; this means: religion unknown/not allocated
end

to create-ethnic-division
  let totalpop count people
  ;; allocate ethnicity
  ; est. 4M Leb => 0.62
  ;      2M Syrian + Palestinian => 0.3
  ;      0.05M high-cast Arab => 0.008
  ;      0.2M high-cast foreigner => 0.03
  ;      0.25M low-cast foreigner => 0.04
  ; total 6.5M
  ;
  ; however, at start we're before Syrian crisis, thus only working (non-refugee) Syrians in Leb.
  ;      4M Leb => 0.85
  ;      0.2M Syrian + Palestinian => 0.042
  ;      0.05M high-cast Arab => 0.01
  ;      0.2M high-cast foreigner => 0.042
  ;      0.25M low-cast foreigner => 0.05
  ; total 4.7M
  ;
  ; hence: allocate total ethnic division (the pool of all people) according to "now", i.e. 2M Syrian,
  ; but populate the city (the citizens) according to the "before", i.e., 0.2M Syrian -- in populate-city-skewed-population below
  ask n-of (round (0.62 * totalpop)) people with [item 0 culture = 5] [set culture replace-item 0 culture 0] ; Lebanese
  ask n-of (round (0.3 * totalpop)) people with [item 0 culture = 5] [ ; refugee
    set culture replace-item 0 culture 1
    set culture replace-item 1 culture 1  ; refugees are Sunni (but see below)
  ]
  ask n-of (round (0.008 * totalpop)) people with [item 0 culture = 5] [ ; high-cast Arab
    set culture replace-item 0 culture 2
    set culture replace-item 1 culture 1  ; other Arabs are Sunni
    ]
  ask n-of (round (0.03 * totalpop)) people with [item 0 culture = 5] [ ; high-cast foreigner
    set culture replace-item 0 culture 3
  ]
  ask n-of (round (0.04 * totalpop)) people with [item 0 culture = 5] [ ; low-cast foreigner
    set culture replace-item 0 culture 4
  ]
  ask people with [item 0 culture = 5] [set culture replace-item 0 culture random traits]
  report-population

  ;; allocate religion
  ; demographic study conducted by the research firm Statistics Lebanon found that approximately 27% of the population was Shia, 27% Sunni, 21% Maronite, 8% Greek Orthodox, 5% Druze, 5% Melkite, and 1% Protestant, with the remaining 6% mostly belonging to smaller non-native to Lebanon Christian denominations
  ask people with [item 0 culture = 0] [ ; Lebanese sects
    ifelse (random-float 1 < 0.39) [set culture replace-item 1 culture 0] [ ; 39% Christian
      ifelse (random-float 1 < 0.44) [set culture replace-item 1 culture 1] [ ; of remaining 61%, 27% / 0.61 = 44% Sunni
        ifelse (random-float 1 < 0.61) [set culture replace-item 1 culture 2] [ ; of remaining 44%, 27% / 0.34 = 61% Shia
          ifelse (random-float 1 < 0.71) [set culture replace-item 1 culture 3] [ ; of remaining 7%, 5% / 0.7 = 71% Druze
            set culture replace-item 1 culture 4  ; other
          ]]]]]
  ask people with [item 0 culture = 1] [ ; some refugees are Shi'a
    if random-float 1 < 0.25 [set culture replace-item 1 culture 2]
  ]
  ask people with [item 1 culture = 5] [set culture replace-item 1 culture random traits]
  report-religion
end

to create-economic-status
  ifelse enable-economy?
  [ask people [set credit-mine credit]]
  [ask people [set credit-mine 1]]

  ifelse real-data? [ ; real numbers
    output-print "Setting Lebanese incomes"
    ;; from https://en.wikipedia.org/wiki/Economy_of_Lebanon, World Bank estimates 2013
    ;;   15% of the Lebanese people lives below poverty line ($2,500)[44]
    ;;   54% of the Lebanese people lives in the moderate middle class ($9,000) annually. Increase 12% from 1998
    ;;   32% of the Lebanese people lives in the upper middle class ($15,000 - $27,000) annually. Decrease 19% from 1998
    ;;   7% of the Lebanese people lives in the highest upper class rich ($30,000 and above) annually. Increase 1% from 1998
    ;;
    ;; and https://publications.credit-suisse.com/tasks/render/file/?fileID=5521F296-D460-2B88-081889DB12817E02, estimates 2014
    ;; set income
    ask people [set income 0]
    let totalpop count people
    ask n-of (round 0.003 * totalpop) people with [income = 0] [set income random-poisson 1000000]
    ask n-of (round 0.03 * totalpop)  people with [income = 0] [set income random-poisson 100000]
    ask n-of (round 0.32 * totalpop)  people with [income = 0] [set income random-poisson 20000]
    ask n-of (round 0.54 * totalpop)  people with [income = 0] [set income random-poisson 9000]
    ask n-of (round 0.13 * totalpop)  people with [income = 0] [set income random-poisson 1000]
    ask people with [income = 0] [set income 2500 + random-poisson 10000]

    ;; adjust income (of non-Lebanese) according to class/ethnicity
    ;DEBUG output-print "Setting non-Lebanese incomes; ownerships..."
    ask people with [item 0 culture = 1] [  ; refugee
      ;; from https://www.oxfam.org/sites/www.oxfam.org/files/rr-bric-livelihoods-syrian-refugees-lebanon-211113-en.pdf
      set income 2000 + random-poisson 2000
      set credit-mine 1  ; refugees never can have credit
    ]
    ask people with [item 0 culture = 2] [
      ;set income 0.7 + random-exponential 0.15]
      set income 100000 + random-poisson 1000000]
    ask people with [item 0 culture = 3] [
      ;set income 0.4 + random-exponential 0.3]
      set income 20000 + random-poisson 100000]
    ask people with [item 0 culture = 4] [  ; maid
      set income 2500 + random-poisson 500
      set credit-mine 1  ; maids never can have credit
    ]
  ][ ; not real numbers
    ;; set income
    ask people [set income random-float 1]  ; initial random
    if random-income? = false [create-skewed-economic-status]  ; skew it towards Gini

    ;; adjust income (of non-Lebanese) according to class/ethnicity
    ;DEBUG output-print "Setting non-Lebanese incomes; ownerships..."

    ask people with [item 0 culture = 1] [  ; refugee
      set income 0.01 + random-exponential 0.1
      set credit-mine 1  ; refugees never can have credit
    ]
    ask people with [item 0 culture = 2] [set income 0.7 + random-exponential 0.15]
    ask people with [item 0 culture = 3] [set income 0.4 + random-exponential 0.3]
    ask people with [item 0 culture = 4] [  ; maid
      set income 0.05 + random-exponential 0.05
      set credit-mine 1  ; maids never can have credit
    ]
    ask people with [income > 1.0] [set income 1.0]
  ]

  ;; ownership TODO
  let totalpop count people
  ask people [set owner? false]
  if have-owners? = true [
    ask n-of (round (ownership * totalpop)) people with [item 0 culture = 0 or item 0 culture = 2]
    [ ;; only Lebanese and high-cast Arab can own property
        set owner? true
        set size ownersize
    ]
  ]
end

;; skewed economic status only for ethnicity 0
;;
;; note: income is (re)set for other ethnicities, later, in create-economic-status
to create-skewed-economic-status
  ;;  adjust one up and one down each time since this method changes the mean - Not SURE this is an improvement!
  let thresh 0.05
  let chng 0.8
  let chng-abs 0
  let med 0
  let local-people people with [item 0 culture = 0]
  let num-local-people count local-people
  output-print (word "(" (count local-people) " local-people have gini " gini local-people " target " init-gini ")")

  let delta init-gini - (gini local-people)
  if delta > thresh [
    ;DEBUG output-print "Skewing economic statuses (down)..."
    ;output-print init-gini - gini local-people
    while [delta > thresh] [
      let num-people-to-change 1 + 200 * delta
      if num-people-to-change > num-local-people [set num-people-to-change num-local-people]
      set med mean ([income] of local-people)
      ask n-of num-people-to-change local-people with [income > med] [
        set chng-abs chng * (income - med)
        set income income - chng-abs
      ]
      ask n-of num-people-to-change local-people with [income < med] [set income max list 0 income - chng-abs]
      set delta init-gini - (gini local-people)
      ;print (word med " " gini local-people)
      ;print (word (gini local-people) " (" delta ") " num-people-to-change)
    ]
  ]
  ;DEBUG print (word "actual " gini local-people " vs desired " init-gini)
  if gini local-people - init-gini > thresh [
    ;DEBUG output-print "Skewing economic statuses (up)..."
    ;output-print gini local-people - init-gini
    while [gini local-people - init-gini > thresh] [
      set med mean ([income] of local-people)
      ask one-of local-people with [income < med] [
        set chng-abs chng * (med - income)
        set income income + chng-abs
      ]
      ask one-of local-people with [income > med] [set income max list 0 income - chng-abs]
      print (word med " " gini local-people)
    ]
  ]

  ask turtles with [income > 1] [set income 1]
  ask turtles with [income < 0.01] [set income 0.01]
end


to populate-city
  ;DEBUG output-print "Populating city..."
  output-print "Populating city..."
  let number-initial-citizens how-many-initial-citizens
  while [count citizens < how-many-initial-citizens] [
    ask one-of people [try-populate-one-person city]
  ]
end

to-report how-many-initial-citizens
  let number-initial-citizens N-Agents
  if non-slum-population > 1 [
    set number-initial-citizens ceiling (N-Agents * non-slum-population * extra-citizen-creation-multiplier)  ; since multiple occupancy is possible, we should create more than N-Agents at start
  ]
  output-print (word "Of " (count people) " people, initial citizens: " number-initial-citizens)
  report number-initial-citizens
end

to try-populate-one-person [where]
  ;DEBUG print (word "trying to populate person to " where)
  set where ensure-patchset-of-city where
  ;DEBUG output-print where
  ;let acceptable where with [not is-slum? and count citizens-here < non-slum-population and price <= ([income] of myself * [credit-mine] of myself)]  ; was: city with ...
  let acceptable where with [not is-slum? and count citizens-here < non-slum-population]
  ;DEBUG if (count acceptable = 0) [print (word "  acceptable 1 is " (count acceptable))]
  set acceptable acceptable with [price <= ([income] of myself * [credit-mine] of myself)]
  ;DEBUG if (count acceptable = 0) [print (word "  acceptable 2 is " (count acceptable))]  ; usually, if can't populate a person it's because he can't afford a place
  set acceptable acceptable with [(not cbd?) or (cbd? and count citizens-here < cbd-population)]  ; enforce CBD density limit from the beginning
  ;DEBUG if (count acceptable = 0) [print (word "  acceptable 3 is " (count acceptable))]
  if owner? [set acceptable acceptable with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else
  ifelse any? acceptable [
    ;; there is somewhere I can afford
    move-to-place-and-spread-out-turtles-and-no-update-occupancy-table one-of acceptable
    set months-here 0
    set breed citizens
    ; owner flag is set according to ethnicity (elsewhere)
    set hidden? false
  ] [
    ;DEBUG print (word "  failed to populate person to " ([area] of one-of where))
  ]
end

to try-populate-one-person-skewed
  ;; randomnly pick a district, according to distribution
  let chosen-district pick-district-skewed
  try-populate-one-person chosen-district
end

to try-populate-one-person-slum [where]
  ;DEBUG print (word "trying to populate slum-person to " where)
  set where ensure-patchset-of-city where
  ;DEBUG output-print where
  let acceptable where with [(not cbd?) or (cbd? and count citizens-here < cbd-population)]   ; enforce cbd density limit from the beginning
  if item 0 culture = 1 [set acceptable acceptable with [not cbd?]]    ; no refugees in cbd from the start
  if owner? [set acceptable acceptable with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else
  set acceptable acceptable with [(not is-slum? and count citizens-here < non-slum-population and price <= ([income] of myself * [credit-mine] of myself)) or  ; a regular place...
      (is-slum? and count citizens-here < slum-population)]   ; ...or a slum
  ifelse any? acceptable [
    ;; there is somewhere I can afford (or a slum)
    move-to-place-and-spread-out-turtles-and-no-update-occupancy-table one-of acceptable
    set months-here 0
    set breed citizens
    set hidden? false
  ] [
    ;DEBUG print (word "  failed to populate slum-person to " ([area] of one-of where))
  ]
end

to try-populate-one-person-slum-skewed
  ;; randomnly pick a district, according to distribution
  let chosen-district ""
  while [chosen-district = ""] [
    set chosen-district pick-district-skewed
    let chosen-district-two (ensure-patchset-of-city chosen-district)
    if (is-cbd? chosen-district-two) [
      ; can't have slum-person in cbd
      set chosen-district ""
    ]
  ]
  try-populate-one-person chosen-district
end

to populate-city-skewed-population
  ; at start we're before Syrian crises, thus only working (non-refugee) Syrians in Leb.
  ;      4M Leb => 0.85
  ;      0.2M Syrian + Palestinian => 0.042
  ;      0.05M high-cast Arab => 0.01
  ;      0.2M high-cast foreigner => 0.042
  ;      0.25M low-cast foreigner => 0.05
  ; total 4.7M
  ;DEBUG output-print "Populating city..."
  let number-initial-citizens how-many-initial-citizens
  ;DEBUG output-print districts
  while [count citizens < number-initial-citizens] [
    ifelse (random-float 1 < 0.85) [ask one-of people with [item 0 culture = 0] [try-populate-one-person-skewed]] [ ; 85% Leb.
      ifelse (random-float 1 < 0.28) [ask one-of people with [item 0 culture = 1] [try-populate-one-person-slum-skewed]] [ ; of remaining 15%, 4.2% / 0.15 = 28% refugee
        ifelse (random-float 1 < 0.09) [ask one-of people with [item 0 culture = 2] [try-populate-one-person-skewed]] [ ; of remaining 10.8%, 1% / 0.108 = 9% high-cast Arab
          ifelse (random-float 1 < 0.43) [ask one-of people with [item 0 culture = 3] [try-populate-one-person-skewed]] [ ; of remaining 9.8, 4.2% / 0.098 = 43% high-cast foreigner
            ask one-of people with [item 0 culture = 4] [try-populate-one-person-slum-skewed] ; low-cast foreigner
          ]]]]
  ]
end

;; Moussaytbeh Medawar Remeil Bachoura Ras Beyrouth Achrafieh Mazraa Port Saifé Ain el-Mreissé Zoukak el-Blatt Minet el-Hosn Beirut Central District
to-report pick-district-skewed
  let districts-sorted (sort districts)  ; TODO save not recompute
  let r random-float 1
  let i 0
  let psum 0
  let chosen-district ""
  while [chosen-district = "" and i < length districts-sorted] [
    let d item i districts-sorted
    let p (proportion-initial-population-in-district d)
    set psum (psum + p)
    if (psum >= r) [
      set chosen-district d
      ;DEBUG output-print (word "chose " d " " psum " " r)
    ]
    set i (i + 1)
  ]
  if chosen-district = "" [
    print "pick-district-skewed: failed to pick a district"
  ]
  report chosen-district
end

;; Beirut district populations from census estimates
to-report proportion-initial-population-in-district [here]
  if here = "Moussaytbeh" [report 0.22]
  if here = "Medawar" [report 0.015]
  if here = "Remeil" [report 0.09]
  if here = "Bachoura" [report 0.04]
  if here = "Ras Beyrouth" [report 0.12]
  if here = "Achrafieh" [report 0.17]
  if here = "Mazraa" [report 0.26]
  if here = "Port" [report 0.00]
  if here = "Saifé" [report 0.01]
  if here = "Ain el-Mreissé" [report 0.02]
  if here = "Zoukak el-Blatt" [report 0.04]
  if here = "Minet el-Hosn" [report 0.005]
  if here = "Beirut Central District" [report 0.01]
  output-print (word "proportion-initial: unknown " here)
  report 0
end


;; Price dependant on condition + location.
;; Not used: the call to create-price is commented-out in setup
to create-price
  set price condition + 0.1
  ;if neighbourhood = "cbd" [set price price * 1.60]
  if cbd? [ ;neighbourhood = "n" [   ; CBD
    set condition condition * 1.6
    set price price * 1.6]
  if neighbourhood = "nw" [  ; Ras Beirut
    set condition condition * 1.2
    set price price * 1.4]
  if neighbourhood = "e" [   ; Ashrafiah
    set condition condition * 1.2
    set price price * 1.4]
  if neighbourhood = "sw" [  ; "southern suburbs"
    set condition condition * 0.6
    set price price * 0.6]
  if condition > 1 [set condition 1]
  if price > 1 [set price 1]
end

;; prices artificial, based on condition
to apply-prices
  ask patches [
    if condition > 1 [set condition 1]
    set price condition + 0.1
    if price > 1 [set price 1]
  ]
end

;; prices from actual data
to apply-prices-by-area
  ifelse real-data? [
    ; condition of Ashrafieh = Rmeil and Port = Mudawar
    ask patches [
      let lb-prob (initial-maintenance-of-district area) * 0.8
      let ub-prob (initial-maintenance-of-district area) * 1.2
      ;; FIXME hack
      if area = "Beirut Central District" [set lb-prob (lb-prob + 0.1)]  ; could check here it's above min condition for CBD
      set condition (random-float-between lb-prob ub-prob)
      if condition < 0.05 [set condition 0.05]
      if condition > 1 [set condition 1]
      set price (initial-price-of-district area) * condition
      ;if price > 1 [set price 1]
      ;DEBUG output-print (word self " in area " area " has price " price)
    ]
  ] [
    print "INFO: apply-prices-by-data -> apply-prices"
    apply-prices
  ]
end

;; initial prices, Beirut ~2010
to-report initial-price-of-district [here]
  ; end-2009 prices in USD from econometric analysis:
  if here = "Moussaytbeh" [report 9340]
  if here = "Medawar" [report 5860]
  if here = "Remeil" [report 6390]
  if here = "Bachoura" [report 3680]
  if here = "Ras Beyrouth" [report 12100]
  if here = "Achrafieh" [report 9160]
  if here = "Mazraa" [report 6260]
  if here = "Port" [report 9300]
  if here = "Saifé" [report 14200]
  if here = "Ain el-Mreissé" [report 9600]
  if here = "Zoukak el-Blatt" [report 6290]
  if here = "Minet el-Hosn" [report 17320]
  if here = "Beirut Central District" [report 9300] ; = Port
  ;DEBUG output-print (word "initial-price-of-district: unknown " here)
  report 0
end

;; initial maintenance conditions, Beirut ~2010
;; back-estimated from 2017 survey, by expert opinion
to-report initial-maintenance-of-district [here]
  if here = "Moussaytbeh" [ ; FIXME hack since district granuality too course for this large district
    let maint-value 0.5 - 0.35 * (1 - abs ((pycor - min-pycor) / min-pycor))
    if maint-value < 0 [set maint-value 0.05]
    report maint-value
    ;report 0.50]  ; guess, because missing (mis-named tab?) in spreadsheet from Sam
  ]
  if here = "Medawar" [report 0.71]
  ;if here = "Remeil" [report 0.50]   ; ?? because note says = maintenace Achrafieh, but Remeil has separate tab in spreadsheet from Sam
  if here = "Remeil" [report 0.72]   ; ?? because note says = maintenace Achrafieh, but Remeil has separate tab in spreadsheet from Sam
  if here = "Bachoura" [report 0.76]
  if here = "Ras Beyrouth" [report 0.77]
  if here = "Achrafieh" [report 0.72]
  if here = "Mazraa" [report 0.63]
  if here = "Port" [report 0.95]   ; = maintenance BCD
  if here = "Saifé" [report 0.90]
  if here = "Ain el-Mreissé" [report 0.90]
  if here = "Zoukak el-Blatt" [report 0.59]  ; guess, because missing (mis-named tab?) in spreadsheet from Sam
  if here = "Minet el-Hosn" [report 0.78]
  if here = "Beirut Central District" [report 0.95]  ; guess, because missing (mis-named tab?) in spreadsheet from Sam
  ; any patch outside the city boundaries won't have its `area' set, hence will be unknown here
  ; can we remove all such patches? or does netlogo need to keep them to have a square grid?
  ;DEBUG output-print (word "initial-maintenance-of-district: unknown " here)
  report 0
end


to set-centres-and-distances
  if kind = "policentric" [
    foreach areas [ ? ->
      let x (([pxcor] of max-one-of city with [area = ?] [pxcor] - [pxcor] of min-one-of city with [area = ?] [pxcor]) / 2) + [pxcor] of min-one-of city with [area = ?] [pxcor]
      let y (([pycor] of max-one-of city with [area = ?] [pycor] - [pycor] of min-one-of city with [area = ?] [pycor]) / 2) + [pycor] of min-one-of city with [area = ?] [pycor]
      if member? patch x y city [
      ask patch x y [
        if not any? neighbors with [not member? self city] [
          set centre? true
          ;set pcolor blue
        ]
      ]
    ] ]
  ] ; endif policentric

  ;; compute dist, which records distance from me to the chosen centre, which in turn depends on type of city
  ask city [
    let centre min-one-of city with [city-zone = "Beirut Central District"] [distance myself]  ;; monocentric city
    if kind = "policentric" [
      set centre min-one-of city with [centre? = true] [distance myself]  ;; policentric city
    ]
    ifelse kind = "no centres" [set dist 1] [set dist distance centre]
  ]
end

to set-neighbourhood
  set neighbourhood area
  ifelse city-zone = "Beirut Central District" [
    set cbd? true
    set plabel "CBD"
  ] [
    set cbd? false
  ]
end

to set-neighbourhood-nogis
  set cbd? false
  if pxcor >= -10 and pxcor < -2 and pycor > 2 and pycor <= 10 [set neighbourhood "nw"]
  if pxcor <= 10 and pxcor > 2 and pycor > 2 and pycor <= 10 [set neighbourhood "ne"]
  if pxcor >= -2 and pxcor <= 2 and pycor >= -2 and pycor <= 2 [set neighbourhood "c"]
  if pxcor >= -10 and pxcor < -2 and pycor < -2 and pycor >= -10 [set neighbourhood "sw"]
  if pxcor >= -10 and pxcor <= -3 and pycor > -4 and pycor < 4 [set neighbourhood "w"]
  if pxcor <= 10 and pxcor > 2 and pycor < -2 and pycor >= -10 [set neighbourhood "se"]
  if pxcor <= 10 and pxcor >= 3 and pycor > -4 and pycor < 4 [set neighbourhood "e"]
  if pxcor >= -3 and pxcor <= 3 and pycor < -2 and pycor >= -10 [set neighbourhood "s"]
  if pxcor >= -3 and pxcor <= 3 and pycor > 2 and pycor <= 10 [
    set neighbourhood "n"
    set cbd? true
    set plabel "CBD"
   ]
end

;; TODO: how should this code relate to slum-threshold?
to set-last-renovated
  if condition <= 0.15 [set last-renovated 120 + random 120]
  if condition > 0.15 and condition <= 0.25 [set last-renovated 60 + random 120]
  if condition > 0.25 and condition <= 0.5 [set last-renovated 60 + random 60]
  if condition > 0.5 and condition <= 0.75 [set last-renovated 24 + random 60]
  if condition > 0.75 [set last-renovated random 48]
end


;; ===== GO! =====

to go
  ;DEBUG type (word "[" ticks "]")
  ifelse ticks = 0 or ticks mod 10 != 0 [type "."] [print "."]
  ifelse ticks mod disp-freq = 0 [set disp? true] [set disp? false]

  ;; optimization: compute and cache occupancys of neighbourhoods, to avoid their recomputation in next code block
  let current-occupancy-percentages table:from-list map [place -> list place (occupancy place "go")] table:keys neighbourhoods-table  ; (needs NetLogo 6.0.2+)
  ;DEBUG print current-occupancy-percentages

  ask city [
    ifelse have-demand-premium [
      ;let occupancy-of-nhood (occupancy neighbourhood "go")
      let occupancy-of-nhood table:get current-occupancy-percentages neighbourhood
      ifelse occupancy-of-nhood >= 0.8   ;; NON DEVE ESSERE SOLO NEIGHBOURHOOD ! [should not be just neighbourhood]
        [set premium 1 + renovation-premium]
      [ifelse occupancy-of-nhood >= 0.5
        [set premium 1 + random-float renovation-premium]
        [set premium 1 + random-float (renovation-premium * 0.8)]
      ]
    ] [set premium 1 + renovation-premium]
    if enable-economy? [
      if gaps = "mean"    [set-gaps-mean]
      if gaps = "max"     [set-gaps-max]
      if gaps = "unified" [set-gaps-unified]
      if gaps = "new"     [set-gaps-new]
      if gaps = "lnd"     [set-gaps-lnd]
      decay
      update-price
    ]
    update-emptiness
    ;update-centreness
  ]

  ;; check-new-allure
  if ticks > 0 and (ticks = 24 or ticks mod 60 = 0) [
    ;DEBUG print (word "go: tick " ticks " starting allure update")
    set-city-allure
    ;ask city [set-allure-for-high-occupancy-districts]
    foreach districts [? -> set-allure-for-high-occupancy-district ? (table:get current-occupancy-percentages ?)]
    ;DEBUG print (word "go: tick " ticks " -- done allure update")
  ]
  ;if ticks < 1000 and any? patches with [allure = 0] [check-new-allure]
  ;if ticks mod 24 = 0 and table:length allured-districts > 0 [check-existing-allure]

  ;; +++++++ ADD SOMETHING HERE TO MONITOR GENTRIFICATION / SEGREGATION (not related to culture) +++++++++++++++

  update-links
  update-dissonance
  update-propensity
  if mixing? [interact]
  ask citizens [
    set months-here months-here + 1
    if decide-moving [seek-place]
    ; set size 0.01 * months-here
    ; if size >= 1.4 [set size 1.4]
  ]

  if ticks > 0 and ticks mod 6 = 0 [  ;; every six months
    if enable-economy? [do-business]
    if immigration? [immigrate]
    if emigration?  [emigrate]
    if social-housing? [
      if (any? city with [social? and not any? citizens-here]) and (table:length housing-waiting-list > 0) [assign-social-housing]
      if any? city with [social?] [check-social-residents]
    ]
    if (regeneration? = "regular") and (random-float 1 > intervention-threshold) [regenerate]
    if (regeneration? = "EU-slums") [regenerate]
  ]

  ask city [colour-patches]
  ;cluster-cultures
  ;update-vacancy-rates
  set av-income mean [income] of citizens
  set sd-income standard-deviation [income] of citizens
  set sd-income+ standard-deviation [income] of citizens with [income >= av-income]
  set sd-income- standard-deviation [income] of citizens with [income <= av-income]
  set occupation-rate count city with [any? citizens-here] / count city
  check-prices

  if record? [vid:record-view]
  tick
  if paper-stuff? and (ticks = 100 or ticks = 200 or ticks = 400) [
    update-reports
    save-screenshots
  ]
  if paper-stuff? and (ticks = end-tick - 1) [update-reports]  ; don't save screenshots at end-tick - 1

  if ticks = end-tick [
    if profile? [
      profiler:stop
      print profiler:report
      profiler:reset
    ]

    if paper-stuff? [save-screenshots]
    if write-csv? or paper-stuff? [save-data]
    if write-csv? or paper-stuff? [save-price-deltas]
    report-population
    report-religion
    output-print word "Of people, final citizens: " (count citizens)
    ;if paper-stuff? [export-plot "Mean prices by neighbourhood" (word save-directory kapital)]
    if record? [vid:save-recording "~/gentaxelling.mov"]
    print "Done"
    stop
  ]
end


;; FIXME moved this fragement of code from go() into this aux method, because anonymous function change of NetLogo 6 couldn't get it to work otherwise
to set-allure-for-high-occupancy-district [d occupancy-of-d]
  ;DEBUG show "start set-allure-for-high-occupancy-district"
  ;if (occupancy d "safhod" > 0.3) [
  if (occupancy-of-d > 0.3) [
    ;DEBUG print (word "go: set allure for " self " because occupancy " (occupancy self))
    let district-as-patchset one-of city with [neighbourhood = d]
    ask district-as-patchset [set-allure d]
    ;DEBUG print (word "go: set allure for " self " -- done")
  ]
  ;DEBUG show "done set-allure-for-high-occupancy-district"
end


;; ===== SOCIAL HOUSING =====

to enter-housing-list [agent place]
  ifelse social-housing? [ ; social housing exists: enter the waiting list
    ;; set housing-waiting-list lput (list(agent)(place)) housing-waiting-list
    table:put housing-waiting-list [who] of agent place
    ask agent [
      remove-turtle-from-occupancy-table
      set breed people
      set hidden? true
    ]
  ]
  [ ; else no social housing => leave city
    do-emigrate
  ]
end

;; time to leave social housing?
to check-social-residents
  ask citizens-on city with [social?] [
    if months-here >= max-time-in-social-housing * 12 [seek-place]
  ]
end

to assign-social-housing
  repeat count city with [social? and not any? citizens-here] [
    let everybody []
    foreach table:keys housing-waiting-list [ ? -> set everybody lput turtle ? everybody ]
    let candidates sublist everybody 0 4
    let housedperson min-one-of turtle-set candidates [income]
    let locality table:get housing-waiting-list [who] of housedperson
    table:remove housing-waiting-list [who] of housedperson
    ifelse locality != "" and any? city with [social? and not any? citizens-here and neighbourhood = locality]
    [move-to-social-housing housedperson locality]
    [move-to-social-housing housedperson ""]
  ]
end

to move-to-social-housing [agent place]
  ask agent [
      set breed citizens
      set months-here 0
      set hidden? false
      ifelse place = ""
      [move-to-place-and-spread-out-turtles one-of city with [social? and not any? citizens-here]]
      [move-to-place-and-spread-out-turtles one-of city with [social? and not any? citizens-here and neighbourhood = place]]
  ]
end


;; ===== "BUSINESS" =====

to check-prices
  foreach districts [ ? ->
    ifelse member? ? declining
    [if median [price] of city with [neighbourhood = ?] >= 0.25 [set declining remove ? declining]]
    [if median [price] of city with [neighbourhood = ?] < 0.25 [set declining fput ? declining]]
    ]
end

;; Renovation happens twice a year according to available capital.
;; FIXME: where exactly does this code do "twice a year"?
to do-business
  let howmany (Kapital * count city) / 2
  let goodgap city with [price-gap >= (price * profit-threshold) and not social? and condition <= 0.8];
  ; let goodgap city with [price-gap >= profit-threshold]; and condition <= 0.75];
  if count goodgap < howmany [set howmany count goodgap]
  let torenovate max-n-of howmany goodgap [price-gap]
  ask torenovate [renovate]
end

to renovate
  set price price + price-gap
  ;if price >= 1 [set price 0.98] ;FIXME-price
  set condition 0.95
  set last-renovated ticks
end

to decay
   let depr monthly-decay
   if is-owned? [set depr depr / 2]   ; slower depreciation if owned
   ;ifelse (is-slum? and count citizens-here >= slum-population)
   ;  [set depr depr * 2]   ; faster depreciation if slum
   ;  [
       if (count citizens-here > non-slum-population) [set depr depr * 1.5]  ; faster depreciation if above normal population density
   ;]

   let time ticks - last-renovated
   if time < 48 [set depr 0]
   if time >= 48 and time <= 60 [set depr depr / 2]
   if time >= 120 and time <= 240 [set depr depr * 2]
   if not any? citizens-here [set depr depr * 1.20]  ; faster depreciation if empty
   ifelse condition - depr <= 0
      [set condition 0]
      [set condition condition - depr]

   if social? and condition <= intervention-threshold [set condition 0.55]  ; renovation of social housing by govt
   if cbd-price-control? and condition <= intervention-threshold and cbd? and random-float 1 < 0.2 [  ; investment in CBD by govt
     set condition 0.4 + random-float 0.35
     ;if price < 0.5 [set price (max list 0.75 condition)]  ; price control
   ]
end

to update-price
  ifelse social? [ ;; social housing
    set price (mean [price] of city) / 2]
  [ ;; not social housing
    let depr yearly-depreciation / 12   ; price naturally "decays" 2% per year
    let time ticks - last-renovated
    if time <= 24 [set depr 0]
    if time > 24 and time <= 60 [set depr depr / 2]
    if time >= 120 and time <= 240 [set depr depr * 2]
    if months-empty > tolerable-vacancy [set depr depr * 2]
    if is-slum? [set depr depr * 2]  ; prices fall faster if above normal population density
    ifelse price - (price * depr) <= 0 [
      ifelse real-data? [set price 10] [set price 0.01]]
      [set price price - (price * depr)]

    if cbd-price-control? and cbd? [  ; CBD has minimum price set by govt
      ;if price < 0.5 [set price (max list 0.75 condition)]  ; price control FIXME-price
      if price < 0.5 * 10000 [set price (max list (0.75 * 10000) (condition * 10000))]  ; price control FIXME-price
    ]
  ]
end

to update-emptiness
  ifelse count citizens-here = 0
  [set months-empty months-empty + 1]
  [set months-empty 0]
end


;; ===== INDIVIDUALS' CHANGES =====

;; immigration happens twice a year
;
; assume no migration of Syrians/refugees before refugee-start-tick
to immigrate
  ;; non-refugees first
  let howmany 1 + ((count citizens * immigration-rate) / 2)
  ;; no refugees if during refugee immigration period, because during this period refugees immigration is handled below
  ;; however, include (possible) refugees as port of the regular immigration outside of (actually, after) the dedicated refugee immigration period
  ifelse ticks > refugee-start-tick and ticks >= refugee-end-tick [
    ask n-of howmany people [do-immigrate]  ; immigrate some (any) people
  ][
    ask n-of howmany people with [item 0 culture != 1] [do-immigrate]  ; immigrate some non-refugees
  ]

  ;; then, refugees
  if ticks >= refugee-start-tick and ticks < refugee-end-tick [
    let howmanyR 1 + ((count citizens * refugee-immigration-rate) / 2)
    let availableRefugeePeople count people with [item 0 culture = 1]
    if (availableRefugeePeople < howmanyR) [
      output-show (word "Only " availableRefugeePeople " people available as refugees; wanted " howmanyR)
      set howmanyR availableRefugeePeople
    ]
    ask n-of howmanyR people with [item 0 culture = 1] [do-immigrate-refugee]  ; immigrate some refugees
  ]
end

to do-immigrate
  ;render-human-after-time-zero
  ;if table:has-key? housing-waiting-list who [table:remove housing-waiting-list who]
  ;set breed citizens
  ;set hidden? false
  ;seek-place

  let myprice income
  if enable-economy? [set myprice (income * credit-mine)]
  ;if any? city with [price <= myprice and not any? citizens-here and not social?]
  ifelse any? city with [price <= myprice and count citizens-here < non-slum-population and not social?] [
    let whoami who
;      ; set income random-float 1
    set breed citizens
    set hidden? false
    seek-place
    ifelse table:has-key? housing-waiting-list whoami   ; if the person is from the housing list we take him away
      [table:remove housing-waiting-list whoami]
      [render-human-after-time-zero]                    ; if not he becomes a citizen
;        set birthday ticks
;        create-culture
;        colour-agent
;        reset-mobility-propensity
;      ]
    set immigrated-count immigrated-count + 1
    if item 0 culture = 1 [set refugee-immigrated-count refugee-immigrated-count + 1]
  ] [
    ;DEBUG output-show "Couldn't find a place to immigrate"
  ]
end

to do-immigrate-refugee
  let myprice income
  if enable-economy? [set myprice (income * credit-mine)]
  ifelse any? city with [(price <= myprice or is-slum?) and count citizens-here < slum-population and not social?] [
    let whoami who
    set breed citizens
    set hidden? false
    seek-place
    ifelse table:has-key? housing-waiting-list whoami   ; if the person is from the housing list we take him away
      [table:remove housing-waiting-list whoami]
      [render-human-after-time-zero]                    ; if not he becomes a citizen
    set immigrated-count immigrated-count + 1
    if item 0 culture = 1 [set refugee-immigrated-count refugee-immigrated-count + 1]
  ] [
    ;DEBUG output-show "[Refugee] Couldn't find a place to immigrate"
  ]
end

;; emigration happens twice a year
to emigrate
  ;; non-refugees first
  let howmany 1 + ((count citizens with [item 0 culture != 1] * emigration-rate) / 2)
  ask n-of howmany citizens with [item 0 culture != 1] [do-emigrate]

  ;; then, refugees
  if ticks > refugee-start-tick [ ;and ticks <= refugee-end-tick [
    let howmanyR 1 + ((count citizens with [item 0 culture = 1] * refugee-emigration-rate) / 2)
    ask n-of howmanyR citizens with [item 0 culture = 1] [do-emigrate]
  ]
end

to do-emigrate
  leave-city
  set emigrated-count emigrated-count + 1
  if item 0 culture = 1 [set refugee-emigrated-count refugee-emigrated-count + 1]
end


;; shall I try to move somewhere else?
to-report decide-moving
  let param income * credit-mine
  ifelse (item 0 culture = 1 or item 0 culture = 4) [ ;; refugee or poor
    if (([price] of patch-here > param and not is-slum?) and ; move if I can't afford the rent (for a non-slum place)
          [social?] of patch-here = false and owner? = false) or
        (random-float 1 < 0.5 * mobility-propensity) [  ; ...or if I'm feeling like it
      set place-changes place-changes + 1
      report true
    ]
  ] [ ;; not refugee or poor
    let my-mobility-propensity mobility-propensity
    if (owner? = true) [set my-mobility-propensity (mobility-propensity * 0.5)]  ; owners less likely to want to move
    if ([price] of patch-here > param and [social?] of patch-here = false and owner? = false) or  ; move if I can't afford the rent...
        (random-float 1 < my-mobility-propensity) [  ; ...or if I'm feeling like it
      set place-changes place-changes + 1
      report true
    ]
  ]
  report false
end

to leave-city
  remove-turtle-from-occupancy-table
  ask my-links [die]
  set breed people
  set hidden? true
end


to update-links
 ask links [    ;; First we check existing links. If the agents are still neighbours we reinforce the relationship, if not we weaken it.
    let stillclose false
    ask one-of both-ends [if neighbourhood = [neighbourhood] of other-end or distance other-end <= 2 [
        set stillclose true] ]
    ifelse stillclose
    [if time-lo < 12 [set time-lo time-lo + 1]]
    [
      set time-lo time-lo - 1
      if time-lo <= 0 [die]
    ]
  ]
  ask citizens [  ;; Then we create new links for neighbours that still don't have one (i.e. new neighbours)
    let myneighbours other citizens with [distance myself <= 2 and link-neighbor? myself = false]
    let goodneighbours myneighbours with [(similarity self myself / traits) >= similarity-for-friendship]
    ask goodneighbours [
      create-link-with myself [
        set time-lo 1
        set hidden? true
      ]
    ]
  ]
end

to update-dissonance
  ask citizens [
    ifelse [condition] of patch-here <= slum-threshold or count citizens-here > non-slum-population or mean [condition] of other city in-radius 2 <= slum-threshold  ; slightly faster than old line below
    ;ifelse [condition] of patch-here <= slum-threshold or mean [condition] of other city in-radius 2 <= slum-threshold or count citizens-here > non-slum-population
      [set time-in-a-slum time-in-a-slum + 1]
      [set time-in-a-slum 0]
    if push? [
      let citizens-on-neighbors citizens-on neighbors
      ;if count citizens-on-neighbors > 0 [
      if any? citizens-on-neighbors [
        let alltraits count citizens-on-neighbors * traits
        let simil 0
        ;; optimize by stopping the ask as soon as simil/alltraits > s-f-d
        ask citizens-on-neighbors [set simil simil + similarity self myself]
        ifelse (simil / alltraits) <= similarity-for-dissonance
          [set dissonance dissonance + 1]
          [set dissonance 0]
      ]
    ]
  ]
end

to update-propensity
  ask citizens [
    let myprice income * credit-mine
    if time-in-a-slum = 0 and dissonance <= tolerable-dissonance [reset-mobility-propensity]
    if ((time-in-a-slum > 12) and (myprice > ([price] of patch-here * 1.20)))   ; If I can afford to, I leave the ghetto
      ;or (income >= ((median [condition] of neighbors) * 1.50))                ; This reflects the preference of middle class people towards status over convenience
      or (median [condition] of neighbors <= slum-threshold)
      [set mobility-propensity mobility-propensity * 1.50]
    if (dissonance > tolerable-dissonance) [
      set mobility-propensity mobility-propensity * 1.50
      if random-float 1 < 0.05 [mutate]
      ]
    if mobility-propensity > 1 [set mobility-propensity 1]
  ]
end

;; The idea here is that prolonged co-location leads to cultural mixing.
;; We need each household to keep track of how long they have been neighbours with each other
to interact
   ask links with [time-lo > 6] [
    let a item 0 sort both-ends
    let b item 1 sort both-ends
    if similarity a b < traits [
      let whichone 1 + random (traits - 1)  ; trait 0 (ethnicity) never changes
      if whichone = 1 [  ; religion unlikely to change
        if random-float 1 > 0.1 [
          set whichone 2 + random (traits - 2)  ; pick another trait
        ]]
      if item whichone [culture] of a != item whichone [culture] of b [
        ifelse random-float 1 <= 0.5
          [ask b [set culture replace-item whichone culture item whichone [culture] of a]]
          [ask a [set culture replace-item whichone culture item whichone [culture] of b]]
      ]
    ]
  ]
end

;to mutate-off
;  ask one-of citizens [
;    set culture replace-item (random traits) culture (random values)
;  ]
;end

to mutate
  let where [neighbourhood] of patch-here
  let trait (1 + random (traits - 1))   ; ethnicity doesn't change
  if trait = 1 [  ; religion unlikely to change
    if random-float 1 > 0.1 [
      set trait 2 + random (traits - 2)  ; pick another trait
    ]]
  let most one-of modes [item trait culture] of citizens-on city with [neighbourhood = where]
  set culture replace-item trait culture most
end


;; ===== INTERVENTIONS =====

to refugee-start-tick-now
  set refugee-start-tick ticks
end

to build-social-housing [howmany]
  if social-housing? [
    let sofar 0
    let zerop min [price] of city
    let zeroc min [condition] of city
    let avg mean [price] of city with [count citizens-here > 0]
    let firstsocial nobody
    let worst city with [not any? citizens-here and price <= zerop and condition <= zeroc]
    ifelse any? worst
    [set firstsocial min-one-of worst [price-gap]]
    [set firstsocial max-one-of city [months-empty]]
    ask firstsocial [
      set social? true
      set price avg / 2
      set condition 0.95
      set sofar sofar + 1
      while [sofar < howmany] [
        ask one-of city in-radius 4 with [not social?] [
          if not any? citizens-here [
            set social? true
            set price avg / 2
            set condition 0.95
            set sofar sofar + 1
          ]
        ]
      ]
    ]
  ]
end

;; Regeneration is intended in the anglo-saxon, "small state but lets-be-compassionate-shall-we" way.
;; Money is put in the areas least desirable to investors (= those with the most narrow price-gap)
;; that are also empty and in run-down condition. These areas are brought to the maximum condition
;; and to the mean price of the city. The idea is to check whether this practice can trigger further private investment.
;;
;; Also, in case of targeted EU regeneration, focus is on (more frequent) small-scale improvement of worst areas.
to regenerate
  let zerop min [price] of city
  let zeroc min [condition] of city
  let avg mean [price] of city with [count citizens-here > 0]
  let worst city with [not any? citizens-here and price = zerop and condition = zeroc]
  if any? worst [
    ask min-one-of worst [price-gap] [
      ifelse regeneration? = "regular" [ ;; regular "anglo-saxon" intervention
        set price avg
        set condition 0.95
        ask neighbors with [not social?] [
          set price avg
          set condition 0.95
        ]]
      [if regeneration? = "EU-slums" [ ;; EU intervention focused on slums
        if zeroc <= max list slum-threshold intervention-threshold [
          set condition max list 0.5 zeroc
          set price avg
        ]
      ]]
    ]
  ]
end


; ===== RESIDENTIAL LOCATION PROCESS =====

;; When seeking a spot we consider vacant affordable places close to the origin (CBD) and with a pleasant cultural mix.
;; This is in line with Jackson et al. 2008, although they have a Schelling-like ethnic (not cultural) mix.
;; In this version agents only evaluate the CULTURAL ALLURE of a district, not the STATUS.
;; If we are to finegrain the model we could also include status in the decision process.

to seek-place
  let howmuch income * credit-mine
  ifelse pull? and table:length allured-districts > 0
  [ ;; alluring districts exist
    let where set-place
    ifelse where != "" [
      ifelse strong-neighbourhood? [relocate-to where howmuch] [weak-relocate-to where howmuch]
    ] [ ;; nowhere particular to relocate to, so go somewhere
      relocate howmuch
    ]
  ] [ ;; else no alluring districts
    relocate howmuch
  ]
end

to-report set-place
  ;let best_ftr 1
  let best_ftr traits * (similarity-threshold * 2)
  let bestdistrict ""
  foreach table:keys allured-districts [ ? ->
    let this_similarity similarity self one-of city with [neighbourhood = ?]  ; FIXME this call to similarity has problem because its two inputs can be different lengths?
    if this_similarity >= best_ftr [
      set best_ftr this_similarity
      set bestdistrict ?
      ]
    ]
  report bestdistrict
end

;; only point in the code where a turtle is assigned to a patch are the next pair of methods
to move-to-place-and-spread-out-turtles-and-no-update-occupancy-table [place]
  move-to place
  spread-out-other-turtles
end

to move-to-place-and-spread-out-turtles [place]
  if (pxcor != 0 and pycor != 0) [  ; (0,0) means no patch
    remove-turtle-from-occupancy-table
  ]
  move-to place
  add-turtle-to-occupancy-table
  spread-out-other-turtles
end

to spread-out-other-turtles
  if (any? citizens-here) [
    ask other turtles-here
      [ ;; spread out all other turtles
        setxy pxcor pycor  ; move to middle of patch
        set heading random 360  ; pick a direction
        fd random-float 0.3 + 0.05   ; move a bit
      ]
  ]
end

to relocate [will]  ;; will is dervied from how much willing to pay (i.e., as rent)
  set months-here 0
  let min-condition 0.05
  let acceptable-number-of-people-in-one-place non-slum-population
  if item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
    set acceptable-number-of-people-in-one-place slum-population
    set min-condition 0
  ]
  set acceptable-number-of-people-in-one-place (acceptable-number-of-people-in-one-place - 1)  ; this many people, besides me

  ;let baseline city with [(count citizens-here <= acceptable-number-of-people-in-one-place) and (condition > min-condition) and (social? = false)]
  let baseline city with [(condition > min-condition) and (social? = false) and (count citizens-here <= acceptable-number-of-people-in-one-place)]
  set baseline baseline with [(not cbd?) or (cbd? and count citizens-here < cbd-population)]  ; impose cbd population limit
  if owner? [set baseline baseline with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else
  ifelse item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
    if item 0 culture = 1 [set baseline baseline with [not cbd?]]  ; refugee CBD exclusion
    if enable-economy? [set baseline baseline with [price <= will or is-slum?]]  ; refugees move somewhere they can afford or to a slum
  ] [ ;; non-refugee
    if enable-economy? [set baseline baseline with [price <= will]]
  ]

  ifelse any? baseline [
;    let testbed n-of 6 city
    let number-of-places-to-benchmark-with min (list 20 (count city / 10))  ; at most look at 20 places to benchmark conditions
    let condi mean [condition] of n-of number-of-places-to-benchmark-with city
    ;let testbed n-of number-of-places-to-benchmark-with city
    ;;let testbed n-of (count city / 10) city
    ;let condi mean [condition] of testbed
;    let secondbest baseline with [(price <= [income] of myself) and (count citizens-here = 0) and (condition >= (mean [condition] of testbed - (mean [condition] of testbed * 0.15)))]  ; if we can't find a place we like then we move to one we can afford
    let secondbest baseline with [condition >= (condi - (condi * 0.15))]  ; if we can't find a place we like then we move to one we can afford
;    let secondbest baseline with [(count citizens-here <= acceptable-number-of-people-in-one-place) and (condition >= (condi - (condi * 0.15)))]  ; if we can't find a place we like then we move to one we can afford
;    if enable-economy? [
;      ifelse item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
;        set secondbest secondbest with [price <= will or is-slum?]
;      ] [ ;; non-refugee
;        set secondbest secondbest with [price <= will]
;      ]
;    ]
    ifelse any? secondbest
    [move-to-place-and-spread-out-turtles min-one-of secondbest [dist]]
    [move-to-place-and-spread-out-turtles min-one-of baseline [dist]]
  ]
  [enter-housing-list self ""]  ; if no place exists we apply for social housing
end

to relocate-to [place will]  ;; will is dervied from how much willing to pay
  set months-here 0
  let min-condition 0.05
  let acceptable-number-of-people-in-one-place non-slum-population
  if item 0 culture = 1 or item 0 culture = 4 [  ;; refugee or poor
    set acceptable-number-of-people-in-one-place slum-population
    set min-condition 0
  ]
  set acceptable-number-of-people-in-one-place (acceptable-number-of-people-in-one-place - 1)  ; this many people, besides me

  let baseline city with [(count citizens-here <= acceptable-number-of-people-in-one-place) and (condition > min-condition) and (social? = false)] ;Add to prevent people from moving to decrepit loc:; and (condition > 0)
  set baseline baseline with [(not cbd?) or (cbd? and count citizens-here < cbd-population)]  ; impose CBD population limit
  if owner? [set baseline baseline with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else
  ifelse item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
    if enable-economy? [set baseline baseline with [(price <= will) or is-slum?]]  ; refugees need somewhere they can afford, or a slum
    if item 0 culture = 1 [set baseline baseline with [(not cbd?)]]  ; refugee CBD exclusion
  ] [ ;; non-refugee
    if enable-economy? [set baseline baseline with [(price <= will)]]
  ]

  ifelse any? baseline [
;    let testbed n-of 6 city
    let testbed n-of (count city / 10) city
    let condi mean [condition] of testbed
    let ideal baseline with [(neighbourhood = place) and (condition >= (condi - (condi * 0.15)))]
    ;let ideal baseline with [(neighbourhood = place) and (condition > 0.25)]
    ifelse any? ideal
      [move-to-place-and-spread-out-turtles min-one-of ideal [local-dist]]
      [ ;; no ideal, try second best
        let apex city with [neighbourhood = place]
        if kind = "policentric" [set apex apex with [centre? = true]]
        let acceptable baseline with [condition >= (condi - (condi / 2))]   ; TODO acceptable condition should vary across agents [i.e. each agent individual]
        let secondbest acceptable with [neighbourhood = place]               ; TODO if nothing is available in place look at the edges
        ifelse any? secondbest
        [move-to-place-and-spread-out-turtles min-one-of secondbest [local-dist]]
        [ ;; no second best, try somewhere acceptable
          ifelse any? acceptable
          [move-to-place-and-spread-out-turtles min-one-of acceptable [distance one-of apex]]             ; If we can't find anything in the desired area we try somewhere close
          [ ; nowhere acceptable, just pick from baseline
            move-to-place-and-spread-out-turtles min-one-of baseline [dist]]
        ]
     ]
  ]
  [enter-housing-list self place]  ; if no place exists we apply for social housing
end

;; this version is used if allure etc is switched off
to weak-relocate-to [place will]
  let acceptable-number-of-people-in-one-place non-slum-population
  if item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
    set acceptable-number-of-people-in-one-place slum-population
  ]
  set acceptable-number-of-people-in-one-place (acceptable-number-of-people-in-one-place - 1)  ; this many people, besides me

  let ideal city with [(count citizens-here <= acceptable-number-of-people-in-one-place) and
    (social? = false) and (neighbourhood = place) and
    (condition >= (mean [condition] of city - (mean [condition] of city * 0.15)))]
  set ideal ideal with [(not cbd?) or (cbd? and count citizens-here < cbd-population)]  ; impose CBD population limit
  if owner? [set ideal ideal with [not is-owned?]]  ; if I'm an owner, I can't move to a place which is already owned by someone else

  ifelse item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
    if enable-economy? [set ideal ideal with [(price <= will) or is-slum?]]  ; refugees need somewhere they can afford, or a slum
    if item 0 culture = 1 [set ideal ideal with [(not cbd?)]]  ; refugee CBD exclusion
  ] [ ;; non-refugee
    if enable-economy? [set ideal ideal with [(price <= will)]]
  ]

  ifelse any? ideal [
    move-to-place-and-spread-out-turtles min-one-of ideal [dist]
  ] [ ;; no ideal place, try second best place
;    let testbed n-of 6 city
    let testbed n-of (count city / 10) city
    let secondbest city with [(count citizens-here <= acceptable-number-of-people-in-one-place) and (condition >= (mean [condition] of testbed - (mean [condition] of testbed * 0.15)))]  ;; if we can't find a place we like then we move to one we can afford
    if enable-economy? [
      ifelse item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
        set secondbest secondbest with [(price <= will) or is-slum?]
      ] [ ;; non-refugee
        set secondbest secondbest with [(price <= will)]
      ]
    ]

    ifelse any? secondbest [
      move-to-place-and-spread-out-turtles min-one-of secondbest [dist]
    ] [ ;; no second best place, try third best place
      let thirdbest city with [(count citizens-here <= acceptable-number-of-people-in-one-place)] ;; Uncomment the following to prevent people from moving in decrepit locations ;and (condition > 0)
      if enable-economy? [
        ifelse item 0 culture = 1 or item 0 culture = 4 [ ;; refugee or poor
          set thirdbest thirdbest with [(price <= will) or is-slum?]
        ] [ ;; non-refugee
          set thirdbest thirdbest with [(price <= will)]
        ]
      ]
      ;ifelse any? thirdbest [move-to min-one-of thirdbest [dist]] [enter-housing-list self place]  ;; if no place exists we leave the city.
      ifelse any? thirdbest [move-to-place-and-spread-out-turtles min-one-of thirdbest [dist]] [do-emigrate]  ;; if no place exists we leave the city.
    ]
  ]
end


; ===== PRICE GAPS =====

to set-gaps-lnd-off
  let sample max [price] of city in-radius 3
  let areaprice median [price] of city with [neighbourhood = [neighbourhood] of myself] * (1 + renovation-premium)
  let whichprice sample
  if areaprice > sample [set whichprice areaprice]
  ifelse whichprice > price
  [set price-gap (sample - price)]
  [set price-gap 0]
end

to set-gaps-lnd
  let whichprice 0
  let neigh-price 0
  let areaprice 0
  let sample city in-radius 2
  let sample-with-people sample with [count citizens-here > 0]
  ifelse any? sample-with-people [
    set areaprice mean [price] of sample-with-people
  ][
    set areaprice (mean [price] of sample) * 0.65
  ]
  set whichprice areaprice * (1 + renovation-premium)
  ifelse whichprice > price [
    let citizens-here-to-consider citizens-here with [owner? = false]
    ifelse any? citizens-here-to-consider with [(income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
    ;ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
;      [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
      [set price-gap (whichprice - (price + compute-resident-eviction-cost citizens-here-to-consider) )]  ; ...which will reduce prospective profit
    ;[set price-gap (whichprice - price)]
    [set price-gap (whichprice - price)]
  ]
  [set price-gap 0]
end

to-report compute-resident-eviction-cost [citizens-to-consider]
  ;let local-non-refugee-removal-cost resident-removal-cost * (count city with [item 0 culture != 1 and income * credit-mine < price])
  ;let local-refugee-removal-cost refugee-removal-cost * (count city with [item 0 culture = 1 and income * credit-mine < price])
  let local-non-refugee-removal-cost resident-removal-cost * price * (count citizens-to-consider with [item 0 culture != 1 and income * credit-mine < price])
  let local-refugee-removal-cost refugee-removal-cost * price * (count citizens-to-consider with [item 0 culture = 1 and income * credit-mine < price])
;  output-show (local-non-refugee-removal-cost + local-refugee-removal-cost)
  report local-non-refugee-removal-cost + local-refugee-removal-cost
end

;; FIXME code not updated c.f. v0.4
to set-gaps-new   ;; Maximum of moore neighbourhood or district median
 let whichprice 0
 let neigh-price 0
 let area-price 0
 set neigh-price max [price] of neighbors
 set area-price median [price] of city with [neighbourhood = [neighbourhood] of myself] * premium
 ifelse neigh-price > area-price
 [set whichprice neigh-price]
 [set whichprice area-price]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end

;; FIXME code not updated c.f. v0.4
to set-gaps-unified  ;;
 let whichprice 0
 let localprice mean [price] of neighbors * (1 + renovation-premium)
 if (occupancy neighbors "sg") >= 0.85 [set localprice max [price] of neighbors * (1 + renovation-premium) ]
 ifelse (occupancy neighbourhood "sg2") >= 0.85
 [set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * (1 + renovation-premium)]
 [set whichprice mean [price] of city with [neighbourhood = [neighbourhood] of myself] * (1 + renovation-premium)]
 if localprice > whichprice [set whichprice localprice]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end

;; FIXME code not updated c.f. v0.4
to set-gaps-mean
 let whichprice 0
 let localprice mean [price] of neighbors * premium
 ;if count citizens-on neighbors / count neighbors >= 0.85 [set localprice max [price] of neighbors * premium ]
 ;ifelse occupancy neighbourhood >= 0.85
 ;[set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * premium]
 set whichprice mean [price] of city with [neighbourhood = [neighbourhood] of myself] * premium
 if localprice > whichprice [set whichprice localprice]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end

;; FIXME code not updated c.f. v0.4
to set-gaps-max
 set premium 1.001
 let whichprice 0
 let localprice max [price] of neighbors * premium
 ;if count citizens-on neighbors / count neighbors >= 0.85 [set localprice max [price] of neighbors * premium ]
 ;ifelse occupancy neighbourhood >= 0.85
 ;[set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * premium]
 set whichprice max [price] of city with [neighbourhood = [neighbourhood] of myself] * premium
 if localprice > whichprice [set whichprice localprice]
 ifelse whichprice > price [
   ifelse any? citizens-here with [owner? = false and (income * credit-mine) < whichprice]   ; we anticipate whether we will have to evict people...
     [set price-gap (whichprice - (price + resident-removal-cost * (count citizens-here with [income < whichprice])))]  ; ...which will reduce prospective profit
     [set price-gap (whichprice - price)]
 ]
 [set price-gap 0]
end


;; ===== ALLURE =====

to-report update-city-culture
  let newallure n-values traits [0]
  let trt 0
  while [trt < traits] [
    let thistrait one-of modes [item trt culture] of citizens
    set newallure replace-item trt newallure thistrait
    set trt trt + 1
  ]
  report newallure
end

to set-city-allure
  let city-culture update-city-culture
  let pallure []
  let num-of-citizens (count citizens)
  let trait 0
  while [trait < traits] [
    set pallure lput ifelse-value (count citizens with
      [(item trait culture = item trait city-culture)] >= (num-of-citizens * 0.3))
    [item trait city-culture] ["x"] pallure
    set trait trait + 1
    ]
  set city-allure pallure
  ;DEBUG show "done set-city-allure"
end

to-report localculture [district] ; FIXME: tidy up
  let people-here0 citizens-on city with [neighbourhood = district]
  ;print (word "localculture " district ": " (count people-here0) " people here: " people-here0)
  ;if any? people-here0 [show "yes, people here"]
  ifelse any? citizens-on city with [neighbourhood = district] [
    let people-here citizens-on city with [neighbourhood = district]  ; netlogo bug? this turtle set always is empty, even though there are people here!
    ;print (word "localculture " district ": " (count people-here) " people here: " people-here)
    ;if any? people-here0 [print (word "yes, people here: " people-here0)]
    let newallure n-values traits [0]
    let trt 0
    while [trt < traits] [
      let most-common-cultural-trait-here modes [item trt culture] of people-here0
      ifelse length most-common-cultural-trait-here > 0 [
        let thistrait one-of most-common-cultural-trait-here
        set newallure replace-item trt newallure thistrait
      ] [
        print (word "localculture " district ": none of " (count people-here) " people here with trait " trt)
      ]
      set trt trt + 1
    ]
    report newallure
   ; ask city with [neighbourhood = district] [set pculture newallure]
  ] [
    print (word "localculture " district ": no people here")
  ]
end

to set-allure [place]
  let ppl citizens-on city with [neighbourhood = place]
  ;DEBUG print (word "set-allure " place ": " (count ppl) " people here")
  let areaculture (localculture place)
  ;DEBUG print (word "set-allure " place ": " (count ppl) " people here and localculture " areaculture) ; ": " ppl)
  let pallure []
  let trait 0
  while [trait < traits] [
    set pallure lput ifelse-value
    (count ppl with [item trait culture = item trait areaculture] > count ppl * 0.3 and
      item trait areaculture != item trait city-allure and
      item trait city-allure != "x"
      )
    [item trait areaculture] ["x"] pallure
    set trait trait + 1
  ]
  let peculiar length filter [ ? -> ? != "x" ] pallure
  ;DEBUG print (word "set-allure " place ": has peculiar " peculiar)

;  set-current-plot "peculiarity"
;  let placepen place
;  if place = "n" [set placepen "cbd"]  ; match the identifier in the allure graph
;  set-current-plot-pen placepen
;  plotxy ticks peculiar

  ifelse peculiar > peculiarity-for-allure [
    ask city with [neighbourhood = place] [set allure pallure]
    if not table:has-key? allured-districts place
    [table:put allured-districts place (list (median [income] of citizens-on city with [neighbourhood = place]) (median [price] of city with [neighbourhood = place]) (occupancy place "sa") (ticks))]
  ]
  [if table:has-key? allured-districts place [
      let longevity ticks - item 3 table:get allured-districts place
      if longevity > 60 [table:remove allured-districts place]]
  ]

  ;DEBUG print (word "set-allure " place " -- done")
end


;; ===== INTERNAL REPORTING AND PHENOMENA CALCULATION =====

to-report how-many-refugees [place]
  report count citizens with [[neighbourhood] of patch-here = place and item 0 culture = 1]
end

to-report median-price-of [place]
  let district-as-patchset place
  if not is-patch-set? place [
    set district-as-patchset table:get neighbourhoods-table place
  ]
  report median [price] of district-as-patchset
end

to-report gini [group]
  let sorted-wealths sort [income] of group
  ;; algorithm from http://shlegeris.com/2016/12/29/gini
  let sum-of-absolute-differences 0
  let subsum 0
  let i 0
  let N count group
  while [i < N] [
    let x item i sorted-wealths
    set sum-of-absolute-differences (sum-of-absolute-differences + i * x - subsum)
    set subsum (subsum + x)
    set i (i + 1)
  ]
  report sum-of-absolute-differences / subsum / N
end

to report-population
  let totalpop count people
  output-print (word "Population " totalpop ": "
    (count people with [item 0 culture = 0]) " Lebanese, "
    (count people with [item 0 culture = 1]) " refugee, "
    (count people with [item 0 culture = 2]) " highcast Arab, "
    (count people with [item 0 culture = 3]) " highcast foreign, "
    (count people with [item 0 culture = 4]) " lowcast foreign")
end

to report-religion
  let totalpop count people
  output-print (word "Religion of " totalpop ": "
    (count people with [item 1 culture = 0]) " Christian, "
    (count people with [item 1 culture = 1]) " Sunni, "
    (count people with [item 1 culture = 2]) " Shi'a, "
    (count people with [item 1 culture = 3]) " Druze, "
    (count people with [item 1 culture = 4]) " other")
end

to-report is-owned?
  if any? citizens-here with [owner?] [report true]
  report false
end

;; place is a slum if condition too low or occupancy to high
to-report is-slum?
  if (condition <= slum-threshold) or (count citizens-here > non-slum-population) [report true]
  report false
end

to-report is-cbd? [place]
  if any? place with [cbd?] [report true]
  report false
end

to-report occupancy [place debug-tag]
  let district-name place
  let district-patches place
  ifelse is-patch-set? place [
    ;DEBUG print (word "setting total as nhood " place)
    set district-name [neighbourhood] of one-of place
  ] [
    ;DEBUG print (word "setting total via table " place)
    set district-patches occupancy-helper place
  ]

  let number-patches-occupied count district-patches with [count citizens-here > 0]
  let total-number-patches count district-patches
  ;DEBUG if (debug-tag != "go") [print (word debug-tag " " district-name " " number-patches-occupied " / " total-number-patches)]
  report safe-division number-patches-occupied total-number-patches
end

to-report occupancy-helper [place]
  report table:get neighbourhoods-table place  ; much faster than line below: since neighbourhoods never change we can use lookup table
  ;report city with [neighbourhood = place]
end

;; overwrite occupancys table based on current neighbourhood of citizens (needs NetLogo 6.0.2+)
to occupancy-table-update
  ;DEBUG print "occupancy-table-update"
  let occupancys-table-agents table:group-agents citizens [neighbourhood]
  ;DEBUG print occupancys-table-agents
  set occupancys-table table:from-list map [place -> list place (count table:get-or-default occupancys-table-agents place no-turtles)] table:keys neighbourhoods-table ;occupancys-table-agents
  ;DEBUG print occupancys-table
  ;DEBUG print "occupancy-table-update -- done"
end

to remove-turtle-from-occupancy-table
;  let current-population-of-old-patch table:get occupancys-table neighbourhood
;  table:put occupancys-table neighbourhood (current-population-of-old-patch - 1)
;  if (current-population-of-old-patch - 1 < 0) [
;    print (word neighbourhood " " current-population-of-old-patch "-=>" (current-population-of-old-patch - 1))
;  ]
end

to add-turtle-to-occupancy-table
;  let current-population-of-new-patch table:get occupancys-table neighbourhood
;  table:put occupancys-table neighbourhood (current-population-of-new-patch + 1)
;  if (current-population-of-new-patch + 1 > count citizens) [
;    print (word neighbourhood " " current-population-of-new-patch "+=>" (current-population-of-new-patch + 1))
;  ]
end

to-report adjacentarea [place]
  let where min-one-of city with [count city with [area = [area] of myself] >= 5] [distance one-of city with [area = place]]
  report [area] of where
end

;; if place is not a patchset, but a string (district name), return the subset of city patches which are in the district place
to-report ensure-patchset-of-city [place]
  if not is-patch-set? place [
    ;DEBUG print (word place " is not a patchset")
    report city with [area = place]  ; HACK works because areas = districts
  ]
  ;DEBUG print (word place " is already a patchset")
  report place
end


to determine-phenomenon [place]
  ifelse median [income] of citizens-on city with [neighbourhood = place] > item 0 table:get allured-districts place
  [
    table:put gentrified-districts place (list median [income] of citizens-on city with [neighbourhood = place] median [price] of city with [neighbourhood = place] occupancy place "DP")
    output-show word "Here is a gentrified neighbourhood: " place
    if write-csv? [export-view (word save-directory-pictures "GENTRIFIED_DISTRICT -" "K" Kapital "-t" ticks "-" place ".png")]
    ]
  [
    table:put downfiltered-districts place (list median [income] of citizens-on city with [neighbourhood = place] median [price] of city with [neighbourhood = place] occupancy place "DP")
    output-show word "Here is a downfiltered neighbourhood: " place
    if write-csv? [export-view (word save-directory-pictures "DOWNFILTERED_DISTRICT -" "K" Kapital "-t" ticks "-" place ".png")]
    ]
end

to determine-super-phenomenon [district case]  ;; when a place lost than regained uniformity. What's happening???
  ifelse case = 0  [   ;; in this case originally gentrification dissolved uniformity
    ifelse median [income] of citizens-on city with [neighbourhood = district] >= (item 0 table:get gentrified-districts district - 0.1) and median [price] of city with [neighbourhood = district] >= (item 1 table:get gentrified-districts district - 0.1)
    [
      output-show word "Here is a recolonised neighbourhood: " district
      if not member? district recolonisation [set recolonisation fput district recolonisation]
      ;set recolonisation recolonisation + 1
      if write-csv? [export-view (word save-directory-pictures "RECOLONISED_DISTRICT -" "K" Kapital "-t" ticks "-" district ".png")]
      ]
    [
      if not member? district degentrification [set degentrification fput district degentrification]
      output-show word "Here is a DEGENTRIFIED neighbourhood: " district
      ;set degentrification degentrification + 1
      ]
  ] [ ; here originally downfiltering dissolved uniformity
  ifelse mean [income] of citizens-on city with [neighbourhood = district] <= (item 0 table:get downfiltered-districts district + 0.1)
  [set recreation fput district recreation]
  [set regentrification fput district regentrification]
  ]
end

to-report entropy [district]
  let common 0
  let thispeople citizens-on city with [neighbourhood = district]
  let pairs (count thispeople * (count thispeople - 1)) / 2
  ask n-of (count thispeople / 2) thispeople [
    ask other thispeople [
      set common common + (similarity self myself / traits)
    ]
  ]
  report safe-division common pairs
end

to-report entropy-old [district]
  let common 0
  let thispeople citizens-on city with [neighbourhood = district]
  let pairs (count thispeople * (count thispeople - 1)) / 2
  ask thispeople [
    ask other thispeople [
      set common common + (similarity self myself / traits)
    ]
  ]
  report safe-division (common / 2) pairs
end


;; ===== DISPLAY ======

to draw-city
  gis:set-drawing-color red
  ;gis:draw prices 0.2
  gis:draw divisions 0.4
end

to highlight-city
  ask city [set pcolor green]
  ask city with [count neighbors with [pcolor = black] > 0]  [set pcolor white]
end

to colour-agent
  ifelse real-data? [
    let median-income-of-citizens median [income] of citizens
    if income > median-income-of-citizens * 2 [set color green - 1]
    if income < median-income-of-citizens * 2 and income > median-income-of-citizens [set color green + 3]
    if income <= median-income-of-citizens and income >= median-income-of-citizens / 2 [set color violet + 2]
    if income < median-income-of-citizens / 2 [set color violet - 2]
  ][
    if income >= 0.80 [set color green - 1]
    if income < 0.80 and income >= 0.5 [set color green + 3]
    if income < 0.5 and income >= 0.25 [set color violet + 2]
    if income < 0.25 [set color violet - 2]
  ]
end

to shape-agent
  if item 0 culture = 0 [set shape "circle"]
  if item 0 culture = 1 [set shape "circle 2"]
  if item 0 culture = 2 [set shape "face happy"]
  if item 0 culture = 3 [set shape "face neutral"]
  if item 0 culture = 4 [set shape "face sad"]
  if owner? = true [set size ownersize]
end

to colour-patches
  if condition > 0.75 [set pcolor white - 1]
  if condition <= 0.75 and condition > 0.50 [set pcolor grey + 1]
  if condition <= 0.50 and condition > 0.25 [set pcolor grey - 1]
  if condition <= 0.25 [set pcolor black + 1]
  if condition <= slum-threshold [set pcolor blue - 3]
  if condition = 0 [set pcolor magenta - 3]
  if social? [set pcolor red]
end


;; ===== SUPPORTING =====

to-report safe-division [a b]
  if a = 0 or b = 0 [report 0]
  report a / b
end

to-report similarity [a b]
  report similarity-of ([culture] of a) (ifelse-value is-turtle? b [[culture] of b] [[allure] of b])
end

;; called a lot: how to speed it up? ideas: https://stackoverflow.com/questions/42877203/netlogo-how-to-filter-a-list-using-a-criterion-which-refers-on-a-corresponding
to-report similarity-of [ls1 ls2]
  ifelse length ls1 != length ls2 [
      output-print (word ls1 "!=" ls2)
      report 0
  ] [
    ;; the imperative is the fastest so far
    let N length ls1
    let i 0
    let mysum 0
    while [i < N] [
      if (item i ls1) = (item i ls2) [set mysum mysum + 1]
      set i i + 1
    ]
    report mysum
  ]
end

to-report add-underscores-to-string [s]
  let i -1
  while [i != false] [
    set i position " " s
    if i != false [set s replace-item i s "_"]
  ]
  report s
end

to-report closest-integer-to [x]
  report round x
end

to-report random-float-between [a b]
  if a > b [
    let tempc b
    set b a
    set a tempc
  ]
  report a + random-float (b - a)
end


;; ===== PLOTTING =====

to plot-ent [dis]
  if disp? [repeat disp-freq [plot entropy dis]]
end

to-report medianincome [place]
  ifelse any? citizens-on city with [neighbourhood = place] [
    report median [income] of citizens-on city with [neighbourhood = place]
  ][report 0]
end


;; ===== DATA OUTPUT =====

to update-reports
  set medianPrices lput median [price] of city with [condition >= slum-threshold] medianPrices
  set median-Income lput median [income] of citizens median-Income
  ;set howmanySlums lput (count city with [condition <= slum-threshold] / count city) howmanySlums
  set howmanySlums lput (count city with [is-slum?] / count city) howmanySlums  ; since is-slum? combines patch-condition and patch-population criteria, there's a case to use instead just condition <= slum-threshold here
  set howmanyWellMaintained lput (count city with [condition > 0.5] / count city) howmanyWellMaintained
  set howmanyOvercrowded lput (count city with [count citizens-here > non-slum-population] / count city) howmanyOvercrowded
  set highestMedianPrice lput add-underscores-to-string item 0 sort-by [ [?1 ?2] -> median-price-of ?1 > median-price-of ?2 ] districts highestMedianPrice  ; neighbourhood with highest price
  set mostRefugee lput add-underscores-to-string item 0 sort-by [ [?1 ?2] -> how-many-refugees ?1 > how-many-refugees ?2 ] districts mostRefugee  ; neighbourhood with most refugees
  set lebanese lput (count citizens with [item 0 culture = 0] / count citizens) lebanese
  set refugees lput (count citizens with [item 0 culture = 1] / count citizens) refugees
  set others lput (count citizens with [item 0 culture > 1] / count citizens) others
end

to save-screenshots
  ;export-view (word save-directory-pictures Kapital "-t" ticks ".png")
  ;export-view (word save-directory-pictures "" "-t" ticks ".png")
  ;print "(not saving screenshots)"
  type "(no png)"
  if ticks = end-tick [
      ;export-interface (word save-directory-pictures "gentax-" version "-interface.png")
  ]
end

to save-data
  ;export-all-plots (word save-directory "gentax-" version "-plots.csv")  ; save all the plots
  set file-name-results (word save-directory "gentax-" version ".csv")   ; save one big csv file with all the measures we want
  let run-number 1  ; NYS was 0
  if behaviorspace-run-number != 0 [set run-number behaviorspace-run-number]
  let should-write-header not (file-exists? file-name-results)
  file-open file-name-results
  if run-number = 0 or should-write-header
    [file-print (word "run#;K;gaps;HaveRefugees;rnetlogo-param-refugees;regeneration; pct-slums-100;pct-slums-200;pct-slums-400;pct-slums-600;pct-good-100;pct-good-200;pct-good-400;pct-good-600;pct-overcrowded-100;pct-overcrowded-200;pct-overcrowded-400;pct-overcrowded-600; medianPrices-100;medianPrices-200;medianPrices-400;medianPrices-600;median-Income-100;median-Income-200;median-Income-400;median-Income-600; highestMedianPrice-100;highestMedianPrice-200;highestMedianPrice-400;highestMedianPrice-600;mostRefugees-100;mostRefugees-200;mostRefugees-400;mostRefugees-600; lebanese-100;lebanese-200;lebanese-400;lebanese-600;refugees-100;refugees-200;refugees-400;refugees-600;others-100;others-200;others-400;others-600")]
  file-print (word run-number ";" Kapital ";" gaps ";" have-refugees? ";" rnetlogo-param-refugees ";" regeneration? ";" howmanySlums ";" howmanyWellMaintained ";" howmanyOvercrowded ";" medianPrices ";" median-Income ";" highestMedianPrice ";" mostRefugee ";" lebanese ";" refugees ";" others)
  file-close-all
end

to save-price-deltas
  file-open (word file-name-results "-district-prices.csv")
  foreach districts [ district ->
    let initialPrice initial-price-of-district district
    let finalPrice median-price-of district
    let deltaPricePercentage ( finalPrice - initialPrice) / initialPrice
    file-print (word (add-underscores-to-string district) ", " initialPrice ", " finalPrice ", " deltaPricePercentage)
  ]
  file-close
end
@#$#@#$#@
GRAPHICS-WINDOW
4
10
1052
1059
-1
-1
16.0
1
9
1
1
1
0
0
0
1
-32
32
-32
32
1
1
1
Months
30.0

SLIDER
1065
217
1239
250
monthly-decay
monthly-decay
0
0.1
0.0015
0.0001
1
NIL
HORIZONTAL

SLIDER
1065
252
1240
285
tolerable-vacancy
tolerable-vacancy
0
24
8.0
1
1
months
HORIZONTAL

SLIDER
1061
491
1197
524
prob-move
prob-move
0
0.01
0.002
0.0001
1
NIL
HORIZONTAL

SLIDER
1213
565
1305
598
traits
traits
2
10
7.0
1
1
NIL
HORIZONTAL

SLIDER
1309
565
1404
598
values
values
5
10
5.0
1
1
NIL
HORIZONTAL

TEXTBOX
1220
439
1281
457
CULTURE
12
0.0
1

SLIDER
1061
566
1196
599
immigration-rate
immigration-rate
0
0.2
0.02
0.0001
1
NIL
HORIZONTAL

SLIDER
1061
750
1196
783
init-gini
init-gini
0
1
0.4
0.001
1
NIL
HORIZONTAL

SLIDER
1059
456
1197
489
N-Agents
N-Agents
0
count patches
500.0
1
1
NIL
HORIZONTAL

SWITCH
1239
357
1374
390
random-income?
random-income?
1
1
-1000

SLIDER
1239
182
1374
215
Kapital
Kapital
0
0.1
0.035
0.001
1
NIL
HORIZONTAL

PLOT
1731
16
2046
231
Mean prices by neighbourhood
NIL
NIL
0.0
600.0
0.0
10000.0
true
true
"" ""
PENS
"C_1" 1.0 0 -16777216 true "" ";plot median [price] of patches with [neighbourhood = \"c\"]"
"NW_1" 1.0 0 -8275240 true "" ";plot median [price] of patches with [neighbourhood = \"nw\"]"
"NE_1" 1.0 0 -5298144 true "" ";plot median [price] of patches with [neighbourhood = \"ne\"]"
"SW_1" 1.0 0 -9276814 true "" ";plot median [price] of patches with [neighbourhood = \"sw\"]"
"SE_1" 1.0 0 -2064490 true "" ";plot median [price] of patches with [neighbourhood = \"se\"]"
"E_1" 1.0 0 -1184463 true "" ";plot median [price] of patches with [neighbourhood = \"e\"]"
"W_1" 1.0 0 -8431303 true "" ";plot median [price] of patches with [neighbourhood = \"w\"]"
"CBD" 1.0 0 -14070903 true "" ";plot median [price] of patches with [neighbourhood = \"n\"]"
"S_1" 1.0 0 -12087248 true "" ";plot median [price] of patches with [neighbourhood = \"s\"]"
"CITY" 1.0 0 -7500403 true "" "plot median [price] of city"

SWITCH
1062
531
1196
564
immigration?
immigration?
0
1
-1000

SLIDER
1213
601
1403
634
tolerable-dissonance
tolerable-dissonance
0
24
5.0
1
1
months
HORIZONTAL

SLIDER
1213
634
1403
667
similarity-for-dissonance
similarity-for-dissonance
0
0.5
0.2
0.01
1
NIL
HORIZONTAL

MONITOR
1063
798
1121
843
Pop
count citizens
0
1
11

SLIDER
1065
182
1237
215
profit-threshold
profit-threshold
0
1
0.125
0.001
1
NIL
HORIZONTAL

PLOT
2046
16
2366
231
Mean income by neighbourhood
NIL
NIL
0.0
600.0
0.0
10000.0
true
true
"" ""
PENS
"C_1" 1.0 0 -16777216 true "" ";if any? citizens-on patches with [neighbourhood = \"c\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"c\"]]"
"NW_1" 1.0 0 -8275240 true "" ";if any? citizens-on patches with [neighbourhood = \"nw\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"nw\"]]"
"NE_1" 1.0 0 -5298144 true "" ";if any? citizens-on patches with [neighbourhood = \"ne\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"ne\"]]"
"SW_1" 1.0 0 -9276814 true "" ";if any? citizens-on patches with [neighbourhood = \"sw\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"sw\"]]"
"SE_1" 1.0 0 -2064490 true "" ";if any? citizens-on patches with [neighbourhood = \"se\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"se\"]]"
"E_1" 1.0 0 -1184463 true "" ";if any? citizens-on patches with [neighbourhood = \"e\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"e\"]]"
"W_1" 1.0 0 -8431303 true "" ";if any? citizens-on patches with [neighbourhood = \"w\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"w\"]]"
"CBD" 1.0 0 -14070903 true "" ";if any? citizens-on patches with [neighbourhood = \"n\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"n\"]]"
"S_1" 1.0 0 -12087248 true "" ";if any? citizens-on patches with [neighbourhood = \"s\"]  [plot mean [income] of citizens-on patches with [neighbourhood = \"s\"]]"

SWITCH
1213
530
1304
563
push?
push?
0
1
-1000

SWITCH
1678
916
1772
949
record?
record?
1
1
-1000

SWITCH
1309
530
1404
563
strong-neighbourhood?
strong-neighbourhood?
0
1
-1000

SWITCH
1309
494
1404
527
pull?
pull?
0
1
-1000

SWITCH
1213
495
1307
528
mixing?
mixing?
0
1
-1000

SWITCH
1678
989
1771
1022
write-csv?
write-csv?
1
1
-1000

BUTTON
1562
921
1647
954
Show allure
ask city with [allure != 0][set pcolor yellow]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1562
886
1647
919
Undisplay
ask city [colour-patches]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
1217
128
1281
146
ECONOMY
12
0.0
1

TEXTBOX
1071
438
1221
456
POPULATION
12
0.0
1

TEXTBOX
1679
891
1743
910
OUTPUT
12
0.0
1

CHOOSER
1416
456
1578
501
kind
kind
"monocentric" "policentric" "no centres"
0

BUTTON
1562
954
1647
987
Show centres
ask city with [centre? = true][set pcolor blue]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
1421
440
1571
458
CITY
12
0.0
1

SLIDER
1416
504
1647
537
peculiarity-for-allure
peculiarity-for-allure
0
traits
3.0
1
1
/ 10 traits
HORIZONTAL

SLIDER
1213
670
1403
703
similarity-for-friendship
similarity-for-friendship
0
1
0.2
0.01
1
NIL
HORIZONTAL

SLIDER
1064
287
1237
320
resident-removal-cost
resident-removal-cost
0
0.25
0.05
0.001
1
NIL
HORIZONTAL

SLIDER
1239
252
1375
285
renovation-premium
renovation-premium
0
1
0.2
0.01
1
NIL
HORIZONTAL

SWITCH
1678
1024
1771
1057
paper-stuff?
paper-stuff?
0
1
-1000

MONITOR
1536
798
1593
843
Mobility
sum [place-changes] of turtles / count turtles
2
1
11

BUTTON
1562
1024
1647
1057
Regeneration
regenerate
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SWITCH
1239
217
1374
250
fixed-premium?
fixed-premium?
0
1
-1000

PLOT
1999
610
2260
760
Income Distribution
Income
Freq
0.0
10000.0
0.0
1.0
true
false
"set-histogram-num-bars 20" "histogram [income] of citizens"
PENS
"default" 1.0 1 -16777216 true "" ""

MONITOR
1729
798
1789
843
Total Gini
ifelse-value (ticks mod 12 = 0) [gini people] [0]
2
1
11

MONITOR
1475
798
1533
843
Occ. R.
occupation-rate
2
1
11

PLOT
1998
457
2259
607
Price Distribution
Price
Freq
0.0
10000.0
0.0
1.0
true
false
"set-histogram-num-bars 20" "histogram [price] of patches with [count citizens-here > 0]"
PENS
"default" 1.0 1 -16777216 true "" ""

SWITCH
1213
457
1403
490
enable-culture?
enable-culture?
0
1
-1000

MONITOR
1599
798
1661
843
% in Slums
100 * (count citizens-on patches with [condition <= slum-threshold or count citizens-here > non-slum-population] / count turtles)
4
1
11

MONITOR
1128
798
1186
843
City Gini
ifelse-value (ticks mod 12 = 0) [gini citizens] [0]
3
1
11

BUTTON
1562
989
1647
1022
Build SH
build-social-housing 4
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
1508
545
1647
578
max-time-in-social-housing
max-time-in-social-housing
0
100
50.0
1
1
NIL
HORIZONTAL

MONITOR
1586
582
1642
627
Waiting list
table:length housing-waiting-list
17
1
11

SLIDER
1065
357
1238
390
yearly-depreciation
yearly-depreciation
0
0.1
0.02
0.001
1
NIL
HORIZONTAL

BUTTON
1066
78
1178
111
RUN!
setup\nrepeat 1440 [go]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1066
42
1121
75
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
1122
42
1177
75
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1242
798
1299
843
Centre
median [income] of citizens-on patches with [dist < 4]
4
1
11

MONITOR
1298
798
1355
843
Semi
median [income] of citizens-on patches with [dist >= 4 and dist < 8]
4
1
11

MONITOR
1352
798
1409
843
Far
median [income] of citizens-on patches with [dist >= 8]
4
1
11

PLOT
2046
230
2366
430
Income: CBD vs. Centre vs. Suburb
NIL
NIL
0.0
600.0
0.0
10000.0
true
true
"" ""
PENS
"Centre" 1.0 0 -16777216 true "" ";plot median [income] of citizens-on patches with [distancexy 0 0 < 4]"
"Semi" 1.0 0 -2674135 true "" ";plot median [income] of citizens-on patches with [distancexy 0 0 >= 4 and distancexy 0 0 < 8]"
"Subs" 1.0 0 -14730904 true "" ";plot median [income] of citizens-on patches with [distancexy 0 0 >= 8]"
"CBD" 1.0 0 -10899396 true "" ";plot median [income] of citizens-on city with [cbd? = true]"

PLOT
1415
230
1730
430
Prices and Income
NIL
NIL
0.0
600.0
0.0
10000.0
true
true
"" ""
PENS
"V" 1.0 0 -16777216 true "" "plot median [price] of city"
"I" 1.0 0 -2674135 true "" "plot median [income] of citizens"

MONITOR
1588
456
1645
501
Allure
table:length allured-districts
17
1
11

PLOT
1415
16
1730
231
Peculiarity
NIL
NIL
0.0
600.0
0.0
11.0
true
true
"" ""
PENS
"nw" 1.0 0 -8990512 true "" ""
"ne" 1.0 0 -2674135 true "" ""
"sw" 1.0 0 -11053225 true "" ""
"se" 1.0 0 -1069655 true "" ""
"s" 1.0 0 -15575016 true "" ""
"CBD" 1.0 0 -14730904 true "" ""
"e" 1.0 0 -1184463 true "" ""
"w" 1.0 0 -10402772 true "" ""
"c" 1.0 0 -16777216 true "" ""

CHOOSER
1065
132
1200
177
gaps
gaps
"mean" "max" "unified" "new" "lnd"
4

SWITCH
1240
392
1373
425
have-demand-premium
have-demand-premium
0
1
-1000

SWITCH
1211
146
1375
179
enable-economy?
enable-economy?
0
1
-1000

SLIDER
1065
392
1236
425
credit
credit
0
15
3.0
1
1
NIL
HORIZONTAL

SLIDER
1213
706
1403
739
similarity-threshold
similarity-threshold
0
1
0.3
0.01
1
NIL
HORIZONTAL

SWITCH
1416
545
1506
578
social-housing?
social-housing?
1
1
-1000

OUTPUT
1062
854
1412
950
8

PLOT
1666
457
1826
607
Ethnicity
NIL
Freq
0.0
5.0
0.0
10.0
true
false
"set-histogram-num-bars 5" "histogram [item 0 culture] of citizens"
PENS
"default" 1.0 1 -16777216 true "" ""

PLOT
1826
456
1986
606
Religion
NIL
NIL
0.0
5.0
0.0
10.0
true
false
"set-histogram-num-bars 5" "histogram [item 1 culture] of citizens"
PENS
"default" 1.0 1 -16777216 true "" ""

PLOT
1666
610
1987
759
Population by ethnicity
NIL
NIL
0.0
600.0
0.0
10.0
true
false
"" ""
PENS
"Lebanese" 1.0 0 -16777216 true "" "plot count citizens with [item 0 culture = 0]"
"Refugee" 1.0 0 -2674135 true "" "plot count citizens with [item 0 culture = 1]"
"Other" 1.0 0 -13345367 true "" "plot count citizens with [item 0 culture > 1]"

SLIDER
1416
650
1538
683
slum-population
slum-population
2
10
6.0
1
1
NIL
HORIZONTAL

SLIDER
1416
685
1539
718
non-slum-population
non-slum-population
1
slum-population - 1
3.0
1
1
NIL
HORIZONTAL

SLIDER
1416
718
1539
751
cbd-population
cbd-population
1
10
1.0
1
1
NIL
HORIZONTAL

SWITCH
1239
287
1374
320
have-owners?
have-owners?
0
1
-1000

SLIDER
1239
322
1374
355
ownership
ownership
0
1
0.25
0.01
1
NIL
HORIZONTAL

MONITOR
1662
798
1726
843
% Owners
100 * (count citizens with [owner? = true] / count citizens)
3
1
11

SLIDER
1061
600
1196
633
refugee-immigration-rate
refugee-immigration-rate
0
0.3
0.15
0.0001
1
NIL
HORIZONTAL

MONITOR
1417
798
1472
843
Avg. Occ.
count citizens-on patches / count patches
4
1
11

INPUTBOX
1418
998
1492
1058
refugee-start-tick
90.0
1
0
Number

SWITCH
1061
640
1196
673
emigration?
emigration?
0
1
-1000

SLIDER
1061
676
1196
709
emigration-rate
emigration-rate
0
0.2
0.019
0.0001
1
NIL
HORIZONTAL

BUTTON
1418
962
1493
995
Refugees
refugee-start-tick-now
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
1061
710
1196
743
refugee-emigration-rate
refugee-emigration-rate
0
0.3
0.012
0.001
1
NIL
HORIZONTAL

SWITCH
1416
756
1536
789
cbd-price-control?
cbd-price-control?
0
1
-1000

SLIDER
1065
322
1239
355
refugee-removal-cost
refugee-removal-cost
0
0.25
0.01
0.001
1
NIL
HORIZONTAL

SLIDER
1416
616
1576
649
slum-threshold
slum-threshold
0
1
0.15
0.01
1
NIL
HORIZONTAL

BUTTON
1493
962
1553
995
Halt
set refugee-start-tick 99999
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
1416
580
1575
613
intervention-threshold
intervention-threshold
0
1
0.2
0.01
1
NIL
HORIZONTAL

MONITOR
1586
631
1644
676
% Slums
count patches with [is-slum?] / count patches
4
1
11

MONITOR
1283
962
1348
1007
Total Immigrated
immigrated-count
17
1
11

MONITOR
1283
1012
1348
1057
Total Emigrated
emigrated-count
17
1
11

MONITOR
1348
962
1413
1007
Refugee Immigrated
refugee-immigrated-count
17
1
11

MONITOR
1348
1012
1413
1057
Refugee Emigrated
refugee-emigrated-count
17
1
11

TEXTBOX
1072
22
1222
40
SIMULATION
11
0.0
1

INPUTBOX
1183
44
1293
104
end-tick
600.0
1
0
Number

PLOT
1731
230
2046
430
Mean condition by neighbourhood
NIL
NIL
0.0
600.0
0.0
1.0
false
true
"" ""
PENS
"C_1" 1.0 0 -16777216 true "" ";plot mean [condition] of patches with [neighbourhood = \"c\"]"
"NW_1" 1.0 0 -8275240 true "" ";plot mean [condition] of patches with [neighbourhood = \"nw\"]"
"NE_1" 1.0 0 -5298144 true "" ";plot mean [condition] of patches with [neighbourhood = \"ne\"]"
"SW_1" 1.0 0 -9276814 true "" ";plot mean [condition] of patches with [neighbourhood = \"sw\"]"
"SE_1" 1.0 0 -2064490 true "" ";plot mean [condition] of patches with [neighbourhood = \"se\"]"
"E_1" 1.0 0 -1184463 true "" ";plot mean [condition] of patches with [neighbourhood = \"e\"]"
"W_1" 1.0 0 -8431303 true "" ";plot mean [condition] of patches with [neighbourhood = \"w\"]"
"CBD" 1.0 0 -14070903 true "" ";plot mean [condition] of patches with [neighbourhood = \"n\"]"
"S_1" 1.0 0 -12087248 true "" ";plot mean [condition] of patches with [neighbourhood = \"s\"]"

CHOOSER
1549
728
1644
773
regeneration?
regeneration?
"none" "regular" "EU-slums"
0

INPUTBOX
1495
998
1554
1058
refugee-end-tick
180.0
1
0
Number

MONITOR
1186
798
1243
843
CBD
median [income] of citizens-on patches with [cbd?]
4
1
11

MONITOR
1586
678
1644
723
% Derelict
count patches with [condition = 0] / count patches
4
1
11

SWITCH
1678
952
1772
985
profile?
profile?
1
1
-1000

SWITCH
1420
924
1554
957
real-data?
real-data?
0
1
-1000

@#$#@#$#@
Agent-Based Simulation of West Asian Urban Dynamics: Impact of Refugees
JASSS
(c) 2014-2016 S. Picascia
(c) 2015-2020 N. Yorke-Smith
licensed under CC BY-NC-ND 4.0
contact: n.yorke-smith@tudelft.nl

This model is derived from S. Picascia's NetLogo model gentaxelling, which describes the basic operation: https://github.com/harrykipper/gentaxelling

This models runs with NetLogo 6.0.4.

Due to copyright reasons, the Beirut GIS data cannot be released.
It should be placed in subfolder GIS/Beirut.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="jasss" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.025"/>
      <value value="0.035"/>
      <value value="0.045"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnetlogo-param-refugees">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rnetlogo-param-regen">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="abmus" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.025"/>
      <value value="0.035"/>
      <value value="0.045"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="refugee-start-tick">
      <value value="90"/>
      <value value="1400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regeneration?">
      <value value="&quot;none&quot;"/>
      <value value="&quot;regular&quot;"/>
      <value value="&quot;EU-slums&quot;"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="pnas" repetitions="40" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1400"/>
    <metric>median [price] of patches</metric>
    <metric>median [income] of turtles</metric>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.02"/>
      <value value="0.025"/>
      <value value="0.03"/>
      <value value="0.04"/>
      <value value="0.05"/>
      <value value="0.06"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="profit-threshold">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="immigration-rate">
      <value value="0.012"/>
      <value value="0.045"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="beirut" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="600"/>
    <enumeratedValueSet variable="Kapital">
      <value value="0.015"/>
      <value value="0.025"/>
      <value value="0.035"/>
      <value value="0.045"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="refugee-start-tick">
      <value value="90"/>
      <value value="1400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="refugee-end-tick">
      <value value="180"/>
      <value value="1400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="regeneration?">
      <value value="&quot;none&quot;"/>
      <value value="&quot;regular&quot;"/>
      <value value="&quot;EU-slums&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
