# PrepareImage.py
# requires pylibtiff
from libtiff import TIFF
import numpy as np
import os

def CheckFileExtension(path: str, formats):
    fname, ext = os.path.splitext(path)
    has_extension = ext != ''
    correct_extension = ext in formats
    return correct_extension, has_extension

def IsTiff(path: str):
    return CheckFileExtension(path, ['.tif', '.tiff', '.TIF', '.TIFF'])

def ReadTiffStack(image : str):
    """
    Reads in a tiff image and returns a numpy ndarray
    image: string with image name, relative or absolute
    """
    correct_extension, has_extension = IsTiff(image)
    if not has_extension:
        image = image + '.tif'
    elif not correct_extension:
        raise Exception('Cannot read image file, ' + image + ' is not an accepted filename!')

    tif = TIFF.open(image, mode='r') # open tiff file in read mode
    # read an image in the current TIFF directory as a numpy array
    imstack = list(tif.iter_images())
    if len(imstack) == 0:
        imstack = imstack[0]
    else:
        # converting to 3d array, note that numpy will set the list as
        # dimension 0, whereas it is more consistent to use the stack as
        # dimension 2
        imstack = np.array(imstack, dtype=type(imstack[0][0,0]))
        imstack = np.moveaxis(imstack, 0, 2)
    return imstack
