
import numpy as np
import TiffUtils
import glob
from tqdm import tqdm


# This script conducts the occupation analysis
# As input, multiple paths can be provided to folders with the 
# relevant segmentated images in TIFF format. As a result,
# a comma separated file (CSV) will be created in each of the provided folders containing
# following information:
# pore space volume (void) in number in voxels
# maximum relative occupied pore space by the non-wetting phase
# minimum relative occupied pore space by the non-wetting phase
# relative occupied pores space by the fluctuations
# relative occupation of the pores space of the non-wetting phase for each timestep

paths = [
    '<path_to_raw_files>/089_raw/',
    '<path_to_raw_files>/100_raw/',
    '<path_to_raw_files>/103_raw/'
]

for base_path in paths:

    # reading all images in the folder
    list_images = glob.glob(base_path + '*.tiff')
    image = TiffUtils.ReadTiffStack(list_images[0])     # reading very first image
    volume_void = float(len(np.nonzero(image)[0]))      # all voxels > 0 are occupied by wetting or non-wetting phase
    print(f'void pixels: {volume_void}')
    min_filter = np.full(image.shape, True)             # initialization of the relative minimum fraction
    max_filter = np.full(image.shape, False)            # initialization of the relative maximum fraction
    num_images = len(list_images)
    volume_frac = np.zeros(num_images, dtype=float)     # initialization of the time series

    # Looping through all images
    for i in tqdm(range(num_images)):
        f_loc = image > 1                   # local occupation with non-wetting phase (value > 1)
        min_filter = min_filter & f_loc     # minimum fraction is given by intersection of non-wetting phase occupation
        max_filter = max_filter | f_loc     # maximum fraction is defined as union of non-wetting phase occupation
        volume_frac[i] = float(len(np.nonzero(f_loc)[0]))   # for each timestep determine the number of voxels occupied with non-wetting phase
        if (i+1) < num_images:
            image = TiffUtils.ReadTiffStack(list_images[i+1])   # reading next image in series

    diff_filter = max_filter != min_filter  # fluctuations are affecting the volume between min and max filter

    # normalize the values with the pores space volume and average over image size
    max_volume_frac = float(len(np.nonzero(max_filter)[0]))
    min_volume_frac = float(len(np.nonzero(min_filter)[0]))
    volume_void = float(len(np.nonzero(image)[0]))
    volume_frac /= volume_void
    max_volume_frac /= volume_void
    min_volume_frac /= volume_void
    diff_filter /= volume_void

    # writing to file
    with open(base_path + 'volume.csv', 'w') as file:
        file.write(f'void_volume,{volume_void}\n')
        file.write(f'max_volume_fraction,{max_volume_frac}\n')
        file.write(f'min_volume_fraction,{min_volume_frac}\n')
        file.write(f'fluctuation_fraction,{diff_filter}')
        file.write('timestep, volume fraction\n')
        for i in range(num_images):
            file.write(f'{i},{volume_frac[i]}\n')
