% This function makes a histogram of correction factor for a GTPase
% concentration that is externally supplied in an .xls(x) file (column
% vector). The function prompts for this file.
%
% You need the useful_colormaps function [1] that was included in the code
% documentation earlier as well, just for the colors in the plots.
%
% Date: 03-05-2023
%
% Bibliography
% [1] Smith, N. J., Walt, S. van der, & Firing, E. (2015). Magma, inferno,
%     plasma and viridis colormaps,
%     https://github.com/BIDS/colormap/blob/master/colormaps.py.

bin_size            = 0.1;
x_start             = 0;
x_end               = 1.6;

plot_font_name      = 'Arial';  % Name of font in plot
plot_font_size      = 8;        % Font size in plot
resol_for_print     = 300;      % Resolution for .tiff outputs
pos_plot            = [0.2 0.5 5 4]; % Position vector of the figure in inches
% [x-coordinate bottom left corner y-coordinate width height] in inches
% Modify if it does not fit your screen
plot_color          = useful_colormaps('viridis', 5); % [1]
plot_color          = plot_color(3, :);

%% Load data
[data_file_name, data_path_name] = uigetfile({'*.xlsx'; '.xls'}, 'Select the file with the correction factor data');
[~, data2, data3]   = xlsread([data_path_name data_file_name]);
y_corr              = nonzeros(cellfun(@(x) str2double(x), data2));
% correction factor data vector
[bar_h, bar_edges]  = histcounts(y_corr, x_start : bin_size : x_end);
% height and location of bars for histogram

%% Cdc42 concentration correction histogram
% Define figure
fig_conc_corr       = figure;
set(fig_conc_corr, 'Renderer', 'painters', 'Units', 'inches', 'Position', pos_plot);

% Define axis
ax_fig_conc_corr    = axes('Parent', fig_conc_corr);
set(ax_fig_conc_corr, 'Units', 'inches', ...
    'Position', [0.8 0.8 fig_conc_corr.Position(3 : 4) - [1 1.6]], ...
    'FontName', plot_font_name, 'FontSize', plot_font_size, ...
    'LabelFontSizeMultiplier', 1, 'TitleFontSizeMultiplier', 1, ...
    'NextPlot', 'add', 'LineWidth', 1, 'Box', 'on', 'color', 'none', ...
    'YScale', 'linear', 'XLim', [0 x_end], ...
    'XTick', bar_edges, 'YLim', [0 ceil(1.1 * max(bar_h))], ...
    'YTick', 0 : 2 : ceil(1.1 * max(bar_h)), ...
    'YMinorTick', 'off', 'YGrid', 'off');
xlabel(ax_fig_conc_corr, 'Correction factor');
ylabel(ax_fig_conc_corr, 'Frequency');

% Make plot
histogram(ax_fig_conc_corr, y_corr, bar_edges, 'FaceColor', plot_color)

% Exporting to .tiff and .pdf
print(fig_conc_corr, '-dtiffn', strcat('-r', num2str(resol_for_print)), 'Correction factors')
fig_conc_corr.PaperOrientation  = 'landscape';
fig_conc_corr.PaperType         = 'A3';
print(fig_conc_corr, '-dpdf', 'Correction factors')