% This function plots the pooled estimate of runs of
% choice (interactive selection), with errorbars denoting
% +/- 1 standard error
% You need the useful_colormaps function [1] that was included in the code
% documentation earlier as well, just for the colors in the plots.
%
% Date: 23-11-2023
%
% Bibliography
% [1] Smith, N. J., Walt, S. van der, & Firing, E. (2015). Magma, inferno,
%     plasma and viridis colormaps,
%     https://github.com/BIDS/colormap/blob/master/colormaps.py.

%% Part that you can modify
close all
clc
clearvars

y_limits            = [1e-3 1e3];   % limits of the y-axis
plot_font_name      = 'Arial';  % Name of font in plot
plot_font_size      = 8;        % Font size in plot
resol_for_print     = 300;      % Resolution for .tiff outputs
pos_plot            = [0.2 0.5 5 4]; % Position vector of the figure in inches

% [x-coordinate bottom left corner y-coordinate width height] in inches
% Modify if it does not fit your screen

%% Interactive selection of assays to be plotted
[assays_file_name, assays_path_name]    = uigetfile('*.mat', 'Select the output of the assay processing (Data_assays.mat)');
load([assays_path_name assays_file_name], 'Assays_processed')

% dialog box for selecting which assays to plot
panel1              = dialog('Name', 'Select assays', 'Position', [50 50 400 600]);
all_assays          = fieldnames(Assays_processed);
% popupmenus for order in which to plot which assays
boxes               = gobjects(numel(all_assays), 1);
for i = 1 : numel(all_assays)
    boxes(i)        = uicontrol('Style', 'popupmenu', 'units', 'normalized', ...
                        'Position', [0.1 0.1 + (i - 1) * 0.8 / numel(all_assays) 0.3 min(0.1, 0.8 /numel(all_assays))], ...
                        'String', ['empty'; all_assays], 'Parent', panel1);
end
% button when done with selecting assays
uicontrol('Style', 'pushbutton', 'units', 'normalized', ...
                        'Position', [0.65 0.1 0.3 0.25], ...
                        'String', 'Start', 'Parent', panel1, 'Callback', 'uiresume(panel1)');
% popupmenu for selecting ref. line in plot, if needed
pop_ref             = uicontrol('Style', 'popupmenu', 'units', 'normalized', ...
                        'Position', [0.65 0.4 0.3 0.1], ...
                        'String', ['empty'; all_assays], 'Parent', panel1);
pop_ref_text        = uicontrol('Style', 'text', 'units', 'normalized', ...
                        'Position', [0.65 0.55 0.3 0.1], ...
                        'String', 'ref. line', 'Parent', panel1);
% Wait until selection is made
uiwait(panel1)
% Indices of selected assays
assay_list          = flipud(nonzeros([boxes.Value] - 1));
% Assay for reference list, if needed
pop_ref_val         = pop_ref.Value - 1;
if pop_ref_val == 0
    assay_ref       = 'empty';
else
    assay_ref       = all_assays(pop_ref_val);
end
% choice is completed, close the dialog box again
close(panel1)

%% Load and process data to remove unncessary assays that won't be plotted
Assay_names         = all_assays(assay_list);
Assays_processed    = rmfield(Assays_processed, setdiff(fieldnames(Assays_processed), Assay_names, 'stable'));
Assays_processed    = orderfields(Assays_processed, Assay_names);
no_Proteins         = structfun(@(x) numel(cat(2, x.Proteins.QuaEff, x.Proteins.LinEff)), Assays_processed);
no_categories       = (no_Proteins == 0) * 2 + (no_Proteins == 1) * 1 + (no_Proteins == 2) * 3;
% 1 category if one protein is varied (= 1 parameter to plot, k3), 2 if
% that protein is a GTPase (k1 and k2), 3 if two proteins varied (k3, X1, k3,X2 and k3, X1-X2)
tot_categories      = [0; cumsum(no_categories)];

%% k2-k3 plot for all runs
% Define figure
fig_interaction     = figure;
set(fig_interaction, 'Renderer', 'painters', 'Units', 'inches', 'Position', pos_plot);

% Define axis
ax_fig_interaction  = axes('Parent', fig_interaction);
set(ax_fig_interaction, 'Units', 'inches', ...
    'Position', [0.8 1.0 fig_interaction.Position(3 : 4) - [1 1.8]], ...
    'FontName', 'Arial', 'FontSize', plot_font_size, ...
    'LabelFontSizeMultiplier', 1, 'TitleFontSizeMultiplier', 1, ...
    'NextPlot', 'add', 'LineWidth', 1, 'Box', 'on', 'color', 'none');
xlabel(ax_fig_interaction, '');
ylabel(ax_fig_interaction, 'K1, K2, or K3,X [h^{-1}] (per 1\muM involved protein(s))');

% colors of the category patches
plot_colors         = useful_colormaps('viridis', numel(no_categories) + 2); % [1]

% Define start values for x-y coordinates in the plots, the labels and the colored patches
run_labels          = cell(tot_categories(end), 1);
[prot_patch, prot_text] = deal(cell(numel(Assay_names), 1));
x                   = (1 : tot_categories(end))';               % x-coordinates
[y, y_low, y_upp]   = deal(zeros(tot_categories(end), 1));      % y-coordinates
[patch_xvertices, patch_yvertices] = deal(zeros(numel(Assay_names), 4)); % patch coordinates
single_run_data     = zeros(0, 5); % [x, y, and color (3 values)]

for p = 1 : tot_categories(end)
    
    assay_ind       = find(p <= tot_categories(2 : end), 1);        % Index of assay in this iteration
    this_assay      = Assays_processed.(Assay_names{assay_ind});    % Assay data of this iteration
    if isempty(cat(2, this_assay.Proteins.QuaEff, this_assay.Proteins.LinEff))
        ind_est     = p - tot_categories(assay_ind);
    else
        ind_est     = 1 + p - tot_categories(assay_ind);
    end
    % Index of the parameter estimate (2, or 3/4 when 2 proteins in the sweep)
    
    % Find the varied proteins
    Proteins_var        = cat(2, this_assay.Proteins.QuaEff, this_assay.Proteins.LinEff);
    % If no effectors, then the varied protein is the GTPase
    if isempty(Proteins_var)
        Proteins_var    = this_assay.Proteins.GTPase{1};
    end
    
    % Make the name for the label at the top
    category_name   = strrep(Assay_names{assay_ind}, '_', '\_');
        
    % Tick labels on x-axis, extracted from the run names
    run_labels{p}   = {'pooled'};

    % Make the name for the label at the top
    if p - tot_categories(assay_ind) <= 2 && ~isempty(cat(2, this_assay.Proteins.QuaEff, this_assay.Proteins.LinEff))
        run_labels{p}   = ['K3,X  X= ' Proteins_var{p - tot_categories(assay_ind)}];
    elseif isempty(cat(2, this_assay.Proteins.QuaEff, this_assay.Proteins.LinEff)) && p == (tot_categories(assay_ind) + 1)
        run_labels{p}   = 'K1 ';
    elseif isempty(cat(2, this_assay.Proteins.QuaEff, this_assay.Proteins.LinEff)) && p == (tot_categories(assay_ind) + 2)
        run_labels{p}   = ['K2 '];
    else
        run_labels{p}   = ['K3,X  X= ' Proteins_var{1} ',' Proteins_var{2}];
    end
    
    % Indices of which runs are part of the pooled estimate
    ind_pool        = logical(this_assay.k_runs.pool);
    
    % Rate numbers
    y(p)            = this_assay.k_tot.est(ind_est);
    y_single        = vertcat(this_assay.k_runs.est);
    y_single        = y_single(ind_pool, ind_est);
    
    % confidence intervals
    y_low(p)        = y(p) - this_assay.k_tot.std_err(ind_est);
    y_upp(p)        = y(p) + this_assay.k_tot.std_err(ind_est);
        
    single_run_data = [single_run_data; p * ones(size(y_single)) y_single ...
                        repmat(plot_colors(1 + assay_ind, :), numel(y_single), 1)];
    
    % only make patch once per category
    if p - tot_categories(assay_ind) > 1, continue, end
        % coordinates for patches of categories
    
    patch_xvertices(assay_ind, :) = x(p) + 0.3 * [-1 1 1 -1] + (tot_categories(assay_ind + 1) - p) * [0 1 1 0];
    patch_yvertices(assay_ind, :) = [1e-5 1e-5 1e5 1e5];
    
    % make patches
	prot_patch{assay_ind}   = patch(patch_xvertices(assay_ind, :), patch_yvertices(assay_ind, :), ...
                                1 - 0.2 * (1 - plot_colors(1 + assay_ind, :)), 'Parent', ax_fig_interaction);
    prot_patch{assay_ind}.EdgeColor = 'none';
    % category labels above patches
    prot_text{assay_ind}    = text(patch_xvertices(assay_ind, 1), y_limits(2) * 1.1, category_name, ...
                        'Parent', ax_fig_interaction, 'FontSize', plot_font_size, 'FontName', plot_font_name);
end

% Corrections for plotting purposes
y_low(y_low < 1e-10) = 1e-10;
% truncate extremely low numbers
run_labels2         = run_labels;
% remove NaNs
run_labels2         = run_labels2(~isnan(y));
y_low               = y_low(~isnan(y));
y_upp               = y_upp(~isnan(y));
y                   = y(~isnan(y));

% Make error bars
for p = 1 : tot_categories(end)
    assay_ind       = find(p <= tot_categories(2 : end), 1);        % Index of assay in this iteration
    errorbar(x(p), y(p), y(p) - y_low(p), y_upp(p) - y(p), 'LineStyle', 'none', 'Marker', 'x', 'MarkerSize', 8, ...
        'Color', plot_colors(1 + assay_ind, :), 'Parent', ax_fig_interaction)
end

% Plot single run points
for i = 1 : size(single_run_data, 1)
    plot(single_run_data(i, 1), single_run_data(i, 2), 'Marker', 'o', 'MarkerSize', 6, ...
        'Color', single_run_data(i, 3 : 5), 'MarkerFaceColor', single_run_data(i, 3 : 5), 'Parent', ax_fig_interaction)
end

% Make ref. line
if ~strcmp(assay_ref, 'empty')
    line([-10 1e3], Assays_processed.(assay_ref{:}).k_tot.est(2) * [1 1], ...
        'LineStyle', '--', 'Color', [0 0 0], 'Parent', ax_fig_interaction);
end

% Adjust appearance axis
set(ax_fig_interaction, 'YScale', 'log', 'XLim', [min(patch_xvertices(:)) - 1 max(patch_xvertices(:)) + 1], ...
    'XTick', x, 'XTickLabels', run_labels2, 'YLim', y_limits, ...
    'YTick', 10 .^ (floor(log10(y_limits(1))) : ceil(log10(y_limits(2)))), ...
    'XTickLabelRotation', 90, 'TickLength', [0 0], ...
    'YMinorTick', 'off', 'YGrid', 'on', 'YMinorGrid', 'off');

% Adjust label positions to avoid overlap
for p = 1 : numel(prot_text)
    prot_text{p}.Position(1) = patch_xvertices(p, 1) + ...
                                (patch_xvertices(p, 2) - patch_xvertices(p, 1)) / 2 - prot_text{p}.Extent(3) / 2;
    prot_text{p}.Position(2)  = (0.7 + exp(mod(p, 3)) * 3)* ax_fig_interaction.YLim(2);
end

% Create second axis to get rid of ticks on x-axis
ax_fig_interaction2  = copyobj(ax_fig_interaction, fig_interaction);
cla(ax_fig_interaction2)
set(ax_fig_interaction2, 'TickLength', [0.01 0.01], 'XTick', [])

% Export figure to .tiff and .pdf
print(fig_interaction, '-dtiffn', strcat('-r', num2str(resol_for_print)), ...
    'Rate contribution per protein')
fig_interaction.PaperOrientation  = 'landscape';
fig_interaction.PaperType         = 'A3';
print(fig_interaction, '-dpdf', 'Rate contribution per protein')