# -*- coding: utf-8 -*-
"""
Created on Tue Jun 22 09:30:27 2021
@author: jeroenverhagen
"""

import pyomo.opt as op
import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import itertools
import numpy.matlib
import os
import logging

from pyomo.core.kernel.sos import *
from matplotlib.colors import ListedColormap, LinearSegmentedColormap
from pyomo.environ import *
from pyomo.mpec import *
from pyomo.util.infeasible import log_infeasible_constraints

# %% Definitions 
def create_model(address_heat_demand_area_type,heat_supply_threshold,initial_heat_demand,minimum_heat_demand,emission_technology,annual_insulation_capacity,address_floor_area,T0,Te,Ts,emission_technology_max,insulation_path):

    #Model
    m = AbstractModel('amsterdam_heat_mix')

    #Sets
    m.address = RangeSet(1,address_heat_demand_area_type.shape[0])
    m.strategy = RangeSet(1,len(heat_supply_threshold))
    m.time = RangeSet(1,int((Te-T1)/Ts)+1)
    m.objectiveTime = RangeSet(2,int((Te-T1)/Ts)+1)
    
    #Parameters
    m.heat_supply_boundary = Param(m.strategy,initialize=heat_supply_boundary)
    m.emission = Param(m.strategy,m.time,initialize=emission_technology)
    m.annual_insulation_capacity = Param(m.time,initialize=annual_insulation_capacity)
    m.address_area = Param(m.address,initialize=address_floor_area)
    m.heat_supply_threshold = Param(m.strategy,initialize=heat_supply_threshold)
    m.initial_heat_demand = Param(m.address,initialize=initial_heat_demand)
    m.minimum_heat_demand = Param(m.address,initialize=minimum_heat_demand)
    m.emission_technology_max = Param(m.time,initialize=emission_technology_max)  
    m.insulation_path = Param(m.time,initialize=insulation_path) 

    #Variables 
    m.x_HT = Var(m.address,m.strategy,domain=Binary)  
    m.x_HF = Var(m.strategy,domain=Binary) 
    
    def heat_demand_bounds(m,a,t):
        return (m.minimum_heat_demand[a],m.initial_heat_demand[a])
    def heat_demand_initialization(m,a,t):
        return m.initial_heat_demand[a]
    m.x_HD = Var(m.address,m.time,domain=NonNegativeReals,bounds=heat_demand_bounds,initialize=heat_demand_initialization)
    
 
    #Constraint 1: supply demand equilibrium
    def equilibrium(m,a):
        return sum(m.x_HT[a,:])==1.0
    m.c_equilibrium = Constraint(m.address,rule = equilibrium)
    
    #Constraint 2: maximum heat supply by technology strategy
    def max_supply(m,a,s,t):
        return 1 - (m.x_HD[a,t]*m.x_HT[a,s]/m.heat_supply_boundary[s])>= 0.0
    m.c_maxSupply = Constraint(m.address,m.strategy,m.time,rule = max_supply)
      
    #Constraint 3: increasing insulation level between years
    def heat_demand_address(m,a,t):
        if t < int((Te-T1)/Ts)+1:
            return (m.x_HD[a,t] - m.x_HD[a,t+1])/m.initial_heat_demand[a] >= 0.0
        else:
            return m.x_HD[a,t]/m.initial_heat_demand[a] >= 0.0
    m.c_insulation = Constraint(m.address,m.time,rule = heat_demand_address)
    
    #Constraint 4: insulation capacity pathway
    def pathway(m,t):
        if t == 1:
            return m.annual_insulation_capacity[t] - sum((m.initial_heat_demand[a]-m.x_HD[a,t])*m.address_area[a] for a in m.address) >= 0.0
        else:
            return m.annual_insulation_capacity[t] - sum((m.x_HD[a,t-1]-m.x_HD[a,t])*m.address_area[a] for a in m.address) >= 0.0
    m.c_insulation_pathway = Constraint(m.time,rule = pathway)

    #Constraint 5: neighborhood threshold level
    def threshold(m,s,t):
        return sum(m.x_HD[a,t]*m.x_HT[a,s]*m.address_area[a] for a in m.address)/heat_supply_threshold[s] - m.x_HF[s] >= 0.0
    m.c_threshold = Constraint(m.strategy,m.time,rule = threshold)

    #Constraint 6: technology level continuity 
    def continuity(m,s):
        return m.x_HF[s] - sum(m.x_HT[a,s] for a in m.address)/address_heat_demand_area_type.shape[0] >= 0.0
    m.c_continuity = Constraint(m.strategy,rule = continuity)

    #Objective: maximize profit
    def emission(m):        
        scaling = sum(m.insulation_path[t]*Ts*sum(m.address_area[a] for a in m.address)*m.emission_technology_max[t]  for t in m.objectiveTime)
        return sum(sum(sum(0.5*(m.x_HD[a,t-1]+m.x_HD[a,t])*m.x_HT[a,s]*m.address_area[a]*Ts*m.emission[s,t] for a in m.address) for s in m.strategy) for t in m.objectiveTime)/scaling
    m.obj = Objective(rule = emission, sense = minimize)     
    
    return m 

        

def electricity_pathway(start,end,pathway,T0,T1,Te,Ts):
    #define pathways from start and end value (both defined on January 1 as 
    #yearly average representation) assuming all change takes places at the 
    #beginning of the year for a choosen configuration; 0 = linear; 
    #1 = no change till T1; -1 = all change before T1
    if pathway == 0:
        yearly_emission = np.linspace(start, end, num=Te-T0+1)
        obj_yearly_emission = yearly_emission[T1-T0:Te-T0+1]  
        obj_period_emission = obj_yearly_emission[np.arange(0,Te-T1+1,Ts)]
        periodic_avg_emission = np.hstack((np.zeros((1)),(obj_period_emission[0:-1]+obj_period_emission[1:])/2))   
    elif pathway==1:
        obj_yearly_emission = np.linspace(start, end, num=Te-T1+1)
        obj_period_emission = obj_yearly_emission[np.arange(0,Te-T1+1,Ts)]
        periodic_avg_emission = np.hstack((np.zeros((1)),(obj_period_emission[0:-1]+obj_period_emission[1:])/2))
    elif pathway==-1:
        obj_yearly_emission = end*np.ones((int((Te-T1+1))))
        periodic_avg_emission = np.hstack((np.zeros((1)),end*np.ones((int((Te-T1)/Ts)))))   
    return periodic_avg_emission,obj_yearly_emission[1:] 

def insulation_capabilities(start,end,pathway,T0,T1,Te,Ts,housing_area):
    #define insulation capabilities from start and end value (both defined on 
    #January 1 as yearly average representations) assuming all change takes 
    #places at the beginning of the period for a choosen configuration; 
    #0 = linear; 1 = no change till T1; -1 = all change before T1
    if pathway == 0:
        annual_cap = (start - end)*housing_area/(Te-T0)
        insulation_cap = np.vstack((annual_cap*(T1-T0), Ts*annual_cap*np.ones((int((Te-T1)/Ts),1))))
    elif pathway==1:
        annual_cap = (start - end)*housing_area/(Te-T1)
        insulation_cap = np.vstack((np.zeros((1)), Ts*annual_cap*np.ones((int((Te-T1)/Ts),1))))
    elif pathway==-1:
        insulation_cap = np.vstack(((start - end)*housing_area, np.zeros((int((Te-T1)/Ts),1))))
    return insulation_cap

def my_callback(cb_m, cb_opt, cb_where):
    if cb_where == GRB.Callback.MIPSOL:
        cb_opt.cbGetSolution(vars=[m.x_HF])


# %% input settings 

#Read scenario settings
scenario_settings = pd.read_excel('Scenario_settings.xlsx')
scenario_name = scenario_settings['Name'].to_numpy()

hydrogen_active = scenario_settings['hydrogen_active'].to_numpy()
neigbourhood_constraint = scenario_settings['neigbourhood_constraint'].to_numpy()
insulation_end = scenario_settings['insulation_end'].to_numpy()
heat_capacity_lt = scenario_settings['heat_capacity_lt'].to_numpy()
electricity_end = scenario_settings['electricity_end'].to_numpy()
incineration_multiplication = scenario_settings['incineration_multiplication'].to_numpy()
T1_scenario = scenario_settings['T1'].to_numpy()
initial_heat_multiplication = scenario_settings['initial_heat_multiplication'].to_numpy()

for sc in range(1,len(scenario_name)):
    
    #Neighbourhood range and scenario name
    neighborhood_range = range(1,4)
    scenarioName = scenario_name[sc]
    
    #Time settings
    T0 = 2020
    T1 = int(T1_scenario[sc])
    Te = 2050
    Ts = 5
    
    #Target for average household heat demand (kWh/m2) on neighbourhood level
    insulation_Te = insulation_end[sc]  
    
    #Current and future emission (kg CO2/GJ) by electricity mix 
    emission_T0 = 154.44
    emission_Te = electricity_end[sc]
    
    
    # %% Load and preprocess data 
    #Load heat technology information
    heat_technology = pd.read_excel('Heat_technology.xlsx',sheet_name='Technology')
    heat_technology.loc[[10,11],['Active']] = hydrogen_active[sc]
    heat_capacity_complete = pd.read_excel('Heat_technology.xlsx',sheet_name='Capacity_constraint')
    heat_capacity_complete.loc[:,['Minimum_heat_density [kWh/ha/year]']] = neigbourhood_constraint[sc]*heat_capacity_complete.loc[:,['Minimum_heat_density [kWh/ha/year]']]
    heat_capacity_complete.loc[[0,1,6,7],['Maximum_heat_capacity [kWh/m2/year]']] = heat_capacity_lt[sc]
    heat_tech_class = np.array(heat_technology['class'])
    heat_tech_active = np.array(heat_technology['Active'])
    
    distribution_loss = pd.read_excel('Heat_technology.xlsx',sheet_name='Distribution_loss')
    ERE_factor = pd.read_excel('Heat_technology.xlsx',sheet_name='ERE_factor')
    emission_factor_fuel = pd.read_excel('Heat_technology.xlsx',sheet_name='Emission_factor_fuel')
    emission_factor_fuel.loc[[0,1,4],['Incineration of fuel (kg CO2/GJ)']] = incineration_multiplication[sc]*emission_factor_fuel.loc[[0,1,4],['Incineration of fuel (kg CO2/GJ)']]
    direct_emission = np.sum(0.0036*(1/(1-distribution_loss.loc[:,['loss [%]']].to_numpy()/100))*(ERE_factor.loc[:,['Biomass','Green gas','Hydrogen','Natural gas','Waste']].to_numpy()*emission_factor_fuel.loc[:,['Incineration of fuel (kg CO2/GJ)']].to_numpy().T),axis=1).reshape(12,1)
    indirect_emission = np.sum(0.0036*(1/(1-distribution_loss.loc[:,['loss [%]']].to_numpy()/100))*(ERE_factor.loc[:,['Biomass','Green gas','Hydrogen','Natural gas','Waste']].to_numpy()*np.sum(emission_factor_fuel.loc[:,['Production of fuel','Transportation of fuel']].to_numpy().T,axis=0)),axis=1).reshape(12,1)
    electricity_emission = 0.0036*(1/(1-distribution_loss.loc[:,['loss [%]']].to_numpy()/100))*ERE_factor.loc[:,['Electricity']].to_numpy()
    emissions_complete = pd.DataFrame(np.column_stack((direct_emission,indirect_emission,electricity_emission)), columns=['direct_emission [kgCO2/kWh]', 'indirect_emission [kgCO2/kWh]', 'electricity_emission [(kgCO2/kWh)/(kgCO2/GJ)]'])
    heat_class = pd.read_excel('Heat_technology.xlsx',sheet_name='Class')
    
    #Load heat demand information 
    building_typology = pd.read_excel('Address_Neigborhood_Heat_Demand.xlsx',sheet_name='Building_type')
    construction_period = pd.read_excel('Address_Neigborhood_Heat_Demand.xlsx',sheet_name='Construction_period')
    address = pd.read_excel('Address_Neigborhood_Heat_Demand.xlsx',sheet_name='Address')
    neighborhood = pd.read_excel('Address_Neigborhood_Heat_Demand.xlsx',sheet_name='Neighborhood')
    heat_demand = pd.read_excel('Address_Neigborhood_Heat_Demand.xlsx',sheet_name='Heat_demand_buidling_type')
    heat_demand.loc[:,['1 [kWh/m2/year]']] = initial_heat_multiplication[sc]*heat_demand.loc[:,['1 [kWh/m2/year]']]
        
    # %% Compute the optimal mix for all neighborhoods, emission and insulation levels
    if not os.path.isdir('Output/'+scenarioName):
        os.mkdir('Output/'+scenarioName)
        
    for neighborhood_idx in neighborhood_range:
            # %% Create data storage folder
            neighbourhood_dir = 'Output/'+scenarioName +'/'+ neighborhood['Name'][neighborhood_idx-1]
            if not os.path.isdir(neighbourhood_dir):
                os.mkdir(neighbourhood_dir)
        
            for pw_e in range(-1,2):    
                for pw_i in range(-1,2):
                    pathway_dir = neighbourhood_dir + '/E' + str(pw_e) +'_I'+ str(pw_i)
                    if not os.path.isdir(pathway_dir):
                        os.mkdir(pathway_dir)
                    
        
                    # %% Prepare data
                    #Initial and minimum heat demand for address in the studied neighbourhood 
                    address_by_neighborhood = address.loc[address['Neighbourhood']==neighborhood_idx]
                    building_types_set = np.unique(np.column_stack((address_by_neighborhood['Building_type'],address_by_neighborhood['Construction_period'])),axis=0)
                    housing_type_name = [building_typology.loc[building_typology['idx']==building_types_set[i,0],'description']  for i in range(0,building_types_set.shape[0])]
                    housing_type_name = [housing_type_name[i].iloc[0] for i in range(0,len(housing_type_name))]
                    housing_type_period = [construction_period.loc[construction_period['idx']==building_types_set[i,1],'period'] for i in range(0,building_types_set.shape[0])]
                    housing_type_period = [housing_type_period[i].iloc[0] for i in range(0,len(housing_type_period))]
                    
                    address_heat_demand_area_type = np.zeros((address_by_neighborhood.shape[0],4))
                    tt = 0
                    for i in address_by_neighborhood['Index']-1:
                        address_heat_demand_area_type[tt,0] = heat_demand.loc[(heat_demand['Building_type']==address_by_neighborhood['Building_type'][i]) & (heat_demand['Construction_period']==address_by_neighborhood['Construction_period'][i]),'1 [kWh/m2/year]'].values[0]
                        address_heat_demand_area_type[tt,1] = heat_demand.loc[(heat_demand['Building_type']==address_by_neighborhood['Building_type'][i]) & (heat_demand['Construction_period']==address_by_neighborhood['Construction_period'][i]),'4 [kWh/m2/year]'].values[0]
                        address_heat_demand_area_type[tt,2] = address_by_neighborhood['Surface_area [m2]'][i]
                        address_heat_demand_area_type[tt,3] = np.where((building_types_set[:,0]==address_by_neighborhood['Building_type'][i]) & (building_types_set[:,1]==address_by_neighborhood['Construction_period'][i]))[0]
                        tt = tt + 1
                                           
                    #Neighbourhood average heat demand (kWh/m2) at T0
                    insulation_T0 = sum(address_heat_demand_area_type[:,0]*address_heat_demand_area_type[:,2])/sum(address_heat_demand_area_type[:,2])
                    housing_area = sum(address_heat_demand_area_type[:,2])
                    
                    #Compute electricity and insulation pathways
                    insulation_capacity = insulation_capabilities(insulation_T0,insulation_Te,pw_i,T0,T1,Te,Ts,housing_area)
                    electricity_emission_path,annual_emission_obj = electricity_pathway(emission_T0,emission_Te,pw_e,T0,T1,Te,Ts)
                    
                    #Yearly minimum heat demand neighbourhood
                    annual_insulation_capacity_array = insulation_capacity
                    annual_insulation_capacity = dict(enumerate(insulation_capacity.flatten(), 1))
                    insulation_path = insulation_T0 - np.cumsum(insulation_capacity/housing_area)
                    insulation_path = np.vstack((np.zeros((1)),(0.5*(insulation_path[0:-1]+insulation_path[1:])).reshape(int((Te-T1)/Ts),1)))
                    insulation_path = dict(enumerate(insulation_path.flatten(),1))
                    
                    #Find the technologies per class having the lowest cumulative emission
                    annual_emissions = emissions_complete['direct_emission [kgCO2/kWh]'].values + emissions_complete['indirect_emission [kgCO2/kWh]'].values + annual_emission_obj.reshape(len(annual_emission_obj),1)*emissions_complete['electricity_emission [(kgCO2/kWh)/(kgCO2/GJ)]'].values
                    cumulative_emission_technology = np.sum(annual_emissions,0)               
                    cumulative_emission_technology[heat_tech_active==0] = 1e5
                    active_technology = np.ones((len(heat_tech_active)))
                    for i in np.unique(heat_tech_class):
                        class_emission = cumulative_emission_technology[heat_tech_class==i]
                        active_technology[heat_tech_class==i] = 1*(class_emission == np.min(class_emission))
                    
                    # Remove inactive technologies 
                    heat_capacity = heat_capacity_complete[active_technology==1]
                    emissions = emissions_complete[active_technology==1]
                    technology_mapping_matrix = np.diag(1*(active_technology==1))
                    technology_mapping_matrix = technology_mapping_matrix[:,~np.all(technology_mapping_matrix == 0, axis=1)]
                    
                    #Minimum required heat demand on neighborhood level     
                    heat_supply_threshold = heat_capacity['Minimum_heat_density [kWh/ha/year]'].values * neighborhood.loc[neighborhood['Index']==neighborhood_idx,'Area [ha]'].values
                    heat_supply_threshold[np.isnan(heat_supply_threshold)] = 1
                    heat_supply_threshold_array = heat_supply_threshold
                    heat_supply_threshold = dict(enumerate(heat_supply_threshold.flatten(), 1))
                    
                    #Maximum feasible heat supply per square meter per technology 
                    heat_supply_boundary = heat_capacity['Maximum_heat_capacity [kWh/m2/year]'].values
                    heat_supply_boundary[np.isnan(heat_supply_boundary)] = 1000
                    heat_supply_boundary_array = heat_supply_boundary
                    heat_supply_boundary = dict(enumerate(heat_supply_boundary.flatten(), 1))
            
                    #Yearly emission of technology composed of direct, indirect and electricity
                    #saved in dictionary of technology x time
                    emission_technology = emissions['direct_emission [kgCO2/kWh]'].values + emissions['indirect_emission [kgCO2/kWh]'].values + electricity_emission_path.reshape(len(electricity_emission_path),1)*emissions['electricity_emission [(kgCO2/kWh)/(kgCO2/GJ)]'].values
                    emission_technology_max = dict(enumerate(emission_technology.max(1).flatten(),1))
                    emission_technology_numpy = emission_technology
                    emission_technology_numpy = np.matmul(technology_mapping_matrix,emission_technology_numpy.T).T
                    emission_technology = dict(((j+1,i+1), emission_technology[i][j]) for i in range(0,int((Te-T1)/Ts+1)) for j in range(0,len(heat_supply_threshold)))
                        
                    #Initial and minimum heat demand on address level saved in a 
                    #dictionary of address x time
                    initial_heat_demand = dict(enumerate(address_heat_demand_area_type[:,0].flatten(), 1))
                    minimum_heat_demand = dict(enumerate(address_heat_demand_area_type[:,1].flatten(), 1))
                    address_floor_area = dict(enumerate(address_heat_demand_area_type[:,2].flatten(),1))
                       
             
                    # %% Model definition 
                    model = create_model(address_heat_demand_area_type,heat_supply_threshold,initial_heat_demand,minimum_heat_demand,emission_technology,annual_insulation_capacity,address_floor_area,T0,Te,Ts,emission_technology_max,insulation_path)
                    instance = model.create_instance()
                    
                    # %% Solve model               
                    opt = op.SolverFactory('gurobi')
                    opt.options['MIPGap']= 1e-5
                    opt.solve(instance, tee=True)
                    
       
                    # %% Extract data and write to folder
                    heating_strategy=np.zeros((len(instance.address),len(instance.strategy)))
                    for i in instance.address:
                        for j in instance.strategy:
                            heating_strategy[i-1,j-1] = instance.x_HT[i,j].value
                    heating_strategy = np.matmul(technology_mapping_matrix,heating_strategy.T).T
                    
                    insulation_level=np.zeros((len(instance.address),len(instance.time)))
                    for i in instance.address:
                        for j in instance.time:
                            insulation_level[i-1,j-1] = instance.x_HD[i,j].value
                            
                    neigbourhood_active_technology=np.zeros((1,len(instance.strategy)))
                    for i in instance.strategy:
                        neigbourhood_active_technology[0,i-1] = instance.x_HF[i].value
                    neigbourhood_active_technology = np.matmul(technology_mapping_matrix,neigbourhood_active_technology.T).T
                    
                    address_heat_demand = np.matmul(np.diag(address_heat_demand_area_type[:,2]),insulation_level[:,1:])
                    address_emission_technology = np.matmul(heating_strategy,emission_technology_numpy[1:,:].T)
                    address_emission = Ts*np.multiply(address_heat_demand,address_emission_technology)
                    
                    emission_by_technology = pd.DataFrame(emission_technology_numpy)
                    heating_strategy = pd.DataFrame(heating_strategy)
                    insulation_level = pd.DataFrame(insulation_level)
                    neigbourhood_active_technology = pd.DataFrame(neigbourhood_active_technology)
                    address_emission = pd.DataFrame(address_emission)
                    annual_insulation_capacity = pd.DataFrame(annual_insulation_capacity_array)
                    annual_emissions = pd.DataFrame(annual_emissions)
                    heat_supply_boundary = pd.DataFrame(np.matmul(technology_mapping_matrix,heat_supply_boundary_array.T).T)
                    heat_supply_threshold = pd.DataFrame(np.matmul(technology_mapping_matrix,heat_supply_threshold_array.T).T)
                    
                    address_by_neighborhood_information = pd.concat([address_by_neighborhood,pd.DataFrame(address_heat_demand_area_type[:,0:2],columns=['intial_heat','min_heat'],index=address_by_neighborhood.index)],axis=1)
                    address_by_neighborhood_information.to_excel(pathway_dir+'/address_by_neighborhood_information.xlsx')
                    insulation_level.to_excel(pathway_dir +'/insulation_level.xlsx')
                    neigbourhood_active_technology.to_excel(pathway_dir +'/neigbourhood_active_technology.xlsx')
                    heating_strategy.to_excel(pathway_dir +'/heating_strategy.xlsx')
                    annual_insulation_capacity.to_excel(pathway_dir +'/annual_insulation_capacity.xlsx')
                    address_emission.to_excel(pathway_dir +'/address_emission.xlsx')
                    emission_by_technology.to_excel(pathway_dir +'/emission_by_technology.xlsx')
                    annual_emissions.to_excel(pathway_dir +'/annual_emissions.xlsx')
                    heat_supply_boundary.to_excel(pathway_dir +'/heat_supply_boundary.xlsx')
                    heat_supply_threshold.to_excel(pathway_dir +'/heat_supply_threshold.xlsx')
                    
                    
                    
                    
                    
                    