% code to compute basic state flow U_0p in Fourier space forced by tide

% Fp_vec: dimensionless tidal forcing in Fourier space (modes -P:P), Fp_vec(i) = conj(Fp_vec(-i))
% s: slip parameter (dimensionless)

% last modified 25/3/2024
% author: Laura Portos-Amill

function [U_0p] = my_basicstate(alpha,Fp_vec,s)

zr = .1; %size of the homogeneous equivalent roughness
N = 1e5; %number of vertical layers
dz = abs((-1+zr/29.8)/(N-1));
z = [0:-dz:-1+zr/29.8]; %vertical coordinate
H = 30; %mean water depth (m)
U = 1; %velocity scale (m/s)
omega = 2*pi/(12*3600 + 25*60); %radial frequency of M2 tidal component (s^-1)

r = 1; % keulegan-carpenter number
L = 7000; %spatial scale (m)

k_vk = 0.4; %von karman constant
A = 1.84; B = -1.56; % blondeaux & vittori 2005

%Dean profile for viscosity
A0 = k_vk.*U./(H.*omega.*5.75.*log10(10.9./zr))/mean(-(1+z)./...
    (4*A + 2*A*z + 9*B + 9*B*z + 3*B*z.^2));
%Uniform viscosity (value from Campmans 2017)
Av_0_un = 0.32*ones(size(z));
% viscosity as linear combination of uniform and depth-dependent
% alpha = 0 means uniform viscosity
Av_0 = Av_0_un + alpha*(Av_0_Dean - Av_0_un);


%% initialisation for basic, first, and second order states

P = floor(length(Fp_vec)/2); % highest mode --> (2P+1) modes in total
Np = 2*P+1; %odd number because -P to P

U_0p = zeros(Np,length(z)); %basic state fourier amplitudes full sol
U_0p_nh = zeros(Np,length(z)); %basic state fourier amplitudes nonhom sol
U_0p_h = zeros(Np,length(z)); %basic state fourier amplitudes hom sol

D_0p = zeros(Np,length(z)); %basic state fourier amplitudes full sol
D_0p_nh = NaN(Np,length(z)); % D0p = Av0*dU_0p/dz
D_0p_h = NaN(Np,length(z));

alphap0 = NaN(Np,1); %vector to scale homogeneous solution for basic state 2(2P+1)

%seabed conditions for basic state
U_0p_nh(:,1) = ones(length(Fp_vec),1); D_0p_nh(:,1) = 0; %seabed conditions, 0 for Dp, random for up
U_0p_h(:,1) = ones(length(Fp_vec),1); D_0p_h(:,1) = 0; %seabed conditions, 0 for Dp, random for up

%% solve basic state

p_vec = [-floor(Np/2):floor(Np/2)].';

for j = 2:length(z)
    %hom
    k1_D = 1i*p_vec.*U_0p_h(:,j-1);
    k1_U = D_0p_h(:,j-1)/Av_0(j-1);
    k2_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k1_U);
    k2_U = (D_0p_h(:,j-1) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k3_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k2_U);
    k3_U = (D_0p_h(:,j-1) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k4_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*k3_U);
    k4_U = (D_0p_h(:,j-1) - dz*k3_D)/Av_0(j);

    D_0p_h(:,j) = D_0p_h(:,j-1) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);
    U_0p_h(:,j) = U_0p_h(:,j-1) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);

    %nonhom
    k1_D = 1i*p_vec.*U_0p_nh(:,j-1) - Fp_vec.';
    k1_U = D_0p_nh(:,j-1)/Av_0(j-1);
    k2_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k1_U) - Fp_vec.';
    k2_U = (D_0p_nh(:,j-1) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k3_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k2_U) - Fp_vec.';
    k3_U = (D_0p_nh(:,j-1) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k4_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*k3_U) - Fp_vec.';
    k4_U = (D_0p_nh(:,j-1) - dz*k3_D)/Av_0(j);

    D_0p_nh(:,j) = D_0p_nh(:,j-1) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);
    U_0p_nh(:,j) = U_0p_nh(:,j-1) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);
end

%compute alpha applying bc at bed
alphap0 = -(s*U_0p_nh(:,end) - D_0p_nh(:,end))./(s*U_0p_h(:,end) - D_0p_h(:,end)); % partial slip

U_0p(:,:) = alphap0.*U_0p_h(:,:) + U_0p_nh(:,:); %general solution as linear combination of nohom and hom
D_0p(:,:) = alphap0.*D_0p_h(:,:) + D_0p_nh(:,:);
end