% compute basic, first and second order flow forced by a tide and affected
% by sand wave topography

% alpha: between 0 and 1, scalar used in eddy viscosity formulation, 
%   0 means uniform viscosity, 1 means Dean profile
% k: dimensionless wave number
% Fp_vec: dimensionless tidal forcing in Fourier space (modes -P:P), Fp_vec(i) = conj(Fp_vec(-i))
% h1: dimensionless bedform amplitude
% s: dimensionless slip parameter

% last modified 21/12/2023
% author: Laura Portos-Amill

function [U_0p,D_0p,U_1p,D_1p,W_1p,Z_1p,U_2p,Av_0,z] = my_secondorder(alpha,k,Fp_vec,h1,s)

zr = .1; %size of the homogeneous equivalent roughness
N = 1e5; %number of vertical layers
dz = abs((-1+zr/29.8)/(N-1));
z = [0:-dz:-1+zr/29.8]; %vertical coordinate
H = 30; %mean water depth
U = 1; %scale of flow velicity (m/s)
omega = 2*pi/(12*3600 + 25*60); %radial frequency of M2 tidal component (s^-1)
A = 1.84; B = -1.56; % blondeaux & vittori 2005

r = 1; % keulegan-carpenter number
L = 7000; %spatial scale (m)

k_vk = 0.4; %von karman constant

%Dean profile for viscosity
A0 = k_vk.*U./(H.*omega.*5.75.*log10(10.9./zr))/mean(-(1+z)./...
    (4*A + 2*A*z + 9*B + 9*B*z + 3*B*z.^2));
Av_0_Dean = -A0.*(1+z)./...
    (4*A + 2*A*z + 9*B + 9*B*z + 3*B*z.^2);
Av_1_Dean = -A0*h1*(2*A*z - 6*B*z.^2 - 3*B*z.^3)./(4*A + 9*B + (2*A + 9*B)*z + 3*B*z.^2).^2;
Av_2_Dean = -2*h1*conj(h1)*A0./(4*A + 9*B + z*(2*A+9*B) + z.^2*3*B).^4.*...
    ((-6*B - 6*B*z).*(4*A + 9*B + z*(2*A+9*B) + z.^2*3*B).^2 - 2*(4*A + 9*B + z*(2*A+9*B) + z.^2*3*B).*...
    (2*A + 9*B + 6*B*z).*(2*A - 6*B*z - 3*B*z.^2)).*z.^2 - 2*h1*conj(h1)*A0*2.*z.*(2*A-6*B*z-3*B*z.^2)./(4*A + 9*B + z*(2*A+9*B) + z.^2*3*B).^2;

%Uniform viscosity (value from Campmans 2017)
Av_0_un = 0.32*ones(size(z));
Av_1_un = zeros(size(z));
Av_2_un = zeros(size(z));

%viscosity as linear combination of uniform and depth-dependent
% alpha = 0 means uniform Av_0 (and no higher order terms)
Av_0 = Av_0_un + alpha*(Av_0_Dean - Av_0_un);
Av_1 = Av_1_un + alpha*(Av_1_Dean - Av_1_un);
Av_2 = Av_2_un + alpha*(Av_2_Dean - Av_2_un);


%% initialisation for basic, first, and second order states

P = floor(length(Fp_vec)/2); % highest mode --> (2P+1) modes in total
Np = 2*P+1; %odd number because -P to P

U_0p = zeros(Np,length(z)); %basic state fourier amplitudes full sol
U_0p_nh = zeros(Np,length(z)); %basic state fourier amplitudes nonhom sol
U_0p_h = zeros(Np,length(z)); %basic state fourier amplitudes hom sol

D_0p = zeros(Np,length(z)); %basic state fourier amplitudes full sol
D_0p_nh = NaN(Np,length(z)); % D0p = Av0*dU_0p/dz
D_0p_h = NaN(Np,length(z));

Z_1p_h = zeros(Np,2*Np); %fourier amplitude of sea surface in perturbed state used for hom sol
U_1p_h = zeros(Np,length(z),2*Np); %1st order state fourier amplitudes hom sol
W_1p_h = zeros(Np,length(z),2*Np); 
D_1p_h = zeros(Np,length(z),2*Np); 

Z_1p_nh = zeros(Np,1); %fourier amplitude of sea surface in perturbed state used for nonhom sol
U_1p_nh = zeros(Np,length(z)); %1st order state fourier amplitudes nonhom sol
W_1p_nh = zeros(Np,length(z)); 
D_1p_nh = zeros(Np,length(z)); 

Z_2p_h = zeros(Np,2*Np); %fourier amplitude of sea surface in perturbed state used for hom sol
U_2p_h = zeros(Np,length(z),Np); %1st order state fourier amplitudes hom sol: p,z,q (mode,depth,homsols)
D_2p_h = zeros(Np,length(z),Np); 

Z_2p_nh = zeros(Np,1); %fourier amplitude of sea surface in perturbed state used for nonhom sol
U_2p_nh = zeros(Np,length(z)); %1st order state fourier amplitudes nonhom sol
D_2p_nh = zeros(Np,length(z)); 

U_2p = zeros(Np,length(z)); D_2p = zeros(Np,length(z)); %full sol

U_1p = zeros(Np,length(z)); W_1p = zeros(Np,length(z)); D_1p = zeros(Np,length(z)); %full sol
Z_1p = zeros(Np,1);


alphap0 = NaN(Np,1); %vector to scale homogeneous solution for basic state 2(2P+1)
alphap1 = NaN(Np*2,1); %vector to scale homogeneous solution for 1st order state 2(2P+1)
alphap2 = NaN(Np,1); %vector to scale homogeneous solution for 2nd order state (2P+1)

%seabed conditions for basic state
U_0p_nh(:,1) = [1;2;3;4;5]; D_0p_nh(:,1) = 0; %seabed conditions, 0 for Dp, random for up
U_0p_h(:,1) = [1;2;3;4;5]; D_0p_h(:,1) = 0; %seabed conditions, 0 for Dp, random for up

%seasurface conditions for 1st order state
U_1p_nh(:,1) = 0; D_1p_nh(:,1) = 0; W_1p_nh(:,1) = 0; %seabed conditions, 0 for Dp and Wp, random for Up
D_1p_h(:,1) = 0; W_1p_h(:,1) = 0; %seabed conditions, 0 for Dp and Wp

%seasurface conditions for 2nd order state
U_2p_nh(:,1) = 0; D_2p_nh(:,1) = 0; W_2p_nh(:,1) = 0; %seabed conditions, 0 for Dp, random for Up
D_2p_h(:,1) = 0; %seabed conditions, 0 for Dp

%% solve basic state

p_vec = [-floor(Np/2):floor(Np/2)].';

for j = 2:length(z)
    %hom
    k1_D = 1i*p_vec.*U_0p_h(:,j-1);
    k1_U = D_0p_h(:,j-1)/Av_0(j-1);
    k2_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k1_U);
    k2_U = (D_0p_h(:,j-1) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k3_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k2_U);
    k3_U = (D_0p_h(:,j-1) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k4_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*k3_U);
    k4_U = (D_0p_h(:,j-1) - dz*k3_D)/Av_0(j);

    D_0p_h(:,j) = D_0p_h(:,j-1) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);
    U_0p_h(:,j) = U_0p_h(:,j-1) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);

    %nonhom
    k1_D = 1i*p_vec.*U_0p_nh(:,j-1) - Fp_vec.';
    k1_U = D_0p_nh(:,j-1)/Av_0(j-1);
    k2_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k1_U) - Fp_vec.';
    k2_U = (D_0p_nh(:,j-1) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k3_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*0.5*k2_U) - Fp_vec.';
    k3_U = (D_0p_nh(:,j-1) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k4_D = 1i*p_vec.*(U_0p_h(:,j-1) - dz*k3_U) - Fp_vec.';
    k4_U = (D_0p_nh(:,j-1) - dz*k3_D)/Av_0(j);

    D_0p_nh(:,j) = D_0p_nh(:,j-1) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);
    U_0p_nh(:,j) = U_0p_nh(:,j-1) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);
end

%compute alpha applying bc at bed
alphap0 = -(s*U_0p_nh(:,end) - D_0p_nh(:,end))./(s*U_0p_h(:,end) - D_0p_h(:,end)); % partial slip

U_0p(:,:) = alphap0.*U_0p_h(:,:) + U_0p_nh(:,:); %general solution as linear combination of nohom and hom
D_0p(:,:) = alphap0.*D_0p_h(:,:) + D_0p_nh(:,:);


%% solve first order state

M_U0_k1 = zeros(Np,Np); M_D0_k1 = zeros(Np,Np); %matrices for convolutions
M_U0_k2 = zeros(Np,Np); M_D0_k2 = zeros(Np,Np); %matrices for convolutions
M_U0_k4 = zeros(Np,Np); M_D0_k4 = zeros(Np,Np); %matrices for convolutions

for j = 2:length(z)

    for ir = 1:Np %create matrices for convolutions, k1, k2/3, k4 are evaluated at different z
        M_U0_k1(ir,:) = [zeros(1,max(0,p_vec(ir))) U_0p(min(Np,ir + P):-1:max(1,ir - P),j-1).' ...
            zeros(1,max(0,p_vec(Np+1-ir)))]; 
        M_D0_k1(ir,:) = [zeros(1,max(0,p_vec(ir))) D_0p(min(Np,ir + P):-1:max(1,ir - P),j-1).'/Av_0(j-1) ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
        
        M_U0_k2(ir,:) = [zeros(1,max(0,p_vec(ir))) ...
            (U_0p(min(Np,ir + P):-1:max(1,ir - P),j-1).'+U_0p(min(Np,ir + P):-1:max(1,ir - P),j).')*0.5 ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
        M_D0_k2(ir,:) = [zeros(1,max(0,p_vec(ir))) ...
            (D_0p(min(Np,ir + P):-1:max(1,ir - P),j-1).'/Av_0(j-1)+D_0p(min(Np,ir + P):-1:max(1,ir - P),j).'/Av_0(j))*0.5 ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
        
        M_U0_k4(ir,:) = [zeros(1,max(0,p_vec(ir))) U_0p(min(Np,ir + P):-1:max(1,ir - P),j).' zeros(1,max(0,p_vec(Np+1-ir)))];
        M_D0_k4(ir,:) = [zeros(1,max(0,p_vec(ir))) D_0p(min(Np,ir + P):-1:max(1,ir - P),j).'/Av_0(j) ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
    end

    for iq = 1:2*Np %solve for each situation with non zero Z1p or U1p (hom sols)
        if iq <= Np
            Z_1p_h(iq,iq) = 1;
        else
            U_1p_h(iq-Np,1,iq) = 1;
        end
        %hom
        k1_U = D_1p_h(:,j-1,iq)/Av_0(j-1);
        k1_D = 1i*p_vec.*U_1p_h(:,j-1,iq) + r*(1i*k*M_U0_k1*U_1p_h(:,j-1,iq) + ...
            M_D0_k1*W_1p_h(:,j-1,iq) + 1i*k*Z_1p_h(:,iq));
        k1_W = -1i*k*U_1p_h(:,j-1,iq);
        k2_U = (D_1p_h(:,j-1,iq) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
        k2_D = 1i*p_vec.*(U_1p_h(:,j-1,iq) - dz*0.5*k1_U) + r*(1i*k*M_U0_k2*(U_1p_h(:,j-1,iq) - dz*0.5*k1_U) + ...
            M_D0_k2*(W_1p_h(:,j-1,iq) - dz*0.5*k1_W) + 1i*k*Z_1p_h(:,iq));
        k2_W = -1i*k*(U_1p_h(:,j-1,iq)-dz*k1_U*0.5);
        k3_U = (D_1p_h(:,j-1,iq) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
        k3_D = 1i*p_vec.*(U_1p_h(:,j-1,iq) - dz*0.5*k2_U) + r*(1i*k*M_U0_k2*(U_1p_h(:,j-1,iq) - dz*0.5*k2_U) + ...
            M_D0_k2*(W_1p_h(:,j-1,iq) - dz*0.5*k2_W) + 1i*k*Z_1p_h(:,iq));
        k3_W = -1i*k*(U_1p_h(:,j-1,iq)-dz*k2_U*0.5);
        k4_U = (D_1p_h(:,j-1,iq) - dz*k3_D)/Av_0(j);
        k4_D = 1i*p_vec.*(U_1p_h(:,j-1,iq) - dz*k3_U) + r*(1i*k*M_U0_k4*(U_1p_h(:,j-1,iq) - dz*k3_U) + ...
            M_D0_k4*(W_1p_h(:,j-1,iq) - dz*k3_W) + 1i*k*Z_1p_h(:,iq));
        k4_W = -1i*k*(U_1p_h(:,j-1,iq)-dz*k3_U);

        U_1p_h(:,j,iq) = U_1p_h(:,j-1,iq) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);
        D_1p_h(:,j,iq) = D_1p_h(:,j-1,iq) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);
        W_1p_h(:,j,iq) = W_1p_h(:,j-1,iq) - dz/6*(k1_W + 2*k2_W + 2*k3_W + k4_W);
    end

    %nonhom
    k1_U = D_1p_nh(:,j-1)/Av_0(j-1);
    if j == 2 %evaluate derivative at surface, second order scheme
        k1_D = 1i*p_vec.*U_1p_nh(:,j-1) + r*(1i*k*M_U0_k1*U_1p_nh(:,j-1) + ...
            M_D0_k1*W_1p_nh(:,j-1) + 1i*k*Z_1p_nh) - ...
            (3*Av_1(j-1)/Av_0(j-1)*D_0p(:,j-1) + Av_1(j+1)/Av_0(j+1)*D_0p(:,j+1) - ...
            4*Av_1(j)/Av_0(j)*D_0p(:,j))/(2*dz);
    else
        k1_D = 1i*p_vec.*U_1p_nh(:,j-1) + r*(1i*k*M_U0_k1*U_1p_nh(:,j-1) + ...
            M_D0_k1*W_1p_nh(:,j-1) + 1i*k*Z_1p_nh) - 0.5/dz*(Av_1(j-2)*D_0p(:,j-2)/Av_0(j-2) - Av_1(j)*D_0p(:,j)/Av_0(j));
    end
    k1_W = -1i*k*U_1p_nh(:,j-1);
    k2_U = (D_1p_nh(:,j-1) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k2_D = 1i*p_vec.*(U_1p_nh(:,j-1) - dz*0.5*k1_U) + r*(1i*k*M_U0_k2*(U_1p_nh(:,j-1) - dz*0.5*k1_U) + ...
        M_D0_k2*(W_1p_nh(:,j-1) - dz*0.5*k1_W) + 1i*k*Z_1p_nh) - 1/dz*(Av_1(j-1)*D_0p(:,j-1)/Av_0(j-1) - Av_1(j)*D_0p(:,j)/Av_0(j));
    k2_W = -1i*k*(U_1p_nh(:,j-1)-dz*k1_U*0.5);
    k3_U = (D_1p_nh(:,j-1) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k3_D = 1i*p_vec.*(U_1p_nh(:,j-1) - dz*0.5*k2_U) + r*(1i*k*M_U0_k2*(U_1p_nh(:,j-1) - dz*0.5*k2_U) + ...
        M_D0_k2*(W_1p_nh(:,j-1) - dz*0.5*k2_W) + 1i*k*Z_1p_nh) - 1/dz*(Av_1(j-1)*D_0p(:,j-1)/Av_0(j-1) - Av_1(j)*D_0p(:,j)/Av_0(j));
    k3_W = -1i*k*(U_1p_nh(:,j-1)-dz*k2_U*0.5);
    k4_U = (D_1p_nh(:,j-1) - dz*k3_D)/Av_0(j);
    if j == length(z) %evalluate derivative at seabed, second order shceme
        k4_D = 1i*p_vec.*(U_1p_nh(:,j-1) - dz*k3_U) + r*(1i*k*M_U0_k4*(U_1p_nh(:,j-1) - dz*k3_U) + ...
            M_D0_k4*(W_1p_nh(:,j-1) - dz*k3_W) + 1i*k*Z_1p_nh) - (4*Av_1(j-1)/Av_0(j-1)*D_0p(:,j-1) - ...
            Av_1(j-2)/Av_0(j-2)*D_0p(:,j-2) - 3*Av_1(j)/Av_0(j)*D_0p(:,j))/(2*dz);
    else
        k4_D = 1i*p_vec.*(U_1p_nh(:,j-1) - dz*k3_U) + r*(1i*k*M_U0_k4*(U_1p_nh(:,j-1) - dz*k3_U) + ...
            M_D0_k4*(W_1p_nh(:,j-1) - dz*k3_W) + 1i*k*Z_1p_nh) - 0.5/dz*(Av_1(j-1)*D_0p(:,j-1)/Av_0(j-1) - ...
            Av_1(j+1)*D_0p(:,j+1)/Av_0(j+1));
    end
    k4_W = -1i*k*(U_1p_nh(:,j-1)-dz*k3_U);

    U_1p_nh(:,j) = U_1p_nh(:,j-1) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);
    D_1p_nh(:,j) = D_1p_nh(:,j-1) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);    
    W_1p_nh(:,j) = W_1p_nh(:,j-1) - dz/6*(k1_W + 2*k2_W + 2*k3_W + k4_W); 
end

% solve alphap A_alpha*alphap = b_alpha
A_alpha = zeros(2*Np,2*Np); b_alpha = zeros(2*Np,1);
for ir = 1:2*Np %row
    for ic = 1:2*Np %column
        if ir <= Np
            A_alpha(ir,ic) = W_1p_h(ir,end,ic);
        else
            A_alpha(ir,ic) = D_1p_h(ir-Np,end,ic) - s*U_1p_h(ir-Np,end,ic);
        end
    end
    if ir <= Np
        b_alpha(ir) = U_0p(ir,end)*1i*k*h1 - W_1p_nh(ir,end);
    else
        b_alpha(ir) = s*U_1p_nh(ir-Np,end) + D_0p(ir-Np,end)/Av_0(end)*(s*h1 - Av_1(end)) - ...
            h1*(4*D_0p(ir-Np,end-1) - D_0p(ir-Np,end-2) -3*D_0p(ir-Np,end))/(2*dz) - D_1p_nh(ir-Np,end);
    end
end
alphap1 = A_alpha\b_alpha;

%general solution for U_1p, W_1p, D_1p, defining also Z_1p
for ip = 1:Np
    U_1p(ip,:) = U_1p_nh(ip,:); %general solution as linear combination of nohom and hom
    W_1p(ip,:) = W_1p_nh(ip,:);
    D_1p(ip,:) = D_1p_nh(ip,:); 
    Z_1p(ip) = Z_1p_nh(ip); 
    for iq = 1:2*Np
        U_1p(ip,:) = U_1p(ip,:) + alphap1(iq)*U_1p_h(ip,:,iq); 
        W_1p(ip,:) = W_1p(ip,:) + alphap1(iq)*W_1p_h(ip,:,iq);
        D_1p(ip,:) = D_1p(ip,:) + alphap1(iq)*D_1p_h(ip,:,iq);
        Z_1p(ip) = Z_1p(ip) + alphap1(iq)*Z_1p_h(ip,iq);
    end
end

%% solve for spatially invariant part of second order state

M_W1_k1 = zeros(Np,Np); M_W1_k2 = zeros(Np,Np); %matrices for convolutions W1*D1
M_W1_k4 = zeros(Np,Np); 

for j = 2:length(z)

    for ir = 1:Np %create matrices for convolutions W1*D1; k1, k2/3, k4 are evaluated at different z
        M_W1_k1(ir,:) = [zeros(1,max(0,p_vec(ir))) W_1p(min(Np,ir + P):-1:max(1,ir - P),j-1).' ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
        M_W1_k2(ir,:) = [zeros(1,max(0,p_vec(ir))) ...
            (W_1p(min(Np,ir + P):-1:max(1,ir - P),j-1).' + W_1p(min(Np,ir + P):-1:max(1,ir - P),j).')*0.5 ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
        M_W1_k4(ir,:) = [zeros(1,max(0,p_vec(ir))) W_1p(min(Np,ir + P):-1:max(1,ir - P),j).' ...
            zeros(1,max(0,p_vec(Np+1-ir)))];
    end

    %hom
    for iq = 1:Np %solve for each situation with non zero U1p (hom sols)
        U_2p_h(iq,1,iq) = 1;

        k1_U = D_2p_h(:,j-1,iq)/Av_0(j-1);
        k1_D = 1i*p_vec.*U_2p_h(:,j-1,iq);
        k2_U = (D_2p_h(:,j-1,iq) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
        k2_D = 1i*p_vec.*(U_2p_h(:,j-1,iq) - dz*0.5*k1_U);
        k3_U = (D_2p_h(:,j-1,iq) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
        k3_D = 1i*p_vec.*(U_2p_h(:,j-1,iq) - dz*0.5*k2_U);
        k4_U = (D_2p_h(:,j-1,iq) - dz*k3_D)/Av_0(j);
        k4_D = 1i*p_vec.*(U_2p_h(:,j-1,iq) - dz*k3_U);

        U_2p_h(:,j,iq) = U_2p_h(:,j-1,iq) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);
        D_2p_h(:,j,iq) = D_2p_h(:,j-1,iq) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);
    end

    %nonhom
    k1_U = D_2p_nh(:,j-1)/Av_0(j-1);
    if j == 2 %evaluate derivative at surface, second order scheme
        k1_D = 1i*p_vec.*U_2p_nh(:,j-1) - (3*Av_2(j-1)*D_0p(:,j-1)/Av_0(j-1) + ...
            Av_2(j+1)*D_0p(:,j+1)/Av_0(j+1) - 4*Av_2(j)*D_0p(:,j)/Av_0(j))/(2*dz) + ...
            r/Av_0(j-1)*(M_W1_k1*conj(D_1p(end:-1:1,j-1)) + M_W1_k1'*D_1p(:,j-1)) - ...
            (3*conj(Av_1(j-1))*D_1p(:,j-1)/Av_0(j-1) + ...
            conj(Av_1(j+1))*D_1p(:,j+1)/Av_0(j+1) - 4*conj(Av_1(j))*D_1p(:,j)/Av_0(j))/(2*dz) - ...
            (3*Av_1(j-1)*conj(D_1p(end:-1:1,j-1))/Av_0(j-1) + ...
            Av_1(j+1)*conj(D_1p(end:-1:1,j+1))/Av_0(j+1) - 4*Av_1(j)*conj(D_1p(end:-1:1,j))/Av_0(j))/(2*dz);
    else
        k1_D = 1i*p_vec.*U_2p_nh(:,j-1) - 0.5/dz*(Av_2(j-2)/Av_0(j-2)*D_0p(:,j-2) - ...
            Av_2(j)/Av_0(j)*D_0p(:,j)) + ...
            r/Av_0(j-1)*(M_W1_k1*conj(D_1p(end:-1:1,j-1)) + M_W1_k1'*D_1p(:,j-1)) - ...
            0.5/dz*(conj(Av_1(j-2))/Av_0(j-2)*D_1p(:,j-2) - conj(Av_1(j))/Av_0(j)*D_1p(:,j)) - ...
            0.5/dz*(Av_1(j-2)/Av_0(j-2)*conj(D_1p(end:-1:1,j-2)) - Av_1(j)/Av_0(j)*conj(D_1p(end:-1:1,j)));
    end
    k2_U = (D_2p_nh(:,j-1) - dz*k1_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k2_D = 1i*p_vec.*(U_2p_nh(:,j-1) - dz*0.5*k1_U) - 1/dz*(Av_2(j-1)/Av_0(j-1)*D_0p(:,j-1) - ...
        Av_2(j)/Av_0(j)*D_0p(:,j)) + ...
        r/((Av_0(j-1)+Av_0(j))*0.5)*(M_W1_k2*conj(D_1p(end:-1:1,j-1) + D_1p(end:-1:1,j))*0.5 + M_W1_k2'*(D_1p(:,j-1) + D_1p(:,j))*0.5) - ...
        1/dz*(conj(Av_1(j-1))/Av_0(j-1)*D_1p(:,j-1) - conj(Av_1(j))/Av_0(j)*D_1p(:,j)) - ...
        1/dz*(Av_1(j-1)/Av_0(j-1)*conj(D_1p(end:-1:1,j-1)) - Av_1(j)/Av_0(j)*conj(D_1p(end:-1:1,j)));
    k3_U = (D_2p_nh(:,j-1) - dz*k2_D*0.5)/(0.5*(Av_0(j-1)+Av_0(j)));
    k3_D = 1i*p_vec.*(U_2p_nh(:,j-1) - dz*0.5*k2_U) - 1/dz*(Av_2(j-1)/Av_0(j-1)*D_0p(:,j-1) - ...
        Av_2(j)/Av_0(j)*D_0p(:,j)) + ...
        r/((Av_0(j-1)+Av_0(j))*0.5)*(M_W1_k2*conj(D_1p(end:-1:1,j-1) + D_1p(end:-1:1,j))*0.5 + M_W1_k2'*(D_1p(:,j-1) + D_1p(:,j))*0.5) - ...
        1/dz*(conj(Av_1(j-1))/Av_0(j-1)*D_1p(:,j-1) - conj(Av_1(j))/Av_0(j)*D_1p(:,j)) - ...
        1/dz*(Av_1(j-1)/Av_0(j-1)*conj(D_1p(end:-1:1,j-1)) - Av_1(j)/Av_0(j)*conj(D_1p(end:-1:1,j)));
    k4_U = (D_2p_nh(:,j-1) - dz*k3_D)/Av_0(j);
    if j == length(z) %evaluate derivative at seabed, second order scheme
        k4_D = 1i*p_vec.*(U_2p_nh(:,j-1) - dz*k3_U) - (-3*Av_2(j)*D_0p(:,j)/Av_0(j) - ...
            Av_2(j-2)*D_0p(:,j-2)/Av_0(j-2) + 4*Av_2(j-1)*D_0p(:,j-1)/Av_0(j-1))/(2*dz) + ...
            r/Av_0(j)*(M_W1_k4*conj(D_1p(end:-1:1,j)) + M_W1_k4'*D_1p(:,j)) - ...
            (-3*conj(Av_1(j))*D_1p(:,j)/Av_0(j) - ...
            conj(Av_1(j-2))*D_1p(:,j-2)/Av_0(j-2) + 4*conj(Av_1(j-1))*D_1p(:,j-1)/Av_0(j-1))/(2*dz) - ...
            (-3*Av_1(j)*conj(D_1p(end:-1:1,j))/Av_0(j) - ...
            Av_1(j-2)*conj(D_1p(end:-1:1,j-2))/Av_0(j-2) + 4*Av_1(j-1)*conj(D_1p(end:-1:1,j-1))/Av_0(j-1))/(2*dz);
    else
        k4_D = 1i*p_vec.*(U_2p_nh(:,j-1) - dz*k3_U) - 0.5/dz*(Av_2(j-1)/Av_0(j-1)*D_0p(:,j-1) - ...
            Av_2(j+1)/Av_0(j+1)*D_0p(:,j+1)) + ...
            r/Av_0(j)*(M_W1_k4*conj(D_1p(end:-1:1,j)) + M_W1_k4'*D_1p(:,j)) - ...
            0.5/dz*(conj(Av_1(j-1))/Av_0(j-1)*D_1p(:,j-1) - conj(Av_1(j+1))/Av_0(j+1)*D_1p(:,j+1)) - ...
            0.5/dz*(Av_1(j-1)/Av_0(j-1)*conj(D_1p(end:-1:1,j-1)) - Av_1(j+1)/Av_0(j+1)*conj(D_1p(end:-1:1,j+1)));
    end

    U_2p_nh(:,j) = U_2p_nh(:,j-1) - dz/6*(k1_U + 2*k2_U + 2*k3_U + k4_U);
    D_2p_nh(:,j) = D_2p_nh(:,j-1) - dz/6*(k1_D + 2*k2_D + 2*k3_D + k4_D);    
end

% solve alphap A_alpha*alphap = b_alpha
A_alpha = zeros(Np,Np); b_alpha = zeros(Np,1);
for ir = 1:Np %row
    b_alpha(ir) = - D_2p_nh(ir,end) + s*U_2p_nh(ir,end) + h1*s/Av_0(end)*conj(D_1p(end-ir+1,end)) + ...
        conj(h1)*s/Av_0(end)*D_1p(ir,end) + 2*h1*conj(h1)*s*(4*D_0p(ir,end-1)/Av_0(end-1) - ...
        D_0p(ir,end-2)/Av_0(end-2) - 3*D_0p(ir,end)/Av_0(end))/(2*dz) - ...
        Av_1(end)/Av_0(end)*conj(D_1p(end-ir+1,end)) - conj(Av_1(end))/Av_0(end)*D_1p(ir,end) - ...
        Av_2(end)/Av_0(end)*D_0p(ir,end) - h1*(4*conj(Av_1(end-1))/Av_0(end-1)*D_0p(ir,end-1) - ...
        conj(Av_1(end-2))/Av_0(end-2)*D_0p(ir,end-2) - 3*conj(Av_1(end))/Av_0(end)*D_0p(ir,end))/(2*dz) - ...
        conj(h1)*(4*Av_1(end-1)/Av_0(end-1)*D_0p(ir,end-1) - Av_1(end-2)/Av_0(end-2)*D_0p(ir,end-2) - ...
        3*Av_1(end)/Av_0(end)*D_0p(ir,end))/(2*dz) - h1*(conj(4*D_1p(end-ir+1,end-1) - D_1p(end-ir+1,end-2) - ...
        3*D_1p(end-ir+1,end)))/(2*dz) - conj(h1)*(4*D_1p(ir,end-1) - D_1p(ir,end-2) - ...
        3*D_1p(ir,end))/(2*dz) - 2*h1*conj(h1)*(D_0p(ir,end-2) - 2*D_0p(ir,end-1) + D_0p(ir,end))/dz^2;
    for ic = 1:Np %column
        A_alpha(ir,ic) = D_2p_h(ir,end,ic) - s*U_2p_h(ir,end,ic);
    end
end
alphap2 = A_alpha\b_alpha;

%general solution for U_1p, W_1p, D_1p, defining also Z_1p
for ip = 1:Np
    U_2p(ip,:) = U_2p_nh(ip,:); %general solution as linear combination of nohom and hom
    D_2p(ip,:) = D_2p_nh(ip,:); 
    for iq = 1:Np
        U_2p(ip,:) = U_2p(ip,:) + alphap2(iq)*U_2p_h(ip,:,iq); 
        D_2p(ip,:) = D_2p(ip,:) + alphap2(iq)*D_2p_h(ip,:,iq);
    end
end

end