from __future__ import print_function
import logging
import sqlalchemy as sa
from collections import namedtuple
import datetime
import traceback
import os

# datatypes
# pairs of cast function and sqlalchemy column type
DATA_TYPES = dict(
    int=(int, sa.Integer),
    str=(str, sa.String),
    datetime=(datetime.datetime.fromtimestamp, sa.DateTime)
)

class DatabaseHandler(logging.Handler):

    attrs = [
        dict(name='name',       type=DATA_TYPES['str']),
        dict(name='levelno',    type=DATA_TYPES['int']),
        dict(name='levelname',  type=DATA_TYPES['str']),
        dict(name='pathname',   type=DATA_TYPES['str']),
        dict(name='lineno',     type=DATA_TYPES['int']),
        dict(name='threadName', type=DATA_TYPES['str']),
        dict(name='processName',type=DATA_TYPES['str']),
        dict(name='msg',        type=DATA_TYPES['str']),
        dict(name='args',       type=DATA_TYPES['str']),
        dict(name='created',    type=DATA_TYPES['datetime']),
        dict(name='exc_text',   type=DATA_TYPES['str'])
    ]

    def __init__(self, db, level=logging.NOTSET, reset=False):
        self.db = db
        self.setup_db(reset=reset)
        super(DatabaseHandler, self).__init__(level=level)

    def setup_db(self, reset):
        self.engine = sa.create_engine(self.db)
        self.connection = self.engine.connect()
        metadata = sa.MetaData(self.connection)

        self.record_table = sa.Table(
            'records', metadata,
            sa.Column('id', sa.Integer, primary_key=True),
            *[sa.Column(attr['name'], attr['type'][1]) for attr in self.attrs])

        if reset:
            self.record_table.drop(self.engine, checkfirst=True)

        self.record_table.create(checkfirst=True)

    def connect(self):
        self.engine.connect()

    def close(self):
        self.connection.close()

    def emit(self, record):
        data = {}
        for attr in self.attrs:
            if attr['name'] == 'exc_text':
                if record.exc_info:
                    v = ''.join(traceback.format_exception(*record.exc_info))
                else:
                    v = ''
            else:
                v = getattr(record, attr['name'])

            try:
                v = attr['type'][0](v)
            except TypeError as e:
                raise TypeError('cannot cast {!r} of type {} to {}'.format(v, type(v).__name__, attr['type'][0]))
            data[attr['name']] = v

        self.record_table.insert().values(**data).execute()

    def iter_all(self, minlevel=logging.DEBUG):
        if isinstance(minlevel, str):
            minlevel = getattr(logging, minlevel)
        Row = namedtuple('LogRecordRow', ['id']+[attr['name'] for attr in self.attrs])
        for row in self.record_table.select().order_by('created').execute():
            if row.levelno >= minlevel:
                yield Row(*row.values())

    def iter_records(self, minlevel=logging.DEBUG):
        if isinstance(minlevel, str):
            minlevel = getattr(logging, minlevel)
        for row in self.record_table.select().order_by('created').execute():
            if row.levelno >= minlevel:
                yield logging.makeLogRecord(dict(row))


class SQLiteDatabaseHandler(DatabaseHandler):

    def __init__(self, db, *args, **kwargs):
        if db == ':memory:':
            dbpath = 'sqlite:///{}'.format(db)
        else:
            dbpath = 'sqlite:///{}'.format(os.path.abspath(db))
        super(SQLiteDatabaseHandler, self).__init__(dbpath, *args, **kwargs)


if __name__ == '__main__':
    logging.basicConfig(level='INFO')

    logger = logging.getLogger(__name__)
    sqlhandler = DatabaseHandler('sqlite:///logging.db')
    logger.addHandler(sqlhandler)

    logging.getLogger(__name__).info('testmessage')
    logging.getLogger(__name__).warning('testmessage')
    try:
        raise ValueError('test')
    except ValueError as e:
        logging.getLogger(__name__).exception('testmessage')

    sqlhandler.close()

    sqlhandler = DatabaseHandler('sqlite:///logging.db')

    for row in sqlhandler.iter_records(minlevel='DEBUG'):
        print(row.levelname, row.msg)
        print(row.exc_text)

