from __future__ import print_function, division
import numpy as np
import warnings


def k_residual(omega, k, dep):
    """
    calculate residual of the wave number iteration
    using $$\left(\frac{2\pi}{T}\right)^2=gk\,tanh(kh)$$
    :param omega: radian frequency
    :param k: wave number
    :param dep: depth
    :return: difference in wave number
    """
    return omega**2 - 9.81 * k * np.tanh(k * dep)


def k_iter(dep, T):
    """
    iteratively determine the wave number from depth and wave period
    using $$\left(\frac{2\pi}{T}\right)^2=gk\,tanh(kh)$$
    :param dep: depth (m)
    :param T: wave period (s)
    :return: wave number
    """
    if T == 0:
        return np.nan
    omega = np.pi * 2 / T
    res_norm = 1e-15
    res = res_norm + 1
    k = 2 * np.pi / 80
    i = 0
    while np.abs(res) > res_norm:
        i += 1
        res = k_residual(omega, k, dep)
        d_res_dk = -9.81 * np.tanh(k * dep) - dep * k * 9.81 * (1 - np.tanh(k * dep)**2)
        dk = -res / d_res_dk
        k += dk

        if i == 1000:
            warnings.warn('wave number did not converge', UserWarning)
            return np.nan

    return k
k_iter = np.vectorize(k_iter)


def orbital_bed_velocity(hsig, T, dep):
    """
    calculate the orbital velocity at the bed
    using $$u_{orb} = \frac{\pi H}{T\,sinh(kh)}$$
    :param hsig: significant wave height
    :param T: wave period
    :param dep: water depth
    :return: orbital velocity at the bed
    """
    return np.pi * hsig / (T * np.sinh(k_iter(dep, T) * dep))


def orbital_diameter(hsig, T, dep):
    """
    calculate the horizontal orbital diameter
    using $$A_{orb} = \frac{H}{sinh(kh)}$$
    :param hsig: significant wave height
    :param T: wave period
    :param dep: water depth
    :return: orbital diameter
    """
    return hsig / np.sinh(k_iter(dep, T) * dep)


def tau_waves(hsig, T, depth, d50):
    """
    calculate the shear stress at the bed due to waves
    using $$\tau=\rho_wu^2_{orb}exp\left[5.213\left(\frac{2.5D_{50}}{A_{orb}}\right)^{0.194}-5.977\right]$$
    :param hsig: significant wave height
    :param T: wave period
    :param dep: water depth
    :param d50: grain size
    :return: shear stress
    """
    uorb = orbital_bed_velocity(hsig, T, depth)
    if not isinstance(uorb, np.ndarray):
        uorb = np.array([uorb])
    uorb[uorb < .001] = np.nan
    return 1000 * uorb**2 * np.exp(-5.977 + 5.213*(2.5 * d50 * 2 / orbital_diameter(hsig, T, depth))**.194)


def shields(hsig, T, depth, d50):
    """
    calculate the shields parameter due to waves
    using $$\Theta=\frac{\tau}{\left(\rho_s-rho_w\right)gD_{50}}$$
    :param hsig: significant wave height
    :param T: wave period
    :param dep: water depth
    :param d50: grain size
    :return: shields parameter
    """
    return tau_waves(hsig, T, depth, d50)/((2650-1000)*9.81*d50)
