"""
Module for landboundary datasets
Datasets consist of space-separated textfiles consisting of line pieces
Each line defines a point (UTM31) and a line is defined as subsequent lines with points
New line pieces are indicated with 'NaN' for the x and y positions

The location of predefined datasets is stored in the variable `datasets`
"""

import numpy as np
from .. import settings


__all__ = ['datasets', 'BoundaryDataset', 'LazyBoundaryDataset', 'NL']


datasets = dict(NL=settings.landboundary_NL)


class BoundaryDataset(object):

    def __init__(self, lines):
        self.x, self.y = lines[:, 0], lines[:, 1]

    @classmethod
    def load(cls, filename):
        rows = []
        with open(filename, 'r') as f:
            for line in f:
                x, y = line.strip().split()
                if x == 'NaN' or y == 'NaN':
                    x, y = np.nan, np.nan
                rows.append([float(x), float(y)])
        return cls(np.ma.masked_invalid(rows))

    def plot(self, ax, scale=1, **kwargs):
        ax.plot(self.x*scale, self.y*scale, **kwargs)


class LazyBoundaryDataset(object):

    def __init__(self, name):
        self.name = name

    def plot(self, ax, **kwargs):
        return BoundaryDataset.load(datasets[self.name]).plot(ax, **kwargs)


NL = LazyBoundaryDataset('NL')


if __name__ == '__main__':
    from matplotlib import pyplot as plt
    NL().plot(plt.gca(), color='k')
    plt.show()

    LazyBoundaryDataset('NL').plot(plt.gca(), color='k')
    plt.show()