import logging
import os
import numpy as np
from ..utils import *


def save_xyz_points(fh, data, columns=None, meta=None, **kw):
    from ...pointdata import PointDataset
    D = PointDataset(data, columns, **(meta or {}))
    D.save(fh, **kw)


def save_csv_points(fh, D, columns=None, meta=None, **kw):
    kw.setdefault('delimiter', ',')
    return save_xyz_points(fh, D, columns=columns, **kw)


def save_asc_grid(fh, x, y, Z, meta=None, **kw):
    import textwrap
    if (x.ndim, y.ndim, Z.ndim) != (1, 1, 2):
        raise ValueError('invalid input data shape')
    rows, cols = Z.shape
    fmt = textwrap.dedent('''\
    ncols  {ncols:.0f}
    nrows  {nrows:.0f}
    xllcorner  {xll:.1f}
    yllcorner  {yll:.1f}
    cellsize  {cellsize:.1f}
    NODATA_value  {nodata}
    ''')

    Z = Z.filled(-999.99)

    header = fmt.format(
        nrows=rows,
        ncols=cols,
        xll=x[0],
        yll=y[0],
        cellsize=(x[1] - x[0]),
        nodata='-999.99')

    fh.write(header.encode('ascii'))

    np.savetxt(fh, np.flipud(Z), fmt='%.5g', delimiter=' ', newline='\n\n\n', **kw)


def save_npz_grid(fh, x, y, Z, meta=None, **kw):
    if isinstance(Z, np.ma.MaskedArray):
        Z = Z.filled(np.nan)
    np.savez(fh, x=x, y=y, Z=Z, meta=meta, **kw)


def save_nc_grid(fh, x, y, Z, meta=None, **kw):
    raise NotImplementedError('save_nc_grid')


def save_nc_gridstack(fh, x, y, t, Z, meta=None, **kw):
    raise NotImplementedError('save_nc_gridstack')


def save_npz_gridstack(fh, x, y, t, Z, meta=None, **kw):
    if isinstance(Z, np.ma.MaskedArray):
        Z = Z.filled(np.nan)
    np.savez(fh, x=x, y=y, t=t, Z=Z, meta=meta, **kw)


POINT_FTYPES = dict(
    XYZ=save_xyz_points,  # space separated, no headers: assume [x y z t0 t1]
    CSV=save_csv_points,  # comma separated with headers
)


GRID_FTYPES = dict(
    ASC=save_asc_grid,
    NC=save_nc_grid,
    NPZ=save_npz_grid
)


GRIDSTACK_FTYPES = dict(
    NC=save_nc_gridstack,
    NPZ=save_npz_gridstack
)


def as_writable(f, ftype=None, meta=None):
    if ftype == 'NC':
        import netCDF4
        return netCDF4.Dataset(f, 'w')
    elif isinstance(f, str):
        if f.startswith('http'):
            raise IOError('web locations not allowed for writing')
        dirname = os.path.dirname(f)
        if dirname == '' or os.path.isdir(dirname):
            return open(f, 'wb')
        raise ValueError('could not make {!r} writable as {}'.format(f, ftype))
    else:
        try:
            f.write
        except AttributeError:
            raise TypeError('not writable')
        else:
            return f


def _save(f, data, ftype=None, datatype='grid', meta=None, **kw):
    if isinstance(f, str):
        fname = f
    else:
        fname = meta.get('source', kw.get('source', None))

    if ftype is None:
        if fname is not None:
            ftype = os.path.splitext(fname)[1].lstrip('.').upper()

    if not ftype:
        ftype = guess_ftype(f, datatype=datatype)

    logging.getLogger(__name__).debug('loading {} as ftype={}'.format(f, ftype or '<undefined>'))

    if isinstance(f, str):
        f = as_writable(f, ftype=ftype, meta=meta)

    saver = dict(grid=GRID_FTYPES, gridstack=GRIDSTACK_FTYPES, points=POINT_FTYPES)[datatype][ftype]

    kw['meta'] = meta
    return saver(f, *data, **kw)


def save_griddata(f, x, y, Z, ftype=None, meta=None, **kw):
    return _save(f, (x, y, Z), ftype=ftype, datatype='grid', meta=meta, **kw)


def save_gridstackdata(f, x, y, t, Z, ftype=None, meta=None, **kw):
    return _save(f, (x, y, t, Z), ftype=ftype, datatype='gridstack', meta=meta, **kw)


def save_pointdata(f, data, columns, ftype=None, meta=None, **kw):
    return _save(f, (data, columns), ftype=ftype, datatype='points', meta=meta, **kw)
