import pandas as pd
import os


__all__ = ['BatchOutput', 'AppendedCSVOutput', 'SeparateBlocksCSVOutput']


class BatchOutput(object):

    def __init__(self, location=None):
        self.location = location

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        return

    def save(self, data, name):
        pass


class AppendedCSVOutput(BatchOutput):

    """write all output to a single file"""

    def __init__(self, *args, **kwargs):
        super(AppendedCSVOutput, self).__init__(*args, **kwargs)
        self.written = False

    def save(self, data, name, **kw):
        df = pd.DataFrame(data=data._asdict())
        if not self.written:
            df.to_csv(self.location, header=True, index=False, float_format='%.4f', mode='w')
            self.written = True
        else:
            df.to_csv(self.location, header=False, index=False, float_format='%.4f', mode='a')

    def load(self):
        return pd.read_csv(self.location)


class SeparateBlocksCSVOutput(BatchOutput):

    """write the output to separate files within one directory"""

    def __enter__(self):
        if not os.path.isdir(self.location):
            raise IOError('directory {!r} not found'.format(self.location))
        return super(SeparateBlocksCSVOutput, self).__enter__()

    def save(self, data, name, **kw):
        df = pd.DataFrame(data=data._asdict())
        df.to_csv(os.path.join(self.location, name+'.csv'),
                  header=True, index=False, float_format='%.4f', mode='w', sep=',')

    def load(self, exclude=()):
        frames = []
        for f in os.listdir(self.location):
            if not f.endswith('.csv'):
                continue
            if f in exclude:
                continue
            df = pd.read_csv(os.path.join(self.location, f))
            frames.append(df)
        if frames:
            return pd.concat(frames, ignore_index=True)
