import numpy as np
from .transform import Transform


class Radon(Transform):

    def __init__(self, theta, b, data, cellsize=25):
        self.theta = theta
        self.b = b
        self.data = data
        self.cellsize = cellsize

    @classmethod
    def transform(cls, x, y, Z):
        """
        returns a radon transform of the data

        the returned value is is Radon object with the following important attributes:
        .theta     rotations
        .b         offsets
        .data      intensities
        .cellsize  cellsize of the input data

        and the following methods:
        .plot(axes)       plot the radon transform on the supplied axis

        :param x: numpy array (1D) with x coordinates
        :param y: numpy array (1D) with y coordinates
        :param Z: numpy array (2D) with data values (no numpy MaskedArray)
        :return: Radon object
        """
        from skimage.transform import radon

        cellsize = np.abs(x[1] - x[0])
        assert cellsize == np.abs(y[1] - y[0]), 'cellsize not equal in both directions'

        theta = np.linspace(0., 180., max(Z.shape), endpoint=False)
        result = radon(Z, theta=theta) * cellsize
        b = cellsize * np.arange(-result.shape[0]//2, result.shape[0] - result.shape[0]//2)

        return cls(theta, b, result, cellsize=cellsize)

