'''
In this code we do the analysis and visualisation for the figures with DALES. The code-file is a part of the publication:

"Turbulent Exchange of CO2 in the Lower Tropical Troposphere Across Clear-to-Cloudy Conditions"

Author: Vincent S. de Feiter. Contact: vincent.defeiter@wur.nl

Version: 29 April, 2025

'''
#------------------------------------------------------------------------------------------------------------
#                                   I M P O R T    M O D U L E S 
#------------------------------------------------------------------------------------------------------------
#%%
import requests
from datetime import datetime, timedelta
import os
import xarray as xr
import numpy as np
import json
import matplotlib  
import math

# Import necessary matplotlib modules
import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
from mpl_toolkits.axes_grid1.inset_locator import inset_axes
import matplotlib.transforms as transforms
from matplotlib.backends.backend_pdf import PdfPages
from matplotlib.colors import LinearSegmentedColormap, ListedColormap, BoundaryNorm
from matplotlib.colors import LinearSegmentedColormap, Normalize
from matplotlib.ticker import AutoMinorLocator, MaxNLocator
import matplotlib.colors as mcolors
from matplotlib.collections import LineCollection
import matplotlib.image as mpimg
from scipy import stats
from matplotlib import patheffects
from matplotlib import rcParams

# Import other necessary libraries
import pandas as pd
import numpy as np

plt.rcParams.update({
    "text.usetex": True,
    "font.family": "serif",
    "font.serif": ["Computer Modern Roman"],
    "text.latex.preamble": r"\usepackage{amsmath}"
})

fontsize = 15
rcParams['font.size'] = fontsize

# # Update the font sizes for ylabel, xlabel, and tick labels
rcParams['axes.labelsize'] = fontsize   # X and Y labels

# # Optionally, you can scale other elements, such as legend size, here
rcParams['legend.fontsize'] = fontsize
rcParams['axes.titlepad'] = 20   # Title padding
rcParams['axes.labelpad'] = 20   # Axis label padding
rcParams['xtick.major.pad']= 5 
rcParams['ytick.major.pad']= 5

rcParams['axes.edgecolor'] = 'k'  # Color of the plot border
rcParams['axes.linewidth'] = 1  # Width of the plot border (spines)
rcParams['xtick.major.width'] = 1  # Width of major ticks on the x-axis
rcParams['ytick.major.width'] = 1  # Width of major ticks on the y-axis
rcParams['xtick.minor.width'] = 1  # Width of minor ticks on the x-axis
rcParams['ytick.minor.width'] = 1  # Width of minor ticks on the y-axis
rcParams['xtick.major.size'] = 6  # Length of major ticks on the x-axis
rcParams['ytick.major.size'] = 6  # Length of major ticks on the y-axis
rcParams['xtick.minor.size'] = 4   # Length of minor ticks on the x-axis
rcParams['ytick.minor.size'] = 4   # Length of minor ticks on the y-axis

axiswidth = 0.8


def find_closest(arr, val):
       idx = np.abs(arr - val).argmin()
       return idx

class NumpyEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, np.ndarray):
            return obj.tolist()
        return json.JSONEncoder.default(self, obj)

def align_yaxis(ax1, val1, ax2, val2):
    """
    Align val1 on ax1 to match val2 on ax2 (visually at the same height).

    Parameters:
    - ax1: First axis (e.g., main axis)
    - val1: Y-value on ax1 to align
    - ax2: Second axis (e.g., twinx axis)
    - val2: Y-value on ax2 to align with val1
    """
    # Get current limits
    y1_min, y1_max = ax1.get_ylim()
    y2_min, y2_max = ax2.get_ylim()

    # Calculate the position of val1 and val2 in figure coordinates
    inv1 = ax1.transData.transform
    inv2 = ax2.transData.inverted().transform

    # Find display coordinate of val1
    coord = inv1((0, val1))

    # Convert that display coordinate into data coordinate on ax2
    aligned_val2 = inv2(coord)[1]

    # Adjust ax2 limits so val2 appears at the same position as val1
    dy = val2 - aligned_val2
    ax2.set_ylim(y2_min + dy, y2_max + dy)



def legend_maker(axis,nrows,ncols,x,y):
    
    # Extract the handles and labels
    handles, labels = axis.get_legend_handles_labels()

    # Specify the number of rows and columns
    n_rows = nrows
    n_cols = ncols 

    # Rearrange handles and labels
    sorted_handles_labels = []
    for col in range(n_cols):
        for row in range(n_rows):
            index = row * n_cols + col
            if index < len(handles):
                sorted_handles_labels.append((handles[index], labels[index]))

    # Unpack sorted handles and labels
    sorted_handles, sorted_labels = zip(*sorted_handles_labels)

    axis.legend(sorted_handles,sorted_labels,loc='lower left', bbox_to_anchor= (x, y), ncols=n_cols,
                    borderaxespad=0, frameon=False) 

#------------------------------------------------------------------------------------------------------------
#                                   P A T H S  &  S E T T I N G S 
#------------------------------------------------------------------------------------------------------------
#%%

#DALES output directory
DALES_nr = '034'
DALES_rootdir = f'/home/.../R{DALES_nr}/'

DALES_nr_no_clouds = '035'
DALES_rootdir_no_clouds = f'/home/.../R{DALES_nr_no_clouds}/'

#Outdir
outdir = '/home/.../Figures/'

#------------------------------------------------------------------------------------------------------------
#                                           L O A D  D A T A 
#------------------------------------------------------------------------------------------------------------
#%%

#Load data (30 min averages)

#1D
profiles = xr.open_dataset(f'{DALES_rootdir}profiles.{DALES_nr}.nc')
profiles_no_clouds = xr.open_dataset(f'{DALES_rootdir_no_clouds}profiles.{DALES_nr_no_clouds}.nc')

#------------------------------------------------------------------------------------------------------------
#                                  1 D   B U D G E T  -  C L O U D S 
#------------------------------------------------------------------------------------------------------------
#%%

#CO2/dt
CO2 = profiles.sv001/1000 #ppb to ppm
time = profiles.time
z_1D = (profiles.zt.values + 40)

CO2_dt = CO2.differentiate('time',datetime_unit='s')
CO2_dt = CO2_dt.coarsen(time=6, boundary='trim').mean()
new_time = time[::6]
CO2_dt = CO2_dt.assign_coords(time=new_time)

#CO2/dz
flux_t = profiles.wsv001t/1000 #ppb m/s to ppm m/s

wCO2_t_dz = (flux_t.differentiate('zm'))
wCO2_t_dz = wCO2_t_dz.coarsen(time=6, boundary='trim').mean()
new_time = time[::6]
wCO2_t_dz = wCO2_t_dz.assign_coords(time=new_time)

total_residual = wCO2_t_dz.values + CO2_dt.values

CO2_dt_1D = CO2_dt.values
wCO2_t_dz_1D = wCO2_t_dz.values

#------------------------------------------------------------------------------------------------------------
#wCO2 (surface)
df = open(f'/home/.../R{DALES_nr}/tmlsm.{DALES_nr}')
headers = None
data = {}

count = 0
for line in df:
    if count == 0:
        split = line.split(None)[1:]
        headers = split
        for item in headers:
            data[item] = []
        count += 1
    elif count == 1:
        count += 1
        continue
    else:
        split = line.split(None)[:]
        sub = 0
        for item in headers:
            data[item].append(split[sub])
            sub += 1
        count += 1

wCO2_s = np.array(data['wco2']).astype(float)/1.8 #mg C/m2 s to ppm/s
wCO2_s_time = np.array(data['time']).astype(float) #5 minute interval

wCO2_s_2 = profiles.wsv001t.values[:,0]/1000
wCO2_s_ds = xr.DataArray(wCO2_s_2,coords=[profiles.time],dims=['time'])
wCO2_s_ds = wCO2_s_ds.coarsen(time=6,boundary='trim').mean()
new_time = wCO2_s_time[::6]
wCO2_s_ds = wCO2_s_ds.assign_coords(time=new_time)
wCO2_s = wCO2_s_ds.values

#------------------------------------------------------------------------------------------------------------
#g (surface)
gCO2_s = np.array(data['gcco2']).astype(float)
gCO2_s_time = np.array(data['time']).astype(float) #5 minute interval

gCO2_s_ds = xr.DataArray(gCO2_s,coords=[gCO2_s_time],dims=['time'])
gCO2_s_ds = gCO2_s_ds.coarsen(time=6,boundary='trim').mean()
new_time = gCO2_s_time[::6]
gCO2_s_ds = gCO2_s_ds.assign_coords(time=new_time)
gCO2_s = gCO2_s_ds.values

#------------------------------------------------------------------------------------------------------------
#Resp (surface)
respCO2_s = np.array(data['Resp']).astype(float)
respCO2_s_time = np.array(data['time']).astype(float) #5 minute interval

respCO2_s_ds = xr.DataArray(respCO2_s,coords=[respCO2_s_time],dims=['time'])
respCO2_s_ds = respCO2_s_ds.coarsen(time=6,boundary='trim').mean()
new_time = respCO2_s_time[::6]
respCO2_s_ds = respCO2_s_ds.assign_coords(time=new_time)
respCO2_s = respCO2_s_ds.values

#------------------------------------------------------------------------------------------------------------
#ra (surface)
ra_s = np.array(data['ra']).astype(float)
ra_s_time = np.array(data['time']).astype(float) #5 minute interval

ra_s_ds = xr.DataArray(ra_s,coords=[ra_s_time],dims=['time'])
ra_s_ds = ra_s_ds.coarsen(time=6,boundary='trim').mean()
new_time = ra_s_time[::6]
ra_s_ds = ra_s_ds.assign_coords(time=new_time)
ra_s = ra_s_ds.values

#------------------------------------------------------------------------------------------------------------
#rsCO2 (surface)
rsCO2_s = np.array(data['rs']).astype(float)*1.6 #rwater to rsCO2
rsCO2_s_time = np.array(data['time']).astype(float) #5 minute interval

rsCO2_s_ds = xr.DataArray(rsCO2_s,coords=[rsCO2_s_time],dims=['time'])
rsCO2_s_ds = rsCO2_s_ds.coarsen(time=6,boundary='trim').mean()
new_time = rsCO2_s_time[::6]
rsCO2_s_ds = rsCO2_s_ds.assign_coords(time=new_time)
rsCO2_s = rsCO2_s_ds.values

#------------------------------------------------------------------------------------------------------------
#An (surface)
An_s = np.array(data['An']).astype(float)
An_s_time = np.array(data['time']).astype(float) #5 minute interval

An_s_ds = xr.DataArray(An_s,coords=[An_s_time],dims=['time'])
An_s_ds = An_s_ds.coarsen(time=6,boundary='trim').mean()
new_time = An_s_time[::6]
An_s_ds = An_s_ds.assign_coords(time=new_time)
An_s = An_s_ds.values

#------------------------------------------------------------------------------------------------------------
#Boundary layer height - 1D
wthvt = profiles.wthvt
time_ref = profiles.time.values

abl_1D = []
idx_1D = []
for i in range(len(time_ref)):
    values = wthvt.sel(time=time_ref[i]).values
    idx = find_closest(values, np.min(values))
    
    idx = np.where(values == np.min(values))[0][0]
    abl_val = profiles.zt[idx].values+40
    
    if abl_val == 0 or abl_val > 2000 or abl_val < 500 or i<10:
        abl_val = np.nan
        idx = np.nan
    
    abl_1D.append(abl_val)
    idx_1D.append(idx)

abl_1D_ds = xr.DataArray(abl_1D,coords=[profiles.time],dims=['time'])
abl_1D_ds = abl_1D_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
abl_1D_ds = abl_1D_ds.assign_coords(time=new_time)

idx_1D_ds = xr.DataArray(idx_1D,coords=[profiles.time],dims=['time'])
idx_1D_ds = idx_1D_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
idx_1D_ds = idx_1D_ds.assign_coords(time=new_time)

#Reference time
ref_time_1D = pd.to_datetime(wCO2_s_ds.time.values, unit='s', origin='unix')
ref_time_1D  = (pd.to_datetime(ref_time_1D) + pd.Timedelta(hours=6)).time
x_DALES_1D_s = [t.strftime('%H:%M:%S') for t in ref_time_1D]
x_DALES_1D =  [datetime.strptime(time_str, '%H:%M:%S').hour + datetime.strptime(time_str, '%H:%M:%S').minute / 60 for time_str in x_DALES_1D_s]

#------------------------------------------------------------------------------------------------------------
#WCO2 at abl height
wCO2 = profiles.wsv001t/1000 #ppb m/s to ppm m/s
idx = idx_1D
wCO2_abl = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        wCO2_abl.append(np.nan)
    else:
        wCO2_val = wCO2.values[i,int(idx[i])]
        wCO2_abl.append(wCO2_val)

wCO2_abl_ds = xr.DataArray(wCO2_abl,coords=[profiles.time],dims=['time'])
wCO2_abl_ds = wCO2_abl_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
wCO2_abl_ds = wCO2_abl_ds.assign_coords(time=new_time)

#------------------------------------------------------------------------------------------------------------
#CO2_dt abl
CO2 = profiles.sv001/1000 #ppb to ppm
idx = idx_1D

CO2_dt_full = CO2.differentiate('time',datetime_unit='s')

CO2_dt_abl = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_dt_abl.append(np.nan)
    else:
        CO2_dt_val = np.nanmean(CO2_dt_full.values[i,:int(idx[i])+1])
        CO2_dt_abl.append(CO2_dt_val)

CO2_dt_abl_ds = xr.DataArray(CO2_dt_abl,coords=[profiles.time],dims=['time'])
CO2_dt_abl_ds = CO2_dt_abl_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
CO2_dt_abl_ds = CO2_dt_abl_ds.assign_coords(time=new_time)

#Residual MXL
LH = CO2_dt_abl_ds.values
RH = (wCO2_s/abl_1D_ds.values - wCO2_abl_ds.values/abl_1D_ds.values)
total_residual_mxl = RH - LH


#------------------------------------------------------------------------------------------------------------
#Cloud base and height - 1D
ql = profiles.ql

cb_1D = []
ct_1D = []

for i in range(len(time_ref)):
    values = ql.sel(time=time_ref[i]).values
    idx = np.where(values > 0)[0]
    
    if len(idx) > 0:
        cb_val = profiles.zt[idx[0]].values
        ct_val = profiles.zt[idx[-1]].values
    else:
        cb_val = np.nan
        ct_val = np.nan
    
    cb_1D.append(cb_val+40)
    ct_1D.append(ct_val+40)

cb_1D_ds = xr.DataArray(cb_1D,coords=[profiles.time],dims=['time'])
cb_1D_ds = cb_1D_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
cb_1D_ds = cb_1D_ds.assign_coords(time=new_time)

ct_1D_ds = xr.DataArray(ct_1D,coords=[profiles.time],dims=['time'])
ct_1D_ds = ct_1D_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
ct_1D_ds = ct_1D_ds.assign_coords(time=new_time)

#------------------------------------------------------------------------------------------------------------
#Mixed-layer concentration
CO2 = profiles.sv001/1000 #ppb to ppm
idx = idx_1D

CO2_mixed = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_mixed.append(np.nan)
    else:
        CO2_val = np.nanmean(CO2.values[i,:int(idx[i])])
        CO2_mixed.append(CO2_val)

CO2_mixed_ds = xr.DataArray(CO2_mixed,coords=[profiles.time],dims=['time'])
CO2_mixed_ds = CO2_mixed_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
CO2_mixed_ds = CO2_mixed_ds.assign_coords(time=new_time)

#------------------------------------------------------------------------------------------------------------
#Free tropospheric concentration
CO2 = profiles.sv001/1000 #ppb to ppm
idx = idx_1D

CO2_ft = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_ft.append(np.nan)
    else:
        idx_val = find_closest(z_1D, abl_1D[i]+100)
        CO2_val = np.nanmean(CO2.values[i,int(idx_val):])
        CO2_ft.append(CO2_val)

CO2_ft_ds = xr.DataArray(CO2_ft,coords=[profiles.time],dims=['time'])
CO2_ft_ds = CO2_ft_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
CO2_ft_ds = CO2_ft_ds.assign_coords(time=new_time)

#------------------------------------------------------------------------------------------------------------
#Lowest level concentration
CO2 = profiles.sv001/1000 #ppb to ppm
CO2 = CO2 * 1.8 #From ppm to mg/m3

CO2_lowest = []
idx = idx_1D

for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_lowest.append(np.nan)
    else:
        CO2_val = CO2.values[i,0]
        CO2_lowest.append(CO2_val)
    
CO2_lowest_ds = xr.DataArray(CO2_lowest,coords=[profiles.time],dims=['time'])
CO2_lowest_ds = CO2_lowest_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_ref[::6]
CO2_lowest_ds = CO2_lowest_ds.assign_coords(time=new_time)

#------------------------------------------------------------------------------------------------------------
#Plant concentration (Ci)
Ci = CO2_lowest_ds.values + (An_s_ds.values * rsCO2_s_ds.values)

#------------------------------------------------------------------------------------------------------------
#Delta_c (Cm - Ci)
delta_c_plant = CO2_mixed_ds.values - Ci/1.8

#------------------------------------------------------------------------------------------------------------
#Delta_c (Cm - Cft)
delta_c_atm = CO2_mixed_ds.values - CO2_ft_ds.values

#------------------------------------------------------------------------------------------------------------
#ratio plant and atm

ratio_plant = (1 - ((Ci/1.8)/CO2_mixed_ds.values))
ratio_plant_without_1 = (Ci/1.8)/CO2_mixed_ds.values
ratio_atm = (1 - (CO2_ft_ds.values/CO2_mixed_ds.values))
ratio_atm_without_1 = CO2_ft_ds.values/CO2_mixed_ds.values

#------------------------------------------------------------------------------------------------------------

#tmser cloud info
tmser = xr.open_dataset(f'{DALES_rootdir}tmser.{DALES_nr}.nc')
time_tmser = tmser.time.values

tmser_cloud_base = tmser.zb.values+40
tmser_cloud_top_av = tmser.zc_av.values+40
tmser_cloud_top_max = tmser.zc_max.values+40

tmser_cloud_base_ds = xr.DataArray(tmser_cloud_base,coords=[time_tmser],dims=['time'])
tmser_cloud_base_ds = tmser_cloud_base_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_tmser[::6]
tmser_cloud_base_ds = tmser_cloud_base_ds.assign_coords(time=new_time)

tmser_cloud_top_av_ds = xr.DataArray(tmser_cloud_top_av,coords=[time_tmser],dims=['time'])
tmser_cloud_top_av_ds = tmser_cloud_top_av_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_tmser[::6]
tmser_cloud_top_av_ds = tmser_cloud_top_av_ds.assign_coords(time=new_time)

tmser_cloud_top_max_ds = xr.DataArray(tmser_cloud_top_max,coords=[time_tmser],dims=['time'])
tmser_cloud_top_max_ds = tmser_cloud_top_max_ds.coarsen(time=6, boundary='trim').mean()
new_time = time_tmser[::6]
tmser_cloud_top_max_ds = tmser_cloud_top_max_ds.assign_coords(time=new_time)

#------------------------------------------------------------------------------------------------------------
#                                   1 D   B U D G E T  -    N O   C L O U D S 
#------------------------------------------------------------------------------------------------------------
#%%

#CO2/dt
CO2_no_clouds = profiles_no_clouds.sv001/1000 #ppb to ppm
time_no_clouds = profiles_no_clouds.time
z_1D_no_clouds = (profiles_no_clouds.zt.values + 40)

CO2_dt_no_clouds = CO2_no_clouds.differentiate('time',datetime_unit='s')
CO2_dt_no_clouds = CO2_dt_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_no_clouds[::6]
CO2_dt_no_clouds = CO2_dt_no_clouds.assign_coords(time=new_time_no_clouds)

#CO2/dz
flux_t_no_clouds = profiles_no_clouds.wsv001t/1000 #ppb m/s to ppm m/s

wCO2_t_dz_no_clouds = (flux_t_no_clouds.differentiate('zm'))
wCO2_t_dz_no_clouds = wCO2_t_dz_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_no_clouds[::6]
wCO2_t_dz_no_clouds = wCO2_t_dz_no_clouds.assign_coords(time=new_time_no_clouds)

total_residual_no_clouds = wCO2_t_dz_no_clouds.values + CO2_dt_no_clouds.values

CO2_dt_1D_no_clouds = CO2_dt_no_clouds.values
wCO2_t_dz_1D_no_clouds = wCO2_t_dz_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#wCO2 (surface)
df = open(f'/home/.../R{DALES_nr_no_clouds}/tmlsm.{DALES_nr_no_clouds}')
headers = None
data = {}

count = 0
for line in df:
    if count == 0:
        split = line.split(None)[1:]
        headers = split
        for item in headers:
            data[item] = []
        count += 1
    elif count == 1:
        count += 1
        continue
    else:
        split = line.split(None)[:]
        sub = 0
        for item in headers:
            data[item].append(split[sub])
            sub += 1
        count += 1

wCO2_s_no_clouds = np.array(data['wco2']).astype(float)/1.8 #mg C/m2 s to ppm/s
wCO2_s_time_no_clouds = np.array(data['time']).astype(float) #5 minute interval

wCO2_s_2_no_clouds = profiles_no_clouds.wsv001t.values[:,0]/1000
wCO2_s_ds_no_clouds = xr.DataArray(wCO2_s_2_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
wCO2_s_ds_no_clouds = wCO2_s_ds_no_clouds.coarsen(time=6,boundary='trim').mean()
new_time_no_clouds = wCO2_s_time_no_clouds[::6]
wCO2_s_ds_no_clouds = wCO2_s_ds_no_clouds.assign_coords(time=new_time_no_clouds)

wCO2_s_no_clouds = wCO2_s_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#g (surface)
gCO2_s_no_clouds = np.array(data['gcco2']).astype(float)
gCO2_s_time_no_clouds = np.array(data['time']).astype(float) #5 minute interval

gCO2_s_ds_no_clouds = xr.DataArray(gCO2_s_no_clouds,coords=[gCO2_s_time_no_clouds],dims=['time'])
gCO2_s_ds_no_clouds = gCO2_s_ds_no_clouds.coarsen(time=6,boundary='trim').mean()
new_time_no_clouds = gCO2_s_time_no_clouds[::6]
gCO2_s_ds_no_clouds = gCO2_s_ds_no_clouds.assign_coords(time=new_time_no_clouds)
gCO2_s_no_clouds = gCO2_s_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#Resp (surface)
respCO2_s_no_clouds = np.array(data['Resp']).astype(float)
respCO2_s_time_no_clouds = np.array(data['time']).astype(float) #5 minute interval

respCO2_s_ds_no_clouds = xr.DataArray(respCO2_s_no_clouds,coords=[respCO2_s_time_no_clouds],dims=['time'])
respCO2_s_ds_no_clouds = respCO2_s_ds_no_clouds.coarsen(time=6,boundary='trim').mean()
new_time_no_clouds = respCO2_s_time_no_clouds[::6]
respCO2_s_ds_no_clouds = respCO2_s_ds_no_clouds.assign_coords(time=new_time_no_clouds)
respCO2_s_no_clouds = respCO2_s_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#ra (surface)
ra_s_no_clouds = np.array(data['ra']).astype(float)
ra_s_time_no_clouds = np.array(data['time']).astype(float) #5 minute interval

ra_s_ds_no_clouds = xr.DataArray(ra_s_no_clouds,coords=[ra_s_time_no_clouds],dims=['time'])
ra_s_ds_no_clouds = ra_s_ds_no_clouds.coarsen(time=6,boundary='trim').mean()
new_time_no_clouds = ra_s_time_no_clouds[::6]
ra_s_ds_no_clouds = ra_s_ds_no_clouds.assign_coords(time=new_time_no_clouds)
ra_s_no_clouds = ra_s_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#rsCO2 (surface)
rsCO2_s_no_clouds = np.array(data['rs']).astype(float)*1.6 #rwater to rsCO2
rsCO2_s_time_no_clouds = np.array(data['time']).astype(float) #5 minute interval

rsCO2_s_ds_no_clouds = xr.DataArray(rsCO2_s_no_clouds,coords=[rsCO2_s_time_no_clouds],dims=['time'])
rsCO2_s_ds_no_clouds = rsCO2_s_ds_no_clouds.coarsen(time=6,boundary='trim').mean()
new_time_no_clouds = rsCO2_s_time_no_clouds[::6]
rsCO2_s_ds_no_clouds = rsCO2_s_ds_no_clouds.assign_coords(time=new_time_no_clouds)
rsCO2_s_no_clouds = rsCO2_s_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#An (surface)
An_s_no_clouds = np.array(data['An']).astype(float)
An_s_time_no_clouds = np.array(data['time']).astype(float) #5 minute interval

An_s_ds_no_clouds = xr.DataArray(An_s_no_clouds,coords=[An_s_time_no_clouds],dims=['time'])
An_s_ds_no_clouds = An_s_ds_no_clouds.coarsen(time=6,boundary='trim').mean()
new_time_no_clouds = An_s_time_no_clouds[::6]
An_s_ds_no_clouds = An_s_ds_no_clouds.assign_coords(time=new_time_no_clouds)
An_s_no_clouds = An_s_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#Boundary layer height - 1D
wthvt_no_clouds = profiles_no_clouds.wthvt
time_ref_no_clouds = profiles_no_clouds.time.values

abl_1D_no_clouds = []
idx_1D_no_clouds = []
for i in range(len(time_ref_no_clouds)):
    values = wthvt_no_clouds.sel(time=time_ref_no_clouds[i]).values
    idx = np.where(values == np.min(values))[0][0]
    abl_val = profiles_no_clouds.zt[idx].values
    
    if abl_val == 0 or abl_val > 2000 or abl_val < 500 or i<10:
        abl_val = np.nan
        idx = np.nan
    
    abl_1D_no_clouds.append(abl_val+40)
    idx_1D_no_clouds.append(idx)

abl_1D_ds_no_clouds = xr.DataArray(abl_1D_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
abl_1D_ds_no_clouds = abl_1D_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
abl_1D_ds_no_clouds = abl_1D_ds_no_clouds.assign_coords(time=new_time_no_clouds)

idx_1D_ds_no_clouds = xr.DataArray(idx_1D_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
idx_1D_ds_no_clouds = idx_1D_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
idx_1D_ds_no_clouds = idx_1D_ds_no_clouds.assign_coords(time=new_time_no_clouds)

#Reference time
ref_time_1D_no_clouds = pd.to_datetime(wCO2_s_ds_no_clouds.time.values, unit='s', origin='unix')
ref_time_1D_no_clouds  = (pd.to_datetime(ref_time_1D_no_clouds) + pd.Timedelta(hours=6)).time
x_DALES_1D_s_no_clouds = [t.strftime('%H:%M:%S') for t in ref_time_1D_no_clouds]
x_DALES_1D_no_clouds =  [datetime.strptime(time_str, '%H:%M:%S').hour + datetime.strptime(time_str, '%H:%M:%S').minute / 60 for time_str in x_DALES_1D_s_no_clouds]

#------------------------------------------------------------------------------------------------------------
#WCO2 at abl height
wCO2_no_clouds = profiles_no_clouds.wsv001t/1000 #ppb m/s to ppm m/s
idx = idx_1D_no_clouds
wCO2_abl_no_clouds = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        wCO2_abl_no_clouds.append(np.nan)
    else:
        wCO2_val = wCO2_no_clouds.values[i,int(idx[i])]
        wCO2_abl_no_clouds.append(wCO2_val)

wCO2_abl_ds_no_clouds = xr.DataArray(wCO2_abl_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
wCO2_abl_ds_no_clouds = wCO2_abl_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
wCO2_abl_ds_no_clouds = wCO2_abl_ds_no_clouds.assign_coords(time=new_time_no_clouds)

#------------------------------------------------------------------------------------------------------------
#CO2_dt abl
CO2_no_clouds = profiles_no_clouds.sv001/1000 #ppb to ppm
idx = idx_1D_no_clouds

CO2_dt_full_no_clouds = CO2_no_clouds.differentiate('time',datetime_unit='s')

CO2_dt_abl_no_clouds = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_dt_abl_no_clouds.append(np.nan)
    else:
        CO2_dt_val = np.nanmean(CO2_dt_full_no_clouds.values[i,:int(idx[i])+1])
        CO2_dt_abl_no_clouds.append(CO2_dt_val)

CO2_dt_abl_ds_no_clouds = xr.DataArray(CO2_dt_abl_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
CO2_dt_abl_ds_no_clouds = CO2_dt_abl_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
CO2_dt_abl_ds_no_clouds = CO2_dt_abl_ds_no_clouds.assign_coords(time=new_time_no_clouds)

#Residual MXL
LH_no_clouds = CO2_dt_abl_ds_no_clouds.values
RH_no_clouds = (wCO2_s_ds_no_clouds.values - wCO2_abl_ds_no_clouds.values)/abl_1D_ds_no_clouds.values
total_residual_mxl_no_clouds = RH_no_clouds - LH_no_clouds

#------------------------------------------------------------------------------------------------------------
#Mixed-layer concentration
CO2_no_clouds = profiles_no_clouds.sv001/1000 #ppb to ppm
idx = idx_1D_no_clouds

CO2_mixed_no_clouds = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_mixed_no_clouds.append(np.nan)
    else:
        CO2_val = np.nanmean(CO2_no_clouds.values[i,:int(idx[i])])
        CO2_mixed_no_clouds.append(CO2_val)

CO2_mixed_ds_no_clouds = xr.DataArray(CO2_mixed_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
CO2_mixed_ds_no_clouds = CO2_mixed_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
CO2_mixed_ds_no_clouds = CO2_mixed_ds_no_clouds.assign_coords(time=new_time_no_clouds)

#------------------------------------------------------------------------------------------------------------
#Free tropospheric concentration
CO2_no_clouds = profiles_no_clouds.sv001/1000 #ppb to ppm
idx = idx_1D_no_clouds

CO2_ft_no_clouds = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_ft_no_clouds.append(np.nan)
    else:
        idx_val = find_closest(z_1D_no_clouds, abl_1D_no_clouds[i]+100)
        CO2_val = np.nanmean(CO2_no_clouds.values[i,int(idx_val):])
        CO2_ft_no_clouds.append(CO2_val)

CO2_ft_ds_no_clouds = xr.DataArray(CO2_ft_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
CO2_ft_ds_no_clouds = CO2_ft_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
CO2_ft_ds_no_clouds = CO2_ft_ds_no_clouds.assign_coords(time=new_time_no_clouds)

#------------------------------------------------------------------------------------------------------------
#Lowest level concentration
CO2_no_clouds = profiles_no_clouds.sv001/1000 #ppb to ppm
CO2_no_clouds = CO2_no_clouds * 1.8 #From ppm to mg/m3

CO2_lowest_no_clouds = []
idx = idx_1D_no_clouds

for i in range(len(idx)):
    if np.isnan(idx[i]):
        CO2_lowest_no_clouds.append(np.nan)
    else:
        CO2_val = CO2_no_clouds.values[i,0]
        CO2_lowest_no_clouds.append(CO2_val)

CO2_lowest_ds_no_clouds = xr.DataArray(CO2_lowest_no_clouds,coords=[profiles_no_clouds.time],dims=['time'])
CO2_lowest_ds_no_clouds = CO2_lowest_ds_no_clouds.coarsen(time=6, boundary='trim').mean()
new_time_no_clouds = time_ref_no_clouds[::6]
CO2_lowest_ds_no_clouds = CO2_lowest_ds_no_clouds.assign_coords(time=new_time_no_clouds)

#------------------------------------------------------------------------------------------------------------
#Plant concentration (Ci)
Ci_no_clouds = CO2_lowest_ds_no_clouds.values + (An_s_ds_no_clouds.values * rsCO2_s_ds_no_clouds.values)

#------------------------------------------------------------------------------------------------------------
#Delta_c (Cm - Ci)
delta_c_plant_no_clouds = CO2_mixed_ds_no_clouds.values - Ci_no_clouds/1.8

#------------------------------------------------------------------------------------------------------------
#Delta_c (Cm - Cft)
delta_c_atm_no_clouds = CO2_mixed_ds_no_clouds.values - CO2_ft_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------
#ratio plant and atm

ratio_plant_no_clouds = (1 - ((Ci_no_clouds/1.8)/CO2_mixed_ds_no_clouds.values))
ratio_plant_no_clouds_without_1 = (Ci_no_clouds/1.8)/CO2_mixed_ds_no_clouds.values
ratio_atm_no_clouds = (1 - (CO2_ft_ds_no_clouds.values/CO2_mixed_ds_no_clouds.values))
ratio_atm_no_clouds_without_1 = CO2_ft_ds_no_clouds.values/CO2_mixed_ds_no_clouds.values

#------------------------------------------------------------------------------------------------------------

#tmser no cloud
tmser_no_clouds = xr.open_dataset(f'{DALES_rootdir_no_clouds}tmser.{DALES_nr_no_clouds}.nc')
time_tmser_no_clouds = tmser_no_clouds.time.values

#------------------------------------------------------------------------------------------------------------
#                                               3 D  B U D G E T
#------------------------------------------------------------------------------------------------------------
#%%

#3D
field_3D = xr.open_dataset(f'{DALES_rootdir}Combined/fielddump.{DALES_nr}.nc')
ref_time_3D = field_3D.time.values

#Reference heights 3D
zt_3D = field_3D.zt.values
zm_3D = field_3D.zm.values
z_3D = field_3D.zt.values[:-1]+40

#3D - no clouds
field_3D_no_clouds = xr.open_dataset(f'{DALES_rootdir_no_clouds}Combined/fielddump.{DALES_nr_no_clouds}.nc')


#%%
#-----------------------------------------------------------------------------------------------------------

# u        (time, zt, yt, xm) 
# v        (time, zt, ym, xt) 
# w        (time, zm, yt, xt) 
# qt       (time, zt, yt, xt) 
# ql       (time, zt, yt, xt) 
# thl      (time, zt, yt, xt) 
# buoy     (time, zt, yt, xt) 
# sv001    (time, zt, yt, xt) 

#conditions for conditional averaging clouds
#1 ql > 0
#2 ql > 0, w' > 0
#3 ql > 0, w' > 0 and theta_v' > 0

wCO2_dz_arr = []
wCO2_dz_arr_inclouds_condition_1 = []
wCO2_dz_arr_inclouds_condition_2 = []
wCO2_dz_arr_inclouds_condition_3 = []

wCO2_dz_arr_environment_condition_1 = []
wCO2_dz_arr_environment_condition_2 = []
wCO2_dz_arr_environment_condition_3 = []

wCO2_dz_arr_no_clouds = []

wCO2_arr = []
wCO2_arr_inclouds_condition_1 = []
wCO2_arr_inclouds_condition_2 = []
wCO2_arr_inclouds_condition_3 = []

wCO2_arr_environment_condition_1 = []
wCO2_arr_environment_condition_2 = []
wCO2_arr_environment_condition_3 = []

wCO2_arr_no_clouds = []

wCO2_arr_inclouds_nofactor_condition_1 = []
wCO2_arr_inclouds_nofactor_condition_2 = []
wCO2_arr_inclouds_nofactor_condition_3 = []

wCO2_arr_environment_nofactor_condition_1 = []
wCO2_arr_environment_nofactor_condition_2 = []
wCO2_arr_environment_nofactor_condition_3 = []

qCO2_arr = []
qCO2_arr_inclouds_condition_1 = []
qCO2_arr_inclouds_condition_2 = []
qCO2_arr_inclouds_condition_3 = []

qCO2_arr_environment_condition_1 = []
qCO2_arr_environment_condition_2 = []
qCO2_arr_environment_condition_3 = []

qCO2_arr_no_clouds = []

wq_arr = []
wq_arr_inclouds_condition_1 = []
wq_arr_inclouds_condition_2 = []
wq_arr_inclouds_condition_3 = []

wq_arr_environment_condition_1 = []
wq_arr_environment_condition_2 = []
wq_arr_environment_condition_3 = []

wq_arr_no_clouds = []

wq_arr_inclouds_nofactor_condition_1 = []
wq_arr_inclouds_nofactor_condition_2 = []
wq_arr_inclouds_nofactor_condition_3 = []

wq_arr_environment_nofactor_condition_1 = []
wq_arr_environment_nofactor_condition_2 = []
wq_arr_environment_nofactor_condition_3 = []

corr_arr = []
corr_arr_inclouds_condition_1 = []
corr_arr_inclouds_condition_2 = []
corr_arr_inclouds_condition_3 = []

corr_arr_environment_condition_1 = []
corr_arr_environment_condition_2 = []
corr_arr_environment_condition_3 = []

corr_arr_no_clouds = []

cfrac_condition_1 = []
cfrac_condition_2 = []
cfrac_condition_3 = []

grad_CO2_arr = []

grad_CO2_arr_inclouds_condition_1 = []
grad_CO2_arr_inclouds_condition_2 = []
grad_CO2_arr_inclouds_condition_3 = []

grad_CO2_arr_environment_condition_1 = []
grad_CO2_arr_environment_condition_2 = []
grad_CO2_arr_environment_condition_3 = []

grad_CO2_arr_no_clouds = []

profile_theta_v_arr = []

profile_theta_v_inclouds_condition_1 = []
profile_theta_v_inclouds_condition_2 = []
profile_theta_v_inclouds_condition_3 = []

profile_theta_v_environment_condition_1 = []
profile_theta_v_environment_condition_2 = []
profile_theta_v_environment_condition_3 = []

profile_theta_v_no_clouds = []

cloud_mask_full_2D_condition_1 = []
cloud_mask_full_2D_condition_2 = []
cloud_mask_full_2D_condition_3 = []

buoy_arr = []

buoy_arr_inclouds_condition_1 = []
buoy_arr_inclouds_condition_2 = []
buoy_arr_inclouds_condition_3 = []

buoy_arr_environment_condition_1 = []
buoy_arr_environment_condition_2 = []
buoy_arr_environment_condition_3 = []

buoy_arr_no_clouds = []

CO2_arr = []

CO2_arr_inclouds_condition_1 = []
CO2_arr_inclouds_condition_2 = []
CO2_arr_inclouds_condition_3 = []

CO2_arr_environment_condition_1 = []
CO2_arr_environment_condition_2 = []
CO2_arr_environment_condition_3 = []

CO2_arr_no_clouds = []


for time in range(len(field_3D.time)):
    
    # time = 36 + 6
    #Print status
    print(time/len(field_3D.time)*100)
    
    #-------------------------------------------------------------------------------------
    #                           R 0 3 4  |  C L O U D S 
    #-------------------------------------------------------------------------------------
    
    #CO2
    CO2 = field_3D.sv001[time,:,:] / 1000
    mean_CO2 = CO2.mean(dim=('yt', 'xt')).values
    var_CO2 = CO2 - mean_CO2[:, None, None]
    
    grad_CO2 = CO2.differentiate('zt')
    grad_CO2 = grad_CO2.mean(dim=('yt', 'xt'))
    grad_CO2_arr.append(grad_CO2.values)
    
    CO2_arr.append(mean_CO2)
    
    #q
    q = field_3D.qt[time,:,:]*1000
    mean_q = q[:,:].mean(dim=('yt', 'xt')).values
    var_q = q[:,:] - mean_q[:, None, None]
    
    #qCO2
    qCO2 = var_CO2[:-1, :, :] * var_q[:-1, :, :]
    qCO2_mean = qCO2.mean(dim=('yt', 'xt'))

    qCO2_arr.append(qCO2_mean.values)
    
    #corr
    std_q = var_q[:-1, :, :].std(dim=('yt', 'xt'))
    std_CO2 = var_CO2[:-1, :, :].std(dim=('yt', 'xt'))
    
    corr = (qCO2)/(std_q*std_CO2)
    corr_mean = corr.mean(dim=('yt', 'xt')).values
    corr_arr.append(corr_mean) 
        
    #Compute w
    w = (field_3D.w[time,:,:][0:, :, :] + field_3D.w[time,:,:][:-1, :, :]) / 2
    mean_w = w.mean(dim=('yt', 'xt'))
    var_w = w - mean_w

    #wCO2/dz
    wCO2 = var_CO2[:-1, :, :] * var_w.values
    wCO2_mean = wCO2.mean(dim=('yt', 'xt'))
    wCO2_dz_mean = wCO2_mean.differentiate('zt')

    wCO2_dz_arr.append(wCO2_dz_mean.values)
    wCO2_arr.append(wCO2_mean.values)

        
    #wq
    wq = var_q[:-1, :, :] * var_w.values
    wq_mean = wq.mean(dim=('yt', 'xt'))
    wq_arr.append(wq_mean.values)
    
    #Theta_v 
    thl = field_3D.thl[time,:,:]
    qt = field_3D.qt[time,:,:]
    theta_v = thl * (1 + 0.61 * qt)
    mean_theta_v = theta_v.mean(dim=('yt', 'xt'))
    var_theta_v = theta_v - mean_theta_v
    
    profile_theta_v_arr.append(mean_theta_v.values)
    
    #Buoyancy
    buoy = field_3D.buoy[time,:,:]
    buoy_mean = buoy.mean(dim=('yt', 'xt'))
    var_buoy = buoy - buoy_mean
    buoy_arr.append(buoy_mean.values)
    
    #-------------------------------------------------------------------------------------
    
    #Cloud mask - Condition 1
    cloud_mask_condition_1 = field_3D.ql[time,:,:,:] > 0
    cloud_mask_2D_condition_1 = cloud_mask_condition_1.mean(dim=('zt')).values
    
    cloud_mask_2D_condition_1 = cloud_mask_2D_condition_1
    cloud_mask_2D_condition_1[cloud_mask_2D_condition_1 > 0] = 1
    cloud_mask_2D_condition_1[cloud_mask_2D_condition_1 < 1] = 0
    
    cloud_mask_3D_condition_1 = cloud_mask_2D_condition_1[np.newaxis,:,:]
    cloud_mask_3D_full_condition_1 = np.broadcast_to(cloud_mask_3D_condition_1, (250,360,360))
    cloud_mask_3D_condition_1 = np.broadcast_to(cloud_mask_3D_condition_1, (249,360,360))
    
    cloud_mask_3D_condition_1 = xr.DataArray(cloud_mask_3D_condition_1,coords=[field_3D.zt[:-1],field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
    cloud_mask_3D_full_condition_1 = xr.DataArray(cloud_mask_3D_full_condition_1,coords=[field_3D.zt,field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
    
    calc_cfrac_condition_1 = np.nansum(cloud_mask_2D_condition_1)/np.size(cloud_mask_2D_condition_1)
    cfrac_condition_1.append(calc_cfrac_condition_1)
    
    cloud_mask_full_2D_condition_1.append(cloud_mask_2D_condition_1)
    
    #-------------------------------------------------------------------------------------
    
    #Cloud mask - Condition 2
    ql_values = field_3D.ql[time,:-1,:,:].values

    x = ql_values
    y = var_w.values
    mask = ((x > 0) & (y>0)).astype(int)

    mask = np.nanmean(mask,axis=0)

    mask[mask > 0] = 1
    mask[mask < 1] = 0

    cloud_mask_2D_condition_2 = mask

    cloud_mask_2D_condition_2 = cloud_mask_2D_condition_2
    cloud_mask_2D_condition_2[cloud_mask_2D_condition_2 > 0] = 1
    cloud_mask_2D_condition_2[cloud_mask_2D_condition_2 < 1] = 0
    
    cloud_mask_3D_condition_2 = cloud_mask_2D_condition_2[np.newaxis,:,:]
    cloud_mask_3D_full_condition_2 = np.broadcast_to(cloud_mask_3D_condition_2, (250,360,360))
    cloud_mask_3D_condition_2 = np.broadcast_to(cloud_mask_3D_condition_2, (249,360,360))
    
    cloud_mask_3D_condition_2 = xr.DataArray(cloud_mask_3D_condition_2,coords=[field_3D.zt[:-1],field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
    cloud_mask_3D_full_condition_2 = xr.DataArray(cloud_mask_3D_full_condition_2,coords=[field_3D.zt,field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
    
    calc_cfrac_condition_2 = np.nansum(cloud_mask_2D_condition_2)/np.size(cloud_mask_2D_condition_2)
    cfrac_condition_2.append(calc_cfrac_condition_2)
    
    cloud_mask_full_2D_condition_2.append(cloud_mask_2D_condition_2)
    
    #-------------------------------------------------------------------------------------
    
    #Cloud mask - Condition 3
    ql_values = field_3D.ql[time,:-1,:,:].values

    x = ql_values
    y = var_buoy.values[:-1,:,:]
    z = var_w.values
    mask = ((x > 0) & (y>0) & (z>0)).astype(int)

    mask = np.nanmean(mask,axis=0)

    mask[mask > 0] = 1
    mask[mask < 1] = 0

    cloud_mask_2D_condition_3 = mask
    
    cloud_mask_2D_condition_3 = cloud_mask_2D_condition_3
    cloud_mask_2D_condition_3[cloud_mask_2D_condition_3 > 0] = 1
    cloud_mask_2D_condition_3[cloud_mask_2D_condition_3 < 1] = 0
    
    cloud_mask_3D_condition_3 = cloud_mask_2D_condition_3[np.newaxis,:,:]
    cloud_mask_3D_full_condition_3 = np.broadcast_to(cloud_mask_3D_condition_3, (250,360,360))
    cloud_mask_3D_condition_3 = np.broadcast_to(cloud_mask_3D_condition_3, (249,360,360))
    
    cloud_mask_3D_condition_3 = xr.DataArray(cloud_mask_3D_condition_3,coords=[field_3D.zt[:-1],field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
    cloud_mask_3D_full_condition_3 = xr.DataArray(cloud_mask_3D_full_condition_3,coords=[field_3D.zt,field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
    
    calc_cfrac_condition_3 = np.nansum(cloud_mask_2D_condition_3)/np.size(cloud_mask_2D_condition_3)
    cfrac_condition_3.append(calc_cfrac_condition_3)
    
    cloud_mask_full_2D_condition_3.append(cloud_mask_2D_condition_3)
    
    #-------------------------------------------------------------------------------------
    
    #Inclouds Contributions  
    try:
        
        #wCO2
        
        #Condition 1
        wCO2_inclouds_condition_1 = wCO2.where(cloud_mask_3D_condition_1==1)
        wCO2_inclouds_mean_condition_1 = wCO2_inclouds_condition_1.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_1)
        wCO2_inclouds_mean_nofactor_condition_1 = wCO2_inclouds_condition_1.mean(dim=('yt', 'xt'))
        
        wCO2_dz_inclouds_condition_1 = wCO2_inclouds_mean_condition_1.differentiate('zt')
        
        wCO2_dz_arr_inclouds_condition_1.append(wCO2_dz_inclouds_condition_1)
        wCO2_arr_inclouds_condition_1.append(wCO2_inclouds_mean_condition_1)   
        wCO2_arr_inclouds_nofactor_condition_1.append(wCO2_inclouds_mean_nofactor_condition_1)
                
        #Condition 2
        wCO2_inclouds_condition_2 = wCO2.where(cloud_mask_3D_condition_2==1)
        wCO2_inclouds_mean_condition_2 = wCO2_inclouds_condition_2.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_2)
        wCO2_inclouds_mean_nofactor_condition_2 = wCO2_inclouds_condition_2.mean(dim=('yt', 'xt'))
        
        wCO2_dz_inclouds_condition_2 = wCO2_inclouds_mean_condition_2.differentiate('zt')
        
        wCO2_dz_arr_inclouds_condition_2.append(wCO2_dz_inclouds_condition_2)
        wCO2_arr_inclouds_condition_2.append(wCO2_inclouds_mean_condition_2)   
        wCO2_arr_inclouds_nofactor_condition_2.append(wCO2_inclouds_mean_nofactor_condition_2)
        
        #Condition 3
        wCO2_inclouds_condition_3 = wCO2.where(cloud_mask_3D_condition_3==1)
        wCO2_inclouds_mean_condition_3 = wCO2_inclouds_condition_3.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_3)
        wCO2_inclouds_mean_nofactor_condition_3 = wCO2_inclouds_condition_3.mean(dim=('yt', 'xt'))
        
        wCO2_dz_inclouds_condition_3 = wCO2_inclouds_mean_condition_3.differentiate('zt')
        
        wCO2_dz_arr_inclouds_condition_3.append(wCO2_dz_inclouds_condition_3)
        wCO2_arr_inclouds_condition_3.append(wCO2_inclouds_mean_condition_3)   
        wCO2_arr_inclouds_nofactor_condition_3.append(wCO2_inclouds_mean_nofactor_condition_3)
    
        
        #-------------------------------------------------------------------------------------

        #qCO2
        
        #Condition 1
        qCO2_inclouds_condition_1 = qCO2.where(cloud_mask_3D_condition_1==1)
        qCO2_inclouds_mean_condition_1 = qCO2_inclouds_condition_1.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_1)
        
        qCO2_arr_inclouds_condition_1.append(qCO2_inclouds_mean_condition_1.values)
        
        #Condition 2
        qCO2_inclouds_condition_2 = qCO2.where(cloud_mask_3D_condition_2==1)
        qCO2_inclouds_mean_condition_2 = qCO2_inclouds_condition_2.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_2)
        
        qCO2_arr_inclouds_condition_2.append(qCO2_inclouds_mean_condition_2.values)
        
        #Condition 3
        qCO2_inclouds_condition_3 = qCO2.where(cloud_mask_3D_condition_3==1)
        qCO2_inclouds_mean_condition_3 = qCO2_inclouds_condition_3.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_3)
        
        qCO2_arr_inclouds_condition_3.append(qCO2_inclouds_mean_condition_3.values)
        
        #-------------------------------------------------------------------------------------
        
        #Correlation
        
        #Condition 1
        corr_inclouds_condition_1 = corr.where(cloud_mask_3D_condition_1==1)
        corr_inclouds_mean_condition_1 = corr_inclouds_condition_1.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_1)
        corr_arr_inclouds_condition_1.append(corr_inclouds_mean_condition_1)
        
        #Condition 2
        corr_inclouds_condition_2 = corr.where(cloud_mask_3D_condition_2==1)
        corr_inclouds_mean_condition_2 = corr_inclouds_condition_2.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_2)
        corr_arr_inclouds_condition_2.append(corr_inclouds_mean_condition_2)
        
        #Condition 3
        corr_inclouds_condition_3 = corr.where(cloud_mask_3D_condition_3==1)
        corr_inclouds_mean_condition_3 = corr_inclouds_condition_3.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_3)
        corr_arr_inclouds_condition_3.append(corr_inclouds_mean_condition_3)
        
        #-------------------------------------------------------------------------------------
        
        #wq
        
        #Condition 1
        wq_inclouds_condition_1 = wq.where(cloud_mask_3D_condition_1==1)
        wq_inclouds_mean_condition_1 = wq_inclouds_condition_1.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_1)
        wq_inclouds_nofactor_condition_1 = wq_inclouds_condition_1.mean(dim=('yt', 'xt'))
        
        wq_arr_inclouds_condition_1.append(wq_inclouds_mean_condition_1.values)
        wq_arr_inclouds_nofactor_condition_1.append(wq_inclouds_nofactor_condition_1.values)
        
        #Condition 2
        wq_inclouds_condition_2 = wq.where(cloud_mask_3D_condition_2==1)
        wq_inclouds_mean_condition_2 = wq_inclouds_condition_2.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_2)
        wq_inclouds_nofactor_condition_2 = wq_inclouds_condition_2.mean(dim=('yt', 'xt'))
        
        wq_arr_inclouds_condition_2.append(wq_inclouds_mean_condition_2.values)
        wq_arr_inclouds_nofactor_condition_2.append(wq_inclouds_nofactor_condition_2.values)
        
        #Condition 3
        wq_inclouds_condition_3 = wq.where(cloud_mask_3D_condition_3==1)
        wq_inclouds_mean_condition_3 = wq_inclouds_condition_3.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_3)
        wq_inclouds_nofactor_condition_3 = wq_inclouds_condition_3.mean(dim=('yt', 'xt'))
        
        wq_arr_inclouds_condition_3.append(wq_inclouds_mean_condition_3.values)
        wq_arr_inclouds_nofactor_condition_3.append(wq_inclouds_nofactor_condition_3.values)
        
        #-------------------------------------------------------------------------------------
        
        #Grad CO2
        
        #Condition 1
        grad_CO2_inclouds_condition_1 = grad_CO2.where(cloud_mask_3D_full_condition_1==1)
        grad_CO2_inclouds_mean_condition_1 = grad_CO2_inclouds_condition_1.mean(dim=('yt', 'xt')).values
        grad_CO2_arr_inclouds_condition_1.append(grad_CO2_inclouds_mean_condition_1)
        
        #Condition 2
        grad_CO2_inclouds_condition_2 = grad_CO2.where(cloud_mask_3D_full_condition_2==1)
        grad_CO2_inclouds_mean_condition_2 = grad_CO2_inclouds_condition_2.mean(dim=('yt', 'xt')).values
        grad_CO2_arr_inclouds_condition_2.append(grad_CO2_inclouds_mean_condition_2)
        
        #Condition 3
        grad_CO2_inclouds_condition_3 = grad_CO2.where(cloud_mask_3D_full_condition_3==1)
        grad_CO2_inclouds_mean_condition_3 = grad_CO2_inclouds_condition_3.mean(dim=('yt', 'xt')).values
        grad_CO2_arr_inclouds_condition_3.append(grad_CO2_inclouds_mean_condition_3)
        
        #-------------------------------------------------------------------------------------
        
        #Theta_v
        
        #Condition 1
        profile_theta_v_inclouds_condition_1.append(theta_v.where(cloud_mask_3D_condition_1==1).mean(dim=('yt', 'xt')).values)
        
        #Condition 2
        profile_theta_v_inclouds_condition_2.append(theta_v.where(cloud_mask_3D_condition_2==1).mean(dim=('yt', 'xt')).values)
        
        #Condition 3
        profile_theta_v_inclouds_condition_3.append(theta_v.where(cloud_mask_3D_condition_3==1).mean(dim=('yt', 'xt')).values)
        
        #-------------------------------------------------------------------------------------
        
        #Buoyancy
        
        #Condition 1
        buoy_inclouds_condition_1 = buoy.where(cloud_mask_3D_full_condition_1==1)
        buoy_inclouds_mean_condition_1 = buoy_inclouds_condition_1.mean(dim=('yt', 'xt')).values
        buoy_arr_inclouds_condition_1.append(buoy_inclouds_mean_condition_1)
        
        #Condition 2
        buoy_inclouds_condition_2 = buoy.where(cloud_mask_3D_full_condition_2==1)
        buoy_inclouds_mean_condition_2 = buoy_inclouds_condition_2.mean(dim=('yt', 'xt')).values
        buoy_arr_inclouds_condition_2.append(buoy_inclouds_mean_condition_2)
        
        #Condition 3
        buoy_inclouds_condition_3 = buoy.where(cloud_mask_3D_full_condition_3==1)
        buoy_inclouds_mean_condition_3 = buoy_inclouds_condition_3.mean(dim=('yt', 'xt')).values
        buoy_arr_inclouds_condition_3.append(buoy_inclouds_mean_condition_3)
        
        #-------------------------------------------------------------------------------------
        
        #CO2
        
        #Condition 1
        CO2_inclouds_condition_1 = CO2.where(cloud_mask_3D_condition_1==1)
        CO2_inclouds_mean_condition_1 = CO2_inclouds_condition_1.mean(dim=('yt', 'xt')).values
        CO2_arr_inclouds_condition_1.append(CO2_inclouds_mean_condition_1)
        
        #Condition 2
        CO2_inclouds_condition_2 = CO2.where(cloud_mask_3D_condition_2==1)
        CO2_inclouds_mean_condition_2 = CO2_inclouds_condition_2.mean(dim=('yt', 'xt')).values
        CO2_arr_inclouds_condition_2.append(CO2_inclouds_mean_condition_2)
        
        #Condition 3
        CO2_inclouds_condition_3 = CO2.where(cloud_mask_3D_condition_3==1)
        CO2_inclouds_mean_condition_3 = CO2_inclouds_condition_3.mean(dim=('yt', 'xt')).values
        CO2_arr_inclouds_condition_3.append(CO2_inclouds_mean_condition_3)   
    
    except:
        wCO2_dz_arr_inclouds_condition_1.append(np.zeros(249))
        wCO2_dz_arr_inclouds_condition_2.append(np.zeros(249))
        wCO2_dz_arr_inclouds_condition_3.append(np.zeros(249))
        
        wCO2_arr_inclouds_condition_1.append(np.zeros(249))
        wCO2_arr_inclouds_condition_2.append(np.zeros(249))
        wCO2_arr_inclouds_condition_3.append(np.zeros(249))
        
        wCO2_arr_inclouds_nofactor_condition_1.append(np.zeros(249))
        wCO2_arr_inclouds_nofactor_condition_2.append(np.zeros(249))
        wCO2_arr_inclouds_nofactor_condition_3.append(np.zeros(249))
                
        qCO2_arr_inclouds_condition_1.append(np.zeros(249))
        qCO2_arr_inclouds_condition_2.append(np.zeros(249))
        qCO2_arr_inclouds_condition_3.append(np.zeros(249))
        
        corr_arr_inclouds_condition_1.append(np.zeros(249))
        corr_arr_inclouds_condition_2.append(np.zeros(249))
        corr_arr_inclouds_condition_3.append(np.zeros(249))
        
        wq_arr_inclouds_condition_1.append(np.zeros(249))
        wq_arr_inclouds_condition_2.append(np.zeros(249))
        wq_arr_inclouds_condition_3.append(np.zeros(249))
        
        wq_arr_inclouds_nofactor_condition_1.append(np.zeros(249))
        wq_arr_inclouds_nofactor_condition_2.append(np.zeros(249))
        wq_arr_inclouds_nofactor_condition_3.append(np.zeros(249))
        
        grad_CO2_arr_inclouds_condition_1.append(np.zeros(250))
        grad_CO2_arr_inclouds_condition_2.append(np.zeros(250))
        grad_CO2_arr_inclouds_condition_3.append(np.zeros(250))
        
        profile_theta_v_inclouds_condition_1.append(np.zeros(250))
        profile_theta_v_inclouds_condition_2.append(np.zeros(250))
        profile_theta_v_inclouds_condition_3.append(np.zeros(250))
        
        buoy_arr_inclouds_condition_1.append(np.zeros(250))
        buoy_arr_inclouds_condition_2.append(np.zeros(250))
        buoy_arr_inclouds_condition_3.append(np.zeros(250))
        
        CO2_arr_inclouds_condition_1.append(np.zeros(250))
        CO2_arr_inclouds_condition_2.append(np.zeros(250))
        CO2_arr_inclouds_condition_3.append(np.zeros(250))

        
    #-------------------------------------------------------------------------------------
    
    #Environment
    
    #wCO2
    
    #Condition 1
    wCO2_environment_condition_1 = wCO2.where(cloud_mask_3D_condition_1==0)
    wCO2_environment_mean_condition_1 = wCO2_environment_condition_1.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_1)
    wCO2_environment_mean_nofactor_condition_1 = wCO2_environment_condition_1.mean(dim=('yt', 'xt'))
    
    wCO2_dz_environment_condition_1 = wCO2_environment_mean_condition_1.differentiate('zt')
    
    wCO2_dz_arr_environment_condition_1.append(wCO2_dz_environment_condition_1)
    wCO2_arr_environment_condition_1.append(wCO2_environment_mean_condition_1)
    wCO2_arr_environment_nofactor_condition_1.append(wCO2_environment_mean_nofactor_condition_1)
        
    #Condition 2
    wCO2_environment_condition_2 = wCO2.where(cloud_mask_3D_condition_2==0)
    wCO2_environment_mean_condition_2 = wCO2_environment_condition_2.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_2)
    wCO2_environment_mean_nofactor_condition_2 = wCO2_environment_condition_2.mean(dim=('yt', 'xt'))
    
    wCO2_dz_environment_condition_2 = wCO2_environment_mean_condition_2.differentiate('zt')
    
    wCO2_dz_arr_environment_condition_2.append(wCO2_dz_environment_condition_2)
    wCO2_arr_environment_condition_2.append(wCO2_environment_mean_condition_2)
    wCO2_arr_environment_nofactor_condition_2.append(wCO2_environment_mean_nofactor_condition_2)
    
    
    #Condition 3
    wCO2_environment_condition_3 = wCO2.where(cloud_mask_3D_condition_3==0)
    wCO2_environment_mean_condition_3 = wCO2_environment_condition_3.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_3)
    wCO2_environment_mean_nofactor_condition_3 = wCO2_environment_condition_3.mean(dim=('yt', 'xt'))
    
    wCO2_dz_environment_condition_3 = wCO2_environment_mean_condition_3.differentiate('zt')
    
    wCO2_dz_arr_environment_condition_3.append(wCO2_dz_environment_condition_3)
    wCO2_arr_environment_condition_3.append(wCO2_environment_mean_condition_3)
    wCO2_arr_environment_nofactor_condition_3.append(wCO2_environment_mean_nofactor_condition_3)
        
    #-------------------------------------------------------------------------------------
    
    #qCO2
    
    #Condition 1
    qCO2_environment_condition_1 = qCO2.where(cloud_mask_3D_condition_1==0)
    qCO2_environment_mean_condition_1 = qCO2_environment_condition_1.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_1)
    
    qCO2_arr_environment_condition_1.append(qCO2_environment_mean_condition_1.values)
    
    #Condition 2
    qCO2_environment_condition_2 = qCO2.where(cloud_mask_3D_condition_2==0)
    qCO2_environment_mean_condition_2 = qCO2_environment_condition_2.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_2)
    
    qCO2_arr_environment_condition_2.append(qCO2_environment_mean_condition_2.values)
    
    #Condition 3
    qCO2_environment_condition_3 = qCO2.where(cloud_mask_3D_condition_3==0)
    qCO2_environment_mean_condition_3 = qCO2_environment_condition_3.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_3)
    
    qCO2_arr_environment_condition_3.append(qCO2_environment_mean_condition_3.values)
    
    #-------------------------------------------------------------------------------------
    
    #Correlation
    
    #Condition 1
    corr_environment_condition_1 = corr.where(cloud_mask_3D_condition_1==0)
    corr_environment_mean_condition_1 = corr_environment_condition_1.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_1)
    corr_arr_environment_condition_1.append(corr_environment_mean_condition_1)
    
    #Condition 2
    corr_environment_condition_2 = corr.where(cloud_mask_3D_condition_2==0)
    corr_environment_mean_condition_2 = corr_environment_condition_2.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_2)
    corr_arr_environment_condition_2.append(corr_environment_mean_condition_2)
    
    #Condition 3
    corr_environment_condition_3 = corr.where(cloud_mask_3D_condition_3==0)
    corr_environment_mean_condition_3 = corr_environment_condition_3.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_3)
    corr_arr_environment_condition_3.append(corr_environment_mean_condition_3)
    
    #-------------------------------------------------------------------------------------
        
    #wq
    
    #Condition 1
    wq_environment_condition_1 = wq.where(cloud_mask_3D_condition_1==0)
    wq_environment_mean_condition_1 = wq_environment_condition_1.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_1)
    wq_environment_nofactor_condition_1 = wq_environment_condition_1.mean(dim=('yt', 'xt'))
    
    wq_arr_environment_condition_1.append(wq_environment_mean_condition_1.values)
    wq_arr_environment_nofactor_condition_1.append(wq_environment_nofactor_condition_1.values)
    
    #Condition 2
    wq_environment_condition_2 = wq.where(cloud_mask_3D_condition_2==0)
    wq_environment_mean_condition_2 = wq_environment_condition_2.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_2)
    wq_environment_nofactor_condition_2 = wq_environment_condition_2.mean(dim=('yt', 'xt'))
    
    wq_arr_environment_condition_2.append(wq_environment_mean_condition_2.values)
    wq_arr_environment_nofactor_condition_2.append(wq_environment_nofactor_condition_2.values)
    
    #Condition 3
    wq_environment_condition_3 = wq.where(cloud_mask_3D_condition_3==0)
    wq_environment_mean_condition_3 = wq_environment_condition_3.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_3)
    wq_environment_nofactor_condition_3 = wq_environment_condition_3.mean(dim=('yt', 'xt'))
    
    wq_arr_environment_condition_3.append(wq_environment_mean_condition_3.values)
    wq_arr_environment_nofactor_condition_3.append(wq_environment_nofactor_condition_3.values)
    
    #-------------------------------------------------------------------------------------
    
    #Grad CO2
    
    #Condition 1
    grad_CO2_environment_condition_1 = grad_CO2.where(cloud_mask_3D_full_condition_1==0)
    grad_CO2_environment_mean_condition_1 = grad_CO2_environment_condition_1.mean(dim=('yt', 'xt')).values
    grad_CO2_arr_environment_condition_1.append(grad_CO2_environment_mean_condition_1)
    
    #Condition 2
    grad_CO2_environment_condition_2 = grad_CO2.where(cloud_mask_3D_full_condition_2==0)
    grad_CO2_environment_mean_condition_2 = grad_CO2_environment_condition_2.mean(dim=('yt', 'xt')).values
    grad_CO2_arr_environment_condition_2.append(grad_CO2_environment_mean_condition_2)
    
    #Condition 3
    grad_CO2_environment_condition_3 = grad_CO2.where(cloud_mask_3D_full_condition_3==0)
    grad_CO2_environment_mean_condition_3 = grad_CO2_environment_condition_3.mean(dim=('yt', 'xt')).values
    grad_CO2_arr_environment_condition_3.append(grad_CO2_environment_mean_condition_3)
    
    #-------------------------------------------------------------------------------------
    
    #Theta_v
    
    #Condition 1
    profile_theta_v_environment_condition_1.append(theta_v.where(cloud_mask_3D_condition_1==0).mean(dim=('yt', 'xt')).values)
    
    #Condition 2
    profile_theta_v_environment_condition_2.append(theta_v.where(cloud_mask_3D_condition_2==0).mean(dim=('yt', 'xt')).values)
    
    #Condition 3
    profile_theta_v_environment_condition_3.append(theta_v.where(cloud_mask_3D_condition_3==0).mean(dim=('yt', 'xt')).values)
    
    #-------------------------------------------------------------------------------------
    
    #Buoyancy
    
    #Condition 1
    buoy_environment_condition_1 = buoy.where(cloud_mask_3D_full_condition_1==0)
    buoy_environment_mean_condition_1 = buoy_environment_condition_1.mean(dim=('yt', 'xt')).values
    buoy_arr_environment_condition_1.append(buoy_environment_mean_condition_1)
    
    #Condition 2
    buoy_environment_condition_2 = buoy.where(cloud_mask_3D_full_condition_2==0)
    buoy_environment_mean_condition_2 = buoy_environment_condition_2.mean(dim=('yt', 'xt')).values
    buoy_arr_environment_condition_2.append(buoy_environment_mean_condition_2)
    
    #Condition 3
    buoy_environment_condition_3 = buoy.where(cloud_mask_3D_full_condition_3==0)
    buoy_environment_mean_condition_3 = buoy_environment_condition_3.mean(dim=('yt', 'xt')).values
    buoy_arr_environment_condition_3.append(buoy_environment_mean_condition_3)
    
    #-------------------------------------------------------------------------------------
    
    #CO2
    
    #Condition 1
    CO2_environment_condition_1 = CO2.where(cloud_mask_3D_condition_1==0)
    CO2_environment_mean_condition_1 = CO2_environment_condition_1.mean(dim=('yt', 'xt')).values
    CO2_arr_environment_condition_1.append(CO2_environment_mean_condition_1)
    
    #Condition 2
    CO2_environment_condition_2 = CO2.where(cloud_mask_3D_condition_2==0)
    CO2_environment_mean_condition_2 = CO2_environment_condition_2.mean(dim=('yt', 'xt')).values
    CO2_arr_environment_condition_2.append(CO2_environment_mean_condition_2)
    
    #Condition 3
    CO2_environment_condition_3 = CO2.where(cloud_mask_3D_condition_3==0)
    CO2_environment_mean_condition_3 = CO2_environment_condition_3.mean(dim=('yt', 'xt')).values
    CO2_arr_environment_condition_3.append(CO2_environment_mean_condition_3)
           
    #-------------------------------------------------------------------------------------
    #                           R 0 3 5  |  N O  C L O U D S 
    #-------------------------------------------------------------------------------------
    
    #CO2
    CO2_no_clouds = field_3D_no_clouds.sv001[time,:,:]/1000
    mean_CO2_no_clouds = CO2_no_clouds.mean(dim=('yt', 'xt')).values
    var_CO2_no_clouds = CO2_no_clouds - mean_CO2_no_clouds[:, None, None]
    
    grad_CO2_no_clouds = CO2_no_clouds.differentiate('zt')
    grad_CO2_no_clouds = grad_CO2_no_clouds.mean(dim=('yt', 'xt')).values
    grad_CO2_arr_no_clouds.append(grad_CO2_no_clouds)
    
    #q
    q_no_clouds = field_3D_no_clouds.qt[time,:,:] * 1000
    mean_q_no_clouds = q_no_clouds[:,:].mean(dim=('yt', 'xt')).values
    var_q_no_clouds = q_no_clouds[:,:] - mean_q_no_clouds[:, None, None]
    
    #qCO2
    qCO2_no_clouds = var_CO2_no_clouds[:-1, :, :] * var_q_no_clouds[:-1, :, :]
    qCO2_mean_no_clouds = qCO2_no_clouds.mean(dim=('yt', 'xt'))
    
    qCO2_arr_no_clouds.append(qCO2_mean_no_clouds.values)
    
    #corr
    std_q_no_clouds = var_q_no_clouds[:-1, :, :].std(dim=('yt', 'xt'))
    std_CO2_no_clouds = var_CO2_no_clouds[:-1, :, :].std(dim=('yt', 'xt'))
    
    corr_no_clouds = (qCO2_no_clouds)/(std_q_no_clouds*std_CO2_no_clouds)
    corr_mean_no_clouds = corr_no_clouds.mean(dim=('yt', 'xt')).values
    corr_arr_no_clouds.append(corr_mean_no_clouds)
    
    #Compute w
    w_no_clouds = (field_3D_no_clouds.w[time,:,:][0:, :, :] + field_3D_no_clouds.w[time,:,:][:-1, :, :]) / 2
    mean_w_no_clouds = w_no_clouds.mean(dim=('yt', 'xt'))
    var_w_no_clouds = w_no_clouds - mean_w_no_clouds
    
    #wCO2/dz
    wCO2_no_clouds = var_CO2_no_clouds[:-1, :, :] * var_w_no_clouds.values
    wCO2_mean_no_clouds = wCO2_no_clouds.mean(dim=('yt', 'xt'))
    wCO2_dz_mean_no_clouds = wCO2_mean_no_clouds.differentiate('zt')
    
    wCO2_dz_arr_no_clouds.append(wCO2_dz_mean_no_clouds.values)
    wCO2_arr_no_clouds.append(wCO2_mean_no_clouds.values)
        
    #wq
    wq_no_clouds = var_q_no_clouds[:-1, :, :] * var_w_no_clouds.values
    wq_mean_no_clouds = wq_no_clouds.mean(dim=('yt', 'xt'))
    wq_arr_no_clouds.append(wq_mean_no_clouds.values)    
    
    #wtheta
    theta_no_clouds = field_3D_no_clouds.thl[time,:,:]
    theta_v_no_clouds = theta_no_clouds * (1 + 0.61 * q_no_clouds/1000)
    theta_v_mean_no_clouds = theta_v_no_clouds.mean(dim=('yt', 'xt'))
    var_theta_v_no_clouds = theta_v_no_clouds - theta_v_mean_no_clouds
    
    #Theta_v 
    thl = field_3D_no_clouds.thl[time,:,:]
    qt = field_3D_no_clouds.qt[time,:,:]
    theta_v = thl * (1 + 0.61 * qt)
    mean_theta_v = theta_v.mean(dim=('yt', 'xt'))
    var_theta_v = theta_v - mean_theta_v
    
    profile_theta_v_no_clouds.append(theta_v.mean(dim=('yt', 'xt')).values)
    
    #Buoyancy
    buoy_no_clouds = field_3D_no_clouds.buoy[time,:,:]
    buoy_mean_no_clouds = buoy_no_clouds.mean(dim=('yt', 'xt'))
    buoy_arr_no_clouds.append(buoy_mean_no_clouds.values)
    
    #CO2
    CO2_no_clouds = field_3D_no_clouds.sv001[time,:,:]/1000
    mean_CO2_no_clouds = CO2_no_clouds.mean(dim=('yt', 'xt')).values
    CO2_arr_no_clouds.append(mean_CO2_no_clouds)

#-----------------------------------------------------------------------------------------------------------
#%%

#Condition 1
cloud_mask_2D_ds_condition_1 = xr.DataArray(cloud_mask_full_2D_condition_1,coords=[field_3D.time,field_3D.yt,field_3D.xt],dims=['time','yt','xt'])
cloud_mask_3D_condition_1 = cloud_mask_2D_ds_condition_1.expand_dims({'zt':field_3D.zt})
cloud_mask_3D_condition_1 = cloud_mask_3D_condition_1.transpose("time", "zt", "yt", "xt")

#Condition 2
cloud_mask_2D_ds_condition_2 = xr.DataArray(cloud_mask_full_2D_condition_2,coords=[field_3D.time,field_3D.yt,field_3D.xt],dims=['time','yt','xt'])
cloud_mask_3D_condition_2 = cloud_mask_2D_ds_condition_2.expand_dims({'zt':field_3D.zt})
cloud_mask_3D_condition_2 = cloud_mask_3D_condition_2.transpose("time", "zt", "yt", "xt")

#Condition 3
cloud_mask_2D_ds_condition_3 = xr.DataArray(cloud_mask_full_2D_condition_3,coords=[field_3D.time,field_3D.yt,field_3D.xt],dims=['time','yt','xt'])
cloud_mask_3D_condition_3 = cloud_mask_2D_ds_condition_3.expand_dims({'zt':field_3D.zt})
cloud_mask_3D_condition_3 = cloud_mask_3D_condition_3.transpose("time", "zt", "yt", "xt")

#-------------------------------------------------------------------------------------
#%%

#CO2/dt - clouds
CO2_diff = field_3D.sv001.differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean = np.nanmean(CO2_diff, axis=(2,3))
CO2_dt_mean = CO2_dt_mean[:,:-1]

#CO2/dt - inclouds

#Condition 1
CO2_diff_inclouds_condition_1 = field_3D.sv001.where(cloud_mask_3D_condition_1==1).differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_inclouds_condition_1 = np.nanmean(CO2_diff_inclouds_condition_1, axis=(2,3))
CO2_dt_mean_inclouds_condition_1 = CO2_dt_mean_inclouds_condition_1[:,:-1]

#Condition 2
CO2_diff_inclouds_condition_2 = field_3D.sv001.where(cloud_mask_3D_condition_2==1).differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_inclouds_condition_2 = np.nanmean(CO2_diff_inclouds_condition_2, axis=(2,3))
CO2_dt_mean_inclouds_condition_2 = CO2_dt_mean_inclouds_condition_2[:,:-1]

#Condition 3
CO2_diff_inclouds_condition_3 = field_3D.sv001.where(cloud_mask_3D_condition_3==1).differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_inclouds_condition_3 = np.nanmean(CO2_diff_inclouds_condition_3, axis=(2,3))
CO2_dt_mean_inclouds_condition_3 = CO2_dt_mean_inclouds_condition_3[:,:-1]

#CO2/dt - environment

#Condition 1
CO2_diff_environment_condition_1 = field_3D.sv001.where(cloud_mask_3D_condition_1==0).differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_environment_condition_1 = np.nanmean(CO2_diff_environment_condition_1, axis=(2,3))
CO2_dt_mean_environment_condition_1 = CO2_dt_mean_environment_condition_1[:,:-1]

#Condition 2
CO2_diff_environment_condition_2 = field_3D.sv001.where(cloud_mask_3D_condition_2==0).differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_environment_condition_2 = np.nanmean(CO2_diff_environment_condition_2, axis=(2,3))
CO2_dt_mean_environment_condition_2 = CO2_dt_mean_environment_condition_2[:,:-1]

#Condition 3
CO2_diff_environment_condition_3 = field_3D.sv001.where(cloud_mask_3D_condition_3==0).differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_environment_condition_3 = np.nanmean(CO2_diff_environment_condition_3, axis=(2,3))
CO2_dt_mean_environment_condition_3 = CO2_dt_mean_environment_condition_3[:,:-1]

#CO2/dt - no clouds
CO2_diff_no_clouds = field_3D_no_clouds.sv001.differentiate('time',datetime_unit='s') / 1000
CO2_dt_mean_no_clouds = np.nanmean(CO2_diff_no_clouds, axis=(2,3))
CO2_dt_mean_no_clouds = CO2_dt_mean_no_clouds[:,:-1]


#--------------------------------------------------------------------------------
#%%

#30 min

#Clouds

#wCO2/dz
wCO2_dz_arr = np.array(wCO2_dz_arr)
wCO2_dz_arr_ds = xr.DataArray(wCO2_dz_arr,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_30min_ds = wCO2_dz_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_30min_ds = wCO2_dz_arr_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_30min = wCO2_dz_arr_30min_ds.values

#CO2/dt
CO2_dt_mean_ds = xr.DataArray(CO2_dt_mean,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_30min_ds = CO2_dt_mean_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CCO2_dt_mean_30min_ds = CO2_dt_mean_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_30min = CO2_dt_mean_30min_ds.values

#wCO2
wCO2_arr = np.array(wCO2_arr)
wCO2_arr_ds = xr.DataArray(wCO2_arr,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_30min_ds = wCO2_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_30min_ds = wCO2_arr_30min_ds.assign_coords(time=new_time)
wCO2_arr_30min = wCO2_arr_30min_ds.values

#qCO2
qCO2_arr = np.array(qCO2_arr)
qCO2_arr_ds = xr.DataArray(qCO2_arr,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_30min_ds = qCO2_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_30min_ds = qCO2_arr_30min_ds.assign_coords(time=new_time)
qCO2_arr_30min = qCO2_arr_30min_ds.values

#corr
corr_arr = np.array(corr_arr)
corr_arr_ds = xr.DataArray(corr_arr,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_30min_ds = corr_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_30min_ds = corr_arr_30min_ds.assign_coords(time=new_time)
corr_arr_30min = corr_arr_30min_ds.values

#wq
wq_arr = np.array(wq_arr)
wq_arr_ds = xr.DataArray(wq_arr,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_30min_ds = wq_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_30min_ds = wq_arr_30min_ds.assign_coords(time=new_time)
wq_arr_30min = wq_arr_30min_ds.values

#Grad CO2
grad_CO2_arr = np.array(grad_CO2_arr)
grad_CO2_arr_ds = xr.DataArray(grad_CO2_arr,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_30min_ds = grad_CO2_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_30min_ds = grad_CO2_arr_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_30min = grad_CO2_arr_30min_ds.values


#theta_v
profile_theta_v = np.array(profile_theta_v_arr)
profile_theta_v_ds = xr.DataArray(profile_theta_v,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
profile_theta_v_30min_ds = profile_theta_v_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_30min_ds = profile_theta_v_30min_ds.assign_coords(time=new_time)
profile_theta_v_30min = profile_theta_v_30min_ds.values


#Buoyancy
buoy_arr = np.array(buoy_arr)
buoy_arr_ds = xr.DataArray(buoy_arr,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoy_arr_30min_ds = buoy_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoy_arr_30min_ds = buoy_arr_30min_ds.assign_coords(time=new_time)
buoy_arr_30min = buoy_arr_30min_ds.values

#CO2
CO2_arr = np.array(CO2_arr)
CO2_arr_ds = xr.DataArray(CO2_arr,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
CO2_arr_30min_ds = CO2_arr_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_30min_ds = CO2_arr_30min_ds.assign_coords(time=new_time)
CO2_arr_30min = CO2_arr_30min_ds.values

#%%
#--------------------------------------------------------------------------------
#Inclouds

#wCO2/dz

#Condition 1
wCO2_dz_arr_inclouds_condition_1 = np.array(wCO2_dz_arr_inclouds_condition_1)
wCO2_dz_arr_inclouds_condition_1_ds = xr.DataArray(wCO2_dz_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_inclouds_condition_1_30min_ds = wCO2_dz_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_inclouds_condition_1_30min_ds = wCO2_dz_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_inclouds_condition_1_30min = wCO2_dz_arr_inclouds_condition_1_30min_ds.values

#Condition 2
wCO2_dz_arr_inclouds_condition_2 = np.array(wCO2_dz_arr_inclouds_condition_2)
wCO2_dz_arr_inclouds_condition_2_ds = xr.DataArray(wCO2_dz_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_inclouds_condition_2_30min_ds = wCO2_dz_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_inclouds_condition_2_30min_ds = wCO2_dz_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_inclouds_condition_2_30min = wCO2_dz_arr_inclouds_condition_2_30min_ds.values

#Condition 3
wCO2_dz_arr_inclouds_condition_3 = np.array(wCO2_dz_arr_inclouds_condition_3)
wCO2_dz_arr_inclouds_condition_3_ds = xr.DataArray(wCO2_dz_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_inclouds_condition_3_30min_ds = wCO2_dz_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_inclouds_condition_3_30min_ds = wCO2_dz_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_inclouds_condition_3_30min = wCO2_dz_arr_inclouds_condition_3_30min_ds.values

#CO2/dt

#Condition 1
CO2_dt_mean_inclouds_condition_1_ds = xr.DataArray(CO2_dt_mean_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_inclouds_condition_1_30min_ds = CO2_dt_mean_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_inclouds_condition_1_30min_ds = CO2_dt_mean_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_inclouds_condition_1_30min = CO2_dt_mean_inclouds_condition_1_30min_ds.values

#Condition 2
CO2_dt_mean_inclouds_condition_2_ds = xr.DataArray(CO2_dt_mean_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_inclouds_condition_2_30min_ds = CO2_dt_mean_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_inclouds_condition_2_30min_ds = CO2_dt_mean_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_inclouds_condition_2_30min = CO2_dt_mean_inclouds_condition_2_30min_ds.values

#Condition 3
CO2_dt_mean_inclouds_condition_3_ds = xr.DataArray(CO2_dt_mean_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_inclouds_condition_3_30min_ds = CO2_dt_mean_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_inclouds_condition_3_30min_ds = CO2_dt_mean_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_inclouds_condition_3_30min = CO2_dt_mean_inclouds_condition_3_30min_ds.values

#wCO2

#Condition 1
wCO2_arr_inclouds_condition_1 = np.array(wCO2_arr_inclouds_condition_1)
wCO2_arr_inclouds_condition_1_ds = xr.DataArray(wCO2_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_inclouds_condition_1_30min_ds = wCO2_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_inclouds_condition_1_30min_ds = wCO2_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
wCO2_arr_inclouds_condition_1_30min = wCO2_arr_inclouds_condition_1_30min_ds.values

wCO2_arr_inclouds_nofactor_condition_1 = np.array(wCO2_arr_inclouds_nofactor_condition_1)
wCO2_arr_inclouds_nofactor_condition_1_ds = xr.DataArray(wCO2_arr_inclouds_nofactor_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_inclouds_nofactor_condition_1_30min_ds = wCO2_arr_inclouds_nofactor_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_inclouds_nofactor_condition_1_30min_ds = wCO2_arr_inclouds_nofactor_condition_1_30min_ds.assign_coords(time=new_time)
wCO2_arr_inclouds_nofactor_condition_1_30min = wCO2_arr_inclouds_nofactor_condition_1_30min_ds.values

#Condition 2
wCO2_arr_inclouds_condition_2 = np.array(wCO2_arr_inclouds_condition_2)
wCO2_arr_inclouds_condition_2_ds = xr.DataArray(wCO2_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_inclouds_condition_2_30min_ds = wCO2_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_inclouds_condition_2_30min_ds = wCO2_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
wCO2_arr_inclouds_condition_2_30min = wCO2_arr_inclouds_condition_2_30min_ds.values

wCO2_arr_inclouds_nofactor_condition_2 = np.array(wCO2_arr_inclouds_nofactor_condition_2)
wCO2_arr_inclouds_nofactor_condition_2_ds = xr.DataArray(wCO2_arr_inclouds_nofactor_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_inclouds_nofactor_condition_2_30min_ds = wCO2_arr_inclouds_nofactor_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_inclouds_nofactor_condition_2_30min_ds = wCO2_arr_inclouds_nofactor_condition_2_30min_ds.assign_coords(time=new_time)
wCO2_arr_inclouds_nofactor_condition_2_30min = wCO2_arr_inclouds_nofactor_condition_2_30min_ds.values

#Condition 3
wCO2_arr_inclouds_condition_3 = np.array(wCO2_arr_inclouds_condition_3)
wCO2_arr_inclouds_condition_3_ds = xr.DataArray(wCO2_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_inclouds_condition_3_30min_ds = wCO2_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_inclouds_condition_3_30min_ds = wCO2_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
wCO2_arr_inclouds_condition_3_30min = wCO2_arr_inclouds_condition_3_30min_ds.values

wCO2_arr_inclouds_nofactor_condition_3 = np.array(wCO2_arr_inclouds_nofactor_condition_3)
wCO2_arr_inclouds_nofactor_condition_3_ds = xr.DataArray(wCO2_arr_inclouds_nofactor_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_inclouds_nofactor_condition_3_30min_ds = wCO2_arr_inclouds_nofactor_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_inclouds_nofactor_condition_3_30min_ds = wCO2_arr_inclouds_nofactor_condition_3_30min_ds.assign_coords(time=new_time)
wCO2_arr_inclouds_nofactor_condition_3_30min = wCO2_arr_inclouds_nofactor_condition_3_30min_ds.values

#qCO2

#Condition 1
qCO2_arr_inclouds_condition_1 = np.array(qCO2_arr_inclouds_condition_1)
qCO2_arr_inclouds_condition_1_ds = xr.DataArray(qCO2_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_inclouds_condition_1_30min_ds = qCO2_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_inclouds_condition_1_30min_ds = qCO2_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
qCO2_arr_inclouds_condition_1_30min = qCO2_arr_inclouds_condition_1_30min_ds.values

#Condition 2
qCO2_arr_inclouds_condition_2 = np.array(qCO2_arr_inclouds_condition_2)
qCO2_arr_inclouds_condition_2_ds = xr.DataArray(qCO2_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_inclouds_condition_2_30min_ds = qCO2_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_inclouds_condition_2_30min_ds = qCO2_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
qCO2_arr_inclouds_condition_2_30min = qCO2_arr_inclouds_condition_2_30min_ds.values

#Condition 3
qCO2_arr_inclouds_condition_3 = np.array(qCO2_arr_inclouds_condition_3)
qCO2_arr_inclouds_condition_3_ds = xr.DataArray(qCO2_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_inclouds_condition_3_30min_ds = qCO2_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_inclouds_condition_3_30min_ds = qCO2_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
qCO2_arr_inclouds_condition_3_30min = qCO2_arr_inclouds_condition_3_30min_ds.values

#corr

#Condition 1
corr_arr_inclouds_condition_1 = np.array(corr_arr_inclouds_condition_1)
corr_arr_inclouds_condition_1_ds = xr.DataArray(corr_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_inclouds_condition_1_30min_ds = corr_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_inclouds_condition_1_30min_ds = corr_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
corr_arr_inclouds_condition_1_30min = corr_arr_inclouds_condition_1_30min_ds.values

#Condition 2
corr_arr_inclouds_condition_2 = np.array(corr_arr_inclouds_condition_2)
corr_arr_inclouds_condition_2_ds = xr.DataArray(corr_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_inclouds_condition_2_30min_ds = corr_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_inclouds_condition_2_30min_ds = corr_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
corr_arr_inclouds_condition_2_30min = corr_arr_inclouds_condition_2_30min_ds.values

#Condition 3
corr_arr_inclouds_condition_3 = np.array(corr_arr_inclouds_condition_3)
corr_arr_inclouds_condition_3_ds = xr.DataArray(corr_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_inclouds_condition_3_30min_ds = corr_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_inclouds_condition_3_30min_ds = corr_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
corr_arr_inclouds_condition_3_30min = corr_arr_inclouds_condition_3_30min_ds.values

#wq

#Condition 1
wq_arr_inclouds_condition_1 = np.array(wq_arr_inclouds_condition_1)
wq_arr_inclouds_condition_1_ds = xr.DataArray(wq_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_inclouds_condition_1_30min_ds = wq_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_inclouds_condition_1_30min_ds = wq_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
wq_arr_inclouds_condition_1_30min = wq_arr_inclouds_condition_1_30min_ds.values

wq_arr_inclouds_nofactor_condition_1 = np.array(wq_arr_inclouds_nofactor_condition_1)
wq_arr_inclouds_nofactor_condition_1_ds = xr.DataArray(wq_arr_inclouds_nofactor_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_inclouds_nofactor_condition_1_30min_ds = wq_arr_inclouds_nofactor_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_inclouds_nofactor_condition_1_30min_ds = wq_arr_inclouds_nofactor_condition_1_30min_ds.assign_coords(time=new_time)
wq_arr_inclouds_nofactor_condition_1_30min = wq_arr_inclouds_nofactor_condition_1_30min_ds.values

#Condition 2
wq_arr_inclouds_condition_2 = np.array(wq_arr_inclouds_condition_2)
wq_arr_inclouds_condition_2_ds = xr.DataArray(wq_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_inclouds_condition_2_30min_ds = wq_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_inclouds_condition_2_30min_ds = wq_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
wq_arr_inclouds_condition_2_30min = wq_arr_inclouds_condition_2_30min_ds.values

wq_arr_inclouds_nofactor_condition_2 = np.array(wq_arr_inclouds_nofactor_condition_2)
wq_arr_inclouds_nofactor_condition_2_ds = xr.DataArray(wq_arr_inclouds_nofactor_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_inclouds_nofactor_condition_2_30min_ds = wq_arr_inclouds_nofactor_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_inclouds_nofactor_condition_2_30min_ds = wq_arr_inclouds_nofactor_condition_2_30min_ds.assign_coords(time=new_time)
wq_arr_inclouds_nofactor_condition_2_30min = wq_arr_inclouds_nofactor_condition_2_30min_ds.values

#Condition 3
wq_arr_inclouds_condition_3 = np.array(wq_arr_inclouds_condition_3)
wq_arr_inclouds_condition_3_ds = xr.DataArray(wq_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_inclouds_condition_3_30min_ds = wq_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_inclouds_condition_3_30min_ds = wq_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
wq_arr_inclouds_condition_3_30min = wq_arr_inclouds_condition_3_30min_ds.values

wq_arr_inclouds_nofactor_condition_3 = np.array(wq_arr_inclouds_nofactor_condition_3)
wq_arr_inclouds_nofactor_condition_3_ds = xr.DataArray(wq_arr_inclouds_nofactor_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_inclouds_nofactor_condition_3_30min_ds = wq_arr_inclouds_nofactor_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_inclouds_nofactor_condition_3_30min_ds = wq_arr_inclouds_nofactor_condition_3_30min_ds.assign_coords(time=new_time)
wq_arr_inclouds_nofactor_condition_3_30min = wq_arr_inclouds_nofactor_condition_3_30min_ds.values

#Grad CO2

#Condition 1
grad_CO2_arr_inclouds_condition_1 = np.array(grad_CO2_arr_inclouds_condition_1)
grad_CO2_arr_inclouds_condition_1_ds = xr.DataArray(grad_CO2_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_inclouds_condition_1_30min_ds = grad_CO2_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_inclouds_condition_1_30min_ds = grad_CO2_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_inclouds_condition_1_30min = grad_CO2_arr_inclouds_condition_1_30min_ds.values

#Condition 2
grad_CO2_arr_inclouds_condition_2 = np.array(grad_CO2_arr_inclouds_condition_2)
grad_CO2_arr_inclouds_condition_2_ds = xr.DataArray(grad_CO2_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_inclouds_condition_2_30min_ds = grad_CO2_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_inclouds_condition_2_30min_ds = grad_CO2_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_inclouds_condition_2_30min = grad_CO2_arr_inclouds_condition_2_30min_ds.values

#Condition 3
grad_CO2_arr_inclouds_condition_3 = np.array(grad_CO2_arr_inclouds_condition_3)
grad_CO2_arr_inclouds_condition_3_ds = xr.DataArray(grad_CO2_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_inclouds_condition_3_30min_ds = grad_CO2_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_inclouds_condition_3_30min_ds = grad_CO2_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_inclouds_condition_3_30min = grad_CO2_arr_inclouds_condition_3_30min_ds.values

#theta_v

#Condition 1
profile_theta_v_inclouds_condition_1 = np.array(profile_theta_v_inclouds_condition_1)
profile_theta_v_inclouds_condition_1_ds = xr.DataArray(profile_theta_v_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
profile_theta_v_inclouds_condition_1_30min_ds = profile_theta_v_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_inclouds_condition_1_30min_ds = profile_theta_v_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
profile_theta_v_inclouds_condition_1_30min = profile_theta_v_inclouds_condition_1_30min_ds.values

#Condition 2
profile_theta_v_inclouds_condition_2 = np.array(profile_theta_v_inclouds_condition_2)
profile_theta_v_inclouds_condition_2_ds = xr.DataArray(profile_theta_v_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
profile_theta_v_inclouds_condition_2_30min_ds = profile_theta_v_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_inclouds_condition_2_30min_ds = profile_theta_v_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
profile_theta_v_inclouds_condition_2_30min = profile_theta_v_inclouds_condition_2_30min_ds.values

#Condition 3
profile_theta_v_inclouds_condition_3 = np.array(profile_theta_v_inclouds_condition_3)
profile_theta_v_inclouds_condition_3_ds = xr.DataArray(profile_theta_v_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
profile_theta_v_inclouds_condition_3_30min_ds = profile_theta_v_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_inclouds_condition_3_30min_ds = profile_theta_v_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
profile_theta_v_inclouds_condition_3_30min = profile_theta_v_inclouds_condition_3_30min_ds.values

#cfrac

#Condition 1
cfrac_condition_1 = np.array(cfrac_condition_1)
cfrac_condition_1_ds = xr.DataArray(cfrac_condition_1,coords=[field_3D.time],dims=['time'])
cfrac_condition_1_30min_ds = cfrac_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
cfrac_condition_1_30min_ds = cfrac_condition_1_30min_ds.assign_coords(time=new_time)
cfrac_condition_1_30min = cfrac_condition_1_30min_ds.values

#Condition 2
cfrac_condition_2 = np.array(cfrac_condition_2)
cfrac_condition_2_ds = xr.DataArray(cfrac_condition_2,coords=[field_3D.time],dims=['time'])
cfrac_condition_2_30min_ds = cfrac_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
cfrac_condition_2_30min_ds = cfrac_condition_2_30min_ds.assign_coords(time=new_time)
cfrac_condition_2_30min = cfrac_condition_2_30min_ds.values

#Condition 3
cfrac_condition_3 = np.array(cfrac_condition_3)
cfrac_condition_3_ds = xr.DataArray(cfrac_condition_3,coords=[field_3D.time],dims=['time'])
cfrac_condition_3_30min_ds = cfrac_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
cfrac_condition_3_30min_ds = cfrac_condition_3_30min_ds.assign_coords(time=new_time)
cfrac_condition_3_30min = cfrac_condition_3_30min_ds.values

#--------------------------------------------------------------------------------

#Buoancy

#Condition 1
buoyancy_arr_inclouds_condition_1 = np.array(buoy_arr_inclouds_condition_1)
buoyancy_arr_inclouds_condition_1_ds = xr.DataArray(buoyancy_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_inclouds_condition_1_30min_ds = buoyancy_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_inclouds_condition_1_30min_ds = buoyancy_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
buoyancy_arr_inclouds_condition_1_30min = buoyancy_arr_inclouds_condition_1_30min_ds.values

#Condition 2
buoyancy_arr_inclouds_condition_2 = np.array(buoy_arr_inclouds_condition_2)
buoyancy_arr_inclouds_condition_2_ds = xr.DataArray(buoyancy_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_inclouds_condition_2_30min_ds = buoyancy_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_inclouds_condition_2_30min_ds = buoyancy_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
buoyancy_arr_inclouds_condition_2_30min = buoyancy_arr_inclouds_condition_2_30min_ds.values

#Condition 3
buoyancy_arr_inclouds_condition_3 = np.array(buoy_arr_inclouds_condition_3)
buoyancy_arr_inclouds_condition_3_ds = xr.DataArray(buoyancy_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_inclouds_condition_3_30min_ds = buoyancy_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_inclouds_condition_3_30min_ds = buoyancy_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
buoyancy_arr_inclouds_condition_3_30min = buoyancy_arr_inclouds_condition_3_30min_ds.values

#--------------------------------------------------------------------------------

#CO2

#Condition 1
CO2_arr_inclouds_condition_1 = np.array(CO2_arr_inclouds_condition_1)
CO2_arr_inclouds_condition_1_ds = xr.DataArray(CO2_arr_inclouds_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_arr_inclouds_condition_1_30min_ds = CO2_arr_inclouds_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_inclouds_condition_1_30min_ds = CO2_arr_inclouds_condition_1_30min_ds.assign_coords(time=new_time)
CO2_arr_inclouds_condition_1_30min = CO2_arr_inclouds_condition_1_30min_ds.values

#Condition 2
CO2_arr_inclouds_condition_2 = np.array(CO2_arr_inclouds_condition_2)
CO2_arr_inclouds_condition_2_ds = xr.DataArray(CO2_arr_inclouds_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_arr_inclouds_condition_2_30min_ds = CO2_arr_inclouds_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_inclouds_condition_2_30min_ds = CO2_arr_inclouds_condition_2_30min_ds.assign_coords(time=new_time)
CO2_arr_inclouds_condition_2_30min = CO2_arr_inclouds_condition_2_30min_ds.values

#Condition 3
CO2_arr_inclouds_condition_3 = np.array(CO2_arr_inclouds_condition_3)
CO2_arr_inclouds_condition_3_ds = xr.DataArray(CO2_arr_inclouds_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_arr_inclouds_condition_3_30min_ds = CO2_arr_inclouds_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_inclouds_condition_3_30min_ds = CO2_arr_inclouds_condition_3_30min_ds.assign_coords(time=new_time)
CO2_arr_inclouds_condition_3_30min = CO2_arr_inclouds_condition_3_30min_ds.values


#%%
#--------------------------------------------------------------------------------

#Environment

#wCO2/dz

#Condition 1
wCO2_dz_arr_environment_condition_1 = np.array(wCO2_dz_arr_environment_condition_1)
wCO2_dz_arr_environment_condition_1_ds = xr.DataArray(wCO2_dz_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_environment_condition_1_30min_ds = wCO2_dz_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_environment_condition_1_30min_ds = wCO2_dz_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_environment_condition_1_30min = wCO2_dz_arr_environment_condition_1_30min_ds.values

#Condition 2
wCO2_dz_arr_environment_condition_2 = np.array(wCO2_dz_arr_environment_condition_2)
wCO2_dz_arr_environment_condition_2_ds = xr.DataArray(wCO2_dz_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_environment_condition_2_30min_ds = wCO2_dz_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_environment_condition_2_30min_ds = wCO2_dz_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_environment_condition_2_30min = wCO2_dz_arr_environment_condition_2_30min_ds.values

#Condition 3
wCO2_dz_arr_environment_condition_3 = np.array(wCO2_dz_arr_environment_condition_3)
wCO2_dz_arr_environment_condition_3_ds = xr.DataArray(wCO2_dz_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_environment_condition_3_30min_ds = wCO2_dz_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_environment_condition_3_30min_ds = wCO2_dz_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_environment_condition_3_30min = wCO2_dz_arr_environment_condition_3_30min_ds.values

#CO2/dt

#Condition 1
CO2_dt_mean_environment_condition_1_ds = xr.DataArray(CO2_dt_mean_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_environment_condition_1_30min_ds = CO2_dt_mean_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_environment_condition_1_30min_ds = CO2_dt_mean_environment_condition_1_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_environment_condition_1_30min = CO2_dt_mean_environment_condition_1_30min_ds.values

#Condition 2
CO2_dt_mean_environment_condition_2_ds = xr.DataArray(CO2_dt_mean_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_environment_condition_2_30min_ds = CO2_dt_mean_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_environment_condition_2_30min_ds = CO2_dt_mean_environment_condition_2_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_environment_condition_2_30min = CO2_dt_mean_environment_condition_2_30min_ds.values

#Condition 3
CO2_dt_mean_environment_condition_3_ds = xr.DataArray(CO2_dt_mean_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_environment_condition_3_30min_ds = CO2_dt_mean_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_environment_condition_3_30min_ds = CO2_dt_mean_environment_condition_3_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_environment_condition_3_30min = CO2_dt_mean_environment_condition_3_30min_ds.values

#wCO2

#Condition 1
wCO2_arr_environment_condition_1 = np.array(wCO2_arr_environment_condition_1)
wCO2_arr_environment_condition_1_ds = xr.DataArray(wCO2_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_environment_condition_1_30min_ds = wCO2_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_environment_condition_1_30min_ds = wCO2_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
wCO2_arr_environment_condition_1_30min = wCO2_arr_environment_condition_1_30min_ds.values

wCO2_arr_environment_nofactor_condition_1 = np.array(wCO2_arr_environment_nofactor_condition_1)
wCO2_arr_environment_nofactor_condition_1_ds = xr.DataArray(wCO2_arr_environment_nofactor_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_environment_nofactor_condition_1_30min_ds = wCO2_arr_environment_nofactor_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_environment_nofactor_condition_1_30min_ds = wCO2_arr_environment_nofactor_condition_1_30min_ds.assign_coords(time=new_time)
wCO2_arr_environment_nofactor_condition_1_30min = wCO2_arr_environment_nofactor_condition_1_30min_ds.values

#Condition 2
wCO2_arr_environment_condition_2 = np.array(wCO2_arr_environment_condition_2)
wCO2_arr_environment_condition_2_ds = xr.DataArray(wCO2_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_environment_condition_2_30min_ds = wCO2_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_environment_condition_2_30min_ds = wCO2_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
wCO2_arr_environment_condition_2_30min = wCO2_arr_environment_condition_2_30min_ds.values

wCO2_arr_environment_nofactor_condition_2 = np.array(wCO2_arr_environment_nofactor_condition_2)
wCO2_arr_environment_nofactor_condition_2_ds = xr.DataArray(wCO2_arr_environment_nofactor_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_environment_nofactor_condition_2_30min_ds = wCO2_arr_environment_nofactor_condition_2_ds.coarsen(time=3, boundary='trim').mean()
wCO2_arr_environment_nofactor_condition_2_30min_ds = wCO2_arr_environment_nofactor_condition_2_30min_ds.assign_coords(time=new_time)
wCO2_arr_environment_nofactor_condition_2_30min = wCO2_arr_environment_nofactor_condition_2_30min_ds.values

#Condition 3
wCO2_arr_environment_condition_3 = np.array(wCO2_arr_environment_condition_3)
wCO2_arr_environment_condition_3_ds = xr.DataArray(wCO2_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_environment_condition_3_30min_ds = wCO2_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_environment_condition_3_30min_ds = wCO2_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
wCO2_arr_environment_condition_3_30min = wCO2_arr_environment_condition_3_30min_ds.values

wCO2_arr_environment_nofactor_condition_3 = np.array(wCO2_arr_environment_nofactor_condition_3)
wCO2_arr_environment_nofactor_condition_3_ds = xr.DataArray(wCO2_arr_environment_nofactor_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_environment_nofactor_condition_3_30min_ds = wCO2_arr_environment_nofactor_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_environment_nofactor_condition_3_30min_ds = wCO2_arr_environment_nofactor_condition_3_30min_ds.assign_coords(time=new_time)
wCO2_arr_environment_nofactor_condition_3_30min = wCO2_arr_environment_nofactor_condition_3_30min_ds.values

#qCO2

#Condition 1
qCO2_arr_environment_condition_1 = np.array(qCO2_arr_environment_condition_1)
qCO2_arr_environment_condition_1_ds = xr.DataArray(qCO2_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_environment_condition_1_30min_ds = qCO2_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_environment_condition_1_30min_ds = qCO2_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
qCO2_arr_environment_condition_1_30min = qCO2_arr_environment_condition_1_30min_ds.values

#Condition 2
qCO2_arr_environment_condition_2 = np.array(qCO2_arr_environment_condition_2)
qCO2_arr_environment_condition_2_ds = xr.DataArray(qCO2_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_environment_condition_2_30min_ds = qCO2_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_environment_condition_2_30min_ds = qCO2_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
qCO2_arr_environment_condition_2_30min = qCO2_arr_environment_condition_2_30min_ds.values

#Condition 3
qCO2_arr_environment_condition_3 = np.array(qCO2_arr_environment_condition_3)
qCO2_arr_environment_condition_3_ds = xr.DataArray(qCO2_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_environment_condition_3_30min_ds = qCO2_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_environment_condition_3_30min_ds = qCO2_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
qCO2_arr_environment_condition_3_30min = qCO2_arr_environment_condition_3_30min_ds.values

#corr

#Condition 1
corr_arr_environment_condition_1 = np.array(corr_arr_environment_condition_1)
corr_arr_environment_condition_1_ds = xr.DataArray(corr_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_environment_condition_1_30min_ds = corr_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_environment_condition_1_30min_ds = corr_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
corr_arr_environment_condition_1_30min = corr_arr_environment_condition_1_30min_ds.values

#Condition 2
corr_arr_environment_condition_2 = np.array(corr_arr_environment_condition_2)
corr_arr_environment_condition_2_ds = xr.DataArray(corr_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_environment_condition_2_30min_ds = corr_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_environment_condition_2_30min_ds = corr_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
corr_arr_environment_condition_2_30min = corr_arr_environment_condition_2_30min_ds.values

#Condition 3
corr_arr_environment_condition_3 = np.array(corr_arr_environment_condition_3)
corr_arr_environment_condition_3_ds = xr.DataArray(corr_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_environment_condition_3_30min_ds = corr_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_environment_condition_3_30min_ds = corr_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
corr_arr_environment_condition_3_30min = corr_arr_environment_condition_3_30min_ds.values

#wq

#Condition 1
wq_arr_environment_condition_1 = np.array(wq_arr_environment_condition_1)
wq_arr_environment_condition_1_ds = xr.DataArray(wq_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_environment_condition_1_30min_ds = wq_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_environment_condition_1_30min_ds = wq_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
wq_arr_environment_condition_1_30min = wq_arr_environment_condition_1_30min_ds.values

wq_arr_environment_nofactor_condition_1 = np.array(wq_arr_environment_nofactor_condition_1)
wq_arr_environment_nofactor_condition_1_ds = xr.DataArray(wq_arr_environment_nofactor_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_environment_nofactor_condition_1_30min_ds = wq_arr_environment_nofactor_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_environment_nofactor_condition_1_30min_ds = wq_arr_environment_nofactor_condition_1_30min_ds.assign_coords(time=new_time)
wq_arr_environment_nofactor_condition_1_30min = wq_arr_environment_nofactor_condition_1_30min_ds.values

#Condition 2
wq_arr_environment_condition_2 = np.array(wq_arr_environment_condition_2)
wq_arr_environment_condition_2_ds = xr.DataArray(wq_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_environment_condition_2_30min_ds = wq_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_environment_condition_2_30min_ds = wq_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
wq_arr_environment_condition_2_30min = wq_arr_environment_condition_2_30min_ds.values

wq_arr_environment_nofactor_condition_2 = np.array(wq_arr_environment_nofactor_condition_2)
wq_arr_environment_nofactor_condition_2_ds = xr.DataArray(wq_arr_environment_nofactor_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_environment_nofactor_condition_2_30min_ds = wq_arr_environment_nofactor_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_environment_nofactor_condition_2_30min_ds = wq_arr_environment_nofactor_condition_2_30min_ds.assign_coords(time=new_time)
wq_arr_environment_nofactor_condition_2_30min = wq_arr_environment_nofactor_condition_2_30min_ds.values

#Condition 3
wq_arr_environment_condition_3 = np.array(wq_arr_environment_condition_3)
wq_arr_environment_condition_3_ds = xr.DataArray(wq_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_environment_condition_3_30min_ds = wq_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_environment_condition_3_30min_ds = wq_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
wq_arr_environment_condition_3_30min = wq_arr_environment_condition_3_30min_ds.values

wq_arr_environment_nofactor_condition_3 = np.array(wq_arr_environment_nofactor_condition_3)
wq_arr_environment_nofactor_condition_3_ds = xr.DataArray(wq_arr_environment_nofactor_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_environment_nofactor_condition_3_30min_ds = wq_arr_environment_nofactor_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_environment_nofactor_condition_3_30min_ds = wq_arr_environment_nofactor_condition_3_30min_ds.assign_coords(time=new_time)
wq_arr_environment_nofactor_condition_3_30min = wq_arr_environment_nofactor_condition_3_30min_ds.values

#Grad CO2

#Condition 1
grad_CO2_arr_environment_condition_1 = np.array(grad_CO2_arr_environment_condition_1)
grad_CO2_arr_environment_condition_1_ds = xr.DataArray(grad_CO2_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_environment_condition_1_30min_ds = grad_CO2_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_environment_condition_1_30min_ds = grad_CO2_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_environment_condition_1_30min = grad_CO2_arr_environment_condition_1_30min_ds.values

#Condition 2
grad_CO2_arr_environment_condition_2 = np.array(grad_CO2_arr_environment_condition_2)
grad_CO2_arr_environment_condition_2_ds = xr.DataArray(grad_CO2_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_environment_condition_2_30min_ds = grad_CO2_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_environment_condition_2_30min_ds = grad_CO2_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_environment_condition_2_30min = grad_CO2_arr_environment_condition_2_30min_ds.values

#Condition 3
grad_CO2_arr_environment_condition_3 = np.array(grad_CO2_arr_environment_condition_3)
grad_CO2_arr_environment_condition_3_ds = xr.DataArray(grad_CO2_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_environment_condition_3_30min_ds = grad_CO2_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_environment_condition_3_30min_ds = grad_CO2_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_environment_condition_3_30min = grad_CO2_arr_environment_condition_3_30min_ds.values

#Theta_v

#Condition 1
profile_theta_v_environment_condition_1 = np.array(profile_theta_v_environment_condition_1)
profile_theta_v_environment_condition_1_ds = xr.DataArray(profile_theta_v_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
profile_theta_v_environment_condition_1_30min_ds = profile_theta_v_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_environment_condition_1_30min_ds = profile_theta_v_environment_condition_1_30min_ds.assign_coords(time=new_time)
profile_theta_v_environment_condition_1_30min = profile_theta_v_environment_condition_1_30min_ds.values

#Condition 2
profile_theta_v_environment_condition_2 = np.array(profile_theta_v_environment_condition_2)
profile_theta_v_environment_condition_2_ds = xr.DataArray(profile_theta_v_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
profile_theta_v_environment_condition_2_30min_ds = profile_theta_v_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_environment_condition_2_30min_ds = profile_theta_v_environment_condition_2_30min_ds.assign_coords(time=new_time)
profile_theta_v_environment_condition_2_30min = profile_theta_v_environment_condition_2_30min_ds.values

#Condition 3
profile_theta_v_environment_condition_3 = np.array(profile_theta_v_environment_condition_3)
profile_theta_v_environment_condition_3_ds = xr.DataArray(profile_theta_v_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
profile_theta_v_environment_condition_3_30min_ds = profile_theta_v_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_environment_condition_3_30min_ds = profile_theta_v_environment_condition_3_30min_ds.assign_coords(time=new_time)
profile_theta_v_environment_condition_3_30min = profile_theta_v_environment_condition_3_30min_ds.values

#Buoancy

#Condition 1
buoyancy_arr_environment_condition_1 = np.array(buoy_arr_environment_condition_1)
buoyancy_arr_environment_condition_1_ds = xr.DataArray(buoyancy_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_environment_condition_1_30min_ds = buoyancy_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_environment_condition_1_30min_ds = buoyancy_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
buoyancy_arr_environment_condition_1_30min = buoyancy_arr_environment_condition_1_30min_ds.values

#Condition 2
buoyancy_arr_environment_condition_2 = np.array(buoy_arr_environment_condition_2)
buoyancy_arr_environment_condition_2_ds = xr.DataArray(buoyancy_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_environment_condition_2_30min_ds = buoyancy_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_environment_condition_2_30min_ds = buoyancy_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
buoyancy_arr_environment_condition_2_30min = buoyancy_arr_environment_condition_2_30min_ds.values

#Condition 3
buoyancy_arr_environment_condition_3 = np.array(buoy_arr_environment_condition_3)
buoyancy_arr_environment_condition_3_ds = xr.DataArray(buoyancy_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_environment_condition_3_30min_ds = buoyancy_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_environment_condition_3_30min_ds = buoyancy_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
buoyancy_arr_environment_condition_3_30min = buoyancy_arr_environment_condition_3_30min_ds.values

#CO2

#Condition 1
CO2_arr_environment_condition_1 = np.array(CO2_arr_environment_condition_1)
CO2_arr_environment_condition_1_ds = xr.DataArray(CO2_arr_environment_condition_1,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_arr_environment_condition_1_30min_ds = CO2_arr_environment_condition_1_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_environment_condition_1_30min_ds = CO2_arr_environment_condition_1_30min_ds.assign_coords(time=new_time)
CO2_arr_environment_condition_1_30min = CO2_arr_environment_condition_1_30min_ds.values

#Condition 2
CO2_arr_environment_condition_2 = np.array(CO2_arr_environment_condition_2)
CO2_arr_environment_condition_2_ds = xr.DataArray(CO2_arr_environment_condition_2,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_arr_environment_condition_2_30min_ds = CO2_arr_environment_condition_2_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_environment_condition_2_30min_ds = CO2_arr_environment_condition_2_30min_ds.assign_coords(time=new_time)
CO2_arr_environment_condition_2_30min = CO2_arr_environment_condition_2_30min_ds.values

#Condition 3
CO2_arr_environment_condition_3 = np.array(CO2_arr_environment_condition_3)
CO2_arr_environment_condition_3_ds = xr.DataArray(CO2_arr_environment_condition_3,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_arr_environment_condition_3_30min_ds = CO2_arr_environment_condition_3_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_environment_condition_3_30min_ds = CO2_arr_environment_condition_3_30min_ds.assign_coords(time=new_time)
CO2_arr_environment_condition_3_30min = CO2_arr_environment_condition_3_30min_ds.values


#%%
#--------------------------------------------------------------------------------

#No clouds

#wCO2/dz
wCO2_dz_arr_no_clouds = np.array(wCO2_dz_arr_no_clouds)
wCO2_dz_arr_no_clouds_ds = xr.DataArray(wCO2_dz_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_dz_arr_no_clouds_30min_ds = wCO2_dz_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_dz_arr_no_clouds_30min_ds = wCO2_dz_arr_no_clouds_30min_ds.assign_coords(time=new_time)
wCO2_dz_arr_no_clouds_30min = wCO2_dz_arr_no_clouds_30min_ds.values

#CO2/dt
CO2_dt_mean_no_clouds_ds = xr.DataArray(CO2_dt_mean_no_clouds,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
CO2_dt_mean_no_clouds_30min_ds = CO2_dt_mean_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_dt_mean_no_clouds_30min_ds = CO2_dt_mean_no_clouds_30min_ds.assign_coords(time=new_time)
CO2_dt_mean_no_clouds_30min = CO2_dt_mean_no_clouds_30min_ds.values

#wCO2
wCO2_arr_no_clouds = np.array(wCO2_arr_no_clouds)
wCO2_arr_no_clouds_ds = xr.DataArray(wCO2_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wCO2_arr_no_clouds_30min_ds = wCO2_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wCO2_arr_no_clouds_30min_ds = wCO2_arr_no_clouds_30min_ds.assign_coords(time=new_time)
wCO2_arr_no_clouds_30min = wCO2_arr_no_clouds_30min_ds.values

#qCO2
qCO2_arr_no_clouds = np.array(qCO2_arr_no_clouds)
qCO2_arr_no_clouds_ds = xr.DataArray(qCO2_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
qCO2_arr_no_clouds_30min_ds = qCO2_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
qCO2_arr_no_clouds_30min_ds = qCO2_arr_no_clouds_30min_ds.assign_coords(time=new_time)
qCO2_arr_no_clouds_30min = qCO2_arr_no_clouds_30min_ds.values

#corr
corr_arr_no_clouds = np.array(corr_arr_no_clouds)
corr_arr_no_clouds_ds = xr.DataArray(corr_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
corr_arr_no_clouds_30min_ds = corr_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
corr_arr_no_clouds_30min_ds = corr_arr_no_clouds_30min_ds.assign_coords(time=new_time)
corr_arr_no_clouds_30min = corr_arr_no_clouds_30min_ds.values

#wq
wq_arr_no_clouds = np.array(wq_arr_no_clouds)
wq_arr_no_clouds_ds = xr.DataArray(wq_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:-1]],dims=['time','zt'])
wq_arr_no_clouds_30min_ds = wq_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
wq_arr_no_clouds_30min_ds = wq_arr_no_clouds_30min_ds.assign_coords(time=new_time)
wq_arr_no_clouds_30min = wq_arr_no_clouds_30min_ds.values

#Grad CO2
grad_CO2_arr_no_clouds = np.array(grad_CO2_arr_no_clouds)
grad_CO2_arr_no_clouds_ds = xr.DataArray(grad_CO2_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
grad_CO2_arr_no_clouds_30min_ds = grad_CO2_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
grad_CO2_arr_no_clouds_30min_ds = grad_CO2_arr_no_clouds_30min_ds.assign_coords(time=new_time)
grad_CO2_arr_no_clouds_30min = grad_CO2_arr_no_clouds_30min_ds.values

#Theta_v
profile_theta_v_no_clouds = np.array(profile_theta_v_no_clouds)
profile_theta_v_no_clouds_ds = xr.DataArray(profile_theta_v_no_clouds,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
profile_theta_v_no_clouds_30min_ds = profile_theta_v_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
profile_theta_v_no_clouds_30min_ds = profile_theta_v_no_clouds_30min_ds.assign_coords(time=new_time)
profile_theta_v_no_clouds_30min = profile_theta_v_no_clouds_30min_ds.values

#Buoyancy
buoyancy_arr_no_clouds = np.array(buoy_arr_no_clouds)
buoyancy_arr_no_clouds_ds = xr.DataArray(buoyancy_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
buoyancy_arr_no_clouds_30min_ds = buoyancy_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
buoyancy_arr_no_clouds_30min_ds = buoyancy_arr_no_clouds_30min_ds.assign_coords(time=new_time)
buoyancy_arr_no_clouds_30min = buoyancy_arr_no_clouds_30min_ds.values

#CO2
CO2_arr_no_clouds = np.array(CO2_arr_no_clouds)
CO2_arr_no_clouds_ds = xr.DataArray(CO2_arr_no_clouds,coords=[field_3D.time,field_3D.zt[:]],dims=['time','zt'])
CO2_arr_no_clouds_30min_ds = CO2_arr_no_clouds_ds.coarsen(time=3, boundary='trim').mean()
new_time = ref_time_3D[::3]
CO2_arr_no_clouds_30min_ds = CO2_arr_no_clouds_30min_ds.assign_coords(time=new_time)
CO2_arr_no_clouds_30min = CO2_arr_no_clouds_30min_ds.values

#--------------------------------------------------------------------------------

#Reference time
ref_time_3D_c = pd.to_datetime(qCO2_arr_no_clouds_30min_ds.time.values, unit='s', origin='unix')
ref_time_3D_c = (pd.to_datetime(ref_time_3D_c) + pd.Timedelta(hours=6)).time
x_DALES_3D_s = [t.strftime('%H:%M:%S') for t in ref_time_3D_c]
x_DALES_3D =  [datetime.strptime(time_str, '%H:%M:%S').hour + datetime.strptime(time_str, '%H:%M:%S').minute / 60 for time_str in x_DALES_3D_s]

#--------------------------------------------------------------------------------
#%%

#MXL

#Clouds

#CO2/dt at ABL
CO2_dt_abl = []
for i in range(len(idx_1D_ds)):
    try:
        val = np.nanmean(CO2_dt_mean_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
    except:
        val = float('nan')
    
    CO2_dt_abl.append(val)


#WCO2 at ABL
wCO2_abl = []
wCO2_abl_complete = []
wCO2_abl_with_no_clouds = []
for i in range(len(idx_1D_ds)):
    try:
        val = (wCO2_arr_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
    except:
        val = float('nan')
        val_2 = float('nan')
        val_3 = float('nan')
    
    wCO2_abl.append(val)
    wCO2_abl_complete.append(val_2)
    wCO2_abl_with_no_clouds.append(val_3)

#inclouds

#CO2/dt at ABL

#Condition 1
CO2_dt_abl_inclouds_condition_1 = []
for i in range(len(idx_1D_ds)):
    
    if np.isnan(idx_1D_ds[i]):
        val = np.nan
    else:
        val = np.nanmean(CO2_dt_mean_inclouds_condition_1_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
        val = float(val)
        
    CO2_dt_abl_inclouds_condition_1.append(val)


#Condition 2
CO2_dt_abl_inclouds_condition_2 = []
for i in range(len(idx_1D_ds)):
    
    if np.isnan(idx_1D_ds[i]):
        val = np.nan
    else:
        val = np.nanmean(CO2_dt_mean_inclouds_condition_2_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
        val = float(val)
        
    CO2_dt_abl_inclouds_condition_2.append(val)


#Condition 3
CO2_dt_abl_inclouds_condition_3 = []
for i in range(len(idx_1D_ds)):
    
    if np.isnan(idx_1D_ds[i]):
        val = np.nan
    else:
        val = np.nanmean(CO2_dt_mean_inclouds_condition_3_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
        val = float(val)
        
    CO2_dt_abl_inclouds_condition_3.append(val)


#WCO2 at ABL

#Condition 1
wCO2_abl_inclouds_condition_1 = []
wCO2_abl_inclouds_condition_1_complete = []
wCO2_abl_inclouds_condition_1_with_no_clouds = []

for i in range(len(idx_1D_ds)):
    
    if np.isnan(idx_1D_ds[i]):
        val = np.nan
        val_2 = np.nan
        val_3 = np.nan
    else:
        val = (wCO2_arr_inclouds_condition_1_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_inclouds_condition_1_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_inclouds_condition_1_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
        
    
    wCO2_abl_inclouds_condition_1.append(val)
    wCO2_abl_inclouds_condition_1_complete.append(val_2)
    wCO2_abl_inclouds_condition_1_with_no_clouds.append(val_3)

#Condition 2
wCO2_abl_inclouds_condition_2 = []
wCO2_abl_inclouds_condition_2_complete = []
wCO2_abl_inclouds_condition_2_with_no_clouds = []

for i in range(len(idx_1D_ds)):
    
    if np.isnan(idx_1D_ds[i]):
        val = np.nan
        val_2 = np.nan
        val_3 = np.nan
    
    else:
        val = (wCO2_arr_inclouds_condition_2_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_inclouds_condition_2_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_inclouds_condition_2_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
    
    wCO2_abl_inclouds_condition_2.append(val)
    wCO2_abl_inclouds_condition_2_complete.append(val_2)
    wCO2_abl_inclouds_condition_2_with_no_clouds.append(val_3)


#Condition 3
wCO2_abl_inclouds_condition_3 = []
wCO2_abl_inclouds_condition_3_complete = []
wCO2_abl_inclouds_condition_3_with_no_clouds = []


for i in range(len(idx_1D_ds)):

    if np.isnan(idx_1D_ds[i]):
        val = np.nan
        val_2 = np.nan
        val_3 = np.nan
    
    else:
        val = (wCO2_arr_inclouds_condition_3_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_inclouds_condition_3_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_inclouds_condition_3_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
       
    wCO2_abl_inclouds_condition_3.append(val)
    wCO2_abl_inclouds_condition_3_complete.append(val_2)
    wCO2_abl_inclouds_condition_3_with_no_clouds.append(val_3)

#--------------------------------------------------------------------------------

#environment

#CO2/dt at ABL

#Condition 1
CO2_dt_abl_environment_condition_1 = []
for i in range(len(idx_1D_ds)):
    try:
        val = np.nanmean(CO2_dt_mean_environment_condition_1_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
    except:
        val = float('nan')
    
    CO2_dt_abl_environment_condition_1.append(val)

#Condition 2
CO2_dt_abl_environment_condition_2 = []
for i in range(len(idx_1D_ds)):
    try:
        val = np.nanmean(CO2_dt_mean_environment_condition_2_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
    except:
        val = float('nan')
    
    CO2_dt_abl_environment_condition_2.append(val)

#Condition 3
CO2_dt_abl_environment_condition_3 = []
for i in range(len(idx_1D_ds)):
    try:
        val = np.nanmean(CO2_dt_mean_environment_condition_3_30min_ds.values[i,:int(idx_1D_ds[i])+1])*3600
    except:
        val = float('nan')
    
    CO2_dt_abl_environment_condition_3.append(val)

#WCO2 at ABL

#Condition 1
wCO2_abl_environment_condition_1 = []
wCO2_abl_environment_condition_1_complete = []
wCO2_abl_environment_condition_1_with_no_clouds = []
for i in range(len(idx_1D_ds)):
    try:
        val = (wCO2_arr_environment_condition_1_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_environment_condition_1_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_environment_condition_1_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
    except:
        val = float('nan')
        val_2 = float('nan')
        val_3 = float('nan')
    
    wCO2_abl_environment_condition_1.append(val)
    wCO2_abl_environment_condition_1_complete.append(val_2)
    wCO2_abl_environment_condition_1_with_no_clouds.append(val_3)

#Condition 2
wCO2_abl_environment_condition_2 = []
wCO2_abl_environment_condition_2_complete = []
wCO2_abl_environment_condition_2_with_no_clouds = []
for i in range(len(idx_1D_ds)):
    try:
        val = (wCO2_arr_environment_condition_2_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_environment_condition_2_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_environment_condition_2_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
    except:
        val = float('nan')
        val_2 = float('nan')
        val_3 = float('nan')
    
    wCO2_abl_environment_condition_2.append(val)
    wCO2_abl_environment_condition_2_complete.append(val_2)
    wCO2_abl_environment_condition_2_with_no_clouds.append(val_3)

#Condition 3
wCO2_abl_environment_condition_3 = []
wCO2_abl_environment_condition_3_complete = []
wCO2_abl_environment_condition_3_with_no_clouds = []
for i in range(len(idx_1D_ds)):
    try:
        val = (wCO2_arr_environment_condition_3_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
        val_2 = wCO2_arr_environment_condition_3_30min_ds.values[i,int(idx_1D_ds[i])]*-1
        val_3 = (wCO2_arr_environment_condition_3_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
    except:
        val = float('nan')
        val_2 = float('nan')
        val_3 = float('nan')
    
    wCO2_abl_environment_condition_3.append(val)
    wCO2_abl_environment_condition_3_complete.append(val_2)
    wCO2_abl_environment_condition_3_with_no_clouds.append(val_3)

#--------------------------------------------------------------------------------

#No clouds

#CO2/dt at ABL
CO2_dt_abl_no_clouds = []
for i in range(len(idx_1D_ds_no_clouds)):
    try:
        val = np.nanmean(CO2_dt_mean_no_clouds_30min_ds.values[i,:int(idx_1D_ds_no_clouds[i])+1])*3600
    except:
        val = float('nan')
    
    CO2_dt_abl_no_clouds.append(val)

#WCO2 at ABL
wCO2_abl_no_clouds = []
wCO2_abl_no_clouds_complete = []
wCO2_abl_no_clouds_with_clouds = []

for i in range(len(idx_1D_ds_no_clouds)):
    try:
        val = (wCO2_arr_no_clouds_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-3600)/abl_1D_ds_no_clouds.values[i]
        val_2 = wCO2_arr_no_clouds_30min_ds.values[i,int(idx_1D_ds_no_clouds[i])]*-1
        val_3 = (wCO2_arr_no_clouds_30min_ds.values[i,int(idx_1D_ds[i])]*-3600)/abl_1D_ds.values[i]
    except:
        val = float('nan')
        val_2 = float('nan')
        val_3 = float('nan')
    
    wCO2_abl_no_clouds.append(val)
    wCO2_abl_no_clouds_complete.append(val_2)
    wCO2_abl_no_clouds_with_clouds.append(val_3)

#------------------------------------------------------------------------------------------------------------
#                                           V I S U A L I S A T I O N
#------------------------------------------------------------------------------------------------------------
#%%
#=================
# Figure 4
#=================

fig,ax = plt.subplots(1,2,dpi=72, figsize=(8.27, 11.69)) #width, height

ax1, ax2 = ax.flatten()

ax1_twin = ax1.twinx()

#Set position
ax1.set_position([0.1,0.8,0.9,0.2]) #left, bottom, width, height
ax1_twin.set_position([0.1,0.8,0.9,0.2]) #left, bottom, width, height
ax2.set_position([0.1,0.35,0.9,0.4]) #left, bottom, width, height

#=================
# Figure 4a
#=================

#CLOUDS
ax1.plot(x_DALES_3D,CO2_dt_abl,linewidth=axiswidth*2,color='darkred',label=r'$\frac{\partial \langle \overline{CO_2} \rangle}{\partial t}$')
ax1.plot(x_DALES_3D,(np.array(wCO2_s)/abl_1D_ds.values)*3600,linewidth=axiswidth*2,color='darkgreen',label=r'$\frac{\overline{\left(w^{\prime}CO_2^{\prime}\right)}_{s}}{z_i}$')
ax1.plot(x_DALES_3D,wCO2_abl,linewidth=axiswidth*2,color='tab:blue',label=r'$- \frac{\overline{\left(w^{\prime}CO_2^{\prime}\right)}_{e}}{z_i}$')

#---------------------------------------------------------------------------------------------------------------

#NO CLOUDS
ax1.plot(x_DALES_3D,CO2_dt_abl_no_clouds,linewidth=axiswidth*2,color='darkred',linestyle='--')
ax1.plot(x_DALES_3D,(np.array(wCO2_s_no_clouds)/abl_1D_ds_no_clouds.values)*3600,linewidth=axiswidth*2,color='darkgreen',linestyle='--')
ax1.plot(x_DALES_3D,wCO2_abl_no_clouds,linewidth=axiswidth*2,color='tab:blue',linestyle='--')

#---------------------------------------------------------------------------------------------------------------

#Shading etc. 

#Interpolate
x_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,x_DALES_1D)
wCO2_abl_inclouds_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl_inclouds_condition_2))
wCO2_abl_environment_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl_environment_condition_2))
wCO2_abl_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl))
wCO2_abl_no_clouds_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl_no_clouds))

#Define regime where simulation initiates till active buoyant clouds start to form
index_0 = find_closest(x_interpolated,8)
index_1_shading = np.where(wCO2_abl_inclouds_interpolated>=wCO2_abl_environment_interpolated)[0][0]
index_1 = np.where(wCO2_abl_inclouds_interpolated>=0.05)[0][0]

ax1.fill_betweenx(np.arange(-6.5,1,1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax1.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 0.23,r'$\textbf{I}$', ha='center', va='bottom', color='k')

#Where CLOUDS > NO CLOUDS
selection = np.where(np.array(wCO2_abl_interpolated)>=np.array(wCO2_abl_no_clouds_interpolated), wCO2_abl_interpolated, np.nan)
index_9LT = find_closest(x_interpolated,9)
index_14LT = find_closest(x_interpolated,14)

cut_selection = selection[index_9LT:index_14LT]
x_cut = x_interpolated[index_9LT:index_14LT]

idx_1 = np.where(~np.isnan(cut_selection))[0][0]
idx_2 = np.where(~np.isnan(cut_selection))[0][-1]

#Where in-cloud > environment
index_1_1 = np.where(wCO2_abl_inclouds_interpolated>=wCO2_abl_environment_interpolated)[0][0]

#Increase in z_i
diff = abs(np.array(CO2_dt_abl)-np.array(CO2_dt_abl_no_clouds))
index_max = np.where(diff==np.nanmax(diff))[0][0]

#Where CLOUDS approaches 0
index_12LT = find_closest(x_interpolated,12)
index_15LT = find_closest(x_interpolated,15)
cut_data = wCO2_abl_interpolated[index_12LT:index_15LT]
cut_x = x_interpolated[index_12LT:index_15LT]

index_2 = find_closest(cut_data,0)

index_2_total = index_12LT+index_2
index_3 = np.where(np.isnan(CO2_dt_abl))[0][-2]-1

ax1.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),-6.5,0.5,color='k',alpha=0.1)

ax1.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 0.23, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax1.text(((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total], 0.23, r'$\textbf{III}$', ha='center', va='bottom', color='k')

index_16LT = find_closest(x_interpolated,16)
cut_data = wCO2_abl_no_clouds_interpolated[index_12LT:index_16LT]
cut_x = x_interpolated[index_12LT:index_16LT]

index_4 = find_closest(cut_data,0)

#Detailing
ax1.tick_params(axis='x')
ax1.tick_params(axis='y')

ax1.spines['top'].set_visible(False)

ax1.yaxis.set_minor_locator(AutoMinorLocator())
ax1.xaxis.set_minor_locator(AutoMinorLocator())
ax1.tick_params(which='major')
ax1.tick_params(which='minor')

ax1.set_ylim(-6.5,0.1)
ax1.set_ylabel('(ppm h$^{-1}$)')

ax1.set_xticks(np.arange(6,19,2),[])
ax1.set_xlim(7.5,18)

ax1.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)

legend = ax1.legend(loc='lower left', bbox_to_anchor= (0.05, 1.2), ncol=3,
                borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size']*1.4)

ax1.text(-0.14,1, '(a)',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)

#Inset axes with bbox
ax1_sub = ax1.inset_axes([0.685, 0.5, 0.1, 0.02]) #

#Turn off axis
ax1_sub.axis('off')
ax1_sub.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='-',label='CLOUDS')
ax1_sub.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='--',label='NO CLOUDS')

ax1_sub.legend(loc='upper left', ncol=1, borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size']*1.2)

#---------------------------------------------------------------------------------------------------------------

#Contribution - Entrainment
idx_start = find_closest(np.array(x_interpolated),8.5)
index_end = find_closest(np.array(x_interpolated),14.3)

idx_1 = find_closest(np.array(x_DALES_3D),8.5)
idx_2 = find_closest(np.array(x_DALES_3D),14.3)

clouds = np.array(wCO2_abl)/np.array(abl_1D_ds.values)
no_clouds = np.array(wCO2_abl_no_clouds)/np.array(abl_1D_ds_no_clouds.values)
clouds = clouds *3600
no_clouds = no_clouds *3600

clouds[clouds>0] = np.nan
no_clouds[no_clouds>0] = np.nan

relative_entrainment =  ((clouds - no_clouds)/no_clouds)*100

#Contribution - Surface flux
clouds = np.array(wCO2_s)/np.array(abl_1D_ds.values)
no_clouds = np.array(wCO2_s_no_clouds)/np.array(abl_1D_ds_no_clouds.values)

clouds = clouds *3600
no_clouds = no_clouds *3600

clouds[clouds>0] = np.nan
no_clouds[no_clouds>0] = np.nan

relative_surface_flux =  ((clouds - no_clouds)/no_clouds)*100

#Contribution - Total tendency
clouds = np.array(CO2_dt_abl)
no_clouds = np.array(CO2_dt_abl_no_clouds)

clouds[clouds>0] = np.nan
no_clouds[no_clouds>0] = np.nan

relative_total_tendency =  ((clouds - no_clouds)/no_clouds)*100

#Stacked bar chart
ax1_twin.bar(x_DALES_3D[idx_1:idx_2], relative_entrainment[idx_1:idx_2], color='tab:blue',width=0.5,alpha=0.2)
ax1_twin.bar(x_DALES_3D[idx_1:idx_2], relative_total_tendency[idx_1:idx_2], color='darkred',width=0.5,alpha=0.2)
ax1_twin.bar(x_DALES_3D[idx_1:idx_2], relative_surface_flux[idx_1:idx_2], color='darkgreen',width=0.5,alpha=0.2)

ax1_twin.set_ylim(0,50)

# Align zero point
align_yaxis(ax1, -7, ax1_twin, 0)

#Detailing
ax1_twin.tick_params(axis='x')
ax1_twin.tick_params(axis='y')

ax1_twin.spines['top'].set_visible(False)

ax1_twin.yaxis.set_minor_locator(AutoMinorLocator())
ax1_twin.xaxis.set_minor_locator(AutoMinorLocator())

ax1_twin.tick_params(which='major')
ax1_twin.tick_params(which='minor')

ax1_twin.set_ylabel('Rel. difference (\%)')
ax1_twin.text(1.12,0.15, 'CLOUDS -- NO CLOUDS',transform=ax1_twin.transAxes,fontsize=plt.rcParams['font.size']*0.6,rotation=90)

ax1_twin.set_yticks(np.arange(0,75,25),np.arange(0,75,25))

ax1_twin.set_xticks(np.arange(6,19,2),[])
ax1_twin.set_xlim(7.5,18)

#---------------------------------------------------------------------------------------------------------------

#=================
# Figure 4b
#=================
ax2.plot(x_DALES_3D,wCO2_abl,linewidth=axiswidth*2.5,color='k')

#Inclouds
ax2.plot(x_DALES_3D,wCO2_abl_inclouds_condition_2,linewidth=axiswidth*2,color='k',linestyle='-.')

#Environment
ax2.plot(x_DALES_3D,wCO2_abl_environment_condition_2,linewidth=axiswidth*2,color='k',linestyle=':')

#---------------------------------------------------------------------------------------------------------------

#lines etc.
ax2.axhline(0, color='k',linewidth=axiswidth*0.5,linestyle='--')

#Interpolate
x_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,x_DALES_1D)
wCO2_abl_inclouds_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl_inclouds_condition_2))
wCO2_abl_environment_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl_environment_condition_2))
wCO2_abl_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl))
wCO2_abl_no_clouds_interpolated = np.abs(np.interp(np.linspace(6,18,1000),x_DALES_1D,wCO2_abl_no_clouds))

#Define regime where simulation initiates till active buoyant clouds start to form
index_0 = find_closest(x_interpolated,8)
index_1_shading = np.where(wCO2_abl_inclouds_interpolated>=wCO2_abl_environment_interpolated)[0][0]
index_1 = find_closest(x_interpolated,8) #np.where(wCO2_abl_inclouds_interpolated>=0.05)[0][0]
time_0 = x_interpolated[index_0]

ax2.fill_betweenx(np.arange(-6.5,1,1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)

ax2.plot(x_interpolated[index_1],-1*wCO2_abl_inclouds_interpolated[index_1],'o',color='k',markersize=7)
ax2.text(x_interpolated[index_1],-1*wCO2_abl_inclouds_interpolated[index_1]+0.1,r'\textbf{A}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)
time_A = x_interpolated[index_1]

#Where in-cloud > environment
index_1_1 = np.where(wCO2_abl_inclouds_interpolated>=wCO2_abl_environment_interpolated)[0][0]
ax2.plot(x_interpolated[index_1_1],-1*wCO2_abl_environment_interpolated[index_1_1],'o',color='k',markersize=7)
ax2.text(x_interpolated[index_1_1]-0.5,-1*wCO2_abl_environment_interpolated[index_1_1]-0.1,r'\textbf{B}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)
time_B = x_interpolated[index_1_1]

#Where CLOUDS == 0
index_12LT = find_closest(x_interpolated,12)
index_15LT = find_closest(x_interpolated,15)
cut_data = wCO2_abl_interpolated[index_12LT:index_15LT]
cut_x = x_interpolated[index_12LT:index_15LT]

index_2 = find_closest(cut_data,0)
ax2.plot(cut_x[index_2],-1*cut_data[index_2],'o',color='k',markersize=7)
ax2.text(cut_x[index_2],-1*cut_data[index_2]+0.1,r'\textbf{C}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)
time_C = cut_x[index_2]

#Increase in z_i
diff = abs(np.array(CO2_dt_abl)-np.array(CO2_dt_abl_no_clouds))
index_max = np.where(diff==np.nanmax(diff))[0][0]
time_D = x_DALES_3D[index_max]

#Where NO CLOUDS approaches 0
index_2_total = index_12LT+index_2
index_3 = np.where(np.isnan(CO2_dt_abl))[0][-2]-1

ax2.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),-6.5,0.5,color='k',alpha=0.1)

index_16LT = find_closest(x_interpolated,16)
cut_data = wCO2_abl_no_clouds_interpolated[index_12LT:index_16LT]
cut_x = x_interpolated[index_12LT:index_16LT]

index_4 = find_closest(cut_data,0)

time_E = cut_x[index_4]

#---------------------------------------------------------------------------------------------------------------

#Detailing
ax2.set_ylim(-3.05,0.1)
ax2.set_xlim(7.8,18)
ax2.set_xticks(np.arange(8,19,2),np.arange(8,19,2))

ax2.tick_params(axis='x')
ax2.tick_params(axis='y')
ax2.spines['right'].set_visible(False)
ax2.spines['top'].set_visible(False)

ax2.yaxis.set_minor_locator(AutoMinorLocator())
ax2.xaxis.set_minor_locator(AutoMinorLocator())
ax2.tick_params(which='major')
ax2.tick_params(which='minor')

ax2.set_xticks(np.arange(8,19,2),np.arange(8,19,2))
ax2.set_xlabel('LT (hour)')

ax2.text(-0.18,0.3, r'$- \frac{\overline{\left(w^{\prime}CO_2^{\prime}\right)}_{e}}{z_i}$',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.4,rotation=90)
ax2.text(-0.16,0.55, '(ppm h$^{-1}$)',transform=ax2.transAxes,rotation=90)

ax2.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)

ax2.text(-0.14,1, '(b)',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)

#=================
#   L E G E N  D
#=================

ax2.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='-',label='CLOUDS')
ax2.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='-.',label='in-cloud ($M$)')
ax2.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle=':',label='environment ($w_e$)')

ax2.legend(loc='upper left', bbox_to_anchor=(0.555,0.5), ncol=1, borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size']*1.2)

#=================
#   E X P O R T
#=================

plt.savefig(f'{outdir}/Figure_4.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_4.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_4.pdf',bbox_inches='tight')
plt.show()
plt.close()


#Print all times

timestep = time_A
hours = int(timestep)
minutes = math.floor((timestep - hours) * 60)
print(f'A: {hours}:{minutes}')

timestep = time_B
hours = int(timestep)
minutes = math.floor((timestep - hours) * 60)
print(f'B: {hours}:{minutes}')

timestep = time_C
hours = int(timestep)
minutes = math.floor((timestep - hours) * 60)
print(f'C: {hours}:{minutes}')

timestep = time_D
hours = int(timestep)
minutes = math.floor((timestep - hours) * 60)
print(f'D: {hours}:{minutes}')

timestep = time_E
hours = int(timestep)
minutes = math.floor((timestep - hours) * 60)
print(f'E: {hours}:{minutes}')

#%%
#------------------------------------------------------------------------------------------------------------
#=================
# Figure 5
#=================

fig, ax =  plt.subplots(2,2, dpi=72, figsize=(8.27*1.5, 11.69))
ax1, ax2, ax3, ax4 = ax.flatten()

#=================
# Figure 5a
#=================

#ABL
ax1.plot(x_DALES_1D,abl_1D_ds.values,color='k',label='CLOUDS',linewidth=axiswidth*2)
ax1.plot(x_DALES_1D,abl_1D_ds_no_clouds.values,color='k',linestyle='--',label='NO CLOUDS',linewidth=axiswidth*2)

#Shading
ax1.fill_betweenx(np.arange(0,3000,1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax1.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),0,3000,color='k',alpha=0.1)

#Marker
x_interpolated_n = np.interp(np.linspace(6,18,1000),x_DALES_1D,x_DALES_1D)
abl_1D_ds_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,abl_1D_ds.values)
abl_1D_ds_no_clouds_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,abl_1D_ds_no_clouds.values)

index_B = find_closest(x_interpolated_n,time_B)
ax1.plot(x_interpolated_n[index_B],abl_1D_ds_interpolated[index_B],'o',color='k',markersize=7)
ax1.text(x_interpolated_n[index_B],abl_1D_ds_interpolated[index_B]+100,r'\textbf{B}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)

#Labels of regimes
ax1.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 2550,r'$\textbf{I}$', ha='center', va='bottom', color='k')
ax1.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 2550, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax1.text(((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total], 2550, r'$\textbf{III}$', ha='center', va='bottom', color='k')

#Detailing
ax1.tick_params(axis='x')
ax1.tick_params(axis='y')
ax1.spines['right'].set_visible(False)
ax1.spines['top'].set_visible(False)

ax1.set_ylabel('z$_i$ (m)',fontsize=plt.rcParams['font.size']*1.2)
ax1.set_yticks(np.arange(0,4000,500),np.arange(0,4000,500))
ax1.set_ylim(0,2500)

ax1.set_xlim(7.5,18)
ax1.set_xticks(np.arange(8,19,2),np.arange(8,19,2))
ax1.set_xlabel('LT (hour)')

ax1.yaxis.set_minor_locator(AutoMinorLocator())
ax1.xaxis.set_minor_locator(AutoMinorLocator())
ax1.tick_params(which='major')
ax1.tick_params(which='minor')

ax1.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)

ax1.text(-0.1,1.1, '(a)',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)

#----------------------------------------------------------------------------------------------------------------

#=================
# Figure 5b
#=================

#Growth
growth = np.gradient(abl_1D_ds.values)/np.nanmean(np.gradient(x_DALES_1D))
growth_no_clouds = np.gradient(abl_1D_ds_no_clouds.values)/np.nanmean(np.gradient(x_DALES_1D))

ax2.plot(x_DALES_1D,growth,color='k',linewidth=axiswidth*2,label='CLOUDS')
ax2.plot(x_DALES_1D,growth_no_clouds,color='k',linestyle='--',linewidth=axiswidth*2,label='NO CLOUDS')

#Shading
ax2.fill_betweenx(np.arange(-300,400,1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax2.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),-300,400,color='k',alpha=0.1)

#Marker
x_interpolated_n = np.interp(np.linspace(6,18,1000),x_DALES_1D,x_DALES_1D)
growth_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,growth)
growth_no_clouds_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,growth_no_clouds)

index_D = find_closest(x_interpolated_n,time_D)
ax2.plot(x_interpolated_n[index_D],growth_no_clouds_interpolated[index_D],'o',color='k',markersize=7)
ax2.text(x_interpolated_n[index_D],growth_no_clouds_interpolated[index_D]-80,r'\textbf{D}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)

#Labels of regimes
ax2.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 355,r'$\textbf{I}$', ha='center', va='bottom', color='k')
ax2.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 355, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax2.text(((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total], 355, r'$\textbf{III}$', ha='center', va='bottom', color='k')

#Detailing
ax2.tick_params(axis='x')
ax2.tick_params(axis='y')

ax2.spines['right'].set_visible(False)
ax2.spines['top'].set_visible(False)

ax2.set_ylim(0,350)
ax2.set_yticks(np.arange(0,400,50),np.arange(0,400,50))

ax2.text(-0.28,0.38,r'$\frac{dz_i}{dt}$',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.4,rotation=90)
ax2.text(-0.25,0.50,'(m h$^{-1}$)',transform=ax2.transAxes,rotation=90,fontsize=plt.rcParams['font.size'])

ax2.text(-0.1,1.1, '(b)',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)

ax2.set_xlim(7.5,18)
ax2.set_xticks(np.arange(8,19,2),np.arange(8,19,2))
ax2.set_xlabel('LT (hour)')

ax2.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax2.axhline(0, color='k',linewidth=axiswidth*0.5,linestyle='--')

#----------------------------------------------------------------------------------------------------------------
#=================
# Figure 5c
#=================

ax3_twin = ax3.twinx()

ax3.plot(x_DALES_1D,ratio_atm_without_1,color='k',linewidth=axiswidth*2)
ax3.plot(x_DALES_1D,ratio_atm_no_clouds_without_1,color='k',linestyle='--',linewidth=axiswidth*2)

ax3_twin.plot(x_DALES_1D,ratio_plant_without_1,color='darkgreen')
ax3_twin.plot(x_DALES_1D,ratio_plant_no_clouds_without_1,color='darkgreen',linestyle='--')

#Shading
ax3.fill_betweenx(np.arange(-2,2,0.2),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax3.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),-2,2,color='k',alpha=0.1)

#Marker
x_interpolated_n = np.interp(np.linspace(6,18,1000),x_DALES_1D,x_DALES_1D)
ratio_atm_without_1_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,ratio_atm_without_1)
ratio_atm_no_clouds_without_1_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,ratio_atm_no_clouds_without_1)
ratio_plant_without_1_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,ratio_plant_without_1)
ratio_plant_no_clouds_without_1_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,ratio_plant_no_clouds_without_1)

index_B = find_closest(x_interpolated_n,time_B)
ax3.plot(x_interpolated_n[index_B],ratio_atm_without_1_interpolated[index_B],'o',color='k',markersize=7)
ax3.text(x_interpolated_n[index_B],ratio_atm_without_1_interpolated[index_B]+0.005,r'\textbf{B}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)
ax3_twin.plot(x_interpolated_n[index_B],ratio_plant_without_1_interpolated[index_B],'o',color='k',markersize=7)

#Labels of regimes
ax3.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 1.011,r'$\textbf{I}$', ha='center', va='bottom', color='k')
ax3.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 1.011, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax3.text(((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total], 1.011, r'$\textbf{III}$', ha='center', va='bottom', color='k')

#Detailing
ax3.tick_params(axis='x')
ax3.tick_params(axis='y')
ax3.tick_params(which='major')
ax3.tick_params(which='minor')

ax3.spines['top'].set_visible(False)
ax3.spines['right'].set_visible(False)

ax3.set_ylim(0.92,1.01)
ax3.set_yticks(np.arange(0.92,1.02,0.02),np.round(np.arange(0.92,1.02,0.02),2))

ax3.set_xlim(7.5,18)
ax3.set_xticks(np.arange(8,19,2),np.arange(8,19,2))
ax3.set_xlabel('LT (hour)')

ax3.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)

ax3.text(-0.1,1.1,'(c)',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.2)


ax3_twin.tick_params(axis='x')
ax3_twin.tick_params(axis='y')
ax3_twin.tick_params(which='major')
ax3_twin.tick_params(which='minor')

ax3_twin.spines['top'].set_visible(False)

ax3_twin.set_ylim(0.74,0.9)
ax3_twin.set_yticks(np.arange(0.74,0.92,0.04),np.round(np.arange(0.74,0.92,0.04),2),color='darkgreen')

ax3_twin.yaxis.set_label_position("right")
ax3_twin.grid(linestyle=':',color='darkgreen',alpha=0.3,linewidth=axiswidth*2)


ax3.text(-0.3,0.3, r'$\frac{C_ft}{\langle CO_2 \rangle}$',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.4,rotation=90)
ax3.text(-0.25,0.47, 'or',transform=ax3.transAxes,fontsize=plt.rcParams['font.size'],rotation=90)
ax3.text(-0.3,0.6, r'$\frac{C_i}{\langle CO_2 \rangle}$',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.4,rotation=90,color='darkgreen')
ax3.text(-0.25,0.78, '(-)',transform=ax3.transAxes,rotation=90,fontsize=plt.rcParams['font.size'])

#Legend
ax3_subber = ax3.inset_axes([-0.24,0,0.35,1]) #
ax3_subber.axis('off')

ax3_subber.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='-',label='CLOUDS')
ax3_subber.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='-.',label='in-cloud ($M$)')
ax3_subber.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle=':',label='environment ($w_e$)')
ax3_subber.plot(x_DALES_3D,[np.nan]*len(x_DALES_3D),linewidth=axiswidth*2,color='k',linestyle='--',label='NO CLOUDS')

ax3_subber.legend(loc='upper left', bbox_to_anchor=(0.76,2.7), ncol=4, borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size'])

#----------------------------------------------------------------------------------------------------------------
#=================
# Figure 5d
#=================

idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_inclouds_condition_2_30min[i,int(idx[i])])

values_at_ABL = np.array(values_at_ABL)/delta_c_atm
values_at_ABL_incloud = values_at_ABL

idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_environment_condition_2_30min[i,int(idx[i])])

values_at_ABL = np.array(values_at_ABL)/delta_c_atm
values_at_ABL_environment = values_at_ABL

idx = idx_1D_ds_no_clouds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_no_clouds_30min[i,int(idx[i])])

values_at_ABL = np.array(values_at_ABL)/delta_c_atm_no_clouds
values_at_ABL_no_clouds = values_at_ABL

ax4.plot(x_DALES_1D,values_at_ABL_incloud/gCO2_s_ds.values,color='k',linestyle='-.')
ax4.plot(x_DALES_1D,values_at_ABL_environment/gCO2_s_ds.values,color='k',linestyle=':')
ax4.plot(x_DALES_1D,values_at_ABL_no_clouds/gCO2_s_ds_no_clouds.values,color='k',linestyle='--')

#Shading
ax4.fill_betweenx(np.arange(0,20,1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax4.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),0,20,color='k',alpha=0.1)

#Marker
x_interpolated_n = np.interp(np.linspace(6,18,1000),x_DALES_1D,x_DALES_1D)
values_at_ABL_incloud_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,values_at_ABL_incloud/gCO2_s_ds.values)
values_at_ABL_environment_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,values_at_ABL_environment/gCO2_s_ds.values)
values_at_ABL_no_clouds_interpolated = np.interp(np.linspace(6,18,1000),x_DALES_1D,values_at_ABL_no_clouds/gCO2_s_ds_no_clouds.values)

#Marker B
index_B = find_closest(x_interpolated_n,time_B)
ax4.plot(x_interpolated_n[index_B],values_at_ABL_incloud_interpolated[index_B],'o',color='k',markersize=7)
ax4.text(x_interpolated_n[index_B]+0.8,values_at_ABL_incloud_interpolated[index_B],r'\textbf{B}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)

#Marker D
index_D = find_closest(x_interpolated_n,time_D)
ax4.plot(x_interpolated_n[index_D],values_at_ABL_no_clouds_interpolated[index_D],'o',color='k',markersize=7)
ax4.text(x_interpolated_n[index_D]-0.4,values_at_ABL_no_clouds_interpolated[index_D]+0.5,r'\textbf{D}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)

#Marker C
index_C = find_closest(x_interpolated_n,time_C)
ax4.plot(x_interpolated_n[index_C],0,'o',color='k',markersize=7)
ax4.text(x_interpolated_n[index_C]+0.3,values_at_ABL_incloud_interpolated[index_C]+1.5,r'\textbf{C}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)

#Marker E
index_E = find_closest(x_interpolated_n,time_E)
ax4.plot(x_interpolated_n[index_E],0,'o',color='k',markersize=7)
ax4.text(x_interpolated_n[index_E]+0.8,values_at_ABL_no_clouds_interpolated[index_E]+0.5,r'\textbf{E}',ha='center', va='bottom', color='k',fontsize=plt.rcParams['font.size']*1.2)

#Labels of regimes
ax4.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 10.1,r'$\textbf{I}$', ha='center', va='bottom', color='k')
ax4.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 10.1, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax4.text(((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total], 10.1, r'$\textbf{III}$', ha='center', va='bottom', color='k')


#Detailing
ax4.tick_params(axis='x')
ax4.tick_params(axis='y')
ax4.tick_params(which='major')
ax4.tick_params(which='minor')

ax4.spines['top'].set_visible(False)
ax4.spines['right'].set_visible(False)

ax4.set_ylim(0,10)
ax4.set_xlim(7.5,18)
ax4.set_xticks(np.arange(8,19,2),np.arange(8,19,2))
ax4.set_xlabel('LT (hour)')

ax4.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax4.text(-0.1,1.1
         ,'(d)',transform=ax4.transAxes,fontsize=plt.rcParams['font.size']*1.2)

ax4.text(-0.28+0.06,0.37,r'$\frac{M}{g_{c,c}}$',transform=ax4.transAxes,fontsize=plt.rcParams['font.size']*1.4,rotation=90)
ax4.text(-0.25+0.06,0.47,'or',transform=ax4.transAxes,fontsize=plt.rcParams['font.size'],rotation=90)
ax4.text(-0.28+0.06,0.6,r'$\frac{w_e}{g_{c,c}}$',transform=ax4.transAxes,fontsize=plt.rcParams['font.size']*1.4,rotation=90)
ax4.text(-0.25+0.06,0.7,'(-)',transform=ax4.transAxes,fontsize=plt.rcParams['font.size'],rotation=90)

plt.subplots_adjust(hspace=0.4, wspace=0.45)



#export
plt.savefig(f'{outdir}/Figure_5.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_5.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_5.pdf',bbox_inches='tight')
plt.show()
plt.close()


#%%

#====================================
# Figure 5 addition (contributions)
#====================================

fig, ax = plt.subplots(1,1,dpi=72, figsize=(8.27, 11.69/2))

cloud_sim = []
idx = idx_1D_ds.values
for i in range(len(idx)):
    if np.isnan(idx[i]):
        cloud_sim.append(float('nan'))
    else:
        cloud_sim.append(wCO2_arr_30min_ds.values[i,int(idx[i])])

cloud_sim = np.nan_to_num(cloud_sim,nan=0)

plt.plot(x_DALES_1D,cloud_sim,linewidth=axiswidth*2,color='k',linestyle='-',label='CLOUDS')


idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_environment_condition_2_30min[i,int(idx[i])])

ax.plot(x_DALES_1D,values_at_ABL,linewidth=axiswidth*2,color='k',linestyle=':',label='environment ($w_e$)')

env = np.nan_to_num(values_at_ABL,nan=0)

idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_inclouds_condition_2_30min[i,int(idx[i])])

ax.plot(x_DALES_1D,values_at_ABL,linewidth=axiswidth*2,color='k',linestyle='-.',label='in-cloud ($M$)')

cloud = np.nan_to_num(values_at_ABL,nan=0)

idx = idx_1D_ds_no_clouds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_no_clouds_30min[i,int(idx[i])])

ax.plot(x_DALES_1D,values_at_ABL,linewidth=axiswidth*2,color='k',linestyle='--',label='NO CLOUDS')

no_cloud = np.nan_to_num(values_at_ABL,nan=0)




#Percentages

x = np.arange(0,len(x_DALES_3D),1)

idx_s = find_closest(np.array(x_DALES_3D),time_A)
idx_e = find_closest(np.array(x_DALES_3D),18)

area_cloud_sim = np.trapz(cloud_sim[idx_s:idx_e+1],x[idx_s:idx_e+1])
area_no_cloud = np.trapz(no_cloud[idx_s:idx_e+1],x[idx_s:idx_e+1])
area_cloud = np.trapz(cloud[idx_s:idx_e+1],x[idx_s:idx_e+1])
area_env = np.trapz(env[idx_s:idx_e+1],x[idx_s:idx_e+1])

print('percentage in-cloud',(area_cloud/area_cloud_sim)*100)
print('percentage env',(area_env/area_cloud_sim)*100)

cloudy = (area_cloud/area_cloud_sim)*100
envy = (area_env/area_cloud_sim)*100

CO2_dt_abl_filled = np.nan_to_num(CO2_dt_abl,nan=0)
wCO2_abl_filled = np.nan_to_num(wCO2_abl,nan=0)
wCO2_s_filled = np.nan_to_num((np.array(wCO2_s)/abl_1D_ds.values)*3600,nan=0)

area_co2_dt = np.trapz(CO2_dt_abl_filled[idx_s:idx_e+1],x[idx_s:idx_e+1])
area_wCO2_abl = np.trapz(wCO2_abl_filled[idx_s:idx_e+1],x[idx_s:idx_e+1])
area_wCO2_s = np.trapz(wCO2_s_filled[idx_s:idx_e+1],x[idx_s:idx_e+1])

print('contribution surface flux', (area_wCO2_s/area_co2_dt)*100)
print('contribution abl flux', (area_wCO2_abl/area_co2_dt)*100)

print('contribution cloudy',cloudy/100*(area_wCO2_abl/area_co2_dt)*100)
print('contribution env',envy/100*(area_wCO2_abl/area_co2_dt)*100)


#detailing
ax.set_xlim(6,18)
ax.set_ylim(0,0.6)
ax.set_ylabel('w$_e$ or M (ppm m s$^{-1}$)')
ax.set_xlabel('LT (hour)')

ax.tick_params(axis='x')
ax.tick_params(axis='y')
ax.spines['right'].set_visible(False)
ax.spines['top'].set_visible(False)

ax.yaxis.set_minor_locator(AutoMinorLocator())
ax.xaxis.set_minor_locator(AutoMinorLocator())
ax.tick_params(which='major')
ax.tick_params(which='minor')

ax.legend(loc='upper left', bbox_to_anchor=(0.05,1.2), ncol=2, borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size'])

ax.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)

plt.savefig(f'{outdir}/Figure_5_addition.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_5_addition.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_5_addition.pdf',bbox_inches='tight')
plt.show()
plt.close()



#%%

#-----------------------------------------------------------------------------------------------------------

#=================
# Figure 6
#=================

fig,ax = plt.subplots(2,3,dpi=72, figsize=(8.27*1.5, 11.69))
ax1, ax2, ax3, ax4, ax5, ax6 = ax.flatten()

times = [9,12,15]

total_dict = {}

count = 0
for time in times:
    
    #dictionaries
    sub_dict = {}
    
    #Clouds
    idx = find_closest(np.array(x_DALES_3D),time)
    sel_CO2_dt = CO2_dt_mean_30min_ds.values[idx,:]*3600
    sel_wCO2_dz = wCO2_dz_arr_30min_ds.values[idx,:]*3600
    
    #No clouds
    sel_CO2_dt_no_clouds = CO2_dt_mean_no_clouds_30min_ds.values[idx,:]*3600
    sel_wCO2_dz_no_clouds = wCO2_dz_arr_no_clouds_30min_ds.values[idx,:]*3600
    
    abl_sel = abl_1D_ds.values[idx]
    cb_sel = tmser_cloud_base_ds.values[idx]
    ct_sel = tmser_cloud_top_max_ds.values[idx]
    
    abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]
    
    #Vis
    locals()[f'ax{count+1}'].plot(sel_CO2_dt,z_3D/abl_sel,linewidth=axiswidth*2,color='darkred',label=r'$\frac{\partial \overline{CO_2}}{\partial t}$')
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',label=r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$')
    
    locals()[f'ax{count+1}'].plot(sel_CO2_dt_no_clouds,z_3D/abl_sel,linewidth=axiswidth*2,color='darkred',linestyle='--')
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz_no_clouds,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle='--')
        
    #horizontal lines and cloud layer
    locals()[f'ax{count+1}'].axhline(abl_sel/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='-',label='z$_i$')
    
    locals()[f'ax{count+1}'].axhline(abl_sel_no_clouds/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='--')
    
    locals()[f'ax{count+1}'].fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1)
    
    locals()[f'ax{count+1}'].axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')
    
    #add values to dictionary
    idx = find_closest(np.array(z_3D),abl_sel)
    sub_dict['dCO2_dt'] = str(sel_CO2_dt[idx])
    sub_dict['dwCO2_dz'] = str(sel_wCO2_dz[idx])
    sub_dict['abl'] = str(abl_sel)
    sub_dict['cb'] = str(cb_sel)
    sub_dict['ct'] = str(ct_sel)
    
    idx = find_closest(np.array(z_3D),abl_sel_no_clouds)
    sub_dict['dCO2_dt_no_clouds'] = str(sel_CO2_dt_no_clouds[idx])
    sub_dict['dwCO2_dz_no_clouds'] = str(sel_wCO2_dz_no_clouds[idx])
    sub_dict['abl_no_clouds'] = str(abl_sel_no_clouds)
       
    total_dict[str(time)] = sub_dict
        
    #Detailing
    locals()[f'ax{count+1}'].set_ylim(0,3)
    
    locals()[f'ax{count+1}'].tick_params(axis='x')
    locals()[f'ax{count+1}'].tick_params(axis='y')
    locals()[f'ax{count+1}'].spines['right'].set_visible(False)
    locals()[f'ax{count+1}'].spines['top'].set_visible(False)
    
    locals()[f'ax{count+1}'].yaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].xaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].tick_params(which='major')
    locals()[f'ax{count+1}'].tick_params(which='minor')

    locals()[f'ax{count+1}'].set_title(f'{time} LT')
    
    count = count + 1

ax1.set_ylabel('z/z$_i$')
ax2.set_xlabel('(ppm h$^{-1}$)')

ax1.set_xlim(-6,6)
ax1.set_ylim(0,2)
ax1.set_xticks(np.arange(-16,17,8),np.arange(-16,17,8))
ax2.set_xlim(-5,5)
ax2.set_xticks(np.arange(-5,6,2.5),np.arange(-5,6,2.5))
ax3.set_xlim(-1,1)
ax3.set_xticks(np.arange(-1,1.1,0.5),np.arange(-1,1.1,0.5))
 
ax1.legend(loc='lower left', bbox_to_anchor= (1.1, -0.45), ncol=3,
            borderaxespad=0, frameon=False)

plt.subplots_adjust(left=0.1, bottom=0.1, right=0.9, top=0.9, wspace=0.5, hspace=0.4)

ax1.fill_betweenx(np.arange(-30,29),0,0,color='k',alpha=0.1,label='cloud layer')
ax1.plot([-30]*len(z_3D),z_3D,linewidth=axiswidth*2,color='k',linestyle='-',label='CLOUDS')
ax1.plot([-30]*len(z_3D),z_3D,linewidth=axiswidth*2,color='k',linestyle='--',label='NO CLOUDS')

legend_maker(ax1,2,3,0.75,-0.65)

#d - f
for time in times:
    
    sub_dict = {}
    
    #Clouds
    idx = find_closest(np.array(x_DALES_3D),time)
    sel_wCO2_dz = wCO2_dz_arr_30min_ds.values[idx,:]*3600
    
    #No clouds
    sel_wCO2_dz_no_clouds = wCO2_dz_arr_no_clouds_30min_ds.values[idx,:]*3600
    
    #inclouds
    sel_wCO2_dz_inclouds = wCO2_dz_arr_inclouds_condition_2_30min_ds.values[idx,:]*3600
    
    #environment
    sel_wCO2_dz_environment = wCO2_dz_arr_environment_condition_2_30min_ds.values[idx,:]*3600
    
    #horizontal lines and cloud layer
    abl_sel = abl_1D_ds.values[idx]
    cb_sel = tmser_cloud_base_ds.values[idx]
    ct_sel = tmser_cloud_top_max_ds.values[idx]
    
    abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]
    
    #Vis
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz,z_3D/abl_sel,linewidth=axiswidth*2.5,color='tab:blue',label='CLOUDS')
    
    #locals()[f'ax{count+1}'].plot(sel_wCO2_dz_no_clouds,z_3D,linewidth=axiswidth*2,color='tab:blue',linestyle='--',label='NO CLOUDS')
    
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz_inclouds,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle='-.',label='in-cloud')
    
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz_environment,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle=':',label = 'environment')
      
    locals()[f'ax{count+1}'].axhline(abl_sel/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='-',label='z$_i$')

    locals()[f'ax{count+1}'].fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1,label='cloud layer')
    
    locals()[f'ax{count+1}'].axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')
        
    #add values to dictionary
    idx = find_closest(np.array(z_3D),abl_sel)
    sub_dict['dwCO2_dz'] = str(sel_wCO2_dz[idx])
    sub_dict['sel_wCO2_dz_inclouds'] = str(sel_wCO2_dz_inclouds[idx])
    sub_dict['sel_wCO2_dz_environment'] = str(sel_wCO2_dz_environment[idx])
    sub_dict['abl'] = str(abl_sel)
    
    total_dict[str(time)] = sub_dict   
    
    #Detailing
    locals()[f'ax{count+1}'].set_ylim(0,3)
    
    locals()[f'ax{count+1}'].tick_params(axis='x')
    locals()[f'ax{count+1}'].tick_params(axis='y')
    locals()[f'ax{count+1}'].spines['right'].set_visible(False)
    locals()[f'ax{count+1}'].spines['top'].set_visible(False)
    
    locals()[f'ax{count+1}'].yaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].xaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].tick_params(which='major')
    locals()[f'ax{count+1}'].tick_params(which='minor')
        
    locals()[f'ax{count+1}'].set_title(f'{time} LT',y=1.02)
    
    count = count + 1

ax4.set_ylabel('z/z$_i$')
ax4.set_xlabel(r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$ (ppm h$^{-1}$)')
ax5.set_xlabel(r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$ (ppm h$^{-1}$)')
ax6.set_xlabel(r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$ (ppm h$^{-1}$)')

ax4.set_ylim(0,2)
ax4.set_xlim(-16,16)
ax4.set_xticks(np.arange(-16,17,8),np.arange(-16,17,8))
ax5.set_xlim(-3,3)
ax5.set_xticks(np.arange(-3,3.5,1),np.arange(-3,3.5,1).astype(int))
ax6.set_xlim(-0.25,1)
ax6.set_xticks(np.arange(0,1.25,0.5),np.arange(0,1.25,0.5))
 
legend_maker(ax4,2,3,0.8,-0.65)

ax1.text(-0.1,1.1, '(a)',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax2.text(-0.1,1.1, '(b)',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax3.text(-0.1,1.1, '(c)',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.2)

ax4.text(-0.1,1.1, '(d)',transform=ax4.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax5.text(-0.1,1.1, '(e)',transform=ax5.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax6.text(-0.1,1.1, '(f)',transform=ax6.transAxes,fontsize=plt.rcParams['font.size']*1.2)

ax1.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax2.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax3.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax4.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax5.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)
ax6.grid(linestyle=':',color='tab:gray',alpha=0.3,linewidth=axiswidth*2)



plt.subplots_adjust(hspace=1)

plt.savefig(f'{outdir}/Figure_6.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_6.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_6.pdf',bbox_inches='tight')
plt.show()
plt.close()

json_string = json.dumps(total_dict)

# Open a file for writing
with open(f'{outdir}/Figure_6_values_at_zi.txt', 'w') as f:
    # Write the JSON string to the file
    f.write(json_string)

#%%
#%%
#====================================
#           Figure 7 
#====================================

#3D
ref_time_3D = field_3D.time.values

#Reference heights 3D
zt_3D = field_3D.zt.values
zm_3D = field_3D.zm.values
z_3D = field_3D.zt.values[:-1]+40


#-----------------------------------------------------------------------------------------------------------

# u        (time, zt, yt, xm) 
# v        (time, zt, ym, xt) 
# w        (time, zm, yt, xt) 
# qt       (time, zt, yt, xt) 
# ql       (time, zt, yt, xt) 
# thl      (time, zt, yt, xt) 
# buoy     (time, zt, yt, xt) 
# sv001    (time, zt, yt, xt) 
    
time = 36+6 #13 LT

#-------------------------------------------------------------------------------------
#                           R 0 3 4  |  C L O U D S 
#-------------------------------------------------------------------------------------

#CO2
mean_CO2 = field_3D.sv001[time,:,:].mean(dim=('yt', 'xt')).values / 1000
var_CO2 = field_3D.sv001[time,:,:]/ 1000  - mean_CO2[:, None, None]

grad_CO2 = field_3D.sv001[time,:,:].differentiate('zt') / 1000
grad_CO2 = grad_CO2.mean(dim=('yt', 'xt'))

#q
q = field_3D.qt[time,:,:] *1000
mean_q = q[:,:].mean(dim=('yt', 'xt')).values
var_q = q[:,:] - mean_q[:, None, None]

#qCO2
qCO2 = var_CO2[:-1, :, :] * var_q[:-1, :, :]
qCO2_mean = qCO2.mean(dim=('yt', 'xt'))


#corr
std_q = var_q[:-1, :, :].std(dim=('yt', 'xt'))
std_CO2 = var_CO2[:-1, :, :].std(dim=('yt', 'xt'))

corr = (qCO2)/(std_q*std_CO2)
corr_mean = corr.mean(dim=('yt', 'xt')).values

    
#Compute w
w = (field_3D.w[time,:,:][0:, :, :] + field_3D.w[time,:,:][:-1, :, :]) / 2
mean_w = w.mean(dim=('yt', 'xt'))
var_w = w - mean_w

#wCO2/dz
wCO2 = var_CO2[:-1, :, :] * var_w.values
wCO2_mean = wCO2.mean(dim=('yt', 'xt'))
wCO2_dz_mean = wCO2_mean.differentiate('zt')

idx = np.where(wCO2_mean.zt.values==50)[0][0]
    
#wq
wq = var_q[:-1, :, :] * var_w.values
wq_mean = wq.mean(dim=('yt', 'xt'))

#Theta_v 
thl = field_3D.thl[time,:,:]
qt = field_3D.qt[time,:,:]
theta_v = thl * (1 + (0.61 * qt))
mean_theta_v = theta_v.mean(dim=('yt', 'xt'))

#Buoyancy
buoy = field_3D.buoy[time,:,:]
mean_buoy = buoy.mean(dim=('yt', 'xt'))
var_buoy = buoy - mean_buoy

#-------------------------------------------------------------------------------------

#Cloud mask - Condition 1
cloud_mask_condition_1 = field_3D.ql[time,:,:,:] > 0
cloud_mask_2D_condition_1 = cloud_mask_condition_1.mean(dim=('zt')).values

cloud_mask_2D_condition_1 = cloud_mask_2D_condition_1
cloud_mask_2D_condition_1[cloud_mask_2D_condition_1 > 0] = 1
cloud_mask_2D_condition_1[cloud_mask_2D_condition_1 < 1] = 0

cloud_mask_3D_condition_1 = cloud_mask_2D_condition_1[np.newaxis,:,:]
cloud_mask_3D_full_condition_1 = np.broadcast_to(cloud_mask_3D_condition_1, (250,360,360))
cloud_mask_3D_condition_1 = np.broadcast_to(cloud_mask_3D_condition_1, (249,360,360))

cloud_mask_3D_condition_1 = xr.DataArray(cloud_mask_3D_condition_1,coords=[field_3D.zt[:-1],field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
cloud_mask_3D_full_condition_1 = xr.DataArray(cloud_mask_3D_full_condition_1,coords=[field_3D.zt,field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])

calc_cfrac_condition_1 = np.nansum(cloud_mask_2D_condition_1)/np.size(cloud_mask_2D_condition_1)

#-------------------------------------------------------------------------------------

#Cloud mask - Condition 2
ql_values = field_3D.ql[time,:-1,:,:].values

x = ql_values
y = var_w.values
mask = ((x > 0) & (y>0)).astype(int)

mask = np.nanmean(mask,axis=0)

mask[mask > 0] = 1
mask[mask < 1] = 0

cloud_mask_2D_condition_2 = mask

cloud_mask_3D_condition_2 = cloud_mask_2D_condition_2[np.newaxis,:,:]
cloud_mask_3D_full_condition_2 = np.broadcast_to(cloud_mask_3D_condition_2, (250,360,360))
cloud_mask_3D_condition_2 = np.broadcast_to(cloud_mask_3D_condition_2, (249,360,360))

cloud_mask_3D_condition_2 = xr.DataArray(cloud_mask_3D_condition_2,coords=[field_3D.zt[:-1],field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
cloud_mask_3D_full_condition_2 = xr.DataArray(cloud_mask_3D_full_condition_2,coords=[field_3D.zt,field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])

calc_cfrac_condition_2 = np.nansum(cloud_mask_2D_condition_2)/np.size(cloud_mask_2D_condition_2)


#-------------------------------------------------------------------------------------

#Cloud mask - Condition 3
ql_values = field_3D.ql[time,:-1,:,:].values

x = ql_values
y = var_buoy.values[:-1,:,:]
z = var_w.values
mask = ((x > 0) & (y>0) & (z>0)).astype(int)

mask = np.nanmean(mask,axis=0)

mask[mask > 0] = 1
mask[mask < 1] = 0

cloud_mask_2D_condition_3 = mask

cloud_mask_3D_condition_3 = cloud_mask_2D_condition_3[np.newaxis,:,:]
cloud_mask_3D_full_condition_3 = np.broadcast_to(cloud_mask_3D_condition_3, (250,360,360))
cloud_mask_3D_condition_3 = np.broadcast_to(cloud_mask_3D_condition_3, (249,360,360))

cloud_mask_3D_condition_3 = xr.DataArray(cloud_mask_3D_condition_3,coords=[field_3D.zt[:-1],field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])
cloud_mask_3D_full_condition_3 = xr.DataArray(cloud_mask_3D_full_condition_3,coords=[field_3D.zt,field_3D.yt,field_3D.xt],dims=['zt','yt','xt'])

calc_cfrac_condition_3 = np.nansum(cloud_mask_2D_condition_3)/np.size(cloud_mask_2D_condition_3)


#-------------------------------------------------------------------------------------

#Inclouds Contributions  


#wCO2

#Condition 1
wCO2_inclouds_condition_1 = wCO2.where(cloud_mask_3D_condition_1==1)
wCO2_inclouds_mean_condition_1 = wCO2_inclouds_condition_1.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_1)
wCO2_inclouds_mean_nofactor_condition_1 = wCO2_inclouds_condition_1.mean(dim=('yt', 'xt'))

wCO2_dz_inclouds_condition_1 = wCO2_inclouds_mean_condition_1.differentiate('zt')



idx = np.where(wCO2_inclouds_mean_condition_1.zt.values+40==50)[0][0]

#Condition 2
wCO2_inclouds_condition_2 = wCO2.where(cloud_mask_3D_condition_2==1)
wCO2_inclouds_mean_condition_2 = wCO2_inclouds_condition_2.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_2)
wCO2_inclouds_mean_nofactor_condition_2 = wCO2_inclouds_condition_2.mean(dim=('yt', 'xt'))

wCO2_dz_inclouds_condition_2 = wCO2_inclouds_mean_condition_2.differentiate('zt')



idx = np.where(wCO2_inclouds_mean_condition_2.zt.values+40==50)[0][0]


#Condition 3
wCO2_inclouds_condition_3 = wCO2.where(cloud_mask_3D_condition_3==1)
wCO2_inclouds_mean_condition_3 = wCO2_inclouds_condition_3.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_3)
wCO2_inclouds_mean_nofactor_condition_3 = wCO2_inclouds_condition_3.mean(dim=('yt', 'xt'))

wCO2_dz_inclouds_condition_3 = wCO2_inclouds_mean_condition_3.differentiate('zt')



idx = np.where(wCO2_inclouds_mean_condition_3.zt.values+40==50)[0][0]


#-------------------------------------------------------------------------------------

#qCO2

#Condition 1
qCO2_inclouds_condition_1 = qCO2.where(cloud_mask_3D_condition_1==1)
qCO2_inclouds_mean_condition_1 = qCO2_inclouds_condition_1.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_1)



#Condition 2
qCO2_inclouds_condition_2 = qCO2.where(cloud_mask_3D_condition_2==1)
qCO2_inclouds_mean_condition_2 = qCO2_inclouds_condition_2.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_2)



#Condition 3
qCO2_inclouds_condition_3 = qCO2.where(cloud_mask_3D_condition_3==1)
qCO2_inclouds_mean_condition_3 = qCO2_inclouds_condition_3.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_3)



#-------------------------------------------------------------------------------------

#Correlation

#Condition 1
corr_inclouds_condition_1 = corr.where(cloud_mask_3D_condition_1==1)
corr_inclouds_mean_condition_1 = corr_inclouds_condition_1.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_1)


#Condition 2
corr_inclouds_condition_2 = corr.where(cloud_mask_3D_condition_2==1)
corr_inclouds_mean_condition_2 = corr_inclouds_condition_2.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_2)


#Condition 3
corr_inclouds_condition_3 = corr.where(cloud_mask_3D_condition_3==1)
corr_inclouds_mean_condition_3 = corr_inclouds_condition_3.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_3)


#-------------------------------------------------------------------------------------

#wq

#Condition 1
wq_inclouds_condition_1 = wq.where(cloud_mask_3D_condition_1==1)
wq_inclouds_mean_condition_1 = wq_inclouds_condition_1.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_1)
wq_inclouds_nofactor_condition_1 = wq_inclouds_condition_1.mean(dim=('yt', 'xt'))


#Condition 2
wq_inclouds_condition_2 = wq.where(cloud_mask_3D_condition_2==1)
wq_inclouds_mean_condition_2 = wq_inclouds_condition_2.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_2)
wq_inclouds_nofactor_condition_2 = wq_inclouds_condition_2.mean(dim=('yt', 'xt'))



#Condition 3
wq_inclouds_condition_3 = wq.where(cloud_mask_3D_condition_3==1)
wq_inclouds_mean_condition_3 = wq_inclouds_condition_3.mean(dim=('yt', 'xt'))*(calc_cfrac_condition_3)
wq_inclouds_nofactor_condition_3 = wq_inclouds_condition_3.mean(dim=('yt', 'xt'))



#-------------------------------------------------------------------------------------

#Grad CO2

#Condition 1
grad_CO2_inclouds_condition_1 = grad_CO2.where(cloud_mask_3D_full_condition_1==1)
grad_CO2_inclouds_mean_condition_1 = grad_CO2_inclouds_condition_1.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_1)


#Condition 2
grad_CO2_inclouds_condition_2 = grad_CO2.where(cloud_mask_3D_full_condition_2==1)
grad_CO2_inclouds_mean_condition_2 = grad_CO2_inclouds_condition_2.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_2)


#Condition 3
grad_CO2_inclouds_condition_3 = grad_CO2.where(cloud_mask_3D_full_condition_3==1)
grad_CO2_inclouds_mean_condition_3 = grad_CO2_inclouds_condition_3.mean(dim=('yt', 'xt')).values*(calc_cfrac_condition_3)


#-------------------------------------------------------------------------------------
    
#-------------------------------------------------------------------------------------

#Environment

#wCO2

#Condition 1
wCO2_environment_condition_1 = wCO2.where(cloud_mask_3D_condition_1==0)
wCO2_environment_mean_condition_1 = wCO2_environment_condition_1.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_1)
wCO2_environment_mean_nofactor_condition_1 = wCO2_environment_condition_1.mean(dim=('yt', 'xt'))

wCO2_dz_environment_condition_1 = wCO2_environment_mean_condition_1.differentiate('zt')



idx = np.where(wCO2_environment_mean_condition_1.zt.values+40==50)[0][0]

#Condition 2
wCO2_environment_condition_2 = wCO2.where(cloud_mask_3D_condition_2==0)
wCO2_environment_mean_condition_2 = wCO2_environment_condition_2.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_2)
wCO2_environment_mean_nofactor_condition_2 = wCO2_environment_condition_2.mean(dim=('yt', 'xt'))

wCO2_dz_environment_condition_2 = wCO2_environment_mean_condition_2.differentiate('zt')


idx = np.where(wCO2_environment_mean_condition_2.zt.values+40==50)[0][0]

#Condition 3
wCO2_environment_condition_3 = wCO2.where(cloud_mask_3D_condition_3==0)
wCO2_environment_mean_condition_3 = wCO2_environment_condition_3.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_3)
wCO2_environment_mean_nofactor_condition_3 = wCO2_environment_condition_3.mean(dim=('yt', 'xt'))

wCO2_dz_environment_condition_3 = wCO2_environment_mean_condition_3.differentiate('zt')


idx = np.where(wCO2_environment_mean_condition_3.zt.values+40==50)[0][0]

#-------------------------------------------------------------------------------------

#qCO2

#Condition 1
qCO2_environment_condition_1 = qCO2.where(cloud_mask_3D_condition_1==0)
qCO2_environment_mean_condition_1 = qCO2_environment_condition_1.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_1)


#Condition 2
qCO2_environment_condition_2 = qCO2.where(cloud_mask_3D_condition_2==0)
qCO2_environment_mean_condition_2 = qCO2_environment_condition_2.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_2)


#Condition 3
qCO2_environment_condition_3 = qCO2.where(cloud_mask_3D_condition_3==0)
qCO2_environment_mean_condition_3 = qCO2_environment_condition_3.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_3)


#-------------------------------------------------------------------------------------

#Correlation

#Condition 1
corr_environment_condition_1 = corr.where(cloud_mask_3D_condition_1==0)
corr_environment_mean_condition_1 = corr_environment_condition_1.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_1)

#Condition 2
corr_environment_condition_2 = corr.where(cloud_mask_3D_condition_2==0)
corr_environment_mean_condition_2 = corr_environment_condition_2.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_2)

#Condition 3
corr_environment_condition_3 = corr.where(cloud_mask_3D_condition_3==0)
corr_environment_mean_condition_3 = corr_environment_condition_3.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_3)

#-------------------------------------------------------------------------------------
    
#wq

#Condition 1
wq_environment_condition_1 = wq.where(cloud_mask_3D_condition_1==0)
wq_environment_mean_condition_1 = wq_environment_condition_1.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_1)
wq_environment_nofactor_condition_1 = wq_environment_condition_1.mean(dim=('yt', 'xt'))


#Condition 2
wq_environment_condition_2 = wq.where(cloud_mask_3D_condition_2==0)
wq_environment_mean_condition_2 = wq_environment_condition_2.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_2)
wq_environment_nofactor_condition_2 = wq_environment_condition_2.mean(dim=('yt', 'xt'))


#Condition 3
wq_environment_condition_3 = wq.where(cloud_mask_3D_condition_3==0)
wq_environment_mean_condition_3 = wq_environment_condition_3.mean(dim=('yt', 'xt'))*(1-calc_cfrac_condition_3)
wq_environment_nofactor_condition_3 = wq_environment_condition_3.mean(dim=('yt', 'xt'))


#-------------------------------------------------------------------------------------

#Grad CO2

#Condition 1
grad_CO2_environment_condition_1 = grad_CO2.where(cloud_mask_3D_full_condition_1==0)
grad_CO2_environment_mean_condition_1 = grad_CO2_environment_condition_1.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_1)

#Condition 2
grad_CO2_environment_condition_2 = grad_CO2.where(cloud_mask_3D_full_condition_2==0)
grad_CO2_environment_mean_condition_2 = grad_CO2_environment_condition_2.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_2)

#Condition 3
grad_CO2_environment_condition_3 = grad_CO2.where(cloud_mask_3D_full_condition_3==0)
grad_CO2_environment_mean_condition_3 = grad_CO2_environment_condition_3.mean(dim=('yt', 'xt')).values*(1-calc_cfrac_condition_3)

#-------------------------------------------------------------------------------------

#Theta_v
    
#-------------------------------------------------------------------------------------
#                           R 0 3 5  |  N O  C L O U D S 
#-------------------------------------------------------------------------------------

#CO2
mean_CO2_no_clouds = field_3D_no_clouds.sv001[time,:,:].mean(dim=('yt', 'xt')).values / 1000
var_CO2_no_clouds = field_3D_no_clouds.sv001[time,:,:]/1000  - mean_CO2_no_clouds[:, None, None]

grad_CO2_no_clouds = field_3D_no_clouds.sv001[time,:,:].differentiate('zt') / 1000
grad_CO2_no_clouds = grad_CO2_no_clouds.mean(dim=('yt', 'xt')).values

#q
q_no_clouds = field_3D_no_clouds.qt[time,:,:] * 1000
mean_q_no_clouds = q_no_clouds[:,:].mean(dim=('yt', 'xt')).values
var_q_no_clouds = q_no_clouds[:,:] - mean_q_no_clouds[:, None, None]

#qCO2
qCO2_no_clouds = var_CO2_no_clouds[:-1, :, :] * var_q_no_clouds[:-1, :, :]
qCO2_mean_no_clouds = qCO2_no_clouds.mean(dim=('yt', 'xt'))


#corr
std_q_no_clouds = var_q_no_clouds[:-1, :, :].std(dim=('yt', 'xt'))
std_CO2_no_clouds = var_CO2_no_clouds[:-1, :, :].std(dim=('yt', 'xt'))

corr_no_clouds = (qCO2_no_clouds)/(std_q_no_clouds*std_CO2_no_clouds)
corr_mean_no_clouds = corr_no_clouds.mean(dim=('yt', 'xt')).values

#Compute w
w_no_clouds = (field_3D_no_clouds.w[time,:,:][0:, :, :] + field_3D_no_clouds.w[time,:,:][:-1, :, :]) / 2
mean_w_no_clouds = w_no_clouds.mean(dim=('yt', 'xt'))
var_w_no_clouds = w_no_clouds - mean_w_no_clouds

#wCO2/dz
wCO2_no_clouds = var_CO2_no_clouds[:-1, :, :] * var_w_no_clouds.values
wCO2_mean_no_clouds = wCO2_no_clouds.mean(dim=('yt', 'xt'))
wCO2_dz_mean_no_clouds = wCO2_mean_no_clouds.differentiate('zt')


idx = np.where(wCO2_mean_no_clouds.zt.values==50)[0][0]

#wq
wq_no_clouds = var_q_no_clouds[:-1, :, :] * var_w_no_clouds.values
wq_mean_no_clouds = wq_no_clouds.mean(dim=('yt', 'xt'))

#wtheta
theta_no_clouds = field_3D_no_clouds.thl[time,:,:]
theta_v_no_clouds = theta_no_clouds * (1 + 0.61 * q_no_clouds/1000)
theta_v_mean_no_clouds = theta_v_no_clouds.mean(dim=('yt', 'xt'))
var_theta_v_no_clouds = theta_v_no_clouds - theta_v_mean_no_clouds

#Theta_v 
thl = field_3D_no_clouds.thl[time,:,:]
qt = field_3D_no_clouds.qt[time,:,:]
theta_v = thl * (1 + 0.61 * qt)
mean_theta_v = theta_v.mean(dim=('yt', 'xt'))
var_theta_v = theta_v - mean_theta_v


#%%
#=============================
#           Figure 7
#=============================

x = 80
xmin, xmax = 180-x,180+x
ymin, ymax = xmin,xmax

dx = 53 #to convert

cmap = 'bwr'

fig, ax = plt.subplots(4,3,dpi=72, figsize=(8.27*1.5, 11.69)) #8.27*1.5, 11.69

# ax1, ax2, ax3 = ax.flatten()
ax1, ax2, ax3, ax4, ax5, ax6, ax7, ax8, ax9, ax10, ax11, ax12 = ax.flatten()

alpha = 0.02

ax1.set_position([0.1, 0.7, 0.25, 0.25]) #left, bottom, width, height
ax2.set_position([0.45+alpha, 0.7, 0.25, 0.25])
ax3.set_position([0.75+alpha*2, 0.7, 0.25, 0.25])
ax4.set_position([1.05+alpha*3, 0.7, 0.25, 0.25])

ax5.set_position([0.1, 0.36, 0.25, 0.25])
ax6.set_position([0.45+alpha, 0.36, 0.25, 0.25])
ax7.set_position([0.75+alpha*2, 0.36, 0.25, 0.25])
ax8.set_position([1.05+alpha*3, 0.36, 0.25, 0.25])

ax9.set_position([0.1, 0.02, 0.25, 0.25])
ax10.set_position([0.45+alpha, 0.02, 0.25, 0.25])
ax11.set_position([0.75+alpha*2, 0.02, 0.25, 0.25])
ax12.set_position([1.05+alpha*3, 0.02, 0.25, 0.25])

# ax13.set_position([0.1, -0.36, 0.25, 0.25])
# ax14.set_position([0.45, -0.36, 0.25, 0.25])
# ax15.set_position([0.75, -0.36, 0.25, 0.25])
# ax16.set_position([1.05, -0.36, 0.25, 0.25])


#Scatter plots
axes = ['ax1','ax5','ax9']
heights = [3000,1100,200]

for i in range(len(axes)):
    
    a = var_CO2[find_closest(z_3D,heights[i]),:,:].values
    b = var_w[find_closest(z_3D,heights[i]),:,:].values
    c = var_q[find_closest(z_3D,heights[i]),:,:].values
    
    a_no_cloud = var_CO2_no_clouds[find_closest(z_3D,heights[i]),:,:].values
    b_no_cloud = var_w_no_clouds[find_closest(z_3D,heights[i]),:,:].values
    c_no_cloud = var_q_no_clouds[find_closest(z_3D,heights[i]),:,:].values
    
    a_cloud = np.where(cloud_mask_2D_condition_2 == 1, a, 0)
    b_cloud = np.where(cloud_mask_2D_condition_2 == 1, b, 0)
    c_cloud = np.where(cloud_mask_2D_condition_2 == 1, c, 0)
    
    a_env = np.where(cloud_mask_2D_condition_2 == 0, a, 0)
    b_env = np.where(cloud_mask_2D_condition_2 == 0, b, 0)
    c_env = np.where(cloud_mask_2D_condition_2 == 0, c, 0)
    
    locals()[f'{axes[i]}'].scatter(a_no_cloud,c_no_cloud,s=0.3,color='green',alpha=0.4,label='NO CLOUDS')
    locals()[f'{axes[i]}'].scatter(a_cloud,c_cloud,s=0.3,color='tab:red',alpha=0.4,label='in-cloud')
    locals()[f'{axes[i]}'].scatter(a_env,c_env,s=0.3,color='tab:blue',alpha=0.4,label='environment')
    
    #Add horizontal and vertical lines through the origin
    locals()[f'{axes[i]}'].axhline(0, color='tab:gray',linewidth=1)
    locals()[f'{axes[i]}'].axvline(0, color='tab:gray',linewidth=1)

    #Regression line - in-cloud
    slope_in_cloud, intercept_in_cloud, r_value_in_cloud, p_value_in_cloud, std_err_in_cloud = stats.linregress(a_cloud.flatten(),c_cloud.flatten())
    res_in_cloud = stats.pearsonr(a_cloud.flatten(),c_cloud.flatten())
      
    x = np.arange(-20, 22, 2)
        
    locals()[f'{axes[i]}'].plot(x, intercept_in_cloud + slope_in_cloud * x, color='darkred',linewidth=axiswidth*2,linestyle='--')

    #Regression line - environment
    slope_env, intercept_env, r_value_env, p_value_env, std_err_env = stats.linregress(a_env.flatten(),c_env.flatten())
    res_env = stats.pearsonr(a_env.flatten(),c_env.flatten())
    
    x = np.arange(-20, 22, 2)
    
    locals()[f'{axes[i]}'].plot(x, intercept_env + slope_env * x, color='darkblue',linewidth=axiswidth*2,linestyle='-.')
    
    #Regression line - no cloud
    slope_no_cloud, intercept_no_cloud, r_value_no_cloud, p_value_no_cloud, std_err_no_cloud = stats.linregress(a_no_cloud.flatten(),c_no_cloud.flatten())
    res_no_cloud = stats.pearsonr(a_no_cloud.flatten(),c_no_cloud.flatten())
    
    maximum_value = np.max([np.max(a_no_cloud),np.max(c_no_cloud)])
    minimum_value = np.min([np.min(a_no_cloud),np.min(c_no_cloud)])
    
    x = np.arange(-20, 22, 2)
    
    locals()[f'{axes[i]}'].plot(x, intercept_no_cloud + slope_no_cloud * x, color='darkgreen',linewidth=axiswidth*2,linestyle='-.')
    
    #Set xlim and ylim
    max_c = np.round(np.max(c))
    min_c = np.round(np.min(c))
    
    max_a = np.round(np.max(a))
    min_a = np.round(np.min(a))
    
    ref_arr = np.arange(-20,22,2)
    
    min_val_c = ref_arr[find_closest(ref_arr,min_c)]
    max_val_c = ref_arr[find_closest(ref_arr,max_c)]
    
    min_val_a = ref_arr[find_closest(ref_arr,min_a)]
    max_val_a = ref_arr[find_closest(ref_arr,max_a)]
    
    locals()[f'{axes[i]}'].set_xticks(np.arange(min_val_a-4,max_val_a+4,1),np.arange(min_val_a-4,max_val_a+4,1))
    
    if i !=0:
        locals()[f'{axes[i]}'].set_yticks(np.arange(min_val_c-4,max_val_c+4,1),np.arange(min_val_c-4,max_val_c+4,1))
    else:   
        locals()[f'{axes[i]}'].set_yticks(np.arange(min_val_c-4,max_val_c+4,2),np.arange(min_val_c-4,max_val_c+4,2))
    
    locals()[f'{axes[i]}'].set_xlim(min_val_a-0.5,max_val_a+2)
    locals()[f'{axes[i]}'].set_ylim(min_val_c,max_val_c+2)
    
    locals()[f'{axes[i]}'].grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.5)

    #Set r values text
    if i==0:
        locals()[f'{axes[i]}'].text(0.69, 0.32+0.16, f' r = {np.round(res_in_cloud[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkred')
        locals()[f'{axes[i]}'].text(0.69, 0.24+0.16, f' r = {np.round(res_env[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkblue')
        locals()[f'{axes[i]}'].text(0.69, 0.16+0.16, f' r = {np.round(res_no_cloud[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkgreen')
    elif i==1:
        locals()[f'{axes[i]}'].text(0.69, 0.32, f' r = {np.round(res_in_cloud[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkred')
        locals()[f'{axes[i]}'].text(0.69, 0.24, f' r = {np.round(res_env[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkblue')
        locals()[f'{axes[i]}'].text(0.69, 0.16, f' r = {np.round(res_no_cloud[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkgreen')
    elif i==2:
        locals()[f'{axes[i]}'].text(0.69, 0.32-0.14, f' r = {np.round(res_in_cloud[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkred')
        locals()[f'{axes[i]}'].text(0.69, 0.24-0.14, f' r = {np.round(res_env[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkblue')
        locals()[f'{axes[i]}'].text(0.69, 0.16-0.14, f' r = {np.round(res_no_cloud[0],2)}',transform=locals()[f'{axes[i]}'].transAxes,fontsize=plt.rcParams['font.size']*1.2,color='darkgreen')
    
    #Detailing
    if i == 0:
        legend = locals()[f'{axes[i]}'].legend(loc='upper left', bbox_to_anchor= (-0.8, 1.38),fontsize=plt.rcParams['font.size']*1.2,frameon=False,markerscale=18,ncols=3) #handlelength=0.5,handletextpad=0.5
        for lh in legend.legend_handles:
            lh.set_alpha(1)
    
    
    locals()[f'{axes[i]}'].tick_params(axis='x')
    locals()[f'{axes[i]}'].tick_params(axis='y')

    locals()[f'{axes[i]}'].yaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'{axes[i]}'].xaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'{axes[i]}'].spines['right'].set_visible(False)
    locals()[f'{axes[i]}'].spines['top'].set_visible(False)
    
#Info
ax1.text(-0.5,1.,'c) '+r'\textbf{z$_H$}',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax5.text(-0.5,1.,'b) '+r'\textbf{z$_M$}',transform=ax5.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax9.text(-0.5,1.,'a) '+r'\textbf{z$_L$}',transform=ax9.transAxes,fontsize=plt.rcParams['font.size']*1.2)

ax1.set_ylabel(r"$\mathrm{H_2 O}'$ (g kg$^{-1}$)",fontsize=plt.rcParams['font.size'])


ax5.set_ylabel(r"$\mathrm{H_2 O}'$ (g kg$^{-1}$)",fontsize=plt.rcParams['font.size'])

ax9.set_xlabel(r"$\mathrm{CO_2}'$ (ppm)",fontsize=plt.rcParams['font.size'])
ax9.set_ylabel(r"$\mathrm{H_2 O}'$ (g kg$^{-1}$)",fontsize=plt.rcParams['font.size'])



#Contour plots

#w'
min_val = -6
max_val = 6

axes = ['ax2','ax6','ax10']
values = [8,4,5]
steps = [2,2,2.5]

for i in range(len(axes)): 
    
    h = heights[i]
    min_val = values[i]*-1
    max_val = values[i]
    step = steps[i]
    
    a_cloud = var_CO2[find_closest(z_3D,h),:,:].values
    b_cloud = var_w[find_closest(z_3D,h),:,:].values
    c_cloud = var_q[find_closest(z_3D,h),:,:].values
           
    norm = Normalize(vmin=min_val, vmax=max_val)
    locals()[f'{axes[i]}'].grid(color='gray',linestyle='--')
    cb = locals()[f'{axes[i]}'].pcolormesh(b_cloud,norm=norm,cmap=cmap)

    if h == 3000:
        locals()[f'{axes[i]}'].contourf(cloud_mask_2D_condition_1,levels=[0.1,1],alpha=0.1,colors='tab:gray')
        locals()[f'{axes[i]}'].contour(cloud_mask_2D_condition_3,levels=[0.1],colors='k',linewidths=axiswidth,linestyle=':')
    
    locals()[f'{axes[i]}'].set_ylim(ymin,ymax)
    locals()[f'{axes[i]}'].set_xlim(xmin,xmax)
    
    locals()[f'{axes[i]}'].yaxis.set_ticks_position('both')
    locals()[f'{axes[i]}'].tick_params(labeltop=False, labelbottom=True, labelright=False,labelleft=True)
    locals()[f'{axes[i]}'].set_xticks(np.arange(xmin,xmax+1,50),np.round(np.arange(ymin,ymax+1,50)*dx/1000,1))
    locals()[f'{axes[i]}'].set_yticks(np.arange(ymin,ymax+1,50),np.round(np.arange(ymin,ymax+1,50)*dx/1000,1))

    subber = locals()[f'{axes[i]}'].inset_axes([0, 1.03, 1, 0.02])
    colorbar = plt.colorbar(cb, cax=subber, orientation='horizontal')
    subber.xaxis.set_ticks_position('top')
    colorbar.set_ticks(np.arange(min_val,max_val+step,step))   

    #set fontsize of ticks
    subber.tick_params(labelsize=plt.rcParams['font.size'])

#h2o'

axes = ['ax3','ax7','ax11']
values = [8,3,2]
steps = [2,1,1]

for i in range(len(axes)): 
    
    h = heights[i]
    min_val = values[i]*-1
    max_val = values[i]
    step = steps[i]
    
    a_cloud = var_CO2[find_closest(z_3D,h),:,:].values
    b_cloud = var_w[find_closest(z_3D,h),:,:].values
    c_cloud = var_q[find_closest(z_3D,h),:,:].values
        
    norm = Normalize(vmin=min_val, vmax=max_val)
    locals()[f'{axes[i]}'].grid(color='gray',linestyle='--')
    cb = locals()[f'{axes[i]}'].pcolormesh(c_cloud,norm=norm,cmap=cmap)

    if h == 3000:
        locals()[f'{axes[i]}'].contourf(cloud_mask_2D_condition_1,levels=[0.1,1],alpha=0.1,colors='tab:gray')
        locals()[f'{axes[i]}'].contour(cloud_mask_2D_condition_3,levels=[0.1],colors='k',linewidths=axiswidth,linestyle=':')

    locals()[f'{axes[i]}'].set_ylim(ymin,ymax)
    locals()[f'{axes[i]}'].set_xlim(xmin,xmax)
    
    locals()[f'{axes[i]}'].yaxis.set_ticks_position('both')
    locals()[f'{axes[i]}'].tick_params(labeltop=False, labelbottom=True, labelright=False,labelleft=False)
    locals()[f'{axes[i]}'].set_xticks(np.arange(xmin,xmax+1,50),np.round(np.arange(ymin,ymax+1,50)*dx/1000,1))
    locals()[f'{axes[i]}'].set_yticks(np.arange(ymin,ymax+1,50),np.round(np.arange(ymin,ymax+1,50)*dx/1000,1))

    subber = locals()[f'{axes[i]}'].inset_axes([0, 1.03, 1, 0.02])
    colorbar = plt.colorbar(cb, cax=subber, orientation='horizontal')
    subber.xaxis.set_ticks_position('top')
    colorbar.set_ticks(np.arange(min_val,max_val+step,step))   

    #set fontsize of ticks
    subber.tick_params(labelsize=plt.rcParams['font.size'])

#co2
axes = ['ax4','ax8','ax12']
values = [2,1,2]
steps = [1,0.5,1]

for i in range(len(axes)): 
    
    h = heights[i]
    min_val = values[i]*-1
    max_val = values[i]
    step = steps[i]
    
    a_cloud = var_CO2[find_closest(z_3D,h),:,:].values
    b_cloud = var_w[find_closest(z_3D,h),:,:].values
    c_cloud = var_q[find_closest(z_3D,h),:,:].values
        
    norm = Normalize(vmin=min_val, vmax=max_val)
    locals()[f'{axes[i]}'].grid(color='gray',linestyle='--')
    cb = locals()[f'{axes[i]}'].pcolormesh(a_cloud,norm=norm,cmap=cmap)

    if h == 3000:
        locals()[f'{axes[i]}'].contourf(cloud_mask_2D_condition_1,levels=[0.1,1],alpha=0.1,colors='tab:gray')
        locals()[f'{axes[i]}'].contour(cloud_mask_2D_condition_3,levels=[0.1],colors='k',linewidths=axiswidth,linestyle=':')

    locals()[f'{axes[i]}'].set_ylim(ymin,ymax)
    locals()[f'{axes[i]}'].set_xlim(xmin,xmax)
    
    locals()[f'{axes[i]}'].yaxis.set_ticks_position('both')
    locals()[f'{axes[i]}'].tick_params(labeltop=False, labelbottom=True, labelright=True,labelleft=False)
    locals()[f'{axes[i]}'].set_xticks(np.arange(xmin,xmax+1,50),np.round(np.arange(ymin,ymax+1,50)*dx/1000,1))
    locals()[f'{axes[i]}'].set_yticks(np.arange(ymin,ymax+1,50),np.round(np.arange(ymin,ymax+1,50)*dx/1000,1))
    
    subber = locals()[f'{axes[i]}'].inset_axes([0, 1.03, 1, 0.02])
    colorbar = plt.colorbar(cb, cax=subber, orientation='horizontal')
    subber.xaxis.set_ticks_position('top')
    colorbar.set_ticks(np.arange(min_val,max_val+step,step))   

    #set fontsize of ticks
    subber.tick_params(labelsize=plt.rcParams['font.size'])


ax2.set_ylabel('y (km)')
ax4.yaxis.set_label_position('right')
ax4.set_ylabel('y (km)',rotation=-90,labelpad=35)

ax6.set_ylabel('y (km)')
ax8.yaxis.set_label_position('right')
ax8.set_ylabel('y (km)',rotation=-90,labelpad=35)

ax10.set_ylabel('y (km)')
ax12.yaxis.set_label_position('right')
ax12.set_ylabel('y (km)',rotation=-90,labelpad=35)
ax10.set_xlabel('x (km)')
ax11.set_xlabel('x (km)')
ax12.set_xlabel('x (km)')

ax2.text(.35,1.25,r"$\mathrm{w}'$ (m s$^{-1}$)",transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax3.text(.35,1.25,r"$\mathrm{H_2 O}'$ (g kg$^{-1})$",transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax4.text(.35,1.25,r"$\mathrm{CO_2}'$ (ppm)",transform=ax4.transAxes,fontsize=plt.rcParams['font.size']*1.2)

#Savefig
plt.savefig(f'{outdir}/Figure_7.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_7.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_7.pdf',bbox_inches='tight')

plt.show()
    
plt.close()


#%%

#=================
# Figure 8
#=================

fig,ax = plt.subplots(1,3,dpi=72, figsize=(8.27*1.5, 11.69/3))
ax1, ax2, ax3 = ax.flatten()

times = [9,12,15]

total_dict = {}

count = 0 
for time in times:
    
    sub_dict = {}
    
    #Clouds
    idx = find_closest(np.array(x_DALES_3D),time)
    sel_corr = corr_arr_30min_ds.values[idx,:]
    
    #No clouds
    sel_corr_no_clouds = corr_arr_no_clouds_30min_ds.values[idx,:]

    #incloud
    sel_corr_inclouds = corr_arr_inclouds_condition_2_30min_ds.values[idx,:]

    #environment
    sel_corr_environment = corr_arr_environment_condition_2_30min_ds.values[idx,:]

    #horizontal lines and cloud layer
    abl_sel = abl_1D_ds.values[idx]
    cb_sel = tmser_cloud_base_ds.values[idx]
    ct_sel = tmser_cloud_top_max_ds.values[idx]
    
    abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]
    
    #Vis
    locals()[f'ax{count+1}'].plot(sel_corr,z_3D/abl_sel,linewidth=axiswidth*2.5,color='tab:blue',label='CLOUDS')
    locals()[f'ax{count+1}'].plot(sel_corr_no_clouds,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle='--',label='NO CLOUDS')
    locals()[f'ax{count+1}'].plot(sel_corr_inclouds,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle='-.',label='in-cloud')
    locals()[f'ax{count+1}'].plot(sel_corr_environment,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle=':',label = 'environment')
    
    locals()[f'ax{count+1}'].axhline(abl_sel/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='-',label='z$_i$')
    locals()[f'ax{count+1}'].axhline(abl_sel_no_clouds/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='--')
    
    locals()[f'ax{count+1}'].fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1,label='cloud layer')
    
    locals()[f'ax{count+1}'].axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')
    
    xmax = 1.3
    
    if count == 0: 
        minner = 10
    else:
        minner = 30
        
    locals()[f'ax{count+1}'].axhline(200/abl_sel, color='gray',linewidth=axiswidth*1.5,linestyle='--')
    locals()[f'ax{count+1}'].axhline(1100/abl_sel, color='gray',linewidth=axiswidth*1.5,linestyle='--')
    locals()[f'ax{count+1}'].axhline(3000/abl_sel, color='gray',linewidth=axiswidth*1.5,linestyle='--')
    
    locals()[f'ax{count+1}'].text(xmax,(200-minner)/abl_sel,'z$_L$',color='k',fontsize=plt.rcParams['font.size'])
    locals()[f'ax{count+1}'].text(xmax,(1100-minner)/abl_sel,'z$_M$',color='k',fontsize=plt.rcParams['font.size'])
    
    if count !=0:
        locals()[f'ax{count+1}'].text(xmax,(3000-minner)/abl_sel,'z$_H$',color='k',fontsize=plt.rcParams['font.size'])

    #add values to dictionary
    idx = find_closest(np.array(z_3D),abl_sel)
    sub_dict['corr'] = str(sel_corr[idx])
    sub_dict['corr_inclouds'] = str(sel_corr_inclouds[idx])
    sub_dict['corr_environment'] = str(sel_corr_environment[idx])
    sub_dict['abl'] = str(abl_sel)
    
    idx = find_closest(np.array(z_3D),abl_sel_no_clouds)
    sub_dict['corr_no_clouds'] = str(sel_corr_no_clouds[idx])
    sub_dict['abl_no_clouds'] = str(abl_sel_no_clouds)
    
    total_dict[str(time)] = sub_dict

    #Detailing
    locals()[f'ax{count+1}'].set_ylim(0,3)
    
    locals()[f'ax{count+1}'].set_xlim(-1.2,1.2)
    locals()[f'ax{count+1}'].set_xticks(np.arange(-1,1.1,0.5),np.round(np.arange(-1,1.1,0.5),2),rotation=45)
    
    locals()[f'ax{count+1}'].tick_params(axis='x')
    locals()[f'ax{count+1}'].tick_params(axis='y')
    locals()[f'ax{count+1}'].spines['right'].set_visible(False)
    locals()[f'ax{count+1}'].spines['top'].set_visible(False)
    
    locals()[f'ax{count+1}'].yaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].xaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].tick_params(which='major')
    locals()[f'ax{count+1}'].tick_params(which='minor')
    
    locals()[f'ax{count+1}'].set_title(f'{time} LT',y=1.02)
    locals()[f'ax{count+1}'].grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)

    count = count + 1

ax1.set_ylabel('z/z$_i$')
ax2.set_xlabel('r (-)')

ax1.set_ylim(0,2)

legend_maker(ax1,2,3,0.75,-0.65)

ax1.text(-0.1,1.1, '(a)',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax2.text(-0.1,1.1, '(b)',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax3.text(-0.1,1.1, '(c)',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.2)

plt.subplots_adjust(left=0.1, bottom=0.1, right=0.9, top=0.9, wspace=0.5, hspace=0.4)

plt.savefig(f'{outdir}/Figure_8.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_8.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_8.pdf',bbox_inches='tight')
plt.show()
plt.close()

#export
json_string = json.dumps(total_dict)

# Open a file for writing
with open(f'{outdir}/Figure_8_values_at_zi.txt', 'w') as f:
    # Write the JSON string to the file
    f.write(json_string)

#%%

#====================================
# Supplements - 1D vs. 3D
#====================================

fig,ax = plt.subplots(3,3,dpi=72, figsize=(8.27*1.5, 11.69))
ax1, ax2, ax3, ax4, ax5, ax6, ax7, ax8, ax9 = ax.flatten()

#Turn off ax8 and ax9
ax8.axis('off')
ax9.axis('off')

#Set position
ax1.set_position([0.1, 0.7, 0.25, 0.25]) #left, bottom, width, height
ax2.set_position([0.4, 0.7, 0.25, 0.25])
ax3.set_position([0.7, 0.7, 0.25, 0.25])

ax4.set_position([0.1, 0.4, 0.25, 0.25])
ax5.set_position([0.4, 0.4, 0.25, 0.25])
ax6.set_position([0.7, 0.4, 0.25, 0.25])

ax7.set_position([0.1, -0.04, 0.85, 0.28])

times = [9,12,15]

total_dict = {}

count = 0 
for time in times:
    
    #1D
    idx = find_closest(np.array(x_DALES_1D),time)
    
    #Clouds
    sel_CO2_dt_1D = CO2_dt_1D[idx]*3600
    sel_wCO2_dz_1D = wCO2_t_dz_1D[idx]*3600
    
    #No clouds
    sel_CO2_dt_no_clouds_1D = CO2_dt_1D_no_clouds[idx]*3600
    sel_wCO2_dz_no_clouds_1D = wCO2_t_dz_1D_no_clouds[idx]*3600
    
    #3D
    idx = find_closest(np.array(x_DALES_3D),time)
   
    #Clouds
    sel_CO2_dt = CO2_dt_mean_30min_ds.values[idx,:]*3600
    sel_wCO2_dz = wCO2_dz_arr_30min_ds.values[idx,:]*3600
    
    #No clouds
    sel_CO2_dt_no_clouds = CO2_dt_mean_no_clouds_30min_ds.values[idx,:]*3600
    sel_wCO2_dz_no_clouds = wCO2_dz_arr_no_clouds_30min_ds.values[idx,:]*3600
    
    #horizontal lines and cloud layer
    abl_sel = abl_1D_ds.values[idx]
    cb_sel = tmser_cloud_base_ds.values[idx]
    ct_sel = tmser_cloud_top_max_ds.values[idx]
    
    abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]
    
    #Visualise
    locals()[f'ax{count+1}'].plot(sel_CO2_dt_1D,z_1D/abl_sel,linewidth=axiswidth*2,color='darkred',label=r'$\frac{\partial \overline{CO_2}}{\partial t}$')
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz_1D,z_1D/abl_sel,linewidth=axiswidth*2,color='tab:blue',label=r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$')
    
    locals()[f'ax{count+1}'].plot(sel_CO2_dt,z_3D/abl_sel,linewidth=axiswidth*2,color='darkred',linestyle='--')
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle='--')
    
    
    locals()[f'ax{count+1}'].axhline(abl_sel/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='-',label='z$_i$')
    
    locals()[f'ax{count+1}'].fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1,label='cloud layer')
    
    locals()[f'ax{count+1}'].axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')
   
    #Detailing
    locals()[f'ax{count+1}'].set_ylim(0,3)
    
    locals()[f'ax{count+1}'].tick_params(axis='x')
    locals()[f'ax{count+1}'].tick_params(axis='y')
    locals()[f'ax{count+1}'].spines['right'].set_visible(False)
    locals()[f'ax{count+1}'].spines['top'].set_visible(False)
    
    locals()[f'ax{count+1}'].yaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].xaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].tick_params(which='major')
    locals()[f'ax{count+1}'].tick_params(which='minor')
    
    locals()[f'ax{count+1}'].set_title(f'{time} LT',y=1.02)
    locals()[f'ax{count+1}'].grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)

    
    count = count + 1

for time in times:
    
    #1D
    idx = find_closest(np.array(x_DALES_1D),time)
    
    #Clouds
    sel_CO2_dt_1D = CO2_dt_1D[idx]*3600
    sel_wCO2_dz_1D = wCO2_t_dz_1D[idx]*3600
    
    #No clouds
    sel_CO2_dt_no_clouds_1D = CO2_dt_1D_no_clouds[idx]*3600
    sel_wCO2_dz_no_clouds_1D = wCO2_t_dz_1D_no_clouds[idx]*3600
    
    #3D
    idx = find_closest(np.array(x_DALES_3D),time)
   
    #Clouds
    sel_CO2_dt = CO2_dt_mean_30min_ds.values[idx,:]*3600
    sel_wCO2_dz = wCO2_dz_arr_30min_ds.values[idx,:]*3600
    
    #No clouds
    sel_CO2_dt_no_clouds = CO2_dt_mean_no_clouds_30min_ds.values[idx,:]*3600
    sel_wCO2_dz_no_clouds = wCO2_dz_arr_no_clouds_30min_ds.values[idx,:]*3600
    
    #horizontal lines and cloud layer
    abl_sel = abl_1D_ds.values[idx]
    cb_sel = tmser_cloud_base_ds.values[idx]
    ct_sel = tmser_cloud_top_max_ds.values[idx]
    
    abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]
    
    #Visualise
    locals()[f'ax{count+1}'].plot(sel_CO2_dt_no_clouds_1D,z_1D/abl_sel,linewidth=axiswidth*2,color='darkred',label=r'$\frac{\partial \overline{CO_2}}{\partial t}$')
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz_no_clouds_1D,z_1D/abl_sel,linewidth=axiswidth*2,color='tab:blue',label=r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$')
    
    locals()[f'ax{count+1}'].plot(sel_CO2_dt_no_clouds,z_3D/abl_sel,linewidth=axiswidth*2,color='darkred',linestyle='--')
    locals()[f'ax{count+1}'].plot(sel_wCO2_dz_no_clouds,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle='--')
            
    locals()[f'ax{count+1}'].axhline(abl_sel_no_clouds/abl_sel, color='tab:purple',linewidth=axiswidth*2,linestyle='-')
    
    locals()[f'ax{count+1}'].fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1,label='cloud layer')
    
    locals()[f'ax{count+1}'].axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')
    
    #Detailing
    locals()[f'ax{count+1}'].set_ylim(0,3)
    
    locals()[f'ax{count+1}'].tick_params(axis='x')
    locals()[f'ax{count+1}'].tick_params(axis='y')
    locals()[f'ax{count+1}'].spines['right'].set_visible(False)
    locals()[f'ax{count+1}'].spines['top'].set_visible(False)
    
    locals()[f'ax{count+1}'].yaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].xaxis.set_minor_locator(AutoMinorLocator())
    locals()[f'ax{count+1}'].tick_params(which='major')
    locals()[f'ax{count+1}'].tick_params(which='minor')
    locals()[f'ax{count+1}'].grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)
    
    count = count + 1


ax1.plot([200]*len(z_3D),z_3D,linewidth=axiswidth*2,color='k',linestyle='-',label='1D')
ax1.plot([200]*len(z_3D),z_3D,linewidth=axiswidth*2,color='k',linestyle='--',label='3D')


ax1.set_ylim(0,2)
ax1.set_xlim(-16,16)
ax1.set_xticks(np.arange(-16,17,8),np.arange(-16,17,8))
ax2.set_xlim(-5,5)
ax2.set_xticks(np.arange(-5,6,2.5),np.arange(-5,6,2.5))
ax3.set_xlim(-1,1)
ax3.set_xticks(np.arange(-1,1.1,0.5),np.arange(-1,1.1,0.5))

ax4.set_ylim(0,2)
ax4.set_xlim(-16,16)
ax4.set_xticks(np.arange(-16,17,8),np.arange(-16,17,8))
ax5.set_xlim(-5,5)
ax5.set_xticks(np.arange(-5,6,2.5),np.arange(-5,6,2.5))
ax6.set_xlim(-1,1)
ax6.set_xticks(np.arange(-1,1.1,0.5),np.arange(-1,1.1,0.5))

ax1.set_ylabel('z/z$_i$')
ax4.set_ylabel('z/z$_i$')

ax5.set_xlabel('(ppm h$^{-1}$)')

legend_maker(ax1,1,6,0.25,1.3)

#Ax7

#1D
ax7.plot(x_DALES_3D,CO2_dt_abl_ds*3600,linewidth=axiswidth*2,color='darkred',label=r'$\frac{\partial \overline{CO_2}}{\partial t}$')
ax7.plot(x_DALES_3D,np.array(wCO2_s_ds)/abl_1D_ds.values*3600,linewidth=axiswidth*2,color='darkgreen',label=r'$\frac{\overline{\left(w^{\prime}CO_2^{\prime}\right)}_{s}}{z_i}$')
ax7.plot(x_DALES_3D,wCO2_abl_ds/abl_1D_ds.values*-3600,linewidth=axiswidth*2,color='tab:blue',label=r'$- \frac{\overline{\left(w^{\prime}CO_2^{\prime}\right)}_{e}}{z_i}$')

#3D
ax7.plot(x_DALES_3D,CO2_dt_abl,linewidth=axiswidth*2,color='darkred',linestyle='--')
ax7.plot(x_DALES_3D,wCO2_abl,linewidth=axiswidth*2,color='tab:blue',linestyle='--')
ax7.plot(x_DALES_3D,np.array(wCO2_s)/abl_1D_ds.values*3600,linewidth=axiswidth*2,color='darkgreen',linestyle='--')

#Regimes
ax7.axhline(0,color='gray',linewidth=axiswidth*2,linestyle='--')

ax7.fill_betweenx(np.arange(-7,2,1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax7.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 1.1,r'$\textbf{I}$', ha='center', va='bottom', color='k')

ax7.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),-7,2,color='k',alpha=0.1)

ax7.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 1.1, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax7.text(((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total], 1.1, r'$\textbf{III}$', ha='center', va='bottom', color='k')

ax7.set_yticks(np.arange(-7,2,1),np.arange(-7,2,1))
ax7.set_ylim(-7,1)

ax7.set_xticks(np.arange(6,20,2),np.arange(6,20,2))
ax7.set_xlim(7.5,18)

#Detailing
ax7.set_ylabel('(ppm h$^{-1}$)')
ax7.set_xlabel('LT (hour)')

ax7.tick_params(axis='x')
ax7.tick_params(axis='y')
ax7.spines['right'].set_visible(False)
ax7.spines['top'].set_visible(False)

ax7.yaxis.set_minor_locator(AutoMinorLocator())
ax7.xaxis.set_minor_locator(AutoMinorLocator())
ax7.tick_params(which='major')
ax7.tick_params(which='minor')

ax7.plot([np.nan],[np.nan],linewidth=axiswidth*2,color='k',label='1D')
ax7.plot([np.nan],[np.nan],linewidth=axiswidth*2,color='k',linestyle='--',label='3D')

ax7.legend(loc='lower left', bbox_to_anchor= (0.15, -0.6), ncol=5,
                 borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size'])

ax7.grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)

ax1.text(-0.1,1.1, '(a)',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax2.text(-0.1,1.1, '(b)',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax3.text(-0.1,1.1, '(c)',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax7.text(-0.015,1.1, '(d)',transform=ax7.transAxes,fontsize=plt.rcParams['font.size']*1.2)


plt.savefig(f'{outdir}/Figure_S4.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_S4.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_S4.pdf',bbox_inches='tight')
plt.show()
plt.close()

#%%
#====================================
#  Supplements - Sensitivity
#====================================

fig, ax = plt.subplots(2,2,dpi=72, figsize=(8.27*1.1, 11.69/1.4))

ax1, ax2, ax3, ax4 = ax.flatten()

#Set position
ax1.set_position([0.1, 0.7, 0.35, 0.25]) #left, bottom, width, height
ax2.set_position([0.60, 0.7, 0.35, 0.25])
ax3.set_position([0.1, -0.02, 0.35, 0.5])
ax4.set_position([0.60, -0.02, 0.35, 0.5])

#--------------------------------------------
#Cloud cover
ac_condition_1 = cfrac_condition_1_30min
ac_condition_2 = cfrac_condition_2_30min
ac_condition_3 = cfrac_condition_3_30min

ax1.plot(x_DALES_3D,ac_condition_1,color='darkred',linewidth=axiswidth*2.5,label='$a_c$')
ax1.plot(x_DALES_3D,ac_condition_2,color='tab:blue',linewidth=axiswidth*2.5,label='$a_{c,w}$')
ax1.plot(x_DALES_3D,ac_condition_3,color='tab:brown',linewidth=axiswidth*2.5,label='$a_{cc}$')

#detailing
ax1.set_ylim(0,0.3)
ax1.set_xlim(6,18)
ax1.set_xticks(np.arange(6,20,2),np.arange(6,20,2))

ax1.set_ylabel('Cloud Cover (-)')
ax1.set_xlabel('LT (hour)')

ax1.tick_params(axis='x')
ax1.tick_params(axis='y')
ax1.spines['right'].set_visible(False)
ax1.spines['top'].set_visible(False)

ax1.yaxis.set_minor_locator(AutoMinorLocator())
ax1.xaxis.set_minor_locator(AutoMinorLocator())
ax1.tick_params(which='major')
ax1.tick_params(which='minor')


ax1.legend(loc='lower left', bbox_to_anchor= (0.45, 1.4), ncol=4,
                 borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size']*1.15)

#--------------------------------------------
#Mass Flux

idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_inclouds_condition_1_30min[i,int(idx[i])])

values_at_ABL = np.array(values_at_ABL)

ax2.plot(x_DALES_3D,values_at_ABL*-3600/abl_1D_ds.values,color='darkred',linewidth=axiswidth*2.5)

idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_inclouds_condition_2_30min[i,int(idx[i])])

values_at_ABL = np.array(values_at_ABL)

ax2.plot(x_DALES_3D,values_at_ABL*-3600/abl_1D_ds.values,color='tab:blue',linewidth=axiswidth*2.5)


idx = idx_1D_ds.values
values_at_ABL = []
for i in range(len(idx)):
    if np.isnan(idx[i]):
        values_at_ABL.append(float('nan'))
    else:
        values_at_ABL.append(wCO2_arr_inclouds_condition_3_30min[i,int(idx[i])])

values_at_ABL = np.array(values_at_ABL)

ax2.plot(x_DALES_3D,values_at_ABL*-3600/abl_1D_ds.values,color='tab:brown',linewidth=axiswidth*2.5)

ax2.axhline(0,color='gray',linewidth=axiswidth*2,linestyle='--')

#shading

ax2.fill_betweenx(np.arange(-2,0.1,0.1),x_interpolated[index_0],x_interpolated[index_1_shading],color='k',alpha=0.1)
ax2.text(((x_interpolated[index_1_shading]-x_interpolated[index_0])/2)+x_interpolated[index_0], 0.2,r'$\textbf{I}$', ha='center', va='bottom', color='k')
ax2.fill_between(np.linspace(x_interpolated[index_2_total],x_DALES_3D[index_3],1000),-2,0.1,color='k',alpha=0.1)
ax2.text(((x_interpolated[index_2_total]-x_interpolated[index_1_shading])/2)+x_interpolated[index_1_shading], 0.2, r'$\textbf{II}$', ha='center', va='bottom', color='k')
ax2.text((((x_DALES_3D[index_3]-x_interpolated[index_2_total])/2)+x_interpolated[index_2_total])-0.5, 0.2, r'$\textbf{III}$', ha='center', va='bottom', color='k')


#detailing
ax2.set_ylim(-2,0.1)
ax2.set_xticks(np.arange(8,16,1),np.arange(8,16,1))
ax2.set_xlim(8,15)

ax2.set_ylabel('M (ppm h$^{-1}$)')
ax2.set_xlabel('LT (hour)')

ax2.tick_params(axis='x')
ax2.tick_params(axis='y')
ax2.spines['right'].set_visible(False)
ax2.spines['top'].set_visible(False)

ax2.yaxis.set_minor_locator(AutoMinorLocator())
ax2.xaxis.set_minor_locator(AutoMinorLocator())
ax2.tick_params(which='major')
ax2.tick_params(which='minor')

#--------------------------------------------
#Vertical flux
times = [9,12,15]

time = 12
    
idx = find_closest(np.array(x_DALES_3D),time)

#inclouds
sel_wCO2_dz_inclouds_1 = wCO2_dz_arr_inclouds_condition_1_30min_ds.values[idx,:]*3600
sel_wCO2_dz_inclouds_2 = wCO2_dz_arr_inclouds_condition_2_30min_ds.values[idx,:]*3600
sel_wCO2_dz_inclouds_3 = wCO2_dz_arr_inclouds_condition_3_30min_ds.values[idx,:]*3600

#environment
sel_wCO2_dz_environment_1 = wCO2_dz_arr_environment_condition_1_30min_ds.values[idx,:]*3600
sel_wCO2_dz_environment_2 = wCO2_dz_arr_environment_condition_2_30min_ds.values[idx,:]*3600
sel_wCO2_dz_environment_3 = wCO2_dz_arr_environment_condition_3_30min_ds.values[idx,:]*3600

#Cloud
sel_wCO2_dz = wCO2_dz_arr_30min_ds.values[idx,:]*3600

#horizontal lines and cloud layer
abl_sel = abl_1D_ds.values[idx]
cb_sel = tmser_cloud_base_ds.values[idx]
ct_sel = tmser_cloud_top_max_ds.values[idx]

abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]

#Vis
ax3.plot(sel_wCO2_dz_inclouds_1,z_3D/abl_sel,linewidth=axiswidth*2.5,color='darkred',linestyle='-.')
ax3.plot(sel_wCO2_dz_environment_1,z_3D/abl_sel,linewidth=axiswidth*2,color='darkred',linestyle=':')

ax3.plot(sel_wCO2_dz_inclouds_2,z_3D/abl_sel,linewidth=axiswidth*2.5,color='tab:blue',linestyle='-.')
ax3.plot(sel_wCO2_dz_environment_2,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle=':')

ax3.plot(sel_wCO2_dz_inclouds_3,z_3D/abl_sel,linewidth=axiswidth*2.5,color='tab:brown',linestyle='-.')
ax3.plot(sel_wCO2_dz_environment_3,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:brown',linestyle=':')

ax3.plot(sel_wCO2_dz,z_3D/abl_sel,linewidth=axiswidth*2.5,color='gray',linestyle='-',alpha=0.5)

ax3.axhline(abl_sel/abl_sel, color='tab:purple',linewidth=axiswidth*2.5,linestyle='-')

ax3.fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1)

ax3.axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')
    
#Detailing
ax3.set_ylim(0,3)

ax3.tick_params(axis='x')
ax3.tick_params(axis='y')
ax3.spines['right'].set_visible(False)
ax3.spines['top'].set_visible(False)

ax3.yaxis.set_minor_locator(AutoMinorLocator())
ax3.xaxis.set_minor_locator(AutoMinorLocator())
ax3.tick_params(which='major')
ax3.tick_params(which='minor')
    
ax3.set_title(f'{time} LT',y=1.02)

ax3.set_xlabel(r'$\frac{\partial \overline{w^{\prime}CO_2^{\prime}}}{\partial z}$ (ppm h$^{-1}$)',fontsize=plt.rcParams['font.size']*1.1)
ax3.set_ylabel('z/z$_i$')
ax3.set_ylim(0,3)
ax3.set_xlim(-3,3)
ax3.set_xticks(np.arange(-3,3.5,1),np.arange(-3,3.5,1).astype(int))

ax3.axhline(-1,color='gray',linewidth=axiswidth*2.5,linestyle='-',alpha=0.5,label='CLOUDS')
ax3.axhline(-1,color='k',linewidth=axiswidth*2.5,linestyle='-.',label='in-cloud')
ax3.axhline(-1,color='k',linewidth=axiswidth*2.5,linestyle=':',label='environment')
ax3.fill_between([-8,-7],0,0.3,color='tab:gray',alpha=0.1,label='cloud layer')
ax3.axhline(-1,color='tab:purple',linewidth=axiswidth*2.5,linestyle='-',label='$z_i$')

ax3.legend(loc='lower left', bbox_to_anchor= (0.3, -0.55), ncol=3,
                 borderaxespad=0, frameon=False,fontsize=plt.rcParams['font.size']*1.15)

#--------------------------------------------
#Correlation

#Clouds
idx = find_closest(np.array(x_DALES_3D),time)
sel_corr = corr_arr_30min_ds.values[idx,:]

#incloud
sel_corr_inclouds_1 = corr_arr_inclouds_condition_1_30min_ds.values[idx,:]
sel_corr_inclouds_2 = corr_arr_inclouds_condition_2_30min_ds.values[idx,:]
sel_corr_inclouds_3 = corr_arr_inclouds_condition_3_30min_ds.values[idx,:]

#environment
sel_corr_environment_1 = corr_arr_environment_condition_1_30min_ds.values[idx,:]
sel_corr_environment_2 = corr_arr_environment_condition_2_30min_ds.values[idx,:]
sel_corr_environment_3 = corr_arr_environment_condition_3_30min_ds.values[idx,:]


#horizontal lines and cloud layer
abl_sel = abl_1D_ds.values[idx]
cb_sel = tmser_cloud_base_ds.values[idx]
ct_sel = tmser_cloud_top_max_ds.values[idx]

abl_sel_no_clouds = abl_1D_ds_no_clouds.values[idx]

#Vis
ax4.plot(sel_corr_inclouds_1,z_3D/abl_sel,linewidth=axiswidth*2.5,color='darkred',linestyle='-.')
ax4.plot(sel_corr_environment_1,z_3D/abl_sel,linewidth=axiswidth*2,color='darkred',linestyle=':')

ax4.plot(sel_corr_inclouds_2,z_3D/abl_sel,linewidth=axiswidth*2.5,color='tab:blue',linestyle='-.')
ax4.plot(sel_corr_environment_2,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:blue',linestyle=':')

ax4.plot(sel_corr_inclouds_3,z_3D/abl_sel,linewidth=axiswidth*2.5,color='tab:brown',linestyle='-.')
ax4.plot(sel_corr_environment_3,z_3D/abl_sel,linewidth=axiswidth*2,color='tab:brown',linestyle=':')

ax4.plot(sel_corr,z_3D/abl_sel,linewidth=axiswidth*2.5,color='gray',linestyle='-',label='corr',alpha=0.5)

ax4.axhline(abl_sel/abl_sel, color='tab:purple',linewidth=axiswidth*2.5,linestyle='-',label='z$_i$')

ax4.fill_between(np.linspace(-30,30,249),abl_sel/abl_sel,ct_sel/abl_sel,color='k',alpha=0.1,label='cloud layer')

ax4.axvline(0,color='k',linewidth=axiswidth*2,linestyle='--')

ax4.axhline(200/abl_sel, color='gray',linewidth=axiswidth*1.5,linestyle='--')
ax4.axhline(1100/abl_sel, color='gray',linewidth=axiswidth*1.5,linestyle='--')
ax4.axhline(3000/abl_sel, color='gray',linewidth=axiswidth*1.5,linestyle='--')

xmax = 1.3

if count == 0: 
    minner = 10
else:
    minner = 30

ax4.text(xmax,(200-minner)/abl_sel,'z$_L$',color='k',fontsize=plt.rcParams['font.size'])

if count !=0:
    ax4.text(xmax,(1100-minner)/abl_sel,'z$_M$',color='k',fontsize=plt.rcParams['font.size'])
    ax4.text(xmax,(3000-minner)/abl_sel,'z$_H$',color='k',fontsize=plt.rcParams['font.size'])

#Detailing
ax4.set_ylim(0,3)
ax4.set_xlim(-1.2,1.2)
ax4.set_xticks(np.arange(-1,1.1,0.5),np.round(np.arange(-1,1.1,0.5),2),rotation=45)

ax4.tick_params(axis='x')
ax4.tick_params(axis='y')
ax4.spines['right'].set_visible(False)
ax4.spines['top'].set_visible(False)

ax4.yaxis.set_minor_locator(AutoMinorLocator())
ax4.xaxis.set_minor_locator(AutoMinorLocator())
ax4.tick_params(which='major')
ax4.tick_params(which='minor')

ax4.set_title(f'{time} LT',y=1.02)

ax4.set_ylabel('z/z$_i$')
ax4.set_xlabel('r (-)')

#Add labels to the plots
ax1.text(-0.1,1.2,'(a)',transform=ax1.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax2.text(-0.1,1.2,'(b)',transform=ax2.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax3.text(-0.1,1.1,'(c)',transform=ax3.transAxes,fontsize=plt.rcParams['font.size']*1.2)
ax4.text(-0.1,1.1,'(d)',transform=ax4.transAxes,fontsize=plt.rcParams['font.size']*1.2)

ax1.grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)
ax2.grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)
ax3.grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)
ax4.grid(color='tab:gray',linestyle=':',linewidth=axiswidth*2,alpha=0.3)


#Save
plt.savefig(f'{outdir}/Figure_S5.jpg',bbox_inches='tight',dpi=500)
plt.savefig(f'{outdir}/Figure_S5.svg',bbox_inches='tight')
plt.savefig(f'{outdir}/Figure_S5.pdf',bbox_inches='tight')
plt.show()
plt.close()

#%%

#Export items of the figures and available data as one dataset

#1D budget clouds

ds_1D_clouds = xr.Dataset()

ds_1D_clouds['time'] = x_DALES_1D
ds_1D_clouds['z'] = z_1D

ds_1D_clouds['CO2_dt'] = xr.DataArray(CO2_dt_1D,coords=[x_DALES_1D,z_1D],dims=['time','z'])
ds_1D_clouds['wCO2_t_dz'] = xr.DataArray(wCO2_t_dz_1D,coords=[x_DALES_1D,z_1D],dims=['time','z'])

ds_1D_clouds['wCO2_s'] = xr.DataArray(wCO2_s_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['gCO2_s'] = xr.DataArray(gCO2_s_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['respCO2_s'] = xr.DataArray(respCO2_s_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['ra_s'] = xr.DataArray(ra_s_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['rsCO2_s'] = xr.DataArray(rsCO2_s_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['An_s'] = xr.DataArray(An_s_ds.values,coords=[x_DALES_1D],dims=['time'])

ds_1D_clouds['abl'] = xr.DataArray(abl_1D_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['wCO2_abl'] = xr.DataArray(wCO2_abl_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['CO2_dt_abl'] = xr.DataArray(CO2_dt_abl_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['cb_1D'] = xr.DataArray(cb_1D_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['ct_1D'] = xr.DataArray(ct_1D_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['CO2_mixed'] = xr.DataArray(CO2_mixed_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['CO2_ft'] = xr.DataArray(CO2_ft_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['CO2_lowest'] = xr.DataArray(CO2_lowest_ds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['Ci'] = xr.DataArray(Ci,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['delta_c_plant'] = xr.DataArray(delta_c_plant,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['delta_c_atm'] = xr.DataArray(delta_c_atm,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['ratio_plant'] = xr.DataArray(ratio_plant,coords=[x_DALES_1D],dims=['time'])
ds_1D_clouds['ratio_atm'] = xr.DataArray(ratio_atm,coords=[x_DALES_1D],dims=['time'])

#Save
ds_1D_clouds.to_netcdf(f'{outdir}/ds_1D_clouds.nc')


#1D budget no clouds

ds_1D_no_clouds = xr.Dataset()

ds_1D_no_clouds['time'] = x_DALES_1D
ds_1D_no_clouds['z'] = z_1D

ds_1D_no_clouds['CO2_dt'] = xr.DataArray(CO2_dt_1D_no_clouds,coords=[x_DALES_1D,z_1D],dims=['time','z'])
ds_1D_no_clouds['wCO2_t_dz'] = xr.DataArray(wCO2_t_dz_1D_no_clouds,coords=[x_DALES_1D,z_1D],dims=['time','z'])

ds_1D_no_clouds['wCO2_s'] = xr.DataArray(wCO2_s_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['gCO2_s'] = xr.DataArray(gCO2_s_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['respCO2_s'] = xr.DataArray(respCO2_s_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['ra_s'] = xr.DataArray(ra_s_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['rsCO2_s'] = xr.DataArray(rsCO2_s_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['An_s'] = xr.DataArray(An_s_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])

ds_1D_no_clouds['abl'] = xr.DataArray(abl_1D_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['wCO2_abl'] = xr.DataArray(wCO2_abl_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['CO2_dt_abl'] = xr.DataArray(CO2_dt_abl_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])

ds_1D_no_clouds['CO2_mixed'] = xr.DataArray(CO2_mixed_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['CO2_ft'] = xr.DataArray(CO2_ft_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['CO2_lowest'] = xr.DataArray(CO2_lowest_ds_no_clouds.values,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['Ci'] = xr.DataArray(Ci_no_clouds,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['delta_c_plant'] = xr.DataArray(delta_c_plant_no_clouds,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['delta_c_atm'] = xr.DataArray(delta_c_atm_no_clouds,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['ratio_plant'] = xr.DataArray(ratio_plant_no_clouds,coords=[x_DALES_1D],dims=['time'])
ds_1D_no_clouds['ratio_atm'] = xr.DataArray(ratio_atm_no_clouds,coords=[x_DALES_1D],dims=['time'])

#Save
ds_1D_no_clouds.to_netcdf(f'{outdir}/ds_1D_no_clouds.nc')

#%%
#3D budget clouds

ds_3D_raw = xr.Dataset()

ds_3D_raw['time'] = field_3D.time.values
ds_3D_raw['z'] = z_3D

ds_3D_raw['wCO2_dz'] = xr.DataArray(wCO2_dz_arr,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_dz_inclouds_condition_1'] = xr.DataArray(wCO2_dz_arr_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_dz_inclouds_condition_2'] = xr.DataArray(wCO2_dz_arr_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_dz_inclouds_condition_3'] = xr.DataArray(wCO2_dz_arr_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wCO2_dz_environment_condition_1'] = xr.DataArray(wCO2_dz_arr_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_dz_environment_condition_2'] = xr.DataArray(wCO2_dz_arr_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_dz_environment_condition_3'] = xr.DataArray(wCO2_dz_arr_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wCO2_dz_no_clouds'] = xr.DataArray(wCO2_dz_arr_no_clouds,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wCO2'] = xr.DataArray(wCO2_arr,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_inclouds_condition_1'] = xr.DataArray(wCO2_arr_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_inclouds_condition_2'] = xr.DataArray(wCO2_arr_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_inclouds_condition_3'] = xr.DataArray(wCO2_arr_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wCO2_environment_condition_1'] = xr.DataArray(wCO2_arr_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_environment_condition_2'] = xr.DataArray(wCO2_arr_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wCO2_environment_condition_3'] = xr.DataArray(wCO2_arr_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wCO2_no_clouds'] = xr.DataArray(wCO2_arr_no_clouds,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['qCO2'] = xr.DataArray(qCO2_arr,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['qCO2_inclouds_condition_1'] = xr.DataArray(qCO2_arr_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['qCO2_inclouds_condition_2'] = xr.DataArray(qCO2_arr_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['qCO2_inclouds_condition_3'] = xr.DataArray(qCO2_arr_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['qCO2_environment_condition_1'] = xr.DataArray(qCO2_arr_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['qCO2_environment_condition_2'] = xr.DataArray(qCO2_arr_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['qCO2_environment_condition_3'] = xr.DataArray(qCO2_arr_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['qCO2_no_clouds'] = xr.DataArray(qCO2_arr_no_clouds,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wq'] = xr.DataArray(wq_arr,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wq_inclouds_condition_1'] = xr.DataArray(wq_arr_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wq_inclouds_condition_2'] = xr.DataArray(wq_arr_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wq_inclouds_condition_3'] = xr.DataArray(wq_arr_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wq_environment_condition_1'] = xr.DataArray(wq_arr_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wq_environment_condition_2'] = xr.DataArray(wq_arr_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['wq_environment_condition_3'] = xr.DataArray(wq_arr_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['wq_no_clouds'] = xr.DataArray(wq_arr_no_clouds,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['corr'] = xr.DataArray(corr_arr,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['corr_inclouds_condition_1'] = xr.DataArray(corr_arr_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['corr_inclouds_condition_2'] = xr.DataArray(corr_arr_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['corr_inclouds_condition_3'] = xr.DataArray(corr_arr_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['corr_environment_condition_1'] = xr.DataArray(corr_arr_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['corr_environment_condition_2'] = xr.DataArray(corr_arr_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['corr_environment_condition_3'] = xr.DataArray(corr_arr_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['corr_no_clouds'] = xr.DataArray(corr_arr_no_clouds,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['cfrac_inclouds_condition_1'] = xr.DataArray(cfrac_condition_1,coords=[field_3D.time.values],dims=['time'])
ds_3D_raw['cfrac_condition_2'] = xr.DataArray(cfrac_condition_2,coords=[field_3D.time.values],dims=['time'])
ds_3D_raw['cfrac_condition_3'] = xr.DataArray(cfrac_condition_3,coords=[field_3D.time.values],dims=['time'])

ds_3D_raw['grad_CO2'] = xr.DataArray(grad_CO2_arr[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['grad_CO2_inclouds_condition_1'] = xr.DataArray(grad_CO2_arr_inclouds_condition_1[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['grad_CO2_inclouds_condition_2'] = xr.DataArray(grad_CO2_arr_inclouds_condition_2[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['grad_CO2_inclouds_condition_3'] = xr.DataArray(grad_CO2_arr_inclouds_condition_3[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['grad_CO2_environment_condition_1'] = xr.DataArray(grad_CO2_arr_environment_condition_1[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['grad_CO2_environment_condition_2'] = xr.DataArray(grad_CO2_arr_environment_condition_2[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['grad_CO2_environment_condition_3'] = xr.DataArray(grad_CO2_arr_environment_condition_3[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['grad_CO2_no_clouds'] = xr.DataArray(grad_CO2_arr_no_clouds[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['profile_theta_v'] = xr.DataArray(profile_theta_v[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['profile_theta_v_inclouds_condition_1'] = xr.DataArray(profile_theta_v_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['profile_theta_v_inclouds_condition_2'] = xr.DataArray(profile_theta_v_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['profile_theta_v_inclouds_condition_3'] = xr.DataArray(profile_theta_v_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['profile_theta_v_environment_condition_1'] = xr.DataArray(profile_theta_v_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['profile_theta_v_environment_condition_2'] = xr.DataArray(profile_theta_v_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['profile_theta_v_environment_condition_3'] = xr.DataArray(profile_theta_v_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['profile_theta_v_no_clouds'] = xr.DataArray(profile_theta_v_no_clouds[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['buoy'] = xr.DataArray(buoy_arr[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['buoy_inclouds_condition_1'] = xr.DataArray(np.array(buoy_arr_inclouds_condition_1)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['buoy_inclouds_condition_2'] = xr.DataArray(np.array(buoy_arr_inclouds_condition_2)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['buoy_inclouds_condition_3'] = xr.DataArray(np.array(buoy_arr_inclouds_condition_3)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['buoy_environment_condition_1'] = xr.DataArray(np.array(buoy_arr_environment_condition_1)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['buoy_environment_condition_2'] = xr.DataArray(np.array(buoy_arr_environment_condition_2)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['buoy_environment_condition_3'] = xr.DataArray(np.array(buoy_arr_environment_condition_3)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['buoy_no_clouds'] = xr.DataArray(np.array(buoy_arr_no_clouds)[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['CO2'] = xr.DataArray(CO2_arr[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['CO2_inclouds_condition_1'] = xr.DataArray(CO2_arr_inclouds_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['CO2_inclouds_condition_2'] = xr.DataArray(CO2_arr_inclouds_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['CO2_inclouds_condition_3'] = xr.DataArray(CO2_arr_inclouds_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['CO2_environment_condition_1'] = xr.DataArray(CO2_arr_environment_condition_1,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['CO2_environment_condition_2'] = xr.DataArray(CO2_arr_environment_condition_2,coords=[field_3D.time.values,z_3D],dims=['time','z'])
ds_3D_raw['CO2_environment_condition_3'] = xr.DataArray(CO2_arr_environment_condition_3,coords=[field_3D.time.values,z_3D],dims=['time','z'])

ds_3D_raw['CO2_no_clouds'] = xr.DataArray(CO2_arr_no_clouds[:,:-1],coords=[field_3D.time.values,z_3D],dims=['time','z'])

#save
ds_3D_raw.to_netcdf(f'{outdir}/ds_3D_raw.nc')

#%%

#3D budget clouds - 30min

ds_3D_clouds = xr.Dataset()

ds_3D_clouds['time'] = x_DALES_3D
ds_3D_clouds['z'] = z_3D

ds_3D_clouds['wCO2_dz'] = xr.DataArray(wCO2_dz_arr_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_dz_inclouds_condition_1'] = xr.DataArray(wCO2_dz_arr_inclouds_condition_1_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_dz_inclouds_condition_2'] = xr.DataArray(wCO2_dz_arr_inclouds_condition_2_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_dz_inclouds_condition_3'] = xr.DataArray(wCO2_dz_arr_inclouds_condition_3_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['wCO2_dz_environment_condition_1'] = xr.DataArray(wCO2_dz_arr_environment_condition_1_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_dz_environment_condition_2'] = xr.DataArray(wCO2_dz_arr_environment_condition_2_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_dz_environment_condition_3'] = xr.DataArray(wCO2_dz_arr_environment_condition_3_30min_ds.values,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['CO2_dt'] = xr.DataArray(CO2_dt_mean_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_dt_inclouds_condition_1'] = xr.DataArray(CO2_dt_mean_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_dt_inclouds_condition_2'] = xr.DataArray(CO2_dt_mean_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_dt_inclouds_condition_3'] = xr.DataArray(CO2_dt_mean_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['CO2_dt_environment_condition_1'] = xr.DataArray(CO2_dt_mean_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_dt_environment_condition_2'] = xr.DataArray(CO2_dt_mean_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_dt_environment_condition_3'] = xr.DataArray(CO2_dt_mean_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['wCO2'] = xr.DataArray(wCO2_arr_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_inclouds_condition_1'] = xr.DataArray(wCO2_arr_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_inclouds_condition_2'] = xr.DataArray(wCO2_arr_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_inclouds_condition_3'] = xr.DataArray(wCO2_arr_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['wCO2_environment_condition_1'] = xr.DataArray(wCO2_arr_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_environment_condition_2'] = xr.DataArray(wCO2_arr_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wCO2_environment_condition_3'] = xr.DataArray(wCO2_arr_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['qCO2'] = xr.DataArray(qCO2_arr_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['qCO2_inclouds_condition_1'] = xr.DataArray(qCO2_arr_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['qCO2_inclouds_condition_2'] = xr.DataArray(qCO2_arr_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['qCO2_inclouds_condition_3'] = xr.DataArray(qCO2_arr_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['qCO2_environment_condition_1'] = xr.DataArray(qCO2_arr_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['qCO2_environment_condition_2'] = xr.DataArray(qCO2_arr_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['qCO2_environment_condition_3'] = xr.DataArray(qCO2_arr_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['wq'] = xr.DataArray(wq_arr_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wq_inclouds_condition_1'] = xr.DataArray(wq_arr_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wq_inclouds_condition_2'] = xr.DataArray(wq_arr_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wq_inclouds_condition_3'] = xr.DataArray(wq_arr_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['wq_environment_condition_1'] = xr.DataArray(wq_arr_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wq_environment_condition_2'] = xr.DataArray(wq_arr_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['wq_environment_condition_3'] = xr.DataArray(wq_arr_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['corr'] = xr.DataArray(corr_arr_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['corr_inclouds_condition_1'] = xr.DataArray(corr_arr_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['corr_inclouds_condition_2'] = xr.DataArray(corr_arr_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['corr_inclouds_condition_3'] = xr.DataArray(corr_arr_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['corr_environment_condition_1'] = xr.DataArray(corr_arr_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['corr_environment_condition_2'] = xr.DataArray(corr_arr_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['corr_environment_condition_3'] = xr.DataArray(corr_arr_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['grad_CO2'] = xr.DataArray(grad_CO2_arr_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['grad_CO2_inclouds_condition_1'] = xr.DataArray(grad_CO2_arr_inclouds_condition_1_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['grad_CO2_inclouds_condition_2'] = xr.DataArray(grad_CO2_arr_inclouds_condition_2_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['grad_CO2_inclouds_condition_3'] = xr.DataArray(grad_CO2_arr_inclouds_condition_3_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['grad_CO2_environment_condition_1'] = xr.DataArray(grad_CO2_arr_environment_condition_1_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['grad_CO2_environment_condition_2'] = xr.DataArray(grad_CO2_arr_environment_condition_2_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['grad_CO2_environment_condition_3'] = xr.DataArray(grad_CO2_arr_environment_condition_3_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['profile_theta_v'] = xr.DataArray(profile_theta_v_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['profile_theta_v_inclouds_condition_1'] = xr.DataArray(profile_theta_v_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['profile_theta_v_inclouds_condition_2'] = xr.DataArray(profile_theta_v_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['profile_theta_v_inclouds_condition_3'] = xr.DataArray(profile_theta_v_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['profile_theta_v_environment_condition_1'] = xr.DataArray(profile_theta_v_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['profile_theta_v_environment_condition_2'] = xr.DataArray(profile_theta_v_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['profile_theta_v_environment_condition_3'] = xr.DataArray(profile_theta_v_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['buoy'] = xr.DataArray(buoy_arr_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['buoy_inclouds_condition_1'] = xr.DataArray(np.array(buoyancy_arr_inclouds_condition_1_30min)[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['buoy_inclouds_condition_2'] = xr.DataArray(np.array(buoyancy_arr_inclouds_condition_2_30min)[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['buoy_inclouds_condition_3'] = xr.DataArray(np.array(buoyancy_arr_inclouds_condition_3_30min)[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['buoy_environment_condition_1'] = xr.DataArray(np.array(buoyancy_arr_environment_condition_1_30min)[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['buoy_environment_condition_2'] = xr.DataArray(np.array(buoyancy_arr_environment_condition_2_30min)[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['buoy_environment_condition_3'] = xr.DataArray(np.array(buoyancy_arr_environment_condition_3_30min)[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['CO2'] = xr.DataArray(CO2_arr_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_inclouds_condition_1'] = xr.DataArray(CO2_arr_inclouds_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_inclouds_condition_2'] = xr.DataArray(CO2_arr_inclouds_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_inclouds_condition_3'] = xr.DataArray(CO2_arr_inclouds_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['CO2_environment_condition_1'] = xr.DataArray(CO2_arr_environment_condition_1_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_environment_condition_2'] = xr.DataArray(CO2_arr_environment_condition_2_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_clouds['CO2_environment_condition_3'] = xr.DataArray(CO2_arr_environment_condition_3_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_clouds['cfrac_condition_1'] = xr.DataArray(cfrac_condition_1_30min,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['cfrac_condition_2'] = xr.DataArray(cfrac_condition_2_30min,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['cfrac_condition_3'] = xr.DataArray(cfrac_condition_3_30min,coords=[x_DALES_3D],dims=['time'])

ds_3D_clouds['CO2_dt_abl'] = xr.DataArray(CO2_dt_abl,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['CO2_dt_abl_condition_1'] = xr.DataArray(CO2_dt_abl_inclouds_condition_1,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['CO2_dt_abl_condition_2'] = xr.DataArray(CO2_dt_abl_inclouds_condition_2,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['CO2_dt_abl_condition_3'] = xr.DataArray(CO2_dt_abl_inclouds_condition_3,coords=[x_DALES_3D],dims=['time'])

ds_3D_clouds['CO2_dt_abl_environment_1'] = xr.DataArray(CO2_dt_abl_environment_condition_1,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['CO2_dt_abl_environment_2'] = xr.DataArray(CO2_dt_abl_environment_condition_2,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['CO2_dt_abl_environment_3'] = xr.DataArray(CO2_dt_abl_environment_condition_3,coords=[x_DALES_3D],dims=['time'])


ds_3D_clouds['wCO2_abl'] = xr.DataArray(wCO2_abl,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['wCO2_abl_condition_1'] = xr.DataArray(wCO2_abl_inclouds_condition_1,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['wCO2_abl_condition_2'] = xr.DataArray(wCO2_abl_inclouds_condition_2,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['wCO2_abl_condition_3'] = xr.DataArray(wCO2_abl_inclouds_condition_3,coords=[x_DALES_3D],dims=['time'])

ds_3D_clouds['wCO2_abl_environment_1'] = xr.DataArray(wCO2_abl_environment_condition_1,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['wCO2_abl_environment_2'] = xr.DataArray(wCO2_abl_environment_condition_2,coords=[x_DALES_3D],dims=['time'])
ds_3D_clouds['wCO2_abl_environment_3'] = xr.DataArray(wCO2_abl_environment_condition_3,coords=[x_DALES_3D],dims=['time'])

#save
ds_3D_clouds.to_netcdf(f'{outdir}/ds_3D_clouds.nc')

#%%

#3D budget no clouds - 30 min

ds_3D_no_clouds = xr.Dataset()

ds_3D_no_clouds['time'] = x_DALES_3D
ds_3D_no_clouds['z'] = z_3D

ds_3D_no_clouds['wCO2_dz'] = xr.DataArray(wCO2_dz_arr_no_clouds_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['CO2_dt'] = xr.DataArray(CO2_dt_mean_no_clouds_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['wCO2'] = xr.DataArray(wCO2_arr_no_clouds_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['qCO2'] = xr.DataArray(qCO2_arr_no_clouds_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['wq'] = xr.DataArray(wq_arr_no_clouds_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['corr'] = xr.DataArray(corr_arr_no_clouds_30min,coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['grad_CO2'] = xr.DataArray(grad_CO2_arr_no_clouds_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['profile_theta_v'] = xr.DataArray(profile_theta_v_no_clouds_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['buoy'] = xr.DataArray(buoyancy_arr_no_clouds_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])
ds_3D_no_clouds['CO2'] = xr.DataArray(CO2_arr_no_clouds_30min[:,:-1],coords=[x_DALES_3D,z_3D],dims=['time','z'])

ds_3D_no_clouds['CO2_dt_abl'] = xr.DataArray(CO2_dt_abl_no_clouds,coords=[x_DALES_3D],dims=['time'])
ds_3D_no_clouds['wCO2_abl'] = xr.DataArray(wCO2_abl_no_clouds,coords=[x_DALES_3D],dims=['time'])

#save
ds_3D_no_clouds.to_netcdf(f'{outdir}/ds_3D_no_clouds.nc')













