%% Non-spatial model for macrophyte-hydrodynamic interactions in a stream
% The model follows the manuscript by Cornacchia et al., June 2017.

% Cornacchia, L., van de Koppel, J., van der Wal, D., Wharton, G., Puijalon, S., Bouma, T.J.
% Landscapes of facilitation: how self-organized patchiness of aquatic
% macrophytes enhances diversity in streams. Submitted on 13 June 2017.

%% Equations for P1 and P2 (reduced model)
%P1 = r1*P1*((k1 - P1 - alpha_21.*P2)./k1) - m_W1.*P1.*NetSpeed;

%P2 = r2*P2*((k2 - P2 - alpha_12.*P1)./k2) - m_W2.*P2.*NetSpeed;

%% Species parameters when P2 = Pb1 (less flow tolerant species)

% Parameters                   Original value      Explanation and Units

k1           =  1        ;     % 1                 Plant carrying capacity for Pf (g DW)
r1           =  1.19     ;     % 1.19              Plant intrinsic growth rate (t-1)   
m_W1         =  9        ;     % 9                 Plant mortality constant due to hydrodynamic stress (Pf) (-)
alpha_12     =  1.42     ;     % 1.42              Competition coefficient of Pf on Pb1 (-)
k2           =  1.95     ;     % 1.95              Plant carrying capacity for Pb1 (g DW)
r2           =  0.94     ;     % 0.94              Plant intrinsic growth rate (t-1)   
m_W2         =  8        ;     % 8                 Plant mortality constant due to hydrodynamic stress (Pb1) (-)
alpha_21     =  0.6      ;     % 0.6               Competition coefficient of Pb1 on Pf (-)

%% Species parameters when P2 = Pb2 (more flow tolerant species)

k1           =  1        ;     % 1                 Plant carrying capacity for Pf (g DW)
r1           =  1.19     ;     % 1.19              Plant intrinsic growth rate (t-1)
m_W1         =  9        ;     % 9                 Plant mortality constant due to hydrodynamic stress (Pf) (-)
alpha_12     =  0.83     ;     % 0.83              Competition coefficient of Pf on Pb2 (-)
k2           =  0.5      ;     % 0.5               Plant carrying capacity for Pb2 (g DW)
r2           =  0.5      ;     % 0.5               Plant intrinsic growth rate (t-1) 
m_W2         =  3        ;     % 3                 Plant mortality constant due to hydrodynamic stress (Pb2) (-)
alpha_21     =  0.5      ;     % 0.5               Competition coefficient of Pb2 on Pf (-)

%% Plot isoclines for P1 and P2

figure(1)
% Set an incoming flow velocity (m s-1)
NetSpeed = 0;
% Set Y1 as a range of Pb values
Y1 = [0:0.1:1.5];
P1 =  -(Y1.*alpha_21*r1 - k1*r1 + NetSpeed*k1*m_W1)/r1;
plot(P1,Y1)
% Set X2 as a range of Pf values
X2 = [0:0.1:1.5];
P2 =  -(X2*alpha_12*r2 - k2*r2 + NetSpeed*k2*m_W2)/r2;
hold on
plot(X2,P2)
axis([0 1.5 0 1.5])
legend('Pf','Pb')
xlabel('Pf')
ylabel('Pb')
% Find Pf and Pb biomass at intersection point
P = InterX([P1;Y1],[X2;P2]);

%% Calculate Pf and Pb biomass without species interactions
% P1 = -(- k1.*r1 + NetSpeed.*k1.*m_W1)/r1;
% 
% P2 = -(- k2.*r2 + NetSpeed.*k2.*m_W2)/r2;

%% Analyze coexistence outcome with increasing flow stress
NetSpeed = [0:0.001:0.17];

% Conditions for stable coexistence
StableCoexistence =  -(-k1.*r1 + NetSpeed.*k1.*m_W1)./r1 < (k2.*r2 - NetSpeed.*k2.*m_W2)./(alpha_12.*r2) & -(- k2.*r2 + NetSpeed.*k2.*m_W2)./r2 < (k1.*r1 - NetSpeed.*k1.*m_W1)./(alpha_21.*r1);
% Convert logical vector to numeric
StableCoexistence = double(StableCoexistence);

% Conditions for unstable coexistence
UnstableCoexistence = -(-k1.*r1 + NetSpeed.*k1.*m_W1)/r1 > (k2.*r2 - NetSpeed.*k2.*m_W2)./(alpha_12.*r2) & -(- k2.*r2 + NetSpeed.*k2.*m_W2)./r2 > (k1.*r1 - NetSpeed.*k1.*m_W1)./(alpha_21.*r1);
% Convert logical vector to numeric
UnstableCoexistence = double(UnstableCoexistence);
UnstableCoexistence(UnstableCoexistence==1)=2;

% Conditions for dominance of P1
P1wins = -(-k1.*r1 + NetSpeed.*k1.*m_W1)./r1 > (k2.*r2 - NetSpeed.*k2.*m_W2)./(alpha_12.*r2) & -(- k2.*r2 + NetSpeed.*k2.*m_W2)./r2 < (k1.*r1 - NetSpeed.*k1.*m_W1)./(alpha_21.*r1);
% Convert logical vector to numeric
P1wins = double(P1wins);
P1wins(P1wins==1)=3;

% Conditions for dominance of P2
P2wins = -(-k1.*r1 + NetSpeed.*k1.*m_W1)./r1 < (k2.*r2 - NetSpeed.*k2.*m_W2)./(alpha_12.*r2) & -(- k2.*r2 + NetSpeed.*k2.*m_W2)./r2 > (k1.*r1 - NetSpeed.*k1.*m_W1)./(alpha_21.*r1);
% Convert logical vector to numeric
P2wins = double(P2wins);
P2wins(P2wins==1)=4;

Coexistence = StableCoexistence + UnstableCoexistence + P1wins + P2wins;

%% Plot results along the flow stress gradient
figure(2)
plot(NetSpeed,Coexistence)
