
rm(list=ls())								## clear workspace ##

# Set working directory

# Load packages
library("readxl")
library("fitdistrplus")
library("lme4")
library("emmeans")
library("lmtest")
library("dplyr")
library("multcomp")
library("DHARMa")
library("RVAideMemoire")
library("MuMIn")
library("sjPlot")
library("ggplot2")
library("vegan")
library("pals")
library("tidyverse")
library("PCAtools")
library("ggeffects")
library("ggfortify")
library("gridExtra")
library("ggvegan")

# function to compare GLMs ####
# Function by Daan Mertens to easily compare models for AIC, BIC and Log-likelyhood
Fit.stats <- function(Model){
  Model.Fit <- as.matrix(c(AIC(Model),BIC(Model),logLik(Model)))
  rownames(Model.Fit) <- c("AIC","BIC","LogLik")
  colnames(Model.Fit) <- deparse(substitute(Model))
  return(Model.Fit)
}


#### load data file ####
Data <- read_xlsx("Data/Large-scale monitoring data.xlsx", sheet = "Processed")

Data$Field <- as.factor(Data$Field)
Data$Treatment <- factor(Data$Treatment,
                         levels = c("Ref",
                                    "Strip",
                                    "Strip_VAR",
                                    "Strip_ADD",
                                    "Strip_ROT"))

Data$Round <- as.factor(Data$Round)
levels(Data$Round) <- c("Round 1", "Round 2")

Data <- Data %>% mutate(Rowtype = 
                  case_when(Row_number == 1 ~ "Outside",
                            Row_number == 2 ~ "Inside",
                            Row_number == 3 ~ "Inside",
                            Row_number == 4 ~ "Outside"),
                .after ="Row_number")
Data$Rowtype <- as.factor(Data$Rowtype)
str(Data)

# Add functional groups
AG_Phloem_feeders <- c("Aphid_other","Brevicoryne_brassicae","mummy","Myzus_persicae")
AG_Chewers <- c("Autographa_gamma","Mamestra_brassicae","Miner","Phyllotreta_atra","Phyllotreta_undulata","Pieris_brassicae","Pieris_rapae", "Plutella_xylostella","Thrips")
BG_Predators <- c("Anthicidae","Carabid_beetle","Carabid_Rove_Larvae","Centipede","Ceratopogonidae_larva","Rove_beetle","Spider","Velvet_Mite")
BG_Detrivores <- c("Dung_beetle","Earthworm","Scarab_Larvae")
BG_Herbivores <- c("Click_beetle","Click_Beetle_Larvae","Cut_Worm")

Data %>% select(AG_Phloem_feeders) %>% rowSums() -> Data$AG_Phloem_feeders
Data %>% select(AG_Chewers) %>% rowSums() -> Data$AG_Chewers
Data %>% select(BG_Predators) %>% rowSums() -> Data$BG_Predators
Data %>% select(BG_Detrivores) %>% rowSums() -> Data$BG_Detrivores
Data %>% select(BG_Herbivores) %>% rowSums() -> Data$BG_Herbivores

# Manually remove some groups
Data <- select(Data, -Ant, -Beetle_Pupa, -Cecidomyiidae_larva)
str(Data)
##

#### Subsets
Data_R1 <- Data[Data$Round == "Round 1",]
Data_R1 <- droplevels(Data_R1)

Data_R2 <- Data[Data$Round == "Round 2",]
Data_R2 <- droplevels(Data_R2)

Data_Strip <- Data[Data$Treatment == "Strip",]
Data_Strip <- droplevels(Data_Strip)

Data_Field10 <- Data[Data$Field == "10",]
Data_Field10 <- droplevels(Data_Field10)

# Color theme
col <- (c("Ref"= "#895E39",
          "Strip" = "#B6DA60",
          "Strip_VAR" = "#308188",
          "Strip_ADD" = "#19399F",
          "Strip_ROT" = "#9671bd"))
## Delia in Strip treatment (Field effect) ####

summary(Data_Strip$Total_Delia)
p1 <- ggplot(Data_Strip, aes(x=Field, y=Total_Delia, fill=Field)) + 
  geom_boxplot(outlier.shape = NA)+
  geom_jitter(width = 0.1, alpha = 0.4)+
  scale_fill_manual(values = rev(ocean.gray(6)[2:5]))+
  facet_wrap(~Round)+
  theme_bw()+
  scale_y_continuous(limits = c(0, 8), oob = scales::squish,
                     expand = expansion(mult = c(0, 0.05)))
p1

# ggsave(file = "Figures/TotalDelia_StripTreatment.pdf", p1,
#        width = 10,
#        height = 6)

Field_mod <- glm(Total_Delia ~ Field, data = Data_Strip, family = poisson())
testDispersion(Field_mod, alternative = "greater") # this model is overdispersed :-(

M4 <- glm.nb(Total_Delia ~ Field * Round, data = Data_Strip)
M3 <- glm.nb(Total_Delia ~ Field + Round, data = Data_Strip)
M2 <- glm.nb(Total_Delia ~ Field, data = Data_Strip)
M1 <- glm.nb(Total_Delia ~ Round, data = Data_Strip)
M0 <- glm.nb(Total_Delia ~ 1, data = Data_Strip)

cbind(Fit.stats(M4),Fit.stats(M3),Fit.stats(M2),Fit.stats(M1),Fit.stats(M0))

sim_fmnb <- simulateResiduals(M4)
plot(sim_fmnb)

lrtest(M4, M3) 
lrtest(M2, M0) 
lrtest(M1, M0) 
tab_model(M4)

# posthoc
posthoc <- emmeans(M4, pairwise ~ Field|Round, type = "response", adjustSigma =T, adjust = "fdr")
posthoc
cld(posthoc$emmeans, type = "response", Letters = "abcdefg", adjust = "fdr")

plot(posthoc)

 
## Delia mono vs. strip ####
# The most accurate comparison of strip vs. monoculture can be made by only comparing within field 10.

summary(Data_Field10$Total_Delia)
p1 <- ggplot(Data_Field10, aes(x=Treatment, y=Total_Delia, fill=Treatment)) + 
  geom_boxplot(outlier.shape = NA)+
  scale_fill_manual(values = col[1:2])+
  geom_jitter(width = 0.1, alpha = 0.4)+
  facet_wrap(~Round)+
  theme_bw()+
  scale_y_continuous(limits = c(0, 8), oob = scales::squish,
                     expand = expansion(mult = c(0, 0.05)))
p1

# ggsave(file = "Figures/TotalDelia_Field10.pdf", p1,
#        width = 10,
#        height = 6)

Mod <- glm(Total_Delia ~ Treatment*Round, data = Data_Field10, family = poisson())
testDispersion(Mod, alternative = "greater") # this model is overdispersed :-(
# Overdispersed -> switch to nb

M4 <- glm.nb(Total_Delia ~ Treatment * Round, data = Data_Field10)
M3 <- glm.nb(Total_Delia ~ Treatment + Round, data = Data_Field10)
M2 <- glm.nb(Total_Delia ~ Treatment, data = Data_Field10)
M1 <- glm.nb(Total_Delia ~ Round, data = Data_Field10)
M0 <- glm.nb(Total_Delia ~ 1, data = Data_Field10)

cbind(Fit.stats(M4),Fit.stats(M3),Fit.stats(M2),Fit.stats(M1),Fit.stats(M0))
# analyse full model

sim_fmnb <- simulateResiduals(M4)
plot(sim_fmnb)

lrtest(M4, M3) 
lrtest(M2, M0) 
lrtest(M1, M0) 
tab_model(M4)

# posthoc
posthoc <- emmeans(M4, pairwise ~ Treatment|Round, type = "response", adjustSigma =T, adjust = "fdr")
posthoc
cld(posthoc$emmeans, type = "response", Letters = "abcdefg", adjust = "fdr")

plot(posthoc)

## Total_Delia by treatment ####

summary(Data$Total_Delia)
p1 <- ggplot(Data, aes(x=Treatment, y=Total_Delia, fill=Treatment)) + 
  geom_boxplot()+
  #geom_jitter(width = 0.1, alpha = 0.4, shape = 16)+
  scale_fill_manual(values = col)+
  facet_wrap(~Round)+
  theme_bw()+
  scale_y_continuous(limits = c(0, 10), oob = scales::squish,
                     expand = expansion(mult = c(0, 0.05)))
p1

# ggsave(file = "Figures/TotalDelia_Treatment.pdf", p1,
#        width = 10,
#        height = 6)

# check overdispersion
Mod <- glmer(Total_Delia ~ Treatment*Round+(1|Field), data = Data, family = poisson())
testDispersion(Mod, alternative = "greater") # this model is overdispersed (only slightly)
RVAideMemoire::overdisp.glmer(Mod)
# Slightly Overdispersed
# sim_fmnb <- simulateResiduals(Mod, refit=T, n=99)
# plotSimulatedResiduals(sim_fmnb)
# Okay, definitely overdispersed


M4 <- glmer.nb(Total_Delia ~ Treatment * Round +(1|Field), data = Data)
M3 <- glmer.nb(Total_Delia ~ Treatment + Round +(1|Field), data = Data)
M2 <- glmer.nb(Total_Delia ~ Treatment +(1|Field), data = Data)
M1 <- glmer.nb(Total_Delia ~ Round +(1|Field), data = Data)
M0 <- glmer.nb(Total_Delia ~ 1 +(1|Field), data = Data)

sim_fmnb <- simulateResiduals(M4)
plotSimulatedResiduals(sim_fmnb)

cbind(Fit.stats(M4),Fit.stats(M3),Fit.stats(M2),Fit.stats(M1),Fit.stats(M0))
# analyse full model

lrtest(M4, M3) 
lrtest(M2, M0) 
lrtest(M1, M0) 
summary(M4)

tab_model(M4)


# posthoc
posthoc <- emmeans(M4, pairwise ~ Treatment|Round, type = "response", adjustSigma =T, adjust = "fdr")
posthoc
cld(posthoc$emmeans, type = "response", Letters = "abcdefg", sort = T, adjust = "fdr")


plot(posthoc)

#### Use eggs as offset

M4 <- glmer.nb(Total_Delia ~ Treatment * Round +(1|Field) + offset(log(Data$Eggs_avg)), data = Data)
M3 <- glmer.nb(Total_Delia ~ Treatment + Round +(1|Field)+ offset(log(Data$Eggs_avg)), data = Data)
M2 <- glmer.nb(Total_Delia ~ Treatment +(1|Field)+ offset(log(Data$Eggs_avg)), data = Data)
M1 <- glmer.nb(Total_Delia ~ Round +(1|Field)+ offset(log(Data$Eggs_avg)), data = Data)
M0 <- glmer.nb(Total_Delia ~ 1 +(1|Field)+ offset(log(Data$Eggs_avg)), data = Data)

sim_fmnb <- simulateResiduals(M4)
plotSimulatedResiduals(sim_fmnb)

cbind(Fit.stats(M4),Fit.stats(M3),Fit.stats(M2),Fit.stats(M1),Fit.stats(M0))
# analyse full model

lrtest(M4, M3) 
lrtest(M2, M0) 
lrtest(M1, M0) 
summary(M4)

tab_model(M4)


### Total Delia by field characteristics #### 
### NOT USED IN PAPER 

## Field characteristics
Mod <- glmer(Total_Delia ~ Strip + Cultivar + Nectar + Diversity +(1|Field), data = Data, family = poisson())
testDispersion(Mod, alternative = "greater") # this model is overdispersed :-(
overdisp.glmer(Mod)
# Overdispersed

MR1 <- glmer.nb(Total_Delia ~ Strip + Cultivar + Nectar + Diversity + (1|Field), data = Data_R1)
MR2 <- glmer.nb(Total_Delia ~ Strip + Cultivar + Nectar + Diversity + (1|Field), data = Data_R2)
MRC <- glmer.nb(Total_Delia ~ Strip + Cultivar + Nectar + Diversity + (1|Field) + (1|Round), data = Data)

drop1(MR1, test = "Chisq")
drop1(MR2, test = "Chisq")
drop1(MRC, test = "Chisq")

summary(MR1)
summary(MR2)

p<- plot_models(MR1, MR2, MRC,
            m.labels = c("Round 1", "Round 2", "combined"),
            show.values = T, 
            show.p = T,
            legend.title = "Collection Round",
            axis.lim = c(0.1, 10),
            line.size = 1,
            dot.size = 4,
            colors = c("black", "gray40"))+
  theme_sjplot2()+
  geom_hline(yintercept = 1, alpha = .4)
p

# ggsave(file = "Figures/TotalDelia_Fieldchar.pdf", p,
#        width = 6,
#        height = 6)



## Stacked bar Large scale monitoring. ####
# Filter data with >10 counts in one of the rounds
Data_Stack_R1 <- as.data.frame(Data_R1 %>% select(Treatment, Round, Total_Delia:Miner))
Data_Stack_R2 <- as.data.frame(Data_R2 %>% select(Treatment, Round, Total_Delia:Miner))
filter1 <- colSums(Data_Stack_R1 %>% select(Total_Delia:Miner)) >= 10
filter2 <- colSums(Data_Stack_R2 %>% select(Total_Delia:Miner)) >= 10
filter <- names(which(c(filter1, filter2), arr.ind = FALSE, useNames = TRUE))
filter <- c("Treatment", "Round", filter)

as.data.frame(colSums(Data %>% select(Total_Delia:Miner)))

Stack_Data <- as.data.frame(Data %>% select(Treatment, Round, Total_Delia:Miner))
Stack_Data <- Stack_Data[, colnames(Stack_Data)%in% filter]
Stack_Data <- gather(Stack_Data, key = Species, value = count, -Treatment, -Round)

Stack_Data$Species <- factor(Stack_Data$Species,
                             levels = c("Myzus_persicae","Aphid_other","mummy","Brevicoryne_brassicae", 
                                        "Thrips","Mamestra_brassicae","Phyllotreta_atra","Autographa_gamma","Miner","Pieris_brassicae","Pieris_rapae", "Phyllotreta_undulata","Plutella_xylostella",
                                        "Anthicidae","Velvet_Mite","Spider","Centipede","Ceratopogonidae_larva","Rove_beetle","Carabid_beetle","Carabid_Rove_Larvae",
                                        "Dung_beetle","Scarab_Larvae","Earthworm",
                                        "Click_beetle","Cut_Worm","Total_Delia","Click_Beetle_Larvae"))


AG_Phloem_feeders <- c("Aphid_other","Brevicoryne_brassicae","mummy","Myzus_persicae")
AG_Chewers <- c("Autographa_gamma","Mamestra_brassicae","Miner","Phyllotreta_atra","Phyllotreta_undulata","Pieris_brassicae","Pieris_rapae", "Plutella_xylostella","Thrips")
BG_Predators <- c("Anthicidae","Carabid_beetle","Carabid_Rove_Larvae","Cecidomyiidae_larva","Centipede","Ceratopogonidae_larva","Rove_beetle","Spider","Velvet_Mite")
BG_Detrivores <- c("Dung_beetle","Earthworm","Scarab_Larvae")
BG_Herbivores <- c("Click_beetle","Click_Beetle_Larvae","Cut_Worm")


color <- c(rev(brewer.brbg(14))[1:4],
           ocean.oxy(11)[1:9],
           ocean.dense(13)[2:9],
           ocean.turbid(6)[2:4],
           ocean.algae(6)[2:5])


p<- ggplot(Stack_Data, aes(fill=Species, y=count, x=Treatment)) + 
  geom_bar(position="stack", stat = "summary", fun = "mean", colour="black", size=0.15)+
  scale_fill_manual(values = color)+
  scale_y_continuous(expand = expansion(mult = c(0, 0.05)))+
  labs(y = "Average count per plant")+
  theme_bw()+
  theme(axis.text.x = element_text(angle = 90, vjust = 1, hjust=1))+
  facet_wrap(~Round)
p

# ggsave("Figures/Stacked_large_monitoring.pdf",p,
#        width = 12,
#        height = 6)  

### Test effect of each group on Total_Delia ####
## Total_Delia ~ functional groups
# AG_Phloem_feeders, AG_Chewers, BG_Predators, BG_Detrivores, BG_Herbivores
MR1 <- glmer.nb(Total_Delia ~ AG_Phloem_feeders + AG_Chewers + BG_Predators + BG_Detrivores + BG_Herbivores +  (1|Field), data = Data_R1,
                na.action = "na.fail")
summary(MR1)
sim_fmnb <- simulateResiduals(MR1)
plot(sim_fmnb)

drop1(MR1, test = "Chisq")
tab_model(MR1)

MR2 <- glmer.nb(Total_Delia ~ AG_Phloem_feeders + AG_Chewers + BG_Predators + BG_Detrivores + BG_Herbivores + (1|Field), data = Data_R2,
                na.action = "na.fail")
summary(MR2)
sim_fmnb <- simulateResiduals(MR2) 
plot(sim_fmnb)
# Some issues with residuals.. 
# When modeling only the BG or AG factors this issue is gone, and results are the same
# I'm pretty confident about the BG p-values

drop1(MR2, test = "Chisq")
tab_model(MR2)

p<- plot_models(MR1, MR2,
                m.labels = c("Round 1", "Round 2"),
                show.values = T, 
                show.p = T,
                legend.title = "Collection Round",
                axis.lim = c(0.5, 2),
                line.size = 1,
                dot.size = 4,
                colors = c("black", "gray40"))+
  theme_sjplot2()+
  geom_hline(yintercept = 1, alpha = .4)
p

# ggsave(file = "Figures/TotalDelia_functional_groups.pdf", p,
#        width = 6,
#        height = 6)

MR1_predict <- ggpredict(MR1)
p<- plot(MR1_predict, add.data = TRUE,show.title = F)

grid.arrange(p$AG_Phloem_feeders, p$AG_Chewers, p$BG_Predators, p$BG_Detrivores, p$BG_Herbivores,
             nrow = 1)
g<- arrangeGrob(p$AG_Phloem_feeders, p$AG_Chewers, p$BG_Predators, p$BG_Detrivores, p$BG_Herbivores,
                nrow = 2)

# ggsave(file = "Figures/ggpredict_R1_func.pdf", g,
#        height = 8,
#        width = 12)

MR2_predict <- ggpredict(MR2)
p<-plot(MR2_predict, add.data = TRUE,show.title = F)
grid.arrange(p$AG_Phloem_feeders, p$AG_Chewers, p$BG_Predators, p$BG_Detrivores, p$BG_Herbivores,
             nrow = 1)
g<- arrangeGrob(p$AG_Phloem_feeders, p$AG_Chewers, p$BG_Predators, p$BG_Detrivores, p$BG_Herbivores,
                nrow = 2)

# ggsave(file = "Figures/ggpredict_R2_func.pdf", g,
#        height = 8,
#        width = 12)



## Plant size
summary(Data$Shoot_dry_weight)

summary(Data_R1$Shoot_dry_weight)
qqnorm(Data_R1$Shoot_dry_weight)
shapiro.test(Data_R1$Shoot_dry_weight)
descdist(Data_R1$Shoot_dry_weight, boot = 1000)
#Round 1 gamma dist

summary(Data_R2$Shoot_dry_weight)
qqnorm(Data_R2$Shoot_dry_weight)
shapiro.test(Data_R2$Shoot_dry_weight)
descdist(Data_R2$Shoot_dry_weight, boot = 1000)
# Round 2 normal dist

## Plot shoot dryweight
p1 <- ggplot(Data, aes(x=Treatment, y=Shoot_dry_weight, fill=Treatment)) + 
  geom_boxplot()+
  #geom_jitter(width = 0.1, alpha = 0.4, shape = 16)+
  scale_fill_manual(values = col)+
  facet_wrap(~Round, scales = "free")+
  theme_bw()+
  scale_y_continuous(oob = scales::squish,
                     expand = expansion(mult = c(0, 0.05)))+
  expand_limits(y = 0)
p1

# ggsave(file = "Figures/Shoot_dryweight.pdf", p1,
#        width = 10,
#        height = 6)

## Round 1

M1 <- glmer(Shoot_dry_weight ~ Treatment +(1|Field),family = Gamma, data = Data_R1)
M0 <- glmer(Shoot_dry_weight ~ 1 +(1|Field),family = Gamma, data = Data_R1)

cbind(Fit.stats(M1),Fit.stats(M0))
# analyse full model
sim_fmnb <- simulateResiduals(M1)
plot(sim_fmnb)

lrtest(M1, M0) 
summary(M1)
tab_model(M1)

# posthoc
posthoc <- emmeans(M1, pairwise ~ Treatment, type = "response", adjustSigma =T, adjust = "fdr")
posthoc
cld(posthoc$emmeans, type = "response", Letters = "abcdefg", sort = T, adjust = "fdr")

plot(posthoc)

## Round 2

M1 <- lmer(Shoot_dry_weight ~ Treatment +(1|Field),data = Data_R2)
M0 <- lmer(Shoot_dry_weight ~ 1 +(1|Field),data = Data_R2)

cbind(Fit.stats(M1),Fit.stats(M0))
# analyse full model
sim_fmnb <- simulateResiduals(M1)
plot(sim_fmnb)

lrtest(M1, M0) 
summary(M1)
tab_model(M1)

# posthoc
posthoc <- emmeans(M1, pairwise ~ Treatment, type = "response", adjustSigma =T, adjust = "fdr")
posthoc
cld(posthoc$emmeans, type = "response", Letters = "abcdefg", sort = T, adjust = "fdr")

 plot(posthoc)



## Delia and shoot dryweight

MR1 <- glmer.nb(Total_Delia ~ Shoot_dry_weight +  (1|Field), data = Data_R1,
                na.action = "na.fail")
M0 <- glmer.nb(Total_Delia ~ 1 +  (1|Field), data = Data_R1,
                na.action = "na.fail")

summary(MR1)
sim_fmnb <- simulateResiduals(MR1)
plot(sim_fmnb)

lrtest(MR1, M0)
tab_model(MR1)


MR2 <- glmer.nb(Total_Delia ~ Shoot_dry_weight + (1|Field), data = Data_R2,
                na.action = "na.fail")
M0 <- glmer.nb(Total_Delia ~ 1 + (1|Field), data = Data_R2,
                na.action = "na.fail")

summary(MR2)
sim_fmnb <- simulateResiduals(MR2)
plot(sim_fmnb)

lrtest(MR2, M0)
tab_model(MR2)


MR1_predict <- ggpredict(MR1)
p<- plot(MR1_predict, add.data = TRUE,show.title = F)
p

MR2_predict <- ggpredict(MR2)
p1<-plot(MR2_predict, add.data = TRUE,show.title = F)
p1

grid.arrange(p$Shoot_dry_weight, p1$Shoot_dry_weight,
             nrow = 1)
g<- arrangeGrob(p$Shoot_dry_weight, p1$Shoot_dry_weight,
                ncol = 1)

# ggsave(file = "Figures/ggpredict_shoot_R1_R2.pdf", g,
#        height = 8,
#        width = 4)

#### Larvae ~ Eggs ####
p1 <- ggplot(Data, aes(x=Treatment, y=Proxy_mortality, fill=Treatment)) + 
  geom_boxplot()+
  #geom_jitter(width = 0.1, alpha = 0.4, shape = 16)+
  scale_fill_manual(values = col)+
  facet_wrap(~Round, scales = "free")+
  theme_bw()+
  scale_y_continuous(oob = scales::squish,
                     expand = expansion(mult = c(0, 0.05)))+
  expand_limits(y = 0)
p1

summary(Data$Mortality_rate)
qqnorm(Data$Mortality_rate)
shapiro.test(Data$Mortality_rate)
descdist(Data$Mortality_rate, boot = 1000)

library(glmmTMB)
M4 <- glmer.nb(Proxy_mortality ~ Treatment * Round +(1|Field), family = Gamma, data = Data)
M3 <- glmer.nb(Mortality_rate ~ Treatment + Round +(1|Field), family = Gamma, data = Data)
M2 <- glmer.nb(Mortality_rate ~ Treatment +(1|Field), family = Gamma, data = Data)
M1 <- glmer.nb(Mortality_rate ~ Round +(1|Field), family = Gamma, data = Data)
M0 <- glmer.nb(Mortality_rate ~ 1 +(1|Field), family = Gamma, data = Data)

sim_fmnb <- simulateResiduals(M4)
plotSimulatedResiduals(sim_fmnb)

cbind(Fit.stats(M4),Fit.stats(M3),Fit.stats(M2),Fit.stats(M1),Fit.stats(M0))
# analyse full model

lrtest(M4, M3) 
lrtest(M2, M0) 
lrtest(M1, M0) 
summary(M4)

tab_model(M4)


# posthoc
posthoc <- emmeans(M4, pairwise ~ Treatment|Round, type = "response", adjustSigma =T, adjust = "fdr")
posthoc
cld(posthoc$emmeans, type = "response", Letters = "abcdefg", sort = T, adjust = "fdr")


plot(posthoc)

M5 <- glmer.nb(Total_Delia ~ Eggs_avg * Treatment + Round + (1|Field), data = Data,
                na.action = "na.fail")
M4 <- glmer.nb(Total_Delia ~ Eggs_avg + Treatment + Round + (1|Field), data = Data,
               na.action = "na.fail")
M3 <- glmer.nb(Total_Delia ~ Eggs_avg + (1|Field), data = Data,
               na.action = "na.fail")
M2 <- glmer.nb(Total_Delia ~ Treatment + (1|Field), data = Data,
               na.action = "na.fail")
M1 <- glmer.nb(Total_Delia ~ Round + (1|Field), data = Data,
               na.action = "na.fail")
M0 <- glmer.nb(Total_Delia ~ 1 + (1|Field), data = Data,
               na.action = "na.fail")

sim_fmnb <- simulateResiduals(M5)
plotSimulatedResiduals(sim_fmnb)

lrtest(M5, M4) 
lrtest(M3, M0) 
lrtest(M2, M0) 
lrtest(M1, M0) 
summary(M5)
tab_model(M5)

M5_predict <- ggpredict(M5, c("Eggs_avg", "Treatment"))

p<- plot(M5_predict, add.data = TRUE,show.title = F)
p

