# This file contains maths and other convenience functions

import scipy
from math import floor, log, sqrt, factorial


def combinatorial_factor(n: int, k: int):
    return factorial(n) / (factorial(k) * factorial(n - k)) if n >= 0 and 0 <= k <= n else 0

def quadratic_formula(a,b,c):
    return (-b + sqrt(b**2 - 4 * a * c )) / (2 * a), (-b - sqrt(b**2 - 4 * a * c )) / (2 * a)


def step_function(rate, rate_0) -> int:
    """
    Implements Θ(R - R_0), taking Θ(0) = 1
    :param rate: Rate to input
    :param rate_0: Critical value for the step function.
    :return: {0,1}
    """
    return 1 if rate >= rate_0 else 0


def beta_function(rate, a, b) -> float:
    """
    Returns F_x(r) =  P[X <= r] where X ~ Beta(a,b). F is a ∫ shaped function, with a and b controlling the shape. Notable values and behaviours are:
        - a = b = 1 gives a linear function
        - a = b gives F(0.5) = 0.5

        The maximum of the gradient occurs at r = (a-1) / (a+b-2)

        The maximum gradient is  (a-1)^(a-1) * (b-1)^(b-1) / β(a,b) * (a+b-2)^(a+b-2)

    :param rate:
    :param a:
    :param b:
    :return:
    """

    return scipy.stats.beta(a=a, b=b).cdf(rate)


def get_number_from_alphabetic_name(name, min_chars: int = 1):

    split_name = [ord(x)-64 for x in list(name.upper())]
    split_name.reverse()

    return sum([26**i*split_name[i] for i in range(len(split_name))]) - sum([26**i for i in range(min_chars)])

def get_name_from_number(number: int):
    """
    converts number into format aa = 0, ab = 1, aaa = 26^2 etc.
    :param number: number to convert
    :return: converted number.
    """

    if number > 0:
        max_power = max(floor(log(number, 26)), 1)
    else:
        max_power = 1

    decomposition = []

    for i in range(max_power, -1, -1):
        decomposition.append(number // (26 ** i))
        number -= 26 ** i * decomposition[-1]

    return "".join([chr(97 + i) for i in decomposition])






README_TABLE_HEADER = """

                                     |                successes per second                 |      Proportion of       |
                                     |    packet generation     |     session execution    |     Demand Satisfied     |
        |    Demand    | Target Rate |  mean  | median |  s.d.  |  mean  | median |  s.d.  |  mean  | median |  s.d.  | Repeats |
        |--------------|-------------|--------|--------|--------|--------|--------|--------|--------|--------|--------|---------|
"""



if __name__ == '__main__':

    print(get_number_from_alphabetic_name('a'))

    assert get_number_from_alphabetic_name('a') == 0
    assert get_number_from_alphabetic_name('aa') == 26
    assert get_number_from_alphabetic_name('aa', min_chars=2) == 0
    assert get_number_from_alphabetic_name('ba', min_chars=2) == 26
