import os.path

import pandas as pd
import numpy as np

from Quantum_Network_Architecture.utils.logging import LogManager as NetworkLogManager
from Quantum_Network_Architecture.networks import Network, HomogeneousStarNetwork
from Quantum_Network_Architecture.schedulers import NetworkScheduler, DummyNetworkScheduler

from Quantum_Network_Architecture.simulation.session_generators import SessionGenerator, SingleSessionGenerator
from Quantum_Network_Architecture.simulation.simple_pga_sim import simple_simulation_of_pgas

from Quantum_Network_Architecture.sessions import ApplicationSessionList


from datetime import timedelta

from rich.console import Console

import pickle
from typing import Optional, Dict
import time


def acquire_output_lock(lock_file: str) -> None:
    while os.path.exists(lock_file):
        time.sleep(1)

    open(lock_file, 'w').close()


def relinquish_output_lock(lock_file: str) -> None:
    try:
        os.remove(lock_file)
    except FileNotFoundError:
        pass


def run_network_simulation_homogeneous_demands(

        configs: Dict[str, str],

        iteration: int,

        session_renewal_rate: float,
        target_packet_generation_rate: float,
        single_session_per_flow: bool = True,

        simulation_length: int = 3600,

        debug: bool = False,

        seed: Optional[int] = None,

        do_qoala_sim: bool = False,
        do_actual_scheduling: bool = True,

        output_dir: Optional[str] = None,
        save_logs: bool = False,
        save_sessions: bool = False,

):
    """
    This function runs a simulation of a homogeneous star network where each pair of nodes are running the same application.
    See the README.md for the format of the outputted pandas dataframe


    @param configs: Dictionary of the form {'network':'/network/config.yaml', 'scheduler':'/scheduler/config.yaml', 'sessions':'/sessions/config.yaml'}. For the format of the YAML files, see README.md
    @param iteration: integer, which iteration number this simulation is
    @param session_renewal_rate: Rate at which new sessions are generated
    @param target_packet_generation_rate: Rate at which nodes demand packets to be generated at
    @param single_session_per_flow: optional boolean default True, if True then only one active session per pair of nodes allowed
    @param simulation_length: optional integer, default 3600. Amount of time to simulate. Should be given in seconds.
    @param debug: optional boolean, default False, if True then enables logging and writing to stdout
    @param seed: optional int = None. seed for the RNG
    @param do_qoala_sim: optional bool = False. Enable/disable using Qoala to simulate the PGAs. Case when True is not implemented
    @param do_actual_scheduling: optional bool = True. If False, then uses the DummyNetworkScheduler to 'compute' schedules.
    @param output_dir: optional string = None, If given then will write pickled pandas dataframe to output_dir/raw-data/R/λ/iteration.pickle
    @param save_logs: optional bool = False, If true then will write log files to file.
    @param save_sessions: optional bool = False. If true then will pickle the final SessionList and write it to file.
    """
    # match (debug, save_logs):
    #     case (True, True):
    #
    #
    # if not debug and not save_logs:
    #     NetworkLogManager.set_scheduler_log_level(999)
    # else:
    #     NetworkLogManager.set_scheduler_log_level("INFO")

    rng = np.random.default_rng(seed)

    if save_logs:
        NetworkLogManager.log_to_file(os.path.join(output_dir, "central-controller.log"), remove_syslog=(not debug))

    elif not debug and not save_logs:
        NetworkLogManager().set_session_generator_log_level(999)
        NetworkLogManager().set_pgasimulator_log_level(999)
        NetworkLogManager().set_scheduler_log_level(999)

    ## ---- simulation ----

    network = HomogeneousStarNetwork.from_yaml(configs['network'])

    root_generator = SingleSessionGenerator if single_session_per_flow else SessionGenerator

    session_generator: SessionGenerator = root_generator.create_homogeneous_sessions_from_yaml(
        config_file=configs["sessions"],
        network=network,
        session_rate=session_renewal_rate,
        packet_rate=target_packet_generation_rate,
        rng=rng,
    )

    if do_actual_scheduling:
        scheduler = NetworkScheduler.from_yaml(configs["scheduler"], network=network, start_time=0, debug=debug)
    else:
        scheduler = DummyNetworkScheduler.from_yaml(configs["scheduler"], network=network, start_time=0, debug=debug)

    compute_schedule = scheduler.compute_schedule()

    all_sessions = ApplicationSessionList()
    while scheduler.current_real_time < simulation_length * 1e9:

        while session_generator.next_session_arrival_time < (
                scheduler.current_real_time - scheduler.real_scheduling_interval) / 1e9:
            new_session, _t = session_generator.next_session()

            all_sessions.append(new_session)

            scheduler.submit_demand(new_session.demand)

            scheduler.logger.info(f"Demand {new_session.demand.identifier} submitted"
                                  f" at time {timedelta(seconds=_t)}")

        scheduler.get_tasks_to_add_to_scheduler()
        next(compute_schedule)

        if do_qoala_sim:
            raise NotImplementedError

        else:

            simple_simulation_of_pgas(
                all_sessions,
                scheduler.current_real_time - 2 * scheduler.real_scheduling_interval,
                scheduler.current_real_time - scheduler.real_scheduling_interval,
                network.timeslot_duration,
                scheduler.pga_success_probability,
                rng=rng
            )

        for session in [s for s in all_sessions if not s.terminated_at is not None]:

            if session.minimal_service:
                scheduler.remove_task(session.demand.packet_generation_task)
                session.terminated_at = (scheduler.current_real_time - scheduler.real_scheduling_interval) / 1e9
                session.is_expired = True
                scheduler.logger.info(f"Session {session.identifier} achieved minimal service, terminating")


            elif session.demand.packet_generation_task is None and session.demand not in scheduler.demand_queue:
                session.terminated_at = scheduler.current_real_time / 1e9 - scheduler.real_scheduling_interval / 1e9
                session.is_expired = True


            elif session.is_expired or session.demand.timing_constraints.expiry < scheduler.current_real_time / 1e9:
                session.is_expired = True
                session.terminated_at = scheduler.current_real_time / 1e9
                scheduler.remove_demand(session.demand)
                scheduler.logger.info(f"Session {session.identifier} expired.")

            else:
                if session.demand.packet_generation_task is not None:
                    scheduler.logger.info(
                        f"Session {session.identifier} is active. ({session.executed_instances}/{session.min_number_of_instances} instances; {timedelta(seconds=scheduler.current_real_time / 1e9)}/{timedelta(seconds=session.expiry_time)})")
                else:
                    scheduler.logger.info(
                        f"Session {session.identifier} is queuing. ({session.executed_instances}/{session.min_number_of_instances} instances; {timedelta(seconds=scheduler.current_real_time / 1e9 - session.demand.submission_time)}/{timedelta(seconds=session.expiry_time - session.demand.submission_time)})")

            if session.is_expired and isinstance(session_generator, SingleSessionGenerator):
                session_generator.renew_session(session.key,
                                                session.terminated_at)  # Offset to account for actual time rather than time schedule computed to.
                scheduler.logger.info(
                    f"Renewed key {session.key} at time {timedelta(seconds=session.terminated_at)}")

    if debug:
        console = Console(width=200, stderr=True)

        console.print(all_sessions.all_tasks.as_table(
            init_order=True,
            show_prop_delivered=False,
            show_no_schedule=True,
            show_dropped_packets=False,
            show_create_time=True,
            show_execution_time=True,
            show_minsep=True,
            show_period=True,
            show_utilisation=True,
            show_Rate=True,
            network_timeslot_length=network.timeslot_duration
        ))

        console.print(all_sessions.as_table())
        console.print(f"Proportion min service: {all_sessions.proportion_pgt_expired_minimal_service}")
        console.print(
            f"Proportion min service | got any service: {all_sessions.proportion_expired_got_service_got_minimal_service}")
        console.print(f"Ave max utilisation: {scheduler.ave_max_utilisation}")

    if not all_sessions:
        pass

    new_row = [
                  len(all_sessions),
                  len(all_sessions.expired_sessions),
                  all_sessions.number_no_service_expired,
                  all_sessions.proportion_pgt_expired_minimal_service,
                  all_sessions.proportion_expired_got_service_got_minimal_service,
                  all_sessions.average_queue_time,
                  all_sessions.average_latency,
                  all_sessions.average_sojourn,
                  all_sessions.average_service_time,
                  all_sessions.all_tasks.average_pga_rate / network.timeslot_duration * 1e9,
                  all_sessions.all_tasks.average_utilisation,
                  scheduler.average_est_pgas_in_schedule,
                  scheduler.ave_max_utilisation
              ] + [
                  scheduler.average_resource_utilisations[r] for r in scheduler.average_resource_utilisations.keys()
              ]
    new_loc = (
        list(session_generator.packet_rates.values())[0], list(session_generator.session_arrival_rates.values())[0],
        iteration)

    df = pd.DataFrame(
        columns=[
                    "number of sessions",
                    "number of expired sessions",
                    "number of expired sessions no service",
                    "proportion minimal service",
                    "proportion minimal service given some service",
                    "average queue time",
                    "average latency to service",
                    "average sojourn time",
                    "average service time",
                    "average pga rate",
                    "average pgt utilisation",
                    "average est pgas per schedule",
                    "average max resource utilisation",
                ] + [
                    f"Average utilisation of link {r}" for r in scheduler.average_resource_utilisations.keys()
                ],
        index=pd.MultiIndex.from_tuples(
            [new_loc], names=["requested packet rate", "session renewal rate", "iteration"])
    )

    df.loc[new_loc] = new_row

    if output_dir is not None:
        save_to_file = os.path.join(output_dir, "raw-data", f"{session_renewal_rate}",
                                    f"{target_packet_generation_rate}", f"{iteration}.pickle")
        if not os.path.isdir(os.path.split(save_to_file)[0]):
            os.makedirs(os.path.split(save_to_file)[0], exist_ok=True)  # exist_ok = True means will not throw error
            # if dir already exists (e.g. if two instances try
            # and create the directory at the same time).

        with open(save_to_file, "wb") as f:
            pickle.dump(df, f)

    if save_sessions:
        sessions_file = os.path.join(output_dir, "pickled_session_lists", f"{session_renewal_rate}",
                                     f"{target_packet_generation_rate}", f"{iteration}.pickle")
        if not os.path.isdir(os.path.split(sessions_file)[0]):
            os.makedirs(os.path.split(sessions_file)[0], exist_ok=True)
        with open(sessions_file, "wb") as f:
            pickle.dump(all_sessions, f)
