
from Quantum_Network_Architecture.networks.nodes import NodeList, Node
from Quantum_Network_Architecture.demands import NetworkDemand
from Quantum_Network_Architecture.tasks import PacketGenerationTask, Taskset

from dataclasses import dataclass

from typing import Optional, List, Iterable

from math import inf

from rich.table import Table
from datetime import timedelta

@dataclass
class ApplicationSession:
    _id: str
    _nodes: NodeList
    _app: str
    _min_number_of_instances: int
    _expiry_time: int
    executed_instances: int = 0
    demand: Optional[NetworkDemand] = None
    _is_expired: bool = False
    terminated_at: Optional[float] = None


    @property
    def queue_time(self) -> Optional[float]:
        if self.demand.queue_exit_time is not None:
            return self.demand.queue_exit_time - self.demand.submission_time
        else:
             return None

    @property
    def is_expired(self):
        return self.demand.packet_generation_task.is_expired if self.demand.packet_generation_task is not None else self._is_expired

    @is_expired.setter
    def is_expired(self, value):
        if isinstance(value, bool):
            self._is_expired = value
        pass

    @property
    def minimal_service(self):
        return self.executed_instances >= self.min_number_of_instances

    @property
    def identifier(self):
        return self._id

    @property
    def key(self):
        return self.identifier.rsplit('-',1)[0]

    @property
    def nodes(self):
        return self._nodes

    @property
    def app(self):
        return self._app

    @property
    def min_number_of_instances(self):
        return self._min_number_of_instances

    @property
    def expiry_time(self):
        return self._expiry_time

    @property
    def sojourn_time(self):
        return self.terminated_at - self.demand.submission_time if self.terminated_at is not None and self.terminated_at - self.demand.submission_time > 0 else None

    @property
    def service_time(self) -> float | None:

        if self.demand.packet_generation_task is not None and self.terminated_at is not None:
            service_time = max(self.terminated_at - self.demand.packet_generation_task.creation_time,0)
        else:
            service_time = None

        return service_time

class ApplicationSessionList(List[ApplicationSession]):


    def __init__(self, _iterable: Optional[Iterable] = None):

        if _iterable is None:
            super().__init__()
        elif all(isinstance(s, ApplicationSession) for s in _iterable):
            super().__init__(_iterable)
        else:
            raise ValueError

    @property
    def expired_sessions(self):
        return ApplicationSessionList([s for s in self if s.is_expired])

    @property
    def not_expired_sessions(self):
        return ApplicationSessionList([s for s in self if not s.is_expired])

    @property
    def identifiers(self):
        return [s.identifier for s in self]

    @property
    def proportion_minimal_service(self):
        return len([s for s in self if s.minimal_service])/len(self) if self else 1

    @property
    def proportion_pgt_expired_minimal_service(self):
        expired_pgts = [s for s in self if s.is_expired]
        return len([s for s in self if s.minimal_service])/len(expired_pgts) if expired_pgts else 1

    @property
    def proportion_expired_got_service_got_minimal_service(self):
        sessions_with_service = [s for s in self if s.demand.packet_generation_task is not None and s.is_expired]
        return len([s for s in sessions_with_service if s.minimal_service])/len(sessions_with_service) if sessions_with_service else 1

    @property
    def number_no_service(self):
        return len([s for s in self if s.demand.packet_generation_task is None])

    @property
    def number_no_service_expired(self):
        return len([s for s in self if s.demand.packet_generation_task is None and s.is_expired])

    @property
    def average_queue_time(self):
        applicable_sessions = [s for s in self if s.queue_time is not None]
        return sum([s.queue_time for s in applicable_sessions]) / len(applicable_sessions) if applicable_sessions else 0


    @property
    def average_latency(self):
        s: ApplicationSession
        served_demand_latencies = [s.demand.latency_to_service for s in self if s.demand.latency_to_service is not None]

        return sum(served_demand_latencies)/len(served_demand_latencies) if served_demand_latencies else 0

    @property
    def average_sojourn(self):
        s: ApplicationSession
        terminated_demands_sojourn = [s.sojourn_time for s in self if s.sojourn_time is not None]

        return sum(terminated_demands_sojourn) / len(terminated_demands_sojourn) if terminated_demands_sojourn else 0

    @property
    def average_service_time(self):
        s: ApplicationSession
        terminated_demands_services = [s.service_time for s in self if s.service_time is not None]
        return sum(terminated_demands_services) / len(terminated_demands_services) if terminated_demands_services else 0

    @property
    def all_demands(self) -> List[NetworkDemand]:
        s: ApplicationSession
        return [s.demand for s in self]

    @property
    def all_tasks(self):
        s: ApplicationSession
        return Taskset(s.demand.packet_generation_task for s in self if s.demand.packet_generation_task is not None)

    def as_table(self, show_id=True, show_app=True, show_executed_instances=True,
                                show_minimal_service=True, show_is_expired=True, show_creation_time=True, show_time_accepted=True,
                                show_latency_to_service=True, show_queue_time:bool = True, show_expiry_time=True, show_termination_time=True, show_sojourn_time: bool = True) -> Table:

        table = Table(title='Application Sessions')


        # Define table columns based on the selected entries
        if show_id:
            table.add_column("ID")
        if show_app:
            table.add_column("App")
        if show_executed_instances:
            table.add_column("Executed Instances")
        if show_minimal_service:
            table.add_column("Minimal Service")
        if show_is_expired:
            table.add_column("Is Expired")

        if show_creation_time:
            table.add_column("Creation Time")
        if show_time_accepted:
            table.add_column("Time Accepted")
        if show_latency_to_service:
            table.add_column("Latency to Service")
        if show_queue_time:
            table.add_column("Queue Time")
        if show_expiry_time:
            table.add_column("Expiry Time")
        if show_termination_time:
            table.add_column("Terminated At")
        if show_sojourn_time:
            table.add_column("Sojourn Time")

        for session in self:
            row = []
            if show_id:
                row.append(session.identifier)
            if show_app:
                row.append(session.app)
            if show_executed_instances:
                row.append(str(session.executed_instances))
            if show_minimal_service:
                row.append(str(session.minimal_service))
            if show_is_expired:
                row.append(str(session.is_expired))
            if show_creation_time:
                creation_time = session.demand.submission_time if session.demand else "N/A"
                row.append(str(timedelta(seconds=creation_time)))
            if show_time_accepted:
                time_accepted = session.demand.packet_generation_task.creation_time if session.demand and session.demand.packet_generation_task else None
                row.append(str(timedelta(seconds=time_accepted)) if time_accepted is not None else "N/A")
            if show_latency_to_service:
                latency_to_service = session.demand.latency_to_service if session.demand else None
                row.append(str(timedelta(seconds=latency_to_service)) if latency_to_service is not None else "N/A")
            if show_queue_time:
                row.append(str(timedelta(seconds=session.queue_time)) if session.queue_time is not None else "N/A")
            if show_expiry_time:
                row.append(str(timedelta(seconds=session.expiry_time)))
            if show_termination_time:
                row.append(str(timedelta(seconds=session.terminated_at)) if session.terminated_at is not None else 'N/A')
            if show_sojourn_time:
                row.append(str(timedelta(seconds=session.sojourn_time)) if session.sojourn_time is not None else 'N/A')

            table.add_row(*row)


        return table








