"""
2022-2024 Paul Moller, Sebastian de Bone (QuTech)
https://github.com/Poeloe/oop_surface_code
_____________________________________________
"""
from circuit_simulation.circuit_simulator import *
from tqdm import tqdm
PBAR: tqdm = None


def create_quantum_circuit(protocol, pbar, **kwargs):
    """
        Initialises a QuantumCircuit object corresponding to the protocol requested.

        Parameters
        ----------
        protocol : str
            Name of the protocol for which the QuantumCircuit object should be initialised

        For other parameters, please see QuantumCircuit class for more information

    """
    global PBAR
    PBAR = pbar

    supop_qubits = None

    if protocol == 'monolithic':
        kwargs.pop('basis_transformation_noise')
        kwargs.pop('no_single_qubit_error')
        qc = QuantumCircuit(9, 2, basis_transformation_noise=True, no_single_qubit_error=False, **kwargs)
        qc.define_node("A", qubits=[1, 3, 5, 7, 0], amount_data_qubits=4)
        qc.define_sub_circuit("A")

    elif 'plain' in protocol:
        qc = QuantumCircuit(16, 2, **kwargs)

        qc.define_node("A", qubits=[14, 7, 6])
        qc.define_node("B", qubits=[12, 5, 4])
        qc.define_node("C", qubits=[10, 3, 2])
        qc.define_node("D", qubits=[8, 1, 0])

    elif protocol in ['modicum', 'modicum_swap']:
        qc = QuantumCircuit(16, 2, **kwargs)

        qc.define_node("A", qubits=[14, 7, 6])
        qc.define_node("B", qubits=[12, 5, 4])
        qc.define_node("C", qubits=[10, 3, 2])
        qc.define_node("D", qubits=[8, 1, 0])

    elif 'medium' in protocol:
        qc = QuantumCircuit(24, 2, **kwargs)

        qc.define_node("A", qubits=[22, 15, 14, 13, 12])
        qc.define_node("B", qubits=[20, 11, 10, 9, 8])
        qc.define_node("C", qubits=[18, 7, 6, 5, 4])
        qc.define_node("D", qubits=[16, 3, 2, 1, 0])

    elif 'refined' in protocol:
        qc = QuantumCircuit(28, 2, **kwargs)

        qc.define_node("A", qubits=[26, 19, 18, 17, 16, 15])
        qc.define_node("B", qubits=[24, 14, 13, 12, 11, 10])
        qc.define_node("C", qubits=[22, 9, 8, 7, 6, 5])
        qc.define_node("D", qubits=[20, 4, 3, 2, 1, 0])

    else:
        qc = QuantumCircuit(20, 2, **kwargs)

        qc.define_node("A", qubits=[18, 11, 10, 9])
        qc.define_node("B", qubits=[16, 8, 7, 6])
        qc.define_node("C", qubits=[14, 5, 4, 3])
        qc.define_node("D", qubits=[12, 2, 1, 0])

    # Common sub circuit defining handled here
    if protocol in ['plain', 'plain_swap', 'expedient', 'expedient_swap', 'stringent',
                    'stringent_swap', 'modicum', 'modicum_swap'] \
            or 'basic' in protocol \
            or 'medium' in protocol \
            or 'refined' in protocol \
            or 'minimum4x_40' in protocol:
        qc.define_sub_circuit("ABCD")
        qc.define_sub_circuit("AB")
        qc.define_sub_circuit("CD", concurrent_sub_circuits="AB")
        qc.define_sub_circuit("AC")
        if 'plain' not in protocol:
            qc.define_sub_circuit("BD", concurrent_sub_circuits="AC")
        qc.define_sub_circuit("A")
        qc.define_sub_circuit("B")
        qc.define_sub_circuit("C")
        qc.define_sub_circuit("D", concurrent_sub_circuits=["A", "B", "C"])

    return qc, supop_qubits


def monolithic(qc: QuantumCircuit, *, operation):
    qc.set_qubit_states({0: ket_p})

    PBAR.update(70) if PBAR is not None else None

    return ["A"]


def plain(qc: QuantumCircuit, *, operation):
    qc.start_sub_circuit("AB")
    qc.create_bell_pair(7, 5)
    qc.start_sub_circuit("CD")
    qc.create_bell_pair(3, 1)
    qc.start_sub_circuit("AC")
    success = qc.single_selection(operation, 6, 2)
    if not success:
        qc.start_sub_circuit("A")
        qc.X(7)
        qc.start_sub_circuit("B")
        qc.X("B-e")

    PBAR.update(70) if PBAR is not None else None

    return ["B", "A", "D", "C"]


def plain_swap(qc: QuantumCircuit, *, operation):
    qc.start_sub_circuit("AB")
    qc.create_bell_pair("B-e", "A-e")
    qc.SWAP("A-e", "A-e+1")
    qc.start_sub_circuit("CD")
    qc.create_bell_pair("D-e", "C-e")
    qc.SWAP("C-e", "C-e+1")
    qc.start_sub_circuit("AC")
    success = qc.single_selection(CZ_gate, "C-e", "A-e", swap=True)
    if not success:
        qc.start_sub_circuit("A")
        qc.X("A-e+1")
        qc.start_sub_circuit("B")
        qc.X("B-e")

    PBAR.update(70) if PBAR is not None else None

    return ["C", "D", "A", "B"]


def modicum(qc: QuantumCircuit, *, operation):
    ghz_success = False
    while not ghz_success:
        PBAR.reset() if PBAR is not None else None
        qc.start_sub_circuit("AC")
        qc.create_bell_pair("A-e", "C-e")

        qc.start_sub_circuit("BD")
        qc.create_bell_pair("D-e", "B-e")

        PBAR.update(40) if PBAR is not None else None

        qc.start_sub_circuit("AB")
        qc.create_bell_pair("B-e+1", "A-e+1")
        qc.apply_gate(CNOT_gate, cqubit="A-e", tqubit="A-e+1", reverse=True)
        qc.apply_gate(CNOT_gate, cqubit="B-e", tqubit="B-e+1", reverse=True)
        meas_parity_ab = qc.measure(["A-e+1", "B-e+1"], basis="Z")

        qc.start_sub_circuit("CD")
        qc.create_bell_pair("C-e+1", "D-e+1")
        meas_parity_cd = qc.single_selection(CZ_gate, "C-e+1", "D-e+1", "C-e", "D-e", create_bell_pair=False)

        qc.start_sub_circuit("ABCD", forced_level=True)
        ghz_success = (len(set(meas_parity_ab)) == 1) == meas_parity_cd if not qc.cut_off_time_reached else True
        if not ghz_success:
            continue
        if not meas_parity_cd:
            qc.X("B-e")
            qc.X("D-e")

        PBAR.update(40) if PBAR is not None else None

    return ["C", "A", "B", "D"]


def modicum_swap(qc: QuantumCircuit, *, operation):
    ghz_success = False
    while not ghz_success:
        PBAR.reset() if PBAR is not None else None
        qc.start_sub_circuit("AC")
        qc.create_bell_pair("A-e", "C-e")
        qc.SWAP("A-e", "A-e+1", efficient=True)
        qc.SWAP("C-e", "C-e+1", efficient=True)

        qc.start_sub_circuit("BD")
        qc.create_bell_pair("D-e", "B-e")
        qc.SWAP("B-e", "B-e+1", efficient=True)
        qc.SWAP("D-e", "D-e+1", efficient=True)

        PBAR.update(40) if PBAR is not None else None

        qc.start_sub_circuit("AB")
        qc.create_bell_pair("B-e", "A-e")

        # Option I
        qc.H("A-e")
        qc.H("B-e")
        qc.apply_gate(CZ_gate, cqubit="A-e", tqubit="A-e+1")
        qc.apply_gate(CZ_gate, cqubit="B-e", tqubit="B-e+1")
        qc.H("A-e")
        qc.H("B-e")
        # # Option II
        # qc.apply_gate(CNOT_gate, cqubit="A-e+1", tqubit="A-e", electron_is_target=True, reverse=True)
        # qc.apply_gate(CNOT_gate, cqubit="B-e+1", tqubit="B-e", electron_is_target=True, reverse=True)

        meas_parity_ab = qc.measure(["A-e", "B-e"], basis="Z")

        qc.start_sub_circuit("CD")
        qc.create_bell_pair("C-e", "D-e")
        meas_parity_cd = qc.single_selection(CZ_gate, "C-e", "D-e", "C-e+1", "D-e+1", create_bell_pair=False)

        qc.start_sub_circuit("ABCD", forced_level=True)
        ghz_success = (len(set(meas_parity_ab)) == 1) == meas_parity_cd if not qc.cut_off_time_reached else True
        if not ghz_success:
            continue
        if not meas_parity_cd:
            qc.X("B-e+1")
            qc.X("D-e+1")

        PBAR.update(40) if PBAR is not None else None
    return ["C", "A", "B", "D"]


def expedient(qc: QuantumCircuit, *, operation):
    ghz_success = False
    while not ghz_success:
        PBAR.reset() if PBAR is not None else None

        # Step 1-2 from Table D.1 (Thesis Naomi Nickerson)
        qc.start_sub_circuit("AB")
        success_ab = False
        while not success_ab:
            qc.create_bell_pair(11, 8)
            success_ab = qc.double_selection(CZ_gate, 10, 7)
            if not success_ab:
                continue
            success_ab = qc.double_selection(CNOT_gate, 10, 7)

        PBAR.update(20) if PBAR is not None else None

        # Step 1-2 from Table D.1 (Thesis Naomi Nickerson)
        qc.start_sub_circuit("CD")
        success_cd = False
        while not success_cd:
            qc.create_bell_pair(5, 2)
            success_cd = qc.double_selection(CZ_gate, 4, 1)
            if not success_cd:
                continue
            success_cd = qc.double_selection(CNOT_gate, 4, 1)

        PBAR.update(20) if PBAR is not None else None

        # Step 3-5 from Table D.1 (Thesis Naomi Nickerson)
        qc.start_sub_circuit('AC')
        outcome_ac = qc.single_dot(CZ_gate, "A-e+1", "C-e+1")
        qc.start_sub_circuit('BD')
        outcome_bd = qc.single_dot(CZ_gate, "B-e+1", "D-e+1")
        qc.start_sub_circuit("ABCD")
        ghz_success = outcome_bd == outcome_ac
        if not ghz_success:
            continue
        if not outcome_bd:
            qc.X("A-e+1")
            qc.X("B-e+1")

        PBAR.update(20) if PBAR is not None else None

        # Step 6-8 from Table D.1 (Thesis Naomi Nickerson)
        qc.start_sub_circuit("AC", forced_level=True)
        ghz_success_1 = qc.single_dot(CZ_gate, 10, 4)
        qc.start_sub_circuit("BD")
        ghz_success_2 = qc.single_dot(CZ_gate, 7, 1)
        if any([not ghz_success_1, not ghz_success_2]):
            ghz_success = False
            continue

        PBAR.update(10) if PBAR is not None else None

    # Step 9 from Table D.1 (Thesis Naomi Nickerson)
    # ORDER IS ON PURPOSE: EVERYTIME THE TOP QUBIT IS MEASURED, WHICH DECREASES RUNTIME SIGNIFICANTLY
    # qc.stabilizer_measurement(operation, nodes=["B", "A", "D", "C"])
    # PBAR.update(20) if PBAR is not None else None
    return ["B", "A", "D", "C"]


def stringent(qc: QuantumCircuit, *, operation):
    ghz_success = False
    while not ghz_success:
        PBAR.reset() if PBAR is not None else None

        # Step 1-8 from Table D.2 (Thesis Naomi Nickerson)
        success_ab = False
        qc.start_sub_circuit("AB")
        while not success_ab:
            qc.create_bell_pair(11, 8)
            success_ab = qc.double_selection(CZ_gate, 10, 7)
            if not success_ab:
                continue
            success_ab = qc.double_selection(CNOT_gate, 10, 7)
            if not success_ab:
                continue

            success_ab = qc.double_dot(CZ_gate, 10, 7)
            if not success_ab:
                continue
            success_ab = qc.double_dot(CNOT_gate, 10, 7)
            if not success_ab:
                continue

        PBAR.update(20) if PBAR is not None else None

        # Step 1-8 from Table D.2 (Thesis Naomi Nickerson)
        success_cd = False
        qc.start_sub_circuit("CD")
        while not success_cd:
            qc.create_bell_pair(5, 2)
            success_cd = qc.double_selection(CZ_gate, 4, 1)
            if not success_cd:
                continue
            success_cd = qc.double_selection(CNOT_gate, 4, 1)
            if not success_cd:
                continue

            success_cd = qc.double_dot(CZ_gate, 4, 1)
            if not success_cd:
                continue
            success_cd = qc.double_dot(CNOT_gate, 4, 1)
            if not success_cd:
                continue

        PBAR.update(20) if PBAR is not None else None

        # Step 9-11 from Table D.2 (Thesis Naomi Nickerson)
        qc.start_sub_circuit('AC')
        outcome_ac, single_selection_success_ac = qc.double_dot(CZ_gate, "A-e+1", "C-e+1")
        qc.start_sub_circuit('BD')
        outcome_bd, single_selection_success_bd = qc.double_dot(CZ_gate, "B-e+1", "D-e+1")
        qc.start_sub_circuit("ABCD", forced_level=True)
        ghz_success = outcome_bd == outcome_ac and single_selection_success_ac and single_selection_success_bd
        if not ghz_success:
            continue
        if not outcome_bd:
            qc.X("A-e+1")
            qc.X("B-e+1")

        PBAR.update(20) if PBAR is not None else None

        # Step 12-14 from Table D.2 (Thesis Naomi Nickerson)
        qc.start_sub_circuit("AC", forced_level=True)
        ghz_success_1 = qc.double_dot(CZ_gate, 10, 4)
        qc.start_sub_circuit("BD")
        ghz_success_2 = qc.double_dot(CZ_gate, 7, 1)
        if any([not ghz_success_1, not ghz_success_2]):
            ghz_success = False
            continue

        PBAR.update(10) if PBAR is not None else None

    # Step 15 from Table D.2 (Thesis Naomi Nickerson)
    # ORDER IS ON PURPOSE: EVERYTIME THE TOP QUBIT IS MEASURED, WHICH DECREASES RUNTIME SIGNIFICANTLY
    # qc.stabilizer_measurement(operation, nodes=["B", "A", "D", "C"])
    # PBAR.update(20) if PBAR is not None else None

    return ["B", "A", "D", "C"]


def expedient_swap(qc: QuantumCircuit, *, operation, tqubit=None):
    ghz_success = False
    while not ghz_success:
        PBAR.reset() if PBAR is not None else None

        qc.start_sub_circuit("AB")
        success_ab = False
        while not success_ab:
            qc.create_bell_pair("A-e", "B-e")
            qc.SWAP("A-e", "A-e+1", efficient=True)
            qc.SWAP("B-e", "B-e+1", efficient=True)
            success_ab = qc.double_selection(CZ_gate, "A-e", "B-e", swap=True)
            if not success_ab:
                continue
            success_ab = qc.double_selection(CNOT_gate, "A-e", "B-e", swap=True)

        PBAR.update(20) if PBAR is not None else None

        qc.start_sub_circuit("CD")
        success_cd = False
        while not success_cd:
            qc.create_bell_pair("C-e", "D-e")
            qc.SWAP("C-e", "C-e+1", efficient=True)
            qc.SWAP("D-e", "D-e+1", efficient=True)
            success_cd = qc.double_selection(CZ_gate, "C-e", "D-e", swap=True)
            if not success_cd:
                continue
            success_cd = qc.double_selection(CNOT_gate, "C-e", "D-e", swap=True)

        PBAR.update(20) if PBAR is not None else None

        qc.start_sub_circuit('AC')
        outcome_ac = qc.single_dot(CZ_gate, "A-e", "C-e", swap=True)
        qc.start_sub_circuit('BD')
        outcome_bd = qc.single_dot(CZ_gate, "B-e", "D-e", swap=True)
        qc.start_sub_circuit("ABCD", forced_level=True)
        ghz_success = outcome_bd == outcome_ac if not qc.cut_off_time_reached else True
        if not ghz_success:
            continue
        if not outcome_bd:
            qc.X("A-e+1")
            qc.X("B-e+1")

        PBAR.update(20) if PBAR is not None else None

        qc.start_sub_circuit('AC', forced_level=True)
        ghz_success_1 = qc.single_dot(CZ_gate, "A-e", "C-e", swap=True)
        qc.start_sub_circuit("BD")
        ghz_success_2 = qc.single_dot(CZ_gate, "B-e", "D-e", swap=True)
        if any([not ghz_success_1, not ghz_success_2]):
            ghz_success = False
            continue

    PBAR.update(10) if PBAR is not None else None

    # ORDER IS ON PURPOSE: EVERYTIME THE TOP QUBIT IS MEASURED, WHICH DECREASES RUNTIME SIGNIFICANTLY
    # qc.stabilizer_measurement(operation, nodes=["B", "A", "D", "C"], swap=True, tqubit=tqubit)
    # PBAR.update(20) if PBAR is not None else None

    return ["B", "A", "D", "C"]


def stringent_swap(qc: QuantumCircuit, *, operation):
    ghz_success = False
    while not ghz_success:
        PBAR.reset() if PBAR is not None else None

        qc.start_sub_circuit("AB")
        success_ab = False
        while not success_ab:
            qc.create_bell_pair(9, 6)
            qc.SWAP(9, 10, efficient=True)
            qc.SWAP(6, 7, efficient=True)
            if not qc.double_selection(CZ_gate, 9, 6, swap=True):
                continue
            if not qc.double_selection(CNOT_gate, 9, 6, swap=True):
                continue
            outcome = qc.double_dot(CZ_gate, 9, 6, swap=True)
            if outcome != SKIP() and not all(outcome):
                continue
            outcome_2 = qc.double_dot(CNOT_gate, 9, 6, swap=True)
            if outcome_2 == SKIP() or all(outcome_2):
                success_ab = True

        PBAR.update(20) if PBAR is not None else None

        qc.start_sub_circuit("CD")
        success_cd = False
        while not success_cd:
            qc.create_bell_pair(3, 0)
            qc.SWAP(3, 4, efficient=True)
            qc.SWAP(0, 1, efficient=True)
            if not qc.double_selection(CZ_gate, 3, 0, swap=True):
                continue
            if not qc.double_selection(CNOT_gate, 3, 0, swap=True):
                continue
            outcome = qc.double_dot(CZ_gate, 3, 0, swap=True)
            if outcome != SKIP() and not all(outcome):
                continue
            outcome_2 = qc.double_dot(CNOT_gate, 3, 0, swap=True)
            if outcome_2 == SKIP() or all(outcome_2):
                success_cd = True

        PBAR.update(20) if PBAR is not None else None

        qc.start_sub_circuit('AC')
        outcome_ac = qc.double_dot(CZ_gate, "A-e", "C-e", swap=True)
        qc.start_sub_circuit('BD')
        outcome_bd = qc.double_dot(CZ_gate, "B-e", "D-e", swap=True)
        qc.start_sub_circuit("ABCD", forced_level=True)
        ghz_success = (outcome_bd[0] == outcome_ac[0] and outcome_ac[1] and outcome_bd[1]
                       if not qc.cut_off_time_reached else True)
        if not ghz_success:
            continue
        if outcome_bd != SKIP() and not outcome_bd[0]:
            qc.X("A-e+1")
            qc.X("B-e+1")

        PBAR.update(20) if PBAR is not None else None

        qc.start_sub_circuit("AC", forced_level=True)
        ghz_success_1 = qc.double_dot(CZ_gate, 9, 3, swap=True)
        ghz_success_1 = all(ghz_success_1) if ghz_success_1 != SKIP() else True
        qc.start_sub_circuit("BD")
        ghz_success_2 = qc.double_dot(CZ_gate, 6, 0, swap=True)
        ghz_success_2 = all(ghz_success_2) if ghz_success_2 != SKIP() else True
        if any([not ghz_success_1, not ghz_success_2]):
            ghz_success = False
            continue

        PBAR.update(10) if PBAR is not None else None

    # ORDER IS ON PURPOSE: EVERYTIME THE TOP QUBIT IS MEASURED, WHICH DECREASES RUNTIME SIGNIFICANTLY
    # qc.stabilizer_measurement(operation, nodes=["B", "A", "D", "C"], swap=True)
    # PBAR.update(20) if PBAR is not None else None

    return ["B", "A", "D", "C"]


def basic_medium_refined_ghz_block(qc: QuantumCircuit, offset=1, prot='basic'):
    ghz_success = False
    while not ghz_success:
        for round in [1, 2]:
            outcomes = []
            sub_circuits = ["AB", "CD"] if round == 1 else ["AC", "BD"]
            for circuit in sub_circuits:
                bell_success = False
                while not bell_success:
                    qc.start_sub_circuit(circuit)
                    qc.create_bell_pair(f"{circuit[0]}-e", f"{circuit[1]}-e")
                    mem = offset if round == 1 else 3
                    if round == 1 or (round == 2 and prot in ['medium', 'refined']):
                        qc.SWAP(f"{circuit[0]}-e", f"{circuit[0]}-e+{mem}", efficient=True)
                        qc.SWAP(f"{circuit[1]}-e", f"{circuit[1]}-e+{mem}", efficient=True)

                    if prot == 'medium':
                        bell_success = qc.single_selection(CZ_gate, f"{circuit[0]}-e", f"{circuit[1]}-e",
                                                           f"{circuit[0]}-e+{mem}", f"{circuit[1]}-e+{mem}")
                        if type(bell_success) == SKIP:
                            break
                    elif prot == 'refined':
                        qc.single_selection(CZ_gate, f"{circuit[0]}-e", f"{circuit[1]}-e",
                                            f"{circuit[0]}-e+{mem}", f"{circuit[1]}-e+{mem}", measure=False)
                        qc.SWAP(f"{circuit[0]}-e", f"{circuit[0]}-e+{round + 2}", efficient=True)
                        qc.SWAP(f"{circuit[1]}-e", f"{circuit[1]}-e+{round + 2}", efficient=True)
                        bell_result_1 = qc.single_selection(CZ_gate, f"{circuit[0]}-e", f"{circuit[1]}-e",
                                                            f"{circuit[0]}-e+{round + 2}", f"{circuit[1]}-e+{round + 2}")
                        if type(bell_result_1) == SKIP:
                            break
                        if bell_result_1 is False:
                            continue
                        qc.SWAP(f"{circuit[0]}-e", f"{circuit[0]}-e+{round + 2}", efficient=True)
                        qc.SWAP(f"{circuit[1]}-e", f"{circuit[1]}-e+{round + 2}", efficient=True)
                        bell_result_2 = qc.measure([f"{circuit[1]}-e", f"{circuit[0]}-e"])
                        if type(bell_result_2) == SKIP:
                            break
                        if bell_result_2[0] != bell_result_2[1]:
                            continue

                        qc.single_selection(CNOT_gate, f"{circuit[0]}-e", f"{circuit[1]}-e",
                                            f"{circuit[0]}-e+{mem}", f"{circuit[1]}-e+{mem}", measure=False)
                        qc.SWAP(f"{circuit[0]}-e", f"{circuit[0]}-e+{round + 2}", efficient=True)
                        qc.SWAP(f"{circuit[1]}-e", f"{circuit[1]}-e+{round + 2}", efficient=True)
                        bell_result_3 = qc.single_selection(CNOT_gate, f"{circuit[0]}-e", f"{circuit[1]}-e",
                                                            f"{circuit[0]}-e+{round + 2}",
                                                            f"{circuit[1]}-e+{round + 2}")
                        if type(bell_result_3) == SKIP:
                            break
                        if bell_result_3 is False:
                            continue
                        qc.SWAP(f"{circuit[0]}-e", f"{circuit[0]}-e+{round + 2}", efficient=True)
                        qc.SWAP(f"{circuit[1]}-e", f"{circuit[1]}-e+{round + 2}", efficient=True)
                        bell_result_4 = qc.measure([f"{circuit[1]}-e", f"{circuit[0]}-e"])
                        if type(bell_result_4) == SKIP:
                            break
                        bell_success = bell_result_4[0] == bell_result_4[1]
                    else:
                        bell_success = True
                if round == 2:
                    if prot in ['medium', 'refined']:
                        qc.SWAP(f"{circuit[0]}-e", f"{circuit[0]}-e+3", efficient=True)
                        qc.SWAP(f"{circuit[1]}-e", f"{circuit[1]}-e+3", efficient=True)
                    outcomes.append(qc.single_selection(CZ_gate, f"{circuit[0]}-e", f"{circuit[1]}-e",
                                                        f"{circuit[0]}-e+{offset}", f"{circuit[1]}-e+{offset}",
                                                        create_bell_pair=False))
        qc.start_sub_circuit("ABCD", forced_level=True)
        ghz_success = outcomes[0] == outcomes[1] if not qc.cut_off_time_reached else True
        if not ghz_success:
            continue
        if not qc.cut_off_time_reached and not outcomes[1]:
            qc.X(f"A-e+{offset}")
            qc.X(f"B-e+{offset}")


def basic_medium_refined_protocol(qc: QuantumCircuit, version=1, prot='basic'):
    ghz_success_main = False
    while not ghz_success_main:
        basic_medium_refined_ghz_block(qc, offset=1, prot=prot)  # Create a GHZ state between A-e+1, B-e+1, C-e+1 and D-e+1
        if not qc.cut_off_time_reached:
            basic_medium_refined_ghz_block(qc, offset=2, prot=prot)  # Create a GHZ state between A-e+2, B-e+2, C-e+2 and D-e+2
        if not qc.cut_off_time_reached:
            meas_res = []
            qc.start_sub_circuit("ABCD", forced_level=True)
            for i_n, node in enumerate(["B", "A", "D", "C"]):
                if version == 1: # The version number indicates which GHZ state is consumed with the distillation step
                    qc.SWAP(f"{node}-e", f"{node}-e+1", efficient=True)
                    qc.CNOT(f"{node}-e", f"{node}-e+2")
                else:
                    qc.SWAP(f"{node}-e", f"{node}-e+2", efficient=True)
                    qc.CNOT(f"{node}-e", f"{node}-e+1")
                result = qc.measure(f"{node}-e")
                if type(result) != SKIP:
                    result = result[0]
                meas_res.append(result)
            ghz_success_main = meas_res.count(1) % 2 == 0 if SKIP() not in meas_res else True
        else:
            break

    return ["B", "A", "D", "C"]


def basic1_swap(qc: QuantumCircuit, *, operation):
    return basic_medium_refined_protocol(qc, version=1, prot='basic')


def basic2_swap(qc: QuantumCircuit, *, operation):
    return basic_medium_refined_protocol(qc, version=2, prot='basic')


def medium1_swap(qc: QuantumCircuit, *, operation):
    return basic_medium_refined_protocol(qc, version=1, prot='medium')


def medium2_swap(qc: QuantumCircuit, *, operation):
    return basic_medium_refined_protocol(qc, version=2, prot='medium')


def refined1_swap(qc: QuantumCircuit, *, operation):
    return basic_medium_refined_protocol(qc, version=1, prot='refined')


def refined2_swap(qc: QuantumCircuit, *, operation):
    return basic_medium_refined_protocol(qc, version=2, prot='refined')
