"""
2022-2024 Sebastian de Bone (QuTech)
https://github.com/sebastiandebone/ghz_prot_II
_____________________________________________
"""
import pandas as pd
import math
from math import sqrt, pi
from termcolor import colored
import scipy.integrate as integrate
import numpy as np

from circuit_simulation.stabilizer_measurement_protocols.run_protocols import \
    update_result_files as update_result_files_so
from oopsc.superoperator.superoperator import Superoperator
from circuit_simulation._superoperator.superoperator_methods import calculate_weighted_sum
from oopsc.threshold.plot import plot_thresholds as plot_fit
from oopsc.threshold.fit import zoomed_in_fit_thresholds as zoomed_fit
import analysis.calculate_thresholds as ct


def yield_settings(explore_new_protocols, set_name, set_for_which_prots_are_found, cut_off_dep_plot_data=None,
                   plot_type=None):
    if plot_type in ["link_efficiency", "bell_quality", "bell_succ_prob", "cot_dependence_subplots",
                     "cot_dependence_one_figure"]:
        for round_yield in [1, 2]:
            for explore_new_protocols_yield in [True, False]:
                if round_yield == 2 and explore_new_protocols_yield is False:
                    continue
                elif (round_yield == 2 or explore_new_protocols_yield is False) \
                        and plot_type in ["cot_dependence_subplots", "cot_dependence_one_figure"]:
                    continue
                else:
                    protocol_found_sets = ["sIIIr", "sIIId", "sIIIq", "sIIIc", "sIIIe", "sIIIf", "sIIIg",
                                           "sVr", "sVq", "sVp", "sVa", "sVb", "sVc", "sVd", "sVe",
                                           "sVIk", "sVIg", "sVIm", "sVIh"]
                    if plot_type == "link_efficiency":
                        associated_sets = ["Set3r", "Set3d", "Set3q", "Set3p", "Set3c", "Set3e", "Set3f", "Set3g",
                                           "Set3m", "Set3h", "Set3k"]
                        # protocol_found_sets = ["sIIIr", "sIIId", "sIIIq", "sIIIp", "sIIIc", "sIIIe", "sIIIf", "sIIIg"]
                    elif plot_type == "bell_succ_prob":
                        # associated_sets = ["Set6d", "Set6e", "Set6f", "Set6k", "Set6g", "Set6m", "Set6h", "Set3e"]
                        # protocol_found_sets = ["sVId", "sVIe", "sVIf", "sVIk", "sVIg", "sVIm", "sVIh", "sIIIe"]
                        associated_sets = ["Set6k", "Set6g", "Set6m", "Set6h", "Set3e"]
                        # protocol_found_sets = ["sVIk", "sVIg", "sVIm", "sVIh", "sIIIe"]
                    elif plot_type == "bell_quality":
                        associated_sets = ["Set5r", "Set5q", "Set5p", "Set5a", "Set5b", "Set5c", "Set5d", "Set5e"]
                        # protocol_found_sets = ["sVr", "sVq", "sVp", "sVa", "sVb", "sVc", "sVd", "sVe"]
                    # for set_name_yield in ["Set3d", "Set3c", "Set3e", "Set3f", "Set3g", "Set3m", "Set3h", "Set3k"]:
                    elif plot_type in ["cot_dependence_subplots", "cot_dependence_one_figure"]:
                        associated_sets = ["Set3c", "Set3g", "Set3g", "Set5p"]  # , "Set5p"]
                        protocol_found_sets = ["sIIIc", None, None, "sVp"]  # , "sVp"]
                        cut_off_dep_plot_data_set = [
                            ("simv3_sIIIc_4_7_1", 0, r"\eta^{\ast}_\mathrm{link}=8\cdot10^2", "Fig. 7"),
                            ("expedient", 1, r"\eta^{\ast}_\mathrm{link}=2\cdot10^5", "Fig. 7"),
                            ("refined2", 2, r"\eta^{\ast}_\mathrm{link}=2\cdot10^5", "Fig. 7"),
                            ("simv2_sVp_4_14_1", 3, r"F_\mathrm{link}\approx0.828", "Fig. 6")]  # ,
                        # ("simv2_sVp_4_15_1", 4, r"F_\mathrm{link}\approx0.828", "Fig. 6")]
                        for combo in zip(associated_sets, protocol_found_sets, cut_off_dep_plot_data_set):
                            set_name_yield = combo[0]
                            set_for_which_prots_are_found_yield = combo[1]
                            explore_new_protocols_yield_co = False if set_for_which_prots_are_found_yield is None else True
                            cut_off_dep_plot_data_yield = combo[2]
                            yield explore_new_protocols_yield_co, set_name_yield, set_for_which_prots_are_found_yield, round_yield, cut_off_dep_plot_data_yield
                        continue
                    for set_name_yield in associated_sets:
                        if explore_new_protocols_yield is True:
                            for set_for_which_prots_are_found_yield in protocol_found_sets:
                                yield explore_new_protocols_yield, set_name_yield, set_for_which_prots_are_found_yield, round_yield, cut_off_dep_plot_data
                        else:
                            yield explore_new_protocols_yield, set_name_yield, set_for_which_prots_are_found, round_yield, cut_off_dep_plot_data
    else:
        if set_for_which_prots_are_found == "all_sV":
            protocol_found_sets = ["sVr", "sVq", "sVp", "sVa", "sVb", "sVc", "sVd", "sVe"]
        elif set_for_which_prots_are_found == "all_sets":
            protocol_found_sets = ["sIIIr", "sIIId", "sIIIq", "sIIIc", "sIIIe", "sIIIf", "sIIIg",
                                   "sVr", "sVq", "sVp", "sVa", "sVb", "sVc", "sVd", "sVe",
                                   "sVIk", "sVIg", "sVIm", "sVIh"]
        else:
            protocol_found_sets = [set_for_which_prots_are_found]
        round_yield = 1
        for set_prots_found in protocol_found_sets:
            yield explore_new_protocols, set_name, set_prots_found, round_yield, cut_off_dep_plot_data


def yield_protocols(n, set_name, explore_new_protocols, set_for_which_prots_are_found, max_prots_per_n,
                    max_prots_per_n_k, decoder, **kwargs):
    if explore_new_protocols:
        for version in ['simv1', 'simv2', 'simv3', 'simv4', 'simv5', 'simv6', 'simv7', 'simv8', 'simv8T']:
            for k in range(5, max_prots_per_n + 1):
                for b in range(1, max_prots_per_n_k + 1):
                    if 1 == 1:
                        prot_name_short = f"{version}_{set_for_which_prots_are_found}_{n}_{k}_{b}"
                        prot_name = f'dyn_prot_{version}_{set_for_which_prots_are_found}_{n}_{k}_{b}'
                        file_name = "recipe_" + prot_name + "_swap_" + set_name + "_toric_graph_3D_" + decoder + "_thres.csv"
                        yield prot_name_short, prot_name, file_name
    else:
        for prot in ['plain', 'modicum', 'basic1', 'basic2', 'medium1', 'medium2', 'minimum4x_22', 'expedient',
                     'refined1', 'refined2', 'minimum4x_40_1', 'minimum4x_40_2', 'stringent']:
            # for prot in ['plain', 'modicum', 'basic1', 'basic2', 'medium1', 'medium2', 'expedient', 'refined1',
            #              'refined2', 'stringent', 'direct_ghz']:
            # for prot in ['basic1', 'medium2', 'minimum4x_22', 'expedient', 'refined1', 'minimum4x_40_1', 'stringent']:
            prot_name_short = prot
            prot_name = "hc_" + prot
            file_name = prot + "_swap_" + set_name + "_toric_graph_3D_" + decoder + "_thres.csv"
            yield prot_name_short, prot_name, file_name


def get_threshold_from_dict(item):
    first_cot = list(item[1].keys())[0]
    return item[1][first_cot]['threshold'][0]


def prefactor(x):
    value = 1
    if x % 2 == 0:
        value /= 2*sqrt(x)
        for y in range(2, x):
            if y % 2 == 1:
                value *= y
            else:
                value /= y
    else:
        value /= pi*sqrt(x)
        for y in range(2, x):
            if y % 2 == 0:
                value *= y
            else:
                value /= y
    return value


def closest_value(input_list, input_value):
    arr = np.asarray(input_list)
    i = (np.abs(arr - input_value)).argmin()
    return arr[i], i


def t_distribution(nu, x):
    return prefactor(nu)*(1+x**2/nu)**(-(nu+1)/2)


def find_t(p=0.95, df=30, start_t=1.5, end_t=5, number_tries=10000):
    results = []
    t_values = []
    for x in range(number_tries):
        t = start_t + x * (end_t - start_t) / number_tries
        result = integrate.quad(lambda x: t_distribution(df, x), -1 * t, t)
        results.append(result[0])
        t_values.append(t)
    clos_val, t = closest_value(results, p)
    # print(f"t-value determined at {t_values[t]}, for df {df}.")
    return clos_val, t_values[t]


def calculate_confidence_interval(samples, p=0.95, start_t=1.9, end_t=5, number_tries=10000, t=None):
    average = np.average(samples)
    n = len(samples)
    error_margin = 0
    for sample in samples:
        error_margin += (sample - average) ** 2
    if t is None:
        t = find_t(p=p, df=n-1, start_t=start_t, end_t=end_t, number_tries=number_tries)[1]
    full_error_margin = t * sqrt(error_margin / (n * (n - 1))) if n > 1 else t * sqrt(error_margin / (n * n))
    return full_error_margin


def define_logical_error_rates_df(
        include_so_stats_dec=False, include_so_stats_stab_fids=False,
        include_so_stats_weighted_sum=False, calculate_cut_off_times=False, **kwargs
):
    columns = ["protocol", "cut_off_time", "est_GHZ_success_rate", "p_g", "GHZ_success_rate",
               "below_threshold", "possibly_below_threshold", "possibly_above_threshold",
               "order", "success_rates", "number_of_iterations"]
    re_order_df_columns = 0
    if include_so_stats_dec:
        re_order_df_columns += 7
        columns += ["avg_prot_time", "link_gen_perc", "dec_T1n_link", "dec_T1n_idle", "dec_T2n_link",
                    "dec_T2n_idle", "n_DD", "written_to"]
        # re_order_threshold_df_columns += 5
        # columns += ["link_gen_perc", "dec_T1n_link", "dec_T1n_idle", "dec_T2n_link", "dec_T2n_idle"]
    if include_so_stats_stab_fids:
        re_order_df_columns += 2
        columns += ["stab_fid_p", "stab_fid_s"]
    if include_so_stats_weighted_sum:
        re_order_df_columns += 12
        columns += ["weighted_sum", "success_rate_12", "success_rate_12_int", "success_rate_4",
                    "weighted_sum_p_x", "weighted_sum_p_z", "weighted_sum_p_y", "weighted_sum_p_meas",
                    "weighted_sum_s_x", "weighted_sum_s_z", "weighted_sum_s_y", "weighted_sum_s_meas"]
    dataframe = pd.DataFrame(columns=columns)
    if calculate_cut_off_times is False:
        dataframe.drop(columns="est_GHZ_success_rate", inplace=True)
        dataframe = dataframe.set_index(['protocol', 'cut_off_time', 'p_g'])
    else:
        dataframe = dataframe.set_index(['protocol', 'cut_off_time', 'est_GHZ_success_rate', 'p_g'])
    return dataframe, re_order_df_columns


def get_folder_name(
        review_old_results, explore_new_protocols_yield, set_name_yield,
        set_for_which_prots_are_found, set_for_which_prots_are_found_yield,
        folder, folder_known_protocols, nDD, add_cut_off_to_location=False, **kwargs
):
    folder = folder + "Old\\Wrong_network_architecture\\" if review_old_results is True else folder
    if explore_new_protocols_yield:
        if add_cut_off_to_location:
            folder_stats = f"{folder}Thresholds_best_prots_{set_for_which_prots_are_found_yield}_vary_cut_off{'_nDD_' + str(nDD) if nDD is not None else ''}\\"
            folder_rel = f"Thresholds_best_prots_{set_for_which_prots_are_found_yield}_vary_cut_off{'_nDD_' + str(nDD) if nDD is not None else ''}\\threshold_sim\\"
        elif set_for_which_prots_are_found == "all_sV":
            folder_stats = f"{folder}Thresholds_best_prots_{set_for_which_prots_are_found}_in_{set_name_yield}\\"
            folder_rel = f"Thresholds_best_prots_{set_for_which_prots_are_found}_in_{set_name_yield}\\threshold_sim\\"
        else:
            folder_stats = f"{folder}Thresholds_best_prots_{set_for_which_prots_are_found_yield}{'_nDD_' + str(nDD) if nDD is not None else ''}\\"
            folder_rel = f"Thresholds_best_prots_{set_for_which_prots_are_found_yield}{'_nDD_' + str(nDD) if nDD is not None else ''}\\threshold_sim\\"
        # folder_stats = f"{folder}Old\\Compare_different_network_architectures_qsurface\\Thresholds_best_prots_sIIIc_new_fit_mi\\"
        # folder_rel = f"Old\\Compare_different_network_architectures_qsurface\\Thresholds_best_prots_sIIIc_new_fit_mi\\threshold_sim\\"
        # folder_stats = f"{folder}Paper_test_comparisons\\Test_vary_nDD\\Thresholds_test_{set_name}_nDD_{nDD}\\"
        # folder_rel = f"Paper_test_comparisons\\Test_vary_nDD\\Thresholds_test_{set_name}_nDD_{nDD}\\threshold_sim\\"
        # folder_stats = f"{folder}Paper_test_comparisons\\Test_isotopically_purified_set10n_idle_coh_times\\Thresholds_best_prots_{set_for_which_prots_are_found_yield}_improved_isotopically_purified_T2_50\\"
        # folder_rel = f"Paper_test_comparisons\\Test_isotopically_purified_set10n_idle_coh_times\\Thresholds_best_prots_{set_for_which_prots_are_found_yield}_improved_isotopically_purified_T2_50\\threshold_sim\\"
        # folder_stats = f"{folder}Paper_test_comparisons\\Thresholds_best_prots_{set_for_which_prots_are_found_yield}_eta_link_test\\"
        # folder_rel = f"Paper_test_comparisons\\Thresholds_best_prots_{set_for_which_prots_are_found_yield}_eta_link_test\\threshold_sim\\"

    elif review_old_results is False:
        if add_cut_off_to_location:
            folder_stats = f"{folder}{folder_known_protocols}_in_{set_name_yield}_vary_cut_off\\"
            folder_rel = f"{folder_known_protocols}_in_{set_name_yield}_vary_cut_off\\threshold_sim\\"
        else:
            folder_stats = f"{folder}{folder_known_protocols}_in_{set_name_yield}\\"
            folder_rel = f"{folder_known_protocols}_in_{set_name_yield}\\threshold_sim\\"
        # folder_stats = f"{folder}{folder_known_protocols}\\"
        # folder_rel = f"{folder_known_protocols}\\threshold_sim\\"
        # folder_rel = f"{csv_folder_name_new_prots}\\threshold_sim\\"
        # folder_stats = f"{folder}{csv_folder_name_new_prots}\\"
    else:
        folder_stats = f"{folder}{folder_known_protocols}\\"
        folder_rel = f"{folder_known_protocols}\\threshold_sim\\"
    return folder_stats, folder_rel


def fill_logical_error_rates_df_dataframe(
        threshold_df, threshold_info,
        prot_name, prot_name_short, set_name, folder_stats,
        update_csv_files=False,
        nDD=None,
        include_so_stats_dec=False,
        include_so_stats_stab_fids=False,
        include_so_stats_weighted_sum=False,
        **kwargs,
):
    cut_off_times = list(threshold_info.keys())
    ghz_success_rates = {}
    for cut_off_time in cut_off_times:
        p_g_value = list(threshold_info[cut_off_time].keys())[0]
        ghz_success_rates[threshold_info[cut_off_time][p_g_value]["GHZ_success"]] = cut_off_time
    for cut_off_time in cut_off_times:
        for p_g_value in list(threshold_info[cut_off_time].keys()):
            index_key = (prot_name_short, cut_off_time, p_g_value)
            threshold_df.sort_index(inplace=True)
            threshold_df.loc[index_key, "below_threshold"] = threshold_info[cut_off_time][p_g_value]["Below_threshold"]
            threshold_df.sort_index(inplace=True)
            threshold_df.loc[index_key, "possibly_below_threshold"] = threshold_info[cut_off_time][p_g_value]["Might_be_below_threshold"]
            threshold_df.sort_index(inplace=True)
            threshold_df.loc[index_key, "possibly_above_threshold"] = threshold_info[cut_off_time][p_g_value]["Might_be_above_threshold"]
            threshold_df.sort_index(inplace=True)
            threshold_df.loc[index_key, "order"] = list(threshold_info[cut_off_time][p_g_value]["Success_rates"].keys())
            threshold_df.sort_index(inplace=True)
            threshold_df.loc[index_key, "success_rates"] = [round(r[0], 5) for r in list(threshold_info[cut_off_time][p_g_value]["Success_rates"].values())]
            threshold_df.sort_index(inplace=True)
            threshold_df.loc[index_key, "GHZ_success_rate"] = threshold_info[cut_off_time][p_g_value]["GHZ_success"]
            threshold_df.sort_index(inplace=True)
            if isinstance(threshold_info[cut_off_time][p_g_value]["Number_of_iterations"], dict):
                threshold_df.loc[index_key, "number_of_iterations"] = list(
                    threshold_info[cut_off_time][p_g_value]["Number_of_iterations"].values())
            else:
                threshold_df.loc[index_key, "number_of_iterations"] = threshold_info[cut_off_time][p_g_value]["Number_of_iterations"]
            threshold_df.sort_index(inplace=True)

            if include_so_stats_stab_fids \
                    or include_so_stats_dec \
                    or include_so_stats_weighted_sum:
                max_tries = 2
                found_supop = False
                try_nr = 1
                while found_supop is False and try_nr <= max_tries:
                    if try_nr == 1:
                        file_prefix = prot_name + "_node-s" + set_name[3:] + f"_p_g-{p_g_value}_p_m-{p_g_value}"
                    elif nDD is not None:
                        file_prefix = prot_name + "_node-s" + set_name[3:] + f"_n_DD-{float(nDD)}" + f"_p_g-{p_g_value}_p_m-{p_g_value}"
                    else:
                        found_supop = False
                        break
                    if "s" + set_name[3:] in ["s3h", "s3k"]:
                        supop_file = file_prefix + "_no-dec_no-prob"
                    elif cut_off_time != np.infty:
                        supop_file = file_prefix + f"_cut_off_time-{cut_off_time}"
                    else:
                        supop_file = file_prefix
                    if update_csv_files:
                        update_result_files_so(folder_stats, supop_file, print_actions=False)
                    try:
                        df_supop = pd.read_csv(folder_stats + supop_file + ".csv", sep=';',
                                               float_precision='round_trip')
                        found_supop = True
                    except FileNotFoundError:
                        found_supop = False
                    try_nr += 1

                if found_supop:
                    index = ['error_config', 'lie'] if 'error_idle' not in df_supop else ['error_stab', 'error_idle',
                                                                                          'lie']
                    df_supop = df_supop.set_index(index)
                    if include_so_stats_dec:
                        average_prot_time = df_supop.loc[('IIII', False), 'avg_duration']
                        total_link_attempts = df_supop.loc[('IIII', False), 'total_link_attempts']
                        total_iterations = df_supop.loc[('IIII', False), 'written_to']
                        link_attempt_time = df_supop.loc[('IIII', False), 't_link']
                        T1n_link = df_supop.loc[('IIII', False), 'T1n_link']
                        T1n_idle = df_supop.loc[('IIII', False), 'T1n_idle']
                        T2n_link = df_supop.loc[('IIII', False), 'T2n_link']
                        T2n_idle = df_supop.loc[('IIII', False), 'T2n_idle']
                        nDD_data = df_supop.loc[('IIII', False), 'n_DD']
                        average_link_generating_time = total_link_attempts / total_iterations / 2 * link_attempt_time
                        threshold_df.loc[index_key, "avg_prot_time"] = average_prot_time
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "link_gen_perc"] = average_link_generating_time / average_prot_time * 100
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "dec_T1n_link"] = 1 - math.exp(-average_link_generating_time / T1n_link)
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "dec_T1n_idle"] = 1 - math.exp(-(average_prot_time - average_link_generating_time) / T1n_idle)
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "dec_T2n_link"] = 1 - math.exp(-average_link_generating_time / T2n_link)
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "dec_T2n_idle"] = 1 - math.exp(-(average_prot_time - average_link_generating_time) / T2n_idle)
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "n_DD"] = nDD_data
                        threshold_df.sort_index(inplace=True)
                    if include_so_stats_stab_fids:
                        stab_fid_p = df_supop.iloc[0, df_supop.columns.get_loc('p')]
                        stab_fid_s = df_supop.iloc[0, df_supop.columns.get_loc('s')]
                        threshold_df.loc[index_key, "stab_fid_p"] = stab_fid_p
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "stab_fid_s"] = stab_fid_s
                        threshold_df.sort_index(inplace=True)
                    if include_so_stats_weighted_sum:
                        superoperator = Superoperator(folder_stats + supop_file + ".csv")
                        weighted_sum_info = calculate_weighted_sum(superoperator)
                        threshold_df.loc[index_key, "weighted_sum"] = weighted_sum_info['s'][1] + \
                                                                      weighted_sum_info['s'][2] + \
                                                                      weighted_sum_info['s'][3] * 2 + \
                                                                      weighted_sum_info['s'][4] / 2 + \
                                                                      weighted_sum_info['p'][1] + \
                                                                      weighted_sum_info['p'][2] + \
                                                                      weighted_sum_info['p'][3] * 2 + \
                                                                      weighted_sum_info['p'][4] / 2
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "success_rate_12"] = \
                        threshold_info[cut_off_time][p_g_value]["Success_rates"][12][0]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "success_rate_12_int"] = \
                        threshold_info[cut_off_time][p_g_value]["Success_rates"][12][1]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "success_rate_4"] = \
                        threshold_info[cut_off_time][p_g_value]["Success_rates"][4][0]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_p_x"] = weighted_sum_info['p'][1]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_p_z"] = weighted_sum_info['p'][2]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_p_y"] = weighted_sum_info['p'][3]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_p_meas"] = weighted_sum_info['p'][4]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_s_x"] = weighted_sum_info['s'][1]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_s_z"] = weighted_sum_info['s'][2]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_s_y"] = weighted_sum_info['s'][3]
                        threshold_df.sort_index(inplace=True)
                        threshold_df.loc[index_key, "weighted_sum_s_meas"] = weighted_sum_info['s'][4]
                        threshold_df.sort_index(inplace=True)
    return threshold_df, True


def perform_and_store_fits(
        thresholds_dict, thresholds_fit_dict, threshold_info, prot_name_short, threshold_df, i_output_fits,
        show_fits, output_fits, zoom_number, zoomed_in_fit,
        font_size_big_start, font_size_small_start, number_scaling_tries,
        plot_style, interactive_plot,
        **kwargs,
):
    thresholds, GHZ_success_rates = ct.calculate_threshold_per_cut_off_time(threshold_info)
    for cut_off_time in thresholds.keys():
        # Add thresholds calculated manually to the dictionary "threshold_dict":
        thresholds_dict[prot_name_short][cut_off_time]["threshold"] = thresholds[cut_off_time]
        thresholds_dict[prot_name_short][cut_off_time]["GHZ_success_rates"] = GHZ_success_rates[cut_off_time]
        key_rate = min(GHZ_success_rates[cut_off_time].keys(), key=lambda x: abs(x - thresholds[cut_off_time][0]))
        GHZ_rate_found = GHZ_success_rates[cut_off_time][key_rate] if key_rate in GHZ_success_rates[
            cut_off_time].keys() else 0
        points_involved_in_threshold = [index_name[2]
                                        for index_name in threshold_df.index
                                        if (index_name[0] == prot_name_short
                                            and index_name[1] == cut_off_time
                                            and index_name[2] >= thresholds[cut_off_time][1][0]
                                            and index_name[2] <= thresholds[cut_off_time][1][1])]
        iterations_involved_in_threshold = [threshold_df.loc[index_name, "number_of_iterations"]
                                            for index_name in threshold_df.index
                                            if (index_name[0] == prot_name_short
                                                and index_name[1] == cut_off_time
                                                and index_name[2] in points_involved_in_threshold)]
        thresholds_dict[prot_name_short][cut_off_time]["number_of_points"] = len(points_involved_in_threshold)
        thresholds_dict[prot_name_short][cut_off_time]["minimum_number_iterations"] = min(
            iterations_involved_in_threshold) if len(iterations_involved_in_threshold) > 0 else 0

        thresholds_fit_dict = perform_and_store_fits_specific_protocol(
            thresholds_fit_dict, threshold_info, cut_off_time, prot_name_short, GHZ_success_rates,
            points_involved_in_threshold, iterations_involved_in_threshold,
            zoomed_in_fit, zoom_number, show_fits, output_fits, i_output_fits,
            font_size_big_start, font_size_small_start, number_scaling_tries,
            plot_style, interactive_plot, GHZ_rate_found
        )
    return thresholds_dict, thresholds_fit_dict


def perform_and_store_fits_specific_protocol(
        thresholds_fit_dict, threshold_info, cut_off_time, prot_name_short, GHZ_success_rates,
        points_involved_in_threshold, iterations_involved_in_threshold,
        zoomed_in_fit, zoom_number,
        show_fits, output_fits, i_output_fits,
        font_size_big_start, font_size_small_start, number_scaling_tries,
        plot_style, interactive_plot,
        GHZ_rate_found
):
    # Add thresholds found with fitting to a different dictionary "threshold_fit_dict":
    # Construct DataFrame required for the fit function to work:
    fit_dataframe = pd.DataFrame(columns=["L", "p_g", "N", "success"])
    fit_dataframe = fit_dataframe.set_index(["L", "p_g"])
    for p_g_value in threshold_info[cut_off_time].keys():
        for lattice_size in sorted(threshold_info[cut_off_time][p_g_value]["Success_rates"].keys()):
            rate = threshold_info[cut_off_time][p_g_value]["Success_rates"][lattice_size]
            iter = threshold_info[cut_off_time][p_g_value]["Number_of_iterations"]
            if isinstance(iter, dict):
                iter = iter[lattice_size]
            fit_dataframe.loc[(lattice_size, p_g_value), "success"] = rate[0] * iter
            fit_dataframe.loc[(lattice_size, p_g_value), "N"] = iter
    try:
        _, par, perr, chisqred = zoomed_fit(fit_dataframe, modified_ansatz=True, print_results=False, limit_bounds=True,
                                            return_chi_squared_red=True, zoomed_in_fit=zoomed_in_fit,
                                            zoom_number=zoom_number)
    except Exception:
        try:
            _, par, perr, chisqred = zoomed_fit(fit_dataframe, modified_ansatz=True, print_results=False,
                                                limit_bounds=False, return_chi_squared_red=True,
                                                zoomed_in_fit=zoomed_in_fit, zoom_number=zoom_number)
        except Exception:
            try:
                _, par, perr, chisqred = zoomed_fit(fit_dataframe, modified_ansatz=True, print_results=False,
                                                    limit_bounds=True, return_chi_squared_red=True,
                                                    zoomed_in_fit=zoomed_in_fit, zoom_number=zoom_number,
                                                    ultimate_try=True)
            except Exception:
                par, perr, chisqred = [0], 0, 0
    degrees_freedom = len(fit_dataframe.index) - len(par)
    tci = find_t(p=0.95, df=degrees_freedom)[1]
    if show_fits or output_fits:
        show_fit_itself = True if par[0] != 0 else False
        output_fits_i = output_fits + f"_{i_output_fits}" if output_fits else ""
        font_size_big_fit = font_size_big_start if output_fits else 20
        font_size_small_fit = font_size_small_start if output_fits else 20 * 0.8
        prot_name_plot = prot_name_short.replace('_', '-') if prot_name_short != "simv3_sIIIc_4_7_1" else r"Septimum ($k=7$)"
        plot_fit(
            modified_ansatz=True, data=fit_dataframe, interactive_plot=interactive_plot,
            plot_title=f"Threshold for {prot_name_plot}" +
                       # "\n" +
                       # r"at $\eta^{\ast}_\mathrm{link}=2\cdot10^3$ in Fig. 7, " +
                       " for " +
                       r"$t_\mathrm{GHZ}\approx $" +
                       f"{round(cut_off_time, 3)} s" +
                       r" ($p_\mathrm{GHZ}\approx $" +
                       f"{round(GHZ_rate_found, 3) * 100}%)",
            par=par, perr=perr * tci,
            accuracy=None, include_fit=show_fit_itself, include_rescaled_error_rates=False,
            chi_squared_red=chisqred, plotstyle=plot_style,
            font_size_big=font_size_big_fit, font_size_small=font_size_small_fit,
            number_scaling_tries=number_scaling_tries,
            output=output_fits_i, show_plot=show_fits
        )
        i_output_fits += 1
    thresholds_fit_dict[prot_name_short][cut_off_time]["threshold"] = par[0], (
        par[0] - perr * tci, par[0] + perr * tci)
    thresholds_fit_dict[prot_name_short][cut_off_time]["GHZ_success_rates"] = GHZ_success_rates[cut_off_time]
    thresholds_fit_dict[prot_name_short][cut_off_time]["number_of_points"] = len(points_involved_in_threshold)
    thresholds_fit_dict[prot_name_short][cut_off_time]["minimum_number_iterations"] = min(
        iterations_involved_in_threshold) if len(iterations_involved_in_threshold) > 0 else 0
    return thresholds_fit_dict


def print_logical_error_rates_df(
        threshold_df, re_order_threshold_df_columns, output_threshold_success_rates, **kwargs,
):
    print(f"\n{colored('All surface code data:', 'grey', 'on_white', attrs=['bold'])}")
    df_columns = threshold_df.columns.tolist()
    threshold_df = threshold_df[
        df_columns[(-1 * re_order_threshold_df_columns):] + df_columns[:(-1 * re_order_threshold_df_columns)]]
    if output_threshold_success_rates:
        threshold_df.to_csv("results/threshold_df.csv")
    print(threshold_df.to_string(formatters={'__index__': lambda x: str(x) if isinstance(x, float) else x}))
    return threshold_df


def print_and_save_calculated_thresholds(
        thresholds_fit_dict, protocol_information,
        best_thresholds_per_set, best_prots_found, best_new_prot_list,
        set_name_yield, explore_new_protocols_yield, round_yield,
        best_new_prot_per_set,
        protocols_calculated_per_set,
        print_results=True,
):
    for dict_used in [thresholds_fit_dict]:  # [thresholds_dict, thresholds_fit_dict]:
        if dict_used:
            if print_results:
                print(f"\n\n{colored('Thresholds found:', 'grey', 'on_white', attrs=['bold'])}")  # bold, dark, underline, blink, reverse, concealed
            dict_used = {prot: dict(sorted(dict_used[prot].items(), key=lambda x: x[1]['threshold'][0], reverse=True))
                         for prot in dict_used.keys()}
            dict_used = dict(sorted(dict_used.items(), key=get_threshold_from_dict, reverse=True))
            for prot in dict_used.keys():
                identical_prots_in_best_prots = []
                similar_prots_in_best_prots = []
                try:
                    identical_protocols = protocol_information[prot]["identical"]
                    similar_protocols = protocol_information[prot]["similar"]
                    for identical_protocol in identical_protocols:
                        if identical_protocol in best_prots_found:
                            identical_prots_in_best_prots.append(identical_protocol)
                    for similar_protocol in similar_protocols:
                        if similar_protocol in best_prots_found:
                            similar_prots_in_best_prots.append(similar_protocol)
                except:
                    pass
                # dict_used[prot] = dict(sorted(dict_used[prot].items(), key=lambda item: item[0]))
                for cut_off_time in dict_used[prot].keys():
                    threshold = dict_used[prot][cut_off_time]['threshold']
                    GHZ_rates = dict_used[prot][cut_off_time]['GHZ_success_rates']
                    key_to_take = min(GHZ_rates.keys(), key=lambda x: abs(x - threshold[0]))
                    GHZ_rate_threshold = GHZ_rates[key_to_take] if key_to_take in GHZ_rates.keys() else 0
                    errorrates = round(threshold[1][0], 6), round(threshold[1][1], 6)
                    if print_results:
                        print(f"{prot}, {cut_off_time} ({colored('{:1.5f}'.format(GHZ_rate_threshold), 'blue')}): "
                              f"{colored(round(threshold[0], 6), 'red', attrs=['bold'])} {colored(errorrates, 'red')}. "
                              f"Numb. points involved: {colored(dict_used[prot][cut_off_time]['number_of_points'], 'blue')}. "
                              f"Min. iterations: {colored(dict_used[prot][cut_off_time]['minimum_number_iterations'], 'blue')}. "
                              f"Identical protocols in list of best protocols: {identical_prots_in_best_prots}. "
                              f"Similar protocols in list of best protocols: {similar_prots_in_best_prots}.")
                        calculated_thresholds_per_set = ct.update_list_of_calculated_thresholds_per_set(
                            set_name_new=set_name_yield, prot_name_new=prot)
                if print_results:
                    print("")

        if print_results:
            print(f"\n{colored(f'Best (completely calculated) threshold found for this set ({set_name_yield}):', 'grey', 'on_white', attrs=['bold'])}")
        max_threshold = 0
        max_threshold_full = None
        prot_mt = None
        prot_mcot = None
        for prot in dict_used.keys():
            max_threshold_prot = 0
            max_threshold_full_prot = 0
            mcot = None
            for cut_off_time in dict_used[prot].keys():
                if dict_used[prot][cut_off_time]['threshold'][0] > max_threshold \
                        and dict_used[prot][cut_off_time]['threshold'][1][0] > 0 \
                        and dict_used[prot][cut_off_time]['threshold'][1][1] < 1:
                    prot_mt = prot
                    prot_mcot = cut_off_time
                    max_threshold_full = dict_used[prot][cut_off_time]['threshold']
                    max_threshold = max_threshold_full[0]
                if dict_used[prot][cut_off_time]['threshold'][0] > max_threshold_prot \
                        and dict_used[prot][cut_off_time]['threshold'][1][0] > 0 \
                        and dict_used[prot][cut_off_time]['threshold'][1][1] < 1:
                    mcot = cut_off_time
                    max_threshold_full_prot = dict_used[prot][cut_off_time]['threshold']
                    max_threshold_prot = max_threshold_full_prot[0]
            if mcot is not None and (explore_new_protocols_yield is False or
                                     (explore_new_protocols_yield is True and prot in best_new_prot_list
                                      and (prot, mcot, max_threshold_full_prot) not in protocols_calculated_per_set[
                                          set_name_yield])):
                save_info = False
                if explore_new_protocols_yield is False:
                    if set_name_yield in ["Set3h", "Set3k"]:
                        if prot in ['basic2', 'medium2', 'refined2', 'minimum4x_40_2']:
                            prot = prot[:-1] if "minimum4x_40" not in prot else prot[:-2]
                            save_info = True
                        elif prot not in ['basic1', 'medium1', 'refined1', 'minimum4x_40_1']:
                            save_info = True
                    else:
                        if prot in ['basic2', 'medium2', 'refined2', 'minimum4x_40_2']:
                            prot = prot[:-1] if "minimum4x_40" not in prot else prot[:-2]
                            save_info = True
                        elif prot not in ['basic1', 'medium1', 'refined1', 'minimum4x_40_1']:
                            save_info = True
                else:
                    save_info = True
                if save_info is True:
                    if print_results:
                        print(colored(f"Appended {(prot, mcot, max_threshold_full_prot)} for set name {set_name_yield}.", "yellow"))
                    protocols_calculated_per_set[set_name_yield].append((prot, mcot, max_threshold_full_prot))

        if max_threshold_full is not None:
            best_thresholds_per_set[set_name_yield] = [(prot_mt, prot_mcot, max_threshold_full)]
            if print_results:
                print(f"{prot_mt}, {prot_mcot}: {round(max_threshold_full[0], 6)} ({round(max_threshold_full[1][0], 6)}, "
                      f"{round(max_threshold_full[1][1], 6)}).\n")
            if explore_new_protocols_yield is True and round_yield == 1:
                best_new_prot_per_set[set_name_yield] = (prot_mt, prot_mcot)

    return best_thresholds_per_set, best_new_prot_per_set, protocols_calculated_per_set


def load_pre_calculated_data(plot_type=None):
    inf = np.inf
    protocols_to_be_plotted_per_set = None
    cut_off_dict_set_plot = None
    if plot_type == "link_efficiency":
        protocols_to_be_plotted_per_set = {'Set3c': [('basic', 0.0394161, (0.0002912535604712159, (0.00015736867411362073, 0.000425138446828811))), ('simv1_sIIIr_4_6_1', 0.0187441, (0.0008050085254869597, (0.0007652977312892646, 0.0008447193196846548))), ('simv3_sIIIc_4_7_1', 0.0248241, (0.001121848792582397, (0.0010220556913012806, 0.0012216418938635135))), ('simv3_sIIIc_4_7_6', 0.0262921, (0.001101306286956197, (0.0009812022838407014, 0.0012214102900716924))), ('simv4_sIIIc_4_7_14', 0.0274641, (0.0010986062521766424, (0.001003676408290987, 0.001193536096062298))), ('simv5_sIIIf_4_11_2', 0.0564401, (0.00016591276736908518, (0.0001178089661081392, 0.00021401656863003117))), ('simv3_sVc_4_10_4', 0.0284721, (0.0007901625828954803, (0.0006970746263810575, 0.0008832505394099032)))], 'Set3e': [('basic', 0.0382001, (0.0010133700047587194, (0.0009327982693102191, 0.00109394174020722))), ('minimum4x_22', 0.0442801, (0.0005158491002163272, (0.00045318304264418133, 0.000578515157788473))), ('expedient', 0.0661681, (0.00023450261856885219, (0.00016480850724761702, 0.00030419672989008735))), ('simv1_sIIIr_4_6_1', 0.0187441, (0.0011402929415552643, (0.0011220417891078711, 0.0011585440940026575))), ('simv3_sIIIc_4_7_1', 0.0236081, (0.001808573478860709, (0.0017766901741722023, 0.0018404567835492158))), ('simv3_sIIIc_4_7_6', 0.0236081, (0.0017979683246157462, (0.0017594025082338025, 0.00183653414099769))), ('simv4_sIIIc_4_7_14', 0.0274641, (0.001772275575253924, (0.0017101136731642666, 0.0018344374773435813))), ('simv5_sIIIf_4_11_2', 0.0588721, (0.0012465080037756804, (0.0011678502395172622, 0.0013251657680340987))), ('simv3_sVc_4_10_4', 0.0296881, (0.0018133171877497785, (0.0017733736604295088, 0.0018532607150700481)))], 'Set3f': [('minimum4x_22', 0.0442801, (0.0017592833424916006, (0.0017147039202514507, 0.0018038627647317504))), ('basic', 0.0382001, (0.0014418216977431693, (0.0013758278317199525, 0.001507815563766386))), ('expedient', 0.0686001, (0.0014099878170523003, (0.0013560737729891297, 0.001463901861115471))), ('medium', 0.0783281, (0.0009031308096932074, (0.0008203211355515595, 0.0009859404838348552))), ('refined', 0.1162963, (0.0008826041667199956, (0.0007976667540840397, 0.0009675415793559515))), ('minimum4x_40', 0.0941012, (0.0007779909864811638, (0.0006921083428341987, 0.0008638736301281288))), ('stringent', 0.1743921, (0.00032535039639465453, (0.0002775532898485434, 0.0003731475029407657))), ('simv1_sIIIr_4_6_1', 0.0187441, (0.0013191465263666705, (0.001301461772826312, 0.0013368312799070291))), ('simv3_sIIIc_4_7_1', 0.0248241, (0.002184897180262896, (0.002144626176419992, 0.0022251681841058))), ('simv4_sIIIc_4_7_14', 0.0274641, (0.0021719077744496327, (0.002136517297132182, 0.0022072982517670836))), ('simv3_sIIIc_4_7_6', 0.0248241, (0.0021698510294745417, (0.0021274952308028053, 0.002212206828146278))), ('simv5_sIIIf_4_11_2', 0.0686001, (0.001953541221818235, (0.0018622097048385463, 0.0020448727387979235))), ('simv3_sVc_4_10_4', 0.0296881, (0.002415734399728291, (0.002376558212963572, 0.00245491058649301)))], 'Set3g': [('minimum4x_22', 0.0442801, (0.0018750796488213675, (0.0018462923494586552, 0.0019038669481840798))), ('expedient', 0.0661681, (0.00150864048474626, (0.0014586471703844032, 0.0015586337991081167))), ('basic', 0.0382001, (0.0014601743419022878, (0.0014106714472183312, 0.0015096772365862444))), ('refined', 0.11555358, (0.0011183421480504459, (0.0010251734716493744, 0.0012115108244515173))), ('minimum4x_40', 0.09496762, (0.0010020923577775055, (0.000877201257467098, 0.0011269834580879131))), ('medium', 0.07743772, (0.0009930082725113786, (0.0009208549032399436, 0.0010651616417828135))), ('stringent', 0.1792561, (0.0005656048992565089, (0.000510001025748853, 0.0006212087727641649))), ('simv1_sIIIr_4_6_1', 0.0187441, (0.001337232934554783, (0.0013117518336566703, 0.0013627140354528957))), ('simv3_sIIIc_4_7_1', 0.0248241, (0.0022197823877677337, (0.0021729075615401503, 0.002266657213995317))), ('simv4_sIIIc_4_7_14', 0.0274641, (0.002212898763223719, (0.0021759536815612866, 0.0022498438448861516))), ('simv3_sIIIc_4_7_6', 0.0248241, (0.0021985883406806083, (0.002166931987735185, 0.0022302446936260318))), ('simv5_sIIIf_4_11_2', 0.06340308, (0.002068398723950525, (0.002013459285202496, 0.0021233381626985536))), ('simv3_sVc_4_10_4', 0.0272561, (0.002403524441777464, (0.0023137731086670327, 0.0024932757748878957)))], 'Set3h': [('stringent', inf, (0.0030926719159396696, (0.00300665486648711, 0.003178688965392229))), ('refined', inf, (0.0025870925342684004, (0.002536471154153283, 0.002637713914383518))), ('minimum4x_40', inf, (0.0024773662430965966, (0.002394483550571871, 0.0025602489356213224))), ('minimum4x_22', inf, (0.002360001769887041, (0.002312812226860952, 0.00240719131291313))), ('expedient', inf, (0.0020635544747414926, (0.0019782807602726706, 0.0021488281892103147))), ('basic', inf, (0.0017313326423497497, (0.001683717481699056, 0.0017789478030004434))), ('medium', inf, (0.001613676190938412, (0.0015748355959919523, 0.0016525167858848719))), ('simv1_sIIIr_4_6_1', inf, (0.0014102880483641265, (0.0013908854968992922, 0.0014296905998289608))), ('simv4_sIIIc_4_7_14', inf, (0.002448842299963221, (0.00241912021068163, 0.002478564389244812))), ('simv3_sIIIc_4_7_6', inf, (0.0024374500238284772, (0.002399702709694824, 0.0024751973379621307))), ('simv3_sIIIc_4_7_1', inf, (0.00235483127926136, (0.0022717791100665095, 0.0024378834484562106))), ('simv5_sIIIf_4_11_2', inf, (0.002672419662149637, (0.002643187635296757, 0.0027016516890025167)))], 'Set3k': [('stringent', inf, (0.00601440160789508, (0.0057229099336595125, 0.006305893282130647))), ('minimum4x_40', inf, (0.005436090213877059, (0.0053322919627976735, 0.005539888464956444))), ('refined', inf, (0.005380112981118627, (0.005289885747794558, 0.005470340214442695))), ('minimum4x_22', inf, (0.005168358643637008, (0.005091817204560112, 0.005244900082713904))), ('expedient', inf, (0.004939702627174901, (0.004859767439793643, 0.005019637814556159))), ('plain', inf, (0.004320269710968336, (0.0008355924324274741, 0.007804946989509199))), ('basic', inf, (0.003655147954589184, (0.0035671064633631564, 0.003743189445815211))), ('medium', inf, (0.0035204278452161774, (0.0034468155601528124, 0.0035940401302795425))), ('simv1_sIIIr_4_6_1', inf, (0.003232544002392771, (0.0031849796711691768, 0.003280108333616365))), ('simv4_sIIIc_4_7_14', inf, (0.004938686064760127, (0.004877554096492128, 0.004999818033028127))), ('simv3_sIIIc_4_7_6', inf, (0.004916490556576346, (0.0048546958828165196, 0.004978285230336172))), ('simv3_sIIIc_4_7_1', inf, (0.004911604191557384, (0.004831454815065121, 0.004991753568049646))), ('simv5_sIIIf_4_11_2', inf, (0.004792715175491646, (0.0046974279719672754, 0.004888002379016016)))], 'Set3r': [('simv1_sIIIr_4_6_1', 0.0187441, (0.0001996196894892424, (0.0001232580675259688, 0.000275981311452516)))], 'Set3d': [('simv1_sIIIr_4_6_1', 0.0187441, (0.000444049164888717, (0.0003807363331928895, 0.0005073619965845445))), ('simv4_sIIId_4_6_17', 0.0187441, (0.00046063578674533737, (0.0003746136651059909, 0.0005466579083846839))), ('simv3_sIIIc_4_7_1', 0.0236081, (0.00045796930762364115, (0.00037947785682522777, 0.0005364607584220545))), ('simv3_sIIIc_4_7_6', 0.0236081, (0.0004497906219954717, (0.000383493878443328, 0.0005160873655476154))), ('simv4_sIIIc_4_7_14', 0.0236081, (0.00041960645951207145, (0.00034169366828472726, 0.0004975192507394156)))], 'Set3q': [('simv1_sIIIr_4_6_1', 0.0187441, (0.0006063295719860612, (0.0005558614175771729, 0.0006567977263949496))), ('simv3_sIIIc_4_7_6', 0.0236081, (0.0007585424224963638, (0.0006991847501963854, 0.0008179000947963422))), ('simv3_sIIIc_4_7_1', 0.0236081, (0.0007336911881945261, (0.0006702940885892379, 0.0007970882877998143))), ('simv4_sIIIc_4_7_14', 0.0236081, (0.0007223599898541096, (0.0006490955576924944, 0.0007956244220157248))), ('simv3_sVc_4_10_4', 0.0272561, (0.00020971427029303769, (0.0001151981656773303, 0.00030423037490874506)))]}
    elif plot_type == "bell_succ_prob":
        protocols_to_be_plotted_per_set = {'Set6g': [('basic', 0.0501361, (4.119424605672646e-06, (1.2971312243597158e-06, 6.941717986985576e-06))), ('simv1_sIIIq_4_6_9', 0.0230641, (0.0006368725264554906, (0.0005794732739825779, 0.0006942717789284032))), ('simv4_sIIIe_4_7_3', 0.0290801, (0.0007689976895498639, (0.0007136831441563166, 0.0008243122349434113))), ('simv3_sVc_4_10_4', 0.0320881, (0.0002581200584785935, (0.0001786129247750348, 0.00033762719218215226)))], 'Set6m': [('basic', 0.0455601, (0.00023859620795924896, (0.00015564095386338823, 0.00032155146205510967))), ('simv1_sIIIq_4_6_9', 0.0216241, (0.0008550073570617018, (0.0008171121227899753, 0.0008929025913334283))), ('simv4_sIIIe_4_7_3', 0.0272561, (0.0011511421048539532, (0.0010868465341166201, 0.0012154376755912862))), ('simv3_sVc_4_10_4', 0.0328881, (0.0008147742775766966, (0.0007530252015810625, 0.0008765233535723307))), ('simv2_sVIm_4_7_6', 0.0286641, (0.001159434774968616, (0.0010509638248445417, 0.0012679057250926903)))], 'Set6h': [('basic', 0.0419201, (0.0006009473089571123, (0.0004875336762080086, 0.000714360941706216))), ('simv1_sIIIq_4_6_9', 0.0205441, (0.0009803919546799323, (0.0009434528592433647, 0.0010173310501165))), ('simv4_sIIIe_4_7_3', 0.0272241, (0.001434230655112338, (0.0013890563101060594, 0.0014794050001186165))), ('simv3_sVc_4_10_4', 0.0298961, (0.0012472273780793853, (0.0011700012939112054, 0.0013244534622475653))), ('simv1_sVIh_4_7_2', 0.0269041, (0.0014483825421433354, (0.001376588426301714, 0.001520176657984957)))], 'Set3e': [('basic', 0.0382001, (0.0010133700047587194, (0.0009327982693102191, 0.00109394174020722))), ('minimum4x_22', 0.0442801, (0.0005158491002163272, (0.00045318304264418133, 0.000578515157788473))), ('expedient', 0.0661681, (0.00023450261856885219, (0.00016480850724761702, 0.00030419672989008735))), ('simv1_sIIIq_4_6_9', 0.0187441, (0.0011833020886558784, (0.001149146458191518, 0.001217457719120239))), ('simv4_sIIIe_4_7_3', 0.0261161, (0.001758219036096265, (0.001681485176444344, 0.0018349528957481858))), ('simv3_sVc_4_10_4', 0.0296881, (0.0018133171877497785, (0.0017733736604295088, 0.0018532607150700481)))], 'Set6k': [('simv1_sIIIq_4_6_9', 0.0248641, (0.0003463742067019622, (0.0003056263770961976, 0.00038712203630772673))), ('simv4_sIIIe_4_7_3', 0.0313601, (0.0002722450359903902, (0.0002161034812808449, 0.0003283865906999355)))]}
    elif plot_type == "bell_quality":
        protocols_to_be_plotted_per_set = {'Set5c': [('minimum4x_22', 0.04770748, (0.00011171032991459728, (8.311878582826577e-05, 0.0001403018740009288))), ('simv3_sIIIc_4_7_1', 0.0309041, (0.0008404750275586297, (0.0007895648938971473, 0.0008913851612201121))), ('simv1_sVr_4_12_9', 0.0369841, (0.0008097549955001449, (0.0007522017489298903, 0.0008673082420703994))), ('simv2_sVp_4_12_1', 0.0309041, (0.0009838050434286003, (0.0009359565672997512, 0.0010316535195574494))), ('simv1_sVc_4_11_8', 0.0345521, (0.001413380467729862, (0.0013688750767407282, 0.001457885858718996))), ('simv3_sVc_4_11_7', 0.0345521, (0.0014072903937405326, (0.0013557358827126296, 0.0014588449047684357))), ('simv3_sVc_4_10_4', 0.0369841, (0.0014055419781167247, (0.0013571481985160932, 0.0014539357577173563))), ('simv1_sVd_4_11_2', 0.0345521, (0.0014066128634444529, (0.0013674175527482425, 0.0014458081741406633)))], 'Set5d': [('minimum4x_22', 0.05013948, (0.00035754848492805736, (0.00030205205398005234, 0.0004130449158760624))), ('basic', 0.0430641, (0.00021922503642695054, (0.00015021293899830917, 0.0002882371338555919))), ('simv3_sIIIc_4_7_1', 0.0248241, (0.0014626670577411404, (0.0014331725899609392, 0.0014921615255213416))), ('simv1_sVr_4_12_9', 0.0333361, (0.0008867820853921249, (0.000826458602021919, 0.0009471055687623308))), ('simv2_sVp_4_12_1', 0.0272561, (0.0011175325045946736, (0.001067287278503035, 0.0011677777306863123))), ('simv3_sVc_4_10_4', 0.0309041, (0.0016648762766653673, (0.0016280498944396156, 0.001701702658891119))), ('simv3_sVc_4_11_7', 0.0321201, (0.0015937797084033464, (0.0015356403872989908, 0.001651919029507702))), ('simv1_sVc_4_11_8', 0.0321201, (0.0015793456482797133, (0.001519410199013999, 0.0016392810975454277))), ('simv1_sVd_4_11_2', 0.0321201, (0.0015856127543969704, (0.001520573228376688, 0.0016506522804172529)))], 'Set5e': [('basic', 0.0357681, (0.0012394835460776822, (0.0011250966371592773, 0.001353870454996087))), ('minimum4x_22', 0.04190158, (0.0005832778334514696, (0.0004867422219357838, 0.0006798134449671555))), ('expedient', 0.0613041, (0.0003737233005360482, (0.00030527192508813785, 0.00044217467598395857))), ('medium', 0.0698161, (0.00028683894124130997, (0.00019778163095784224, 0.0003758962515247777))), ('simv3_sIIIc_4_7_1', 0.0248241, (0.0019324145765431666, (0.0018919513769393105, 0.0019728777761470226))), ('simv1_sVr_4_12_9', 0.0321201, (0.0009496354701819008, (0.0008994411087884865, 0.000999829831575315))), ('simv2_sVp_4_12_1', 0.0236081, (0.0012112207428514239, (0.001177359591563749, 0.0012450818941390988))), ('simv3_sVc_4_10_4', 0.0272561, (0.0018528518559752093, (0.0018094110727947808, 0.0018962926391556379))), ('simv3_sVc_4_11_7', 0.0296881, (0.0017249684377409358, (0.0016804462714196551, 0.0017694906040622166))), ('simv1_sVc_4_11_8', 0.0296881, (0.0017085349641293437, (0.001675467165209353, 0.0017416027630493343))), ('simv1_sVd_4_11_2', 0.0296881, (0.0017306490058168708, (0.0016828821988760505, 0.001778415812757691)))], 'Set5r': [('simv1_sVr_4_12_9', 0.0552241, (6.577833360483307e-05, (4.145277303726576e-05, 9.010389417240038e-05)))], 'Set5q': [('simv1_sVr_4_12_9', 0.0515761, (0.0002868761306665339, (0.00022930619513522076, 0.000344446066197847))), ('simv2_sVp_4_12_1', 0.0479281, (0.00024199859224052653, (0.00018717377530291454, 0.0002968234091781385)))], 'Set5p': [('simv1_sVr_4_12_9', 0.0467121, (0.00046511126652775184, (0.00035934718689280504, 0.0005708753461626986))), ('simv2_sVp_4_12_1', 0.0430641, (0.0004995215447876563, (0.00043231511312089385, 0.0005667279764544188))), ('simv3_sVc_4_11_7', 0.0442801, (0.0003686365168541826, (0.0003166389211784403, 0.0004206341125299249))), ('simv1_sVc_4_11_8', 0.0442801, (0.00036580014490020495, (0.0002961287589885902, 0.00043547153081181973))), ('simv3_sVc_4_10_4', 0.0527921, (0.00021499688527767065, (0.000155233651616311, 0.0002747601189390303))), ('simv1_sVd_4_11_2', 0.0442801, (0.0003639155619563184, (0.0003044264714095807, 0.00042340465250305614)))], 'Set5a': [('simv1_sVr_4_12_9', 0.0430641, (0.0005926613652756151, (0.00046272681966639, 0.0007225959108848401))), ('simv2_sVp_4_12_1', 0.0382001, (0.0006960255027313281, (0.0006187459933833305, 0.0007733050120793257))), ('simv3_sVc_4_11_7', 0.0394161, (0.0008376329630367065, (0.0007839746653122186, 0.0008912912607611944))), ('simv1_sVc_4_11_8', 0.0394161, (0.0008310971463180375, (0.000778870840695286, 0.0008833234519407889))), ('simv3_sVc_4_10_4', 0.0491441, (0.0006980199382104492, (0.0006233680924036098, 0.0007726717840172886))), ('simv1_sVd_4_11_2', 0.0394161, (0.0007705624768073024, (0.0006953834114676202, 0.0008457415421469846)))], 'Set5b': [('simv1_sVr_4_12_9', 0.0394161, (0.0007249616983953364, (0.0006464379914142645, 0.0008034854053764084))), ('simv2_sVp_4_12_1', 0.0345521, (0.0008509358807819589, (0.0008031629479596604, 0.0008987088136042574))), ('simv1_sVc_4_11_8', 0.0369841, (0.0011601577395364275, (0.0011118915039603691, 0.001208423975112486))), ('simv3_sVc_4_11_7', 0.0369841, (0.0011503260568502616, (0.0011022343237070235, 0.0011984177899934997))), ('simv3_sVc_4_10_4', 0.0430641, (0.001093993065795253, (0.0010264126261103179, 0.0011615735054801883))), ('simv1_sVd_4_11_2', 0.0369841, (0.001161683167874297, (0.0011084549468104179, 0.0012149113889381763)))]}
    elif plot_type in ["cot_dependence_one_figure", "cot_dependence_subplots"]:
        cut_off_dict_set_plot = {0: {0.0150961: {'threshold': (0.0005743938988740482, (0.000503583904522898, 0.0006452038932251984)), 'GHZ_success_rates': {0.0004: 0.9132, 0.00055: 0.9105}, 'number_of_points': 6, 'minimum_number_iterations': 50000}, 0.0163121: {'threshold': (0.0007418498451764077, (0.0007029939877914512, 0.0007807057025613643)), 'GHZ_success_rates': {0.0004: 0.9366, 0.0007: 0.9323}, 'number_of_points': 8, 'minimum_number_iterations': 50000}, 0.0164681: {'threshold': (0.0007194873195265122, (0.0006066865764756691, 0.0008322880625773553)), 'GHZ_success_rates': {0.0007: 0.932}, 'number_of_points': 2, 'minimum_number_iterations': 50000}, 0.0175281: {'threshold': (0.0008638314394854218, (0.0008242935227431411, 0.0009033693562277025)), 'GHZ_success_rates': {0.0007: 0.9542, 0.00085: 0.9526}, 'number_of_points': 5, 'minimum_number_iterations': 150000}, 0.0187441: {'threshold': (0.000975985776436757, (0.0009254578146491928, 0.001026513738224321)), 'GHZ_success_rates': {0.0007: 0.9717, 0.00095: 0.9694}, 'number_of_points': 5, 'minimum_number_iterations': 150000}, 0.0199601: {'threshold': (0.0010368568211900563, (0.0009840547737837602, 0.0010896588685963525)), 'GHZ_success_rates': {0.0007: 0.982, 0.00095: 0.9802}, 'number_of_points': 5, 'minimum_number_iterations': 150000}, 0.0211761: {'threshold': (0.0010423153998406747, (0.000988224781940247, 0.0010964060177411025)), 'GHZ_success_rates': {0.0009: 0.9869, 0.00105: 0.986}, 'number_of_points': 4, 'minimum_number_iterations': 150000}, 0.0223921: {'threshold': (0.0011004188676653776, (0.0010575790391057587, 0.0011432586962249966)), 'GHZ_success_rates': {0.00095: 0.9922, 0.0011: 0.9917}, 'number_of_points': 6, 'minimum_number_iterations': 50000}, 0.0236081: {'threshold': (0.001104709609520086, (0.0010502475852307504, 0.0011591716338094217)), 'GHZ_success_rates': {0.00095: 0.9947, 0.00105: 0.9944}, 'number_of_points': 6, 'minimum_number_iterations': 150000}, 0.0248241: {'threshold': (0.001109688681917355, (0.0010568449457351746, 0.0011625324180995354)), 'GHZ_success_rates': {0.0009: 0.9964, 0.0011: 0.996}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.0284721: {'threshold': (0.0010882341402350185, (0.0010453949586948306, 0.0011310733217752063)), 'GHZ_success_rates': {0.0009: 0.9986, 0.00105: 0.9984}, 'number_of_points': 5, 'minimum_number_iterations': 150000}, 0.0309041: {'threshold': (0.0010888649431051482, (0.001044822993070359, 0.0011329068931399374)), 'GHZ_success_rates': {0.0009: 0.9994, 0.00105: 0.9994}, 'number_of_points': 5, 'minimum_number_iterations': 150000}, 0.0369841: {'threshold': (0.0010000253285013577, (0.000936776433303844, 0.0010632742236988714)), 'GHZ_success_rates': {0.00085: 0.9999, 0.001: 0.9999}, 'number_of_points': 7, 'minimum_number_iterations': 100000}, 0.0382001: {'threshold': (0.0009993136492000149, (0.0009389974965369364, 0.0010596298018630934)), 'GHZ_success_rates': {0.00075: 1.0, 0.00095: 1.0}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.0442801: {'threshold': (0.000933881175572018, (0.0008800237461495761, 0.0009877386049944599)), 'GHZ_success_rates': {0.00075: 1.0, 0.0009: 1.0}, 'number_of_points': 7, 'minimum_number_iterations': 100000}}, 1: {0.0406321: {'threshold': (0.0009129720252904336, (0.0008642747878660966, 0.0009616692627147705)), 'GHZ_success_rates': {0.0007: 0.8496, 0.0009: 0.8406}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.0430641: {'threshold': (0.0010148283164787323, (0.00095999590233126, 0.0010696607306262046)), 'GHZ_success_rates': {0.0007: 0.8878, 0.001: 0.8751}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0454961: {'threshold': (0.00114768246047089, (0.0010932997147837307, 0.0012020652061580494)), 'GHZ_success_rates': {0.0007: 0.9154, 0.0011: 0.8981}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0479281: {'threshold': (0.0012670559443590688, (0.0012438217803170924, 0.0012902901084010452)), 'GHZ_success_rates': {0.00095: 0.926, 0.00125: 0.9146}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.0503601: {'threshold': (0.0013395041046610496, (0.0013124187984340804, 0.0013665894108880188)), 'GHZ_success_rates': {0.0012: 0.9332, 0.0013: 0.9297}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0527921: {'threshold': (0.0013896077604916078, (0.0013603350845382816, 0.001418880436444934)), 'GHZ_success_rates': {0.0012: 0.9481, 0.0013: 0.9447}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0552241: {'threshold': (0.0014224740714775506, (0.0013898212734623455, 0.0014551268694927557)), 'GHZ_success_rates': {0.0012: 0.9587, 0.0014: 0.9532}, 'number_of_points': 3, 'minimum_number_iterations': 100000}, 0.0576561: {'threshold': (0.0014623763764196826, (0.0014340366307544021, 0.001490716122084963)), 'GHZ_success_rates': {0.0012: 0.9672, 0.0014: 0.9624}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0613041: {'threshold': (0.0014858751835952135, (0.0014465668291962937, 0.0015251835379941332)), 'GHZ_success_rates': {0.0012: 0.9772, 0.00145: 0.9729}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0649521: {'threshold': (0.0015135593393655416, (0.0014776283811759926, 0.0015494902975550905)), 'GHZ_success_rates': {0.0012: 0.9833, 0.0015: 0.9798}, 'number_of_points': 3, 'minimum_number_iterations': 100000}, 0.0673841: {'threshold': (0.0015153780040470964, (0.0014755253051425572, 0.0015552307029516356)), 'GHZ_success_rates': {0.0012: 0.9873, 0.0015: 0.984}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.0698161: {'threshold': (0.0014950800453340108, (0.001465063963889723, 0.0015250961267782987)), 'GHZ_success_rates': {0.0007: 0.994, 0.001475: 0.9875}, 'number_of_points': 8, 'minimum_number_iterations': 100000}, 0.0722481: {'threshold': (0.0014948764852672524, (0.0014365904675488128, 0.001553162502985692)), 'GHZ_success_rates': {0.0012: 0.9924, 0.0015: 0.9904}, 'number_of_points': 3, 'minimum_number_iterations': 100000}, 0.0795441: {'threshold': (0.0014573395450224401, (0.0014167691077747772, 0.001497909982270103)), 'GHZ_success_rates': {0.0005: 0.9988, 0.00145: 0.9954}, 'number_of_points': 7, 'minimum_number_iterations': 100000}, 0.0831921: {'threshold': (0.0014407461495117146, (0.001406411857029798, 0.001475080441993631)), 'GHZ_success_rates': {0.0005: 0.9994, 0.00135: 0.9972}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.1014321: {'threshold': (0.0012762507875131686, (0.0012270484472763292, 0.0013254531277500081)), 'GHZ_success_rates': {0.0003: 1.0, 0.00125: 0.9995}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.1038641: {'threshold': (0.001244166745910367, (0.0011892505821870685, 0.0012990829096336655)), 'GHZ_success_rates': {0.0001: 1.0, 0.0012: 0.9998}, 'number_of_points': 7, 'minimum_number_iterations': 100000}}, 2: {0.0758961: {'threshold': (0.0007177424993600469, (0.0006614962231737328, 0.0007739887755463609)), 'GHZ_success_rates': {0.0003: 0.8606, 0.00065: 0.8377}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.0783281: {'threshold': (0.0008113961700268459, (0.00076875218638052, 0.0008540401536731719)), 'GHZ_success_rates': {0.00045: 0.8743, 0.0007: 0.86}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.08117396: {'threshold': (0.0008378182621479549, (0.0007896694552839515, 0.0008859670690119582)), 'GHZ_success_rates': {0.00065: 0.8731, 0.0008: 0.8664}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.0844081: {'threshold': (0.000909616362038195, (0.0008609211587329378, 0.0009583115653434523)), 'GHZ_success_rates': {0.00065: 0.8952, 0.0009: 0.8834}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.0868401: {'threshold': (0.0010196094549152694, (0.0009710666157393488, 0.0010681522940911901)), 'GHZ_success_rates': {0.0001: 0.9477, 0.00095: 0.9076}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.09411158: {'threshold': (0.0010803299491927067, (0.0010389902278407639, 0.0011216696705446496)), 'GHZ_success_rates': {0.0001: 0.9674, 0.0011: 0.9258}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.10038958: {'threshold': (0.001118947051354998, (0.001076611782363526, 0.00116128232034647)), 'GHZ_success_rates': {0.0001: 0.9741, 0.0011: 0.9413}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.1099441: {'threshold': (0.0010886735198406845, (0.0010391673990460214, 0.0011381796406353475)), 'GHZ_success_rates': {0.0001: 0.9783, 0.00105: 0.9528}, 'number_of_points': 4, 'minimum_number_iterations': 100000}, 0.1148081: {'threshold': (0.0011117448700772503, (0.0010442818361102002, 0.0011792079040443004)), 'GHZ_success_rates': {0.0001: 0.9832, 0.0011: 0.962}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.12496706: {'threshold': (0.001065966677647953, (0.0009950482344146243, 0.0011368851208812816)), 'GHZ_success_rates': {0.0005: 0.9817, 0.00105: 0.9722}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.13223344: {'threshold': (0.001071989628059657, (0.0010104203785211853, 0.0011335588775981287)), 'GHZ_success_rates': {0.0005: 0.9879, 0.0011: 0.981}, 'number_of_points': 5, 'minimum_number_iterations': 100000}, 0.1395961: {'threshold': (0.0010237078333689327, (0.0009555676550909709, 0.0010918480116468945)), 'GHZ_success_rates': {0.0005: 0.9919, 0.001: 0.9869}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.14804948: {'threshold': (0.0009839475753385557, (0.0009245182445141664, 0.001043376906162945)), 'GHZ_success_rates': {0.0005: 0.9957, 0.001: 0.9915}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.1518883: {'threshold': (0.0009424533261367481, (0.0008854221294227116, 0.0009994845228507847)), 'GHZ_success_rates': {0.0005: 0.9969, 0.001: 0.9931}, 'number_of_points': 7, 'minimum_number_iterations': 100000}, 0.15934372: {'threshold': (0.0008905216468922002, (0.0008243140156462037, 0.0009567292781381967)), 'GHZ_success_rates': {0.0005: 0.9978, 0.0009: 0.9949}, 'number_of_points': 7, 'minimum_number_iterations': 50000}, 0.18245158: {'threshold': (0.000723810368547834, (0.0006635175445174944, 0.0007841031925781736)), 'GHZ_success_rates': {0.0003: 0.9997, 0.0007: 0.9989}, 'number_of_points': 6, 'minimum_number_iterations': 50000}, 0.21783452: {'threshold': (0.00040849241254773354, (0.0003588152684849564, 0.0004581695566105107)), 'GHZ_success_rates': {0.0002: 1.0, 0.0004: 1.0}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.23440386: {'threshold': (0.00022787301894340773, (0.00016401392328445151, 0.00029173211460236395)), 'GHZ_success_rates': {1e-05: 1.0, 0.00025: 1.0}, 'number_of_points': 6, 'minimum_number_iterations': 100000}, 0.26349148: {'threshold': (1.0000000000000002e-06, (-6.558492753819893e-05, 6.758492753819893e-05)), 'GHZ_success_rates': {1e-06: 1.0}, 'number_of_points': 9, 'minimum_number_iterations': 100000}}, 3: {0.0345521: {'threshold': (0.0001574728196519305, (0.0001177159925421144, 0.0001972296467617466)), 'GHZ_success_rates': {7.5e-05: 0.9133, 0.0002: 0.9111}, 'number_of_points': 11, 'minimum_number_iterations': 100000}, 0.0382001: {'threshold': (0.0003175921281470081, (0.0002960599592352736, 0.00033912429705874257)), 'GHZ_success_rates': {0.000125: 0.9424, 0.000315: 0.9396}, 'number_of_points': 23, 'minimum_number_iterations': 100000}, 0.0406321: {'threshold': (0.0003756529027387934, (0.0003392292078322504, 0.0004120765976453364)), 'GHZ_success_rates': {0.000225: 0.9556, 0.0004: 0.9538}, 'number_of_points': 21, 'minimum_number_iterations': 100000}, 0.0418481: {'threshold': (0.0004087867484949176, (0.00036941595694321277, 0.0004481575400466224)), 'GHZ_success_rates': {0.000275: 0.9611, 0.00041: 0.9601}, 'number_of_points': 17, 'minimum_number_iterations': 100000}, 0.0430641: {'threshold': (0.00043255006653365117, (0.00038997259324096675, 0.0004751275398263356)), 'GHZ_success_rates': {0.000275: 0.967, 0.000455: 0.9655}, 'number_of_points': 14, 'minimum_number_iterations': 100000}, 0.0467121: {'threshold': (0.00046000000000000007, (0.0003976710097634948, 0.0005223289902365054)), 'GHZ_success_rates': {0.000325: 0.9792, 0.00047: 0.978}, 'number_of_points': 16, 'minimum_number_iterations': 100000}, 0.0503601: {'threshold': (0.00046499999999999997, (0.0004266903826683576, 0.0005033096173316424)), 'GHZ_success_rates': {0.000375: 0.9868, 0.000495: 0.9863}, 'number_of_points': 16, 'minimum_number_iterations': 100000}, 0.0527921: {'threshold': (0.00046999999999999993, (0.00041208703430651846, 0.0005279129656934814)), 'GHZ_success_rates': {0.000375: 0.9907, 0.00049: 0.9903}, 'number_of_points': 16, 'minimum_number_iterations': 100000}, 0.0552241: {'threshold': (0.0004978567163333487, (0.0004547832934370526, 0.0005409301392296449)), 'GHZ_success_rates': {0.000375: 0.9932, 0.000505: 0.9928}, 'number_of_points': 17, 'minimum_number_iterations': 100000}, 0.0576561: {'threshold': (0.0004730951118354175, (0.00044426026267366063, 0.0005019299609971743)), 'GHZ_success_rates': {0.000375: 0.9949, 0.000495: 0.9947}, 'number_of_points': 22, 'minimum_number_iterations': 100000}, 0.0600881: {'threshold': (0.00046004323537756367, (0.0004264523202207015, 0.0004936341505344259)), 'GHZ_success_rates': {0.000375: 0.9963, 0.000485: 0.9962}, 'number_of_points': 18, 'minimum_number_iterations': 100000}, 0.0649521: {'threshold': (0.00043437266672449585, (0.00041065970285197287, 0.00045808563059701883)), 'GHZ_success_rates': {0.000375: 0.9983, 0.000445: 0.9983}, 'number_of_points': 12, 'minimum_number_iterations': 100000}, 0.0746801: {'threshold': (0.00033479404327924745, (0.00029640705383581287, 0.00037318103272268203)), 'GHZ_success_rates': {0.0003: 0.9997, 0.00036: 0.9997}, 'number_of_points': 15, 'minimum_number_iterations': 100000}, 0.0904881: {'threshold': (0.00018000000000000004, (0.00013879599830476504, 0.00022120400169523504)), 'GHZ_success_rates': {0.00018: 1.0, 0.000215: 1.0}, 'number_of_points': 17, 'minimum_number_iterations': 100000}, 0.0977841: {'threshold': (0.00014244045771621202, (0.00011007950078901964, 0.0001748014146434044)), 'GHZ_success_rates': {0.0001: 1.0, 0.000175: 1.0}, 'number_of_points': 11, 'minimum_number_iterations': 100000}}}
    return protocols_to_be_plotted_per_set, cut_off_dict_set_plot
