"""
2021-2024 Sébastian de Bone (QuTech)
https://github.com/sebastiandebone/ghz_prot_II/
"""
from termcolor import colored
import os
from circuit_simulation.gates.gates import *
import signal
import platform
import pandas as pd
import dill
import sys
import copy
from multiprocessing import cpu_count

from circuit_simulation.stabilizer_measurement_protocols.run_protocols import main_series, main_threaded
from oopsc.threshold.sim import sim_thresholds
from oopsc.threshold.plot import plot_thresholds as plot_fit
from oopsc.threshold.fit import fit_thresholds as fit_function

from circuit_simulation.node.open_sets_as_csv import get_hardware_data_sets
hardware_data_sets = get_hardware_data_sets()

PROTOCOL_EXPORT_ID = 0


def increase_export_id():
    global PROTOCOL_EXPORT_ID
    PROTOCOL_EXPORT_ID += 1


def initial_threshold_search(prot_name, protocol, set_name, p_g_values=None, lattices=None, so_iters=5000,
                             sc_iters=50000, results_dictionary=None, multithreading=True, max_nr_threads=128,
                             show_fits=True, zoomed_in_fit=False, zoom_number=2, alpha=None, n_DD=None,
                             interactive_plot=True):
    p_g_values = [0.0008, 0.00105, 0.0013, 0.00155] if p_g_values is None else p_g_values
    lattices = [4, 6, 8, 10, 12] if lattices is None else lattices
    fit_dataframe = pd.DataFrame(columns=["L", "p_g", "N", "success"])
    fit_dataframe = fit_dataframe.set_index(["L", "p_g"])
    if not isinstance(results_dictionary, dict):
        results_dictionary = {}
        for p_g in p_g_values:
            results_dictionary[p_g] = {}
            characteristics = simulate_protocol_recipe(protocol, seeds=[*range(so_iters)], set_name=set_name,
                                                       metric="logical_success", p_g=p_g, print_results=False,
                                                       lattices=lattices, sc_iters=sc_iters,
                                                       multithreading=multithreading, max_nr_threads=max_nr_threads,
                                                       merge_results=False, alpha=alpha, n_DD=n_DD)

            print(f"For {p_g} and {lattices}: {characteristics}.")
            for i_L, L in enumerate(lattices):
                fit_dataframe.loc[(L, p_g), "success"] = characteristics[i_L] * sc_iters
                fit_dataframe.loc[(L, p_g), "N"] = sc_iters
                results_dictionary[p_g][L] = {}
                results_dictionary[p_g][L]["success"] = characteristics[i_L] * sc_iters
                results_dictionary[p_g][L]["N"] = sc_iters
    else:
        p_g_values = []
        for p_g in results_dictionary.keys():
            p_g_values.append(p_g)
            for L in results_dictionary[p_g].keys():
                fit_dataframe.loc[(L, p_g), "success"] = results_dictionary[p_g][L]["success"]
                fit_dataframe.loc[(L, p_g), "N"] = results_dictionary[p_g][L]["N"]
    try:
        _, par, perr, chisqrred = fit_function(
            fit_dataframe, modified_ansatz=True, print_results=False, limit_bounds=True, return_chi_squared_red=True
        )
    except RuntimeError:
        par, perr = [0, 0, 0, 0, 0, 0, 0], 0
        show_fits = False
    except ValueError:
        par, perr = [0, 0, 0, 0, 0, 0, 0], 0
        show_fits = False

    # 'zoom_number' is the number of p_g values taken on both sides of the threshold for the fit
    if zoomed_in_fit:
        if len(p_g_values) < 2 * zoom_number:
            zoom_number = int(len(p_g_values)/2)
        fit_dataframe_zoom = copy.deepcopy(fit_dataframe)
        p_g_values = sorted(p_g_values)
        p_g_thres = par[0]
        p_g_closest = min(p_g_values, key=lambda x: abs(x - p_g_thres))
        p_g_index = p_g_values.index(p_g_closest)
        corr = 1 if p_g_thres < p_g_values[p_g_index] else 0
        if p_g_index+zoom_number+1-corr >= len(p_g_values):
            p_g_to_keep = p_g_values[(-1*zoom_number*2):]
        elif p_g_index-zoom_number+1-corr < 0:
            p_g_to_keep = p_g_values[:(zoom_number*2)]
        else:
            p_g_to_keep = p_g_values[(p_g_index-zoom_number+1-corr):(p_g_index+zoom_number+1-corr)]
        for index in fit_dataframe.index:
            if index[1] not in p_g_to_keep:
                fit_dataframe_zoom.drop(index, inplace=True)
        try:
            _, par, perr, chisqrred = fit_function(
                fit_dataframe_zoom, modified_ansatz=True, print_results=False, limit_bounds=True,
                return_chi_squared_red=True
            )
        except RuntimeError:
            par, perr = [0, 0, 0, 0, 0, 0, 0], 0
            show_fits = False

    if show_fits:
        plot_fit(
            modified_ansatz=True, data=fit_dataframe, interactive_plot=interactive_plot,
            plot_title=f"Threshold for {prot_name.replace('_','-')} without cut-off", par=par, perr=perr,
            accuracy=None, include_fit=True, include_rescaled_error_rates=False, chi_squared_red=chisqrred,
            plotstyle="paper",
        )
    return par, perr, results_dictionary


def simulate_protocol_recipe(protocol_recipe, numb_of_iter=1, protocol_name="auto_generated_swap", print_results=False,
                             seeds=None, set_name="SetJ", cpu_number=None, failed_protocol_prefix=None, p_g=None,
                             metric=None, lattices=None, sc_iters=None, multithreading=False, max_nr_threads=128,
                             merge_results=True, print_information=False, alpha=None, n_DD=None, return_bell_par=False):

    characteristics_full = {'dur': [], 'stab_fid': [], 'ghz_fid': [], 'weighted_sum': [], 'logical_success': []}

    arguments = {'protocol': protocol_name, 'protocol_recipe': protocol_recipe, 'seed_number': seeds, 'color': False,
                 'save_latex_pdf': False, 'csv_filename': None, 'cp_path': None,
                 'to_console': False, 'draw_circuit': False, 'single_qubit_gate_lookup': None,
                 'two_qubit_gate_lookup': None, # 'gate_duration_file': 'circuit_simulation\\gate_duration.txt',
                 'progress_bar': False, 'cut_off': None, 'iterations': numb_of_iter, 'threaded': multithreading,
                 'force_run': True, 'combine': False, 'cutoff_search': False, 'do_not_merge_superoperators': True,
                 'threads': max_nr_threads if cpu_count() > max_nr_threads else cpu_count()}

    if set_name.lower() in ["setj", "setiia"]:
        gates_dict = {'X_gate': (0.013, 1.4e-07), 'Y_gate': (0.013, 1.4e-07), 'Z_gate': (0.0065, 1e-07),
                      'H_gate': (0.0065, 1e-07), 'CNOT_gate': (0.025, None), 'CZ_gate': (0.025, None),
                      'CiY_gate': (0.025, None), 'SWAP_gate': (0.075, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.005, "mu": 0.995, "lambda": 1,
                                "eta": 0.142}

        arguments = arguments | {'p_g': 0.0001, 'p_m': 0.0001, 'p_m_1': 0.0001, 'p_n': None, 'lde_success': None,
                     'fixed_lde_attempts': 600, 'pulse_duration': 13e-3, '_node': 'Pur', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'pm_equals_pg': True,
                     'probabilistic': True, 'measurement_duration': 1e-06, 'lde_duration': 1e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1_idle': 300.0,
                     'T2_idle': 10.0, 'T1_lde': 1.2, 'T2_lde': 1.2, 'T1_idle_electron': 300, 'T2_idle_electron': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "seti":
        gates_dict = {'X_gate': (0.013, 1.4e-07), 'Y_gate': (0.013, 1.4e-07), 'Z_gate': (0.0065, 1e-07),
                      'H_gate': (0.0065, 1e-07), 'CNOT_gate': (0.025, None), 'CZ_gate': (0.025, None),
                      'CiY_gate': (0.025, None), 'SWAP_gate': (0.075, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "single_click", "F_prep": 0.999, "p_DE": 0.02, "mu": 0.925,
                                "lambda": 0.991832, "eta": 0.0793, "alpha": 0.06325}

        arguments = arguments | {'pg': 0.0001, 'pm': 0.0001, 'pm_1': 0.0001, 'pn': None, 'lde_success': None,
                     'fixed_lde_attempts': 324, 'pulse_duration': 13e-3, '_node': 'Pur', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'pm_equals_pg': True,
                     'probabilistic': True, 'measurement_duration': 4e-06, 'lde_duration': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1_idle': 300.0,
                     'T2_idle': 10.0, 'T1_lde': 1.2, 'T2_lde': 1.2, 'T1_idle_electron': 300, 'T2_idle_electron': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiib":
        gates_dict = {'X_gate': (0.013, 1.4e-07), 'Y_gate': (0.013, 1.4e-07), 'Z_gate': (0.0065, 1e-07),
                      'H_gate': (0.0065, 1e-07), 'CNOT_gate': (0.025, None), 'CZ_gate': (0.025, None),
                      'CiY_gate': (0.025, None), 'SWAP_gate': (0.075, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'pg': 0.0001, 'pm': 0.0001, 'pm_1': 0.0001, 'pn': None, 'lde_success': None,
                     'fixed_lde_attempts': 69, 'pulse_duration': 13e-3, '_node': 'Pur', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'pm_equals_pg': True,
                     'probabilistic': True, 'measurement_duration': 1e-06, 'lde_duration': 1e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1_idle': 300.0,
                     'T2_idle': 10.0, 'T1_lde': 1.2, 'T2_lde': 1.2, 'T1_idle_electron': 300, 'T2_idle_electron': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() in ["setiii", "setiiia"]:
        gates_dict = {'X_gate': (0.4e-4, 1.4e-07), 'Y_gate': (0.4e-4, 1.4e-07), 'Z_gate': (0.2e-4, 1e-07),
                      'H_gate': (0.2e-4, 1e-07), 'CNOT_gate': (0.2e-4, None), 'CZ_gate': (0.2e-4, None),
                      'CiY_gate': (0.2e-4, None), 'SWAP_gate': (0.6e-4, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'pg': 0.0001, 'pm': 0.0001, 'pm_1': 0.0001, 'pn': None, 'lde_success': None,
                     'fixed_lde_attempts': 500, 'pulse_duration': 0.4e-4, '_node': 'Pur', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'pm_equals_pg': True,
                     'probabilistic': True, 'measurement_duration': 1e-06, 'lde_duration': 1e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1_idle': 300.0,
                     'T2_idle': 10.0, 'T1_lde': 1.6e-2, 'T2_lde': 0.4e-2, 'T1_idle_electron': 300, 'T2_idle_electron': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiib":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3b', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.03, 'T2n_link': 0.0075, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiiu":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3b', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.045, 'T2n_link': 0.01125, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiir":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3b', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.06, 'T2n_link': 0.015, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiid":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3d', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.075, 'T2n_link': 0.01875, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiiq":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.0001, 'p_m': 0.0001, 'p_m_1': None, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3p', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.09, 'T2n_link': 0.0225, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiip":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.0001, 'p_m': 0.0001, 'p_m_1': None, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3p', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.105, 'T2n_link': 0.02625, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiic":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.0006, 'p_m': 0.0006, 'p_m_1': None, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3c', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.12, 'T2n_link': 0.03, 'T1e_idle': 300.0, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiie":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3e', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.3, 'T2n_link': 0.075, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiif":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3f', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 3.0, 'T2n_link': 0.75, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiig":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3g', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 30.0, 'T2n_link': 7.5, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiim":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3m', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 300.0, 'T2n_link': 10.0, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiih":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 0, 't_pulse': 0.0, '_node': 's3h', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': False, 'p_m_equals_p_g': True,
                     'probabilistic': False, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 30.0, 'T2n_link': 10.0, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() == "setiiik":
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}

        arguments = arguments | {'p_g': 0.001, 'p_m': 0.001, 'p_m_1': 0.001, 'F_link': None, 'p_link': None,
                     'n_DD': 0, 't_pulse': 0.0, '_node': 's3k', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': False, 'p_m_equals_p_g': True,
                     'probabilistic': False, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': True, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 300.0, 'T2n_link': 10.0, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}

    elif set_name.lower() in ["setiiiiv", "setiiiir", "setiiiiq", "setiiiip", "setiiiia", "setiiiib", "setiiiic", "setiiiid", "setiiiie",
                              "setvu", "setvr", "setvq", "setvp", "setva", "setvb", "setvc", "setvd", "setve"]:
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.4472}
        if "setiiii" in set_name.lower():
            bell_pair_parameters["ent_prot"] = "single_click"
            bell_pair_parameters["lambda"] = 0.984
            bell_pair_parameters["alpha"] = 0.005
        if set_name[-1].lower() == "u":
            bell_pair_parameters["p_DE"] = 0.1387984
        if set_name[-1].lower() == "r":
            bell_pair_parameters["p_DE"] = 0.1210398
        if set_name[-1].lower() == "q":
            bell_pair_parameters["p_DE"] = 0.103633
        if set_name[-1].lower() == "p":
            bell_pair_parameters["p_DE"] = 0.086556
        if set_name[-1].lower() == "a":
            bell_pair_parameters["p_DE"] = 0.069796
        if set_name[-1].lower() == "b":
            bell_pair_parameters["p_DE"] = 0.0533309
        if set_name[-1].lower() == "c":
            bell_pair_parameters["p_DE"] = 0.037147
        if set_name[-1].lower() == "d":
            bell_pair_parameters["p_DE"] = 0.021231
        if set_name[-1].lower() == "e":
            bell_pair_parameters["p_DE"] = 0.0055701

        arguments = arguments | {'p_g': 0.0001, 'p_m': 0.0001, 'p_m_1': 0.0001, 'F_link': None, 'p_link': None,
                     'n_DD': 18, 't_pulse': 1.0e-3, '_node': 's3e', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.3, 'T2n_link': 0.075, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}
        arguments["_node"] = 's' + ("4" if "setiiii" in set_name.lower() else "5") + set_name[-1].lower()
        if "setiiii" in set_name.lower():
            arguments["n_DD"] = 154


    elif set_name.lower() in ["setvia", "setvib", "setvic", "setvid", "setvie", "setvif", "setvik", "setvig", "setvim", "setvih",
                              "setviia", "setviib", "setviic", "setviid", "setviie", "setviif", "setviik", "setviig", "setviim", "setviih",
                              "setviiia", "setviiib", "setviiic", "setviiid", "setviiie", "setviiif", "setviiik", "setviiig", "setviiim", "setviiih",
                              "setixa", "setixb", "setixc", "setixd", "setixe", "setixf", "setixk", "setixg", "setixm", "setixh"]:
        gates_dict = {'X_gate': (1.0e-3, 1.4e-07), 'Y_gate': (1.0e-3, 1.4e-07), 'Z_gate': (0.5e-3, 1e-07),
                      'H_gate': (0.5e-3, 1e-07), 'CNOT_gate': (0.5e-3, None), 'CZ_gate': (0.5e-3, None),
                      'CiY_gate': (0.5e-3, None), 'SWAP_gate': (1.5e-3, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.1061}
        if set_name.lower() in ["setviia", "setviib", "setviic", "setviid", "setviie", "setviif", "setviik", "setviig", "setviim", "setviih",
                                "setviiia", "setviiib", "setviiic", "setviiid", "setviiie", "setviiif", "setviiik", "setviiig", "setviiim", "setviiih"]:
            bell_pair_parameters["ent_prot"] = "single_click"
            bell_pair_parameters["mu"] = 0.9
            bell_pair_parameters["lambda"] = 0.984
            bell_pair_parameters["F_prep"] = 0.99
            bell_pair_parameters["alpha"] = 0.02655
        elif set_name.lower() in ["setixa", "setixb", "setixc", "setixd", "setixe", "setixf", "setixk", "setixg", "setixm", "setixh"]:
            bell_pair_parameters["ent_prot"] = "single_click"
            bell_pair_parameters["mu"] = 0.95
            bell_pair_parameters["lambda"] = 0.984
            bell_pair_parameters["F_prep"] = 0.999
            bell_pair_parameters["alpha"] = 0.05

        if set_name.lower() in ["setviia", "setviib", "setviic", "setviid", "setviie", "setviif", "setviik", "setviig", "setviim", "setviih"]:
            bell_pair_parameters["p_DE"] = 0.04
        elif set_name.lower() in ["setviiia", "setviiib", "setviiic", "setviiid", "setviiie", "setviiif", "setviiik", "setviiig", "setviiim", "setviiih"]:
            bell_pair_parameters["p_DE"] = 0.07
        else:
            bell_pair_parameters["p_DE"] = 0.01

        n_DD_int = 198
        if set_name[-1].lower() == "a":
            bell_pair_parameters["eta"] = 0.0447
        elif set_name[-1].lower() == "b":
            bell_pair_parameters["eta"] = 0.0596
        elif set_name[-1].lower() == "c":
            bell_pair_parameters["eta"] = 0.0795
        elif set_name[-1].lower() == "d":
            bell_pair_parameters["eta"] = 0.1061
            n_DD_int = 134
        elif set_name[-1].lower() == "e":
            bell_pair_parameters["eta"] = 0.1414
        elif set_name[-1].lower() == "f":
            bell_pair_parameters["eta"] = 0.1886
            n_DD_int = 64
        elif set_name[-1].lower() == "k":
            bell_pair_parameters["eta"] = 0.217778
            n_DD_int = 52
        elif set_name[-1].lower() == "g":
            bell_pair_parameters["eta"] = 0.2515
            n_DD_int = 42
        elif set_name[-1].lower() == "m":
            bell_pair_parameters["eta"] = 0.2904123
            n_DD_int = 34
        elif set_name[-1].lower() == "h":
            bell_pair_parameters["eta"] = 0.3354
            n_DD_int = 28

        arguments = arguments | {'p_g': 0.0001, 'p_m': 0.0001, 'p_m_1': 0.0001, 'F_link': None, 'p_link': None,
                     'n_DD': n_DD_int, 't_pulse': 1.0e-3, '_node': 's3e', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 0.3, 'T2n_link': 0.075, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}
        arguments["_node"] = 's' + ("8" if set_name.lower()[3:7] == "viii" else ("7" if set_name.lower()[3:6] == "vii" else ("6" if set_name.lower()[3:5] == "vi" else "9"))) + set_name[-1].lower()



    elif set_name.lower() in ["setxa", "setxb", "setxc", "setxd", "setxe", "setxf", "setxg", "setxh"]:
        gates_dict = {'X_gate': (0.013, 1.4e-07), 'Y_gate': (0.013, 1.4e-07), 'Z_gate': (0.0065, 1e-07),
                      'H_gate': (0.0065, 1e-07), 'CNOT_gate': (0.025, None), 'CZ_gate': (0.025, None),
                      'CiY_gate': (0.025, None), 'SWAP_gate': (0.075, None)}
        set_duration_of_known_gates(gates_dict)

        bell_pair_parameters = {"ent_prot": "double_click", "F_prep": 0.999, "p_DE": 0.01, "mu": 0.95,
                                "lambda": 1, "eta": 0.1061}

        n_DD_int = 198
        if set_name[-1].lower() == "a":
            bell_pair_parameters["eta"] = 0.0447
        elif set_name[-1].lower() == "b":
            bell_pair_parameters["eta"] = 0.0596
        elif set_name[-1].lower() == "c":
            bell_pair_parameters["eta"] = 0.0795
        elif set_name[-1].lower() == "d":
            bell_pair_parameters["eta"] = 0.1061
            n_DD_int = 134
        elif set_name[-1].lower() == "e":
            bell_pair_parameters["eta"] = 0.1414
        elif set_name[-1].lower() == "f":
            bell_pair_parameters["eta"] = 0.1886
            n_DD_int = 64
        elif set_name[-1].lower() == "g":
            bell_pair_parameters["eta"] = 0.2515
            n_DD_int = 42
        elif set_name[-1].lower() == "h":
            bell_pair_parameters["eta"] = 0.3354
            n_DD_int = 28

        arguments = arguments | {'p_g': 0.0001, 'p_m': 0.0001, 'p_m_1': 0.0001, 'F_link': None, 'p_link': None,
                     'n_DD': n_DD_int, 't_pulse': 13e-3, '_node': 's10' + set_name[-1].lower(), 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': True, 'p_m_equals_p_g': True,
                     'probabilistic': True, 't_meas': 4e-06, 't_link': 6e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1n_idle': 300.0,
                     'T2n_idle': 10.0, 'T1n_link': 1.2, 'T2n_link': 1.2, 'T1e_idle': 300, 'T2e_idle': 1.0,
                     'bell_pair_parameters': bell_pair_parameters, 'gate_durations': gates_dict}


    elif set_name.lower() == "setyves":
        gates_dict = {'X_gate': (1., 1.), 'Y_gate': (1., 1.), 'Z_gate': (1., 1.),
                      'H_gate': (1., 1.), 'CNOT_gate': (1., None), 'CZ_gate': (1., None),
                      'CiY_gate': (1., None), 'SWAP_gate': (0., None)}
        set_duration_of_known_gates(gates_dict)

        dec_time = 99.4991624734

        arguments = arguments | {'pg': 0.001, 'pm': 0.001, 'pm_1': 0.001, 'pn': 0.05, 'lde_success': 1.,
                     'fixed_lde_attempts': 1e-15, 'pulse_duration': 0., '_node': 'Pur', 'cut_off_time': np.inf,
                     'stabilizer_type': ['X', 'Z'], 'decoherence': False, 'pm_equals_pg': True,
                     'probabilistic': True, 'measurement_duration': 1., 'lde_duration': 1., 'use_swap_gates': True,
                     'noiseless_swap': True, 'bell_pair_type': 0, 'network_noise_type': 0, 'T1_idle': dec_time,
                     'T2_idle': dec_time, 'T1_lde': dec_time, 'T2_lde': dec_time, 'T1_idle_electron': dec_time,
                     'T2_idle_electron': dec_time,
                     'bell_pair_parameters': None, 'gate_durations': gates_dict}

    elif set_name == 0:
        arguments = arguments | {'gate_duration_file': 'circuit_simulation\\gate_duration.txt',
                     'pg': 0.0001, 'pm': 0.0001, 'pm_1': 0.0001, 'pn': 0.015, 'lde_success': 0.017,
                     'fixed_lde_attempts': 100, 'pulse_duration': 13e-3, '_node': 'Pur', 'cut_off_time': np.inf,
                     'stabilizer_type': 'Z', 'decoherence': True, 'pm_equals_pg': True,
                     'probabilistic': True, 'measurement_duration': 1e-06, 'lde_duration': 1e-06, 'use_swap_gates': True,
                     'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 3, 'T1_idle': 300.0,
                     'T2_idle': 10.0, 'T1_lde': 1.2, 'T2_lde': 1.2, 'T1_idle_electron': 300, 'T2_idle_electron': 1.0}
    else:
        # The set is not (immediately) recognized from the default sets above: we are going to check if it happens to
        # be present in the "sets.csv" file of the circuit simulator. Otherwise we select the most basic set.
        if set_name in hardware_data_sets.index:
            # print("Set is recognized from the sets.csv file of the circuit simulator.")
            hds = hardware_data_sets.loc[set_name]
            gates_dict = {'X_gate': (float(hds['tn_X']), float(hds['te_X'])),
                          'Y_gate': (float(hds['tn_Y']), float(hds['te_Y'])),
                          'Z_gate': (float(hds['tn_Z']), float(hds['te_Z'])),
                          'H_gate': (float(hds['tn_H']), float(hds['te_Z'])),
                          'CNOT_gate': (float(hds['t_CX']), None), 'CZ_gate': (float(hds['t_CZ']), None),
                          'CiY_gate': (float(hds['t_CiY']), None), 'SWAP_gate': (float(hds['t_SWAP']), None)}
            set_duration_of_known_gates(gates_dict)

            if hardware_data_sets.loc[set_name]['F_link'] is None \
                    or hardware_data_sets.loc[set_name]['p_link'] is None:
                bell_pair_parameters = {"ent_prot": hds['ent_prot']}
                for param in ['F_prep', 'p_DE', 'mu', 'lambda', 'eta']:
                    bell_pair_parameters[param] = hds[param] if hds[param] is None else float(hds[param])
                if bell_pair_parameters['ent_prot'] == 'single_click':
                    bell_pair_parameters["alpha"] = float(hds['alpha'])
                data_F_link, data_p_link = None, None
            else:
                bell_pair_parameters = None
                data_F_link, data_p_link = float(hds['F_link']), float(hds['p_link'])

            for param in ['p_g', 'p_m', 'p_m_1', 't_pulse', 't_meas', 't_link',
                          'T1n_idle', 'T2n_idle', 'T1n_link', 'T2n_link', 'T1e_idle', 'T2e_idle']:
                arguments[param] = hds[param] if hds[param] is None else float(hds[param])
            for param in ['n_DD', 'bell_pair_type', 'network_noise_type']:
                arguments[param] = hds[param] if hds[param] is None else int(hds[param])
            for param in ['decoherence', 'noiseless_swap', 'probabilistic']:    # The 'bool' conversion is very
                arguments[param] = False if hds[param] is None else bool(hds[param])    # important here, as it converts
                # a np.bool_ type to a pure bool type. This is important because np.bool_ is not recognized as False.

            arguments = arguments | {'F_link': data_F_link, 'p_link': data_p_link, '_node': set_name,
                                     'cut_off_time': np.inf, 'stabilizer_type': ['X', 'Z'], 'p_m_equals_p_g': True,
                                     'use_swap_gates': True, 'bell_pair_parameters': bell_pair_parameters,
                                     'gate_durations': gates_dict}
        else:
            if set_name != 1:
                print("Set was not recognized.", file=sys.stderr)
            arguments = arguments | {'gate_duration_file': 'circuit_simulation\\gate_duration.txt',
                 'pg': 0.001, 'pm': 0.001, 'pm_1': 0.001, 'pn': 0.013, 'lde_success': 0.01,
                 'fixed_lde_attempts': 100, 'pulse_duration': 13e-3, '_node': 'Pur', 'cut_off_time': np.inf,
                 'stabilizer_type': 'Z', 'decoherence': True, 'pm_equals_pg': True,
                 'probabilistic': True, 'measurement_duration': 1e-06, 'lde_duration': 1e-06,
                 'use_swap_gates': True,
                 'noiseless_swap': False, 'bell_pair_type': 3, 'network_noise_type': 0, 'T1_idle': 300.0,
                 'T2_idle': 10.0, 'T1_lde': 1.2, 'T2_lde': 1.2, 'T1_idle_electron': 300, 'T2_idle_electron': 1.0}

    if isinstance(seeds, list):
        number_of_seeds = len(seeds)
    else:
        number_of_seeds = 1
        seeds = [seeds]

    if 'SolidState' in set_name or 'NeutralAtom' in set_name or set_name == "SetNV":
        arguments['network_noise_type'] = 99
        p_m_equals_p_g_extra_noisy_measurement = True
    else:
        p_m_equals_p_g_extra_noisy_measurement = False
    arguments['bell_pair_type'] = 3 if arguments['bell_pair_type'] > 3 else arguments['bell_pair_type']
    arguments['bell_pair_type'] = 0 if arguments['network_noise_type'] == 99 else arguments['bell_pair_type']
    arguments['network_noise_type'] = 3 if (arguments['network_noise_type'] > 22 and
                                            arguments['network_noise_type'] != 99) else arguments['network_noise_type']
    if alpha is not None and isinstance(alpha, float) and alpha <= 1.0 and alpha >= 0.0 \
            and isinstance(arguments['bell_pair_parameters'], dict) \
            and 'ent_prot' in arguments['bell_pair_parameters'] \
            and arguments['bell_pair_parameters']['ent_prot'] == 'single_click':
        arguments['bell_pair_parameters']['alpha'] = alpha
    if return_bell_par:
        if 'bell_pair_parameters' in arguments and arguments['bell_pair_parameters'] is not None:
            return arguments['bell_pair_parameters'] | {'t_pulse': arguments['t_pulse'], 't_link': arguments['t_link']}
        else:
            return {'p_link': arguments['p_link'], 'F_link': arguments['F_link'], 't_pulse': arguments['t_pulse'],
                    't_link': arguments['t_link']}
    if p_g is not None and isinstance(p_g, float) and p_g <= 1.0 and p_g >= 0.0:
        arguments['p_g'] = p_g
        arguments['p_m'] = p_g if p_m_equals_p_g_extra_noisy_measurement is False \
            else 5 * p_g / 3 - 4 * p_g ** 2 / 9 - 2 * p_g ** 3 / 9
        arguments['p_m_1'] = arguments['p_m']
    if n_DD is not None and (isinstance(n_DD, float) or isinstance(n_DD, int)) and n_DD > 0:
        arguments['n_DD'] = int(n_DD)

    # print(f"CPU {cpu_number} starts calculating a protocol with {number_of_seeds}.")
    failed_protocol_prefix = "" if failed_protocol_prefix is None \
        else (failed_protocol_prefix if failed_protocol_prefix[-1] != "/" else failed_protocol_prefix[:-1])
    main_folder = f"results/protocols/{failed_protocol_prefix}/"
    if not os.path.exists(main_folder):
        os.mkdir(main_folder)

    # Close session
    def handler(signum, frame):
        raise Exception("Action took too much time")
    if platform.system() == "Linux":
        signal.signal(signal.SIGALRM, handler)

    if metric == "logical_success":
        lattices = [4, 8] if lattices is None else lattices
        sc_iters = 500 if sc_iters is None else sc_iters
        decoders = __import__("oopsc.decoder", fromlist=['uf'])
        decode = getattr(decoders, 'uf')
        surface_code_args = {'lattice_type': 'toric', 'iters': sc_iters, 'lattices': lattices, 'perror': [],
                             'superoperator_filenames': None,
                             'superoperator_filenames_failed': None,
                             'GHZ_successes': [1.0],
                             'supop_date_and_time_values': None,
                             'superoperator_filenames_additional': None, 'superoperator_filenames_additional_failed': None,
                             'networked_architecture': True, 'measurement_error': True, 'multithreading': multithreading,
                             'threads': max_nr_threads if cpu_count() > max_nr_threads else cpu_count(),
                             'modified_ansatz': False, 'save_result': False, 'show_result': False, 'progressbar': False,
                             'fbloom': 0.5, 'dg_connections': False, 'directed_graph': False, 'debug': False,
                             'cycles': None, 'iid': False, 'space_weight': 2, 'skip_surface_code': False,
                             'network_architecture_type': 'weight-4'}
        arguments['iterations'] = number_of_seeds if seeds[0] is not None else numb_of_iter
        arguments['iter_offset'] = seeds[0] if seeds[0] is not None else 0
        if not ((len(seeds) == 1 and seeds[0] is None)
                or all([seeds[offset] == seeds[0] + offset for offset in range(len(seeds))])):
            raise ValueError("To use the metric 'logical_success', the seeds used have to be ascending integers.")
        number_of_seeds = 1

    main = main_threaded if multithreading else main_series

    for i_seed in range(number_of_seeds):
        # if platform.system() == "Linux":
        #     # We wait a maximum of 10 minutes per protocol iteration: the whole process gets killed after that time.
        #     signal.alarm(100*60)  # Set the parameter to the amount of seconds you want to wait
        if cpu_number is not None:
            try:
                # print(f"Calculating seed {seeds[i_seed]} for cpu {cpu_number}.")
                arguments['seed_number'] = seeds[i_seed] if metric != "logical_success" else None
                (succeeded, cut_off), print_lines, characteristics = main(fn=(None, None), **arguments)
                if print_information:
                    print(f"Stabilizer fidelity is ({succeeded.loc[('IIII', False), 'p']}, {succeeded.loc[('IIII', False), 's']}).")
                if metric == "logical_success":
                    surface_code_args['superoperator_filenames'] = [succeeded]
                    surface_code_args['superoperator_filenames_failed'] = [cut_off] if cut_off is not None else None
                    successful_iterations = succeeded.loc[('IIII', False), 'written_to']
                    failed_iterations = cut_off.loc[('IIII', False), 'written_to'] if cut_off is not None else 0
                    surface_code_args['GHZ_successes'] = [float(successful_iterations) / (float(successful_iterations) + float(failed_iterations))]
                    data = sim_thresholds(decode, **surface_code_args)
                    for i_lat in range(len(lattices)):
                        characteristics_full['logical_success'].append(data.loc[data.index[i_lat], 'success'] / data.loc[data.index[i_lat], 'N'])

                characteristics_full['dur'] += characteristics['dur']
                characteristics_full['ghz_fid'] += characteristics['ghz_fid']
                characteristics_full['stab_fid'] += characteristics['stab_fid']
                characteristics_full['weighted_sum'] += characteristics['weighted_sum']
            except:
                print(f"Protocol crashed for seed {arguments['seed_number']}.")
                folder = f"{main_folder}{cpu_number}/" if cpu_number is not None else "protocols/"
                if not os.path.exists(folder):
                    os.mkdir(folder)
                file_names = [file for file in os.listdir(folder) if file.startswith("rec_failed_")]
                prot_new_name = f"rec_failed_{set_name}_{len(file_names)}_{arguments['seed_number']}"
                dill.dump(protocol_recipe, open(folder + prot_new_name, "wb"))
                print(f"Exported protocol {prot_new_name} which gave problems for seed {arguments['seed_number']}.")
                characteristics_full['dur'] += [None]
                characteristics_full['ghz_fid'] += [None]
                characteristics_full['stab_fid'] += [0]
                characteristics_full['weighted_sum'] += [None]
                if metric == "logical_success":
                    characteristics_full['logical_success'].append(0)
                    for i_lat in range(len(lattices) - 1):
                        characteristics_full['logical_success'].append(-0.5)
                break
        else:
            arguments['seed_number'] = seeds[i_seed] if metric != "logical_success" else None
            (succeeded, cut_off), print_lines, characteristics = main(fn=(None, None), **arguments)
            if print_information:
                print(f"Stabilizer fidelity is ({succeeded.loc[('IIII', False), 'p']}, {succeeded.loc[('IIII', False), 's']}).")
            if metric == "logical_success":
                surface_code_args['superoperator_filenames'] = [succeeded]
                surface_code_args['superoperator_filenames_failed'] = [cut_off] if cut_off is not None else None
                successful_iterations = succeeded.loc[('IIII', False), 'written_to']
                failed_iterations = cut_off.loc[('IIII', False), 'written_to'] if cut_off is not None else 0
                surface_code_args['GHZ_successes'] = [float(successful_iterations) / (float(successful_iterations) + float(failed_iterations))]
                data = sim_thresholds(decode, **surface_code_args)
                for i_lat in range(len(lattices)):
                    characteristics_full['logical_success'].append(data.loc[data.index[i_lat], 'success'] / data.loc[data.index[i_lat], 'N'])

            characteristics_full['dur'] += characteristics['dur']
            characteristics_full['ghz_fid'] += characteristics['ghz_fid']
            characteristics_full['stab_fid'] += characteristics['stab_fid']
            characteristics_full['weighted_sum'] += characteristics['weighted_sum']

    if metric == "logical_success" and len(lattices) == 2 and merge_results:
        result = characteristics_full["logical_success"]
        characteristics_full["logical_success"] = [0.5 - (result[0] - result[1])]

    # # We switch off the alarm again:
    # if platform.system() == "Linux":
    #     signal.alarm(0)  # Disables the alarm

    # print(f"CPU {cpu_number} stops with calculating a protocol.")
    # os.remove(prot_folder + prot_name)

    if print_results:
        print(colored("Stabilizer fidelities:", "green"))
        print(characteristics_full["stab_fid"])
        print("The average is " + str(np.average(characteristics_full["stab_fid"])) + ".")
        print(colored("\nGHZ fidelities:", "green"))
        print(characteristics_full["ghz_fid"])
        print("The average is " + str(np.average(characteristics_full["ghz_fid"])) + ".")
        print(colored("\nDuration:", "green"))
        print(characteristics_full["dur"])
        print("The average is " + str(np.average(characteristics_full["dur"])) + ".")
        print(colored("\nWeighted error probability:", "green"))
        print(characteristics_full["weighted_sum"])
        print("The average is " + str(np.average(characteristics_full["weighted_sum"])) + ".")

    if metric is None:
        if set_name == 0 or "set" in set_name.lower() \
                or "solidstate" in set_name.lower() or "neutralatom" in set_name.lower():
            metric = "stab_fid"
        elif set_name == 1:
            metric = "ghz_fid"
        else:
            metric = "all"

    if metric in ["stab_fid", "ghz_fid", "weighted_sum", "logical_success"]:
        return characteristics_full[metric]
    else:
        return characteristics_full
