"""
2021-2024 Sébastian de Bone (QuTech)
https://github.com/sebastiandebone/ghz_prot_II/
"""
import copy
import analysis.evaluate_protocols as ep
import GHZ_prot_II.da_protocols as dap
from GHZ_prot_II.simulate_protocol.protocol_recipe_sub_functions import *
from GHZ_prot_II.simulate_protocol.protocol_recipe_sub_classes import *


class ProtocolRecipe:
    """
        Class that describes protocol recipe objects. The protocol recipes are constructed from a binary tree protocol
        produced by the dynamic program: such a binary tree protocol is imported as "protocol" here. With "id_elem", it
        is possible to import a custom order in which the elementary links are generated. If this argument is not given,
        the "standard" procedure is used.

        A general description of how a protocol recipe is constructed can be found in the supplementary document
        "Protocol recipe construction and execution procedure.pdf" that comes with this repository.
    """
    def __init__(self, protocol, id_elem=None):
        self.n = protocol.value.n
        # self.k = protocol.value.k
        self.protocol = dap.protocol_add_meta_data(protocol)
        self.id_elem = ep.id_nrs_elementary_links(self.protocol) if id_elem is None else id_elem
        self.id_linked = ep.match_id_elem_n_4(self.id_elem)
        self.print_debug = False
        if self.print_debug is True:
            print(self.id_elem)
            print([id[0] for id in self.id_elem])
            print(self.id_linked)
        self.link_values = []
        self.link_children = []
        self.link_parent_id = []
        self.link_nodes = []
        self.link_memory_loc = []
        self.qubit_memory = []
        self.all_operations = []
        self.all_operations2 = {}
        self.swap_count = 0
        self.time_blocks = []
        self.subsystems = {}
        self.id_link_structure = {}
        self.delayed_distillation_check = []
        self.fusion_corrections = []

        self._identify_link_information()
        self._identify_operations()
        if self.print_debug is True:
            for operation in self.all_operations:
                operation.print_operation()
                if operation.type == "FUSE":
                    for fc in operation.fusion_corrections:
                        fc.print_fusion_correction()
        self._group_operations()

        self.qubit_memory_per_time_step = self._create_time_step_qubit_memory()
        self._adjust_failure_reset_levels()

    def _identify_link_information(self):
        if self.protocol == None:
            return
        myStack = []
        node = self.protocol
        parent_id = None
        while node or myStack:
            while node:
                self.link_values.append(node.value)
                self.link_parent_id.append(parent_id)
                self.link_nodes.append(node.node_nrs)
                self.link_memory_loc.append(None)
                if node.left is None:
                    self.link_children.append([None, None])
                else:
                    self.link_children.append([node.left.id, node.right.id])
                myStack.append(node)
                parent_id = node.id
                node = node.left
            node = myStack.pop()
            parent_id = node.id
            node = node.right

    def _identify_operations(self):
        """
        Function that creates a recipe list for a certain full binary tree protocol, calculates how big the memory for
        each of the physical network nodes must be, and calculates the number of swap gates.

        Parameters
        ----------
        protocol : binary tree with purification and distillation operations
        print_prot : Boolean (default=True)
            Indicates whether the list should be printed in the console
        return_marker: Boolean (default=False)
            Makes it possible to mark protocols that have specific properties that are discovered in this function

        Returns
        -------
        prot_recipe_out : multi-dimensional array with strings
            Contains the recipe list for the protocol per time step.
        qubit_memory : array of integers
            The first dimensions refers to the node numbers, and the second one to individual memory locations of the nodes.
            Here the integers refer to the link id that the qubits stored at the location belongs to.
        swap_count : positive integer
            Integer that indicates how many swap-gates have been carried out in the memory.
        node_list : list of multiple elements
            List of links ordered after the id of the nodes. Each element of the list contains four of five elements itself:
            node.value of the link, the id of the child on the left, the id of the child on the right, the id of its
            parent, a list with network node numbers on which the link is created, and (if the link is created) a list with
            locations in the memory of the concerning network node numbers.
        marker : Boolean
            Output parameter that is only returned if return_marker is set to True. It makes it possible to mark specific
            properties that have specific properties that are discovered in this function for further processing. The
            default value of this parameter is False, and when these properties are met, it can be set to True.
        """
        swap_count = 0
        self.qubit_memory = [None] * self.n
        for i in range(self.n):
            self.qubit_memory[i] = [None]
        id_created = [False] * len(self.link_values)  # List of all protocol links (elementary or operation) indicating
        i = 0
        while all(id_created) is False:
            while id_created[self.id_elem[i][0]] is True:
                # If the link is already created because it was coupled to an earlier link (simultaneously carried out),
                # we move on
                i += 1
            coupled_link_id = self.id_linked[i]
            simultaneous_events = 2 if coupled_link_id is not None else 1
            for se in range(simultaneous_events):
                link_id = self.id_elem[i][0] if se == 0 else self.id_elem[coupled_link_id][0]
                nodes_nrs = self.id_elem[i][1] if se == 0 else self.id_elem[coupled_link_id][1]

                id_created[link_id] = True

                swapped_qb = self._update_qubit_memory("create_link", link_id)
                self.swap_count += len(swapped_qb)
                for swap in swapped_qb:
                    self.all_operations.append(Operation([swap[0]], [[swap[0], swap[1]]], [[swap[0], swap[2]]], "SWAP"))
                    self.all_operations2["SWAP" + str(swap_count)] = Operation([swap[0]], [[swap[0], swap[1]]], [[swap[0], swap[2]]], "SWAP")
                    swap_count += 1

                self.all_operations.append(Operation(nodes_nrs, [[nodes_nrs[0], 0], [nodes_nrs[1], 0]], None,
                                                     "CREATE_LINK", link_id=link_id, nodes_final_state=nodes_nrs))
                self.all_operations2[link_id] = Operation(nodes_nrs, [[nodes_nrs[0], 0], [nodes_nrs[1], 0]], None,
                                                          "CREATE_LINK", link_id=link_id, nodes_final_state=nodes_nrs,
                                                          state_mem=[[nodes_nrs[0], 0], [nodes_nrs[1], 0]])

            for se in range(simultaneous_events):
                link_id = self.id_elem[i][0] if se == 0 else self.id_elem[coupled_link_id][0]
                parent_ready = True
                while parent_ready is True:
                    parent_id = self.link_parent_id[link_id]
                    if parent_id is not None and id_created[parent_id] is False:
                        parent_ready = self._check_if_parent_is_ready(parent_id, id_created)
                        if parent_ready is True:
                            op = "Distill" if self.link_values[parent_id].p_or_f == 0 else "Fuse"

                            # Here, we define the id numbers of the left and right children of the operation, and we
                            # obtain a list "inv_nodes" of all nodes that cover the final state of the operation:
                            left_child_id = self.link_children[parent_id][0]
                            right_child_id = self.link_children[parent_id][1]
                            inv_nodes = self.link_nodes[parent_id]

                            # We make a list of qubits that we need to flip if the fusion operation produces a -1
                            # result. This list will just consist of the qubits of the child of the right, minus the
                            # qubit that is been measured out (we will remove that one later):
                            if op == "Fuse":
                                correction_nodes = self.link_nodes[right_child_id]
                                correction_qubits = self.link_memory_loc[right_child_id]
                                correction_qb_zip = [[qb[0], qb[1]] for qb in zip(correction_nodes, correction_qubits)]

                            # Here, we are updating the "qubit_memory" and "node_list", and we obtain a list of qubits
                            # that need to be swapped in order for the current fusion or distillation operation to take
                            # place:
                            swapped_qb = self._update_qubit_memory(op, left_child_id, right_child_id)

                            # Here, we are going to add SWAP gates to the protocol recipe (the first qubit involved in
                            # each SWAP gate, "swap[1]", is always qubit 0 of the node "swap[0]"):
                            self.swap_count += len(swapped_qb)
                            for swap in swapped_qb:
                                self.all_operations.append(
                                    Operation([swap[0]], [[swap[0], swap[1]]], [[swap[0], swap[2]]], "SWAP"))
                                self.all_operations2["SWAP" + str(swap_count)] = Operation([swap[0]], [[swap[0], swap[1]]], [[swap[0], swap[2]]], "SWAP")
                                swap_count += 1

                            # The array "operation_nodes" contains all nodes that are involved in the fusion or
                            # distillation operations. This is different from "inv_nodes": "inv_nodes" are all nodes of
                            # the final state, but the operations can only act on a subset of "inv_nodes".
                            if op == "Fuse":
                                operation_nodes = [
                                    find_fused_node(self.link_nodes[left_child_id], self.link_nodes[right_child_id])]
                                correction_nodes_qb = []
                                for node_qb in correction_qb_zip:
                                    if node_qb[0] != operation_nodes[0]:
                                        correction_nodes_qb.append(node_qb)
                                dist_op = None
                            else:
                                operation_nodes = self.link_nodes[right_child_id]
                                dist_operator, dist_op = \
                                    find_distillation_operator(self.link_values[parent_id].dec, inv_nodes)

                            # To the array "mem_locs" we are going to add the qubits that are involved in the two-qubit
                            # operations that are part of the fusion or distillation operations, but are not measured
                            # out.
                            mem_locs = []
                            for i_node in range(len(inv_nodes)):
                                node_nr = inv_nodes[i_node]
                                for node_nr_op in operation_nodes:
                                    if node_nr == node_nr_op:
                                        mem_locs.append([node_nr, self.link_memory_loc[parent_id][i_node]])
                                        break

                            # Here, we are finding the children of the current link, plus its Failure Reset Level:
                            children_ids, frl_id = self._find_reset_id(parent_id)

                            # Here, we add the operation itself to the protocol recipe:
                            e_qubits = [[qb, 0] for qb in operation_nodes]

                            state_mem = [list(mem_qub) for mem_qub in
                                         list(zip(inv_nodes, self.link_memory_loc[parent_id]))]

                            self.all_operations.append(Operation(operation_nodes, e_qubits, mem_locs, op.upper(),
                                                                 dist_op, link_id=parent_id, children=[left_child_id,
                                                                                                       right_child_id],
                                                                 family_tree=children_ids, frl=frl_id, frl_id=frl_id,
                                                                 nodes_final_state=inv_nodes, state_mem=state_mem))
                            self.all_operations2[parent_id] = Operation(operation_nodes, e_qubits, mem_locs, op.upper(),
                                                                 dist_op, link_id=parent_id, children=[left_child_id,
                                                                                                       right_child_id],
                                                                 family_tree=children_ids, frl=frl_id, frl_id=frl_id,
                                                                 nodes_final_state=inv_nodes, state_mem=state_mem)

                            if op == "Fuse":
                                for qubit in correction_nodes_qb:
                                    self.all_operations[-1].fusion_corrections.append(
                                        FusionCorrection(qubit=qubit, condition=[[], [parent_id]]))
                                    self.all_operations2[parent_id].fusion_corrections.append(
                                        FusionCorrection(qubit=qubit, condition=[[], [parent_id]]))

                            id_created[parent_id] = True
                            link_id = parent_id
                    else:
                        break
            i += 1

        for i_op, operation in enumerate(self.all_operations):
            operation.i_op = i_op

    def _update_qubit_memory(self, type, link1_id, link2_id=None):
        nodes_nrs = self.link_nodes[link1_id]
        swapped_qubits = []
        parent_id = None

        if type == "create_link":
            for node_nr in nodes_nrs:
                if self.qubit_memory[node_nr][0] is not None:  # If the first memory slot of this node is not None
                    stored_link_id = self.qubit_memory[node_nr][0]  # we have to swap the qubit sitting there
                    allocated_memory = False
                    mem_loc = 1
                    while allocated_memory is False:
                        if len(self.qubit_memory[node_nr]) == mem_loc:  # If the number of qubits in node nodes_nr is not
                            self.qubit_memory[node_nr].append(None)  # big enough, we add one empty qubit memory slot
                        if self.qubit_memory[node_nr][mem_loc] is None:  # Replace the state at the first empty memory slot
                            self.qubit_memory[node_nr][mem_loc] = stored_link_id
                            swapped_qubits.append([node_nr, 0, mem_loc])
                            # Here, we update "self.link_memory_loc" to make sure the entry "stored_link_id" has the
                            # correct memory location again for node "node_nr" (because it is swapped with the new
                            # link):
                            index_link_memory_loc = self.link_nodes[stored_link_id].index(node_nr)
                            self.link_memory_loc[stored_link_id][index_link_memory_loc] = mem_loc
                            allocated_memory = True
                        mem_loc = mem_loc + 1
                self.qubit_memory[node_nr][0] = link1_id  # Add the new state to the e- qubit of the memory
            self.link_memory_loc[link1_id] = [0] * len(nodes_nrs)  # Add the list of memory locations to node_list

        elif type == "Distill":
            assert link2_id is not None, "For a distillation step there are two links necessary to update the memory."
            # Find the state that is measured out in the memory, move it to the first memory position (the electron
            # qubit), and then measure it out and clear out the memory:
            for i_node_nr_2, node_nr_2 in enumerate(self.link_nodes[link2_id]):
                stored_link_id = self.qubit_memory[node_nr_2][0]
                if link2_id != stored_link_id:
                    # If this is the case, there is another state at the electron qubit and the two states have to be
                    # swapped first:
                    mem_loc = self.link_memory_loc[link2_id][i_node_nr_2]
                    self.qubit_memory[node_nr_2][mem_loc] = stored_link_id      # The state originally at the e- qubit
                    swapped_qubits.append([node_nr_2, 0, mem_loc])
                    if stored_link_id is not None:                  # It could be that the electron qubit was empty
                        index_link_memory_loc = self.link_nodes[stored_link_id].index(node_nr_2)
                        assert self.link_memory_loc[stored_link_id][index_link_memory_loc] == 0, "Error!"
                        self.link_memory_loc[stored_link_id][index_link_memory_loc] = mem_loc
                    else:
                        pass  # swap_count if efficient if stored_link_id is None (if the e- qubit was empty)
                self.qubit_memory[node_nr_2][0] = None      # "Measure out" the qubits of the second state
            self.link_memory_loc[link2_id] = None           # Update "link_memory_loc" for removing the second state
            parent_id = self.link_parent_id[link1_id]
            mem_locs = self.link_memory_loc[link1_id]
            for i_nn, node_nr in enumerate(nodes_nrs):
                self.qubit_memory[node_nr][mem_locs[i_nn]] = parent_id
            self.link_memory_loc[parent_id] = mem_locs      # Add the locations to node_list of the parent_id
            self.link_memory_loc[link1_id] = None           # Remove the state from the id of the state itself

        elif type == "Fuse":
            assert link2_id is not None, "For a fusion step there are two links necessary to update the memory."
            # Find the node number of the fused node:
            fused_node_nr = find_fused_node(self.link_nodes[link1_id], self.link_nodes[link2_id])
            # Find the index of state "link2_id" at the fused node "fused_node_nr":
            i_fused_node = self.link_nodes[link2_id].index(fused_node_nr)
            # "Measure out" the overlapping qubit by first putting it on the electron qubit:
            stored_link_id = self.qubit_memory[fused_node_nr][0]
            if link2_id != stored_link_id:
                # If this is the case, there is another state at the e- qubit and the two have to be swapped first:
                mem_loc = self.link_memory_loc[link2_id][i_fused_node]
                self.qubit_memory[fused_node_nr][mem_loc] = stored_link_id  # State originally at e- is put at new place
                swapped_qubits.append([fused_node_nr, 0, mem_loc])
                if stored_link_id is not None:  # It could be that the e- qubit was empty
                    index_link_memory_loc = self.link_nodes[stored_link_id].index(fused_node_nr)
                    assert self.link_memory_loc[stored_link_id][index_link_memory_loc] == 0, "Error!"
                    self.link_memory_loc[stored_link_id][index_link_memory_loc] = mem_loc
                else:
                    pass  # swap_count if efficient if stored_link_id is None
            self.qubit_memory[fused_node_nr][0] = None  # "Measure out" the qubit by removing it from the memory
            parent_id = self.link_parent_id[link1_id]
            parent_nodes_nrs = self.link_nodes[parent_id]
            mem_locs = [None] * len(parent_nodes_nrs)
            # Loop over all node nrs of the parent to update the link id numbers and the memory locations of the old
            # and new state:
            for i_nnp, node_nr_p in enumerate(parent_nodes_nrs):
                for link_id in [link1_id, link2_id]:
                    if node_nr_p in self.link_nodes[link_id] and \
                            not (node_nr_p == fused_node_nr and link_id == link2_id):
                        index_link_memory_loc = self.link_nodes[link_id].index(node_nr_p)
                        mem_loc = self.link_memory_loc[link_id][index_link_memory_loc]
                        # Here, we update the memory (replace the old link value with the new one). We save "mem_locs"
                        # to in the end add to "link_memory_loc[parent_id]". The node that is measured out is already
                        # set to "None" above already.
                        self.qubit_memory[node_nr_p][mem_loc] = parent_id
                        mem_locs[i_nnp] = mem_loc
            self.link_memory_loc[link2_id] = None  # Clear out the memory information in node_list for the two states
            self.link_memory_loc[link1_id] = None  # involved in the fusion.
            self.link_memory_loc[parent_id] = mem_locs  # Add the memory location to the parent (the new state after fusion).

        for nod_qub in swapped_qubits:
            for qb in range(1, 3):
                id_nr = self.qubit_memory[nod_qub[0]][nod_qub[qb]]
                if id_nr not in [link1_id, link2_id, parent_id] and id_nr is not None:
                    mem_ind = self.all_operations2[id_nr].state_mem.index([nod_qub[0], nod_qub[(qb + 1) % 2]])
                    self.all_operations2[id_nr].state_mem[mem_ind] = [nod_qub[0], nod_qub[qb]]

        return swapped_qubits

    def _check_if_parent_is_ready(self, parent_id, id_created):
        """
        Function that checks if a protocol link can be created by checking if its two children are created.

        Parameters
        ----------
        parent_id : int
            id number of the link in the protocol
        id_created : list with Boolean values
            Indicates whether a protocol link is created (True) or not (False)

        Returns
        -------
        Boolean indicating whether the link can be created (True) or not (False)
        """
        left_child_id = self.link_children[parent_id][0]
        right_child_id = self.link_children[parent_id][1]
        if id_created[left_child_id] is True and id_created[right_child_id] is True:
            return True
        else:
            return False

    def _find_reset_id(self, node_id):
        """
        Function that checks what the id is of the link that forms the failure reset level of the current node, and
        what all the children id's are of the current node (all these nodes have to be done again when a distillation
        step fails).

        Parameters
        ----------
        node_id : int
            id number of the distillation of fusion link in the protocol

        Returns
        -------
        children : list
            List of id numbers of all the children of node_id.
        frl_id : int
            id number of the failure reset level elementary link of node_id.
        """
        frl_id = None
        new_node_id = self.link_children[node_id][0]
        subtree_stack = [self.link_children[node_id][1]]
        subtree = []
        while new_node_id or subtree_stack:
            while new_node_id:
                if self.link_children[new_node_id][1]:
                    subtree_stack.append(self.link_children[new_node_id][1])
                subtree.append(new_node_id)
                new_node_id = self.link_children[new_node_id][0]
            if subtree_stack:
                new_node_id = subtree_stack.pop()
        for i in range(len(self.id_elem)):
            if self.id_elem[i][0] in subtree:
                frl_id = self.id_elem[i][0]
                break
        return subtree, frl_id

    def _group_operations(self):
        """
        Function that prints the protocol recipe and groups the events smartly in such a way that consecutive operations
        between the same nodes are grouped in one time event.

        Parameters
        ----------
        # prot_ops : list of operations
        #     Contains the operations list for the protocol.
        # fusion_corrections : dictionary with possible corrections necessary with fusion operations
        #     The keys of the dictionary are the link_id numbers of the fusion links in the binary tree protocol. Every time
        #     a fusion operation is carried out and results in a -1 result, we have carry out a Pauli correction somewhere
        #     in the network. Ideally, we want to carry out the correction at the end of a time step.
        # swap_count : positive integer
        #     Integer that indicates how many swap-gates have been carried out in the memory.
        # print_recipe : Boolean (default=True)
        #     Parameter that determines whether the eventual recipe, now grouped together more smartly in time steps, should
        #     be printed in the console.

        Returns
        -------
        # marker : Boolean
        #     Output marker that makes it possible to mark protocols that have specific properties.
        """
        show_correction = False

        list_ops = self.all_operations

        marker = False
        tsteps = [-1, -1]
        nodes_per_time_step = []
        for operation in list_ops:
            inv_nodes = operation.nodes  # The involved nodes at the current operation
            max_ts = max(tsteps)
            ts = max_ts + 1
            found_subsystem = False
            no_subsystem_possible = False
            found_empty_ssys = False
            while found_subsystem is False and no_subsystem_possible is False and ts > 0:
                ts -= 1
                for ssys in range(2):
                    overlapping_nodes = nodes_overlap(inv_nodes, nodes_per_time_step[ts][ssys])
                    if overlapping_nodes == sorted(inv_nodes):
                        found_subsystem = True
                        break
                    elif overlapping_nodes:
                        no_subsystem_possible = True
                        # TO DO: In some situations, it is still possible to move an operation in front of an operation
                        # involving one or more of the same nodes: if these operations don't have each other in their
                        # subtrees. Implementing this requires making a function that deals with changing the order
                        # of operations from how they were determined at "self._identify_operations".
                        break
                    elif nodes_per_time_step[ts][ssys] is None:
                        found_empty_ssys = [ts, ssys]
                        for ts2 in range(ts, max_ts + 1):
                            for ssys2 in range(2):
                                if nodes_overlap(inv_nodes, nodes_per_time_step[ts2][ssys2]):
                                    found_empty_ssys = False
            if found_subsystem:
                tstep = ts
                ib = ssys
            elif found_empty_ssys:
                tstep = found_empty_ssys[0]
                ib = found_empty_ssys[1]
                tsteps[ib] = tstep
                if not tuple(sorted(inv_nodes)) in self.subsystems.keys():
                    self.subsystems[tuple(sorted(inv_nodes))] = SubSystem(sorted(inv_nodes), self.n)
                nodes_per_time_step[tstep][ib] = inv_nodes
            else:
                # Make new subsystem
                nodes_per_time_step.append([None, None])
                self.time_blocks.append([TimeBlock(), TimeBlock()])
                self.fusion_corrections.append({})
                self.delayed_distillation_check.append({})

                tstep = max_ts + 1
                ib = 0
                tsteps[ib] = tstep
                if not tuple(sorted(inv_nodes)) in self.subsystems.keys():
                    self.subsystems[tuple(sorted(inv_nodes))] = SubSystem(sorted(inv_nodes), self.n)
                nodes_per_time_step[tstep][ib] = inv_nodes

            if operation.type == "CREATE_LINK":  # Count the number of LDE links per time step
                self.time_blocks[tstep][ib].elem_links += 1  # and per subsystem.

            if operation.type == "FUSE":
                for fusion_correction in operation.fusion_corrections:
                    if not tuple(fusion_correction.qubit) in self.fusion_corrections[tstep].keys():
                        self.fusion_corrections[tstep][tuple(fusion_correction.qubit)] = fusion_correction
                        # protocol_recipe.fusion_corrections[tstep][tuple(fusion_correction.qubit)].sub_block.append(ib)
                    else:
                        # Here we only have to include "condition[1]" of the fusion correction listed at the operation,
                        # because by default there are only X-corrections involved when an operations is just added.
                        self.fusion_corrections[tstep][tuple(fusion_correction.qubit)].condition[1].append(
                            fusion_correction.condition[1][0])
                        # if ib not in protocol_recipe.fusion_corrections[tstep][tuple(fusion_correction.qubit)].sub_block:
                        #     protocol_recipe.fusion_corrections[tstep][tuple(fusion_correction.qubit)].sub_block.append(ib)

            self._update_operations_based_on_fusion_corrections(operation, tstep)

            current_subsystem = self.subsystems[tuple(sorted(nodes_per_time_step[tstep][ib]))]
            operation.subsystem = current_subsystem

            # Here, we add the operation to the list of operations of the protocol recipe. We also make a dictionary
            # indicating in which sub block, and in which time step of the sub block, each operation is carried out.
            # The dictionary is indexed by the link_id of each operation.
            if operation.type != "SWAP":
                if not operation.link_id in self.id_link_structure.keys():
                    self.id_link_structure[operation.link_id] = [tstep,
                                                                 ib,
                                                                 len(self.time_blocks[tstep][ib].list_of_operations)]
            self.time_blocks[tstep][ib].list_of_operations.append(operation)
            if self.time_blocks[tstep][ib].subsystem is None:
                self.time_blocks[tstep][ib].subsystem = current_subsystem

        # Old way of making the dictionary indicating in which sub block an operation is stored. I am making the
        # id_link_structure again here from scratch, to make sure the dictionary items are placed in the dictionary
        # in the correct order again (not in the order in which the operations are placed in the protocol recipe). This
        # is required for finding the failure reset level of the distillation steps: it needs to find the first link
        # in the recipe that needs to be recreated.
        self.id_link_structure = {}
        for i_ts, ts in enumerate(self.time_blocks):
            for i_ssys, ssys in enumerate(ts):
                for i_op, operation in enumerate(ssys.list_of_operations):
                    link_id = operation.link_id
                    if operation.type != "SWAP":
                        if not link_id in self.id_link_structure.keys():
                            self.id_link_structure[link_id] = [i_ts, i_ssys, i_op]

        # Make sure the subsystem with all nodes is in the list of subsystems, for the fusion corrections:
        if tuple([*range(self.n)]) not in self.subsystems.keys():
            self.subsystems[tuple([*range(self.n)])] = SubSystem([*range(self.n)], self.n)

        # Here we check for all distillation whether or not the post-selection should be evaluated directly after the
        # measurements, or should be delayed until after the end of the sub block (in case fusion operation corrections
        # might influence the distillation measurement result that we call successful). We also update the failure
        # reset levels of all distillation operations. (IS THIS LAST STATEMENT TRUE? I DON'T THINK SO.)
        self._identify_distillation_post_measurement_delays()

    def _update_operations_based_on_fusion_corrections(self, operation, tstep):
        """
            Here, we update operations and fusion corrections based on the list of fusion corrections that are being
            carried out after the time steps.
        """
        if self.print_debug is True:
            print("")
            print(tstep)
            operation.print_operation()
            if operation.type == "FUSE":
                for fc in operation.fusion_corrections:
                    fc.print_fusion_correction()
            print(self.fusion_corrections)

        for i_ts in range(len(self.fusion_corrections) - 1, tstep - 1, -1):
            if len(self.fusion_corrections[i_ts]) > 0:
                fusion_correction_qubits = [*self.fusion_corrections[i_ts].keys()]
                # print(operation.i_op, fusion_correction_qubits)
                if operation.type == "SWAP":
                    self._update_swap_operations_fusion_corr(operation, fusion_correction_qubits, i_ts)
                elif operation.type == "FUSE":
                    self._update_fuse_operations_fusion_corr(operation, fusion_correction_qubits, i_ts, tstep)
                elif operation.type == "DISTILL":
                    self._update_distill_operations_fusion_corr(operation, fusion_correction_qubits, i_ts)
                # fusion_correction_qubits_after = [*self.fusion_corrections[i_ts].keys()]
                # print(operation.i_op, fusion_correction_qubits_after)
        if operation.type == "FUSE" and self.print_debug is True:
            for fc in operation.fusion_corrections:
                fc.print_fusion_correction()


    def _update_swap_operations_fusion_corr(self, operation, fusion_correction_qubits, i_ts):
        """
            In case of a swap operation it is easy: we just have to adjust the qubits on which fusion correction
            depends based on the swap operation.
        """
        corrected_qubits = []   # List with correction that are already being taken into account.
        for fc_qubit in fusion_correction_qubits:
            if list(fc_qubit) not in corrected_qubits and \
                    (list(fc_qubit) == operation.e_qubits[0] or list(fc_qubit) == operation.m_qubits[0]):
                new_qubit = operation.e_qubits[0] if list(fc_qubit) == operation.m_qubits[0] \
                    else operation.m_qubits[0]
                if not tuple(new_qubit) in self.fusion_corrections[i_ts].keys():
                    self.fusion_corrections[i_ts][tuple(fc_qubit)].qubit = new_qubit
                    self.fusion_corrections[i_ts][tuple(new_qubit)] = self.fusion_corrections[i_ts][tuple(fc_qubit)]
                    del self.fusion_corrections[i_ts][tuple(fc_qubit)]
                else:
                    corrected_qubits.append(new_qubit)
                    swap_corr = self.fusion_corrections[i_ts][tuple(fc_qubit)]
                    self.fusion_corrections[i_ts][tuple(fc_qubit)] = self.fusion_corrections[i_ts][tuple(new_qubit)]
                    self.fusion_corrections[i_ts][tuple(new_qubit)] = swap_corr
                    self.fusion_corrections[i_ts][tuple(new_qubit)].qubit = new_qubit
                    self.fusion_corrections[i_ts][tuple(fc_qubit)].qubit = list(fc_qubit)

    def _update_fuse_operations_fusion_corr(self, operation, fusion_correction_qubits, i_ts, tstep):
        """
            At each fusion operation, there are two qubits involved. If there is a possible correction from an earlier
            fusion operation attached to the electron qubit of the current fusion operation, this essentially flips the
            current fusion operation: we add the condition of the previous fusion measurement to the fusion corrections
            associated with the current fusion operation, and remove the fusion correction from the list.
        """
        for fc_qubit in fusion_correction_qubits:
            condition_prev = self.fusion_corrections[i_ts][fc_qubit].condition
            if list(fc_qubit) == operation.e_qubits[0]:
                if condition_prev[0]:
                    # In this case, there are possible Z gates applied, which carry over as Z gate corrections to the
                    # "m_qubits[0]" qubit of the current fusion operation.
                    Z_corr_qb = operation.m_qubits[0]
                    if not tuple(Z_corr_qb) in self.fusion_corrections[i_ts].keys():
                        self.fusion_corrections[i_ts][tuple(Z_corr_qb)] = \
                            FusionCorrection(Z_corr_qb, condition=[[], []])
                    for qb_prev in condition_prev[0]:
                        self.fusion_corrections[i_ts][tuple(Z_corr_qb)].condition[0].append(qb_prev)
                    if condition_prev[1]:
                        self.fusion_corrections[i_ts][tuple(fc_qubit)].condition[0] = []
                    else:
                        del self.fusion_corrections[i_ts][tuple(fc_qubit)]
                if condition_prev[1]:
                    # In this case there are possible X gates applied, which result in a possible "flip" of the
                    # measurement outcome of the current fusion operation.
                    for fusion_correction in operation.fusion_corrections:
                        qb_curr = fusion_correction.qubit
                        added_link_dependencies = []
                        for cond_prev in condition_prev[1]:
                            for i_cond in range(2):
                                if self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond]:
                                    if cond_prev not in self.fusion_corrections[tstep][tuple(qb_curr)].condition[
                                        i_cond]:
                                        self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond].append(
                                            cond_prev)
                                        added_link_dependencies.append(cond_prev)
                                    else:
                                        del self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond][
                                            self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond].index(
                                                cond_prev)]
                        if i_ts > tstep:
                            if len(added_link_dependencies) > 1:
                                print("Something strange is happening.")
                            else:
                                if added_link_dependencies:
                                    loc_id = self.id_link_structure[added_link_dependencies[0]]
                                    self._move_fusion_corrections_to_later_time_step([qb_curr], tstep, loc_id[0],
                                                                                     [loc_id[1], loc_id[2] + 1])
                    if condition_prev[0]:
                        self.fusion_corrections[i_ts][tuple(fc_qubit)].condition[1] = []
                    else:
                        del self.fusion_corrections[i_ts][tuple(fc_qubit)]
            if list(fc_qubit) == operation.m_qubits[0]:
                if condition_prev[1]:
                    # In this case there are X gates applied: they carry over to after the operation (so they stay), but
                    # they also possibly flip the measurement outcome of the current fusion operation.
                    for fusion_correction in operation.fusion_corrections:
                        qb_curr = fusion_correction.qubit
                        added_link_dependencies = []
                        for cond_prev in condition_prev[1]:
                            for i_cond in range(2):
                                if self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond]:
                                    if cond_prev not in self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond]:
                                        self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond].append(
                                            cond_prev)
                                        added_link_dependencies.append(cond_prev)
                                    else:
                                        del self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond][
                                            self.fusion_corrections[tstep][tuple(qb_curr)].condition[i_cond].index(
                                                cond_prev)]
                        if i_ts > tstep:
                            if len(added_link_dependencies) > 1:
                                print("Something strange is happening.")
                            else:
                                if added_link_dependencies:
                                    loc_id = self.id_link_structure[added_link_dependencies[0]]
                                    self._move_fusion_corrections_to_later_time_step([qb_curr], tstep, loc_id[0],
                                                                                     [loc_id[1], loc_id[2] + 1])

    def _move_fusion_corrections_to_later_time_step(self, fusion_correction_qubits, tstep_old, tstep_new, i_opblock=None):
        """
            This function can be used when the system has to move a fusion_correction to a later time step. This
            involves updating the fusion_correction with all intermediate operations acting on the same qubit.
        """
        for fcq in fusion_correction_qubits:
            fusion_correction = self.fusion_corrections[tstep_old][tuple(fcq)]
            new_fusion_corrections = {tuple(fcq): fusion_correction}

            if i_opblock is not None:
                tstep_start = tstep_new
                i_op_start = i_opblock[0]
                i_ssys_start = i_opblock[1]
            else:
                tstep_start = tstep_old + 1
                i_op_start = 0
                i_ssys_start = 0
            for i_ts in range(tstep_start, tstep_new + 1):
                i_ssys_max = len((self.time_blocks[i_ts])) if i_opblock is not None else i_ssys_start + 1
                for i_ssys in range(i_ssys_start, i_ssys_max):
                    for i_op in range(i_op_start, len(self.time_blocks[i_ts][i_ssys].list_of_operations)):
                        operation = self.time_blocks[i_ts][i_ssys].list_of_operations[i_op]
                        new_fusion_correction_qubits = list(new_fusion_corrections.keys())
                        for c_qubit in new_fusion_correction_qubits:
                            fcc = new_fusion_corrections[tuple(c_qubit)]
                            condition_prev = fcc.condition
                            if operation.type == "SWAP":
                                if list(c_qubit) == operation.e_qubits[0] or list(c_qubit) == operation.m_qubits[0]:
                                    new_qubit = operation.e_qubits[0] if list(c_qubit) == operation.m_qubits[0] \
                                        else operation.m_qubits[0]
                                    if not tuple(new_qubit) in new_fusion_corrections.keys():
                                        new_fusion_corrections[tuple(c_qubit)].qubit = new_qubit
                                        new_fusion_corrections[tuple(new_qubit)] = new_fusion_corrections[tuple(c_qubit)]
                                        del new_fusion_corrections[tuple(c_qubit)]
                                    else:
                                        swap_corr = new_fusion_corrections[tuple(c_qubit)]
                                        new_fusion_corrections[tuple(c_qubit)] = new_fusion_corrections[tuple(new_qubit)]
                                        new_fusion_corrections[tuple(new_qubit)] = swap_corr
                                        new_fusion_corrections[tuple(new_qubit)].qubit = new_qubit
                                        new_fusion_corrections[tuple(c_qubit)].qubit = c_qubit
                            elif operation.type == "FUSE":
                                if list(c_qubit) == operation.e_qubits[0]:
                                    if condition_prev[0]:
                                        # In this case, there are possible Z gates applied, which carry over as Z gate corrections to the
                                        # "m_qubits[0]" qubit of the current fusion operation.
                                        Z_corr_qb = operation.m_qubits[0]
                                        if not tuple(Z_corr_qb) in new_fusion_corrections.keys():
                                            new_fusion_corrections[tuple(Z_corr_qb)] = \
                                                FusionCorrection(Z_corr_qb, condition=[condition_prev[0], []])
                                        else:
                                            for qb_prev in condition_prev[0]:
                                                new_fusion_corrections[tuple(Z_corr_qb)].condition[0].append(qb_prev)
                                        if condition_prev[1]:
                                            new_fusion_corrections[tuple(c_qubit)].condition[0] = []
                                        else:
                                            del new_fusion_corrections[tuple(c_qubit)]
                                    if condition_prev[1]:
                                        # In this case there are possible X gates applied, which result in a possible "flip" of the
                                        # measurement outcome of the current fusion operation.
                                        for fusion_correction in operation.fusion_corrections:
                                            qb_curr = fusion_correction.qubit
                                            if not tuple(qb_curr) in new_fusion_corrections.keys():
                                                new_fusion_corrections[tuple(qb_curr)] = \
                                                    FusionCorrection(qb_curr, condition=[[], condition_prev[1]])
                                            else:
                                                for cond_prev in condition_prev[1]:
                                                    new_fusion_corrections[tuple(qb_curr)].condition[1].append(cond_prev)
                                        if condition_prev[0]:
                                            new_fusion_corrections[tuple(c_qubit)].condition[1] = []
                                        else:
                                            del new_fusion_corrections[tuple(c_qubit)]
                                if list(c_qubit) == operation.m_qubits[0]:
                                    if condition_prev[1]:
                                        # In this case there are X gates applied: they carry over to after the operation (so they stay), but
                                        # they also possibly flip the measurement outcome of the current fusion operation.
                                        for fusion_correction in operation.fusion_corrections:
                                            qb_curr = fusion_correction.qubit
                                            if not tuple(qb_curr) in new_fusion_corrections.keys():
                                                new_fusion_corrections[tuple(qb_curr)] = \
                                                    FusionCorrection(qb_curr, condition=[[], condition_prev[1]])
                                            else:
                                                for cond_prev in condition_prev[1]:
                                                    new_fusion_corrections[tuple(qb_curr)].condition[1].append(cond_prev)
                            elif operation.type == "DISTILL":
                                for i_dist in range(len(operation.operator)):
                                    op_e_qubit = operation.e_qubits[i_dist]
                                    if list(c_qubit) == op_e_qubit:
                                        # For all conditional Z operators executed on this qubit we adjust the success dependency of
                                        # this distillation operation: if the success of the distillation operation already depended
                                        # on this previous operation measurement result, we remove, and if it didn't already depend
                                        # on it, we add it to the list:
                                        for cond in condition_prev[0]:
                                            if cond in operation.success_dep:
                                                del operation.success_dep[operation.success_dep.index(cond)]
                                            else:
                                                operation.success_dep.append(cond)
                                        # We also have to take into account the influence of all conditional X corrections executed
                                        # on this qubit. These do not influence the distillation measurement outcome, but move as
                                        # a conditional Z correction to the associated memory qubit in the same node.
                                        Z_corr_qb = operation.m_qubits[i_dist]
                                        if condition_prev[1]:
                                            if not tuple(Z_corr_qb) in new_fusion_corrections.keys():
                                                new_fusion_corrections[tuple(Z_corr_qb)] = \
                                                    FusionCorrection(Z_corr_qb, condition=[condition_prev[1], []])
                                            else:
                                                for cond_prev in condition_prev[1]:
                                                    new_fusion_corrections[tuple(Z_corr_qb)].condition[0].append(cond_prev)
                                        # This conditional correction can now be removed from the list, because it is applied to
                                        # a qubit that is measured out in the operation:
                                        del new_fusion_corrections[tuple(c_qubit)]
                                        # We can also break the for loop over the electron qubits of the operation, because we have
                                        # found a qubit that overlaps with "c_qubit", and there is only one maximally:
                                        break
                                    op_m_qubit = operation.m_qubits[i_dist]
                                    if list(c_qubit) == op_m_qubit:
                                        dist_op = operation.operator[i_dist]
                                        if dist_op == 2 or dist_op == 3:
                                            # In this case a conditional Z operation flips the measurement result of the
                                            # distillation operation.
                                            for cond in condition_prev[0]:
                                                if cond in operation.success_dep:
                                                    del operation.success_dep[operation.success_dep.index(cond)]
                                                else:
                                                    operation.success_dep.append(cond)
                                        if dist_op == 1 or dist_op == 3:
                                            # In this case a conditional X operation flips the measurement result of the
                                            # distillation operation.
                                            for cond in condition_prev[1]:
                                                if cond in operation.success_dep:
                                                    del operation.success_dep[operation.success_dep.index(cond)]
                                                else:
                                                    operation.success_dep.append(cond)
                                        break

            for c_qubit in new_fusion_corrections.keys():
                fcc = new_fusion_corrections[tuple(c_qubit)]
                if not tuple(c_qubit) in self.fusion_corrections[tstep_new]:
                    self.fusion_corrections[tstep_new][tuple(fcq)] = fcc
                else:
                    for i_cond in range(2):
                        for cond_qb in fcc.condition[i_cond]:
                            if cond_qb in self.fusion_corrections[tstep_new][tuple(c_qubit)].condition[i_cond]:
                                del self.fusion_corrections[tstep_new][tuple(c_qubit)].condition[i_cond][
                                    self.fusion_corrections[tstep_new][tuple(c_qubit)].condition[i_cond].index(cond_qb)]
                            else:
                                self.fusion_corrections[tstep_new][tuple(c_qubit)].condition[i_cond].append(cond_qb)
            del self.fusion_corrections[tstep_old][tuple(fcq)]

    def _update_distill_operations_fusion_corr(self, operation, fusion_correction_qubits, i_ts):
        """
            Here, we have to establish whether or not the fusion correction options commute with the operator
            that is been measured in the distillation operation. If it commutes with all correction possibilities,
            the distillation operation can just proceed as usual. If it anti-commutes with correction whose
            execution is determined by a fusion operation from another subsystem (i.e., with information that is
            not available in the current subsystem), we have to delay the evaluation of the measurement results
            of the distillation operation till after both subsystems have finished. We do this by setting the
            "delay_after_sub_block" property of the current operation to True.

            Now, we check if the distillation operation has qubits that are also in the list of qubits that might
            receive a correction from a fusion operation in the same time step. This should also mean that the
            link_id of this fusion operation is part of the subtree of the distillation operation. If this occurs,
            and the correction operator does not commute with the operator that is measured non-locally in the
            distillation procedure, we have to come up with some mechanism to delay the evaluation of the measurement
            result till after the time step is fully completed. This is of course only relevant if the overlapping
            qubits in the subsystem in question are triggered by operations in the other subsystem (this should be
            checked).
        """
        for fc_qubit in fusion_correction_qubits:
            condition_prev = self.fusion_corrections[i_ts][tuple(fc_qubit)].condition
            for i_dist, dist_op in enumerate(operation.operator):
                op_e_qubit = operation.e_qubits[i_dist]
                if list(fc_qubit) == op_e_qubit:
                    # For all conditional Z operators executed on this qubit we adjust the success dependency of
                    # this distillation operation: if the success of the distillation operation already depended
                    # on this previous operation measurement result, we remove, and if it didn't already depend
                    # on it, we add it to the list. We have to be careful with Y operations on the electron qubit.
                    # For those the dependency only changes when there is either a Z or an X operator on the electron
                    # qubit; not when there are both.
                    if dist_op in [1, 2]:
                        for cond in condition_prev[0]:
                            if cond in operation.success_dep:
                                del operation.success_dep[operation.success_dep.index(cond)]
                            else:
                                operation.success_dep.append(cond)
                    elif dist_op == 3:
                        for cond in condition_prev[1]:
                            if cond not in condition_prev[0] and cond in operation.success_dep:
                                del operation.success_dep[operation.success_dep.index(cond)]
                            else:
                                operation.success_dep.append(cond)
                        for cond in condition_prev[0]:
                            if cond not in condition_prev[1] and cond in operation.success_dep:
                                del operation.success_dep[operation.success_dep.index(cond)]
                            else:
                                operation.success_dep.append(cond)
                    # We also have to take into account the influence of all conditional X corrections executed
                    # on this qubit. These do not influence the distillation measurement outcome, but move as
                    # a conditional Z correction (for a Z distillation operator), an X correction (for an X distillation
                    # operator) or a Y correction (for a Y distillation operator) to the associated memory qubit in the
                    # same node.
                    Z_corr_qb = operation.m_qubits[i_dist]
                    if condition_prev[1]:
                        if not tuple(Z_corr_qb) in self.fusion_corrections[i_ts].keys():
                            self.fusion_corrections[i_ts][tuple(Z_corr_qb)] = \
                                FusionCorrection(Z_corr_qb, condition=[[], []])
                        for cond_prev in condition_prev[1]:
                            if dist_op in [1, 3]:
                                self.fusion_corrections[i_ts][tuple(Z_corr_qb)].condition[0].append(
                                    cond_prev)
                            if dist_op in [2, 3]:
                                self.fusion_corrections[i_ts][tuple(Z_corr_qb)].condition[1].append(
                                    cond_prev)
                    # This conditional correction can now be removed from the list, because it is applied to
                    # a qubit that is measured out in the operation:
                    del self.fusion_corrections[i_ts][fc_qubit]
                    # We can also break the for loop over the electron qubits of the operation, because we have
                    # found a qubit that overlaps with "fc_qubit", and there is only one maximally:
                    break
                op_m_qubit = operation.m_qubits[i_dist]
                if list(fc_qubit) == op_m_qubit:
                    if dist_op == 2 or dist_op == 3:
                        # In this case a conditional Z operation flips the measurement result of the
                        # distillation operation.
                        for cond in condition_prev[0]:
                            if cond in operation.success_dep:
                                del operation.success_dep[operation.success_dep.index(cond)]
                            else:
                                operation.success_dep.append(cond)
                    if dist_op == 1 or dist_op == 3:
                        # In this case a conditional X operation flips the measurement result of the
                        # distillation operation.
                        for cond in condition_prev[1]:
                            if cond in operation.success_dep:
                                del operation.success_dep[operation.success_dep.index(cond)]
                            else:
                                operation.success_dep.append(cond)
                    break

    def _identify_distillation_post_measurement_delays(self):
        """
            Here, we figure out if we have to delay the evaluation of the distillation measurements (i.e., determining
            whether the distillation was successful or not) until after the complete time step (i.e., at the time where
            the fusion corrections are executed as well).
        """
        for i_ts, ts in enumerate(self.time_blocks):
            for i_ssys, ssys in enumerate(ts):
                for i_op, operation in enumerate(ssys.list_of_operations):
                    if operation.type == "DISTILL":
                        # Check if we have to delay the evaluation of the distillation measurement until after the sub
                        # block has finished:
                        if len(operation.success_dep) > 1:
                            ids_succ = [False] * len(operation.success_dep)
                            for i_sd, id_succ_dep in enumerate(operation.success_dep):
                                id_coor = self.id_link_structure[id_succ_dep]
                                if (id_coor[0] < i_ts) \
                                        or (id_coor[0] == i_ts and id_coor[1] == i_ssys and id_coor[2] <= i_op):
                                    ids_succ[i_sd] = True
                            if not all(ids_succ):
                                operation.delay_after_sub_block = True
                                self.delayed_distillation_check[i_ts][operation.link_id] = operation.success_dep

    def _create_time_step_qubit_memory(self):
        # In this function, we create an object that contains the links stored in the qubit memory register of the nodes
        # in each of the time steps of the final protocol recipe. We also assign two link_id numbers to the SWAP gates
        # involved.
        ts_qubit_mem = []
        for node in range(len(self.qubit_memory)):
            ts_qubit_mem.append([None] * len(self.qubit_memory[node]))
        tb_qubit_mems = []
        for i_ts, ts in enumerate(self.time_blocks):
            tb_qubit_mems.append([])
            for i_ssys, ssys in enumerate(ts):
                if ssys.list_of_operations:
                    tb_qubit_mems[i_ts].append([])
                    for i_op, operation in enumerate(ssys.list_of_operations):
                        tb_qubit_mems[i_ts][i_ssys].append(copy.deepcopy(ts_qubit_mem))
                        if operation.type == "DISTILL" or operation.type == "FUSE":
                            for i_node, node_qubit in enumerate(tb_qubit_mems[i_ts][i_ssys][i_op]):
                                for i_qubit, qubit in enumerate(node_qubit):
                                    if qubit in operation.family_tree:
                                        id_to_add = operation.link_id
                                        parent_in_memory = True
                                        while parent_in_memory:
                                            parent_in_memory = False
                                            parent_id = self.link_parent_id[id_to_add]
                                            if parent_id is not None:
                                                for node_qubit in tb_qubit_mems[i_ts][i_ssys][i_op]:
                                                    if parent_id in node_qubit:
                                                        parent_in_memory = True
                                                        id_to_add = parent_id
                                                        break
                                        tb_qubit_mems[i_ts][i_ssys][i_op][i_node][i_qubit] = id_to_add
                            for [node, qubit] in operation.e_qubits:
                                tb_qubit_mems[i_ts][i_ssys][i_op][node][qubit] = None
                        elif operation.type == "CREATE_LINK":
                            [node1, qubit1] = operation.e_qubits[0]
                            [node2, qubit2] = operation.e_qubits[1]
                            tb_qubit_mems[i_ts][i_ssys][i_op][node1][qubit1] = operation.link_id
                            tb_qubit_mems[i_ts][i_ssys][i_op][node2][qubit2] = operation.link_id
                        elif operation.type == "SWAP":
                            [node1, qubit1] = operation.e_qubits[0]
                            [node2, qubit2] = operation.m_qubits[0]
                            link_id1 = tb_qubit_mems[i_ts][i_ssys][i_op][node1][qubit1]
                            link_id2 = tb_qubit_mems[i_ts][i_ssys][i_op][node2][qubit2]
                            tb_qubit_mems[i_ts][i_ssys][i_op][node1][qubit1] = link_id2
                            tb_qubit_mems[i_ts][i_ssys][i_op][node2][qubit2] = link_id1
                            operation.link_id = [link_id1, link_id2]
                        ts_qubit_mem = tb_qubit_mems[i_ts][i_ssys][i_op]
        return tb_qubit_mems

    def _adjust_failure_reset_levels(self):
        """
            In this function, we adjust the failure reset levels and assign a list of link_id's that have be reset
            whenever the distillation operation in question fails.
        """
        for i_ts, ts in enumerate(self.time_blocks):
            for i_ssys, ssys in enumerate(ts):
                for i_op, operation in enumerate(ssys.list_of_operations):
                    if operation.type == "DISTILL":
                        links_to_be_reset = copy.deepcopy(operation.family_tree)
                        links_to_be_reset.append(operation.link_id)
                        if operation.delay_after_sub_block is True:
                            # In this situation we have to also take into account the operations that take place between
                            # the distillation measurement and the evaluation of the measurement: if these operations
                            # also use "operation" (or any of its links in the subtree), we have to redo them as well.
                            for i_ssys2 in range(i_ssys, len(ts)):
                                for i_op2 in range(i_op, len(self.time_blocks[i_ts][i_ssys2].list_of_operations)):
                                    operation2 = self.time_blocks[i_ts][i_ssys2].list_of_operations[i_op2]
                                    if operation2.type != "SWAP" and operation2.family_tree is not None:
                                        for l_id_ft in operation2.family_tree:
                                            if l_id_ft in links_to_be_reset:
                                                # If this is the case, "operation2" and "operation" have overlapping
                                                # elements in their subtrees, meaning that if we have to make the state
                                                # of "operation" again, we also have to make the state of "operation2"
                                                # again. However, in that case, we also have to make all other links in
                                                # the subtree of "operation2" again (they might not be included to make
                                                # "operation"). Therefore, we also include its entire subtree, and then
                                                # we remove all duplicates.
                                                links_to_be_reset.append(operation2.link_id)
                                                links_to_be_reset += operation2.family_tree
                                                links_to_be_reset = list(dict.fromkeys(links_to_be_reset))
                                                break
                            # In this situation we have to compare all operation with the qubit memory at the time of
                            # the evaluation point after the full time step.
                            i_ssys_c = len(ts) - 1
                            i_op_c = len(self.time_blocks[i_ts][i_ssys_c].list_of_operations) - 1
                        else:
                            i_ssys_c = i_ssys
                            i_op_c = i_op
                        qubit_register = copy.deepcopy(self.qubit_memory_per_time_step[i_ts][i_ssys_c][i_op_c])

                        # Here we check whether any of the parents of the links in links_to_be_reset is already executed
                        # at this point in time (which is most likely a fusion operation in that case): these links
                        # should of course also be recreated when the program is run again. The way the qubit_register
                        # is created now, if this is the case, the id of the parent should be present in the
                        # qubit_register:
                        for link_id_reset in links_to_be_reset:
                            parent_id_reset = self.link_parent_id[link_id_reset]
                            while parent_id_reset is not None:
                                for i_node in range(len(qubit_register)):
                                    for i_qubit in range(len(qubit_register[i_node])):
                                        if qubit_register[i_node][i_qubit] == parent_id_reset:
                                            links_to_be_reset.append(parent_id_reset)
                                            links_to_be_reset += self.all_operations2[parent_id_reset].family_tree
                                links_to_be_reset = list(dict.fromkeys(links_to_be_reset))
                                parent_id_reset = self.link_parent_id[parent_id_reset]

                        # Now, we check if we also need to recreate any other links that happen to sit on the memory
                        # locations that we require to make the links in links_to_be_reset:
                        found_frl = False
                        while found_frl is False:
                            # We remove the link_id's of all links that we are going to recreate to our local memory:
                            qubit_register = remove_link_ids_from_qubit_memory(qubit_register, links_to_be_reset)
                            for l_id in self.id_link_structure:
                                if l_id in links_to_be_reset:
                                    # This is the first link of "links_to_be_reset" that is created in the protocol.
                                    operation.frl_id = l_id
                                    break
                            frl_coor = self.id_link_structure[operation.frl_id]
                            # Here we collect all link_id's of links between "frl_coor" and "[i_ts, i_ssys_c, i_op_c]"
                            # that use the same memory locations as the links we are going to recreate. Therefore, we
                            # also need to recreate these links.
                            overlap_links = self._check_overlap_with_qubit_register(qubit_register, frl_coor,
                                                                                    [i_ts, i_ssys_c, i_op_c],
                                                                                    links_to_be_reset)
                            if len(overlap_links) == 0:
                                # If there were no overlapping links found, we know that this is the true failure reset
                                # level for this version of the protocol.
                                found_frl = True
                            else:
                                for link_id in overlap_links:
                                    link_c = self.id_link_structure[link_id]
                                    operation2 = self.time_blocks[link_c[0]][link_c[1]].list_of_operations[link_c[2]]
                                    if operation2.family_tree is not None:
                                        # To recreate all overlapping links, we have to of course first create all the
                                        # links in its subtree.
                                        links_to_be_reset += operation2.family_tree
                                    links_to_be_reset.append(link_id)
                                    # We remove possible duplicates:
                                    links_to_be_reset = list(dict.fromkeys(links_to_be_reset))
                            # found_frl = True
                        operation.frl = frl_coor
                        operation.fr_list = list(dict.fromkeys(links_to_be_reset))


    def _check_overlap_with_qubit_register(self, qubit_register, start_ts, end_ts, links_to_check):
        overlapping_links = []
        i_ts = start_ts[0]
        i_ssys = start_ts[1]
        i_op = start_ts[2]
        end_reached = False
        while not end_reached:
            if self.time_blocks[i_ts][i_ssys].list_of_operations:
                operation = self.time_blocks[i_ts][i_ssys].list_of_operations[i_op]
                if operation.type != "SWAP":
                    if operation.link_id in links_to_check:
                        qubits = [operation.e_qubits, operation.m_qubits]
                        for type_qubits in range(2):
                            if qubits[type_qubits] is not None:
                                for [node, qubit] in qubits[type_qubits]:
                                    if qubit_register[node][qubit] is not None: # and qubit_register[node][qubit] not in links_that_stay:
                                        overlapping_links.append(qubit_register[node][qubit])
                elif operation.type == "SWAP":
                    if operation.link_id[0] in links_to_check or operation.link_id[1] in links_to_check:
                        # If one of the link_id's that is used in the SWAP operation is actually a link_id that we have
                        # to recreate, we have to carry out this SWAP gate, and we have to include the link_id's that
                        # are stored at the qubit memory locations that the SWAP acts on, if they aren't already a part
                        # of links_to_check:
                        [[node1, qubit1], [node2, qubit2]] = [operation.e_qubits[0], operation.m_qubits[0]]
                        if qubit_register[node1][qubit1] is not None and qubit_register[node1][qubit1] not in links_to_check:
                            overlapping_links.append(qubit_register[node1][qubit1])
                        if qubit_register[node2][qubit2] is not None and qubit_register[node2][qubit2] not in links_to_check:
                            overlapping_links.append(qubit_register[node2][qubit2])
                if [i_ts, i_ssys, i_op] == end_ts:
                    end_reached = True
                i_op += 1
            if i_op == len(self.time_blocks[i_ts][i_ssys].list_of_operations):
                i_ssys += 1
                i_op = 0
                if i_ssys == len(self.time_blocks[i_ts]):
                    i_ts += 1
                    i_ssys = 0
        return list(dict.fromkeys(overlapping_links))

    def print_protocol_recipe(self, print_intermediate_memory=False):
        tb_qubit_mems = self._create_time_step_qubit_memory()
        print(colored("\nElementary link IDs in the order of id_elem:", "green"))
        print([elem_link[0] for elem_link in self.id_elem])
        print(colored("\nLink IDs of all link creation, fusion and distillation steps in order of execution:", "green"))
        print([id_link_struc for id_link_struc in self.id_link_structure])
        print(colored("\nProtocol recipe:", "green"))
        for i_ts, ts in enumerate(self.time_blocks):
            if i_ts != 0:
                print("")
            print(colored("Time step " + str(i_ts) + ":", "white"))
            for i_ssys, ssys in enumerate(ts):
                if ssys.list_of_operations:
                    print(colored(
                        "SUBSYSTEM " + str(ssys.subsystem.nodes) + " (" + str(ssys.elem_links) + " LDE attempts):",
                        "red"))
                    for i_op, operation in enumerate(ssys.list_of_operations):
                        operation.print_operation()
                        if print_intermediate_memory:
                            print(tb_qubit_mems[i_ts][i_ssys][i_op])
            if self.delayed_distillation_check[i_ts]:
                print(colored("CLASSICAL EVALUATION:", "red"))
            for dist_id in self.delayed_distillation_check[i_ts]:
                print_line = colored("EVALUATE", "magenta") + " the success of "
                oper_coor = self.id_link_structure[dist_id]
                oper_frl = self.time_blocks[oper_coor[0]][oper_coor[1]].list_of_operations[oper_coor[2]].frl
                oper_frl_id = self.time_blocks[oper_coor[0]][oper_coor[1]].list_of_operations[oper_coor[2]].frl_id
                oper_i_op = self.time_blocks[oper_coor[0]][oper_coor[1]].list_of_operations[oper_coor[2]].i_op
                print_line += "(" + str(oper_i_op) + ", " + colored(str(dist_id), "cyan") + "). "
                print_line += "It's success depends on results "
                print_line += str(self.delayed_distillation_check[i_ts][dist_id])
                print_line += ". FRL level: " + str(oper_frl)
                print_line += ", FRL ID: " + str(oper_frl_id) + "."
                print(print_line)
            if self.fusion_corrections[i_ts]:
                print(colored("SUBSYSTEM " + str([*range(self.n)]) + ":", "red"))
            for fc_qubit in self.fusion_corrections[i_ts]:
                fusion_correction = self.fusion_corrections[i_ts][fc_qubit]
                fusion_correction.print_fusion_correction()
        print("")
        print(colored("Used subsystems:", "green"))
        for sub_sys in self.subsystems.keys():
            subsystem = self.subsystems[sub_sys]
            print(subsystem.nodes, "has (possible) concurrent subsystems", subsystem.concurrent_subsystems)

        print("")
        print(colored("Total number of swap gates necessary = " + str(self.swap_count), "green"))
        print("")
