"""
2020-2024 Runsheng Ouyang, Sebastian de Bone (QuTech)
https://github.com/sebastiandebone/ghz_prot_II
"""
import sys
import numpy as np
import time
import pickle
import random
import math
import multiprocessing
import os
import itertools
from math import floor
from collections import namedtuple
from copy import deepcopy

import GHZ_prot_II.operations as op
import GHZ_prot_II.ancilla_rotations as ar
import GHZ_prot_II.da_protocols as dap
import GHZ_prot_II.plot_protocol as plotp
import analysis.evaluate_protocols as ep
import GHZ_prot_II.simulate_protocol.protocol_recipe as pr
import GHZ_prot_II.simulate_protocol.run_auto_generated_protocol as ragp
from GHZ_prot_II.auxiliaries_and_help_files.calculate_confidence_interval import calculate_confidence_interval, find_t


ParameterSearchGenerator = namedtuple('ParameterSearchGenerator', 'type dec n k n1 n2 k1 k2 i j a b r1 r2')
globals()["ParameterSearchGenerator"] = ParameterSearchGenerator


class Path:
    """
    A class used to capture the structure of the dynamic algorithm that store multiple states per value of n and k;
    for each combination [n][k] (where n is the number of parties of the n-qubit GHZ diagonal state and k is the number
    of isotropic Bell diagonal states used) a new object of this class is created.

    ...

    Attributes
    ----------
    p_or_f : 0, 1 or -1
        0 for purification, 1 for fusion, and -1 if (n, k) = (2, 1), which is an elementary isotropic Bell
        diagonal state
    z_or_xy : 0, 1 or -1
        0 for z purification, 1 for xy purification, -1 for fusion or (n, k) = (2, 1)
    n : positive integer
        number of parties over which the n-qubit GHZ diagonal state is created
    k : positive integer
        number of Bell diagonal states used to create  the n-qubit GHZ diagonal state
    n2 : positive integer, or -1
        number of the previous n (size of the ancillary state), with n2 = -1 if (n, k) = (2, 1)
    k2 : positive integer, or -1
        number of the previous k (number of states used to create the ancillary state), with k2 = -1 if (n, k) = (2, 1)
    dec : integer, -1 or bigger
        describes the type of stablizer used in purification, with dec = -1 if fusion or if (n, k) = (2, 1)
    i : integer, -1 or bigger
        describes which qubit of the first state is used in the fusion process, with i = -1 if purification or if
        (n, k) = (2, 1)
    j : integer, -1 or bigger
        describes which qubit of the second state is used in the fusion process, with j = -1 if purification or if
        (n, k) = (2, 1)
    state : one-dimensional numpy-vector of length 2**n
        the best n-qubit GHZ diagonal state found for these values of n and k
    t1 : non-negative integer
        describes the memory position of the first state that is used to create the current state (at data[n1][k1][t1])
    t2 : non-negative integer
        describes the memory position of the second state that is used to create the current state (at data[n2][k2][t2])
    r1 : non-negative integer
        describes the ancillary state rotation applied to the first state at data[n1][k1][t1]
    r2 : non-negative integer
        describes the ancillary state rotation applied to the second state at data[n2][k2][t2]

    Methods
    -------
    created(self, n, k)
        Prints status information about the dynamic program in case this is requested by the user
    """
    def __init__(self, p_or_f, z_or_xy, n, k, t, n2, k2, dec, i, j, state, t1, t2, r1, r2, id_elem=None):
        self.p_or_f = p_or_f
        self.z_or_xy = z_or_xy
        self.n = n
        self.k = k
        self.t = t
        self.n2 = n2
        self.k2 = k2
        self.dec = dec
        self.i = i
        self.j = j
        self.state = state
        self.t1 = t1
        self.t2 = t2
        self.r1 = r1
        self.r2 = r2
        self.id_elem = id_elem

    # noinspection PyMethodMayBeStatic
    def created(self, n, k, parameter=None):
        if parameter:
            print('(', n, ',', k, ')')
        else:
            print('(', n, ',', k, ',', parameter, ')')


def compare_F(A, B):
    """
    This function compares the fidelity between GHZ diagonal states A and B

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B

    Returns
    -------
    best : int
        +1 means A has higher fidelity, 0 means A and B have equal F, -1 means B has higher F
    """
    if A[0] > B[0]:
        best = 1
    elif A[0] == B[0]:
        best = 0
    elif A[0] < B[0]:
        best = -1
    return best


def compare_min(A, B):
    """
    This function compares which of the two GHZ diagonal states A and B has the smallest value for the smallest
    coefficient

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B

    Returns
    -------
    best : int
        +1 means A has the smallest value for the smallest coefficient, 0 means A and B have equal F, -1 means B has
        the smallest values for the smallest coefficient
    """
    if min(A) < min(B):
        best = 1
    elif min(A) == min(B):
        best = 0
    elif min(A) > min(B):
        best = -1
    return best


def compare_gap(A, B):
    """
    This function compares which of the two GHZ diagonal states A and B has the biggest gap between the biggest
    coefficient and the smallest coefficient

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B

    Returns
    -------
    best : int
        +1 means A has the biggest coefficient gap, 0 means A and B have equal F, -1 means B has higher gap
    """
    if np.size(A) != np.size(B):
        sys.exit("The input states in compare function should have the same dimension")
    if (max(A) - min(A)) > (max(B) - min(B)):
        best = 1
    elif (max(A) - min(A)) == (max(B) - min(B)):
        best = 0
    elif (max(A) - min(A)) < (max(B) - min(B)):
        best = -1
    return best


def compare_entropy(A, B):
    """
    This function compares which of the two GHZ diagonal states A and B has the lowest entropy

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B

    Returns
    -------
    best : int
        +1 means A has the lowest entropy, 0 means A and B have equal F, -1 means B has the lowest entropy
    """
    S_A = 0
    S_B = 0
    for i in range(np.size(A)):
        if A[i] != 0:
            S_A = S_A - A[i] * np.log2(A[i])
        if B[i] != 0:
            S_B = S_B - B[i] * np.log2(B[i])
    if S_A < S_B:
        best = 1
    elif S_A == S_B:
        best = 0
    elif S_A > S_B:
        best = -1
    return best


def compare_gap_coeff_sum(A, B):
    """
    This function compares which of the two GHZ diagonal states A and B has the biggest gap between the biggest
    coefficient and the smallest coefficient sum configuration.

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B

    Returns
    -------
    best : int
        +1 means A has the biggest gap, 0 means A and B have equal F, -1 means B has the biggest gap
    """
    if (max(A) - ar.smallest_coefficient_sum(A)) > (max(B) - ar.smallest_coefficient_sum(B)):
        best = 1
    elif (max(A) - ar.smallest_coefficient_sum(A)) == (max(B) - ar.smallest_coefficient_sum(B)):
        best = 0
    elif (max(A) - ar.smallest_coefficient_sum(A)) < (max(B) - ar.smallest_coefficient_sum(B)):
        best = -1
    return best


def compare_min_coeff_sum(A, B):
    """
    This function compares which of the two GHZ diagonal states A and B has the smallest value for the coefficient
    sum

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B

    Returns
    -------
    best : int
        +1 means A has the smallest coefficient sum, 0 means A and B have equal F, -1 means B has the smallest
        coefficient sum
    """
    if ar.smallest_coefficient_sum(A) < ar.smallest_coefficient_sum(B):
        best = 1
    elif ar.smallest_coefficient_sum(A) == ar.smallest_coefficient_sum(B):
        best = 0
    elif ar.smallest_coefficient_sum(A) > ar.smallest_coefficient_sum(B):
        best = -1
    return best


def compare(A, B, t):
    """
    This function compares which of the two GHZ diagonal states A and B has the best statistics based on a certain
    comparison determined by the input parameter t.

    Parameters
    ----------
    A : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as B
        GHZ diagonal state  A
    B : one-dimensional numpy object containing the coefficients of a GHZ diagonal state of the same size as A
        GHZ diagonal state  B
    t : integer in range [0,4]
        determines which is of the comparisons is used

    Returns
    -------
    best : int
        +1 means A has the best stats, 0 means A and B have equal F, -1 means B has the best stats
    """
    if isinstance(A, Path):
        A = A.state
    if isinstance(B, Path):
        B = B.state
    if A is not None:
        if isinstance(A, float) or isinstance(A, int):
            A = [A]
    if B is not None:
        if isinstance(B, float) or isinstance(B, int):
            B = [B]
    if isinstance(A, np.ndarray) and isinstance(B, np.ndarray):
        if np.size(A) != np.size(B):
            sys.exit("The input states in compare function should have the same dimension")
    if A is not None and B is None:
        return 1
    if A is None and B is not None:
        return -1
    if A is None and B is None:
        return None
    if t == 0:
        best = compare_F(A, B)
    elif t == 2:
        best = compare_min(A, B)
    elif t == 1:
        best = compare_min_coeff_sum(A, B)
    elif t == 3:
        best = compare_gap_coeff_sum(A, B)
    elif t == 4:
        best = compare_entropy(A, B)
    return best


def create_storage(n_max, k_max, ntype, input_data_file=None, calculate_prot=False):
    """
    This function creates a matrix of size (n_max+1, k_max+1, ntype), and fills it on places that are feasible elements
    in the dynamic program (e.g., n=4 and k=2 is infeasible, because it is impossible to make an 4-qubit GHZ state with
    only 2 Bell pairs).

    Parameters
    ----------
    n_max : positive integer
        Maximum number of qubits of the GHZ diagonal state that will be created
    k_max : positive integer
        Maximum number of isotropic Bell diagonal states used to create and distill the final GHZ diagonal state
    ntype : positive integer
        Parameter that determines how many protocols are stored per value of n and k (each of them based on the a
        different comparison; the first ntype comparisons of the compare function above)

    Returns
    -------
    data : (n_max+1, k_max+1, ntype) matrix with objects of class Path
        Each element of this matrix is used to store information about its element itself and how it's made
    """
    if input_data_file is not None:
        if os.path.exists(input_data_file):
            input_data = pickle.load(open(input_data_file, "rb"))
            n_max_input = len(input_data) - 1
            k_max_inputs = []
            for n in range(n_max_input + 1):
                k_max_inputs.append(len(input_data[n]) - 1)
        else:
            raise FileExistsError(f"Input data file {input_data_file} does not exist.")
    else:
        input_data = None
        n_max_input = n_max
        k_max_inputs = [k_max] * (n_max + 1)

    n_max_used = max(n_max, n_max_input)
    data = [None] * (n_max_used + 1)
    precalculated_data = [None] * (n_max_used + 1)
    for n in range(n_max_used + 1):
        k_max_used = max(k_max, k_max_inputs[n]) if len(k_max_inputs) > n else k_max
        data[n] = [None] * (k_max_used + 1)
        precalculated_data[n] = [None] * (k_max_used + 1)
        for k in range(k_max_used + 1):
            if input_data and (len(input_data) > n and len(input_data[n]) > k and input_data[n][k]):
                data[n][k] = input_data[n][k]
                precalculated_data[n][k] = True if data[n][k][0] is not None else False
            elif k >= (n - 1):
                data[n][k] = [None] * ntype
                precalculated_data[n][k] = False
                for t in range(ntype):
                    data[n][k][t] = None
            else:
                data[n][k] = None

    return data, precalculated_data


def yield_generators_parameter_search(data, n, k, inc_rot):
    for dec in range(1, 2 ** n):
        if dec < 2 ** (n - 1):  # Z_purification
            n2 = np.size(op.transform_list_ind(dec, n, 'Z'))
        else:  # XY_purification
            n2 = n
        for k2 in range(n2 - 1, k - n + 1 + 1):
            for a in range(len(data[n][k-k2])):
                for b in range(len(data[n2][k2])):
                    for r in range((2 ** (n2 - 1) + floor(2 / n2) * 2) ** inc_rot):
                        yield ParameterSearchGenerator(type="Distill", dec=dec, n=n, k=k, n1=n, n2=n2, k1=k-k2, k2=k2, i=None, j=None,
                                        a=a, b=b, r1=None, r2=r)

    for n2 in range(2, n - 1 + 1):
        for k2 in range(n2 - 1, k - n + n2 + 1):
            n1 = n + 1 - n2
            for i in range(n1):
                for j in range(n2):
                    for a in range(len(data[n1][k-k2])):
                        for b in range(len(data[n2][k2])):
                            for r1 in range((2 ** (n1 - 1) + floor(2 / n1) * 2) ** inc_rot):
                                for r2 in range((2 ** (n2 - 1) + floor(2 / n2) * 2) ** inc_rot):
                                    yield ParameterSearchGenerator(type="Fuse", dec=None, n=n, k=k, n1=n1, n2=n2, k1=k-k2, k2=k2, i=i,
                                                    j=j, a=a, b=b, r1=r1, r2=r2)


def update_m_prot(data, n, k, input_state, show_or_not, nstate, inc_rot, da_type='sp', marker_cnt=0,
                  calculate_prot=False, nmb_id_elems=1, seeds=None, number_of_threads=1, start_time=None,
                  failed_protocol_prefix=None, init_seed_search=True, skip_similar_prots='identical',
                  reshuffle_protocols=True, p_g=None, alpha=None, n_DD=None, metric=None):
    """
    Function that updates data[n][k][t] in the deterministic version of the algorithm.

    Parameters
    ----------
    data : matrix with objects of class Path
    n : positive integer smaller than or equal to n_max
        Number of parties for which we want to update the element in data
    k : positive integer smaller than or equal to k_max
        Number of Bell diagonal states for which we want to update the element in data
    n_max : positive integer
        Maximum number of qubits of the GHZ diagonal state stored in data
    k_max : positive integer
        Maximum number of isotropic Bell diagonal states stored in data
    input_state : string
        A string describing the input Bell pair fidelity involved in the search. If this is set to a float between 0
        and 1, the Bell pairs are set to Werner states with this fidelity. Fidelity of the isotropic Bell diagonal
        states used.
    show_or_not : Boolean
        Determines if status update about a matrix entry being update is printed in the console
    nstate : positive integer
        Number of protocols stored per value of n and k in data
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    da_type : string
        String describing what version of the deterministic random the algorithm should carry out: 'sp' is a single
        protocol per value of n and k, 'mpc' is multiple protocols per value of n and k based on different conditions,
        'mpF' is multiple protocols per n and k based on the highest fidelity)
    marker_cnt : integer
        Integer that, upon request by the user, keeps track of a parameter during the search
    calculate_prot : Boolean
        Determines if the program should convert the protocols to a list of instructions and use the density matrix
        simulator to calculate the quality/fitness of the protocols (in case of True), or should just operate with
        perfect operations and measurements to calculate the quality/fitness of the protocols (in case of False)
    nmb_id_elems : Integer
        Determines how many (random) permutations are considered for the order in which the Bell pairs of the protocols
        are created
    seeds : List of integers
        Determines, in case nmb_id_elems > 1, for what random seed numbers each protocol should be calculated. The
        length of this list describes how many iterations of the circuit simulation are averaged to determine the
        quality (or fitness) of a considered protocol
    init_seed_search : True
        If true, the program first runs a protocol for 10 seeds and estimates the upper bound (with 90% confidence) of
        the average stabilizer fidelity out of these first 10 iterations. If this upper bound is below the average
        stabilizer fidelity of the worst protocol in the buffer, the iterations with the remaining seeds are skipped.
        If the number of seeds used is set to 20 or lower, the program uses the first ceiling(#seeds/2) seeds to get a
        first estimate of the average stabilizer fidelity.
    skip_similar_prots : 'identical'
        A string describing if a protocol should be considered (i.e., considered in the search) if it is either
        'identical' or 'similar' (in its binary tree structure) to a protocol that already has been considered in the
        search. Passing a different string from these two means no protocols are skipped.
    reshuffle_protocols : True
        A boolean describing whether the protocol should be 'sliced' before given to the available CPU cores (in order
        to have a more 'balanced' distribution of them over the available CPUs.
    p_g : None
        Float value with gate and measurement error probability used in the search.
    alpha : None
        Float value with the bright-state population in case the single-click protocol is used.
    n_DD : None
        Integer describing how many entanglement generation attempts fit in half of a dynamical decoupling sequence.
    metric : None
        String describing what metric we want to use to determine if one protocol is better than another protocol: None
        will use the default setting, "ghz_fid" will use the fidelity of the GHZ state, "stab_fid" will use the
        stabilizer fidelity, and "weighted_sum" will use the weighted sum of the superoperator. Further,
        "logical_success" will compare protocols by looking at the logical success rate over 5000 of their
        superoperator for a distance 4 toric code. This will, however, only work for n=4: for all other values of n,
        this setting will default back to using "stab_fid" as compare metric.

    Returns
    -------
    data : (n_max+1, k_max+1, nstate) matrix with objects of class Path
        Each element (n, k, t) of this matrix is used to store information about (n, k, t) itself and how it's made
    """

    if show_or_not == 1:
        if start_time is None:
            print(f"\nStarted calculating for (n, k) = ({n}, {k}).")
        else:
            print(f"\nStarted calculating for (n, k) = ({n}, {k}), at time {time.time() - start_time}.")

    if metric == "logical_success" and n != 4:
        metric = "stab_fid"
    print(f"Metric used is {metric} with init_seed_search set at {init_seed_search}")

    if da_type == 'sp':
        nstate = 1
        cp = 1
    elif da_type == 'mpc':
        cp = 1
    elif da_type in ['mpF', 'mpuF', 'mpuP']:
        cp = 0  # This parameter makes sure for 'mpF', states are only compared based on fidelity
    else:
        raise ValueError("Dynamic algorithm type unknown.")

    swap_count = None
    prot_save = False

    F = input_state if calculate_prot is False else None
    state_set = input_state if calculate_prot else None
    nmb_id_elems = 1 if calculate_prot is False else nmb_id_elems

    if nmb_id_elems > 1 and seeds is None:
        seeds = [1]

    if init_seed_search:
        if isinstance(seeds, list):
            df = 9 if len(seeds) > 20 else (int(len(seeds) / 2) if len(seeds) > 2 else len(seeds))
        else:
            df = 1
        tci = find_t(p=0.9, df=df)[1]
    else:
        tci = None

    check_multiple_id_elem = True if nmb_id_elems > 1 else False

    avg_iteration_time = [0]

    if n == 2 and k == 1:
        state = F if calculate_prot else op.set_isotropic_state(F, 2)
        data[2][1][0] = Path(p_or_f=-1, z_or_xy=-1, n=2, k=1, t=0, n2=-1, k2=-1, dec=-1, i=-1, j=-1, state=state,
                             t1=-1, t2=-1, r1=-1, r2=-1, id_elem=None)
        for t in range(1, len(data[2][1])):
            # data[2][1][t] = Path(p_or_f=-1, z_or_xy=-1, n=2, k=1, t=t, n2=-1, k2=-1, dec=-1, i=-1, j=-1, state=state,
            #                      t1=-1, t2=-1, r1=-1, r2=-1, id_elem=None)
            del data[2][1][-1]
        marker_cnt += 1
        # marker_cnt += nstate
    else:
        # Yield parameter search generators for distillation and fusion:
        parameter_search_generators = list(yield_generators_parameter_search(data, n, k, inc_rot))
        print(f"Number of protocols considered: {len(parameter_search_generators)}.")

        if number_of_threads == "auto" or number_of_threads > 1:
            threads = multiprocessing.cpu_count() if number_of_threads == "auto" else number_of_threads
            pool = multiprocessing.Pool(threads)

            nmb_par_sets = len(parameter_search_generators)
            marker_cnt += nmb_par_sets

            iterations, remaining_iterations = divmod(nmb_par_sets, threads)
            # sets_per_thread = math.ceil(nmb_par_sets / threads)

            # Explore the full parameter space for these values of n and k:
            results = []
            for i in range(threads):
                if reshuffle_protocols is False:
                    iterations_thread = iterations + 1 if i < remaining_iterations else iterations
                    iter_offset = iterations * i + min(i, remaining_iterations)
                    parameters = deepcopy(parameter_search_generators[iter_offset: iter_offset + iterations_thread])
                    # parameters = deepcopy(
                    #     parameter_search_generators[i * sets_per_thread: min((i + 1) * sets_per_thread, nmb_par_sets)])
                else:
                    parameters = deepcopy(itertools.islice(parameter_search_generators, i, None, threads))
                results.append(pool.apply_async(browse_search_parameters,
                                                args=[data, n, k, nstate, state_set, da_type, parameters,
                                                      calculate_prot, check_multiple_id_elem, nmb_id_elems, seeds,
                                                      marker_cnt, cp, i, tci, failed_protocol_prefix,
                                                      skip_similar_prots, p_g, alpha, n_DD, metric]))

            result = results[0].get()
            buffer = result[0]
            if da_type == 'mpuP':
                buffer_prots = result[1]
                buffer_prots_rem = []
                prots_counter_skip = [result[2][0]]
                prots_counter_part = [result[2][1]]
                prots_counter_full = [result[2][2]]
                avg_iteration_time = [result[3]]

            buffer_rem = []
            for result in results[1:]:
                result_full = result.get()
                buffer_rem += result_full[0]
                if da_type == 'mpuP':
                    buffer_prots_rem += result_full[1]
                    prots_counter_skip.append(result_full[2][0])
                    prots_counter_part.append(result_full[2][1])
                    prots_counter_full.append(result_full[2][2])
                    avg_iteration_time.append(result_full[3])
            pool.close()
            protocols_were_skipped = False
            for t1 in range(len(buffer_rem)):
                size_buffer = len(buffer)
                if da_type == 'mpuP' and ep.check_if_protocols_are_identical(buffer_prots_rem[t1], buffer_prots, skip_similar_prots):
                    protocols_were_skipped = True
                    continue
                if size_buffer < nstate:
                    size_buffer += 1
                    buffer.append(None)
                    if da_type == 'mpuP':
                        buffer_prots.append(None)
                for t2 in range(size_buffer):
                    if da_type == 'mpuF' and compare(buffer_rem[t1], buffer[t2], 0) == 0:
                        if buffer[-1] is None:
                            del buffer[-1]
                        break  # In this mode we don't save states with fidelities that we already have.
                    if compare(buffer_rem[t1], buffer[t2], t2 * cp) == 1 or \
                            (compare(buffer_rem[t1], buffer[t2], t2 * cp) != -1
                             and buffer_rem[t1].t1 + buffer[t2].t2 == 0):
                        if da_type in ['mpF', 'mpuF', 'mpuP']:
                            for v in range(size_buffer - t2 - 1):  # Shift existing states to the
                                u = size_buffer - 1 - v  # right from the last to the number t+1
                                buffer[u] = buffer[u - 1]
                                if da_type == 'mpuP':
                                    buffer_prots[u] = buffer_prots[u - 1]
                        buffer[t2] = buffer_rem[t1]
                        if da_type == 'mpuP':
                            buffer_prots[t2] = buffer_prots_rem[t1]
                        if da_type in ['mpF', 'mpuF', 'mpuP']:  # Breaks the "for t in
                            break  # range(nstate)" loop at the point where the first
                            # better state is found

            # # Alternative way of sorting the buffer with results from each of the threads:
            # buffer.sort(key=lambda x: x.state[0], reverse=True)
            if da_type == 'mpuP':
                if show_or_not == 1:
                    print(f"Protocol skip: {prots_counter_skip}.")
                    print(f"Protocol part: {prots_counter_part}.")
                    print(f"Protocol full: {prots_counter_full}.")
                    print(f"Average time per protocol iteration: {avg_iteration_time}.")
                    if protocols_were_skipped:
                        print(f"Some protocols were skipped in constructing the final combined buffer over all threads.")

        else:
            if reshuffle_protocols:
                # Reshuffle 'parameter_search_generators' to match the order encountered by the version with
                # multiprocessing above:
                number_of_threads_used = 8
                parameters = []
                for i_th in range(number_of_threads_used):
                    parameters += itertools.islice(parameter_search_generators, i_th, None, number_of_threads_used)
            else:
                parameters = parameter_search_generators

            result = browse_search_parameters(data, n, k, nstate, state_set, da_type, parameters, calculate_prot,
                                              check_multiple_id_elem, nmb_id_elems, seeds, marker_cnt,
                                              cp, tci=tci, failed_protocol_prefix=failed_protocol_prefix,
                                              skip_similar_prots=skip_similar_prots, p_g=p_g, alpha=alpha, n_DD=n_DD,
                                              metric=metric)
            buffer = result[0]
            if da_type == 'mpuP':
                buffer_prots = result[1]
                if show_or_not == 1:
                    print(f"Protocol skip: {[result[2][0]]}.")
                    print(f"Protocol part: {[result[2][1]]}.")
                    print(f"Protocol full: {[result[2][2]]}.")
                    print(f"Average time per protocol iteration: {[result[3]]}.")
                avg_iteration_time = [result[3]]
            marker_cnt += len(parameter_search_generators)

        for t in range(nstate):
            if t >= len(buffer):
                del data[n][k][-1]
            else:
                data[n][k][t] = buffer[t]

    if da_type == 'mpuP':
        avg_iteration_time = [time for time in avg_iteration_time if time is not None]
        return marker_cnt, sum(avg_iteration_time)/len(avg_iteration_time)
    else:
        return marker_cnt


def browse_search_parameters(data, n, k, nstate, state_set, da_type, parameter_search_generators, calculate_prot,
                             check_multiple_id_elem, nmb_id_elems, seeds, marker_cnt, cp, cpu_number=None, tci=None,
                             failed_protocol_prefix=None, skip_similar_prots='identical', p_g=None, alpha=None,
                             n_DD=None, metric=None):
    # Create a buffer for the states that will be found:
    # buffer = [None] * nstate    # np.empty(nstate, dtype=object)
    # for t in range(nstate):
    #     buffer[t] = data[n][k][t]
    buffer = []

    if da_type == 'mpuP':
        buffer_prots = []

    if tci is not None and isinstance(seeds, list) and len(seeds) > 20:
        seeds_initial = seeds[:10]     # [*range(max(seeds) + 1, max(seeds) + 11)]
        seeds_remaining = seeds[10:]
    elif tci is not None and isinstance(seeds, list) and len(seeds) > 2:
        n_init = int(len(seeds)/2) + 1
        seeds_initial = seeds[:n_init]  # seeds[[*range(max(seeds) + 1, max(seeds) + int(len(seeds)/2) + 2)]
        seeds_remaining = seeds[n_init:]
    else:
        tci = None
        seeds_initial = seeds
        seeds_remaining = []

    count_prots_considered_full = 0
    count_prots_considered_part = 0
    count_prots_skipped = 0

    start_time = time.time()

    print_information = False

    i = 0
    for par in parameter_search_generators:
        if print_information:
            i += 1
            print(f"\nInvestigate protocol {i}.")
        if not (par.n1 == 2 and k - par.k2 == 1) and data[par.n1][k - par.k2][par.a].p_or_f == -1:
            save_state = [False]  # We skip this step if one of the resource states is the starting
            to_compare = [None]
        elif not (par.n2 == 2 and par.k2 == 1) and data[par.n2][par.k2][par.b].p_or_f == -1:  # state but (n,k) != (2,1)
            save_state = [False]
            to_compare = [None]
        else:
            if calculate_prot or da_type == 'mpuP':
                if par.type == "Distill":
                    protocol = dap.Node(
                        Path(p_or_f=0, z_or_xy=par.dec >> (n - 1), n=n, k=k, t=0, n2=par.n2, k2=par.k2, dec=par.dec,
                             i=-1, j=-1, state=None, t1=par.a, t2=par.b, r1=-1, r2=par.r2),
                        id=0, root=None, lr=None)
                    protocol.left = dap.identify_protocol(data, n, k - par.k2, par.a, 0, protocol, 0)
                    protocol.right = dap.identify_protocol(data, par.n2, par.k2, par.b, 0, protocol, 1)
                elif par.type == "Fuse":
                    protocol = dap.Node(
                        Path(p_or_f=1, z_or_xy=-1, n=n, k=k, t=0, n2=par.n2, k2=par.k2, dec=-1, i=par.i, j=par.j,
                             state=None, t1=par.a, t2=par.b, r1=par.r1, r2=par.r2), id=0, root=None, lr=None)
                    protocol.left = dap.identify_protocol(data, n + 1 - par.n2, k - par.k2, par.a, 0,
                                                          protocol, 0)
                    protocol.right = dap.identify_protocol(data, par.n2, par.k2, par.b, 0, protocol, 1)

            protocol_should_be_considered_for_buffer = True
            if da_type == 'mpuP':
                protocol = dap.protocol_add_meta_data(protocol)
                if ep.check_if_protocols_are_identical(protocol, buffer_prots, skip_similar_prots):
                    to_compare = []
                    protocol_should_be_considered_for_buffer = False
                    count_prots_skipped += 1
                    if print_information:
                        print("Protocol is skipped because the same protocol is already in the buffer.")

            if protocol_should_be_considered_for_buffer:
                if calculate_prot:
                    if check_multiple_id_elem:
                        id_elems = ep.other_versions_of_binary_tree_protocol(protocol,
                                                                             number_of_orderings_checked=nmb_id_elems)
                        to_compare = [None] * len(id_elems)
                        for i_ide, id_elem in enumerate(id_elems):
                            protocol_recipe = pr.ProtocolRecipe(protocol, id_elem)
                            max_qb_per_node = max([len(node) for node in protocol_recipe.qubit_memory])
                            fidelities = ragp.simulate_protocol_recipe(protocol_recipe, seeds=seeds_initial,
                                                                       set_name=state_set, cpu_number=cpu_number,
                                                                       failed_protocol_prefix=failed_protocol_prefix,
                                                                       p_g=p_g, alpha=alpha, n_DD=n_DD, metric=metric)
                            if tci is not None:
                                max_fidelity = np.average(fidelities) + calculate_confidence_interval(fidelities, t=tci)
                                if len(buffer) < nstate or compare(max_fidelity, buffer[-1], 0) != -1 and seeds_remaining:
                                    count_prots_considered_full += 1
                                    # We are going to do more iterations because this protocol is worth investigating
                                    fidelities += ragp.simulate_protocol_recipe(protocol_recipe, seeds=seeds_remaining,
                                                                                set_name=state_set, cpu_number=cpu_number,
                                                                                failed_protocol_prefix=failed_protocol_prefix,
                                                                                p_g=p_g, alpha=alpha, n_DD=n_DD,
                                                                                metric=metric)
                                else:
                                    count_prots_considered_part += 1
                            else:
                                count_prots_considered_full += 1
                            to_compare[i_ide] = np.average(fidelities)
                        save_state = [True] * len(to_compare)
                    else:
                        protocol_recipe = pr.ProtocolRecipe(protocol)
                        # print(protocol_recipe.id_elem)
                        max_qb_per_node = max([len(node) for node in protocol_recipe.qubit_memory])
                        # to_compare = [np.average(ragp.simulate_protocol_recipe(protocol_recipe, seeds=seeds, set_name=state_set, cpu_number=cpu_number))]
                        fidelities = ragp.simulate_protocol_recipe(protocol_recipe, seeds=seeds_initial,
                                                                   set_name=state_set, cpu_number=cpu_number,
                                                                   failed_protocol_prefix=failed_protocol_prefix,
                                                                   p_g=p_g, alpha=alpha, n_DD=n_DD, metric=metric)
                        if tci is not None:
                            max_avg_fidelity = np.average(fidelities) + calculate_confidence_interval(fidelities, t=tci)
                            if print_information:
                                print(f"Determined initial max_fidelity = {max_avg_fidelity}. Buffer = {[item.state for item in buffer]}.")
                            if len(buffer) < nstate or compare(max_avg_fidelity, buffer[-1], 0) != -1 and seeds_remaining:
                                count_prots_considered_full += 1
                                # We are going to do more iterations because this protocol is worth investigating
                                fidelities += ragp.simulate_protocol_recipe(protocol_recipe, seeds=seeds_remaining,
                                                                            set_name=state_set, cpu_number=cpu_number,
                                                                            failed_protocol_prefix=failed_protocol_prefix,
                                                                            p_g=p_g, alpha=alpha, n_DD=n_DD,
                                                                            metric=metric)
                                if print_information:
                                    new_max_avg_fidelity = np.average(fidelities) + calculate_confidence_interval(fidelities, t=tci)
                                    print(f"More calculations are performed: new_max_avg_fidelity = {new_max_avg_fidelity}. Buffer = {[item.state for item in buffer]}.")
                            else:
                                count_prots_considered_part += 1
                        else:
                            count_prots_considered_full += 1
                        to_compare = [np.average(fidelities)]
                        if print_information:
                            print(f"Average fidelity = {to_compare}.")
                        save_state = [True]

                else:
                    if par.type == "Distill":
                        to_compare = [op.purification(data[n][k - par.k2][par.a].state,
                                                      ar.ancilla_rotation(data[par.n2][par.k2][par.b].state, par.r2),
                                                      par.dec)]
                    if par.type == "Fuse":
                        to_compare = [op.fuse_GHZ_local(
                            ar.ancilla_rotation(data[par.n1][k - par.k2][par.a].state, par.r1),
                            ar.ancilla_rotation(data[par.n2][par.k2][par.b].state, par.r2), par.i, par.j)]
                    count_prots_considered_full += 1
                    save_state = [True]
                    marker_cnt += 1

            if da_type == 'mpuF':
                for itc, to_comp in enumerate(to_compare):
                    for t in range(len(buffer)):
                        # to_comp = to_comp[0] if isinstance(to_comp, list) else to_comp
                        if compare(to_comp, buffer[t], 0) == 0:
                            save_state[itc] = False  # In this mode we don't save states with
                            break  # fidelities that we already have.

            for itc, to_comp in enumerate(to_compare):
                # if to_comp:     # The number of id_elems requested can be lower than the maximum nmb
                #     # of id_elems: then some elements of to_compare are None from the definition.
                if len(buffer) == nstate and compare(to_comp, buffer[-1], 0) == -1:
                    if print_information:
                        print("Protocol is not saved because buffer is full and average fidelity is too low.")
                    save_state[itc] = False  # Don't save the new state if it is not better than
                    # the last one in the register

        for itc, to_comp in enumerate(to_compare):
            if save_state[itc]:
                size_buffer = len(buffer)
                if size_buffer < nstate:
                    size_buffer += 1
                    buffer.append(None)
                    if da_type == 'mpuP':
                        buffer_prots.append(None)
                for t in range(size_buffer):
                    if compare(to_comp, buffer[t], t * cp) == 1 or \
                            (compare(to_comp, buffer[t], t * cp) == 0 and par.a + par.b == 0):
                        if da_type in ['mpF', 'mpuF', 'mpuP']:  # Shift existing states to the right,
                            for v in range(size_buffer - t - 1):  # from the last to the number t+1
                                u = size_buffer - 1 - v
                                buffer[u] = buffer[u - 1]
                                if da_type == 'mpuP':
                                    buffer_prots[u] = buffer_prots[u - 1]
                        save_id_elem = id_elems[itc] if check_multiple_id_elem else None
                        if par.type == "Distill":
                            buffer[t] = Path(p_or_f=0, z_or_xy=par.dec >> (n - 1), n=n, k=k, t=t, n2=par.n2,
                                             k2=par.k2, dec=par.dec, i=-1, j=-1, state=to_comp, t1=par.a, t2=par.b,
                                             r1=-1, r2=par.r2, id_elem=save_id_elem)
                        elif par.type == "Fuse":
                            buffer[t] = Path(p_or_f=1, z_or_xy=-1, n=n, k=k, t=t, n2=par.n2,
                                             k2=par.k2, dec=-1, i=par.i, j=par.j, state=to_comp, t1=par.a,
                                             t2=par.b, r1=par.r1, r2=par.r2, id_elem=save_id_elem)
                        if da_type == 'mpuP':
                            buffer_prots[t] = protocol
                        if da_type in ['mpF', 'mpuF', 'mpuP']:  # Breaks the "for t in
                            break  # range(nstate)" loop at the point where the first
                            # better state is found
            if print_information:
                print(f"Protocol is added to buffer: {[item.state for item in buffer]}")

    end_time = time.time()
    if not isinstance(seeds, list):
        seeds_initial = [None]
        seeds_remaining = [None]
    number_iterations_full = count_prots_considered_part * len(seeds_initial) + count_prots_considered_full * (len(seeds_initial) + len(seeds_remaining))
    if number_iterations_full > 0:
        average_time_per_iteration = (end_time - start_time) / number_iterations_full
    else:
        average_time_per_iteration = None

    if da_type == 'mpuP':
        return buffer, buffer_prots, (count_prots_skipped, count_prots_considered_part, count_prots_considered_full), average_time_per_iteration
    else:
        return buffer,


def update_random(data, n, k, F, show_or_not, nstate, inc_rot, T):
    """
    Function that updates data[n][k][t] in the situation where we want a randomized version of the dynamic algorithm.

    Parameters
    ----------
    data : matrix with objects of class Path
    n : positive integer smaller than or equal to n_max
        Number of parties for which we want to update the element in data
    k : positive integer smaller than or equal to k_max
        Number of Bell diagonal states for which we want to update the element in data
    n_max : positive integer
        Maximum number of qubits of the GHZ diagonal state stored in data
    k_max : positive integer
        Maximum number of isotropic Bell diagonal states stored in data
    F : float between 0 and 1
        Fidelity of the isotropic Bell diagonal states used
    show_or_not : Boolean
        Determines if status update about a matrix entry being update is printed in the console
    nstate : positive integer
        Number of protocols stored per value of n and k in data
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    T : float
        temperature in exp**(delta_F/T). A higher T will allow this function accept more states with low fidelity

    Returns
    -------
    data : (n_max+1, k_max+1, nstate) matrix with objects of class Path
        Each element (n, k, t) of this matrix is used to store information about (n, k, t) itself and how it's made
    """

    if n == 2 and k == 1:   # At the (n, k) = (2, 1) spots in the memory we add isotropic Bell pairs
        state = op.set_isotropic_state(F, 2)
        for t in range(nstate):
            data[2][1][t] = Path(p_or_f=-1, z_or_xy=-1, n=2, k=1, t=t, n2=-1, k2=-1, dec=-1, i=-1, j=-1,
                                 state=state, t1=-1, t2=-1, r1=-1, r2=-1)
    else:
        # Create a buffer for the states that will be found
        buffer = np.empty(nstate, dtype=object)
        for t in range(nstate):
            buffer[t] = data[n][k][t]

        # Create first state to compare with:
        prev_state = op.set_isotropic_state(1 / 2 ** n, n)

        for t in range(nstate):
            if n == 2:  # In this situation we can only do purification
                p_or_f = 0
            elif k == n - 1:    # In this situation we can only do fusion
                p_or_f = 1
            else:
                p_or_f = random.randint(0, 1)

            if p_or_f == 0:     # Purification
                n2 = 0
                dec = 0
                while (n2 == 0) | ((n2 - 1) > (k - n + 1)):  # to satisfy randint input range
                    dec = random.randint(1, 2 ** n - 1)
                    if dec < 2 ** (n - 1):  # Z_purification
                        n2 = np.size(op.transform_list_ind(dec, n, 'Z'))
                    else:  # XY_purification
                        n2 = n
                z_or_xy = dec >> (n - 1)
                k2 = random.randint(n2 - 1, k - n + 1)
                i = -1
                j = -1
                a = random.randint(0, nstate - 1)
                b = random.randint(0, nstate - 1)
                r1 = -1
                r2 = random.randint(0, (2 ** (n2 - 1) + floor(2 / n2) * 2) ** inc_rot - 1)
                to_compare = op.purification(data[n][k - k2][a].state,
                                             ar.ancilla_rotation(data[n2][k2][b].state, r2),
                                             dec)
            else:   # Fusion
                z_or_xy = -1
                dec = -1
                n2 = random.randint(2, n - 1)
                n1 = n + 1 - n2
                if k == n - 1:
                    k2 = n2 - 1
                else:
                    k2 = random.randint(n2 - 1, k - n + n2)
                i = random.randint(0, n1 - 1)
                j = random.randint(0, n2 - 1)
                a = random.randint(0, nstate - 1)
                b = random.randint(0, nstate - 1)
                r1 = random.randint(0, (2 ** (n1 - 1) + floor(2 / n1) * 2) ** inc_rot - 1)
                r2 = random.randint(0, (2 ** (n2 - 1) + floor(2 / n2) * 2) ** inc_rot - 1)
                to_compare = op.fuse_GHZ_local(ar.ancilla_rotation(data[n1][k - k2][a].state, r1),
                                               ar.ancilla_rotation(data[n2][k2][b].state, r2),
                                               i, j)

            if compare(to_compare, prev_state, 0) == -1:    # Triggers if the previous state was better
                rate = math.exp((to_compare[0] - prev_state[0]) / T)  # Accept rate
                psi = random.uniform(0, 1)  # Throw a dice to see if the state is accepted
                if psi < rate:
                    accept_state = 1
                else:
                    accept_state = 0
            else:   # Triggers if the new state is better or the same as the previous state
                accept_state = 1

            if accept_state == 1:  # Accept the new state in the memory
                buffer[t].p_or_f = p_or_f
                buffer[t].z_or_xy = z_or_xy
                buffer[t].n = n
                buffer[t].k = k
                buffer[t].t = t
                buffer[t].n2 = n2
                buffer[t].k2 = k2
                buffer[t].dec = dec
                buffer[t].i = i
                buffer[t].j = j
                buffer[t].state = to_compare
                buffer[t].t1 = a
                buffer[t].t2 = b
                buffer[t].r1 = r1
                buffer[t].r2 = r2
                prev_state = to_compare
            else:   # Use the previous state in the memory
                buffer[t].p_or_f = buffer[t - 1].p_or_f
                buffer[t].z_or_xy = buffer[t - 1].z_or_xy
                buffer[t].n = buffer[t - 1].n
                buffer[t].k = buffer[t - 1].k
                buffer[t].t = buffer[t - 1].t
                buffer[t].n2 = buffer[t - 1].n2
                buffer[t].k2 = buffer[t - 1].k2
                buffer[t].dec = buffer[t - 1].dec
                buffer[t].i = buffer[t - 1].i
                buffer[t].j = buffer[t - 1].j
                buffer[t].state = buffer[t - 1].state
                buffer[t].t1 = buffer[t - 1].t1
                buffer[t].t2 = buffer[t - 1].t2
                buffer[t].r1 = buffer[t - 1].r1
                buffer[t].r2 = buffer[t - 1].r2
                prev_state = prev_state

        # Sort buffer from highest fidelity to lowest
        to_sort = np.empty(nstate, dtype=object)
        for t in range(nstate):
            to_sort[t] = buffer[t].state[0]
        sorted_buf = np.argsort(to_sort)
        # np.argsort() return to a list with element from 0 to nstate-1, from smallest F to the biggest
        for t in range(nstate):
            u = nstate - 1 - t  # From biggest F to smallest F
            data[n][k][t] = buffer[sorted_buf[u]]
            data[n][k][t].t = t

    if show_or_not == 1:
        Path.created(data[n][k], n, k)
    return data


def dynamic_algorithm(n_max, k_max, input_state, input_data_file=None, n_start=2, k_start=1, da_type="mpc", nstate=2,
                      rot_type="none", show_or_not=0, seeds=None, T=0.0009, calculate_prot=False, nmb_id_elems=1,
                      number_of_threads=1, failed_protocol_prefix=None, init_seed_search=True,
                      skip_similar_prots='identical', reshuffle_protocols=True, p_g=None, alpha=None, n_DD=None,
                      metric=None):
    """
    Function that carries out the dynamic algorithm and updates the object data in this process (using the update
    function).

    Parameters
    ----------
    n_max : positive integer
        Maximum number of qubits of the GHZ diagonal state stored in data
    k_max : positive integer
        Maximum number of isotropic Bell diagonal states stored in data
    input_state : string
        A string describing the input Bell pair fidelity involved in the search. If this is set to a float between 0
        and 1, the Bell pairs are set to Werner states with this fidelity. Fidelity of the isotropic Bell diagonal
        states used.
    input_data_file : string
        A string containing the data structure of an earlier iteration of this algorithm (that can be used to further
        extend on).
    n_start : positive integer
        Integer describing at which n value the search should start.
    k_start : positive integer
        Integer describing at which k value the search should start.
    da_type : string
        String describing what version of the deterministic random the algorithm should carry out: 'sp' is a single
        protocol per value of n and k, 'mpc' is multiple protocols per value of n and k based on different conditions,
        'mpF' is multiple protocols per n and k based on the highest fidelity, and 'random' is the randomized version
        of the dynamic program
    nstate : positive integer
        Number of protocols stored per value of n and k in data
    rot_type : string
        String describing if rotation of ancillary states should be included in the search process. "k_max" means
        rotations are only carried out whenever k == k_max, "n_max" means rotations are only carried out when k == k_max
        and n == n_max, "always" means rotations are carried out at every value of n and k, and "none" means no
        rotations are include.
    show_or_not : Boolean
        Determines if status update about a matrix entry being update is printed in the console
    seeds : Integer or List of integers
        Determines, for the randomized version of the dynamic program, what seed should be used or, in case
        check_multiple_id_elem=True, for what random seed numbers each protocol should be calculated. The length of this
        list describes how many iterations of the circuit simulation are averaged to determine the quality (or fitness)
        of a considered protocol
    T : float
        temperature in exp**(delta_F/T). A higher T will allow this function accept more states with low fidelity
        calculate_prot : Boolean
        Determines if the program should convert the protocols to a list of instructions and use the density matrix
        simulator to calculate the quality/fitness of the protocols (in case of True), or should just operate with
        perfect operations and measurements to calculate the quality/fitness of the protocols (in case of False)
    nmb_id_elems : Integer
        Determines how many (random) permutations are considered for the order in which the Bell pairs of the protocols
        are created
    init_seed_search : True
        If true, the program first runs a protocol for 10 seeds and estimates the upper bound (with 90% confidence) of
        the average stabilizer fidelity out of these first 10 iterations. If this upper bound is below the average
        stabilizer fidelity of the worst protocol in the buffer, the iterations with the remaining seeds are skipped.
        If the number of seeds used is set to 20 or lower, the program uses the first ceiling(#seeds/2) seeds to get a
        first estimate of the average stabilizer fidelity.
    skip_similar_prots : 'identical'
        A string describing if a protocol should be considered (i.e., considered in the search) if it is either
        'identical' or 'similar' (in its binary tree structure) to a protocol that already has been considered in the
        search. Passing a different string from these two means no protocols are skipped.
    reshuffle_protocols : True
        A boolean describing whether the protocol should be 'sliced' before given to the available CPU cores (in order
        to have a more 'balanced' distribution of them over the available CPUs.
    p_g : None
        Float value with gate and measurement error probability used in the search.
    alpha : None
        Float value with bright-state population used in case the single-click protocol is used.
    n_DD : None
        Integer describing how many entanglement generation attempts fit in half of a dynamical decoupling sequence.
    metric : None
        String describing what metric we want to use to determine if one protocol is better than another protocol: None
        will use the default setting, "ghz_fid" will use the fidelity of the GHZ state, "stab_fid" will use the
        stabilizer fidelity, and "weighted_sum" will use the weighted sum of the superoperator. Using a metric
        "weighted_sum_max" will only use "weighted_sum" on (n_max,k_max), and "stab_fid" for protocols before that;
        this setting will also overwrite a pre-calculated (n_max,k_max) buffer that was included with the
        input_data_file. Further, the metric "logical_success" will compare protocols by looking at the logical success
        rate over 5000 of their superoperator for a distance 4 toric code. This will, however, only work for n=4: for
        all other values of n, this setting will default back to using "stab_fid" as compare metric. If the
        "logical_success" setting is active, "init_seed_search" is set to False.

    Returns
    -------
    data : (n_max+1, k_max+1) matrix with objects of class Path
        Each element (n, k) of this matrix is used to store information about (n, k) itself and how it's made
    """
    start_time = time.time()
    if da_type == "random":
        random.seed(seeds[0]) if isinstance(seeds, list) else random.seed(seeds)
    try:
        input_state = float(input_state)
    except ValueError:
        if calculate_prot is False:
            raise ValueError("The parameter 'input state' should indicate the fidelity of Werner states used as "
                             "input when 'calculate_prot' is set to False.")
        input_state = input_state
    if calculate_prot and isinstance(seeds, int):
        seeds = [*range(seeds)]
    data, precalculated_data = create_storage(n_max, k_max, nstate, input_data_file, calculate_prot)
    # dynamic algorithm by using update function
    marker_cnt = 0
    first_n = True

    if metric == "weighted_sum_max":
        precalculated_data[n_max][k_max] = False
    if n_start > 2:
        for n in range(n_start, n_max + 1):
            first_k = k_start if n == n_start else n - 1
            for k in range(first_k, k_max + 1 - n_max + n):
                precalculated_data[n][k] = False
                data[n][k] = [None] * nstate

    avg_iteration_times = [0] * len(data)
    for i in range(len(data)):
        avg_iteration_times[i] = [0] * len(data[i])

    for n in range(n_start, n_max + 1):
        k_start = k_start if first_n else n - 1
        for k in range(k_start, k_max + 1 - n_max + n): # k_max + 1
            if precalculated_data[n][k] is False:
                start_time_n_k = time.time()

                if metric == "weighted_sum_max":
                    if n == n_max and k == k_max:
                        metric_used = "weighted_sum"
                    else:
                        metric_used = "stab_fid"
                else:
                    metric_used = metric

                if (rot_type == "k_max" and k == k_max) or (rot_type == "n_max" and k == k_max and n == n_max) \
                        or rot_type == "always":
                    inc_rot = 1
                else:
                    inc_rot = 0
                if da_type == "random":
                    update_random(data, n, k, input_state, show_or_not, nstate, inc_rot, T)
                else:
                    output = update_m_prot(data, n, k, input_state, show_or_not, nstate, inc_rot, da_type,
                                           marker_cnt=marker_cnt, calculate_prot=calculate_prot,
                                           nmb_id_elems=nmb_id_elems, seeds=seeds,
                                           number_of_threads=number_of_threads, start_time=start_time,
                                           failed_protocol_prefix=failed_protocol_prefix,
                                           init_seed_search=init_seed_search,
                                           skip_similar_prots=skip_similar_prots,
                                           reshuffle_protocols=reshuffle_protocols,
                                           p_g=p_g, alpha=alpha, n_DD=n_DD, metric=metric_used)
                    if da_type == "mpuP":
                        marker_cnt, avg_iteration_times[n][k] = output
                        print(f"Average calculation time for this block: {avg_iteration_times[n][k]}.")
                    else:
                        marker_cnt = output
                print(f"This block took {time.time() - start_time_n_k} seconds to calculate. "
                      f"{len(data[n][k])} protocols were stored in the buffer at location ({n}, {k}).")
        first_n = False

    print("\nAverage calculation times:")
    for i in range(2, len(avg_iteration_times)):
        print(f"avg_iteration_times[{i}] = {avg_iteration_times[i]}")

    print("\nEstimations for next possible next steps:")
    extra_k_values = 3
    data_next, _ = create_storage(n_max, k_max + extra_k_values, nstate, input_data_file, calculate_prot)

    est_iteration_times = [0] * len(data_next)
    est_extra_calc_times = [0] * len(data_next)
    extra_prot_numbers = [0] * len(data_next)
    for i in range(len(data_next)):
        est_iteration_times[i] = [0] * len(data_next[i])
        est_extra_calc_times[i] = [0] * len(data_next[i])
        extra_prot_numbers[i] = [0] * len(data_next[i])
    for i in range(len(data)):
        for j in range(len(data[i])):
            est_iteration_times[i][j] = avg_iteration_times[i][j]

    for extra_k in range(1, extra_k_values + 1):
        for n in range(n_start, n_max + 1):
            for k in range(k_max + extra_k - n_max + n, k_max + extra_k + 1 - n_max + n):
                if (rot_type == "k_max" and k == k_max + extra_k) \
                        or (rot_type == "n_max" and k == k_max + extra_k and n == n_max) \
                        or rot_type == "always":
                    inc_rot = 1
                else:
                    inc_rot = 0
                extra_prot_numbers[n][k] = len(list(yield_generators_parameter_search(data_next, n, k, inc_rot)))
                if da_type == "mpuP":
                    est_iteration_times[n][k] = (est_iteration_times[n][k - 1])**2 / est_iteration_times[n][k - 2] \
                        if k > 2 and est_iteration_times[n][k - 2] > 0 else est_iteration_times[n][k - 1]
                    est_extra_calc_times[n][k] = extra_prot_numbers[n][k] * est_iteration_times[n][k]
                    if n > 2:
                        extra_prot_numbers[n][k] += extra_prot_numbers[n-1][k-1]
                        est_extra_calc_times[n][k] += est_extra_calc_times[n-1][k-1]
                    if extra_k > 1 and n == n_max:
                        extra_prot_numbers[n][k] += extra_prot_numbers[n][k - 1]
                        est_extra_calc_times[n][k] += est_extra_calc_times[n][k - 1]
                    if n == n_max and k == k_max + extra_k - n_max + n:
                        print(f"For (n, k) = ({n}, {k}) we need a total of {extra_prot_numbers[n][k]} extra protocols to calculate. This is estimated to take {est_extra_calc_times[n][k]} time.")

    return data, marker_cnt


def dynamic_algorith_calculation_estimation():
    n_max = 4
    k_max = 30
    nstate = 15
    inc_rot = 0
    nmb_workers = 128
    nmb_seeds = 1

    data, _ = create_storage(n_max, k_max, nstate)

    data[2][1] = [None]
    data[2][2] = [None, None, None]
    data[3][2] = [None]
    data[3][3] = [None] * 9
    data[4][3] = [None] * 6

    # Estimates for set IIIc with p_g=p_m=0.0006:
    avg_iteration_times = [None] * (n_max + 1)
    avg_iteration_times[2] = [None, 0, 0.52, 0.63, 0.74, 0.85, 0.96, 1.04, 1.15, 1.29, 1.41, None, None]
    avg_iteration_times[3] = [None, None, 0.92, 1.06, 1.20, 1.34, 1.46, 1.60, 1.70, 1.84, 1.97, 2.10, None]
    avg_iteration_times[4] = [None, None, None, 2.67, 2.91, 3.06, 3.27, 3.47, 3.62, 3.79, 3.94, 4.24, 4.46]
    # avg_iteration_times[2] = [0, 0.0, 0.4050491491953532, 0.4910610238711039, 0.6509821523974333, 0, 0]
    # avg_iteration_times[3] = [0, 0, 0.7143664824962616, 0.8234660841506204, 1.0038190987706184, 1.1327383585540312, 0]
    # avg_iteration_times[4] = [0, 0, 0, 2.068492341306474, 2.4720977962451673, 2.6305897465321637, 2.7995756462759758]

    # Based on these numbers, the 4,9 additional step would take approximately 51 hours
    # In the program, it is estimated to take about 35 hours

    time = 0

    for n in range(2, n_max + 1):
        for k in range(n - 1, k_max + 1 - n_max + n):
            if len(avg_iteration_times[n]) <= k:
                avg_iteration_times[n].append(None)
            if avg_iteration_times[n][k] is None:
                avg_iteration_times[n][k] = avg_iteration_times[n][k - 1]**2 / avg_iteration_times[n][k - 2]
            time += math.ceil(len(list(yield_generators_parameter_search(data, n, k, inc_rot))) / nmb_workers) * nmb_seeds * avg_iteration_times[n][k]

    print(time/3600)


def store_data_intF(n, k, da_type='mpc', F_min=0, F_max=1, seg=100, nstate=2, inc_rot=0, seed=10, T=0.0009):
    """
    Function that executes the dynamic program over a range of values F for the isotropic Bell diagonal states used,
    and stores the data file in a .txt file so that it can be loaded later again.

    Parameters
    ----------
    n : positive integer
        Contains the maximum number of parties for which we want to apply the dynamic program
    k : positive integer
        Contains the maximum number of isotropic Bell diagonal states for which we want to apply the dynamic program
    da_type : string
        String describing what version of the deterministic random the algorithm should carry out: 'sp' is a single
        protocol per value of n and k, 'mpc' is multiple protocols per value of n and k based on different conditions,
        'mpF' is multiple protocols per n and k based on the highest fidelity, and 'random' is the randomized version
        of the dynamic program
    F_min : float between 0 and 1, smaller than F_max
        Minimum fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    F_max : float between 0 and 1
        Maximum fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    seg : positive integer
        number of fidelity values used in the data structure stored in the .txt file
    nstate : positive integer
        Number of protocols stored per value of n and k in data
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    seed : integer or list of length seg
        Integer that determines which should be used in the randomized version of the dynamic program. This parameter
        can also be an array of list seg: in this situation for each fidelity in the range a different seed will be
        used.
    T : float
        temperature in exp**(delta_F/T). A higher T will allow this function accept more states with low fidelity
    """
    if (n < 2) | (k < (n - 1)):
        sys.exit('In save_data_F_new: should be not((n < 2) | (k < (n - 1)))')

    if da_type == 'random':
        if isinstance(seed, list) is False:
            filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_intF_' + str(F_min) + '_' + str(F_max) \
                       + '_' + str(seg) + '_' + str(nstate) + '_' + str(inc_rot) + '_' + str(seed) + '_' + str(T) \
                       + '.txt'
        elif isinstance(seed, list) is True and len(seed) == seg + 1:
            filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_intF_' + str(F_min) + '_' + str(F_max) \
                       + '_' + str(seg) + '_' + str(nstate) + '_' + str(inc_rot) + '_' + str(seed[0]) + '-' \
                       + str(seed[seg]) + '_' + str(T) + '.txt'
        else:
            sys.exit("In store_data_inF: Input parameter seed must of length 1 or length seg.")
    else:
        filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_intF_' + str(F_min) + '_' + str(F_max) \
                   + '_' + str(seg) + '_' + str(nstate) + '_' + str(inc_rot) + '.txt'

    if os.path.isfile(filename):
        print("The file", filename, "is already available in the directory.")
    else:
        dataF = np.empty((seg + 1), dtype=object)
        for i in range(seg + 1):
            current_time = time.ctime(time.time())
            F = F_min + (F_max - F_min) * i / seg
            print(i, current_time, F)
            if isinstance(seed, list) is False:
                dataF[i] = dynamic_algorithm(n, k, F, da_type, nstate, inc_rot, 0, seed, T)
            else:
                dataF[i] = dynamic_algorithm(n, k, F, da_type, nstate, inc_rot, 0, seed[i], T)
        pickle.dump(dataF, open(filename, "wb"))
    return


def import_data_intF(n, k, da_type='mpc', F_min=0, F_max=1, seg=100, nstate=2, inc_rot=0, seed=10, T=0.0009):
    if (n < 2) | (k < (n - 1)):
        sys.exit('In save_data_F_new: should be not((n < 2) | (k < (n - 1))).')

    if da_type == 'random':
        if isinstance(seed, list) is False:
            filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_intF_' + str(F_min) + '_' + str(F_max) \
                       + '_' + str(seg) + '_' + str(nstate) + '_' + str(inc_rot) + '_' + str(seed) + '_' + str(T) \
                       + '.txt'
        elif isinstance(seed, list) is True and len(seed) == seg + 1:
            filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_intF_' + str(F_min) + '_' + str(F_max) \
                       + '_' + str(seg) + '_' + str(nstate) + '_' + str(inc_rot) + '_' + str(seed[0]) + '-' \
                       + str(seed[seg]) + '_' + str(T) + '.txt'
        else:
            sys.exit("In import_data_inF: Input parameter seed must of length 1 or length seg.")
    else:
        filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_intF_' + str(F_min) + '_' + str(F_max) \
                   + '_' + str(seg) + '_' + str(nstate) + '_' + str(inc_rot) + '.txt'

    if os.path.isfile(filename):
        dataF = pickle.load(open(filename, 'rb'))
    else:
        print('The data file ' + filename + ' is not created yet. It is being created now.')
        store_data_intF(n, k, da_type, F_min, F_max, seg, nstate, inc_rot, seed, T)
        dataF = pickle.load(open(filename, 'rb'))

    return dataF


def store_data_varT(n, k, da_type='random', F=0.9, nstate=200, inc_rot=0, seed=10, T=0.0009):
    """
    Function that executes the randomized version of the dynamic program over a range of temperatures, and stores the
    data file in a .txt file so that it can be loaded later again.

    Parameters
    ----------
    n : positive integer
        Contains the maximum number of parties for which we want to apply the dynamic program
    k : positive integer
        Contains the maximum number of isotropic Bell diagonal states for which we want to apply the dynamic program
    da_type : string
        String describing what version of the deterministic random the algorithm should carry out: 'sp' is a single
        protocol per value of n and k, 'mpc' is multiple protocols per value of n and k based on different conditions,
        'mpF' is multiple protocols per n and k based on the highest fidelity, and 'random' is the randomized version
        of the dynamic program
    F : float between 0 and 1
        Fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    nstate : positive integer
        Number of protocols stored per value of n and k in data
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    seed : integer
        Integer that determines which should be used in the randomized version of the dynamic program
    T : float
        temperature in exp**(delta_F/T). A higher T will allow this function accept more states with low fidelity
    """
    if (n < 2) | (k < (n - 1)):
        sys.exit('In save_data_F_new: should be not((n < 2)|(k < (n-1)))')
    if da_type != 'random':
        sys.exit('The function import_data_varT is indended for da_type random.')

    T_list = [0.00001, 0.00002, 0.00003, 0.00004, 0.00005, 0.00006, 0.00007, 0.00008, 0.00009, 0.0001, 0.0002, 0.0003,
              0.0004, 0.0005, 0.0006, 0.0007, 0.0008, 0.0009]
    seed_list = [3316, 7876, 570, 1264, 9343, 6959, 1162, 2100, 5177, 8559, 5454, 8917, 6232, 2994, 9603, 9296, 8193,
                 9321, 4319, 4239, 4010, 7355, 9398, 9047, 273, 9697, 6637, 8965, 2599, 5148, 6372, 5911, 3844, 17,
                 5263, 200, 4720, 787, 5339, 7157, 8184, 5289, 9342, 9304, 3409, 4122, 2967, 1789, 3048, 4734, 4831,
                 6272, 6897, 8397, 3360, 1109, 8164, 1361, 9541, 5428, 6766, 1837, 8560, 1043, 6328, 701, 1082, 3725,
                 852, 6029, 7106, 8174, 2556, 7533, 6013, 9076, 7502, 4950, 8562, 4164, 561, 6941, 1464, 4496, 4230,
                 8111, 9981, 5976, 9707, 8695, 2589, 3038, 1126, 7144, 6165, 845, 1555, 8660, 9783, 6466, 9075, 9674,
                 1526, 1222, 4328, 4231, 1820, 6602, 6091, 1714, 2421]
    nT = np.size(T_list)

    dataT = np.empty((nT, n + 1, k + 1, nstate), dtype=object)
    for seed in seed_list:
        filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_varT_' + str(F) + '_' + str(nstate) \
                   + '_' + str(inc_rot) + '_' + str(seed) + '.txt'

        if os.path.isfile(filename):
            print("The file", filename, "is already available in the directory.")

        else:
            print("-----")
            localtime = time.asctime(time.localtime(time.time()))
            print(seed, '\t', localtime)
            for num in range(nT):
                T = T_list[num]
                data_random = dynamic_algorithm(n, k, F, da_type, nstate, inc_rot, 0, seed, T)
                for n in range(2, n + 1):
                    for k in range(n - 1, k + 1):
                        for t in range(nstate):
                            dataT[num][n][k][t] = data_random[n][k][t].state[0]
                localtime = time.asctime(time.localtime(time.time()))
                print(data_random[n][k][0].state[0], '\t', localtime)
            pickle.dump(dataT, open(filename, "wb"))
            print("-----")
    return


def import_data_varT(n, k, da_type='random', F=0.9, nstate=200, inc_rot=0, seed=10, T=0.0009):
    if (n < 2) | (k < (n - 1)):
        sys.exit('In save_data_F_new: should be not((n < 2) | (k < (n - 1))).')
    if da_type != 'random':
        sys.exit('The function import_data_varT is intended for da_type random.')

    filename = 'calc_data/' + da_type + '_' + str(n) + '_' + str(k) + '_varT_' + str(F) + '_' + str(nstate) + '_' \
               + str(inc_rot) + '_' + str(seed) + '.txt'

    if os.path.isfile(filename):
        dataT = pickle.load(open(filename, 'rb'))
    else:
        sys.exit(str('The data file ' + filename + ' is not created yet.'))

    return dataT


def spike(n, k, da_type='mpc', F_min=0.8, F_max=1, seg=100, nstate=2, inc_rot=0, F=0.85, seed=10, T=0.0009):
    """
    Function that allows one to find the best protocol found in a range of protocols are the input fidelity.

    Parameters
    ----------
    filename : string
        Contains the location of the .txt file containing the data structure calculated with the dynamic program
    n : positive integer
        Contains the number of parties for which we find the best protocol
    k : positive integer
        Contains the number of isotropic Bell diagonal states for which we find the best protocol
    da_type : string
        String describing what version of the deterministic random the algorithm should carry out: 'sp' is a single
        protocol per value of n and k, 'mpc' is multiple protocols per value of n and k based on different conditions,
        'mpF' is multiple protocols per n and k based on the highest fidelity, and 'random' is the randomized version
        of the dynamic program
    F_min : float between 0 and 1, smaller than F_max
        Minimum fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    F_max : float between 0 and 1
        Maximum fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    seg : positive integer
        number of fidelity values used in the data structure stored in the .txt file
    nstate : positive integer
        Number of protocols stored per value of n and k in data
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    F : float between 0 and 1
        Fidelity of the isotropic Bell diagonal states for which we want to find the best protocol
    seed : integer
        Integer that determines which should be used in the randomized version of the dynamic program
    T : float
        temperature in exp**(delta_F/T). A higher T will allow this function accept more states with low fidelity

    Returns
    -------
    i_spike : integer
        Describes at what fidelity the protocol can be found that leads to highest output fidelity using F as input
        fidelity
    """
    dataF = import_data_intF(n, k, da_type, F_min, F_max, seg, nstate, inc_rot, seed, T)
    to_compare = 0  # F when input EPR F is 0.98
    i_spike = 0
    for i in range(seg + 1):
        for j in range(nstate):
            protocol = dap.identify_protocol(dataF[i], n, k, j)
            # print(n, k, i, j)
            tmp = dap.operate_protocol(protocol, nstate, F)[0]
            if tmp > to_compare:
                i_spike = i
                to_compare = tmp
    return i_spike


# Multiprocessing version of the algorithm
def save_dataF_mp(n, k, F_min=0, F_max=1, seg=100, ntype=3, inc_rot=0):
    """
    Function that executes the dynamic program and stores the data file in a .txt file so that it can be loaded later
    again, and uses multiprocessing to do so.

    Parameters
    ----------
    n : positive integer
        Contains the maximum number of parties for which we want to apply the dynamic program
    k : positive integer
        Contains the maximum number of isotropic Bell diagonal states for which we want to apply the dynamic program
    F_min : float between 0 and 1, smaller than F_max
        Minimum fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    F_max : float between 0 and 1
        Maximum fidelity of the isotropic Bell diagonal states for which we want to apply the dynamic program
    seg : positive integer
        number of fidelity values used in the data structure stored in the .txt file
    ntype : positive integer
        number of protocols stored per value of n and k in the search process where the protocol was found
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    """
    if (n < 2) | (k < (n - 1)):
        sys.exit('In save_data_F_new: should be !((n<2)|(k<(n-1)))')

    return_dict = multiprocessing.Manager().dict()
    n_cpu = 2
    n_steps = floor(seg / n_cpu)
    for j in range(n_steps + 1):
        processes = []
        for i in range(j * n_cpu, min((j + 1) * n_cpu, seg + 1)):
            current_time = time.ctime(time.time())
            print(i, current_time)
            F = F_min + (F_max - F_min) * i / seg
            p = multiprocessing.Process(target=mp_dynamic_algorithm, args=(return_dict, i, n, k, F, ntype, inc_rot))
            processes.append(p)
            p.start()
        for process in processes:
            process.join()

    dataF_new = np.empty((seg + 1), dtype=object)
    for i in range(seg + 1):
        dataF_new[i] = return_dict[i]

    pickle.dump(dataF_new, open('calc_data/' + 'mpc' + '_' + str(n) + '_' + str(k) + '_' + str(F_min) + '_' + str(F_max)
                                + '_' + str(seg) + '_' + str(ntype) + '_' + str(inc_rot) + '_mp.txt', "wb"))


def mp_dynamic_algorithm(return_dict, i, n, k, F, ntype, inc_rot):
    """
    Function that carries out the dynamic algorithm and updates the object data in this process (using the update
    function), catered for multiprocessing.

    Parameters
    ----------
    return_dict : global object in the multiprocessing process
    i : nonnegative integer
        number indicating the memory spot in return_dict
    n : positive integer
        Maximum number of qubits of the GHZ diagonal state stored in data
    k : positive integer
        Maximum number of isotropic Bell diagonal states stored in data
    F : float between 0 and 1
        Fidelity of the isotropic Bell diagonal states used
    ntype : positive integer
        Number of protocols stored per value of n and k in data
    inc_rot : Boolean
        0 means not ancillary permutations are included, 1 means they are included
    """
    return_dict[i] = dynamic_algorithm(n, k, F, 'mpc', ntype, inc_rot, 0)


def find_symmetric_protocols(n, k, seeds, Tvalues, input_fid=0.9, thr_fid=0.95, nstate=200, inc_rot=0):
    """
    Function that runs the random algorithm for a list of seeds and temperature values and evaluates if the protocols
    found have 1) high enough fidelity and 2) are symmetric (meaning at all times two links can be created at both
    sides of the network).

    Parameters
    ----------
    n : positive integer
        Maximum number of qubits of the GHZ diagonal state stored in data.
    k : positive integer
        Maximum number of isotropic Bell diagonal states stored in data.
    seeds: list of integers
        Seed values for which the random dynamic algorithm runs.
    Tvalues : list of floats
        Temperature values for which the random dynamic algorithm runs.
    input_fid : float between 0 and 1
        Fidelity of the isotropic Bell diagonal states used
    thr_fid : float between 0 and 1 (default value 0.95)
        Minimum fidelity of the final GHZ state that protocol has to have to be considered "good enough" by the
        algorithm.
    nstate : integer bigger than 0 (default value 200)
        Number of protocols stored per value of n and k.
    inc_rot : Boolean (default value 0)
        0 means not ancillary permutations are included, 1 means they are included.

    Returns
    -------
    protocols_found : list of protocol, final fidelity of protocol based on input_fid, seed and temperature used to
                      find the protocol
    """
    protocols_found = []
    for seed in seeds:
        for T in Tvalues:
            data = dynamic_algorithm(n, k, input_fid, 'random', nstate, inc_rot, 0, seed, T)
            for t in range(nstate):
                if data[n][k][t].state[0] > thr_fid:
                    dyn_prot = dap.identify_protocol(data, n, k, t)
                    if dap.is_prot_symmetric(dyn_prot):
                        protocols_found.append([dap.protocol_add_meta_data(dyn_prot), data[n][k][t].state, seed, T])
                        print("Symmetric protocol found with fidelity {} at seed = {} and T = {}.".format(data[n][k][t].state, seed, T))
                else:
                    break
                # # Part I used before for (n,k) = (4,6):
                # found_one = True
                # for link in range(i):
                #     if id_linked[link] is None:
                #         found_one = False
                # if found_one is True and data[4][6][nr_state].state[0] == 0.8550360733409689:
                #     # print(data[4][6][nr_state].state[0])
                #     # print(dap.operate_protocol(dyn_prot, nstate, 0.9)[0])
                #     # print(id_elem)
                #     links = [min(id_elem[0][1], id_elem[id_linked[0]][1])]
                #     links_skip = [id_linked[0]]
                #     for link in range(1, i):
                #         if link not in links_skip:
                #             links.append(min(id_elem[link][1], id_elem[id_linked[link]][1]))
                #             links_skip.append(id_linked[link])
                #     if links[0] == links[1] or links[1] == links[2]:
                #         print(links)
                #         nr_prot = nr_prot + 1
                #         if nr_prot < 10:
                #             name = "protocols/dyn_prot_4_" + str(i) + "_sym_" + str(nr_prot)
                #             pickle.dump(dyn_prot, open(name, "wb"))
                #             name = "dyn_prot_4_" + str(i) + "_sym_" + str(nr_prot)
                #             plotp.plot_protocol(dyn_prot, 1, name=name, print_reg_slot=False, print_id=True)
                #     if data[4][6][nr_state].state[0] > F_max:
                #         F_max = data[4][6][nr_state].state[0]
    # print("nr_prot = {}".format(nr_prot))
    # print(F_max)
    return protocols_found
