"""
2023-2024 Sébastian de Bone (QuTech)
https://github.com/sebastiandebone/ghz_prot_II/
"""
import numpy as np
from math import ceil, sqrt


def number_of_entanglement_attempts_per_echo_pulse(p_link=0.1, t_link=6e-6, t_pulse=1e-3, show_tot_prob=False,
                                                   version=2):
    kMin = 1                            # Minimum number of link generation attempts per echo tried
    kMax = max(100, int(1 / p_link))    # Maximum number of link generation attempts per echo tried
    # tLink = 6e-6                      # Time of a single link generation attempt
    # tEcho = 1e-3                      # Time of an echo pulse
    # pLink = 0.00447                   # Success probability of a single link generation attempt
    N = int(10 / p_link)                # Maximum number of entanglement attempts used in the calculation
    tLink = t_link
    pLink = p_link
    tEcho = t_pulse

    averageTotalTime = [0] * (kMax - kMin + 1)
    totalProbability = 0
    for i_k, k in enumerate(range(kMin, kMax + 1)):
        if show_tot_prob:
            print(i_k)
        averageTime = 0
        for n1 in range(1, N + 1):
            if version == 3:
                number_seq_needed_a_1 = ceil(n1 / (2 * k))
                for n2 in range(1, N + 1):
                    number_seq_needed_a_2 = ceil(n2 / (2 * k))
                    prob_a = pLink ** 2 * (1 - pLink) ** (n1 + n2 - 2)
                    for n3 in range(1, N + 1):
                        number_seq_needed_b_3 = ceil(n3 / (2 * k))
                        for n4 in range(1, N + 1):
                            number_seq_needed_b_4 = ceil(n4 / (2 * k))
                            number_seq_needed = max(number_seq_needed_a_1 + number_seq_needed_a_2,
                                                    number_seq_needed_b_3 + number_seq_needed_b_4)
                            prob_b = pLink ** 2 * (1 - pLink) ** (n3 + n4 - 2)
                            averageTime += prob_a * prob_b * number_seq_needed * (2 * k * tLink + tEcho)
                            if i_k == 0:
                                totalProbability += prob_a * prob_b
            elif version == 2:
                for n2 in range(1, N + 1):
                    number_seq_needed = ceil(max(n1, n2) / (2 * k))
                    seq_time = number_seq_needed * (2 * k * tLink + tEcho)
                    averageTime += pLink ** 2 * (1 - pLink) ** (n1 + n2 - 2) * seq_time
                    if i_k == 0:
                        totalProbability += pLink ** 2 * (1 - pLink) ** (n1 + n2 - 2)
            else:
                number_seq_needed = ceil(n1 / (2 * k))
                seq_time = number_seq_needed * (2 * k * tLink + tEcho)
                averageTime += pLink * (1 - pLink) ** (n1 - 1) * seq_time
                if i_k == 0:
                    totalProbability += pLink * (1 - pLink) ** (n1 - 1)
        # print(averageTime)
        averageTotalTime[i_k] = averageTime     # Average time per entangled link
    if show_tot_prob:
        print(f"totalProbability = {totalProbability}")

    return int(np.argmin(averageTotalTime) + kMin)


def calculate_bell_state_properties(bell_pair_parameters):
    mu = bell_pair_parameters['mu']
    F_prep = bell_pair_parameters['F_prep']
    p_DE = bell_pair_parameters['p_DE']
    phi = sqrt(mu) * ((2 * F_prep - 1) ** 2) * ((1 - p_DE) ** 2)
    eta = bell_pair_parameters['eta']
    labda = bell_pair_parameters['lambda']

    alpha = bell_pair_parameters['alpha'] if 'alpha' in bell_pair_parameters else None
    if bell_pair_parameters['ent_prot'] == 'single_click':
        if alpha is None:
            raise ValueError(f"The parameter 'alpha' was not set, but is required for the single-click protocol.")
        p_link = 2 * eta * alpha + 0.5 * (eta * alpha) ** 2 * (mu - 3)
        F_link = 1 / p_link * (1 + phi * (2 * labda - 1)) * eta * alpha * (1 - alpha)
    elif bell_pair_parameters['ent_prot'] == 'double_click':
        p_link = eta ** 2 / 2
        F_link = 0.5 * (1 + phi ** 2)
    else:
        raise ValueError(f"The key {bell_pair_parameters['ent_prot']} given at 'ent_prot' is not understood.")

    return p_link, F_link


if __name__ == "__main__":
    print(number_of_entanglement_attempts_per_echo_pulse(p_link=0.01, show_tot_prob=True, version=2))
