/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2020 ENERCON GmbH
    Copyright (C) 2018-2021 OpenCFD Ltd
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "flyingActuationDiskSource.H"
//#include "geometricOneField.H"
#include "cellSet.H"
#include "addToRunTimeSelectionTable.H"

// * * * * * * * * * * * * * * Static Data Members * * * * * * * * * * * * * //

namespace Foam
{
namespace fv
{
    defineTypeNameAndDebug(flyingActuationDiskSource, 0);
    addToRunTimeSelectionTable(option, flyingActuationDiskSource, dictionary);
}
}


const Foam::Enum
<
    Foam::fv::flyingActuationDiskSource::forceMethodType
>
Foam::fv::flyingActuationDiskSource::forceMethodTypeNames
({
    { forceMethodType::FROUDE, "Froude" },
    { forceMethodType::VARIABLE_SCALING, "variableScaling" },
});


const Foam::Enum
<
    Foam::fv::flyingActuationDiskSource::monitorMethodType
>
Foam::fv::flyingActuationDiskSource::monitorMethodTypeNames
({
    { monitorMethodType::POINTS, "points" },
    { monitorMethodType::CELLSET, "cellSet" },
});


// * * * * * * * * * * * * Protected Member Functions  * * * * * * * * * * * //

void Foam::fv::flyingActuationDiskSource::writeFileHeader(Ostream& os)
{
    writeFile::writeHeader(os, "Actuation disk source");
    writeFile::writeCommented(os, "Time");
    writeFile::writeCommented(os, "Uref");
    writeFile::writeCommented(os, "rotorCpStar");
    writeFile::writeCommented(os, "rotorCtStar");

    if (forceMethod_ == forceMethodType::FROUDE)
    {
        writeFile::writeCommented(os, "a");
        writeFile::writeCommented(os, "rotorT");
        writeFile::writeCommented(os, "froce vector");
    }
    else if (forceMethod_ == forceMethodType::VARIABLE_SCALING)
    {
        writeFile::writeCommented(os, "Udisk");
        writeFile::writeCommented(os, "rotorT");
        writeFile::writeCommented(os, "rotorP");
        writeFile::writeCommented(os, "bladeT");
        writeFile::writeCommented(os, "bladeP");
        writeFile::writeCommented(os, "total force");
        writeFile::writeCommented(os, "rotor force");
        writeFile::writeCommented(os, "blade force");
        writeFile::writeCommented(os, "Calc rotor Power");
        writeFile::writeCommented(os, "Calc blade Power");
    }

    os  << endl;
}


// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

void Foam::fv::flyingActuationDiskSource::setMonitorCells(const dictionary& dict)
{
    switch (monitorMethod_)
    {
        case monitorMethodType::POINTS:
        {
            Info<< "    - selecting cells using points" << endl;

            labelHashSet selectedCells;

            List<point> monitorPoints;

            const dictionary* coeffsDictPtr = dict.findDict("monitorCoeffs");
            if (coeffsDictPtr)
            {
                coeffsDictPtr->readIfPresent("points", monitorPoints);
            }
            else
            {
                monitorPoints.resize(1);
                dict.readEntry("upstreamPoint", monitorPoints.first());
            }

            for (const auto& monitorPoint : monitorPoints)
            {
                const label celli = mesh_.findCell(monitorPoint);
                if (celli >= 0)
                {
                    selectedCells.insert(celli);
                }

                const label globalCelli = returnReduce(celli, maxOp<label>());
                if (globalCelli < 0)
                {
                    WarningInFunction
                        << "Unable to find owner cell for point "
                        << monitorPoint << endl;
                }
            }

            monitorCells_ = selectedCells.sortedToc();
            break;
        }
        case monitorMethodType::CELLSET:
        {
            Info<< "    - selecting cells using cellSet "
                << zoneName() << endl;

            monitorCells_ = cellSet(mesh_, zoneName()).sortedToc();
            break;
        }
        default:
        {
            FatalErrorInFunction
                << "Unknown type for monitoring of incoming velocity"
                << monitorMethodTypeNames[monitorMethod_]
                << ". Valid monitor method types : "
                << monitorMethodTypeNames
                << exit(FatalError);
        }
    }
}


// Added by Clem 2023_10_25
// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

Foam::fv::flyingActuationDiskSource::~flyingActuationDiskSource()
{}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::fv::flyingActuationDiskSource::flyingActuationDiskSource
(
    const word& name,
    const word& modelType,
    const dictionary& dict,
    const fvMesh& mesh
)
:
    cellSetOption(name, modelType, dict, mesh),
    writeFile(mesh, name, modelType, coeffs_),
    forceMethod_
    (
        forceMethodTypeNames.getOrDefault
        (
            "variant",
            coeffs_,
            forceMethodType::FROUDE
        )
    ),
    monitorMethod_
    (
        monitorMethodTypeNames.getOrDefault
        (
            "monitorMethod",
            coeffs_,
            monitorMethodType::POINTS
        )
    ),
    sink_
    (
        coeffs_.getOrDefault<bool>("sink", true)
      ? 1
      : -1
    ),
    writeFileStart_(coeffs_.getOrDefault<scalar>("writeFileStart", 0)),
    writeFileEnd_(coeffs_.getOrDefault<scalar>("writeFileEnd", VGREAT)),
    diskArea_
    (
        coeffs_.getCheck<scalar>
        (
            "diskArea",
            scalarMinMax::ge(VSMALL)
        )
    ),
    diskDir_
    (
        coeffs_.getCheck<vector>
        (
            "diskDir",
            [&](const vector& vec){ return mag(vec) > VSMALL; }
        ).normalise()
    ),
    rotorUvsCpPtr_(Function1<scalar>::New("rotorCp", coeffs_)),
    rotorUvsCtPtr_(Function1<scalar>::New("rotorCt", coeffs_)),
    bladeUvsCpPtr_(Function1<scalar>::New("bladeCp", coeffs_)),
    bladeUvsCtPtr_(Function1<scalar>::New("bladeCt", coeffs_)),    
    monitorCells_(),
    rotorCompensationFactor_(coeffs_.getOrDefault<scalar>("rotorCompensationFactor", 1.0)),
    bladeCompensationFactor_(coeffs_.getOrDefault<scalar>("bladeCompensationFactor", 1.0)),
    totalForce_(Zero),
    localForce_(Zero),
    totalForceRotor_(Zero),
    localForceRotor_(Zero),
    totalForceBlade_(Zero),
    localForceBlade_(Zero),
    rotorForcePrescribed_(coeffs_.getOrDefault<label>("rotorForcePrescribed", 0)),
    rotorTPrescribed_(coeffs_.getOrDefault<scalar>("rotorTPrescribed", 0.0)),
    bladeForcePrescribed_(coeffs_.getOrDefault<label>("bladeForcePrescribed", 0)),
    bladeTPrescribed_(coeffs_.getOrDefault<scalar>("bladeTPrescribed", 0.0)),
    bladeChord_(coeffs_.getOrDefault<scalar>("bladeChord", 0.0)),
    bladeCl_(coeffs_.getOrDefault<scalar>("bladeCl", 0.0)),
    bladeCd_(coeffs_.getOrDefault<scalar>("bladeCd", 0.0)),
    bladeNumber_(coeffs_.getOrDefault<label>("bladeNumber", 1)),
    bladeEndEffectsActive_(coeffs_.getOrDefault<label>("bladeEndEffectsActive", 0)),
    bladeSpanPerAL_(coeffs_.getOrDefault<scalar>("bladeSpanPerAL", 0.0)),
    nonUniformDiskLoad_(coeffs_.getOrDefault<label>("nonUniformDiskLoad", 0)),
    rotorStreamWisePosition_(coeffs_.getOrDefault<scalar>("rotorStreamWisePosition", -1.0)),
    widthOfDisk_(coeffs_.getOrDefault<scalar>("widthOfDisk", 0.0)),
    heightOfDisk_(coeffs_.getOrDefault<scalar>("heightOfDisk", 0.0)),
    velSamplingRadius_(coeffs_.getOrDefault<scalar>("velSamplingRadius", 0.0)),
    velSamplingNumber_(coeffs_.getOrDefault<label>("velSamplingNumber", 0)),
    timeStartAdjustPitch_(coeffs_.getOrDefault<scalar>("timeStartAdjustPitch", -1.0)),
    bladePitchTimeScale_(coeffs_.getOrDefault<scalar>("bladePitchTimeScale", 30.0)),
    bladePitchTol_(coeffs_.getOrDefault<scalar>("bladePitchTol", 0.5)),
    bladePitchSpeed_(coeffs_.getOrDefault<scalar>("bladePitchSpeed", 0.1)),
    meshBoundBox_(mesh_.points(), false),
    inflowVelocity_(Zero),
    forceField_
    (
        IOobject
        (
            "force." + name_,
            mesh_.time().timeName(),
            mesh_,
            IOobject::NO_READ,
            IOobject::AUTO_WRITE
        ),
        mesh_,
        dimensionedVector
        (
            "force",
            dimForce/dimVolume/dimDensity,
            vector::zero
        )
    )
{
    meshBoundBox_.inflate(1e-6);  // Added By Clem 2024_02_06

    coeffs_.lookup("bladeEndPositions") >> bladeEndPositions_; // Added By Clem 2024_01_22

    coeffs_.lookup("bladeActuatorPointsPositions") >> bladeActuatorPointsPositions_; // Added By Clem 2024_02_07

    coeffs_.lookup("rotorActuatorPointsPositions") >> rotorActuatorPointsPositions_; // Added By Clem 2024_09_10

    coeffs_.lookup("alphaVSClCd") >> alphaVSClCd_; // Added By Clem 2024_02_12

    alphaList_ = scalarList(alphaVSClCd_.size(), 0.0); // Added By Clem 2024_02_12

    clList_ = scalarList(alphaVSClCd_.size(), 0.0);    // Added By Clem 2024_02_12

    cdList_ = scalarList(alphaVSClCd_.size(), 0.0);    // Added By Clem 2024_02_12

    misTargetAOAFiltered_ = scalarList(bladeNumber_, 0.0); // Added By Clem 2024_11_11

    forAll(alphaVSClCd_, jj)
    {
        alphaList_[jj] = alphaVSClCd_[jj][0];
        clList_[jj] = alphaVSClCd_[jj][1];
        cdList_[jj] = alphaVSClCd_[jj][2];
    }

    // Added By Clem 2024_03_05 Adjusted by Clem 2024_11_06
    dict.readIfPresent("bladeSmearingLengthX", bladeSmearingLengthX_);
    dict.readIfPresent("bladeSmearingLengthY", bladeSmearingLengthY_);
    dict.readIfPresent("bladeSmearingLengthZ", bladeSmearingLengthZ_);

    // Added By Clem 2024_09_10
    dict.readIfPresent("rotorSmearingLengthX", rotorSmearingLengthX_);
    dict.readIfPresent("rotorSmearingLengthY", rotorSmearingLengthY_);
    dict.readIfPresent("rotorSmearingLengthZ", rotorSmearingLengthZ_);

    // Added By Clem 2024_09_10
    dict.readIfPresent("rotorFrontalAreaPerCell", rotorFrontalAreaPerCell_);

    setMonitorCells(coeffs_);

    fieldNames_.resize(1, "U");

    // Something Special HERE
    fv::option::resetApplied();

    Info<< "    - creating actuation disk zone: " << this->name() << endl;

    Info<< "    - force computation method: "
        << forceMethodTypeNames[forceMethod_] << endl;

    writeFileHeader(file());

    forceField_.write();

    // Set the domain to project Force of rotor Added by Clem 2023_10_30
    // dict.readEntry("rotorDomainSetName", rotorDomainSetName_);

    // rotorDomainCells_ = cellSet(mesh_, rotorDomainSetName_).sortedToc();

    // Set cells that project Force of blade Added by Clem 2023_10_30
    //Commented by Clem. Now the locations of actuator points are specified explicitly
    //dict.readEntry("bladeSetName", bladeSetName_);
    // bladeCells_ = cellSet(mesh_, bladeSetName_).sortedToc();


    // Set the domain to project Force of blade Added by Clem 2023_10_30
    // dict.readEntry("bladeDomainSetName", bladeDomainSetName_);

    // bladeDomainCells_ = cellSet(mesh_, bladeDomainSetName_).sortedToc();


    // Added by Clem 2023_10_30
    dict.readIfPresent("freeStreamDir", freeStreamDir_);
    freeStreamDir_.normalise();
    if (mag(freeStreamDir_) < VSMALL)
    {
        FatalIOErrorInFunction(dict)
            << "Free stream direction vector is zero: "
            << "freeStreamDir = " << freeStreamDir_
            << exit(FatalIOError);
    }


    // Added by Clem 2023_10_30
    dict.readIfPresent("bladeDir", bladeDir_);
    bladeDir_.normalise();
    if (mag(bladeDir_) < VSMALL)
    {
        FatalIOErrorInFunction(dict)
            << "Free stream direction vector is zero: "
            << "bladeDir = " << bladeDir_
            << exit(FatalIOError);
    }

    // Added by Clem 2024_01_23       
    //lateralDirection_(coeffs_.getOrDefault<vector>("lateralDirection", vector::zero)),
    dict.readIfPresent("lateralDirection", lateralDirection_);
    lateralDirection_.normalise();
    if (mag(lateralDirection_) < VSMALL)
    {
        FatalIOErrorInFunction(dict)
            << "Free stream direction vector is zero: "
            << "lateralDirection = " << lateralDirection_
            << exit(FatalIOError);
    }


    //Start added by Clem 2023_11_12

    //label rotorCurrentIndex = 0;

    //Added by Clem 2024_02_12
    //if (cells_.size() > 0)
    //{
    //    rotorStreamWiseShift_ = vector(rotorStreamWisePosition_ - mesh_.C()[cells_[0]].x(), 0.0, 0.0);
    //}

    //for (const label celli : cells_)
    //{
    //    label rotorOrBlade = 0;
    //    vector centerToSmear = mesh_.C()[celli]; //Adjusted by Clem 2024_02_08  Get the position vector
    //    centerToSmear = centerToSmear + rotorStreamWiseShift_; //Added by Clem 2024_02_12
    //    selectingCelltoSmear(centerToSmear, rotorCurrentIndex, rotorSmearingLength_, rotorOrBlade);
    //    rotorCurrentIndex += 1;

    //}



    // Start added By Clem 2024_09_10
    rotorNumberofActuatorPoints_ = 0;

    for (
        label rotorActuatorPointsCounter = 0; 
        rotorActuatorPointsCounter < rotorActuatorPointsPositions_.size(); 
        rotorActuatorPointsCounter++
    )
    {
        vector vectorHolder = rotorActuatorPointsPositions_[rotorActuatorPointsCounter];
        if (meshBoundBox_.containsInside(vectorHolder))
        {
            rotorNumberofActuatorPoints_ += 1;
        }
    }
    Pout << name_ << ": rotorNumberofActuatorPoints: " << rotorNumberofActuatorPoints_ << endl;


    label rotorCurrentIndex = 0;

    for ( 
        label rotorActuatorPointsCounter = 0; 
        rotorActuatorPointsCounter < rotorNumberofActuatorPoints_; 
        rotorActuatorPointsCounter++
    )
    {
        vector centerToSmear = rotorActuatorPointsPositions_[rotorActuatorPointsCounter];
        label rotorOrBlade = 0;
        scalar largestSmearingLength = max(rotorSmearingLengthX_[rotorActuatorPointsCounter], rotorSmearingLengthY_[rotorActuatorPointsCounter]);
        largestSmearingLength = max(largestSmearingLength, rotorSmearingLengthZ_[rotorActuatorPointsCounter]);
        selectingCelltoSmear(centerToSmear, rotorCurrentIndex, largestSmearingLength, rotorOrBlade);
        rotorCurrentIndex += 1;

    }
    // End added By Clem 2024_09_10




    // Added By Clem 2024_02_07
    bladeNumberofActuatorPoints_ = 0;

    for (
        label actuatorPointsCounter = 0; 
        actuatorPointsCounter < bladeActuatorPointsPositions_.size(); 
        actuatorPointsCounter++
    )
    {
        vector vectorHolder = bladeActuatorPointsPositions_[actuatorPointsCounter];
        //Pout << "vectorHolder: " << vectorHolder << endl;
        //label celli = findCell(vectorHolder);
        //Pout << "celli: " << celli << endl;
        //if (celli >= 0)
        if (meshBoundBox_.containsInside(vectorHolder))
        {
            bladeNumberofActuatorPoints_ += 1;
        }
    }
    Pout << name_ << ": bladeNumberofActuatorPoints: " << bladeNumberofActuatorPoints_ << endl;

    label bladeCurrentIndex = 0;

    // Adjusted by Clem 2024_02_07
    for ( 
        label actuatorPointsCounter = 0; 
        actuatorPointsCounter < bladeNumberofActuatorPoints_; 
        actuatorPointsCounter++
    )
    {
        vector centerToSmear = bladeActuatorPointsPositions_[actuatorPointsCounter];
        label rotorOrBlade = 1;
        scalar largestSmearingLength = max(bladeSmearingLengthX_[actuatorPointsCounter], bladeSmearingLengthY_[actuatorPointsCounter]);
        largestSmearingLength = max(largestSmearingLength, bladeSmearingLengthZ_[actuatorPointsCounter]);
        selectingCelltoSmear(centerToSmear, bladeCurrentIndex, largestSmearingLength, rotorOrBlade);
        bladeCurrentIndex += 1;

    }

    //Added by Clem 2024_01_25
    //Adjusted by Clem 2024_02_07
    if (bladeNumberofActuatorPoints_ > 0)
    {
        bladeRelDistToTipCalc();
    }
    
    //Added by Clem 2024_02_16
    dict.readIfPresent("bladeTwist", bladeTwist_);

    //Added by Clem 2024_02_29
    dict.readIfPresent("bladePitch", bladePitch_);

    //Added by Clem 2024_02_29
    dict.readIfPresent("bladeAOATarget", bladeAOATarget_);

    //Added by Clem 2024_02_29
    dict.readIfPresent("bladeNumIndicator", bladeNumIndicator_);

    //Added by Clem 2024_02_29
    dict.readIfPresent("indexForBladeMid", indexForBladeMid_);


    //Added by Clem 2024_02_29
    if (bladeNumberofActuatorPoints_ > 0)
    {
        bladeAOA_ = scalarList(bladeNumberofActuatorPoints_, 0.0);
        bladeAOAMidofThatBlade_ = scalarList(bladeNumber_, 0.0);
    }


    //Added by Clem 2024_01_27
    if (bladeEndEffectsActive_ == 1)
    {
        dict.readIfPresent("tipCorrectionRelSpanRef", tipCorrectionRelSpanRef_);
        dict.readIfPresent("tipCorrectionFactorRef", tipCorrectionFactorRef_);

        if (bladeNumberofActuatorPoints_ > 0)
        {
            tipCorrectionFactor_ = scalarList(bladeRelativeDistToRoot_.size(), 1.0);

            forAll(bladeRelativeDistToRoot_, relDistIndex)
            {
                tipCorrectionFactor_[relDistIndex] = 
                    interpolate(bladeRelativeDistToRoot_[relDistIndex], 
                        tipCorrectionRelSpanRef_, tipCorrectionFactorRef_);

            Pout << "bladeRelativeDistToRoot :" << bladeRelativeDistToRoot_[relDistIndex] << 
                ";  tipCorrectionFactor: " << tipCorrectionFactor_[relDistIndex] << endl;
            }
        }
    }

    //Added by Clem 2024_01_21
    createOutputFile();

    Pout << endl;
    Pout << "howManyPointtoProject (" << name_ << " Rotor): " << rotorCurrentIndex << endl;
    Pout << "howManyPointtoProject (" << name_ << " Blade): " << bladeCurrentIndex << endl;
    //End added by Clem 2023_11_12

    Info << endl;
    Info << "Summary of " << name_ << ":" << endl;
    Info << endl;
    Info << "widthOfDisk: " << widthOfDisk_ << endl;
    Info << "heightOfDisk: " << heightOfDisk_ << endl;
    Info << "freeStreamDir: " << freeStreamDir_ << endl;
    Info << "lateralDirection: " << lateralDirection_ << endl;
    Info << "rotorSmearingLengthX[0]: " << rotorSmearingLengthX_[0] << endl;
    Info << "rotorSmearingLengthY[0]: " << rotorSmearingLengthY_[0] << endl;
    Info << "rotorSmearingLengthZ[0]: " << rotorSmearingLengthZ_[0] << endl;
    Info << "rotorCompensationFactor: " << rotorCompensationFactor_ << endl;
    Info << "diskDir: " << diskDir_ << endl;
    Info << "rotorForcePrescribed: " << rotorForcePrescribed_ << endl;
    Info << "rotorTPrescribed: " << rotorTPrescribed_ << endl;
    Info << "nonUniformDiskLoad: " << nonUniformDiskLoad_ << endl;
    Info << "rotorFrontalAreaPerCell[0]: " << rotorFrontalAreaPerCell_[0] << endl;
    Info << endl;
    Info << endl;
    Info << "bladeSmearingLengthX[0]: " << bladeSmearingLengthX_[0] << endl;
    Info << "bladeSmearingLengthY[0]: " << bladeSmearingLengthY_[0] << endl;
    Info << "bladeSmearingLengthZ[0]: " << bladeSmearingLengthZ_[0] << endl;
    Info << "bladeCompensationFactor: " << bladeCompensationFactor_ << endl;
    Info << "bladeDir: " << bladeDir_ << endl;
    Info << "bladeForcePrescribed: " << bladeForcePrescribed_ << endl;
    Info << "bladeTPrescribed: " << bladeTPrescribed_ << endl;
    Info << "bladeNumber: " << bladeNumber_ << endl;
    Info << "bladeChord: " << bladeChord_ << endl;
    Info << "bladeSpanPerAL: " << bladeSpanPerAL_ << endl;
    Info << "bladeCl: " << bladeCl_ << endl;
    Info << "bladeCd: " << bladeCd_ << endl;
    Info << "bladeEndEffectsActive: " << bladeEndEffectsActive_ << endl;
    Info << "bladeTwist_[0]: " << bladeTwist_[0] << " deg" << endl;

    Info << "velSamplingRadius: " << velSamplingRadius_ << endl;
    Info << "velSamplingNumber: " << velSamplingNumber_ << endl;
    //Info << "bladeEndPositions[1]: " << bladeEndPositions_[1] << endl;

    Info << "clList_[4]: " << clList_[4] << endl;
    Info << "timeStartAdjustPitch: " << timeStartAdjustPitch_ << endl;
    Info << "bladePitchTol: " << bladePitchTol_ << endl;

    scalar bladeEndPositionsCounter = 0;
    for (const vector bladeEndPositionsTemp : bladeEndPositions_)
        {
            Info << "bladeEndPositions_" + Foam::name(bladeEndPositionsCounter) 
                + ": " << bladeEndPositionsTemp << endl;
        }


    Info << endl;
    Info << endl;

}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::fv::flyingActuationDiskSource::addSup
(
    fvMatrix<vector>& eqn,
    const label fieldi
)
{
    //interpolationCellPoint<vector> is invalid inside if statement
    const volVectorField& Uin(eqn.psi());                     //Added by Clem 2024_02_06
    //Pout << "Before interpolationCellPoint<vector>" << endl;  //Added by Clem 2024_02_06
    interpolationCellPoint<vector> UInterp(Uin);             //Added by Clem 2024_02_06
    //Pout << "After interpolationCellPoint<vector>" << endl;   //Added by Clem 2024_02_06

    if (V() > VSMALL)
    {
        calc(geometricOneField(), geometricOneField(), eqn, UInterp);
    }
}


void Foam::fv::flyingActuationDiskSource::addSup
(
    const volScalarField& rho,
    fvMatrix<vector>& eqn,
    const label fieldi
)
{
    //interpolationCellPoint<vector> is invalid inside if statement
    const volVectorField& Uin(eqn.psi());                     //Added by Clem 2024_02_06
    Pout << "Before interpolationCellPoint<vector>" << endl;  //Added by Clem 2024_02_06
    interpolationCellPoint<vector> UInterp(Uin);             //Added by Clem 2024_02_06
    Pout << "After interpolationCellPoint<vector>" << endl;   //Added by Clem 2024_02_06

    if (V() > VSMALL)
    {
        calc(geometricOneField(), rho, eqn, UInterp);
    }
}


void Foam::fv::flyingActuationDiskSource::addSup
(
    const volScalarField& alpha,
    const volScalarField& rho,
    fvMatrix<vector>& eqn,
    const label fieldi
)
{
    //interpolationCellPoint<vector> is invalid inside if statement
    const volVectorField& Uin(eqn.psi());                     //Added by Clem 2024_02_06
    Pout << "Before interpolationCellPoint<vector>" << endl;  //Added by Clem 2024_02_06
    interpolationCellPoint<vector> UInterp(Uin);             //Added by Clem 2024_02_06
    Pout << "After interpolationCellPoint<vector>" << endl;   //Added by Clem 2024_02_06

    if (V() > VSMALL)
    {
        calc(alpha, rho, eqn, UInterp);
    }
}


bool Foam::fv::flyingActuationDiskSource::read(const dictionary& dict)
{
    if (cellSetOption::read(dict) && writeFile::read(dict))
    {
        dict.readIfPresent("sink", sink_);
        dict.readIfPresent("writeFileStart", writeFileStart_);
        dict.readIfPresent("writeFileEnd", writeFileEnd_);
        dict.readIfPresent("diskArea", diskArea_);
        if (diskArea_ < VSMALL)
        {
            FatalIOErrorInFunction(dict)
                << "Actuator disk has zero area: "
                << "diskArea = " << diskArea_
                << exit(FatalIOError);
        }

        dict.readIfPresent("diskDir", diskDir_);
        diskDir_.normalise();
        if (mag(diskDir_) < VSMALL)
        {
            FatalIOErrorInFunction(dict)
                << "Actuator disk surface-normal vector is zero: "
                << "diskDir = " << diskDir_
                << exit(FatalIOError);
        }


        return true;
    }

    return false;
}


// Adjusted by Clem 2024_02_08
void Foam::fv::flyingActuationDiskSource::selectingCelltoSmear(
    vector centerToSmear, label currentIndex, scalar smearingLength, label rotorOrBlade)
{
    scalar sphereRadius = smearingLength * 3;
    
    scalar smearCellCounter = 0;

    labelList selectedCelltoSmear;

    forAll(mesh_.cells(), cellI)
    {
        // Adjusted by Clem 2024_02_08
        scalar dis = mag(mesh_.C()[cellI] - centerToSmear);

        if(dis <= sphereRadius)
        {
            // if(smearCellCounter == 0){
            //     labelList selectedCelltoSmear = labelList(Foam::one, cellI);
            //     smearCellCounter += 1;
            // }else{
            //     selectedCelltoSmear.append(cellI);
            //     smearCellCounter += 1;
            // }
            selectedCelltoSmear.append(cellI);
            smearCellCounter += 1;
        }
    }

    if(rotorOrBlade == 0){
        rotorLableListtoSmear_.append(selectedCelltoSmear);
        Pout << "currentIndex (" << name_ << " rotor): " << currentIndex << ", Position at: " << centerToSmear << endl;
        Pout << "smearCellCounter during initializing (" << name_ << " rotor): " << smearCellCounter << endl;
    }

    if(rotorOrBlade == 1){
        bladeLableListtoSmear_.append(selectedCelltoSmear);
        Pout << "currentIndex (" << name_ << " blade): " << currentIndex << ", Position at: " << centerToSmear << endl;
        Pout << "smearCellCounter during initializing (" << name_ << " blade): " << smearCellCounter << endl;
    }
      
    // Pout << endl;

}


// Added by Clem 2024_01_25
void Foam::fv::flyingActuationDiskSource::bladeRelDistToTipCalc()
{
    //Adejusted by Clem 2024_02_07
    bladeRelativeDistToRoot_ = scalarList(bladeNumberofActuatorPoints_, 1.0);

    for (label ii = 0; ii < bladeNumberofActuatorPoints_; ii++) 
    {

        scalar relDistTemp = 1.0;

        for (label tt = 0; tt < bladeNumber_ * 2; tt++) 
        {
            scalar disToTip = 
                mag(bladeActuatorPointsPositions_[ii] - bladeEndPositions_[tt]) / widthOfDisk_;
            relDistTemp = min(relDistTemp, disToTip);
        }

        bladeRelativeDistToRoot_[ii] = max( (1.0 - relDistTemp), 0.001);

        Pout << "bladeRelativeDistToRoot_ at " << bladeActuatorPointsPositions_[ii]
            << " : " << bladeRelativeDistToRoot_[ii] << endl;
    }
}


// Added by Clem 2024_01_21
void Foam::fv::flyingActuationDiskSource::createOutputFile()
{
    fileName dir;

    if (Pstream::parRun())
    {
        dir = mesh_.time().path()/"../postProcessing/detailAboutBlade"
            / mesh_.time().timeName() / name_;
    }
    else
    {
        dir = mesh_.time().path()/"postProcessing/detailAboutBlade"
            / mesh_.time().timeName() / name_;
    }

    if (not isDir(dir))
    {
        mkDir(dir);
    }

    Pout << "createOutputFile(), processor: " << Pstream::myProcNo() << endl;
    Pout << "createOutputFile(), bladeNumberofActuatorPoints: " << bladeNumberofActuatorPoints_ << endl;

    if (rotorNumberofActuatorPoints_ > 0)
    {

        outputFileDisk_ = new OFstream(dir/"proc_" + Foam::name(Pstream::myProcNo()) + "_disk.csv");
        *outputFileDisk_<< "Time and then (x, y, z), (u, v, w), (fx, fy, fz):" << endl;
    }

    if (bladeNumberofActuatorPoints_ > 0)
    {

        outputFile_ = new OFstream(dir/"proc_" + Foam::name(Pstream::myProcNo()) + "_blades.csv");
        *outputFile_<< "Time and then (x, y, z), (avg_u, avg_v, avg_w), (u, v, w), (fx, fy, fz)_lift, (fx, fy, fz)_drag, tipLossFactor, bladeLocalCl, bladePitch:" << endl;

        //Pout << "Create: Master " << endl;

        //for (const label celli : bladeCells_)
        //{
        //    //Pout << "Actuator: INNN" << endl;
        //    vector cellCenterNow = mesh_.C()[celli];
        //    //Pout << "Actuator: " << name_ << ", cellCenterNow.x() is: " << cellCenterNow.x() << endl;
        //    *outputFile_ << cellCenterNow.x() << "," ;
        //}
        //*outputFile_ << endl;

        //for (const label celli : bladeCells_)
        //{
        //    vector cellCenterNow = mesh_.C()[celli];
        //    *outputFile_ << cellCenterNow.y() << "," ;
        //}
        //*outputFile_ << endl;

        //for (const label celli : bladeCells_)
        //{
        //    vector cellCenterNow = mesh_.C()[celli];
        //    *outputFile_ << cellCenterNow.z() << "," ;
        //}
        //*outputFile_ << endl;

    }

    
}


//added by Clem 2024_01_27
Foam::scalar Foam::fv::flyingActuationDiskSource::interpolate
(
    scalar xNew,
    scalarList& xOld,
    scalarList& yOld
)
{
    label index = 0;
    label indexP = 0;
    label indexM = 0;
    scalar error = 1.0E30;
    forAll(xOld, i)
    {
        scalar diff = mag(xNew - xOld[i]);
        if (diff < error)
        {
            index = i;
            error = diff;
        }
    }
    if (xNew < xOld[index])
    {
        if (index == 0)
        {
            indexP = 1;
            indexM = indexP - 1;
        }
        else
        {
            indexP = index;
            indexM = indexP - 1;
        }
        return yOld[indexM]
               + ((yOld[indexP]
               - yOld[indexM])/(xOld[indexP]
               - xOld[indexM]))*(xNew - xOld[indexM]);
    }
    else if (xNew > xOld[index])
    {
        if (index == xOld.size() - 1)
        {
            indexP = xOld.size() - 1;
            indexM = indexP - 1;
        }
        else
        {
            indexP = index + 1;
            indexM = indexP - 1;
        }
        return yOld[indexM] + ((yOld[indexP]
               - yOld[indexM])/(xOld[indexP]
               - xOld[indexM]))*(xNew - xOld[indexM]);
    }
    else if (xNew == xOld[index])
    {
        return yOld[index];
    }
    else
    {
        return 0.0;
    }
}


// ************************************************************************* //
