% Data selection 
%
% This file will filter and select the correct data segments using a
% specified method. 
%   1.  First, the force plate data will be loaded. This includes the force
%       data (COP), as double-checked by computing manually with the 
%       separate channels.
%   2.  The relevant force plate data will then be loaded onto a separate
%       struct. This struct will include the events and force plate data.
%   3.  Load and filter the IMU data with a specified cut-off frequency for
%       the low-pass filter.
%   4.  Select quiet sitting data using the specified method.
%   5.  The outcome metrics (RMS of jerk, RMS of COP velocity, RMS of
%       acceleration, and RMS of angular velocity will be computed. These
%       metrics are then compiled into a table.
%   6.  Statistical comparisons will be computed, comparing the outcome
%       metrics to the RMS of the resultant of COP velocity.

%% Define parameters

fs = 100;           % Sampling frequency (Hz)
cutOffLowPass = 10; % Cut-off frequency for low-pass filter
filtOrder = 2;      % Filter order (second-order Butterworth)
qsWindow = 30;      % Window for analysis of quiet sitting (s)
numWindows = 3;     % Number of windows to compare
nSubjs = 6;      % Number of subjects

%% Load data

for idx = 1:nSubjs
    load(['Data\TS' num2str(idx) '.mat']);
    TS(idx).age = age;  % Store the child's age
    TS(idx).FP  = FP;   % Store the force plate data
    TS(idx).IMU = IMU;  % Store the IMU data
end
clear idx age FP IMU

%% Filter data

for idx = 1:nSubjs
    % Subject-specific data
    FP = TS(idx).FP; IMU = TS(idx).IMU;
    
    % Iterate over all files
    for ifile = 1:length(FP)
        % Define filter for force plates
        fn = FP(ifile).Force.Frequency / 2; % Nyquist frequency (Hz)
        wn = cutOffLowPass / fn;            % Normalized cut-off frequency
        [B,A] = butter(filtOrder,wn);       % 2nd order Butterworth filter
        
        % Filter and store: COP
        FP(ifile).COP = filter(B, A, FP(ifile).Force.COP, [], 2)';
        
        % Define filter for IMU
        fn = fs / 2;                    % Nyquist frequency (Hz)
        wn = cutOffLowPass / fn;        % Normalized cut-off frequency
        [B,A] = butter(filtOrder,wn);   % 2nd order Butterworth filter
        
        % Filter and store: IMU
        temp = structfun(@(x) filter(B,A,x), IMU(ifile), 'UniformOutput', false);
        IMU(ifile) = temp;
    end
    
    % Store FP, IMU
    TS(idx).FP = FP; TS(idx).IMU = IMU;
end
clear FP IMU fn wn B A temp idx

%% Select quiet sitting data

% Quiet sitting selection parameters
samples = qsWindow * fs * ones(numWindows,1);

% Iterate over subjects
for idx = 1:nSubjs
    % Subject-specific data
    FP = TS(idx).FP; IMU = TS(idx).IMU; filenum = length(TS(idx).IMU);
    
    % Initialize index, dataWindow, minRMS arrays
    index = ones(numWindows, 1);        % Index of minimum RMS vel starting frame
    windowFile = ones(numWindows, 1);   % Data file in that index window
    minRMS = ones(numWindows, 1) * 30;  % Minimum RMS vel in that window (initialize high)
    
    % Iterate over number of windows
    for idx_seg = 1:numWindows
        for idx_file = 1:filenum
            % Initialize a boolean array of usable data indices
            boolArray = true(length(IMU(idx_file).vel_x),1);
            
            % Prevent overlapping
            if idx_seg > 1 && idx_file == windowFile(1) % ...in the first window
                boolArray(index(1):(index(1)+samples(1)-1)) = false;
            end
            if idx_seg > 2 && idx_file == windowFile(2) % ...in the second window
                boolArray(index(2):(index(2)+samples(2)-1)) = false;
            end
            
            % Find indices to exclude based on manual events
            excludeStart = []; excludeEnd = []; % Initialize indices to exclude
            for idx_event = 1:length(FP(idx_file).Events) % Iterate over events
                eventtemp = FP(idx_file).Events(idx_event);
                
                % Is it "not useful" & "start" or "end"?
                if contains(eventtemp.Label, "useful", 'IgnoreCase', true) && ...
                        contains(eventtemp.Label, "start", 'IgnoreCase', true)
                    excludeStart(end+1) = eventtemp.Frame;
                elseif contains(eventtemp.Label, "useful", 'IgnoreCase', true) && ...
                        contains(eventtemp.Label, "end", 'IgnoreCase', true)
                    excludeEnd(end+1) = eventtemp.Frame;
                end
            end
            
            % Remove indices from boolean array
            for idx_event = 1:length(excludeStart)
                boolArray(round(excludeStart(idx_event)):round(excludeEnd(idx_event))) = false;
            end
            
            % Compute RMS of velocity
            for idx_frame = 1:length(IMU(idx_file).vel_x) - samples(idx_seg) + 1
                if all(boolArray(idx_frame:idx_frame + samples(idx_seg) -1)) % Are all frames OK?
                    vel_x = IMU(idx_file).vel_x(idx_frame:(idx_frame+samples(idx_seg)-1)); % AP angular velocity
                    vel_z = IMU(idx_file).vel_z(idx_frame:(idx_frame+samples(idx_seg)-1)); % ML angular velocity
                    
                    vel_res = sqrt(vel_x.^2 + vel_z.^2); % Resultant
                    if rms(vel_res) < minRMS(idx_seg) % Is this a new minimum?
                        % Redefine
                        index(idx_seg) = idx_frame;
                        windowFile(idx_seg) = idx_file;
                        minRMS(idx_seg) = rms(vel_res);
                    end
                end
            end
        end
        
        % Store
        TS(idx).QS(idx_seg).IMU = structfun(@(x) x(index(idx_seg):(index(idx_seg)+samples(idx_seg)-1)), ...
            IMU(windowFile(idx_seg)), 'UniformOutput', false);
        TS(idx).QS(idx_seg).IMU.rot_GS = IMU(windowFile(idx_seg)).rot_GS(:,:,index(idx_seg):(index(idx_seg)+samples(idx_seg)-1));
        TS(idx).QS(idx_seg).COP = FP(windowFile(idx_seg)).COP(5*index(idx_seg):(5*(index(idx_seg)+samples(idx_seg))-1),:);
    end
disp(idx)
end
clear idx* samples FP IMU filenum index windowFile minRMS boolArray vel_*

%% Compute outcome metrics and compile into a table

% Begin table with age
age = [TS(:).age]';
datatable = table(age);

for idx = 1:nSubjs
    % Initialize Jerk and RMSJerk structs
    TS(idx).Jerk = struct('Res',[],'AP',[],'ML',[]);
    TS(idx).RMSJerk = struct('Res',[],'AP',[],'ML',[]);
    
    % Compute jerk: for all segments
    for idx_seg = 1:length(TS(idx).QS)
        % Compute jerk in the components and resultant
        zjerktemp   = firstDiff(TS(idx).QS(idx_seg).IMU.acc_z, 1/fs);
        xjerktemp   = firstDiff(TS(idx).QS(idx_seg).IMU.acc_x, 1/fs);
        resjerktemp = sqrt(zjerktemp.^2 + xjerktemp.^2);
        
        % Compute RMS jerk and store
        TS(idx).Jerk.Res(idx_seg,1:length(resjerktemp)) = resjerktemp;
        TS(idx).Jerk.AP(idx_seg,1:length(resjerktemp))  = zjerktemp;
        TS(idx).Jerk.ML(idx_seg,1:length(resjerktemp))  = xjerktemp;
        TS(idx).RMSJerk.Res(idx_seg,1) = rms(resjerktemp);
        TS(idx).RMSJerk.AP(idx_seg,1)  = rms(zjerktemp);
        TS(idx).RMSJerk.ML(idx_seg,1)  = rms(xjerktemp);
    end
    clear *temp
    
    % Compute COP velocity (just idx_seg = 1)
    ytemp = firstDiff(TS(idx).QS(1).COP(:,2), 1/(5*fs));
    xtemp = firstDiff(TS(idx).QS(1).COP(:,1), 1/(5*fs));
    
    % Downsample and compute resultant
    ydowntemp  = downsample(ytemp, 5); xdowntemp = downsample(xtemp, 5);
    resdowntemp = downsample(sqrt(ytemp.^2 + xtemp.^2), 5);
    
    % Compute RMS COP velocity and store
    TS(idx).RMSCOP.Res = rms(resdowntemp);
    TS(idx).RMSCOP.AP  = rms(ydowntemp);
    TS(idx).RMSCOP.ML  = rms(xdowntemp);
    clear *temp
    
    % Compute RMS linear acceleration and store (just idx_seg = 1)
    TS(idx).RMSacc.AP = rms(TS(idx).QS(1).IMU.acc_z);
    TS(idx).RMSacc.ML = rms(TS(idx).QS(1).IMU.acc_x);
    
    % Compute RMS angular velocity and store (just idx_seg = 1)
    TS(idx).RMSvel.AP = rms(TS(idx).QS(1).IMU.vel_x);
    TS(idx).RMSvel.ML = rms(TS(idx).QS(1).IMU.vel_z);
    
    % Add to table
    datatable.RMSJerkRes(idx) = TS(idx).RMSJerk.Res(1);
    datatable.RMSJerkAP(idx)  = TS(idx).RMSJerk.AP(1);
    datatable.RMSJerkML(idx)  = TS(idx).RMSJerk.ML(1);
    
    datatable.RMSJerkRes2(idx) = TS(idx).RMSJerk.Res(2);
    datatable.RMSJerkAP2(idx)  = TS(idx).RMSJerk.AP(2);
    datatable.RMSJerkML2(idx)  = TS(idx).RMSJerk.ML(2);
    datatable.RMSJerkRes3(idx) = TS(idx).RMSJerk.Res(3);
    datatable.RMSJerkAP3(idx)  = TS(idx).RMSJerk.AP(3);
    datatable.RMSJerkML3(idx)  = TS(idx).RMSJerk.ML(3);
    
    datatable.RMSCOPRes(idx) = TS(idx).RMSCOP.Res;
    
    datatable.RMSaccAP(idx)  = TS(idx).RMSacc.AP;
    datatable.RMSaccML(idx)  = TS(idx).RMSacc.ML;
    datatable.RMSvelAP(idx)  = TS(idx).RMSvel.AP;
    datatable.RMSvelML(idx)  = TS(idx).RMSvel.ML;
end
clear age idx

%% Compute correlations and t-test

% Paired t-test: RMS of jerk in AP vs. ML directions
[~,pval,~,stats] = ttest(datatable.RMSJerkAP - datatable.RMSJerkML);
disp(['RMS(j), AP vs. ML: t(' num2str(stats.df) ') = ' num2str(stats.tstat) ...
    ', p = ' num2str(pval)])

% Validation: RMS jerk vs. RMS COP velocity
[rho,pval] = corr(datatable.RMSJerkRes,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(j Res) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])
[rho,pval] = corr(datatable.RMSJerkAP,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(j AP) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])
[rho,pval] = corr(datatable.RMSJerkML,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(j ML) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])

% Validation: RMS acceleration vs. RMS COP velocity
[rho,pval] = corr(datatable.RMSaccAP,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(a AP) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])
[rho,pval] = corr(datatable.RMSaccML,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(a ML) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])

% Validation: RMS angular velocity vs. RMS COP velocity
[rho,pval] = corr(datatable.RMSvelAP,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(w AP) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])
[rho,pval] = corr(datatable.RMSvelML,datatable.RMSCOPRes,'Type','Pearson');
disp(['RMS(w ML) vs. RMS(v COP): r(4) = ' num2str(rho) ', p = ' num2str(pval)])

