#################################################################################
# Tillage-Controlled Runoff Pattern Model (TCRP model)                  	#
# Method 2                                                              	#
#										#
# Model to calculate a runoff pattern (Ldd-map), whereby water will		# 
# flow in the direction of the plough-lines.                             	#
#                                                                       	#
# The model is described in Takken, I., Jetten, V., Govers, G., 		#
# Nachtergaele, N. and Steegen, A., 2001. Geomorphology, 37, in press.  	#
#                                                                       	#
# Ingrid Takken, 17 December 2000                                       	#
#										#
# Last changes:							    		#
# - calculation of 'flowdir' changed						#
# - calculation of new output maps: TillDir2.map and angle.map			#
# - all comments translated in english						#
#																				# 
# - Adjusted for new pcraster version (4.4.0) (2022-12) Meindert Commelin		#
#################################################################################


binding

# INPUT
  Dem = dem.map;              # Digital elevation model
  Fields = fields.map;        # Nominal Input map, no fields with id 0 or 1!!
  TillDir = tilldir.map;     # Input map, scalar (directions 0-180 (-1 voor notill))
  Chan = chan.map;            # Boolean input map with channels
  LddChan = chanldd.map;      # Ldd for channels

# OUTPUT
  Edge = mapedge.map;         # The edge of the fields.map  
  LddTill = lddtill.map;      # Tillage-controlled ldd
  SlopeTill = SlopeTil.map;   # Slope in direction of flow
  FlowDirTill = flowdir.map;
  Ang = angle.map;            # The angle between tillage and aspect direction
  TillDir2 = tilldir2.map;    # Output map with tillage directions, including headlands
  LddTop = lddtop.map;        # ldd based only on topography
  thalweg = thalweg.map;

  #test = test.map;

initial

  Chan = boolean(Chan);
  dummy1 = boolean(cover(1, Dem));
  dx = celllength();
  dy = celllength();


# CALCULATION OF SLOPE, ASPECT AND LDD IN TOPOGRAPHIC DIRECTION

  Dem = lddcreatedem(Dem,1e31,1e31,1e31,1e31);
report  LddTop = lddcreate(Dem,1e31,1e31,1e31,1e31);
  Aspect = aspect(Dem);
  SlopeTop = slope(Dem);


# CALCULATION EDGES OF FIELDS AND OF TOTAL MAP-SHEET

 # Edges of the map
    Ldd2 = spatial(ldd(2));
    Ldd4 = spatial(ldd(4));
    Ldd6 = spatial(ldd(6));
    Ldd8 = spatial(ldd(8));
    down2 = boolean(downstream(Ldd2, dummy1));
    down4 = boolean(downstream(Ldd4, dummy1));
    down6 = boolean(downstream(Ldd6, dummy1));
    down8 = boolean(downstream(Ldd8, dummy1));
    rand = down2 and down4 and down6 and down8;
    rand = boolean(if(rand,0));
    rand = cover(rand,1);
    # The Fields map is reduced if the fields adjoin the map edge.
    Fields = if(rand eq 0, Fields);

 # For pitremove: make a zone on the map edge to select (and keep) pits on the edge
   mask = scalar(if(Fields ne 0, 0, 1));
   mask = cover(mask,1);
   # Calculate a map with the highest value in the middle of the fields and 
   # the lowest values along the field borders: 
    Spread = spread(nominal(mask),0,1);
    Edge = boolean(if(Spread le (4*dx), 1, 0));
    # If the Fields.map would adjoin the map edge, Edge would only get missing values here...
report Edge = boolean(if(boolean(mask) ne Edge, 1, 0));

# MC - different data types of TillDir maps

# TillDir = scalar, mv outside fields, -1 no tillages, 0-180 tillage direction
# TillDirD = directional, mv on all fields etc without tillage
   TillDirD = if(TillDir ne -1, directional(TillDir));
# TillDirS = scalar, mv on all fields etc without tillage
   TillDirS = scalar(TillDirD);


# CALCULATION OF SLOPE IN TILLAGE DIRECTION

    Ldd1 = spatial(ldd(1));
    Ldd2 = spatial(ldd(2));
    Ldd3 = spatial(ldd(3));
    Ldd4 = spatial(ldd(4));
    Ldd6 = spatial(ldd(6));
    Ldd7 = spatial(ldd(7));
    Ldd8 = spatial(ldd(8));
    Ldd9 = spatial(ldd(9));
    Dem1 = downstream(Ldd1, Dem);
    Dem2 = downstream(Ldd2, Dem);
    Dem3 = downstream(Ldd3, Dem);
    Dem4 = downstream(Ldd4, Dem);
    Dem6 = downstream(Ldd6, Dem);
    Dem7 = downstream(Ldd7, Dem);
    Dem8 = downstream(Ldd8, Dem);
    Dem9 = downstream(Ldd9, Dem);

  # dz/dx and dz/dy calculated according to Horn (1981).
    dz_dx = ((Dem1+2*Dem4+Dem7)/4 - (Dem3+2*Dem6+Dem9)/4) / (2*dx);
    dz_dy = ((Dem1+2*Dem2+Dem3)/4 - (Dem7+2*Dem8+Dem9)/4) / (2*dx);

  # Note: TillDir is positive in clockwise direction with 0 to the North,
  #       then the y-axis is the adjacent side of the angle.

   SlopeTill = scalar((dz_dx)*sin(TillDirD)+(dz_dy)*cos(TillDirD));


# CALCULATION OF DOWNSTREAM TILLAGE DIRECTION


#   TillDir1 = TillDir; # (maintains the orgignal input map)
# same as TillDirS
#   TillDir = scalar(if(scalar(TillDir) ne -1, TillDir)); # This gives missing values for 'no direction'
   
   Till = boolean(if(TillDir ne -1, 1, 0)); # this is used for the slope calculation
   TillDirDown = scalar(if(SlopeTill lt 0, TillDirS-180, TillDirS));
   TillDirDown = scalar(if(TillDirDown lt 0, TillDirDown+360 ,TillDirDown));
   SlopeTill = abs(SlopeTill);


# CALCULATION OF FIELD BORDERS AND ANGLE PERPENDICULAR TO BORDER.

  downField = downstream(Ldd1,Fields);
  grens1 = boolean(if(Fields ne downField then 1 else 0));
  grens1 = cover(grens1, 0);
  somx = if(grens1, -0.7071, 0);
  somy = if(grens1, -0.7071, 0);
  downField = downstream(Ldd2,Fields);
  grens2 = boolean(if(Fields ne downField then 1 else 0));
  grens2 = cover(grens2, 0);
  somx = if(grens2, somx, somx);
  somy = if(grens2, somy-1, somy);
  downField = downstream(Ldd3,Fields);
  grens3 = boolean(if(Fields ne downField then 1 else 0));
  grens3 = cover(grens3, 0);
  somx = if(grens3, somx+0.7071, somx);
  somy = if(grens3, somy-0.7071, somy);
  downField = downstream(Ldd4,Fields);
  grens4 = boolean(if(Fields ne downField then 1 else 0));
  grens4 = cover(grens4, 0);
  somx = if(grens4, somx-1, somx);
  somy = if(grens4, somy, somy);
  downField = downstream(Ldd6,Fields);
  grens6 = boolean(if(Fields ne downField then 1 else 0));
  grens6 = cover(grens6, 0);
  somx = if(grens6, somx+1, somx);
  somy = if(grens6, somy, somy);
  downField = downstream(Ldd7,Fields);
  grens7 = boolean(if(Fields ne downField then 1 else 0));
  grens7 = cover(grens7, 0);
  somx = if(grens7, somx-0.7071, somx);
  somy = if(grens7, somy+0.7071, somy);
  downField = downstream(Ldd8,Fields);
  grens8= boolean(if(Fields ne downField then 1 else 0));
  grens8 = cover(grens8, 0);
  somx = if(grens8, somx, somx);
  somy = if(grens8, somy+1, somy);
  downField = downstream(Ldd9,Fields);
  grens9= boolean(if(Fields ne downField then 1 else 0));
  grens9 = cover(grens9, 0);
  somx = if(grens9, somx+0.7071, somx);
  somy = if(grens9, somy+0.7071, somy);
  pgrens = grens2 or grens4 or grens6 or grens8;

# CALCULATION OF DIRECTION PERPENDICULAR TO FIELD BORDER (out of field)

 # The four directions of the axis:
  Angle = directional(if(somx eq 0 and somy gt 0, 90, -1)); 
  Angle = if(somx eq 0 and somy lt 0, 270, Angle);
  Angle = if(somy eq 0 and somx gt 0, 0, Angle);
  Angle = if(somy eq 0 and somx lt 0, 180, Angle);
 # The four quadrants:
  Angle = directional(if(somy gt 0 and somx gt 0, scalar(atan(somy/somx)), scalar(Angle)));
  Angle = directional(if(somy gt 0 and somx lt 0, 180+scalar(atan(somy/somx)), scalar(Angle)));
  Angle = directional(if(somy lt 0 and somx lt 0, 180+scalar(atan(somy/somx)), scalar(Angle)));
  Angle = directional(if(somy lt 0 and somx gt 0, 360+scalar(atan(somy/somx)), scalar(Angle)));
  Angle = if(pgrens, Angle);

 # Convert Angle to the 'tilldir co-ordinate system'.
  Angle = directional(90-scalar(Angle));
  Angle = directional(if(scalar(Angle) lt 0, scalar(Angle)+360, scalar(Angle)));


# CALCULATE LDD FOR PARCELBORDERS.

  grenzen = ordinal(if(pgrens, Fields)); # Every field gets its own border number 
  mask = scalar(if(dummy1, 9999));

   dir1 = directional(if(downstream(Ldd1, grenzen) eq grenzen,225));
   dif1 = scalar(abs(scalar(dir1)-scalar(Angle)));
   dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1));
   # 'Dif' may not be zero, this happens only where a small road is located diagonally within a field.
   dif1 = if(dif1 lt 36, 9999, dif1);
   dif1 = cover(dif1, mask); # The Angle between the possible ldd-direction and "Angle" (ea. the direction prependicular to the parcel border)
   dir2 = directional(if(downstream(Ldd2, grenzen) eq grenzen,180));
   dif2 = scalar(abs(scalar(dir2)-scalar(Angle)));
   dif2 = scalar(if(dif2 gt 180, 360-dif2, dif2)); 
   dif2 = cover(dif2, mask);
   dir3 = directional(if(downstream(Ldd3, grenzen) eq grenzen,135));
   dif3 = scalar(abs(scalar(dir3)-scalar(Angle)));
   dif3 = scalar(if(dif3 gt 180, 360-dif3, dif3)); 
   dif3 = if(dif3 lt 36, 9999, dif3);
   dif3 = cover(dif3, mask);
   dir4 = directional(if(downstream(Ldd4, grenzen) eq grenzen,270));
   dif4 = scalar(abs(scalar(dir4)-scalar(Angle)));
   dif4 = scalar(if(dif4 gt 180, 360-dif4, dif4)); 
   dif4 = cover(dif4, mask);
   dir6 = directional(if(downstream(Ldd6, grenzen) eq grenzen,90));
   dif6 = scalar(abs(scalar(dir6)-scalar(Angle)));
   dif6 = scalar(if(dif6 gt 180, 360-dif6, dif6)); 
   dif6 = cover(dif6, mask);
   dir7 = directional(if(downstream(Ldd7, grenzen) eq grenzen,315));
   dif7 = scalar(abs(scalar(dir7)-scalar(Angle)));
   dif7 = scalar(if(dif7 gt 180, 360-dif7, dif7)); 
   dif7 = if(dif7 lt 36, 9999, dif7);
   dif7 = cover(dif7, mask);
   dir8 = directional(if(downstream(Ldd8, grenzen) eq grenzen,0));
   dif8 = scalar(abs(scalar(dir8)-scalar(Angle)));
   dif8 = scalar(if(dif8 gt 180, 360-dif8, dif8)); 
   dif8 = cover(dif8, mask);
   dir9 = directional(if(downstream(Ldd9, grenzen) eq grenzen,45));
   dif9 = scalar(abs(scalar(dir9)-scalar(Angle)));
   dif9 = scalar(if(dif9 gt 180, 360-dif9, dif9)); 
   dif9 = if(dif9 lt 36, 9999, dif9);
   dif9 = cover(dif9, mask);
   minim = min(dif1, dif2, dif3, dif4, dif6, dif7, dif8, dif9);
   minim = if(pgrens, minim);
   min1 = ldd(if(dif1 eq minim, 1));
   min2 = ldd(if(dif2 eq minim, 2));
   poss1 = cover(min2, min1);
   min3 = ldd(if(dif3 eq minim, 3));
   poss1 = cover(min3, poss1);
   min4 = ldd(if(dif4 eq minim, 4));
   poss1 = cover(min4, poss1);
   min6 = ldd(if(dif6 eq minim, 6));
   poss1 = cover(min6, poss1);
   min7 = ldd(if(dif7 eq minim, 7));
   poss1 = cover(min7, poss1);
   min8 = ldd(if(dif8 eq minim, 8));
   poss1 = cover(min8, poss1);
   min9 = ldd(if(dif9 eq minim, 9));
   poss1 = cover(min9, poss1);
   minim = if(poss1 eq 1, min(dif2, dif3, dif4, dif6, dif7, dif8, dif9), 9999);
   minim = if(poss1 eq 2, min(dif1, dif3, dif4, dif6, dif7, dif8, dif9), minim);
   minim = if(poss1 eq 3, min(dif1, dif2, dif4, dif6, dif7, dif8, dif9), minim);
   minim = if(poss1 eq 4, min(dif1, dif2, dif3, dif6, dif7, dif8, dif9), minim);
   minim = if(poss1 eq 6, min(dif1, dif2, dif3, dif4, dif7, dif8, dif9), minim);
   minim = if(poss1 eq 7, min(dif1, dif2, dif3, dif4, dif6, dif8, dif9), minim);
   minim = if(poss1 eq 8, min(dif1, dif2, dif3, dif4, dif6, dif7, dif9), minim);
   minim = if(poss1 eq 9, min(dif1, dif2, dif3, dif4, dif6, dif7, dif8), minim);
   minim2 = minim;
   min1 = ldd(if(dif1 eq minim and poss1 ne 1, 1));
   min2 = ldd(if(dif2 eq minim and poss1 ne 2, 2));
   poss2 = cover(min2, min1);
   min3 = ldd(if(dif3 eq minim and poss1 ne 3, 3));
   poss2 = cover(min3, poss2);
   min4 = ldd(if(dif4 eq minim and poss1 ne 4, 4));
   poss2 = cover(min4, poss2);
   min6 = ldd(if(dif6 eq minim and poss1 ne 6, 6));
   poss2 = cover(min6, poss2);
   min7 = ldd(if(dif7 eq minim and poss1 ne 7, 7));
   poss2 = cover(min7, poss2);
   min8 = ldd(if(dif8 eq minim and poss1 ne 8, 8));
   poss2 = cover(min8, poss2);
   min9 = ldd(if(dif9 eq minim and poss1 ne 9, 9));
   poss2= cover(min9, poss2);
   # In case that there is no second possibility... 
   poss2= if(minim eq 9999, poss1, poss2);


# SEPARATION OF HEADLANDS AND FIELD BORDERS PARALLEL TO THE MAIN TILLAGE DIRECTION

# Define whether the border is along a headland or not:
   # Angle between "Angle" (direction perpendicular to the border) and "TillDir".
   dif1 = scalar(abs(scalar(directional(Angle))-scalar(TillDirDown)));
   dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1));
   evenw = boolean(if(dif1 gt 60 and dif1 lt 120, 1, 0));
   wa = boolean(if(evenw eq 1, 0, 1));


# CALCULATION OF LDD FOR PARCEL BORDERS

# Ldd for parcel borders based on the angle between the possible direction and "TillDirDown" (ea downstlope tillage direction).
   # Angle between possible flow direction 1 (poss1) and TillDirDown.
   dif1 = scalar(abs(scalar(directional(poss1))-scalar(TillDirDown)));
   dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1));
   # Angle between possible flow direction 2 (poss2) and TillDirDown.
   dif2 = scalar(abs(scalar(directional(poss2))-scalar(TillDirDown)));
   dif2 = scalar(if(dif2 gt 180, 360-dif2, dif2));
   Lddgrens = ldd(if(dif1 lt dif2, poss1, poss2));

# Ldd for headlands based on the angle between the possible flow direction and "GrensDirDown" (ea downslope border direction).
   # The direction of the parcel border is: "Angle"-90 (ea The "angle" map was the direction perp. to the border). 
   GrensDir = scalar(directional(scalar(Angle)+90));
   SlopeGrens = scalar((dz_dx)*sin(GrensDir)+(dz_dy)*cos(GrensDir));
   GrensDirDown = scalar(if(SlopeGrens lt 0, GrensDir-180, GrensDir));
   GrensDirDown = scalar(if(GrensDirDown lt 0, GrensDirDown+360 ,GrensDirDown));
   # Angle between possible flow direction 1 (poss1) and GrensDirDown. 
   dif1 = scalar(abs(scalar(directional(poss1))-scalar(GrensDirDown)));
   dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1));
   # Angle between possible flow direction 2 (poss2) and GrensDirDown. 
   dif2 = scalar(abs(scalar(directional(poss2))-scalar(GrensDirDown)));
   dif2 = scalar(if(dif2 gt 180, 360-dif2, dif2));
   Lddgrens = ldd(if(wa and dif1 lt dif2, poss1, Lddgrens));
   Lddgrens = ldd(if(wa and dif2 lt dif1, poss2, Lddgrens));

   opposite = scalar(directional(Lddgrens))-180;
   opposite = scalar(if(opposite lt 0, opposite+360, opposite));
   toestroom = boolean(if(downstream(Lddgrens,scalar(directional(Lddgrens))) eq opposite,1,0));
   toestroom = cover(toestroom, 0);
   hoger = boolean(if(downstream(Lddgrens, Dem) gt Dem,1,0));
   hoger = cover(hoger,0);
   Lddgrens = if(toestroom and hoger, 5, Lddgrens);

   # Only flow along parcel borders for tilled fields!!
   #Till = boolean(if(scalar(TillDir) ne -1, 1, 0));
   Lddgrens = if(Till, Lddgrens);

   # Required in order to find converging flow on flat parts in the DEM (no 'hoger');
   Lddgrens = lddrepair(Lddgrens);
      
   SlopeGrens = scalar((dz_dx)*sin(GrensDirDown)+(dz_dy)*cos(GrensDirDown));
   SlopeGrens = if(wa, SlopeGrens, SlopeTill);
   SlopeGrens = if(Till, SlopeGrens);

# CALCULATION OF TILLAGE LINES - SCRIPT VICTOR JETTEN

  # Correction on maps for script of Victor:
  # From: TillDir defined from 0 (north) clockwise,
  # To: 0 = horizontal and 90 is north, so anti-clockwise.
  dir_p = scalar(if(scalar(TillDir) eq -1, 0, 90-scalar(TillDir)));
  dir_p = scalar(if(dir_p lt 0, dir_p+180, dir_p));

  # take x and y vectors of tillage direction
  # take scalar because tan is "directional" type
  # avoid tan(90) degrees => solve with y direction
    directionx = scalar(tan(dir_p));
    directionx = scalar(if(dir_p == 90, 1, tan(dir_p)));
    directiony = scalar(if(dir_p == 90, 0, 1));
# report dirx = directionx;
  # cumulative x and ycoordinates
       # xcoor = xcoordinate(1);
       # ycoor = ycoordinate(1);
       # # if y increases from top to bottom...
       # ycoor = mapmaximum(ycoor)-ycoor + mapminimum(ycoor);

 # The above 4 lines replaced by the createion of a rows and columns map.
    lddcol = ldd(6);
    lddcol = lddrepair(lddcol);
    lddrow = ldd(8);
    lddrow = lddrepair(lddrow);
    columns = ordinal(accuflux(lddcol,1));
    rows = ordinal(accuflux(lddrow,1));
    xcoor = scalar(columns)*dx;
    ycoor = scalar(rows)*dx;

  # map with sloping surface in direction perpendicular to tillage direction per field
    xy = directionx*(xcoor-mapminimum(xcoor)) -
         directiony*(ycoor-mapminimum(ycoor));
    xy -= mapminimum(xy); # zelfde als: xy = xy-mapminimum(xy);

  # MAKE 3 LINES (wta, wtb, wtc) PER FIELD TO FROCE LDD's AND COMBINE.
    directionx=max(abs(directionx), 1); 
    mask = scalar(if(dummy1, 9999));
    wta = scalar(xy mod (3*dx*directionx) lt (dx*directionx));
       dir1 = directional(if(downstream(Ldd1, wta) eq 1,225));
       dif1 = scalar(abs(scalar(dir1)-scalar(TillDirDown)));
       dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1)); 
       dif1 = cover(dif1, mask); # Angle between the possible ldd-direction and the downstream tillage direction
       dir2 = directional(if(downstream(Ldd2, wta) eq 1,180));
       dif2 = scalar(abs(scalar(dir2)-scalar(TillDirDown)));
       dif2 = scalar(if(dif2 gt 180, 360-dif2, dif2)); 
       dif2 = cover(dif2, mask);
       dir3 = directional(if(downstream(Ldd3, wta) eq 1,135));
       dif3 = scalar(abs(scalar(dir3)-scalar(TillDirDown)));
       dif3 = scalar(if(dif3 gt 180, 360-dif3, dif3)); 
       dif3 = cover(dif3, mask);
       dir4 = directional(if(downstream(Ldd4, wta) eq 1,270));
       dif4 = scalar(abs(scalar(dir4)-scalar(TillDirDown)));
       dif4 = scalar(if(dif4 gt 180, 360-dif4, dif4)); 
       dif4 = cover(dif4, mask);
       dir6 = directional(if(downstream(Ldd6, wta) eq 1,90));
       dif6 = scalar(abs(scalar(dir6)-scalar(TillDirDown)));
       dif6 = scalar(if(dif6 gt 180, 360-dif6, dif6)); 
       dif6 = cover(dif6, mask);
       dir7 = directional(if(downstream(Ldd7, wta) eq 1,315));
       dif7 = scalar(abs(scalar(dir7)-scalar(TillDirDown)));
       dif7 = scalar(if(dif7 gt 180, 360-dif7, dif7)); 
       dif7 = cover(dif7, mask);
       dir8 = directional(if(downstream(Ldd8, wta) eq 1,0));
       dif8 = scalar(abs(scalar(dir8)-scalar(TillDirDown)));
       dif8 = scalar(if(dif8 gt 180, 360-dif8, dif8)); 
       dif8 = cover(dif8, mask);
       dir9 = directional(if(downstream(Ldd9, wta) eq 1,45));
       dif9 = scalar(abs(scalar(dir9)-scalar(TillDirDown)));
       dif9 = scalar(if(dif9 gt 180, 360-dif9, dif9)); 
       dif9 = cover(dif9, mask);
       minim = min(dif1, dif2, dif3, dif4, dif6, dif7, dif8, dif9);
       dir1 = directional(if(dif1 eq minim, 225));
       dir2 = directional(if(dif2 eq minim, 180));
       dirtmp = cover(dir1, dir2);
       dir3 = directional(if(dif3 eq minim, 135));
       dirtmp = cover(dir3, dirtmp);
       dir4 = directional(if(dif4 eq minim, 270));
       dirtmp = cover(dir4, dirtmp);
       dir6 = directional(if(dif6 eq minim, 90));
       dirtmp = cover(dir6, dirtmp);
       dir7 = directional(if(dif7 eq minim, 315));
       dirtmp = cover(dir7, dirtmp);
       dir8 = directional(if(dif8 eq minim, 0));
       dirtmp = cover(dir8, dirtmp);
       dir9 = directional(if(dif9 eq minim, 45));
       dirtmp = cover(dir9, dirtmp);
       hoek = abs(TillDirDown-scalar(dirtmp));
       hoek = if(hoek gt 180, 360-hoek, hoek);
       dirtmp = if(hoek gt 90, directional(TillDirDown), dirtmp);
    Ldda = ldd(if(wta eq 1, dirtmp));

    wtb = scalar((xy + dx*directionx) mod (3*dx*directionx) lt (dx*directionx));
       dir1 = directional(if(downstream(Ldd1, wtb) eq 1,225));
       dif1 = scalar(abs(scalar(dir1)-scalar(TillDirDown)));
       dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1)); 
       dif1 = cover(dif1, mask);
       dir2 = directional(if(downstream(Ldd2, wtb) eq 1,180));
       dif2 = scalar(abs(scalar(dir2)-scalar(TillDirDown)));
       dif2 = scalar(if(dif2 gt 180, 360-dif2, dif2)); 
       dif2 = cover(dif2, mask);
       dir3 = directional(if(downstream(Ldd3, wtb) eq 1,135));
       dif3 = scalar(abs(scalar(dir3)-scalar(TillDirDown)));
       dif3 = scalar(if(dif3 gt 180, 360-dif3, dif3)); 
       dif3 = cover(dif3, mask);
       dir4 = directional(if(downstream(Ldd4, wtb) eq 1,270));
       dif4 = scalar(abs(scalar(dir4)-scalar(TillDirDown)));
       dif4 = scalar(if(dif4 gt 180, 360-dif4, dif4)); 
       dif4 = cover(dif4, mask);
       dir6 = directional(if(downstream(Ldd6, wtb) eq 1,90));
       dif6 = scalar(abs(scalar(dir6)-scalar(TillDirDown)));
       dif6 = scalar(if(dif6 gt 180, 360-dif6, dif6)); 
       dif6 = cover(dif6, mask);
       dir7 = directional(if(downstream(Ldd7, wtb) eq 1,315));
       dif7 = scalar(abs(scalar(dir7)-scalar(TillDirDown)));
       dif7 = scalar(if(dif7 gt 180, 360-dif7, dif7)); 
       dif7 = cover(dif7, mask);
       dir8 = directional(if(downstream(Ldd8, wtb) eq 1,0));
       dif8 = scalar(abs(scalar(dir8)-scalar(TillDirDown)));
       dif8 = scalar(if(dif8 gt 180, 360-dif8, dif8)); 
       dif8 = cover(dif8, mask);
       dir9 = directional(if(downstream(Ldd9, wtb) eq 1,45));
       dif9 = scalar(abs(scalar(dir9)-scalar(TillDirDown)));
       dif9 = scalar(if(dif9 gt 180, 360-dif9, dif9)); 
       dif9 = cover(dif9, mask);
       minim = min(dif1, dif2, dif3, dif4, dif6, dif7, dif8, dif9);
       dir1 = directional(if(dif1 eq minim, 225));
       dir2 = directional(if(dif2 eq minim, 180));
       dirtmp = cover(dir1, dir2);
       dir3 = directional(if(dif3 eq minim, 135));
       dirtmp = cover(dir3, dirtmp);
       dir4 = directional(if(dif4 eq minim, 270));
       dirtmp = cover(dir4, dirtmp);
       dir6 = directional(if(dif6 eq minim, 90));
       dirtmp = cover(dir6, dirtmp);
       dir7 = directional(if(dif7 eq minim, 315));
       dirtmp = cover(dir7, dirtmp);
       dir8 = directional(if(dif8 eq minim, 0));
       dirtmp = cover(dir8, dirtmp);
       dir9 = directional(if(dif9 eq minim, 45));
       dirtmp = cover(dir9, dirtmp);
       hoek = abs(TillDirDown-scalar(dirtmp));
       hoek = if(hoek gt 180, 360-hoek, hoek);
       dirtmp = if(hoek gt 90, directional(TillDirDown), dirtmp);
    Lddb = ldd(if(wtb eq 1, dirtmp));

    wtc = scalar((xy + 2*dx*directionx) mod (3*dx*directionx) lt (dx*directionx));
       dir1 = directional(if(downstream(Ldd1, wtc) eq 1,225));
       dif1 = scalar(abs(scalar(dir1)-scalar(TillDirDown)));
       dif1 = scalar(if(dif1 gt 180, 360-dif1, dif1)); 
       dif1 = cover(dif1, mask);
       dir2 = directional(if(downstream(Ldd2, wtc) eq 1,180));
       dif2 = scalar(abs(scalar(dir2)-scalar(TillDirDown)));
       dif2 = scalar(if(dif2 gt 180, 360-dif2, dif2)); 
       dif2 = cover(dif2, mask);
       dir3 = directional(if(downstream(Ldd3, wtc) eq 1,135));
       dif3 = scalar(abs(scalar(dir3)-scalar(TillDirDown)));
       dif3 = scalar(if(dif3 gt 180, 360-dif3, dif3)); 
       dif3 = cover(dif3, mask);
       dir4 = directional(if(downstream(Ldd4, wtc) eq 1,270));
       dif4 = scalar(abs(scalar(dir4)-scalar(TillDirDown)));
       dif4 = scalar(if(dif4 gt 180, 360-dif4, dif4)); 
       dif4 = cover(dif4, mask);
       dir6 = directional(if(downstream(Ldd6, wtc) eq 1,90));
       dif6 = scalar(abs(scalar(dir6)-scalar(TillDirDown)));
       dif6 = scalar(if(dif6 gt 180, 360-dif6, dif6)); 
       dif6 = cover(dif6, mask);
       dir7 = directional(if(downstream(Ldd7, wtc) eq 1,315));
       dif7 = scalar(abs(scalar(dir7)-scalar(TillDirDown)));
       dif7 = scalar(if(dif7 gt 180, 360-dif7, dif7)); 
       dif7 = cover(dif7, mask);
       dir8 = directional(if(downstream(Ldd8, wtc) eq 1,0));
       dif8 = scalar(abs(scalar(dir8)-scalar(TillDirDown)));
       dif8 = scalar(if(dif8 gt 180, 360-dif8, dif8)); 
       dif8 = cover(dif8, mask);
       dir9 = directional(if(downstream(Ldd9, wtc) eq 1,45));
       dif9 = scalar(abs(scalar(dir9)-scalar(TillDirDown)));
       dif9 = scalar(if(dif9 gt 180, 360-dif9, dif9)); 
       dif9 = cover(dif9, mask);
       minim = min(dif1, dif2, dif3, dif4, dif6, dif7, dif8, dif9);
       dir1 = directional(if(dif1 eq minim, 225));
       dir2 = directional(if(dif2 eq minim, 180));
       dirtmp = cover(dir1, dir2);
       dir3 = directional(if(dif3 eq minim, 135));
       dirtmp = cover(dir3, dirtmp);
       dir4 = directional(if(dif4 eq minim, 270));
       dirtmp = cover(dir4, dirtmp);
       dir6 = directional(if(dif6 eq minim, 90));
       dirtmp = cover(dir6, dirtmp);
       dir7 = directional(if(dif7 eq minim, 315));
       dirtmp = cover(dir7, dirtmp);
       dir8 = directional(if(dif8 eq minim, 0));
       dirtmp = cover(dir8, dirtmp);
       dir9 = directional(if(dif9 eq minim, 45));
       dirtmp = cover(dir9, dirtmp);
       hoek = abs(TillDirDown-scalar(dirtmp));
       hoek = if(hoek gt 180, 360-hoek, hoek);
       dirtmp = if(hoek gt 90, directional(TillDirDown), dirtmp);
    Lddc = ldd(if(wtc eq 1, dirtmp));

# COMBINE
    LddTill = cover(cover(Ldda,Lddb),Lddc);
    # Indien de lijnen verspringen ontstaan missing values...
    wtcheck = scalar(wta)+scalar(wtb)+scalar(wtc);
    LddTill = if(wtcheck eq 0, ldd(directional(TillDirDown)), LddTill);

# FIND PITS
    opposite = scalar(directional(LddTill))-180;
    opposite = scalar(if(opposite lt 0, opposite+360, opposite));
    toestroom = boolean(if(downstream(LddTill,scalar(directional(LddTill))) eq opposite,1,0));
    toestroom = cover(toestroom, 0);
    hoger = boolean(if(downstream(LddTill, Dem) gt Dem,1,0));
    hoger = cover(hoger,0);
    LddTill = if(toestroom and hoger, 5, LddTill);
# report temp0 = LddTill;

# SUPERIMPOSE PARCEL BORDER LDD
    LddTill = cover(Lddgrens, LddTill);
# report temp1 = LddTill;

# SUPERIMPOSE LDD FOR TOPOGRAPHIC ZONES (forests, pastures)
   LddTill = if(scalar(TillDir) == -1, LddTop, LddTill);
# report temp2 = LddTill;

# SUPERIMPOSE CHANNEL DIRECTIONS
    LddTill= cover(LddChan, LddTill);

# FIND PITS AND CIRCULAR FLOW
    LddTill = lddrepair(LddTill);
    Pits = boolean(pit(LddTill));
# report temp3 = LddTill;

# IF PIT THEN TOP-DIRECTION 
    LddTill = if(Pits and not Edge, LddTop, LddTill);
    LddTill = lddrepair(LddTill);
# report temp4 = LddTill;


# SLOPE CALCULATION (1)

  SlopeTill = scalar(if(TillDir ne -1, SlopeTill, SlopeTop));
  DirChan = directional(LddChan);
  SlopeChan = scalar((dz_dx)*sin(DirChan)+(dz_dy)*cos(DirChan));
  SlopeTill = cover(SlopeChan, SlopeTill);
  SlopeTill = cover(SlopeGrens, SlopeTill);
  SlopeTill = if(Pits eq 1, SlopeTop, SlopeTill);
  LddTmp = LddTill;   # Required for SLOPE CALCULATION (2).

# MC- FlowDirTop = if(Aspect ne -1, Aspect, directional(LddTop));
# MC - the -1 with directional does not work, rewrite with same results.
  LddTopDir = directional(LddTop);
  FlowDirTop = directional(cover(scalar(Aspect), scalar(LddTopDir)));
  FlowDirTill = if(TillDir ne -1, directional(TillDirDown), FlowDirTop);
  FlowDirTill = cover(DirChan,FlowDirTill);
  GrDir = directional(if(wa, GrensDirDown, TillDirDown));
  GrDir = if(SlopeGrens gt -1, GrDir); # How can SlopeGrens be -1???
  FlowDirTill = cover(GrDir, FlowDirTill);
  FlowDirTill = if(Pits eq 1, FlowDirTop, FlowDirTill);
  # SlopeTill must thus correspond to slope in FlowDirTill.


# CORRECTION FOR CROSSING FLOW LINES.

# It is possible that flow lines cross each other diagonally.
# Locations where this happens are defined and corrections are made.
 
 # Problems of crossing flow lines can only occur where grid cells have a 
 # flow direction in the ldd-directions 1, 3, 7 or 9.

# Firstly, the locations where crossing flow occurs will be defined:
 # Detection of the problem.
 risk = boolean(if(LddTill eq 1 or LddTill eq 3 or LddTill eq 7 or LddTill eq 9 then 1 else 0));
 contr1 = ldd(directional(scalar(directional(LddTill))-45));
 contr2 = ldd(directional(scalar(directional(LddTill))+45));
 lddcontr1 = ldd(downstream(contr1,LddTill));
 lddcontr1 = cover(lddcontr1,LddTill);
 lddcontr2 = ldd(downstream(contr2,LddTill));
 lddcontr2 = cover(lddcontr2,LddTill);
 error1 = boolean(if(lddcontr1 eq ldd(directional(scalar(directional(LddTill))+90)) then 1 else 0));
 error2 = boolean(if(lddcontr2 eq ldd(directional(scalar(directional(LddTill))-90)) then 1 else 0));
 error1 = risk and error1 and not error2;
 error2 = risk and error2 and not error1;

# There are three situations that may cause these crossing flow lines:
 # A.
 # Water may not cross a small road/channel (oriented diagonally and 1 grid cell wide).
 # If such a small road is located within a field with a tillage direction perpendicular 
 # to the road, this can happen. In this case the flow direction of the field cells 
 # will be corrected. (Hereto a road or channel map is required ("Weg")).
 # B. 
 # At locations where the flow direction of thalweg cells was cahnged from tillage
 # direction to topographic doirection, crossing flow may occur if the thalweg
 # is located diagonally over the grid pattern and perpendicular to the tillage direction. 
 # C.
 # A similar situation to the roads or channels (A) may occur along parcel borders (headlands). 
 # In this case the parcel borders are considered as channels. 

# Check for the three situations of which cell the flow direction should be changed (error1 or error2).

# A. Channels
# No change in channel direction, except for the location of pits. 
 grens = boolean(if(Lddgrens ne 5,1,0));
 grens = cover(grens,0);
 Chan = boolean(if(LddChan ne 5,1,0));
 Chan = cover(Chan,0);
 Chan = Chan or grens; # All 'channels' that should not be changed...

 BuurChan1  = boolean(if(downstream(contr1,Chan) eq 1,1,0));
 BuurChan1 = boolean(if((BuurChan1 eq 1) and not Chan,1,0));
 BuurChan1 = cover(BuurChan1,0); # to avoid missing values along the catchemnt border
 BuurChan2 = boolean(if(downstream(contr2,Chan) eq 1,1,0));   
 BuurChan2 = boolean(if((BuurChan2 eq 1) and not Chan,1,0));
 BuurChan2 = cover(BuurChan2,0);
 BuurChan = BuurChan1 and BuurChan2; # Both cardinal neighbours are Channel

# B. Thalwegs 
 # Thalwegs in a field, whereby pits where replaced by topographic direction.
 
 Pits = Pits;
 BuurPit1 = downstream(contr1, Pits);
 BuurPit1 = cover(BuurPit1, 0);
 BuurPit2 = downstream(contr2, Pits);
 BuurPit2 = cover(BuurPit2, 0);
 Thalw1 = error1 and BuurPit1;
 Thalw2 = error2 and BuurPit2;
report thalweg = Thalw1 or Thalw2;

# C. Crossing flow over field borders.
 # Locations where the flow direction changes due to other land use (Chan, field border).
 downField = downstream(LddTill,Fields);
 downField = cover(downField,Fields);
 overgrens1 = boolean(if(Fields ne downField,1,0));
 downTillDir = downstream(LddTill,TillDir);
 downTillDir = cover(downTillDir,TillDir);
 overgrens2 = boolean(if(TillDir ne downTillDir,1,0));
 overgrens2 = cover(overgrens2, 0);
 overgrens = overgrens1 or overgrens2;

# Cells of which the flow direction should be changed:
 lokatie = BuurChan or thalweg or overgrens;

# Corrections:
 corr1 = boolean(if(lokatie and error1 then 1 else 0));
 corr2 = boolean(if(lokatie and error2 then 1 else 0));
 LddTemp = ldd(if(corr1 then ldd(directional(scalar(directional(LddTill))+45)) else LddTill));
 LddTemp = ldd(if(corr2 then ldd(directional(scalar(directional(LddTemp))-45)) else LddTemp));
 LddTill = (if(not Chan then LddTemp else LddTill));
# report temp6 = LddTill;


# It is now still possible that some crossing flow lines remained (but this happens seldomly).
# For example flow from field1 to field1 crossing flow from field2 to field2,
# where a small tilled field in located within another field.
# In this situation it is difficult to judge which flow direction should get priority,
# therefore both errors will be changed. 

# Check again if there are crossing flow lines (and correct):
 risk = boolean(if(LddTill eq 1 or LddTill eq 3 or LddTill eq 7 or LddTill eq 9 then 1 else 0));
 contr1 = ldd(directional(scalar(directional(LddTill))-45));
 contr2 = ldd(directional(scalar(directional(LddTill))+45));
 lddcontr1 = ldd(downstream(contr1,LddTill));
 lddcontr1 = cover(lddcontr1,LddTill);
 lddcontr2 = ldd(downstream(contr2,LddTill));
 lddcontr2 = cover(lddcontr2,LddTill);
 error1 = boolean(if(lddcontr1 eq ldd(directional(scalar(directional(LddTill))+90)) then 1 else 0));
 error2 = boolean(if(lddcontr2 eq ldd(directional(scalar(directional(LddTill))-90)) then 1 else 0));
 error1 = risk and error1 and not error2;
 error2 = risk and error2 and not error1;
 LddTemp = ldd(if(error1 then ldd(directional(scalar(directional(LddTill))+45)) else LddTill));
 LddTemp = ldd(if(error2 then ldd(directional(scalar(directional(LddTemp))-45)) else LddTemp));
 LddTill = (if(not Chan then LddTemp else LddTill));


# FIND PITS AND CIRCULAR FLOW
# MC - this does result is some 'mv' cells, by using cover with the lddtopo for the 'fields' region
# all 'mv' values are filled with lddtopo values
 lddTopoFields = ldd(scalar(LddTop) * (scalar(Fields)*0+1)); 
 LddTill = cover(LddTill, lddTopoFields);
 report LddTill = lddrepair(LddTill);


# SLOPE CALCULATION (2)

 Dif = boolean(if(LddTill ne LddTmp, 1, 0));
 Dir = directional(LddTill);
 SlopeDir = scalar((dz_dx)*sin(Dir)+(dz_dy)*cos(Dir));
 SlopeTill = scalar(if(Dif, SlopeDir, SlopeTill));
 report SlopeTill = if(SlopeTill lt 0.0010, 0.0010, SlopeTill); # voor Lisem

 report FlowDirTill = if(Dif, Dir, FlowDirTill);

# Calculate the angle between the tillage direction and the aspect direction. 
  wa = cover(wa,0);
  GrensDir = if(GrensDir ge 180, GrensDir - 180, GrensDir);
  GrensDirwa = if(wa, GrensDir);
# MC - creating directional maps with -1 for 'no directions' does not seem possible with
# data types other than ldd.
report TillDir2 = cover(directional(GrensDirwa), TillDirD);
  Ang = if(scalar(TillDir) ne -1, abs(scalar(Aspect)-scalar(TillDir2)));
  Ang = scalar(if(Ang gt 180, 360-Ang, Ang));
report Ang = scalar(if(Ang gt 90, 180-Ang, Ang));
