%% flume_sensor_analysis_v2.m

% This script can be used to analyze and plot the output for flume
% experiments of an array of Seatek transducers (accoustic sensors that
% measure the distance to the sediment bed).

% Version 1 - 07 October 2024 - All OK.
% Version 2 - 21 October 2024 - ALl OK.
% Changes in Version 2: included versions of the (x,t)-diagram that have
% more representative grid spacing in the x and t directions (grid spacing
% in t-direction is calculated as time step times celerity).
% PLEASE NOTE THAT THIS SCRIPT WORKS WELL FOR "file = 1" (SEE BELOW). FOR
% THE OTHER FILE NUMBERS, THIS SCRIPT IS NOT (YET) FULLY SUITABLE.

% Script written by: Roeland C. van de Vijsel
% Hydrology & Environmental Hydraulics Group, Wageningen University, the Netherlands

% This script belongs to the following publication:
% "Bimodality in subaqueous dune height suggests flickering behavior at high flow", published in Nature Communications (2025).
% Sjoukje I. de Lange(1,2,*), Roeland C. van de Vijsel(1), Paul J.J. F. Torfs(3), Nick P. Wallerstein(1), and A.J.F (Ton) Hoitink(1)
% 1 Wageningen University, Department of Environmental Sciences, Hydrology and Environmental Hydraulics Group, Wageningen, the Netherlands
% 2 HKV lijn in water, Lelystad, the Netherlands
% 3 independent researcher
% * Corresponding author: Sjoukje de Lange (delangesjoukje@gmail.com)

clear all
close all
clc

% OK.

%% Add paths where input data is stored and output should be saved

pathMain = 'Data_experiments\';
pathSeatek = 'Data_experiments\Seatek\';
pathStilling = 'Data_experiments\StillingWells\';
pathOutput = 'output\';

addpath(pathMain)
addpath(pathSeatek) 
addpath(pathStilling)
addpath(pathOutput)

% OK.

%% Parameter choices %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

file = 1 % Choose an input data file, as ordened in the list "datanames" below

sensor = 16; % Select one of the sensors (1:seatek.N_sensors) for all subsequent plots showing timeseries from one sensor

% OK. %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Select the input file 

% List the names of all input data files
dataname_list = {'20240710_1533 File 1', ... % first run, ran throughout the night - time stamps are lacking.
             'File2_start_6_30pm_Thursday', ... % test to see if crashed transducer work again (check)
             'File3_start_6_35pm_Thursday', ... % continuation of above. Flume crashed around 4 am --- file should give only zeros from that moment on.
             '20240712_1107', ... % omit last 5 minutes - already changed flume settings but forgot to turn off transducers. 
             '2024_7_12_17_03', ...
             '2024_7_13_12_16', ... % software crashed, no data
             '2024_7_14_12_43'};

% Define the corresponding starting datetimes
startdatetime_list = {  '2024-07-10 15:33:00.000', ...
                        '2024-07-11 18:30:00.000', ...
                        '2024-07-11 18:35:00.000', ...
                        '2024-07-12 11:07:00.000', ...
                        '2024-07-12 17:03:00.000', ...
                        '2024-07-13 12:16:00.000', ... 
                        '2024-07-14 12:43:00.000'};

% Convert the starting datetime to datetime format    
startdatetime = datetime(startdatetime_list{file}, 'Format', 'yyyy-MM-dd HH:mm:ss.SSS'); 

dataname = dataname_list{file}; % Select the corresponding file name
disp(['Selected data input file: ''', dataname,'.txt'''])

if file <= 4 % These input files are for the first experiment (lower discharge)
    
    flume.Q = 72.5; % [L/s] Discharge (discharge consists of discharge flume plus discharge sediment pump (4.5 L/s))
    flume.S0 = 0.004; % [m/m] Flume bed slope
    flume.h_weir = 0.26; % [m] Flume weir height

else % These input files are for the second experiment (higher discharge)

    flume.Q = 93.5; % [L/s] Discharge (discharge consists of discharge flume plus discharge sediment pump (4.5 L/s))
    flume.S0 = 0.005; % [m/m] Flume bed slope
    flume.h_weir = 0.225; % [m] Flume weir height

end

% OK.

%% Flume and sensor settings

seatek.N_sensors = 32; % Number of Seatek sensors used

flume.x_sensors = [1110 1080 1050	1020 990 960 930 900 870 840 810 795 780, ...
    765	750	735	720	705	690	675	660	645	630	615	600	570	540, ...
    510	480	450	420	390]/100; % [m] Distance of the sensors along the flume

seatek.range_min = 0.05; % [m] Minimum range (blanking distance) of the sensors
seatek.range_max = 0.25; % [m] Maximum range of the sensors
seatek.N_pings = 1; % [-] Number of pings over which averaging has taken place
seatek.T_sample = 0.100; % [s] Sample period
seatek.Fs = 1/seatek.T_sample; % [Hz] Sampling frequency
seatek.V_threshold = 2500; % [mV] Threshold voltage
flume.temp = 20; % [deg C] Water temperature
flume.sal = 0; % Water salinity

% OK.

%% Load the input data (time and range)

data = readtable([pathSeatek,dataname,'.txt']); % Raw input data

if file == 1 % For this file, no measurement times are given

    time = seatek.T_sample*(0:size(data,1)-1)'; % [s] Time variable, starting at 0

    data = data(:,1:seatek.N_sensors); % [cm] Range measured from sensor to sediment bed (rows: time instances; columns: sensors)
    data = table2array(data)/100; % [m] Converted to meters, and to an array

else % For the other files, the last column contains measurement times
    
    time = data(:,seatek.N_sensors+1); % [ms] Time variable
    time = table2array(time)/1000; % [s] Converted to seconds
    time = time - time(1); % Subtract first time stamp, to start at 0

    data = data(:,1:seatek.N_sensors); % [cm] Range measured from sensor to sediment bed (rows: time instances; columns: sensors)
    data = table2array(data)/100; % [m] Converted to meters, and to an array

end

% Convert times to datetime array
datetimeArray = startdatetime + seconds(time);

% OK.

%% Replace values beyond the expected range with NaNs

elmin = find(data < seatek.range_min); % Find values lower than the minimal range
elmax = find(data > seatek.range_max); % Find values larger than the maximal range
data(elmin) = NaN; % Set these values to NaN
data(elmax) = NaN; % Set these values to NaN

NaN_perc = sum(isnan(data(:)))/numel(data)*100; % Percentage of NaNs in the dataset
if NaN_perc > 1 % Define the maximal percentage of NaNs that is permitted in the dataset
    error('Too many NaNs in this dataset')
else
    disp(['Acceptable percentage of NaNs in dataset: ', num2str(NaN_perc),'%'])
end

% OK.

%% Convert from range (which increases downward) to elevation above the mean

data = nanmean(data(:)) - data; % Convert range to elevation above the mean, with mean elevation set to zero.

data_fill = fillmissing(data,'spline'); % First fill in missing values (NaNs), using spline interpolation

datamin = nanmin(data(:));
datamax = nanmax(data(:));

% OK.

%% Plot the entire dataset (bed elevation for all sensors, as a function of time)

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

x_spaces = linspace(0,1,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    ax = axes('Position', [0.028+x_spaces(ii), 0.08, 0.138, 0.90]); % Define the dimensions of each subplot

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time) 
        elmax = length(time); 
    end 
    
    time_subplot = time(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    
    caxis([datamin datamax]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    cb = colorbar;
    
    yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',get(gca,'FontSize'));
    yl.Position(1) = (yl.Position(1))*1.2; % Slightly shift the colorbar label to prevent overlap

    xlabel('Distance along flume [m]','FontSize',get(gca,'FontSize'))
    ylabel('Time [min]','FontSize',get(gca,'FontSize'))
    title([strrep(dataname,'_','\_'),' (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'.fig'])

% OK.

%% Idem, but finetune to make the final figure for the article's Supplementary Info

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

x_spaces = linspace(0,1,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    ax = axes('Position', [0.026+x_spaces(ii), 0.08, 0.135, 0.90]); % Define the dimensions of each subplot

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time) 
        elmax = length(time); 
    end 
    
    time_subplot = time(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin datamax]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    xline(flume.x_sensors(sensor),'k--','LineWidth',1)

    cb = colorbar;
    cb.FontSize = get(gca,'FontSize')+2;
    yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',get(gca,'FontSize')+2);
    yl.Position(1) = (yl.Position(1))*1.10; % Slightly shift the colorbar label to prevent overlap

    xlabel('Distance along flume [m]','FontSize',get(gca,'FontSize')+2)
    ylh = ylabel('Time [min]','FontSize',get(gca,'FontSize')+2);
    if ii == 1
        ylh.Position(1)=ylh.Position(1)+0.01;
    end

    % title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

    set(gca,'FontSize',get(gca,'FontSize')+2)
end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data.fig'])

% OK.

%% Idem, but correct the Aspect Ratio

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots
close all
figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure
fs = 12;
x_spaces = linspace(0,0.7,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    x_sp = 0.01;
    y_sp = 0.09;
    w_sp = 0.135;
    h_sp = 0.90;

    if ii <numplots
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.67, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    else 
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.69, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    end

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time) 
        elmax = length(time); 
    end 
    
    % Determine the true aspect ratio of each subplot
    LX = flume.x_sensors(1)-flume.x_sensors(end); % [m] Length in x-direction
    DT = seatek.T_sample; % Sensor sample period [s]
    celerity = LX/(20*60); % [m/s] Estimated dune celerity ( = 6mm/s)
    DY = celerity*DT; % Step size in the "y" direction (i.e. equivalent distance between each time step) (0.6mm)
    NY = length(elmin:elmax); % Number of steps in the y-direction in each subplot 
    LY = NY*DY; % equivalent length in the y-direction [m]

    time_subplot = time(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin datamax]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    xline(flume.x_sensors(sensor),'k--','LineWidth',1)

    
    pbaspect([LX LY 1])
    
    if ii == numplots
        cb = colorbar;
        cb.FontSize = fs;%get(gca,'FontSize')+2;
        yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);%get(gca,'FontSize')+2);
        % yl.Position(1) = (yl.Position(1))*1.10; % Slightly shift the colorbar label to prevent overlap
    end

    set(gca,'XTick',[])
    if ii == 3
       xlabel('Distance along flume [m]','FontSize',fs,'HorizontalAlignment','center')
       set(gca,'XTick',[flume.x_sensors(end) flume.x_sensors(sensor) flume.x_sensors(1)])
    elseif ii == 1
        ylh = ylabel('Flume run time [min]','FontSize',fs);
        % ylh.Position(1)=ylh.Position(1)+0.01;
    end

    % title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

    set(gca,'FontSize',fs)
    
end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_AR.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_AR.pdf'],'Resolution',dpi)
        % saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_AR.fig'])
exportgraphics(gcf,[pathOutput, 'FigureS3.pdf'],'Resolution',dpi)
        

% OK.

%% Apply a lowpass filter to filter out high-frequency data (spikes)

data_lowpass = -999*ones(size(data_fill)); % Prepare an array that will store the lowpass-filtered data

Fpass = 0.01; % [Hz] Passband frequency: frequencies below this threshold can pass; higher frequencies are filtered out

for i = 1:length(flume.x_sensors) % Apply the lowpass filter separately to the timeseries of each sensor

    data_lowpass(:,i) = lowpass(data_fill(:,i),Fpass,seatek.Fs);

end

% Select the data from one sensor, for plotting purposes
timeseries = data(:,sensor)*100; % [cm] Bed elevation timeseries at the selected sensor; original data
timeseries_lowpass = data_lowpass(:,sensor)*100; % [cm] Bed elevation timeseries at the selected sensor; lowpass-filtered data

% Plot the result of the lowpass filter
lw = 1; % LineWidth

% Zoomed-in regions
elzoom1 = 490;
elzoom2 = 530;
elzoom3 = 499;
elzoom4 = 504;

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

subplot(2,2,1:2) % Main figure (entire timeseries)
plot(time/60,timeseries,'LineWidth',lw)
hold on
plot(time/60,timeseries_lowpass,'LineWidth',lw)
yline(0,'LineWidth',lw)
legend('original','lowpass','Location','South')
xlabel('time [min]')
ylabel('Bed elevation relative to mean [cm]')
set(gca,'XLim',[min(time/60) max(time/60)])
xline([elzoom1, elzoom2],'--k','LineWidth',lw)
title(['sensor ', num2str(sensor),' (x = ',num2str(flume.x_sensors(sensor)),' m), lowpass filter with F_{pass} = ',num2str(Fpass),' Hz'])

subplot(2,2,3) % First zoomed-in part of the timeseries
plot(time/60,timeseries,'LineWidth',lw)
hold on
plot(time/60,timeseries_lowpass,'LineWidth',lw)
yline(0,'LineWidth',lw)
% legend('original','lowpass','Location','South')
xlabel('time [min]')
ylabel('Bed elevation relative to mean [cm]')
set(gca,'XLim',[elzoom1 elzoom2])
xline([elzoom3, elzoom4],'--k','LineWidth',lw)

subplot(2,2,4) % Second, extra zoomed-in part of the timeseries
plot(time/60,timeseries,'LineWidth',lw)
hold on
plot(time/60,timeseries_lowpass,'LineWidth',lw)
yline(0,'LineWidth',lw)
% legend('original','lowpass','Location','South')
xlabel('time [min]')
ylabel('Bed elevation relative to mean [cm]')
set(gca,'XLim',[elzoom3 elzoom4])

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_lowpass.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_lowpass.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_lowpass.fig'])

% OK.

%% Wavelet transform of the original and lowpass-filtered data (for one sensor)

% Original data
close(figure(3)); figure(3)
cwt(timeseries,seatek.Fs); % Analyze the elevation data, in [cm]
title(['Magnitude scalogram; original data (sensor ', num2str(sensor),')'])
% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_cwt.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_cwt.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_cwt.fig'])

% Lowpass-filtered data
close(figure(4)); figure(4)
cwt(timeseries_lowpass,seatek.Fs); % Analyze the elevation data, in [cm]
yline(Fpass,'r','LineWidth',lw) % Indicate the passband frequency applied in the lowpass filter
title(['Magnitude scalogram; lowpass-filtered data (sensor ', num2str(sensor),', F_{pass} = ',num2str(Fpass),' Hz)'])
% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_lowpass.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_lowpass.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_cwt_lowpass.fig'])

% OK.

%% Normalize time-averaged wavelet spectrum

% Original data
close(figure(5)); figure(5)
fb = cwtfilterbank('SignalLength',length(timeseries),'SamplingPeriod',seconds(seatek.T_sample)); % filterbank
timeSpectrum(fb,timeseries)
sgtitle(['original data, sensor ', num2str(sensor),' (x = ',num2str(flume.x_sensors(sensor)),' m)'],'FontSize',get(gca,'FontSize')+2);
% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_timeaveraged.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_timeaveraged.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_cwt_timeaveraged.fig'])

% Lowpass-filtered data
close(figure(6)); figure(6)
fb = cwtfilterbank('SignalLength',length(timeseries_lowpass),'SamplingPeriod',seconds(seatek.T_sample)); % filterbank
timeSpectrum(fb,timeseries_lowpass)
yline(1/Fpass,'r','LineWidth',lw/2) % Indicate the passband frequency (here: period) applied in the lowpass filter
sgtitle(['lowpass-filtered data, sensor ', num2str(sensor),' (x = ',num2str(flume.x_sensors(sensor)),' m), F_{pass} = ',num2str(Fpass),' Hz'],'FontSize',get(gca,'FontSize')+2);
% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_timeaveraged_lowpass.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_timeaveraged_lowpass.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_cwt_timeaveraged_lowpass.fig'])

% OK.

%% Select the minimal/low-quantile values over a moving window - use 50% the window width as moving window stride

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%% 1-second moving windows; 10th, 5th, 1st quantile %%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Prepare arrays to store the filtered data (q## = the ##th quantile, ##s = ## seconds moving windows, d## = stride length = ##% of window width)
data_q10_01s_d50 = -999*ones(size(data)); 
data_q05_01s_d50 = -999*ones(size(data)); 
data_q01_01s_d50 = -999*ones(size(data)); 

T_mov = 1.1; % [s] Time period to be covered by the moving window (add 0.1s to make sure N_mov is odd)
N_mov = round(T_mov/seatek.T_sample); % [-] Corresponding number of data points
d_mov = floor(0.50*N_mov); % Window stride (50% of window width)

ii = 0; % Counter

for i = 1:d_mov:(length(timeseries) - (N_mov - 1)) % Move the window over the timeseries of this sensor
    
    ii = ii + 1;

    % Determine the initial, end, and midpoint indices for the window
    el_ini = i;
    el_end = i + N_mov - 1;
    el_mid = (el_ini + el_end)/2;

    DATA = data(el_ini:el_end,:); % Select that part of the data set

    % Calculate quantiles (for all columns (sensors) at once, as indicated by dimension = 1)
    data_q10_01s_d50(el_mid,:) = quantile(DATA, 0.10, 1);
    data_q05_01s_d50(el_mid,:) = quantile(DATA, 0.05, 1);
    data_q01_01s_d50(el_mid,:) = quantile(DATA, 0.01, 1);
   
    % Show progress 
    clc; disp([num2str(100*ii/length(1:d_mov:(length(timeseries) - (N_mov - 1)))),'% done (1s)'])  

end

% OK

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%% 5-second moving windows; 10th, 5th, 1st quantile %%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Prepare arrays to store the filtered data (q## = the ##th quantile, ##s = ## seconds moving windows, d## = stride length = ##% of window width)
data_q10_05s_d50 = -999*ones(size(data)); 
data_q05_05s_d50 = -999*ones(size(data)); 
data_q01_05s_d50 = -999*ones(size(data)); 

T_mov = 5.1; % [s] Time period to be covered by the moving window (add 0.1s to make sure N_mov is odd)
N_mov = round(T_mov/seatek.T_sample); % [-] Corresponding number of data points
d_mov = floor(0.50*N_mov); % Window stride (50% of window width)

ii = 0; % Counter

for i = 1:d_mov:(length(timeseries) - (N_mov - 1)) % Move the window over the timeseries of this sensor

    ii = ii + 1;

    % Determine the initial, end, and midpoint indices for the window
    el_ini = i;
    el_end = i + N_mov - 1;
    el_mid = (el_ini + el_end)/2;

    DATA = data(el_ini:el_end,:); % Select that part of the data set

    % Calculate quantiles (for all columns (sensors) at once, as indicated by dimension = 1)
    data_q10_05s_d50(el_mid,:) = quantile(DATA, 0.10, 1);
    data_q05_05s_d50(el_mid,:) = quantile(DATA, 0.05, 1);
    data_q01_05s_d50(el_mid,:) = quantile(DATA, 0.01, 1);
   
    % Show progress 
    clc; disp([num2str(100*ii/length(1:d_mov:(length(timeseries) - (N_mov - 1)))),'% done (5s)']) 

end

% OK

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%% 10-second moving windows; 10th, 5th, 1st quantile %%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Prepare arrays to store the filtered data (q## = the ##th quantile, ##s = ## seconds moving windows, d## = stride length = ##% of window width)
data_q10_10s_d50 = -999*ones(size(data)); 
data_q05_10s_d50 = -999*ones(size(data)); 
data_q01_10s_d50 = -999*ones(size(data)); 

T_mov = 10.1; % [s] Time period to be covered by the moving window (add 0.1s to make sure N_mov is odd)
N_mov = round(T_mov/seatek.T_sample); % [-] Corresponding number of data points
d_mov = floor(0.50*N_mov); % Window stride (50% of window width)

ii = 0; % Counter

for i = 1:d_mov:(length(timeseries) - (N_mov - 1)) % Move the window over the timeseries of this sensor

    ii = ii + 1;

    % Determine the initial, end, and midpoint indices for the window
    el_ini = i;
    el_end = i + N_mov - 1;
    el_mid = (el_ini + el_end)/2;

    DATA = data(el_ini:el_end,:); % Select that part of the data set

    % Calculate quantiles (for all columns (sensors) at once, as indicated by dimension = 1)
    data_q10_10s_d50(el_mid,:) = quantile(DATA, 0.10, 1);
    data_q05_10s_d50(el_mid,:) = quantile(DATA, 0.05, 1);
    data_q01_10s_d50(el_mid,:) = quantile(DATA, 0.01, 1);
   
    % Show progress
    clc; disp([num2str(100*ii/length(1:d_mov:(length(timeseries) - (N_mov - 1)))),'% done (10s)']) 

end

% OK

%% Filter out the missing data (value = -999)

% 01s
el_choose_01s = (data_q10_01s_d50(:,1)~=-999); % Choose the meaningful elements in the data series (those elements where the data is not equal to -999)
data_q10_01s_d50 = data_q10_01s_d50(el_choose_01s,:); % Filter out the non-meaningful elements from the data arrays
data_q05_01s_d50 = data_q05_01s_d50(el_choose_01s,:);
data_q01_01s_d50 = data_q01_01s_d50(el_choose_01s,:);
datetime_01s_d50 = datetimeArray(el_choose_01s); % Also only select the meaningful elements from the time vector
    time_01s_d50 = time(el_choose_01s); % Idem, but for the numerical time vector

% 05s
el_choose_05s = (data_q10_05s_d50(:,1)~=-999); % Choose the meaningful elements in the data series (those elements where the data is not equal to -999)
data_q10_05s_d50 = data_q10_05s_d50(el_choose_05s,:); % Filter out the non-meaningful elements from the data arrays
data_q05_05s_d50 = data_q05_05s_d50(el_choose_05s,:);
data_q01_05s_d50 = data_q01_05s_d50(el_choose_05s,:);
datetime_05s_d50 = datetimeArray(el_choose_05s); % Also only select the meaningful elements from the time vector
    time_05s_d50 = time(el_choose_05s); % Idem, but for the numerical time vector

% 10s
el_choose_10s = (data_q10_10s_d50(:,1)~=-999); % Choose the meaningful elements in the data series (those elements where the data is not equal to -999)
data_q10_10s_d50 = data_q10_10s_d50(el_choose_10s,:); % Filter out the non-meaningful elements from the data arrays
data_q05_10s_d50 = data_q05_10s_d50(el_choose_10s,:);
data_q01_10s_d50 = data_q01_10s_d50(el_choose_10s,:);
datetime_10s_d50 = datetimeArray(el_choose_10s); % Also only select the meaningful elements from the time vector
    time_10s_d50 = time(el_choose_10s); % Idem, but for the numerical time vector

% OK.


%% Plot the timeseries at one sensor, and zoom in, to show the filter method (FOR THE SUPPLEMENTARY FIGURE)

plot_timeseries = (data(:,sensor) - mean(data(:,sensor)))*100; % Subtract total mean for this sensor, and convert from m to cm
plot_timeseries_filtered = (data_q05_05s_d50(:,sensor) - mean(data(:,sensor)))*100; % Idem, but filtered data

% Plot the result of the lowpass filter
lw2 = 1.5; % LineWidth
lw1 = 1;
fs = 11;

% Zoomed-in regions
elzoom1 = 490;
elzoom2 = 530;
elzoom3 = 499;
elzoom4 = 504;
elzoom5 = 500;
elzoom6 = 501;

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

ylims = [-10 8];

subplot(2,3,1:3) % Main figure (entire timeseries);
xlims=[min(time/60) max(time/60)];
hold on
pl1=plot(time/60,plot_timeseries,'.','LineWidth',lw2);
pl2=plot(time_05s_d50/60,plot_timeseries_filtered,'LineWidth',lw2);
pl3=plot([elzoom1, elzoom1],ylims,'--k','LineWidth',lw1);
pl4=plot([elzoom2, elzoom2],ylims,'--k','LineWidth',lw1);
pl5=plot([min(time/60) max(time/60)],[0 0],'k','LineWidth',lw1);
xlabel('Time [min]','FontSize',fs)
ylabel('Bed elevation relative to mean [cm]','FontSize',fs)
set(gca,'XLim',xlims,'YLim',ylims,'FontSize',fs)
legend([pl1,pl2,pl3],'original','filtered','zoom bounds','Location','South','FontSize',fs)
text(-diff(xlims)/32,ylims(2),'a)','FontSize',fs,'FontWeight','bold')

subplot(2,3,4) % First zoomed-in part of the timeseries
xlims=[elzoom1 elzoom2];
hold on
pl1=plot(time/60,plot_timeseries,'.','LineWidth',lw2);
pl2=plot(time_05s_d50/60,plot_timeseries_filtered,'LineWidth',lw2);
pl3=plot([elzoom3, elzoom3],ylims,'--k','LineWidth',lw1);
pl4=plot([elzoom4, elzoom4],ylims,'--k','LineWidth',lw1);
pl5=plot([min(time/60) max(time/60)],[0 0],'k','LineWidth',lw1);
% legend('original','filtered','Location','South')
xlabel('Time [min]','FontSize',fs)
ylabel('Bed elevation relative to mean [cm]','FontSize',fs)
set(gca,'XLim',xlims,'YLim',ylims,'FontSize',fs)
text(xlims(1)-diff(xlims)/8.5,ylims(2),'b)','FontSize',fs,'FontWeight','bold')

subplot(2,3,5) % Second, extra zoomed-in part of the timeseries
xlims=[elzoom3 elzoom4];
hold on
pl1=plot(time/60,plot_timeseries,'.','LineWidth',lw2);
pl2=plot(time_05s_d50/60,plot_timeseries_filtered,'LineWidth',lw2);
pl3=plot([elzoom5, elzoom5],ylims,'--k','LineWidth',lw1);
pl4=plot([elzoom6, elzoom6],ylims,'--k','LineWidth',lw1);
pl5=plot([min(time/60) max(time/60)],[0 0],'k','LineWidth',lw1);
% legend('original','filtered','Location','South')
xlabel('Time [min]','FontSize',fs)
ylabel('Bed elevation relative to mean [cm]','FontSize',fs)
set(gca,'XLim',xlims,'YLim',ylims,'FontSize',fs)
text(xlims(1)-diff(xlims)/8.5,ylims(2),'c)','FontSize',fs,'FontWeight','bold')

subplot(2,3,6) % Second, extra zoomed-in part of the timeseries
xlims=[elzoom5 elzoom6];
hold on
pl1=plot(time/60,plot_timeseries,'.','LineWidth',lw2);
pl2=plot(time_05s_d50/60,plot_timeseries_filtered,'-o','LineWidth',lw2);
pl3=plot([elzoom3, elzoom3],ylims,'--k','LineWidth',lw1);
pl4=plot([elzoom4, elzoom4],ylims,'--k','LineWidth',lw1);
pl5=plot([min(time/60) max(time/60)],[0 0],'k','LineWidth',lw1);
legend('original','filtered','Location','SouthEast')
xlabel('Time [min]','FontSize',fs)
ylabel('Bed elevation relative to mean [cm]','FontSize',fs)
set(gca,'XLim',xlims,'YLim',ylims,'FontSize',fs)
text(xlims(1)-diff(xlims)/8.5,ylims(2),'d)','FontSize',fs,'FontWeight','bold')

dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_timeseries_filter_method.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_timeseries_filter_method.pdf'],'Resolution',dpi)
        % saveas(gcf,[pathOutput, 'SI_Figure_XX_timeseries_filter_method.fig'])
exportgraphics(gcf,[pathOutput, 'FigureS4.pdf'],'Resolution',dpi)

% OK.

clear plot_timeseries_filtered plot_timeseries

%% Plot all the different quantile-based, filtered outputs

close(figure(7)); figure(7)
set(gcf,'units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

tiledlayout(1,10, 'Padding', 'none', 'TileSpacing', 'compact'); 

nexttile
ax(1)=imagesc(data);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('original')

nexttile
ax(2)=imagesc(data_q10_01s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q10\_01s\_d50')
nexttile

ax(3)=imagesc(data_q10_05s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q10\_05s\_d50')

nexttile
ax(4)=imagesc(data_q10_10s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q10\_10s\_d50')

nexttile
ax(5)=imagesc(data_q05_01s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q05\_01s\_d50')

nexttile
ax(6)=imagesc(data_q05_05s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q05\_05s\_d50')

nexttile
ax(7)=imagesc(data_q05_10s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q05\_10s\_d50')

nexttile
ax(8)=imagesc(data_q01_01s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q01\_01s\_d50')

nexttile
ax(9)=imagesc(data_q01_05s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q01\_05s\_d50')

nexttile
ax(10)=imagesc(data_q01_10s_d50);
% colorbar
caxis([datamin datamax])
set(gca,'XTick',[],'YTick',[])
title('q01\_10s\_d50')

cmap = cmocean('delta','pivot',0);
colormap(cmap)

% exportgraphics(gcf,[pathOutput, dataname,'_quantiles_all.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_quantiles_all.pdf'],'Resolution',dpi) 
%         saveas(gcf,[pathOutput, dataname,'_quantiles_all.fig']) 

% OK.

%% Plot all the different quantile-based, filtered outputs

close(figure(8)); figure(8)
set(gcf,'units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

tiledlayout(1,3, 'Padding', 'none', 'TileSpacing', 'compact'); 

% 10% quantiles, stride is 50% the window size
nexttile
set(gca,'DefaultLineLineWidth',lw)
hold on
plot(datetimeArray,data(:,sensor));
plot(datetime_01s_d50,data_q10_01s_d50(:,sensor));
plot(datetime_05s_d50,data_q10_05s_d50(:,sensor));
plot(datetime_10s_d50,data_q10_10s_d50(:,sensor));
set(gca,'YLim',[datamin datamax],'XLim',[datetimeArray(1) datetimeArray(end)])
ax(1) = gca;
legend('original','q10\_01s\_d50','q10\_05s\_d50','q10\_10s\_d50','Location','South')
title('10% quantiles - stride 50%')

% 5% quantiles, stride is 50% the window size
nexttile
set(gca,'DefaultLineLineWidth',lw)
hold on
plot(datetimeArray,data(:,sensor));
plot(datetime_01s_d50,data_q05_01s_d50(:,sensor));
plot(datetime_05s_d50,data_q05_05s_d50(:,sensor));
plot(datetime_10s_d50,data_q05_10s_d50(:,sensor));
set(gca,'YLim',[datamin datamax],'XLim',[datetimeArray(1) datetimeArray(end)])
ax(2) = gca;
legend('original','q05\_01s\_d50','q05\_05s\_d50','q05\_10s\_d50','Location','South')
title('5% quantiles - stride 50%')

% 1% quantiles, stride is 50% the window size
nexttile
set(gca,'DefaultLineLineWidth',lw)
hold on
plot(datetimeArray,data(:,sensor));
plot(datetime_01s_d50,data_q01_01s_d50(:,sensor));
plot(datetime_05s_d50,data_q01_05s_d50(:,sensor));
plot(datetime_10s_d50,data_q01_10s_d50(:,sensor));
set(gca,'YLim',[datamin datamax],'XLim',[datetimeArray(1) datetimeArray(end)])
ax(3) = gca;
legend('original','q01\_01s\_d50','q01\_05s\_d50','q01\_10s\_d50','Location','South')
title('1% quantiles - stride 50%')

linkaxes(ax,'xy')

% exportgraphics(gcf,[pathOutput, dataname,'_filtermethods_all.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_filtermethods_all.pdf'],'Resolution',dpi) 
%         saveas(gcf,[pathOutput, dataname,'_filtermethods_all.fig']) 

%% Subtract the means, such that each filtered dataset is averaged around 0

data_q10_01s_d50 = data_q10_01s_d50 - nanmean(data_q10_01s_d50(:));
data_q05_01s_d50 = data_q05_01s_d50 - nanmean(data_q05_01s_d50(:));
data_q01_01s_d50 = data_q01_01s_d50 - nanmean(data_q01_01s_d50(:));

data_q10_05s_d50 = data_q10_05s_d50 - nanmean(data_q10_05s_d50(:));
data_q05_05s_d50 = data_q05_05s_d50 - nanmean(data_q05_05s_d50(:));
data_q01_05s_d50 = data_q01_05s_d50 - nanmean(data_q01_05s_d50(:));

data_q10_10s_d50 = data_q10_10s_d50 - nanmean(data_q10_10s_d50(:));
data_q05_10s_d50 = data_q05_10s_d50 - nanmean(data_q05_10s_d50(:));
data_q01_10s_d50 = data_q01_10s_d50 - nanmean(data_q01_10s_d50(:));

% OK.

%% Plot the entire dataset (bed elevation for all sensors, as a function of time), for the filtered data (q05_05s_d50)

datamin_q05_05s_d50 = min(data_q05_05s_d50(:));
datamax_q05_05s_d50 = max(data_q05_05s_d50(:));

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

x_spaces = linspace(0,1,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    ax = axes('Position', [0.028+x_spaces(ii), 0.08, 0.138, 0.90]); % Define the dimensions of each subplot

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time_05s_d50)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time_05s_d50)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time_05s_d50) 
        elmax = length(time_05s_d50); 
    end 
    
    time_subplot = time_05s_d50(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data_q05_05s_d50(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin_q05_05s_d50 datamax_q05_05s_d50]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    cb = colorbar;
    
    yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',get(gca,'FontSize'));
    yl.Position(1) = (yl.Position(1))*1.2; % Slightly shift the colorbar label to prevent overlap

    xlabel('Distance along flume [m]','FontSize',get(gca,'FontSize'))
    ylabel('Time [min]','FontSize',get(gca,'FontSize'))
    title([strrep(dataname,'_','\_'),' q05\_05s\_d50 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,' q05_05s_d50.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,' q05_05s_d50.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,' q05_05s_d50.fig'])

% OK.

%% Save the final (filtered) data clearly

% Final time vector (filtered according to 5% quantiles, 5 second sliding windows with 50% window stride). Time step size is 2.5 sec.
time_final = time_05s_d50;

% Final data (filtered according to 5% quantiles, 5 second sliding windows with 50% window stride). The average of the ENTIRE dataset is set to zero.
data_final = data_q05_05s_d50;

% Idem, but now make sure the average of each SENSOR is also zero (so not only the average of the entire dataset).
data_final0 = data_final; 
for i = 1:seatek.N_sensors 
    data_final0(:,i) = data_final(:,i) - nanmean(data_final(:,i));
end

% OK.

%% Plot the entire dataset (bed elevation for all sensors, as a function of time), for the dataset "data_final0"

datamin_final0 = min(data_final0(:));
datamax_final0 = max(data_final0(:));

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

x_spaces = linspace(0,1,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    ax = axes('Position', [0.028+x_spaces(ii), 0.08, 0.138, 0.90]); % Define the dimensions of each subplot

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time_final)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time_final)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time_final) 
        elmax = length(time_final); 
    end 
    
    time_subplot = time_final(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data_final0(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin_final0 datamax_final0]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    cb = colorbar;
    
    yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',get(gca,'FontSize'));
    yl.Position(1) = (yl.Position(1))*1.2; % Slightly shift the colorbar label to prevent overlap

    xlabel('Distance along flume [m]','FontSize',get(gca,'FontSize'))
    ylabel('Time [min]','FontSize',get(gca,'FontSize'))
    title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,' final0.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,' final0.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,' final0.fig'])

% OK.

%% Idem, but finetune to make the final figure for the article's Supplementary Info

datamin_final0 = min(data_final0(:));
datamax_final0 = max(data_final0(:));

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

x_spaces = linspace(0,1,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    ax = axes('Position', [0.026+x_spaces(ii), 0.08, 0.135, 0.90]); % Define the dimensions of each subplot

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time_final)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time_final)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time_final) 
        elmax = length(time_final); 
    end 
    
    time_subplot = time_final(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data_final0(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin_final0 datamax_final0]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    xline(flume.x_sensors(sensor),'k--','LineWidth',1)

    cb = colorbar;
    cb.FontSize = get(gca,'FontSize')+2;
    yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',get(gca,'FontSize')+2);
    yl.Position(1) = (yl.Position(1))*1.10; % Slightly shift the colorbar label to prevent overlap

    xlabel('Distance along flume [m]','FontSize',get(gca,'FontSize')+2)
    ylh = ylabel('Time [min]','FontSize',get(gca,'FontSize')+2);
    if ii == 1
        ylh.Position(1)=ylh.Position(1)+0.01;
    end

    % title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

    set(gca,'FontSize',get(gca,'FontSize')+2)
end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered.fig'])

% OK.

%% Idem, but correct the Aspect Ratio

datamin_final0 = min(data_final0(:));
datamax_final0 = max(data_final0(:));

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots
close all
figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure
fs = 12;
x_spaces = linspace(0,0.7,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    x_sp = 0.01;
    y_sp = 0.09;
    w_sp = 0.135;
    h_sp = 0.90;

    if ii <numplots
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.67, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    else 
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.69, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    end

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time_final)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time_final)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time_final) 
        elmax = length(time_final); 
    end 
    
    % Determine the true aspect ratio of each subplot
    LX = flume.x_sensors(1)-flume.x_sensors(end); % [m] Length in x-direction
    DT = median(diff(time_final));%seatek.T_sample; % Sensor sample period [s]
    celerity = LX/(20*60); % [m/s] Estimated dune celerity ( = 6mm/s)
    DY = celerity*DT; % Step size in the "y" direction (i.e. equivalent distance between each time step) (0.6mm)
    NY = length(elmin:elmax); % Number of steps in the y-direction in each subplot 
    LY = NY*DY; % equivalent length in the y-direction [m]

    time_subplot = time_final(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    
    data_subplot = data_final0(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors,time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes

    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin_final0 datamax_final0]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    xline(flume.x_sensors(sensor),'k--','LineWidth',1)

    
    pbaspect([LX LY 1])
    
    if ii == numplots
        cb = colorbar;
        cb.FontSize = fs;%get(gca,'FontSize')+2;
        yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);%get(gca,'FontSize')+2);
        % yl.Position(1) = (yl.Position(1))*1.10; % Slightly shift the colorbar label to prevent overlap
    end

    set(gca,'XTick',[])
    if ii == 3
       xlabel('Distance along flume [m]','FontSize',fs,'HorizontalAlignment','center')
       set(gca,'XTick',[flume.x_sensors(end) flume.x_sensors(sensor) flume.x_sensors(1)])
    elseif ii == 1
        ylh = ylabel('Flume run time [min]','FontSize',fs);
        % ylh.Position(1)=ylh.Position(1)+0.01;
    end

    % title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

    set(gca,'FontSize',fs)
    
end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR.pdf'],'Resolution',dpi)
        % saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR.fig'])
exportgraphics(gcf,[pathOutput, 'FigureS5.pdf'],'Resolution',dpi)

% OK.

%% Idem, but make sure that DX is equidistant and that the DT-equivalent-DY is the same as DX

datamin_final0 = min(data_final0(:));
datamax_final0 = max(data_final0(:));

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots
close all
figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure
fs = 12;
x_spaces = linspace(0,0.7,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    x_sp = 0.01;
    y_sp = 0.09;
    w_sp = 0.135;
    h_sp = 0.90;

    if ii <numplots
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.67, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    else 
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.69, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    end

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time_final)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time_final)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time_final) 
        elmax = length(time_final); 
    end 
    
    % Determine the true aspect ratio of each subplot
    LX = flume.x_sensors(1)-flume.x_sensors(end); % [m] Length in x-direction
    DT = median(diff(time_final));%seatek.T_sample; % Sensor sample period [s]
    celerity = LX/(20*60); % [m/s] Estimated dune celerity ( = 6mm/s)
    DY = celerity*DT; % Step size in the "y" direction (i.e. equivalent distance between each time step) (0.6mm)
    NY = length(elmin:elmax); % Number of steps in the y-direction in each subplot 
    LY = NY*DY; % equivalent length in the y-direction [m]

    % Resample, such that x-grid is equidistant and the t-grid has the same
    % step size, if t is converted to distance y via the celerity
    x_sensors_sel = [1:11,13:2:25,26:32]; % The indices of the sensors to select, such that x-grid is equidistant
    DX = median(-diff(flume.x_sensors(x_sensors_sel))); % Grid spacing [m] in x-direction
    Nstep = round(DX/DY); % Step size at which to resample the t-axis, such that DY = DX

    time_subplot = time_final(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    time_subplot = time_subplot(1:Nstep:end); % Resample, such that the DT-equivalent-DY is equal to DX

    data_subplot = data_final0(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    data_subplot = data_subplot(1:Nstep:end,x_sensors_sel); % Resample, such that the DT-equivalent-DY is equal to DX, and the x-grid is equidistant with spacing DX
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors(x_sensors_sel),time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes
    
    surf(x_grid,t_grid,data_subplot,'EdgeColor','none'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin_final0 datamax_final0]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    xline(flume.x_sensors(sensor),'k--','LineWidth',1)

    
    pbaspect([LX LY 1])
    
    if ii == numplots
        cb = colorbar;
        cb.FontSize = fs;%get(gca,'FontSize')+2;
        yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);%get(gca,'FontSize')+2);
        % yl.Position(1) = (yl.Position(1))*1.10; % Slightly shift the colorbar label to prevent overlap
    end

    set(gca,'XTick',[])
    if ii == 3
       xlabel('Distance along flume [m]','FontSize',fs,'HorizontalAlignment','center')
       set(gca,'XTick',[flume.x_sensors(end) flume.x_sensors(sensor) flume.x_sensors(1)])
    elseif ii == 1
        ylh = ylabel('Flume run time [min]','FontSize',fs);
        % ylh.Position(1)=ylh.Position(1)+0.01;
    end

    % title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

    set(gca,'FontSize',fs)
    
end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR_equidistant.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR_equidistant.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR_equidistant.fig'])

% OK.

%% Idem, but show interpolated surf-plot

datamin_final0 = min(data_final0(:));
datamax_final0 = max(data_final0(:));

numplots = 6; % Number of subplots; in case of a long timeseries, split it up into this number ofsubplots
close all
figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure
fs = 12;
x_spaces = linspace(0,0.7,numplots+1); % Horizontal positions to start each subplot from

for ii = 1:numplots

    x_sp = 0.01;
    y_sp = 0.09;
    w_sp = 0.135;
    h_sp = 0.90;

    if ii <numplots
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.67, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    else 
        ax = axes('Position', [x_sp+(ii-1)*w_sp*0.69, y_sp, w_sp, h_sp]); % Define the dimensions of each subplot
    end

    % Split up the time-axis into equal parts, to be plotted in each subplot
    elmin = (ii-1)*floor(length(time_final)/numplots) + 1; % The starting element of this part of the time-axis (rounding down to lower integer)
    elmax = ii*ceil(length(time_final)/numplots); % The final element of this part of the time-axis (rounding up to higher integer)
    
    % Prevent an index exceeding the length of the time axis
    if elmax > length(time_final) 
        elmax = length(time_final); 
    end 
    
    % Determine the true aspect ratio of each subplot
    LX = flume.x_sensors(1)-flume.x_sensors(end); % [m] Length in x-direction
    DT = median(diff(time_final));%seatek.T_sample; % Sensor sample period [s]
    celerity = LX/(20*60); % [m/s] Estimated dune celerity ( = 6mm/s)
    DY = celerity*DT; % Step size in the "y" direction (i.e. equivalent distance between each time step) (0.6mm)
    NY = length(elmin:elmax); % Number of steps in the y-direction in each subplot 
    LY = NY*DY; % equivalent length in the y-direction [m]

    % Resample, such that x-grid is equidistant and the t-grid has the same
    % step size, if t is converted to distance y via the celerity
    x_sensors_sel = [1:11,13:2:25,26:32]; % The indices of the sensors to select, such that x-grid is equidistant
    DX = median(-diff(flume.x_sensors(x_sensors_sel))); % Grid spacing [m] in x-direction
    Nstep = round(DX/DY); % Step size at which to resample the t-axis, such that DY = DX

    time_subplot = time_final(elmin:elmax)/60; % [min] The corresponding part of the time-axis to be shown along this subplot
    time_subplot = time_subplot(1:Nstep:end); % Resample, such that the DT-equivalent-DY is equal to DX

    data_subplot = data_final0(elmin:elmax,:)*100; % [cm] The corresponding selection of the full dataset
    data_subplot = data_subplot(1:Nstep:end,x_sensors_sel); % Resample, such that the DT-equivalent-DY is equal to DX, and the x-grid is equidistant with spacing DX
    
    [x_grid,t_grid] = meshgrid(flume.x_sensors(x_sensors_sel),time_subplot); % Create a spatial [m] and temporal [min] grid, for plotting purposes
    
    surf(x_grid,t_grid,data_subplot,'EdgeColor','none','FaceColor','interp'); view([0 90]) % Show the data, taking into account the non-equidistant spacing between sensors
    caxis([datamin_final0 datamax_final0]*100) % Color axis range for the bed elevation measurements [cm]
    set(gca,'YDir','Normal') % Make sure the time-axis increases upwards
    axis([min(flume.x_sensors) max(flume.x_sensors) min(time_subplot) max(time_subplot)]) % Distance [m] on x-axis, time [min] on y-axis
    
    xline(flume.x_sensors(sensor),'k--','LineWidth',1)

    
    pbaspect([LX LY 1])
    
    if ii == numplots
        cb = colorbar;
        cb.FontSize = fs;%get(gca,'FontSize')+2;
        yl = ylabel(cb,'Elevation relative to mean [cm]','Rotation',270,'FontSize',fs);%get(gca,'FontSize')+2);
        % yl.Position(1) = (yl.Position(1))*1.10; % Slightly shift the colorbar label to prevent overlap
    end

    set(gca,'XTick',[])
    if ii == 3
       xlabel('Distance along flume [m]','FontSize',fs,'HorizontalAlignment','center')
       set(gca,'XTick',[flume.x_sensors(end) flume.x_sensors(sensor) flume.x_sensors(1)])
    elseif ii == 1
        ylh = ylabel('Flume run time [min]','FontSize',fs);
        % ylh.Position(1)=ylh.Position(1)+0.01;
    end

    % title([strrep(dataname,'_','\_'),' final0 (',num2str(ii),' of ',num2str(numplots),')'],'FontSize',get(gca,'FontSize'))

    set(gca,'FontSize',fs)
    
end

cmap = cmocean('delta','pivot',0); % Choose a color scheme
colormap(cmap);

dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR_equidistant_interp.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR_equidistant_interp.pdf'],'Resolution',dpi)
        % saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_data_filtered_AR_equidistant_interp.fig'])
exportgraphics(gcf,[pathOutput, 'Figure4.pdf'],'Resolution',dpi)

% OK.

%% Delete unnecessary variables, to save space

clearvars -except data_final data_final0 dataname file flume seatek path* sensor time_final

% OK.

%% Do a wavelet transformation

samplingperiod = seconds(2.5); % Define the sampling timeperiod
filterbank = cwtfilterbank('SignalLength',length(time_final),'SamplingPeriod',samplingperiod); % filterbank

%% Make a time-integrated wavelet transform of this final filtered data
figure

[tavgp,period] = timeSpectrum(filterbank,data_final0(:,sensor));
timeSpectrum(filterbank,data_final0(:,sensor))
sgtitle(['filtered data (final0), sensor ', num2str(sensor),' (x = ',num2str(flume.x_sensors(sensor)),' m)'],'FontSize',get(gca,'FontSize')+2);
% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_timeaveraged_final0.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_timeaveraged_final0.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_cwt_timeaveraged_final0.fig'])

% OK.

%% Find the peaks in the time-integrated wavelet spectrum

[tavgp_pks tavgp_locs] = findpeaks(tavgp);

figure
semilogx(seconds(period),tavgp)
hold on
semilogx(seconds(period(tavgp_locs)),tavgp_pks,'o')

% OK.

%% Make a normal wavelet transform of this final filtered data
figure

[wt,period,coi] = cwt(data_final0(:,sensor),samplingperiod); % Wavelet
cwt(data_final0(:,sensor),samplingperiod); % Plot the wavelet

% OK.

%% Determine the location of the wavelet spectrum, for each timestep

cwt_max_periods = -999*ones(size(time_final)); % Create an array to store the periods belonging to the maxima

for i = 1:length(time_final) % loop over each timestep

    [~,I] = max(wt(:,i)); % Find the location of the maximum, for this timestep
    cwt_max_periods(i) = seconds(period(I)); % Find and store the corresponding timeperiod, in seconds

end

% OK.

%% Plot the wavelet spectrum, the locations of the maxima, and the time-integrated wavelet spectrum

lw = 1; % LineWidth
col1 = [0 0.4470 0.7410]; % blue
col2 = [0.8500 0.3250 0.0980]; % orange
col3 = [0.9290 0.6940 0.1250]; % yellow
col4 = [0.4940 0.1840 0.5560]; % purple
col6 = [0.3010 0.7450 0.9330]; % light blue

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

ax(1) = subplot(1,3,1:2); % Left subplot
imagesc(time_final,seconds(period),abs(wt)) % plot the wavelet spectrum
set(gca,'YScale','log','YDir','Normal')
colorbar
hold on
semilogy(time_final,cwt_max_periods,'.','Color',col3,'LineWidth',lw) % Show the locations of the spectral maxima
patch([time_final' fliplr(time_final')],[seconds(coi)' max(ylim)*ones(size(seconds(coi)))'],'k','FaceAlpha', 0.5,'EdgeColor','none') % mask out the cone of influence
cmap = flipud(cmocean('ice')); % Choose a color scheme
colormap(cmap);
yline(seconds(period(tavgp_locs(1:3))),'--','Color',col4,'LineWidth',lw) % Show the locations of the spectral peaks
xlabel('time [s]')
ylabel('period [s]')
title('Magnitude scalogram')
axis([min(time_final) max(time_final) min(seconds(period)) max(seconds(period))])

% Right subplot
ax(2) = subplot(1,3,3);
nbins = 50; % Choose the number of histogram bins
bins = 10.^(linspace(log10(min(seconds(period))),log10(max(seconds(period))),nbins)); % Bin edges (logarithmic spacing)
histogram(cwt_max_periods, bins,'Normalization','count','FaceColor',col3,'FaceAlpha',1) % Histogram of the maxima in the left wavelet spectrum
set(gca, "XScale", "log",'XLim',[min(seconds(period)) max(seconds(period))])
hold on
semilogx(seconds(period),tavgp*2*10^8,'Color',col4,'LineWidth',lw) % Also show the time-averaged wavelet spectrum
xline(seconds(period(tavgp_locs(1:3))),'--','Color',col4,'LineWidth',lw) % Also show the locations of the 3 dominant peaks in this time-averaged wavelet spectrum
set(gca,'XLim',[min(seconds(period)) max(seconds(period))])
xlabel('period [s]')
ylabel('bincount; power x 2\cdot10^{8}')
legend('max periods (bincount)','time-avg wavelet spectrum (power x 2\cdot10^{8})')
title('Time-averaged wavelet spectrum')
for i = 1:3
    yl = get(gca,'YLim');
    text(1.1*seconds(period(tavgp_locs(i))),(4400/5000)*yl(2),[num2str(seconds(period(tavgp_locs(i))),'%.0f'),' s'])
end
set(gca,'view',[90 -90])

sgtitle(['filtered data (final0), sensor ', num2str(sensor),' (x = ',num2str(flume.x_sensors(sensor)),' m)'],'FontSize',get(gca,'FontSize')+2);

% Save plot
% dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_final0.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, dataname,'_cwt_final0.pdf'],'Resolution',dpi)
%         saveas(gcf,[pathOutput, dataname,'_cwt_final0.fig'])

% OK.

%% Idem, but finetune the figure (for Supplementary Information figure)

lw = 1; % LineWidth
fs = 11;

celerity = (max(flume.x_sensors)-min(flume.x_sensors))/(20*60); % [m/s] Approximate dune celerity (covers measured section in ca. 20min)
L_flume = 15; % Total flume length [m]
T_bar = (L_flume/celerity)/60; % Expected period of a circulating bar [min]

col1 = [0 0.4470 0.7410]; % blue
col2 = [0.8500 0.3250 0.0980]; % orange
col3 = [0.9290 0.6940 0.1250]; % yellow
col4 = [0.4940 0.1840 0.5560]; % purple
col6 = [0.3010 0.7450 0.9330]; % light blue

figure('units','normalized','outerposition',[0 0 1 1]) % Open a screen-sized figure

ax(1) = subplot(1,3,1:2); % Left subplot
imagesc(time_final/60,minutes(period),abs(wt)) % plot the wavelet spectrum
set(gca,'YScale','log','YDir','Normal')
cb = colorbar;
cb.FontSize = fs;
yl = ylabel(cb,'Magnitude','Rotation',270,'FontSize',fs);
hold on
% semilogy(time_final/60,cwt_max_periods/60,'.','Color',col3,'LineWidth',lw) % Show the locations of the spectral maxima
patch([time_final'/60 fliplr(time_final'/60)],[minutes(coi)' max(ylim)*ones(size(minutes(coi)))'],'k','FaceAlpha', 0.5,'EdgeColor','none') % mask out the cone of influence
cmap = flipud(cmocean('ice')); % Choose a color scheme
colormap(cmap);
yline(minutes(period(tavgp_locs(1:3))),'--','Color',col2,'LineWidth',lw+1) % Show the locations of the spectral peaks
yline(T_bar,':','Color',[0.6 0.6 0.6],'LineWidth',lw+1) % Expected period of a recirculating bar
xlabel('Flume run time [min]')
ylabel('Period [min]')
axis([min(time_final/60) max(time_final/60) min(minutes(period)) max(minutes(period))])
set(gca,'FontSize',fs)
title('a) Magnitude scalogram','FontSize',fs)
ax = gca;
ax.TitleHorizontalAlignment = 'left';

% Right subplot
ax(2) = subplot(1,3,3);
nbins = 50; % Choose the number of histogram bins
bins = 10.^(linspace(log10(min(minutes(period))),log10(max(minutes(period))),nbins)); % Bin edges (logarithmic spacing)
% histogram(cwt_max_periods, bins,'Normalization','count','FaceColor',col3,'FaceAlpha',1) % Histogram of the maxima in the left wavelet spectrum
set(gca, "XScale", "log",'XLim',[min(seconds(period)) max(seconds(period))])
hold on
semilogx(minutes(period),tavgp,'Color',col2,'LineWidth',lw+1) % Also show the time-averaged wavelet spectrum
xline(minutes(period(tavgp_locs(1:3))),'--','Color',col2,'LineWidth',lw+1) % Also show the locations of the 3 dominant peaks in this time-averaged wavelet spectrum
xline(T_bar,':','Color',[0.6 0.6 0.6],'LineWidth',lw+1) % Expected period of a recirculating bar
set(gca,'XLim',[min(minutes(period)) max(minutes(period))])
xlabel('Period [min]')
ylabel('Power')
% legend('Time-avg wavelet spectrum (power x 2\cdot10^{8})')
for i = 1:3
    yl = get(gca,'YLim');
    text(1.1*minutes(period(tavgp_locs(i))),0.1*10^-5,[num2str(minutes(period(tavgp_locs(i))),'%.1f'),' min'],'FontSize',fs)
end
set(gca,'view',[90 -90])
set(gca,'FontSize',fs)
title('b) Time-averaged wavelet spectrum','FontSize',fs)
ax = gca;
ax.TitleHorizontalAlignment = 'left';

% sgtitle(['filtered data (final0), sensor ', num2str(sensor),' (x = ',num2str(flume.x_sensors(sensor)),' m)'],'FontSize',get(gca,'FontSize')+2);

% Save plot
dpi = 500; % Resolution
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_wavelet.png'],'Resolution',dpi)
% exportgraphics(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_wavelet.pdf'],'Resolution',dpi)
        % saveas(gcf,[pathOutput, 'SI_Figure_XX_flume_exp_wavelet.fig'])
exportgraphics(gcf,[pathOutput, 'FigureS10.pdf'],'Resolution',dpi)

% OK.

%% Save wavelet output and circulating bar estimates, and then clear redundant information

wavelet.samplingperiod = samplingperiod; % Sampling period [s] of the final, filtered data
wavelet.filterbank = filterbank; % Settings for wavelet transform
wavelet.period = period; % Periods [s] along the wavelet spectrum
wavelet.wt = wt; % wavelet spectrum
wavelet.coi = coi; % Cone of influence
wavelet.tavgp = tavgp; % time-averaged wavelet
wavelet.tavgp_pks = tavgp_pks; % peaks in the tavgp
wavelet.tavgp_locs = tavgp_locs; % Locations (time periods) of these peaks
wavelet.cwt_max_periods = cwt_max_periods; % periods where the wavelet is maximum, for each timestep

flume.celerity = celerity; % Estimated dune celerity [m/s]
flume.L_flume = L_flume; % Total length of the flume [m]
flume.T_bar = T_bar; % Estimated period of a recirculating bar [min]

clear samplingperiod filterbank period wt coi tavgp tavgp_pks tavgp_locs cwt_max_periods
clear ax bins cmap col* dpi i I lw nbins
clear celerity L_flume T_bar cb file fs yl

% OK.

%% Save all remaining output

close all
clc

save([pathOutput, dataname,'_output.mat'])

% OK.