clear all;close all;clc %#ok<CLALL>

%%
CompositeOnline=[0.32497846515584 % from Tabone, W., Happee, R., García, J., Lee, Y. M., Lupetti, M. L., Merat, N., & de Winter, J. (2023). Augmented reality interfaces for pedestrian-vehicle interactions: An online study. Transportation research part F: traffic psychology and behaviour, 94, 170-189.
    -0.260153657514941
    -0.350969688371509
    -0.122852113425989
    0.28135465699983
    0.0367235618369252
    -0.522883616413057
    0.36722628454074
    0.246576107192162];

%                     non-yielding AV           yielding AV
IntuitivenessOnline =[5.66834677419355          5.52822580645161 % from Tabone, W., Happee, R., García, J., Lee, Y. M., Lupetti, M. L., Merat, N., & de Winter, J. (2023). Augmented reality interfaces for pedestrian-vehicle interactions: An online study. Transportation research part F: traffic psychology and behaviour, 94, 170-189.
    4.61088709677419          4.85383064516129
    4.64112903225806          4.6491935483871
    4.94858870967742          4.83870967741935
    5.62399193548387          5.57056451612903
    5.00100806451613          5.47782258064516
    4.06754032258065          4.2883064516129
    5.67741935483871          5.6875
    5.5866935483871           5.49596774193548];

is = csvread('trial_conditions.csv',1); %#ok<CSVRD> % 3600 x 6 matrix, with 1: participant number, 2: trial number, 3: interface number, 4: non-yielding/yielding, 5: circle position, 6: intuitiveness rating

CD = is(:,1:5); % Conditions
% 1: Participant number (1 to 30)
% 2: Trial number (1 to 120)
% 3: AR interface number (1 to 10)
% 4: Car (AV) behaviour (0: non-yielding, 1: yielding)
% 5: Attention-attractor circle position (0: left, 1: middle, 2: right)

AOIlabels = {'Road1','Road2', 'Road3', 'Road', 'Car', 'Circle', 'ARConcept', 'HudReg', 'TLightReg', 'WindScrReg', 'Other'};
AOIlabelsForFigure = {'Road1','Road2', 'Road3', 'Road', 'Car', 'Circle', 'AR interface', 'HUD region', 'Traffic light region', 'Windscreen region', 'Other'};
Labels={'Augmented zebra crossing','Planes on vehicle','Conspicuous looming planes','Field of safe travel','Fixed pedestrian lights','Virtual fence','Phantom car','Nudge HUD','Pedestrian lights HUD','Baseline'};

%% Read all Hiker data

% 1. SystemTime: time stamp of the current time
% 2. t: trail start time
% 3. P-St: Participant status - Ready (first tile of pavement), Crossing (on the road), Crossed (on the tile)
% 4. P.x: Participant position (m)
% 5. P.z: Participant position (m)
% 6. P.speed: Participant movement speed (mph)
% 7. P.r.w
% 8. P.r.x
% 9. P.r.y
% 10. P.r.z: Participant quaternion rotations
% 11. P.rEu.x
% 12. P.rEu.y
% 13. P.rEu.z: Participant Euler angle locations (deg),
% 14. LookingAt: Where the head (tracker) direction is (Road1, Road2, Road3, Road, Car, Circles, ARConcept, HudReg, TrafficLightReg, WindScrReg, Other);
% 15. GazingAt: Where gaze (eye-tracker) direction is (Road1, Road2, Road3, Road, Car, Circles, ARConcept, HudReg, TrafficLightReg, WindScrReg, Other);
% 16. GazeHitPoint.x
% 17. GazeHitPoint.y
% 18. GazeHitPoint.z
% 19. GazeVec.x
% 20. GazeVec.y
% 21. GazeVec.z: Gaze directional vector
% 22. Circle-St: Circle status: (Y: active, N: deactive)
% 23. Buzzer-St: Buzzer status (Y: active, N: deactive)
% 24. AR-St: AR status: (Y: active, N: deactive)
% 25. AR.x
% 26. AR.y
% 27. AR.z: AR concept position
% 28. CarLength: Car length (m)
% 29. Car1.x
% 30. Car1.z: Car position in the scene (m)
% 31. CarSpeed (mph)

read_raw_Hiker_data = 1;
if read_raw_Hiker_data == 1
    counter = 0;
    HikerData = NaN(3600,4500,10); % 3600 trials (30 participants x 120 trials) x 4500 data points per trial x 10 variabless
    for p = 1:30 % loop over 30 participants
        for i = 1:120 % loop over 120 trials
            disp(['Participant: ' num2str(p) ', trial number: ' num2str(i)])
            folderfile = ['Logs\Participant_' sprintf('%0.2d',p) '\Sub' num2str(p) '_Trial' num2str(i) '_*.*']; % folder and first part of file
            filename = dir(folderfile); % retrieve file name of log file
            filename = filename.name; % extract file name
            hikerdata = readtable(['Logs\Participant_' sprintf('%0.2d',p) '\' filename],'VariableNamingRule','preserve','Delimiter',{' ', ','},'ConsecutiveDelimitersRule','join'); % read the Hiker data file for participant p, trial i

            time = table2array(hikerdata(:,2));
            pedestrianx = table2array(hikerdata(:,4));
            pedestrianz = table2array(hikerdata(:,5));

            LookingAt = table2array(hikerdata(:,14));
            LookingAtNum = zeros(size(LookingAt));
            for l = 1:length(AOIlabels)
                LookingAtNum = LookingAtNum + l.*strcmp(LookingAt,AOIlabels(l));
            end

            GazingAt = table2array(hikerdata(:,15));
            GazingAtNum = zeros(size(GazingAt));
            for l = 1:length(AOIlabels)
                GazingAtNum = GazingAtNum + l.*strcmp(GazingAt,AOIlabels(l));
            end

            CircleStatus = table2array(hikerdata(:,22)); CircleStatus = 1*strcmp(CircleStatus,'N') + 2*strcmp(CircleStatus,'Y');
            BuzzerStatus = table2array(hikerdata(:,23)); BuzzerStatus = 1*strcmp(BuzzerStatus,'N') + 2*strcmp(BuzzerStatus,'Y');

            ARStatus = table2array(hikerdata(:,24));
            if iscell(ARStatus)
                ARStatus= 1*strcmp(ARStatus,'N')+ 2*strcmp(ARStatus,'Y');
            end

            carx = table2array(hikerdata(:,29));
            carz = table2array(hikerdata(:,30));
           
            hikerdatareduced = [time pedestrianx pedestrianz LookingAtNum GazingAtNum CircleStatus BuzzerStatus ARStatus carx carz];
            counter=counter+1;

            HikerData(counter,1:size(hikerdatareduced,1),1:size(hikerdatareduced,2))= hikerdatareduced; % store the Hiker data file for participant p, trial i
        end
    end
    save('HikerData.mat','HikerData','-v7.3'); % save .mat file
else
    load('HikerData.mat') % load .mat file
end
%% Process the data in such a way that the results are time-locked to the moment the attention-attractor circle disappears

process_Hiker_data = 0;
if process_Hiker_data==1
    HikerDataShifted = NaN(3600,4000,9);
    for t = 1:3600 % loop over 3600 trials (30 participants x 120 trials)
        disp(['Trial number: ' num2str(t) ' out of 3600'])
        hikerdatareduced = squeeze(HikerData(t,:,:));
        hikerdatareduced = hikerdatareduced(1: find(~isnan(hikerdatareduced(:,1)),1,'last'),:); % retain up to the last numerical value (i.e., discard the NaNs)
        hikerdatareduced(isnan(hikerdatareduced(:,1)),:)=[]; % remove missing measurement points (usually only 2 or 3 per trial)
        hikerdatareducedi = interp1(hikerdatareduced(:,1),hikerdatareduced(:,2:end),0.01:0.01:40,'nearest'); % interpolate to a new time vector (100 Hz)
        AACDix = find(hikerdatareducedi(:,5)==2,1,'last') + 1; % index where the attention-attractor circle has just disappeared
        data_to_store = hikerdatareducedi(AACDix-916:end,:);
        HikerDataShifted(t,1:size(data_to_store,1),1:size(data_to_store,2)) = data_to_store;
    end
    save('HikerDataShifted.mat','HikerDataShifted','-v7.3'); % save .mat file
else
    load('HikerDataShifted.mat')
end

time = -9.17:0.01:30.82; % correspoding time vector, where t = 0 s is the moment the circle has just disappeared

% Variables of the HikerDataShifted .mat file:
% 1. Pedestrian x (parallel to the road)
% 2. Pedestrian z (perpendicular to the road)
% 3. LookingAtNum (according to AOIlabels)
% 4. GazingAtNum (according to AOIlabels)
% 5. Circle Status (1: off, 2: on)
% 6. Buzzer Status (1: off, 2: on)
% 7. AR Status (1: off, 2: on)
% 8. Car x (parallel to the road)
% 9. Car z (perpendicular to the road)

%% Figure 11. Plots of the gaze percentage for each area of interest region in the environment.
AOI = HikerDataShifted(:,:,4);
figure;
for i=1:4
    if i==1
        retain_indices=find(CD(:,4)==1 & ismember(CD(:,3),[2 3 4 7]) & CD(:,5)==0);
        TTL='Vehicle-locked interface, Circle: Left';
    elseif i==2
        retain_indices=find(CD(:,4)==1 & ismember(CD(:,3),[2 3 4 7]) & CD(:,5)==2);
        TTL='Vehicle-locked interface, Circle: Right';
    elseif i==3
        retain_indices=find(CD(:,4)==1 & ismember(CD(:,3),[8 9]) & CD(:,5)==0);
        TTL='HUD interface, Circle: Left';
    elseif i==4
        retain_indices=find(CD(:,4)==1 & ismember(CD(:,3),[8 9]) & CD(:,5)==2);
        TTL='HUD interface, Circle: Right';
    end

    AOIr = AOI(retain_indices,:);
    h = histc(AOIr,1:11); %#ok<HISTC> % histogram counts
    p = 100*h./sum(h); % percentage
    p = movmedian(p',20)';
    p = 100*p./sum(p);

    subplot(2,2,i)
    set(gcf,'renderer','painters')

    newcolors = [0.3 0.3 0.3; % Road 1
        0.45 0.45 0.45; % Road 2
        0.6 0.6 0.6; % Road 3
        0.75 0.75 0.75 % Road
        0 0 1 % Car (blue)
        0 1 1  % Circle (cyan)
        0 1 0 % AR concept (green)
        1 165/255 0 % HUD (orange)
        1 0 0 % traffic light (red)
        234/255 221/255 202/255 % windscreen
        1 1 1]; % other
    colororder(newcolors)

    a = area(time,p');
    title(TTL)
    if i==3 || i==4
        xlabel('Time (s)')
    end
    if i==1 || i==3
        ylabel('Gaze target (%)')
    end
    ylim([0 100]);
    xlim([-9 8.6]);
    set(gca,'xtick',-10:2:10)
    if i==2
        l=legend(flip(a),flip(AOIlabelsForFigure),'Position',[0.82 0.608 0.161 0.363]);
    end
    if i==2 || i==4
        set(gca,'yticklabel',[])
    end
    fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',23)

    set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
    if i==1
        set(gca,'position',[0.055 0.565 0.35 0.4])
    elseif i==2
        set(gca,'position',[0.44 0.565 0.35 0.4])
    elseif i==3
        set(gca,'position',[0.055 0.082 0.35 0.4])
    elseif i==4
        set(gca,'position',[0.44 0.082 0.35 0.4])
    end
    set(gca,'tickdir','out')
end
set(gcf, 'Units', 'normalized', 'OuterPosition', [0 0 1 1]);
print('Figure 11', '-dpng','-r600')
%% Calculate participants' intuitiveness scores for each AR interface condition and circle position
[ISy,ISny] = deal(NaN(30,10,3));

for i=1:10 % loop over 10 AR interfaces conditions
    for i2=1:3 % loop over 3 circle positions
        for p=1:30 % loop over 30 participants
            retain_indices = is(:,1)==p & is(:,4)==1 & is(:,3)==i & is(:,5)==i2-1; % yielding AV
            ISy(p,i,i2)=mean(is(retain_indices,6));

            retain_indices = is(:,1)==p & is(:,4)==0 & is(:,3)==i & is(:,5)==i2-1; % non-yielding AV
            ISny(p,i,i2)=mean(is(retain_indices,6));
        end
    end
end

%% Calculate crossing initiation time
Pz = HikerDataShifted(:,:,2);
Pz(Pz==1.34) = NaN;  % 1.34 are bug values that should be removed
Pz = fillmissing(Pz','nearest')';
Pz = -Pz + 4.876;

Pcit = NaN(size(Pz,1),1);
for i=1:size(Pz,1)
    cit=500+find(Pz(i,501:end)>2.050,1,'first');
    if ~isempty(cit)
        Pcit(i)=time(cit);
    end
end

CIT=NaN(30,10,3); % preallocate crossing initiation time matrix
NCIT=NaN(30,10,3,2); % preallocate matrix for sample sizes
for i = 1:10 % loop over 10 interface conditions
    for i2 = 1:3 % loop over 3 circle positions
        for p = 1:30 % loop over 30 participants
            cit = Pcit(CD(:,1)==p & CD(:,4)==1 & CD(:,3)==i & CD(:,5)==i2-1); % only take the 3 yielding trials (CIT does not exist for non-yieding trials)
            CIT(p,i,i2) = mean(cit,'omitnan'); % crossing initiation time; take the mean across 3 trials under identical conditions
            NCIT(p,i,i2,1) = sum(~isnan(cit)); % number of crossing initiation times
            NCIT(p,i,i2,2) = length(cit); % number of trials (this is always 3)
        end
    end
end

% Lane boundaries are from 2.826 m till -1.363 m
% This corresponds to 2.050 m till 6.239 m, after correcting the z position
% so that z = 0 m corresponds to the starting position of the pedestrian
% (reversal of sign, and 4.876 m addition)

%% Figure 7. Scatter plot of the mean intuitiveness rating per AR interface in their non-yielding state, as reported in the current HIKER experiment and the previous online questionnaire study (Tabone et al., 2023).
v1=mean(squeeze(mean(ISny(:,1:9,:),3)))';
v2=IntuitivenessOnline(:,1);
disp('Correlation coefficient corresponding to Figure 7 - Non-yielding AVs')
disp(corr(v1,v2))

cm = colormap(turbo(9));
Figure7TextPos=[4.4841928721174             5.68230903834449                         0
                5.50180293501048            4.52975502130249                         0
                5.63555555555556            4.64112903225806                         0
                6.18                        4.94858870967742                         0
                6.40222222222222            5.62399193548387                         0
                5.84058700209644            5.08968730979915                         0
                4.58                        4.06754032258065                         0
                6.29656184486373            5.74345709068777                         0
                6.2093501048218             5.51876902008521                         0];
figure
set(gcf,'renderer','painters')
for i = 1:9
    scatter1 = scatter(v1(i),v2(i), 400, 'markerfacecolor', cm(i, :),  'markeredgecolor', cm(i, :));
    hold on
    grid on;box on
    scatter1.MarkerFaceAlpha = 0.8;
    text(v1(i) + 0.05, v2(i), char(Labels(i)), 'horizontalalignment', 'left', 'fontsize', 28,'position',Figure7TextPos(i,:));
end
plot([1 7],[1 7],':','color',[0.5 0.5 0.5],'Linewidth',3)
set(gca,'XLim',[4 7])
set(gca,'YLim',[4 7])
set(gca,'fontsize', 28)
axis square
xlabel('Intuitiveness score, non-yielding AV, HIKER study (1–7)','fontsize', 28)
ylabel('Intuitiveness score, non-yielding AV, Online study (1–7)','fontsize', 28)
set(gcf, 'Units', 'normalized', 'OuterPosition', [0 0 1 1]);
print('Figure 7', '-dpng','-r600')
%% Figure 8. Scatter plot of the mean intuitiveness rating per AR interface in their yielding state, as reported in the current HIKER experiment and the previous online questionnaire study (Tabone et al., 2023).
v1=mean(squeeze(mean(ISy(:,1:9,:),3)))';
v2=IntuitivenessOnline(:,2);
disp('Correlation coefficient corresponding to Figure 8 - Yielding AVs')
disp(corr(v1,v2))
cm = colormap(turbo(9));

figure
set(gcf,'renderer','painters')
Figure8TextPos=[4.66944793850454          5.54143335362142                         0
                6.04777777777778          4.85383064516129                         0
                5.78481481481481           4.6491935483871                         0
                4.50850454227813          4.84248326232501                         0
                6.30158630328442          5.59131923311016                         0
                5.53889587700909           5.4117848447961                         0
                4.99222222222222           4.2883064516129                         0
                6.22185185185185          5.70070754716981                         0
                6.37081481481481          5.46389227023737                         0];
for i = 1:9
    scatter1 = scatter(v1(i),v2(i), 400, 'markerfacecolor', cm(i, :),  'markeredgecolor', cm(i, :));
    hold on
    grid on;box on
    scatter1.MarkerFaceAlpha = 0.8;
    text(v1(i) + 0.05, v2(i), char(Labels(i)), 'horizontalalignment', 'left', 'fontsize', 28,'Position',Figure8TextPos(i,:));
end
plot([1 7],[1 7],':','color',[0.5 0.5 0.5],'Linewidth',3)
set(gca,'XLim',[4 7])
set(gca,'YLim',[4 7])
set(gca,'fontsize', 28)
axis square
xlabel('Intuitiveness score, yielding AV, HIKER study (1–7)','fontsize', 28)
ylabel('Intuitiveness score, yielding AV, Online study (1–7)','fontsize', 28)
set(gcf, 'Units', 'normalized', 'OuterPosition', [0 0 1 1]);
print('Figure 8', '-dpng','-r600')
%% Figure 9. Bar plot of the mean crossing initiation time as affected by the attention-attractor position for each AR interface. Vertical lines delineate the road-mapped interfaces, vehicle-mapped interfaces, HUDs, and the baseline condition.
LabelsS=Labels([1 5 6   2 3 4 7  8 9  10]);
figure;
set(gcf,'renderer','painters')

CITS=CIT(:,[1 5 6   2 3 4 7  8 9  10],:); % sorted on AI interfaces (30 participants x 10 interfaces x 3 yielding trials matrix)
temp=[squeeze(CITS(:,1,1:3)) squeeze(CITS(:,2,1:3)) squeeze(CITS(:,3,1:3)) squeeze(CITS(:,4,1:3)) squeeze(CITS(:,5,1:3)) squeeze(CITS(:,6,1:3)) squeeze(CITS(:,7,1:3)) squeeze(CITS(:,8,1:3)) squeeze(CITS(:,9,1:3)) squeeze(CITS(:,10,1:3))];

ci=NaN(1,30);PV=NaN(10,3);
for i=1:10
    si=i*3-2;
    ei=si+2;
    ci(si:ei)=within_subject_ci(temp(:,si:ei));
    [~,PV(i,1)] = ttest(temp(:,si),temp(:,si+1)); % p-value for left vs middle
    [~,PV(i,2)] = ttest(temp(:,si),temp(:,si+2)); % p-value for left vs right
    [~,PV(i,3)] = ttest(temp(:,si+1),temp(:,si+2)); % p-value for middle vs right
end

b = bar(squeeze(mean(CITS,'omitnan')));hold on
errorbar([b(1).XEndPoints' b(2).XEndPoints' b(3).XEndPoints'],squeeze(mean(CITS,'omitnan')),[ci(1:3:end)' ci(2:3:end)' ci(3:3:end)'],'ko','Linewidth',3)
VC=5.10;
O=-0.03;
for i = 1:10
    if PV(i,1) < 0.05/3 % left vs middle
        line([i-0.22; i],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i; i-0.22 i],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)
        text(i-0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')
    end
    if PV(i,3) < 0.05/3 % middle vs right
        line([i;i+0.22],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i i+0.22; i i+0.22],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)
        text(i+0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')

    end
    if PV(i,2) < 0.05/3 % left vs right
        line([i-0.22; i+0.22],[VC+0.08; VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i+0.22; i-0.22 i+0.22],[VC+0.06 VC+0.06; VC+0.08 VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        text(i,VC+0.085,'*','FontSize',24,'HorizontalAlignment','center')
    end
end

plot([3.5 3.5],[0 100],'k:','Linewidth',3)
plot([7.5 7.5],[0 100],'k:','Linewidth',3)
plot([9.5 9.5],[0 100],'k:','Linewidth',3)

legend('Left','Centre','Right','location','southwest','orientation','horizontal');
set(gca,'xticklabel',LabelsS)
ylabel('Crossing initiation time (s)')
set(gca,'TickDir','out')
set(gca,'ylim',[0.9 5.3])
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
h=findobj('FontName','Helvetica'); set(h,'FontSize',25,'Fontname','Arial')
set(gcf, 'Units', 'normalized', 'OuterPosition', [0 0 1 1]);
print('Figure 9', '-dpng','-r600')

temp=[CIT(:,:,1) CIT(:,:,2) CIT(:,:,3)]; % crossing initiation time
clear table;t=table(temp(:,1),temp(:,2),temp(:,3),temp(:,4),temp(:,5),temp(:,6),temp(:,7),temp(:,8),temp(:,9),temp(:,10), ...
    temp(:,11),temp(:,12),temp(:,13),temp(:,14),temp(:,15),temp(:,16),temp(:,17),temp(:,18),temp(:,19),temp(:,20), ...
    temp(:,21),temp(:,22),temp(:,23),temp(:,24),temp(:,25),temp(:,26),temp(:,27),temp(:,28),temp(:,29),temp(:,30), ...
    'VariableNames',{'t1','t2','t3','t4','t5','t6','t7','t8','t9','t10','t11','t12','t13','t14','t15','t16','t17','t18','t19','t20','t21','t22','t23','t24','t25','t26','t27','t28','t29','t30'});
Conditions1={'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10'}'; % interface number
Conditions2={'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1'  'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2'  'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3'}'; % circle position number
rm0=fitrm(t,'t1-t30~1','WithinDesign',table(Conditions1,Conditions2,'VariableNames',{'Interface','Circle'}));
rm=ranova(rm0,'WithinModel','Interface*Circle');
disp(rm)
disp('Partial eta squared')
rma=table2array(rm);
disp(round([rma(3,1)/(rma(3,1)+rma(4,1)) rma(5,1)/(rma(5,1)+rma(6,1)) rma(7,1)/(rma(7,1)+rma(8,1))],2))
%% Determine first look at AOI
FirstGlanceAR_Time=NaN(30,10,3);
FirstGlanceAR_N=NaN(30,10,3,2);
FirstGlanceCAR_N=NaN(30,10,3);
for i=1:10
    for i2=1:3
        for p=1:30
            retain_indices=find(CD(:,1)==p & CD(:,3)==i & CD(:,5)==i2-1);
            T_AR=NaN(length(retain_indices),1);
            T_CAR=NaN(length(retain_indices),1);
            for t=1:length(retain_indices)
                first_glace_AR_idx = find(ismember(AOI(retain_indices(t),:),7),1,'first'); % first glance at AR concept index
                first_glace_car_idx = find(ismember(AOI(retain_indices(t),:),[5 10]),1,'first'); % first glance at the car or windscreen region
                % 1. 'Road1'
                % 2. 'Road2'
                % 3. 'Road3'
                % 4. 'Road'
                % 5. 'Car'
                % 6. 'Circle'
                % 7. 'AR Concept'
                % 8. 'HUDReg'
                % 9. 'TrafficLightReg'
                % 10. 'WindScrReg'
                % 11. 'Other');

                if ~isempty(first_glace_AR_idx)
                    T_AR(t)=time(first_glace_AR_idx);
                end
                if ~isempty(first_glace_car_idx)
                    T_CAR(t)=time(first_glace_car_idx);
                end
                FirstGlanceAR_Time(p,i,i2) = mean(T_AR,'omitnan');
                FirstGlanceAR_N(p,i,i2,1) = sum(~isnan(T_AR)); % number of glance times
                FirstGlanceAR_N(p,i,i2,2) = length(T_AR); % number of trials (this is always 4)
                FirstGlanceCAR_N(p,i,i2) = sum(isnan(T_CAR)); % number of trials that the participant has not glanced at the car at all
            end
        end
    end
end
%% Figure 10. Bar plot of the moment in which the participants first glanced at the AR interface as affected by the attention-attractor position for each AR interfflaace.
figure;
FirstGlanceAR_TimeS=FirstGlanceAR_Time(:,[1 5 6   2 3 4 7  8 9  10],:); % sorted on AI interfaces
temp=[squeeze(FirstGlanceAR_TimeS(:,1,1:3)) squeeze(FirstGlanceAR_TimeS(:,2,1:3)) squeeze(FirstGlanceAR_TimeS(:,3,1:3)) squeeze(FirstGlanceAR_TimeS(:,4,1:3)) squeeze(FirstGlanceAR_TimeS(:,5,1:3)) squeeze(FirstGlanceAR_TimeS(:,6,1:3)) squeeze(FirstGlanceAR_TimeS(:,7,1:3)) squeeze(FirstGlanceAR_TimeS(:,8,1:3)) squeeze(FirstGlanceAR_TimeS(:,9,1:3)) squeeze(FirstGlanceAR_TimeS(:,10,1:3))];

ci=NaN(1,30);
PV=NaN(10,3);
for i=1:10
    si=i*3-2;
    ei=si+2;
    ci(si:ei)=within_subject_ci(temp(:,si:ei));
    [~,PV(i,1)] = ttest(temp(:,si),temp(:,si+1)); % p-value for left vs middle
    [~,PV(i,2)] = ttest(temp(:,si),temp(:,si+2)); % p-value for left vs right
    [~,PV(i,3)] = ttest(temp(:,si+1),temp(:,si+2)); % p-value for middle vs right
end

b = bar(squeeze(mean(FirstGlanceAR_TimeS,'omitnan')));hold on
errorbar([b(1).XEndPoints' b(2).XEndPoints' b(3).XEndPoints'],squeeze(mean(FirstGlanceAR_TimeS,'omitnan')),[ci(1:3:end)' ci(2:3:end)' ci(3:3:end)'],'ko','Linewidth',3)

VC=3.0;
O=-0.03;
for i = 1:10
    if PV(i,1) < 0.05/3 % left vs middle
        line([i-0.22; i],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i; i-0.22 i],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)

        text(i-0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')
    end
    if PV(i,3) < 0.05/3 % middle vs right
        line([i;i+0.22],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i i+0.22; i i+0.22],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)
        text(i+0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')

    end
    if PV(i,2) < 0.05/3 % left vs right
        line([i-0.22; i+0.22],[VC+0.08; VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i+0.22; i-0.22 i+0.22],[VC+0.06 VC+0.06; VC+0.08 VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        text(i,VC+0.085,'*','FontSize',24,'HorizontalAlignment','center')
    end
end

plot([3.5 3.5],[-100 100],'k:','Linewidth',3)
plot([7.5 7.5],[-100 100],'k:','Linewidth',3)

legend('Left','Centre','Right','location','southwest','orientation','horizontal');
set(gca,'xticklabel',LabelsS(1:9))
ylabel('First glance at AR interface (s)')
set(gca,'TickDir','out')
set(gca,'ylim',[-1.2 3.2],'xlim',[0.5 9.5])
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
h=findobj('FontName','Helvetica'); set(h,'FontSize',25,'Fontname','Arial')
set(gcf, 'Units', 'normalized', 'OuterPosition', [0 0 1 1]);
print('Figure 10', '-dpng','-r600')

temp=[FirstGlanceAR_TimeS(:,:,1) FirstGlanceAR_TimeS(:,:,2) FirstGlanceAR_TimeS(:,:,3)]; % first glance at AR interface
clear table;t=table(temp(:,1),temp(:,2),temp(:,3),temp(:,4),temp(:,5),temp(:,6),temp(:,7),temp(:,8),temp(:,9), ...
    temp(:,11),temp(:,12),temp(:,13),temp(:,14),temp(:,15),temp(:,16),temp(:,17),temp(:,18),temp(:,19), ...
    temp(:,21),temp(:,22),temp(:,23),temp(:,24),temp(:,25),temp(:,26),temp(:,27),temp(:,28),temp(:,29), ...
    'VariableNames',{'t1','t2','t3','t4','t5','t6','t7','t8','t9','t10','t11','t12','t13','t14','t15','t16','t17','t18','t19','t20','t21','t22','t23','t24','t25','t26','t27'});
Conditions1={'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9'}'; % interface number
Conditions2={'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1'  'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3'}'; % circle position number
rm0=fitrm(t,'t1-t27~1','WithinDesign',table(Conditions1,Conditions2,'VariableNames',{'Interface','Circle'}));
rm=ranova(rm0,'WithinModel','Interface*Circle');
disp(rm)
disp('Partial eta squared')
rma=table2array(rm);
disp(round([rma(3,1)/(rma(3,1)+rma(4,1)) rma(5,1)/(rma(5,1)+rma(6,1)) rma(7,1)/(rma(7,1)+rma(8,1))],2))
%%
disp('Crossing intiation time:')
disp(['Out of a total of ' num2str(sum(sum(sum(NCIT(:,:,:,2))))) ' trials (10 AR interface conditions × 3 yielding trials × 3 repetitions x 30 participants), participants did not initiate crossing in ' num2str(-sum(sum(sum(NCIT(:,:,:,1))))+sum(sum(sum(NCIT(:,:,:,2))))) ' instances.'])
disp(['Out of ' num2str(size(CIT,1)*size(CIT,2)*size(CIT,3)) ' data points (10 AR interface conditions × 3 circle positions x 30 participants) that underlie the statistical test, ' num2str(sum(sum(sum(isnan(CIT)))))  ' data points were unavailable because participants did not initiate crossing.'])
disp('First glance at AR interface:')
disp(['Out of a total of ' num2str(sum(sum(sum(FirstGlanceAR_N(:,1:9,:,2))))) ' trials (9 AR interfaces × 4 trials × 3 repetitions x 30 participants), participants did not glance at the AR interface in ' num2str(-sum(sum(sum(FirstGlanceAR_N(:,1:9,:,1))))+sum(sum(sum(FirstGlanceAR_N(:,1:9,:,2))))) ' instances.'])
disp(['Out of ' num2str(size(FirstGlanceAR_Time,1)*(size(FirstGlanceAR_Time,2)-1)*size(FirstGlanceAR_Time,3)) ' data points (9 AR interfaces × 3 circle positions x 30 participants) that underlie the statistical test, ' num2str(sum(sum(sum(isnan(FirstGlanceAR_Time(:,1:9,:)))))) ' data points were unavailable because participants did not look at the AR interface.'])
disp(newline)
disp('Number of trials in which the participant has not glanced at the car at all for the 10 respective AR interface conditions')
for i=1:10
    disp([Labels{i} ': ' num2str(sum(sum(FirstGlanceCAR_N(:,i,:),3),1))])
end
%% Extra figure: intuitiveness score for non-yielding AVs
figure;
set(gcf,'renderer','painters')

ISnyS=ISny(:,[1 5 6   2 3 4 7  8 9  10],:); % sorted on AI interfaces (30 participants x 10 interfaces x 3 yielding trials matrix)
temp=[squeeze(ISnyS(:,1,1:3)) squeeze(ISnyS(:,2,1:3)) squeeze(ISnyS(:,3,1:3)) squeeze(ISnyS(:,4,1:3)) squeeze(ISnyS(:,5,1:3)) squeeze(ISnyS(:,6,1:3)) squeeze(ISnyS(:,7,1:3)) squeeze(ISnyS(:,8,1:3)) squeeze(ISnyS(:,9,1:3)) squeeze(ISnyS(:,10,1:3))];

ci=NaN(1,30);
PV=NaN(10,3);
for i=1:10
    si=i*3-2;
    ei=si+2;
    ci(si:ei)=within_subject_ci(temp(:,si:ei));
    [~,PV(i,1)] = ttest(temp(:,si),temp(:,si+1));
    [~,PV(i,2)] = ttest(temp(:,si),temp(:,si+2));
    [~,PV(i,3)] = ttest(temp(:,si+1),temp(:,si+2));
end

b = bar(squeeze(mean(ISnyS,'omitnan')));hold on
errorbar([b(1).XEndPoints' b(2).XEndPoints' b(3).XEndPoints'],squeeze(mean(ISnyS,'omitnan')),[ci(1:3:end)' ci(2:3:end)' ci(3:3:end)'],'ko','Linewidth',3)

VC=7.30;
O=-0.03;
for i = 1:10
    if PV(i,1) < 0.05/3 % left vs middle
        line([i-0.22; i],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i; i-0.22 i],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)

        text(i-0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')
    end
    if PV(i,3) < 0.05/3 % middle vs right
        line([i;i+0.22],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i i+0.22; i i+0.22],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)
        text(i+0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')
    end
    if PV(i,2) < 0.05/3 % left vs right
        line([i-0.22; i+0.22],[VC+0.08; VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i+0.22; i-0.22 i+0.22],[VC+0.06 VC+0.06; VC+0.08 VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        text(i,VC+0.085,'*','FontSize',24,'HorizontalAlignment','center')
    end
end

plot([3.5 3.5],[0 100],'k:','Linewidth',3)
plot([7.5 7.5],[0 100],'k:','Linewidth',3)
plot([9.5 9.5],[0 100],'k:','Linewidth',3)

legend('Left','Centre','Right','location','southwest','orientation','horizontal');
set(gca,'xticklabel',LabelsS)
ylabel('Intuitiveness score, non-yielding AV (1–7)')
set(gca,'TickDir','out')
set(gca,'ylim',[3.5 7.5])
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
h=findobj('FontName','Helvetica'); set(h,'FontSize',25,'Fontname','Arial')

temp=[ISnyS(:,:,1) ISnyS(:,:,2) ISnyS(:,:,3)]; % crossing initiation time
clear table;t=table(temp(:,1),temp(:,2),temp(:,3),temp(:,4),temp(:,5),temp(:,6),temp(:,7),temp(:,8),temp(:,9),temp(:,10), ...
    temp(:,11),temp(:,12),temp(:,13),temp(:,14),temp(:,15),temp(:,16),temp(:,17),temp(:,18),temp(:,19),temp(:,20), ...
    temp(:,21),temp(:,22),temp(:,23),temp(:,24),temp(:,25),temp(:,26),temp(:,27),temp(:,28),temp(:,29),temp(:,30), ...
    'VariableNames',{'t1','t2','t3','t4','t5','t6','t7','t8','t9','t10','t11','t12','t13','t14','t15','t16','t17','t18','t19','t20','t21','t22','t23','t24','t25','t26','t27','t28','t29','t30'});
Conditions1={'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10'}'; % interface number
Conditions2={'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1'  'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2'  'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3'}'; % circle position number
rm0=fitrm(t,'t1-t30~1','WithinDesign',table(Conditions1,Conditions2,'VariableNames',{'Interface','Circle'}));
rm=ranova(rm0,'WithinModel','Interface*Circle');
disp(rm)
disp('Partial eta squared')
rma=table2array(rm);
disp(round([rma(3,1)/(rma(3,1)+rma(4,1)) rma(5,1)/(rma(5,1)+rma(6,1)) rma(7,1)/(rma(7,1)+rma(8,1))],2))

%% Extra figure: intuitiveness score for yielding AVs
figure;
set(gcf,'renderer','painters')

ISyS=ISy(:,[1 5 6   2 3 4 7  8 9  10],:); % sorted on AI interfaces (30 participants x 10 interfaces x 3 yielding trials matrix)
temp=[squeeze(ISyS(:,1,1:3)) squeeze(ISyS(:,2,1:3)) squeeze(ISyS(:,3,1:3)) squeeze(ISyS(:,4,1:3)) squeeze(ISyS(:,5,1:3)) squeeze(ISyS(:,6,1:3)) squeeze(ISyS(:,7,1:3)) squeeze(ISyS(:,8,1:3)) squeeze(ISyS(:,9,1:3)) squeeze(ISyS(:,10,1:3))];

ci=NaN(1,30);
PV=NaN(10,3);
for i=1:10
    si=i*3-2;
    ei=si+2;
    ci(si:ei)=within_subject_ci(temp(:,si:ei));
    [~,PV(i,1)] = ttest(temp(:,si),temp(:,si+1)); % p-value for left vs middle
    [~,PV(i,2)] = ttest(temp(:,si),temp(:,si+2)); % p-value for left vs right
    [~,PV(i,3)] = ttest(temp(:,si+1),temp(:,si+2)); % p-value for middle vs right
end

b=bar(squeeze(mean(ISyS,'omitnan')));hold on
errorbar([b(1).XEndPoints' b(2).XEndPoints' b(3).XEndPoints'],squeeze(mean(ISyS,'omitnan')),[ci(1:3:end)' ci(2:3:end)' ci(3:3:end)'],'ko','Linewidth',3)
VC=7.30;
O=-0.03;
for i = 1:10
    if PV(i,1) < 0.05/3 % left vs middle
        line([i-0.22; i],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i; i-0.22 i],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)
        text(i-0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')
    end
    if PV(i,3) < 0.05/3 % middle vs right
        line([i;i+0.22],[VC+O; VC+O],'Color',[0 0 0],'LineWidth',1.2)
        line([i i+0.22; i i+0.22],[VC+O-0.03 VC+O-0.03; VC+O VC+O],'Color',[0 0 0],'LineWidth',1.2)
        text(i+0.11,VC+0.005+O,'*','FontSize',24,'HorizontalAlignment','center')

    end
    if PV(i,2) < 0.05/3 % left vs right
        line([i-0.22; i+0.22],[VC+0.08; VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        line([i-0.22 i+0.22; i-0.22 i+0.22],[VC+0.06 VC+0.06; VC+0.08 VC+0.08],'Color',[0 0 0],'LineWidth',1.2)
        text(i,VC+0.085,'*','FontSize',24,'HorizontalAlignment','center')
    end
end

plot([3.5 3.5],[0 100],'k:','Linewidth',3)
plot([7.5 7.5],[0 100],'k:','Linewidth',3)
plot([9.5 9.5],[0 100],'k:','Linewidth',3)

legend('Left','Centre','Right','location','southwest','orientation','horizontal');
set(gca,'xticklabel',LabelsS)
ylabel('Intuitiveness score, yielding AV (1–7)')
set(gca,'TickDir','out')
set(gca,'ylim',[3.5 7.5])
set(gca,'TickDir','out','LooseInset',[0.01 0.01 0.01 0.01])
h=findobj('FontName','Helvetica'); set(h,'FontSize',25,'Fontname','Arial')

temp=[ISyS(:,:,1) ISyS(:,:,2) ISyS(:,:,3)]; % crossing initiation time
clear table;t=table(temp(:,1),temp(:,2),temp(:,3),temp(:,4),temp(:,5),temp(:,6),temp(:,7),temp(:,8),temp(:,9),temp(:,10), ...
    temp(:,11),temp(:,12),temp(:,13),temp(:,14),temp(:,15),temp(:,16),temp(:,17),temp(:,18),temp(:,19),temp(:,20), ...
    temp(:,21),temp(:,22),temp(:,23),temp(:,24),temp(:,25),temp(:,26),temp(:,27),temp(:,28),temp(:,29),temp(:,30), ...
    'VariableNames',{'t1','t2','t3','t4','t5','t6','t7','t8','t9','t10','t11','t12','t13','t14','t15','t16','t17','t18','t19','t20','t21','t22','t23','t24','t25','t26','t27','t28','t29','t30'});
Conditions1={'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10' 'S1' 'S2' 'S3' 'S4' 'S5' 'S6' 'S7' 'S8' 'S9' 'S10'}'; % interface number
Conditions2={'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1' 'N1'  'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2' 'N2'  'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3' 'N3'}'; % circle position number
rm0=fitrm(t,'t1-t30~1','WithinDesign',table(Conditions1,Conditions2,'VariableNames',{'Interface','Circle'}));
rm=ranova(rm0,'WithinModel','Interface*Circle');
disp(rm)
disp('Partial eta squared')
rma=table2array(rm);
disp(round([rma(3,1)/(rma(3,1)+rma(4,1)) rma(5,1)/(rma(5,1)+rma(6,1)) rma(7,1)/(rma(7,1)+rma(8,1))],2))

%% Transcribe think-aloud using OpenAI's WHISPER
transcribe = 0;
if transcribe ==1
    folderbase = 'Data\Voice Recordings\';
    clear folderfiles
    for p = 1:30 % loop over 30 participants
        for i=1:10 % loop over 10 interface conditions
            folderfile = [folderbase 'P' num2str(p) '_B' num2str(i) '.MP3'];
            folderfiles{p,i} = folderfile; %#ok<SAGROW>
        end
    end
    clear Transcript
    for p = 1:30 % loop over 30 participants
        for i = 1:10 % % loop over 10 trial numbers
            disp([p i])
            import matlab.net.* matlab.net.http.* matlab.net.http.io.*

            header = [HeaderField('Content-Type', 'multipart/form-data')
                HeaderField('Authorization', 'Bearer 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')]';

            uri = URI('https://api.openai.com/v1/audio/transcriptions');
            body = MultipartFormProvider('file', FileProvider(folderfiles{p,i}), 'model', FormProvider('whisper-1'),'language','en');
            response = RequestMessage('post', header, body).send(uri.EncodedURI);
            disp(response.Body.Data)
            Transcript{p,i}=response.Body.Data; %#ok<SAGROW>
            if p==14 && i==3 % recording was split into two files, for participant 14, trial 3
                body = MultipartFormProvider('file', FileProvider('Data\Voice Recordings\P14_B3_2.MP3'), 'model', FormProvider('whisper-1'),'language','en');
                response = RequestMessage('post', header, body).send(uri.EncodedURI);
                disp(response.Body.Data)
                ExtraP14C3=response.Body.Data;
            end
        end
        save('Transcript_Whisper_intermediate.mat','Transcript')
    end
    save('Transcript_Whisper.mat','Transcript','ExtraP14C3')
else
    load('Transcript_Whisper.mat')
end

%% Perform VADER sentiment analysis on interview stranscripts
disp('Perform VADER sentiment analysis')
clear AllInterviewTranscripts
for i=1:10 % loop over 10 AR interfaces of participant
    TranscriptsOfInterface = [];
    for p=1:30 % loop over 30 participants
        block_number = ceil(CD(find(CD(:,1)==p & CD(:,3)==i,1,'first'),2)/12); % block number corresponding to interface i
        response_of_participant = struct2array(Transcript{p,block_number});
        TranscriptsOfInterface = [TranscriptsOfInterface response_of_participant]; %#ok<AGROW>
    end
    AllInterviewTranscripts{i}=TranscriptsOfInterface; %#ok<SAGROW>
end

SentimentScores = zscore(vaderSentimentScores(tokenizedDocument(AllInterviewTranscripts)));
%% Table 2. Correlation matrix for measures relating to the online questionnaire study (Tabone et al., 2023), and the simulator study conducted in the HIKER
disp('Table 2. Correlation matrix for measures relating to the online questionnaire study (Tabone et al., 2023), and the simulator study conducted in the HIKER')
Rankings = csvread('interface_ranking.csv',1); %#ok<CSVRD> % read ranking data (not avalable for one participant)
MR = mean(Rankings(:,2:end))'; % HIKER, mean preference rank for the 9 AR interfaces

XM=[ mean(squeeze(mean(ISny,3)))' mean(squeeze(mean(ISy,3)))' SentimentScores'  [MR;NaN] mean(mean(CIT,3,'omitnan'))'  [IntuitivenessOnline;NaN NaN]];  % HIKER
c=corr(XM,'rows','pairwise');
disp(round(c,2))
%% Table 3. Interview summaries provided by ChatGPT-4o. In the summaries for the Baseline condition, an ‘AR interface’ is still alluded to. The reason is that the ChatGPT prompt only mentioned ‘AR interfaces’.
disp('Table 3. Interview summaries provided by ChatGPT-4o. In the summaries for the Baseline condition, an ‘AR interface’ is still alluded to. The reason is that the ChatGPT prompt only mentioned ‘AR interfaces’')
summarize_interviews=0;
if summarize_interviews==1
    for i=1:10 % loop over 10 AR interfaces of participant
        key
        import matlab.net.* matlab.net.http.* matlab.net.http.io.*
        header = HeaderField('Authorization', api_key);
        uri = URI('https://api.openai.com/v1/chat/completions');

        disp(newline)
        Prompt = ['Based on the participants'' interview responses, summarise the strengths and weaknesses of the interface concisely. Do not overly mention general characteristics such as ''the interface is intuitive'', but report specific aspects that could aid designers. Keep it short, specific, and interesting, with a maximum of 4 sentences in total. Start the strengths with “STRENGTHS:” and start the weaknesses with “WEAKNESSES: Do not refer to individual participants.' newline];
        disp(newline)
        for p=1:30 % loop over 30 participants
            block_number = ceil(CD(find(CD(:,1)==p & CD(:,3)==i,1,'first'),2)/12); % block number corresponding to interface i
            response_of_participant = struct2array(Transcript{p,block_number});
            Prompt = [Prompt newline 'Participant number: ' num2str(p) '. ' response_of_participant]; %#ok<AGROW>
        end
        body = JSONProvider(struct('model','gpt-4o','temperature',0,'n',1,'messages', {{struct('role', 'user','content', Prompt')}}));
        response = RequestMessage('post', [HeaderField('Content-Type', 'application/json') HeaderField('Authorization', api_key)], body).send(uri.EncodedURI);

        output=response.Body.Data.choices.message.content;
        disp(Labels{i})
        disp(output)
    end

    % 1. Participant number (1 to 30)
    % 2. Trial number (1 to 12)
    % 3. AR Interface number (1 to 10)
    % 4. YieldCond (0: non-yielding, 1: yielding)
    % 5. CirclePos (0: left, 1: middle, or 2: right)
end
