### Visualize validation CDFs
{
  ## Set working directory
  setwd("repository/scripts")
  
  ## Load packages
  library(data.table)
  library(ggplot2)
  
  ## Load simulation results
  files = list.files("../simulations/rhino_movement/",
                     full.names = TRUE,
                     recursive = TRUE,
                     include.dirs = FALSE)
  dat = list()
  pb = txtProgressBar(min=0, max=1, initial=0, char="-", style=3)
  for(i in seq_along(files)) {
    dat[[i]] = readRDS(files[i])[[1]]
    setTxtProgressBar(pb, i/length(files))
  }
  dat = rbindlist(dat)
  sum(is.na(dat))==0 #TRUE
  max(dat$encounterdist0h/dat$poacherturndist)<unique(dat$nturns) #TRUE
  setkeyv(dat,c("encounterradius","rhinoturndist","rhinospeed","poacher"))
  setcolorder(dat)
  
  ## Create output table for all selected parameter combinations
  dat_cmb = expand.grid(encounterradius = unique(dat$encounterradius),
                        rhinoturndist = unique(dat$rhinoturndist),
                        rhinospeed = unique(dat$rhinospeed))
  
  ## Compute encountered proportion of rhinos vs moved distance
  dat_cdf = list()
  
  pb = txtProgressBar(min=0, max=1, initial=0, char="-", style=3)
  for(i in seq_len(nrow(dat_cmb))) {
    dats = dat[encounterradius==dat_cmb$encounterradius[i] &
                 rhinoturndist==dat_cmb$rhinoturndist[i] &
                 rhinospeed==dat_cmb$rhinospeed[i],]
    dats[, encounterdist := as.integer(round(encounterdist0h))]
    dats = dats[, lapply(.SD, length), 
                by=c("encounterdist"), 
                .SDcols="poacher"]
    
    dat_cdf[[i]] = 
      data.table(encounterdist = 0:max(dats$encounterdist))
    
    setkeyv(dats, "encounterdist")
    setkeyv(dat_cdf[[i]], "encounterdist")
    dat_cdf[[i]] = dats[dat_cdf[[i]],]
    dat_cdf[[i]][is.na(poacher), poacher := 0]
    dat_cdf[[i]][, encounters := cumsum(poacher)/sum(poacher)]
    dat_cdf[[i]] = dat_cdf[[i]][,-"poacher"]
    dat_cdf[[i]] = cbind(dat_cdf[[i]],dat_cmb[i,])
    setkeyv(dat_cdf[[i]], c(colnames(dat_cmb),"encounterdist"))
    setcolorder(dat_cdf[[i]])
    
    dat_cdf1 = dat_cdf[[i]][!duplicated(encounters),]
    dat_cdf2 = dat_cdf[[i]][nrow(dat_cdf[[i]]):1,][!duplicated(encounters),][-1,]
    dat_cdf[[i]] = rbind(dat_cdf1,dat_cdf2)
    setkeyv(dat_cdf[[i]], c(colnames(dat_cmb),"encounterdist"))
    
    setTxtProgressBar(pb, i/nrow(dat_cmb))
  }
  
  dat_cdf = rbindlist(dat_cdf)
  
  ## Process data for plotting
  dat_cdf[, `:=` ("encounterdist" = encounterdist/1e3,
                  "rhino speed vs poacher" = as.factor(rhinospeed),
                  "mean rhino interturn distance vs poacher" = as.factor(rhinoturndist),
                  "poacher observation radius (m)" = as.factor(encounterradius))]
  
  ## Visualize figure
  p = ggplot(dat_cdf, 
             aes(x=encounterdist, y=encounters,
                 color = rhinospeed,
                 group = rhinospeed)) + 
    geom_line() +
    coord_cartesian(xlim=c(0,2500),
                    ylim=c(0,1),
                    expand=0) +
    facet_grid(`poacher observation radius (m)`~`mean rhino interturn distance vs poacher`,
               labeller = "label_both") +
    scale_colour_gradient(breaks = c(0,.125,.25,.5,1),
                          labels = c(expression(0),
                                     expression(8^-1),
                                     expression(4^-1),
                                     expression(2^-1),
                                     expression(1))) +
    labs(x = "poacher movement distance (km)",
         y = "cumulative probability of first rhino encounter",
         color = "rhino speed vs poacher") +
    theme(panel.spacing = unit(1, "lines"),
          legend.position="top")
  
  ## Save figure
  ggsave("../plots/fig7.pdf",
         p,
         width = 3000,
         height = 4000,
         units = "px")
}
