### Load HPC packages
library(rslurm)
library(data.table)
library(zoo)

### Create parameter grid
params = expand.grid(
  'npoacher' = seq_len(1e4) # number of poaching attempts
)

### Create simulation function
poachersearch = function(npoacher) {
  
  ## Setup output table
  results = as.data.table(expand.grid(
    'poacher' = npoacher, # identifier of poaching attempt
    'knplength' = 360*1e2, # 10% of KNP N-S length in meters
    'knpwidth' = 55*1e2, # 10% of KNP E-W width in meters
    'encounterradius' = c(.5,1,2)*1e3, # maximum distance in meters of poacher detecting rhino
    'nturns' = 2e3, # max number of turns by poacher
    'steps' = 1, # poacher displacement in meters for which to compute distance to rhino
    'poacherturndist' = 2.5*1e3, # distance in meters between poacher turns
    'poacherspeed' = 2.5*1e3, # poacher speed in meters per hour
    'rhinoturndist' = c(.01,.1,1), # mean distance in meters between rhino turns relative to poacher
    'rhinospeed' = c(0,.125,.25,.5,1), # rhino speed relative to poacher
    'encounterdist0h'=NA_real_,
    'encounterdist1h'=NA_real_,
    'encounterdist2h'=NA_real_,
    'encounterdist4h'=NA_real_,
    'encounterdist8h'=NA_real_,
    'encounterdist16h'=NA_real_,
    'encounterdist32h'=NA_real_,
    'encounterdist64h'=NA_real_
  ))
  
  for(i in seq_len(nrow(results))) {
    
    ## Attach data
    attach(results[i,2:10],
           warn.conflicts = FALSE)
    
    ## Unique simulation seed
    set.seed(npoacher)
    
    ## Poacher movement
    poacher = data.table("t"=rep(NA_real_,nturns),
                         "px"=rep(NA_real_,nturns),
                         "py"=rep(NA_real_,nturns))
    ta = runif(nturns-1,
               0,
               2*pi)
    sl = poacherturndist
    dx = sl*cos(ta)
    dy = sl*sin(ta)
    poacher[, `:=` (t = (0:(nturns-1))*(poacherturndist/poacherspeed),
                    px = cumsum(c(knpwidth/2, dx)),
                    py = cumsum(c(knplength/2, dy)))]
    
    ## Rhino movement
    if(rhinospeed==0) {
      rhino = data.table("t"=rep(NA_real_,2),
                         "rx"=rep(NA_real_,2),
                         "ry"=rep(NA_real_,2))
      rhino[, `:=` (t = c(0,(nturns-1))*(poacherturndist/poacherspeed),
                    rx = runif(1,0,knpwidth),
                    ry = runif(1,0,knplength))]
    } else {
      rhino = data.table("t"=rep(NA_real_,((nturns+64*(poacherspeed/poacherturndist))*
                                             rhinospeed/rhinoturndist)),
                         "rx"=rep(NA_real_,((nturns+64*(poacherspeed/poacherturndist))*
                                              rhinospeed/rhinoturndist)),
                         "ry"=rep(NA_real_,((nturns+64*(poacherspeed/poacherturndist))*
                                              rhinospeed/rhinoturndist)))
      ta = runif(nrow(rhino)-1,
                 0,
                 2*pi)
      sl = abs(rnorm(nrow(rhino)-1,
                     sd = (poacherturndist*rhinoturndist)/sqrt(2/pi)))
      dx = sl*cos(ta)
      dy = sl*sin(ta)
      rhino[, `:=` (t = (round(cumsum(c(0,
                                        sl)/
                                        (poacherspeed*rhinospeed))
                               *poacherspeed)/
                           poacherspeed-
                           64),
                    rx = cumsum(c(runif(1,
                                        0,
                                        knpwidth),
                                  dx)),
                    ry = cumsum(c(runif(1,
                                        0,
                                        knplength),
                                  dy)))]
    }
    
    ## Join poacher and rhino data
    dt = data.table("t" = seq(rhino$t[1],
                              max(rhino$t[nrow(rhino)],
                                  poacher$t[nrow(poacher)]),
                              steps/poacherspeed))
    setkeyv(dt, "t")
    setkeyv(poacher, "t")
    setkeyv(rhino, "t")
    poacher = poacher[dt,]
    rhino = rhino[dt,]
    dt = poacher[rhino,]
    maxt = min(rev(poacher[!is.na(px),]$t)[1],
               rev(rhino[!is.na(rx),]$t)[1])
    rm(ta,sl,dx,dy,poacher,rhino)
    invisible(gc())
    
    ## Fill in straight moving lines
    dt[, `:=` (px = na.approx(px, na.rm = FALSE) %% knpwidth,
               py = na.approx(py, na.rm = FALSE) %% knplength,
               rx = na.approx(rx, na.rm = FALSE) %% knpwidth,
               ry = na.approx(ry, na.rm = FALSE) %% knplength)]
    dt = dt[!(t > maxt),]
    rm(maxt)
    invisible(gc())
    
    ## Compute poacher-rhino distances
    dt[, `:=` (ddx = abs(px-rx),
               ddy = abs(py-ry))]
    dt[ddx>(knpwidth/2), ddx := knpwidth - ddx]
    dt[ddy>(knplength/2), ddy := knplength - ddy]
    dt[, d := sqrt(ddx^2+ddy^2)]
    
    ## 0h tracks
    results[i, encounterdist0h := (poacherturndist*
                                     dt[which(d<encounterradius)[1],]$t)]
    if(rhinospeed==0) {
      results[i, `:=` (encounterdist1h = encounterdist0h,
                       encounterdist2h = encounterdist0h,
                       encounterdist4h = encounterdist0h,
                       encounterdist8h = encounterdist0h,
                       encounterdist16h = encounterdist0h,
                       encounterdist32h = encounterdist0h,
                       encounterdist64h = encounterdist0h)]
      
      ## Clear memory
      rm(dt)
      invisible(gc())
      
    } else {
      ## 1h tracks
      dt = dt[1:which(d<encounterradius)[1],]
      dt[, `:=` (ddx = NULL,
                 ddy = NULL,
                 d = NULL)]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt1 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-1)]
        dt1[, `:=` (ddx = abs(dt$px[j+n]-rx),
                    ddy = abs(dt$py[j+n]-ry))]
        dt1[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt1[ddy>(knplength/2), ddy := knplength - ddy]
        dt1[, d := sqrt(ddx^2+ddy^2)]
        mindt1 = min(dt1$d)
        if(mindt1<encounterradius) {
          results[i, encounterdist1h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt1 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## 2h tracks
      dt = dt[1:(j+n),]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt2 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-2)]
        dt2[, `:=` (ddx = abs(dt$px[j+n]-rx),
                    ddy = abs(dt$py[j+n]-ry))]
        dt2[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt2[ddy>(knplength/2), ddy := knplength - ddy]
        dt2[, d := sqrt(ddx^2+ddy^2)]
        mindt2 = min(dt2$d)
        if(mindt2<encounterradius) {
          results[i, encounterdist2h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt2 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## 4h tracks
      dt = dt[1:(j+n),]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt4 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-4)]
        dt4[, `:=` (ddx = abs(dt$px[j+n]-rx),
                    ddy = abs(dt$py[j+n]-ry))]
        dt4[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt4[ddy>(knplength/2), ddy := knplength - ddy]
        dt4[, d := sqrt(ddx^2+ddy^2)]
        mindt4 = min(dt4$d)
        if(mindt4<encounterradius) {
          results[i, encounterdist4h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt4 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## 8h tracks
      dt = dt[1:(j+n),]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt8 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-8)]
        dt8[, `:=` (ddx = abs(dt$px[j+n]-rx),
                    ddy = abs(dt$py[j+n]-ry))]
        dt8[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt8[ddy>(knplength/2), ddy := knplength - ddy]
        dt8[, d := sqrt(ddx^2+ddy^2)]
        mindt8 = min(dt8$d)
        if(mindt8<encounterradius) {
          results[i, encounterdist8h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt8 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## 16h tracks
      dt = dt[1:(j+n),]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt16 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-16)]
        dt16[, `:=` (ddx = abs(dt$px[j+n]-rx),
                     ddy = abs(dt$py[j+n]-ry))]
        dt16[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt16[ddy>(knplength/2), ddy := knplength - ddy]
        dt16[, d := sqrt(ddx^2+ddy^2)]
        mindt16 = min(dt16$d)
        if(mindt16<encounterradius) {
          results[i, encounterdist16h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt16 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## 32h tracks
      dt = dt[1:(j+n),]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt32 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-32)]
        dt32[, `:=` (ddx = abs(dt$px[j+n]-rx),
                     ddy = abs(dt$py[j+n]-ry))]
        dt32[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt32[ddy>(knplength/2), ddy := knplength - ddy]
        dt32[, d := sqrt(ddx^2+ddy^2)]
        mindt32 = min(dt32$d)
        if(mindt32<encounterradius) {
          results[i, encounterdist32h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt32 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## 64h tracks
      dt = dt[1:(j+n),]
      n = nrow(dt[t<0,])
      invisible(gc())
      j=1
      while(j <= (nrow(dt)-n)) {
        dt64 = dt[t<=dt$t[j+n] & t>=(dt$t[j+n]-64)]
        dt64[, `:=` (ddx = abs(dt$px[j+n]-rx),
                     ddy = abs(dt$py[j+n]-ry))]
        dt64[ddx>(knpwidth/2), ddx := knpwidth - ddx]
        dt64[ddy>(knplength/2), ddy := knplength - ddy]
        dt64[, d := sqrt(ddx^2+ddy^2)]
        mindt64 = min(dt64$d)
        if(mindt64<encounterradius) {
          results[i, encounterdist64h := (poacherturndist*dt$t[j+n])]
          break
        } else {
          j = j + ceiling((((mindt64 - encounterradius)/(steps*(1+rhinospeed)))))
        }
      }
      
      ## Clear memory
      rm(dt,dt1,dt2,dt4,dt8,dt16,dt32,dt64)
      invisible(gc())
    }
  }
  
  ## Return output
  return(results)
}

### Start HPC array job
sopt = list('time' = '0-0:30:0',
            'error' = 'error_%a.txt',
            'qos' = 'low',
            'mem-per-cpu' = '2500',
            'cpus-per-task' = '1')

sjob = slurm_apply(f = poachersearch,
                   params = params,
                   jobname = 'rhino_movement',
                   nodes = nrow(params),
                   cpus_per_node = 1,
                   job_array_task_limit = 100,
                   slurm_options = sopt,
                   submit = TRUE)
