/*
 *  Authors:
 *    Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *    Christian Schulte, 2008-2019
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software, to deal in the software without restriction,
 *  including without limitation the rights to use, copy, modify, merge,
 *  publish, distribute, sublicense, and/or sell copies of the software,
 *  and to permit persons to whom the software is furnished to do so, subject
 *  to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#pragma push_macro("slots")
#undef slots
#include "Python.h"
#pragma pop_macro("slots")
#include <gecode/brancher/data-collector.hh>

#include <gecode/int.hh>

#include <iostream>
#include <random>

using namespace Gecode;
using namespace std;

class ProbingSearch : public Brancher {
protected:
	ViewArray<Int::IntView> x;
	// Asset id to identify and asset in portfolio search
	int a_id;
	mutable vector<int> var_assigned;
	// choice definition
	class PosVal : public Choice {
	public:
		int pos; int val;
		PosVal(const ProbingSearch& b, int p, int v)
			: Choice(b, 2), pos(p), val(v) {}
		virtual void archive(Archive& e) const {
			Choice::archive(e);
			e << pos << val;
		}
	};
	
public:
	ProbingSearch(Home home, ViewArray<Int::IntView>& x0, int asset_id)
		: Brancher(home), x(x0), a_id(asset_id) {}
	static void post(Home home, ViewArray<Int::IntView>& x, int asset_id) {
		(void) new (home) ProbingSearch(home, x, asset_id);
	}
	virtual size_t dispose(Space& home) {
		(void)Brancher::dispose(home);
		return sizeof(*this);
	}
	ProbingSearch(Space& home, ProbingSearch& b, int asset_id)
		: Brancher(home, b), a_id(asset_id), var_assigned(b.var_assigned) {
		x.update(home, b.x);
	}
	virtual Brancher* copy(Space& home) {
		return new (home) ProbingSearch(home, *this, a_id);
	}
	// status
	virtual bool status(const Space& home) const {
		for (int i = 0; i < x.size(); i++)
			if (!x[i].assigned()) {
				return true;
			}
		return false;
	}
	// choice
	virtual Choice* choice(Space& home) {
		vector<int> assigned;
		vector<int> unassigned;
		//cout << "x size: " << x.size() << endl;
		
		for (int i = 0; i < x.size() - 1; i++){
			if(x[i].assigned()) {
				assigned.push_back(i);
				//cout << "IS ASSIGNED: " << i << " with val: " << x[i].val() << " and start value: " << start << endl;
			} else {
				unassigned.push_back(i);
			}
			//assigned.push_back(x[i].val());
		}
		if(unassigned.empty()){
			unassigned.push_back(x.size() - 1);
		}
		if(assigned.empty())
			var_assigned.clear();

		// cout << "Var Assigned: [ ";
		// for(int i = 0; i < var_assigned.size(); ++i){
		// 	cout << var_assigned.at(i) << " ";
		// }
		// cout << "]" << endl;

		// cout << "Assigned: [ ";
		// for(int i = 0; i < assigned.size(); ++i){
		// 	cout << assigned.at(i) << " ";
		// }
		// cout << "]" << endl;

		// cout << "Unassigned: [ ";
		// for(int i = 0; i < unassigned.size(); ++i){
		// 	cout << unassigned.at(i) << " ";
		// }
		// cout << "]" << endl;

		bool seen = true;
		int i2remove = 0;

		for (int i = 0; i < var_assigned.size(); i++){
			seen = false;
			for (int j = 0; j < assigned.size(); j++) {
				if( var_assigned.at(i) == assigned.at(j) ){
					//cout << "Seen this var: " << var_assigned.at(i) << endl;
					seen = true;
				}		
			}
			if(!seen) {
				i2remove = i;
				goto remove_rest;
			}
		}
		remove_rest:
		if(!seen){
			//cout << "i2remove: " << i2remove << endl;
			var_assigned.erase(var_assigned.begin() + i2remove,var_assigned.end());
		}
			


		int index = 0;

		//Random val selection
		random_device rd; 
		mt19937 eng(rd()); 
		uniform_int_distribution<> distr(0, unassigned.size() - 1);
		int var = unassigned.at(distr(eng));
		uniform_int_distribution<> distr2(0, x[var].size());
		int val_index = distr2(eng);

		int val = x[var].min();
		// Node is skipped if there is only 1 value in the domain
		int second_to_last_min_val = 1; 
		for (Int::ViewValues<Int::IntView> j(x[var]); j(); ++j) {
			if(index == 1)
				second_to_last_min_val = j.val();
			if(index == val_index){
				val = j.val();
				break;
			}
			index++;
		}
		//}	
			
		DataCollector* dc = DataCollector::getInstance();
		
		var_assigned.push_back(var);
		int var_position = var_assigned.size();
		// cout << "Var Assigned: [ ";
		// for(int i = 0; i < var_assigned.size(); ++i){
		// 	cout << var_assigned.at(i) << " ";
		// }
		// cout << "]" << endl;
		//cout << "Var pos chosen: " << var << " with val " << val << endl;
		dc->storeMLNode(var_assigned, x[var].size(), MLUtils::computeDomainSize(x), var, var_position, val, val_index, x[var].min(), x[var].max(), x[var].regret_min(), x[var].regret_max());
		
		return new PosVal(*this, var, val);
	}
	virtual Choice* choice(const Space&, Archive& e) {
		int pos, val;
		e >> pos >> val;
		return new PosVal(*this, pos, val);
	}
	// commit
	virtual ExecStatus commit(Space& home,
		const Choice& c,
		unsigned int a) {

		const PosVal& pv = static_cast<const PosVal&>(c);
		int pos = pv.pos, val = pv.val;
		if (a == 0)
			return me_failed(x[pos].eq(home, val)) ? ES_FAILED : ES_OK;
		else
			return me_failed(x[pos].nq(home, val)) ? ES_FAILED : ES_OK;
	}
	// print
	virtual void print(const Space& home, const Choice& c,
		unsigned int a,
		ostream& o) const {
		const PosVal& pv = static_cast<const PosVal&>(c);
		int pos = pv.pos, val = pv.val;
		if (a == 0)
			o << "x[" << pos << "] = " << val;
		else
			o << "x[" << pos << "] != " << val;
	}
};
void probing_search(Home home, const IntVarArgs& x, int asset_id) {
	if (home.failed()) return;
	ViewArray<Int::IntView> y(home, x);
	//cout << "Setting root domain and posting" << endl;
	//DataCollector::getInstance()->setRootDomain(y);
	ProbingSearch::post(home, y, asset_id);
}