import numpy as np
from model_cpwresonator import fun_k, cpw_ind_total, cpw_cap_total

'''
Basic functions for mapping of coplanar waveguide resonator to lumped element resonator. 
'''

def le_ind(ind_m, length, ind_nw):
    '''Returns inductance of equivalent lumped element resonator for a given loaded CPW resonator.'''
    return 8*length * ind_m / np.pi**2 + ind_nw


def le_cap(cap_m, length, cap_term):
    '''Returns capacitance of equivalent lumped element resonator for a given loaded CPW resonator.'''
    return length/2 * cap_m + cap_term


def le_freq(ind, cap):
    '''Returns resonance frequency of equivalent lumped element resonator for given capacitance and inductance.'''
    return 1 / (2*np.pi * np.sqrt(ind*cap))


def le_imp(ind, cap):
    '''Returns resonator impedance of equivalent lumped element resonator for given capacitance and inductance.'''
    return np.sqrt(ind/cap)
  

def le_cpwresonator(width, spacing, length, ind_kin_sq, epsilon_r, ind_nw, cap_term):
    '''Returns inductance, capacitance, resonance frequency and resonator impedance of equivalent lumped element resonator
    for given CPW geometry (width, spacing, length),
    material properties (kinetic inductance, dielectric constant) and load (inductive, capacitice).'''

    # cpw resonator 
    k = fun_k(width, spacing)
    cap_m = cpw_cap_total(k, epsilon_r)
    ind_m = cpw_ind_total(k, ind_kin_sq, width)
    
    # conversion to lumped element resonator model
    ind = le_ind(ind_m, length, ind_nw)
    cap = le_cap(cap_m, length, cap_term)
    freq = le_freq(ind, cap)
    imp = le_imp(ind, cap)

    return np.array([ind, cap, freq, imp], dtype=object)
  
def le_qc(imp_le, cap_le, imp_fl, cap_c):
    '''Returns coupling quality factor for notch type resonator
    (lumped element resonator capacitively coupled to transmission line)
    for given transmission line impedance, coupling capacitance and lumped element resonator properties.'''
    q_c = 2* imp_le/imp_fl*(cap_le/cap_c)**2
    return q_c

def cap_coup(imp_le, cap_le, imp_fl, q_c):
    '''Returns coupling capacitance for notch type resonator
    (lumped element resonator capacitively coupled to transmission line)
    for given transmission line impedance, lumped element resonator properties and targeted coupling quality factor.'''
    return np.sqrt(2 * imp_le / (imp_fl * q_c)) * cap_le
