import numpy as np

import scipy.constants as pyc

from uncertainties import ufloat
import uncertainties.umath as umath
from uncertainties.umath import *
from uncertainties import unumpy


'''
Functions for numerical analysis of planar transmission line resonators 
based conformal mapping technique presented in D. M. Pozar, Microwave engineering (Fourth edition. Hoboken, NJ : Wiley, 2012)
'''


def fun_k(width, spacing):
    '''Returns geometrical factor for width central conductor and spacing to ground of transmission line'''
    return width / (width + 2 * spacing)


def cpw_cap_diel(k, epsilon_r):
    '''Returns capacitance per unit length of the central conductor to ground through the dielectric.
    Takes geometrical factor and dielectric constant.'''
    if k <= 1 / np.sqrt(2):
        kprime = np.sqrt(1 - k ** 2)
        f = np.pi / np.log(2 * (1 + np.sqrt(kprime)) / (1 - np.sqrt(kprime)))
    else:
        f = np.log(2 * (1 + np.sqrt(k)) / (1 - np.sqrt(k))) / np.pi

    return 2 * pyc.epsilon_0 * epsilon_r * f


def cpw_cap_total(k, epsilon_r):
    '''Returns total capacitance per unit length to ground taking air into account and assuming equal participation. '''
    return cpw_cap_diel(k, epsilon_r=1) + cpw_cap_diel(k, epsilon_r=epsilon_r)


def cpw_ind_geo(k):
    '''Returns geometric inductance per unit length of the central conductor.
    Takes geometrical factor.'''
    if k <= 1 / np.sqrt(2):
        kprime = np.sqrt(1 - k ** 2)
        f = np.pi / np.log(2 * (1 + np.sqrt(kprime)) / (1 - np.sqrt(kprime)))
    else:
        f = np.log(2 * (1 + np.sqrt(k)) / (1 - np.sqrt(k))) / np.pi

    return pyc.mu_0 / 4 / f


def cpw_ind_kin(ind_kin_sq, width):
    '''Returns kinetic inductance per unit length of the central conductor in zero order approximation.
    Takes kinetic sheet inductance per square and width of central conductor.'''
    return ind_kin_sq / width


def cpw_ind_total(k, ind_kin_sq, width):
    '''Returns total inductance per unit length as sum of geometric and kinetic inductance.'''
    return cpw_ind_geo(k) + cpw_ind_kin(ind_kin_sq, width)
  
  
def cpw_bare_resonator(width, spacing, length, ind_kin_sq, epsilon_r):
    '''Returns resonance frequency and characteristic impedance of a quarter-wavelength transmission line resonator
    for a given geometry (width, spacing, length) and material properties (kinetic inductance, dielectric constant).'''
    k = fun_k(width, spacing)
    cap_m = cpw_cap_total(k, epsilon_r)
    ind_m = cpw_ind_total(k, ind_kin_sq, width)

    freq = 1/(4*length*umath.sqrt(ind_m*cap_m))
    imp = umath.sqrt(ind_m/cap_m)

    return np.array([freq, imp], dtype=object)


def cpw_resonator_length(width, spacing, freq_nw, ind_kin_sq, ind_nw, epsilon_r):
    '''Returns required resonator length for a quarter-wavelength transmission line resonator
    for a given geometry (width, spacing),
    material properties (kinetic inductance, nanowire inductance, dielectric constant)
    and targeted resonance frequency (freq_nw).'''

    # properties of unloaded cpw resonator
    k = fun_k(width, spacing)
    cap_m = cpw_cap_total(k, epsilon_r)
    ind_m = cpw_ind_total(k, ind_kin_sq, width)
    imp0 = umath.sqrt(ind_m / cap_m)

    # based on characteristic impedance and target frequency, find the right resonator length
    f0 = freq_nw / (1 - 4 * freq_nw * ind_nw / imp0)
    length = 1 / (4 * f0 * np.sqrt(ind_m * cap_m))

    return length
