%% CHARACTERIZATION of SIDE CHANNELS
% This scripts represents a simple 1D bifurcation model which is based on
% Kleinhans et al. 2011 and Van Denderen et al. 2018. More detail is given
% in the publication (van Denderen et al. 2019: doi.org/10.1002/rra.3462)
%
% The code consits of three parts:
% - River characteristics (river, initial and boundary conditions)
% - function compute whcih is the main loop of the numerical model
% - List of functions which are all the functions used in the model
%
% Questions? r.p.vandenderen@utwente.nl

clear all;
%% river characteristics
    
    Qsyear=2E5;         % Yearly sediment supply of sand
    Qmyear=5.5E5;       % Yearly sediment supply of fines
    ws=7E-5;            % Fall velocity of fines (Middelkoop 1998 and Asselman 2002)

    % The following parameters are given for the:
    % [upstream channel; 
    %  main channel; 
    %  side channel; 
    % downstream channel]
    %
    % The following conditions represent the East channel at Gameren 
    Q=[2900; 2870;  30; 2900];              % [m^3/s]   -   initial discharge during backfull discharge (of each channel)
    k={0.5; 0.5; 0.2; 0.5};                 % [m]       -   bed roughness in each channel (note side channels are generally less rough)
    W={ 280;  280;  28; 280};               % [m]       -   Channel width of each channel
    D={ .75E-3;  .75E-3;  .3E-3; .75E-3};   % [m]       -   Grain size in each channel (note side channel grain size is generally finer)
    L={ 1000;340;440;2000};                 % [m]       -   Channel length
    ib={1E-4; 1E-4; 7.7273e-05; 1E-4;}; 	% [-]       -   Channel slope

    theta_c=0.035;                          % critical shields stress used for bed load transport
    
    
% % Side channel characteristics for suspended bed-material and suspended
% load dominant side channels.
% If size(variable)>1 then loop occurs over array
% %
    Qfrac_range=0.01; % discharge fraction
    L_range=440;      % Length side channel
    W_range=28;       % Width side channel
    k_range=0.2;      % Roughness side channel
    D_range=0.3E-3;   % Grain size side cahnnel
    dTendyr=100;      % number of years during which the discharge is constant before stopping computation

% % Side channel characteristics for bed load dominant side channels
%     Qfrac_range=0.1; % discharge fraction
%     L_range=440;     % Length side channel
%     W_range=28;      % Width side channel
%     k_range=0.5;     % Roughness side channel
%     D_range=0.75E-3; % Grain size side cahnnel
%     dTendyr=50;      % number of years during which the discharge is constant before stopping computation


    wh_range=logspace(0,2,35);
    L_range=linspace(340,2*340,35);
% 
    name='results_L_wh_susbed'; % name of mat file created
    sedtype='susbed'; % bed, susbed or sus
    disp(name)
    report=cell(length(L_range),length(Qfrac_range),length(W_range),length(k_range),length(D_range),length(wh_range));
    parfor ii=1:length(L_range)
        tmp1=cell(length(Qfrac_range),length(W_range),length(k_range),length(D_range),length(wh_range));
        for jj=1:length(Qfrac_range)
            for kk=1:length(W_range)
                for ll=1:length(k_range)
                    for mm=1:length(D_range)
                        for nn=1:length(wh_range)
                            k_tmp=k;
                            k_tmp{3}=k_range(ll);
                            D_tmp=D;
                            D_tmp{3}=D_range(mm);

                            if length(wh_range)>1
                                wh_val=wh_range(nn);
                            else
                                wh_val=[];
                            end

                            tmp=compute([Q(1,1); (1-Qfrac_range(jj))*Q(1,1); (Qfrac_range(jj))*Q(1,1); Q(4,1)],...
                                                     k_tmp,...
                                                     {W{1,1}; W{2,1}; W_range(kk); W{4,1}} ,...
                                                     D_tmp,...
                                                     {L{1,1}; L{2,1}; L_range(ii); L{4,1}},...
                                                     {ib{1,1}; ib{2,1}; ib{2,1}/(L_range(ii)./L{2,1}); ib{4,1}},...
                                                     wh_val,...
                                                     Qsyear,Qmyear,ws,{sedtype},dTendyr,theta_c);
                            tmp1{jj,kk,ll,mm,nn}=tmp;
                        end
                    end
                end
            end
            
        end
        report(ii,:,:,:,:,:)=tmp1
        disp(num2str(ii))
    end
  
    
    % inicond.
    inicond=struct('Qsyear',Qsyear,'Qmyear',Qmyear,'ws',ws,'Q',Q,'k',k,'W',W,'D',D,'L',L,'ib',ib);
    parsave(['results\' name '\' name '_1.mat'],report,Qfrac_range,L_range,W_range,k_range,D_range,wh_range,dTendyr,theta_c,inicond)

function parsave(filename,report,Qfrac_range,L_range,W_range,k_range,D_range,wh_range,dTendyr,theta_c,inicond)
    save(filename,'report','Qfrac_range','L_range','W_range','k_range','D_range','wh_range','dTendyr','theta_c','inicond','-v7.3')
end


    %%
    % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% %
    %                                                                   %  
    %                                                                   %
    %                                                                   %
    %               !!!!!!!! do not touch below !!!!!!!!!               %
    %                                                                   %
    %                                                                   %
    %                                                                   %
    % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% %
function report=compute(Q,k,W,D,L,ib,wh_val,Qsyear,Qmyear,ws,transtype,dTendyr,theta_c)

    g=9.81;                             % gravitational acceleration
    delta=1.65;                         % Relative density sediment
    C_18=(sqrt(g)/log10(exp(1))/0.4);   % Darcy weibach parameter (about 18)
    C_12=12.2;                          % Darcy weibach parameter (about 12)
    dt=0.005 *365.25*24*3600;           % Time step
    por=0.3;                            % Sediment porosity
    
        
    % nodal point variables
    bol_alpha=2;                        % bol_alpha*W_upstream is the distance upstream over which tranvserse sediment flux occurs. Alpha parameter in Bolla Pittaluga 2003 and Kleinhans 2008
    bol_r=1.2;                          % parameter in the transver bed slope effect BollaPittaluga (2003)
    
    % solving parameters
    hpres= 1E-20;                       % water level precision in the discharge partitioning iteration
    options=optimset('TolX',hpres);     % options for the discharge partitioning iteration

    % set logicals
    logback=[true;true;true;true];      % true= solve using backwater, false= uniform channel depth approach
    
    transload=logical(zeros(3,1));      % logical, which type of sediment partioning and capacity should be used
    if any(strcmp(transtype,'bed'))     % uses Kleinhans 2008
        transload(1)=true;
    end
    if any(strcmp(transtype,'susbed'))  % uses Wang with k=1
        transload(2)=true;
    end
    if any(strcmp(transtype,'sus'))     % uses Wang with k=1 and Middelkoop (1998)
        transload(3)=true;
    end
    
    % Set the dx in the channels and other initial values. 
    dx=zeros(length(L),1);
    xid=zeros(length(L),1);
    Qs=zeros(length(L),1);
    htst=cell(length(L),1);
    for ii=1:length(L)
        dx(ii,1)=L{ii,1}/20; %dx is different in each channel
        if ii==3
            dx(ii,1)=10;
        end
    end
    xi0=0;


    % Get initial uniform water depth to set the inital bed level in the
    % channels.
    [h,C,W]=solve_h_ini_b(Q,k,W,ib,xi0,L,wh_val,g,C_18,C_12,options);
    
    % create grid, downstream water levels of each channel and the bed
    % level of each channel
    for ii=length(h):-1:1
        if logback(ii,1)
            x{ii,1}=0:dx(ii,1):L{ii,1};
        else
            x{ii,1}=0;
        end
        if ii==4
            xid(ii,1)=xi0;
        elseif any(ii==2:3)
            xid(ii,1)=xi0+ib{4,1}*L{4,1};
        else
            xid(ii,1)=xi0+ib{4,1}*L{4,1}+ib{2,1}*L{2,1};
        end

        if length(x{ii,1})>1
            eta{ii,1}=xid(ii,1)-h{ii,1}+ib{ii,1}*fliplr(x{ii,1}(1,1:(end-1))+diff(x{ii,1})/2);
        else
            eta{ii,1}=xid(ii,1)-h{ii,1};
        end

        h{ii,1}=zeros(1,length(x{ii,1})) +h{ii,1};  
    end

    % Initial estimate of the sediment transport to compute the
    % 'intermittiency factor'(see VanDenderen et al. 2018)
    if transload(1)
        [S, u]=meyerpeter(Q,D,W,h,theta_c,C,g,delta,C_18,C_12);
    else
        [S, u]=engelund(Q,D,W,h,C,g,delta);
    end
    If=Qsyear./(S{1,1}(1,1)*3600*24*365.25);
    Qs(1,1)=S{1,1}(1,1);    % Set upstream sediment supply of upstream channel (equilibrium condition)
    T=0;
    branch=1:4;
    lognum=branch(logback);

    % T=0, save initial output paramets
    report.eta=eta{3,1}(1,1); % average bed level side channel
    report.etam=mean(eta{3,1}(1,:)); % average bed level side channel
    report.h=h{3,1}(1,1);
    report.hm=mean(h{3,1}(1,:));
    report.C=C{3,1}(1,1);
    report.ib=ib{3,1}(1,1);
    report.S=S{3,1}(1,1);
    report.Q=Q;
    report.W=W{3,1}(1,1);
    report.D=D{3,1}(1,1);
    report.L=L{3,1}(1,1);
    report.k=k{3,1}(1,1);
    report.If=If;
    report.T=T;
    report.Qs=zeros(4,1);
    if transload(3)
        report.E   =NaN;
        report.SE  =NaN;
        report.deta=0;
        report.cumdeta=0;
    end

    %% computation
    p=2;    % track the number of reported values
    n=2;    % track the number of timesteps

    Tend=1E6*3600*24*365.25;
    dTend=dTendyr*3600*24*365.25;
    Tstop=0;
    while p==2||T<=Tend
        n=n+1;

        % compute waterdepth downstream channel
        if any(lognum==4)
            [h{4,1}, C{4,1}]=solve_h_back(Q(4,1),k{4,1},W{4,1},ib{4,1},eta{4,1},x{4,1},xid(4,1),g,C_18,C_12);
        else
            [h(4,1), C(4,1), htst(4,1)]=solve_h_uni(Q(4,1),k(4,1),W(4,1),ib(4,1),eta(4,1),xid(4,1),L(4,1),h(4,1),C_18,C_12,options);
        end
        % set the water level at the confluence
        for ii=3:-1:2
            if logback(ii)
                if logback(4)
                    xid(ii,1)=eta{4,1}(1,1)+ib{4,1}*dx(ii,1)/2+h{4,1}(1,1);
    %                 xid(ii,1)=interp1((x{4,1}(1,1:(end-1))+diff(x{4,1})/2),eta{4,1},0,'linear','extrap')+h{4,1}(1,1);
                else
                    xid(ii,1)=xi0-h{4,1}+ib{4,1}*L{4,1}+htst{4,1};
                end
            end
        end

        % compute the discharge partitioning
        [Q, fval]=compute_Q(Q,k,W,ib,eta,x,xid,L,h,lognum,g,C_18,C_12,options);
        
        % end condition if one of the channels closes or when no solution
        % is found for the iteration
        if any(Q<.1)||fval>10||any(isnan(fval))||(T>(dTend+dt)&&(max(abs(report.Q(3,((p-(dTendyr/0.05)):end))-Q(3,1)))<(0.0001*Q(3,1))))||(transload(3)&&~transload(2)&&Tstop)
            report.eta(:,p)=eta{3,1}(1,1); % average bed level side channel
            report.etam(:,p)=mean(eta{3,1}(1,:)); % average bed level side channel
            report.h(:,p)=h{3,1}(1,1);
            report.hm(:,p)=mean(h{3,1}(1,:));
            report.C(:,p)   =C{3,1}(1,1);
            report.S(:,p)   =S{3,1}(1,1);
            report.Q(:,p)   =Q;
            report.Qs(:,p)  =Qs;
            if transload(3)
                if p==2
                    Sm=0;
                    SE=0;
                    deta=0;
                    cumdeta=0;
                    taumin=0;
                end
                report.E(:,p)   =((Sm(1,1)-Sm(1,end))./Sm(1,1));
                report.SE(:,p)  =mean(SE);
                report.deta(:,p)=mean(deta);
                report.cumdeta(:,p)=report.cumdeta(:,p-1)+cumdeta;
                report.Tstop=Tstop;
                report.taumin=taumin;
            end
            report.T(1,p)   =T/3600/24/365.25;
            report.dTend    =dTend;
            p=p+1;
            break
        end

        % compute the water level at the bifurcation
        for ii=3:-1:1
            if ii==1
                if logback(ii)
                    if logback(2)
                        xid(ii,1)=eta{2,1}(1,1)+ib{2,1}*dx(ii,1)/2+h{2,1}(1,1);
                    elseif logback(3)
                        xid(ii,1)=eta{3,1}(1,1)+ib{3,1}*dx(ii,1)/2+h{3,1}(1,1);
                    else
                        xid(ii,1)=eta{2,1}(1,1)+ib{2,1}*L{2,1}+htst{2,1};
                    end
                end
            end
            
            % solve water depth in the upstream channel
            if logback(ii)
                [h{ii,1}, C{ii,1}]=solve_h_back(Q(ii,1),k{ii,1},W{ii,1},ib{ii,1},eta{ii,1},x{ii,1},xid(ii,1),g,C_18,C_12);
            else
                [h(ii,1), C(ii,1)]=solve_h_uni(Q(ii,1),k(ii,1),W(ii,1),ib(ii,1),eta(ii,1),xid(ii,1),L(ii,1),h(ii,1),C_18,C_12,options);
            end
        end

        % compute sediment trapping of fines (Middelkoop 1998 or Asselman 2002)
        deta=0;
        if transload(3)
            [E,SE,Sm, deta,cumdeta,Tstop,taumin]=compute_trapping(Q,W{3,1},h{3,1},u{3,1},C{3,1},x{3,1},ws,Qmyear,dt);
            dV={};
        end
        
        % compute sediment transport and bed level changes for bed load and
        % suspended bed-material load
        if any(transload(1:2))
            if transload(1)
                [S, u]=meyerpeter(Q,D,W,h,theta_c,C,g,delta,C_18,C_12);
            else
                [S, u]=engelund(Q,D,W,h,C,g,delta);
            end
            Qs=nodal(Q,D,W,h,S,C,ib,eta,u,x,Qs,L,delta,bol_alpha,bol_r,transload);
            dV=dQsdx(Qs,S,x,L);
            
        end
        
        % check if the courant number of the bed wave celerity is smaller
        % than 0.5 in all channels.
        if ~isempty(lognum)&&any(transload(1:2))
            for ii=lognum
                cs=compute_bedwave(W{ii,1},h{ii,1},u{ii,1},S{ii,1},If,por,g);
                sigma=max(cs*dt/dx(ii,1));
                if sigma>0.8
                    disp('bedwave prob')
                   warning(['The courant number is ' num2str(sigma) '. Try dt=' num2str(0.5*dx(ii,1)/max(cs)/365.25/24/3600)])
                    break
                   
                end
            end
        end

        % bed level changes
        [eta, ib]=update_bed(dV,eta,W,If,dt,ib,por,deta);

        % save the variables to report if the channel is not yet closed
        T=T+dt;
        if (T>=((report.T(1,p-1)+0.05)*3600*24*365.25))||abs(T-(report.T(1,p-1)+0.05)*3600*24*365.25)<(dt/100)
            report.eta(:,p)=eta{3,1}(1,1); % average bed level side channel
            report.etam(:,p)=mean(eta{3,1}(1,:)); % average bed level side channel
            report.h(:,p)=h{3,1}(1,1);
            report.hm(:,p)=mean(h{3,1}(1,:));
            report.C(:,p)   =C{3,1}(1,1);
            report.S(:,p)   =S{3,1}(1,1);
            report.Q(:,p)   =Q;
            report.Qs(:,p)  =Qs;
            if transload(3)
                report.E(:,p)   =((Sm(1,1)-Sm(1,end))./Sm(1,1));
                report.SE(:,p)  =mean(SE);
                report.deta(:,p)=mean(deta);
                report.cumdeta(:,p)=report.cumdeta(:,p-1)+cumdeta;
            end
            report.T(1,p)   =T/3600/24/365.25;
            p=p+1;
        end

    end
end

%% Functions

%% initial water depth and bed level using backwater effect

function [h,C,W]=solve_h_ini_b(Q,k,W,ib,xi0,L,wh_val,g,C_18,C_12,options)
    h=cell(4,1);
    for ii=[4 2 3 1]
        if any(ii==[4,2,1])
            [h(ii,1), tmp]=solve_h_uni(Q(ii),{k{ii}},{W{ii}},{ib{ii}},{},{},{L{ii,1}},{5},C_18,C_12,options);
            C{ii,1}=tmp{1};
            if ii==4
                eta{ii,1}=xi0-h{ii};
            elseif ii==1
                xid(1,1)=eta{3,1}(1)+h{3,1}(1);
                eta{ii,1}=xid(ii,1)-h{ii};
            else
                xid(2,1)=xi0+ib{4,1}*L{4,1};
                eta{ii,1}=xid(ii,1)-h{ii};
            end
        elseif 3
            x{ii,1}=0:L{ii,1}/20:L{ii,1};
            xid(ii,1)=xi0+ib{4,1}*L{4,1};
            
            if isempty(wh_val)
                [htmp, tmp]=solve_h_uni(Q(ii),{k{ii}},{W{ii}},{ib{ii}},{},{},{L{ii,1}},{5},C_18,C_12,options);
                C{ii,1}=tmp{1};
                hlim=[0 100];
                dh=100;
                while abs(dh)>1E-10
                    eta{ii,1}=xid(ii,1)-htmp{1}+ib{ii,1}*fliplr(x{ii,1}(1,1:(end-1))+diff(x{ii,1})/2);
                    h{ii,1}=solve_h_back(Q(ii,1),k{ii,1},W{ii,1},ib{ii,1},eta{ii,1},x{ii,1},xid(ii,1),g,C_18,C_12);
                    dh=(xi0+ib{4,1}*L{4,1}+ib{2,1}*L{2,1})-(eta{3,1}(1)+ib{3,1}*diff(x{3,1}(1,1:2))/2+h{3,1}(1));

                    if dh>0
                        hlim(2)=htmp{1};
                    else
                        hlim(1)=htmp{1};
                    end
                    htmp{1}=mean(hlim);
                end
            else
                Wlim=[0 1000];
                wh_test=0;
                W3tst=W{ii};
                while abs(wh_val-wh_test)>1E-4
                    [htmp, tmp]=solve_h_uni(Q(ii),{k{ii}},{W3tst},{ib{ii}},{},{},{L{ii,1}},{5},C_18,C_12,options);
                    C{ii,1}=tmp{1};
                    hlim=[0 100];
                    dh=100;
                    while abs(dh)>1E-10
                        eta{ii,1}=xid(ii,1)-htmp{1}+ib{ii,1}*fliplr(x{ii,1}(1,1:(end-1))+diff(x{ii,1})/2);
                        h{ii,1}=solve_h_back(Q(ii,1),k{ii,1},W3tst,ib{ii,1},eta{ii,1},x{ii,1},xid(ii,1),g,C_18,C_12);
                        dh=(xi0+ib{4,1}*L{4,1}+ib{2,1}*L{2,1})-(eta{3,1}(1)+ib{3,1}*diff(x{3,1}(1,1:2))/2+h{3,1}(1));

                        if dh>0
                            hlim(2)=htmp{1};
                        else
                            hlim(1)=htmp{1};
                        end
                        htmp{1}=mean(hlim);
                    end
                    wh_test=W3tst./htmp{1};
                    
                    if wh_test>wh_val
                        Wlim(2)=W3tst;
                    else
                        Wlim(1)=W3tst;
                    end
                    W3tst=mean(Wlim);
                end
                W{3,1}=W3tst;
                h{3,1}=htmp{1};
            end
        end


    end

end


%% hydrodynamic functions

function [h,C,htst]=solve_h_uni(Q,k,W,ib,eta,xid,L,h_old,C_18,C_12,options)
    % Computes the uniform water depth in each channel based on the
    % discharge (Q) the roughness (k), the channel width (W), the
    % equilibrium bed slope (ib), the bed level (eta) (if eta is non empty
    % then an estimation of the backwater is used), the downstream water
    % depth (xid), the channel length (L), h_old= first estimate.
    %
    %
    h=cell(length(Q),1);
    C=cell(length(Q),1);
    htst=cell(length(Q),1);
    for ii=1:length(Q)
        if Q(ii,1)>0.1
            fun=@(X)h_solver_uni(X,Q(ii,1),W{ii,1},k{ii,1},ib{ii,1},C_18,C_12);
            %h_old{ii,1}
            [h{ii,1}]=fzero(fun,h_old{ii,1},options);
        else 
            h{ii,1}=-1E10;
        end
        C{ii,1}=C_18*log10(C_12*(W{ii,1}.*h{ii,1})./(W{ii,1}+2.*h{ii,1})./k{ii,1});
        
        if ~isempty(eta)
            if h{ii,1}>0
                hc=((Q(ii,1)/W{ii,1})^2/9.81)^(1/3);
                h0=xid(ii,1)-eta{ii,1};

                Ladj=(h{ii,1}^3-hc^3)/h{ii,1}^2/ib{ii,1};
                htst{ii,1}=h{ii,1}+(h0-h{ii,1})*exp(3*(-1*L{ii,1})/Ladj);
            else
                htst{ii,1}=h{ii,1};
            end
        end
    end
    
end

function dQ=h_solver_uni(h,Q,W,k,ib,C_18,C_12)
    % Solves the discharge partitioning given the water depth if a uniform
    % channel depth is assumed
    
    if h>0.1
        R=(W.*h)./(W+2.*h);
        C=C_18*log10(C_12.*R./k);

        htst=Q./W./C./sqrt(R.*ib);
        dQ=h-htst;
    else
        dQ=-1E10;
    end
end

function [hval,Cval]=solve_h_back(Q,kval,Wval,ib,etval,x,xid,g,C_18,C_12)
    % Solves the water depth in the branch based on a backwater relation.
    %
    nx=length(x);
    hval=zeros(1,nx);
    Cval=zeros(1,nx);
    if Q>0
        dx=(x(2:end)-x(1:(end-1)));
        etaL=etval(1,end)-ib*dx(end)/2;

        hval(1,nx)=xid-etaL;
        R=(Wval*hval(1,nx))./(Wval+2*hval(1,nx));
        Cval(1,nx)=C_18*log10(C_12*R./kval);
        utmp=Q./Wval./hval(1,nx);

        eta0=etval(1,1)+ib*dx(1)/2;
        dzdx=([eta0, etval(1:(end-1))]-etval)./dx;

        for jj=(nx-1):-1:1    
            Fr=utmp./sqrt(hval(1,jj+1)*g);
            if Fr>=1||(Fr~=real(Fr))
                hval(1,1:nx)=1E10;
                break
            end

            F1=(dzdx(1,jj)-Fr.^2*g/Cval(1,jj+1)^2)./(1-Fr.^2);
            htmp=hval(1,jj+1)-F1*dx(jj);
            utmp=Q./Wval./htmp;

            R=(Wval*htmp)./(Wval+2*htmp);
            Ctmp=C_18*log10(C_12*R./kval);
            Fr=utmp./sqrt(htmp*g);
            if Fr>=1||(Fr~=real(Fr))
                hval(1,1:nx)=1E10;
                break
            end
            F2=(dzdx(1,jj)-Fr.^2*g/Ctmp^2)./(1-Fr.^2);

            hval(1,jj)=hval(1,jj+1)-0.5*(F1+F2)*dx(jj);
            R=(Wval*hval(1,jj))./(Wval+2*hval(1,jj));
            Cval(1,jj)=C_18*log10(C_12*R./kval);
            utmp=Q./Wval./hval(1,jj);
        end
    else
        hval(1,1:nx)=-1E10;
    end
end

function [Q, fval]=compute_Q(Q,k,W,ib,eta,x,xid,L,h_old,lognum,g,C_18,C_12,options)
    % Solve the discharge paritioning at the bifurcation using the
    % backwater relation.
    %
    if length(x{2,1})>1
        eta2=eta{2,1}(1,1)+ib{2,1}*diff(x{2,1}(1,1:2))/2;
    else
        eta2=eta{2,1}+ib{2,1}*L{2,1};
    end
    if length(x{3,1})>1
        eta3=eta{3,1}(1,1)+ib{3,1}*diff(x{3,1}(1,1:2))/2;
    else
        eta3=eta{3,1}+ib{3,1}*L{3,1};
    end
    fun=@(X)Q_solver(X,Q,k,W,ib,eta,eta2,eta3,x,xid,L,h_old,lognum,g,C_18,C_12,options);
    [Q2,fval]=fzero(fun, Q(2,1),options);
    if fval>10||isnan(fval)
        return 
    end
    Q(2,1)=Q2;
    Q(3,1)=Q(1,1)-Q2;
end

function dxi=Q_solver(Q2,Q,k,W,ib,eta,eta2,eta3,x,xid,L,h_old,lognum,g,C_18,C_12,options)
    % function used in the fzero of compute_Q that used backwater relations
    % to compute the water level difference at the bifurcation.
    %
    Q(2,1)=Q2;
    Q(3,1)=Q(1,1)-Q2;

    h=cell(length(Q),1);
    for ii=2:3
        if any(lognum==ii)
            h{ii,1}=solve_h_back(Q(ii,1),k{ii,1},W{ii,1},ib{ii,1},eta{ii,1},x{ii,1},xid(ii,1),g,C_18,C_12);
        else
            [~,~,h(ii,1)]=solve_h_uni(Q(ii,1),k(ii,1),W(ii,1),ib(ii,1),eta(ii,1),xid(ii,1),L(ii,1),h_old(ii,1),C_18,C_12,options);
        end
    end
    
    dxi=(eta2+h{2,1}(1,1))-(eta3+h{3,1}(1,1));
    
end

%% sediment transport
function [S, u]=engelund(Q,D,W,h,C,g,delta)
    % Sediment transport and flow velocity following engelund hansen

    S=cell(length(Q),1);
    u=cell(length(Q),1);
    for ii=1:length(h)
        u{ii,1}=Q(ii,1)./W{ii,1}./h{ii,1};
        S{ii,1}=0.05*u{ii,1}.^5./(sqrt(g)*C{ii,1}.^3.*delta^2.*D{ii,1}).*W{ii,1};
    end
end

function [S, u]=meyerpeter(Q,D,W,h,theta_c,C,g,delta,C_18,C_12)
    % Sediment transport and flow velocity following meyer peter

    S=cell(length(Q),1);
    u=cell(length(Q),1);
    for ii=1:length(h)
        u{ii,1}=Q(ii,1)./W{ii,1}./h{ii,1};
        Cs=C_18.*log10(C_12.*h{ii,1}/3/0.002586915582003);
        mu=(C{ii,1}./Cs).^(3/2);
        psi=mu.*u{ii,1}.^2./C{ii,1}.^2/delta/D{ii,1};
        phi=zeros(1,length(h{ii,1}));
        phi(psi>theta_c)=8*(psi(psi>theta_c)-theta_c).^(3/2);
        S{ii,1}=phi.*D{ii,1}^(3/2)*sqrt(g*delta)*W{ii,1};
    end
end


%% Sediment functions
%
function dV=dQsdx(Qs,S,x,L)
    % Compute the volume change in sediment in each grid cell
    dV=cell(length(S),1);
    for ii=1:length(S)
        if length(x{ii,1})>1
            dV{ii,1}=-1*diff([Qs(ii,1),S{ii,1}(1,2:end)])./diff(x{ii,1});
        else
            dV{ii,1}=(Qs(ii,1)-S{ii,1})./L{ii,1};
        end
    end
end

function [eta, ib]=update_bed(dV,eta,W,If,dt,ib,por,deta)
    % Compute the bed level change and update the bed level based on the
    % volume  change and the trapping of sediment
    if ~isempty(dV)
        for ii=1:length(dV)
            if ii~=3
                eta{ii,1}=eta{ii,1}+(dV{ii,1}./W{ii,1}./(1-por)).*(If.*dt);
            else
                eta{ii,1}=eta{ii,1}+(dV{ii,1}./W{ii,1}./(1-por)).*(If.*dt)+deta;
            end
        end
    else
        eta{3,1}=eta{3,1}+deta;
    end
end

function cs=compute_bedwave(W,h,u,S,If,por,g)
    % Compute the bed wave celerity
    Fr=u./sqrt(g*h);
    cs=If.*S./W.*5./(1-por)./(1-Fr.^2)./h;
end


%% nodal point relation
function Qs=nodal(Q,D,W,h,S,C,ib,eta,u,x,Qs,L,delta,bol_alpha,bol_r,transload)
    % Switch for the nodal point relation. 
    
    if any(transload(2:3))       
        Qs=wang(Q,S,Qs);
    end
    if transload(1)
        Qs=kleinhans(Q,D,W,h,S,C,ib,eta,u,x,Qs,L,delta,bol_alpha);
    end

end

function Qs=wang(Q,S,Qs)
    % wang with k = 1
    Qs(2,1)=Q(2,1)./Q(1,1)*S{1,1}(1,end);
    Qs(3,1)=S{1,1}(1,end)-Qs(2,1);
    
    Qs(4,1)=S{2,1}(1,end)+S{3,1}(1,end);
end

function Qs=kleinhans(Q,D,W,h,S,C,ib,eta,u,x,Qs,L,delta,bol_alpha)
    % The nodal point relation using  Kleinhans (2008;2011)
    
    if length(x{2,1})>1
        eta2=eta{2,1}(1,1)+ib{2,1}*diff(x{2,1}(1,1:2))/2;
    else
        eta2=eta{2,1}+ib{2,1}*L{2,1};
    end
    if length(x{3,1})>1
        eta3=eta{3,1}(1,1)+ib{3,1}*diff(x{3,1}(1,1:2))/2;
    else
        eta3=eta{3,1}+ib{3,1}*L{3,1};
    end
    
    Qy=0.5*(Q(2,1)-Q(3,1)-Q(1,1)*(W{2,1}(1,1)-W{3,1}(1,1))/(W{2,1}(1,1)+W{3,1}(1,1)));
    dzdy=(eta2-eta3)./0.5/W{1,1}(1,end);
    theta=(u{1,1}(1,end))^2/C{1,1}(1,end)^2/delta/D{3,1}(1,end);
    %talmon 1995
    ftheta = 9*(D{1,1}/h{1,1}(1,end))^0.3 *theta^0.5;

    v=Qy/(h{1,1}(1,end)*bol_alpha*W{1,1}(1,end));
    a_delta=atan(v/u{1,1}(1,end));
    
    tanalpha=(sin(a_delta) - dzdy/ftheta)/...
             (cos(a_delta) + ib{1,1}(1,end)/ftheta);
    qsy=S{1,1}(1,end)/W{1,1}(1,end)*tanalpha;
    Qsy=qsy*bol_alpha*W{1,1}(1,end);
    
    Qs(2,1)=max(Qsy + W{2,1}(1,1)/(W{2,1}(1,1)+W{3,1}(1,1))*S{1,1}(1,end),0);
    Qs(3,1)=S{1,1}(1,end)-Qs(2,1);
    
    if Qs(3,1)<0
        Qs(3,1)=0;
        Qs(2,1)=S{1,1}(1,end);
    end
        
    Qs(4,1)=S{2,1}(1,end)+S{3,1}(1,end);

end

%% mud trapping

function [E,SE,Sm, deta,cumdeta,Tstop,taumin]=compute_trapping(Q,W,h,u,C,x,ws,Qmyear,dt)
    % Mud trapping based on Middelkoop (sediflux) or Asselman (exponential
    % function)
    %
    dx=(x(2:end)-x(1:(end-1)));
    uc=u(1,1:(end-1))+u(1,2:end)/2; % Flow velocity in cell center
    hc=h(1,1:(end-1))+h(1,2:end)/2; % Water depth in cell center
    Cc=C(1,1:(end-1))+C(1,2:end)/2; % Chezy in cell center
    tau=1000*9.81*uc.^2./Cc.^2;     % Bed shear stress in cell cente
    tc=2;                           % Critical bed shear stress.
    
    %% Assleman
    E=1-exp(-dx./uc*ws./hc);   
    E(tau>tc)=0;
    
    Sm=zeros(1,length(h));
    SE=zeros(1,length(dx));
    Sm(1,1)=Qmyear/(365.25*24*3600)*Q(3,1)/Q(1,1);
    for ii=1:length(dx)
        SE(1,ii)=E(1,ii)*Sm(1,ii);
        Sm(1,ii+1)=Sm(1,ii)-SE(1,ii);
    end

    deta=(SE*dt/W./dx)./(1-0.3);   % bed level change per cell per time step
    cumdeta=mean(deta); % average bed level change
    Tstop=all(tau>tc);
    taumin=min(tau);
end
