function [rms,Mphi,omega,SpecIn] = DLGpre
% This script is to generate a few basic files required later in DLG
% based on the input files and data.
%
% Input:    CaseParameters.dat
%           TransferFunction.dat
%           InputSpectrum.dat (When SpecType in CaseParameters.dat is 0)
%      
% Output:   a.dat
%           a_excitation.dat
%           RunTimeStats.dat
%           TransferPhase.dat
%           omega.dat
%
% July 31, 2011
% By Daehyun Kim   

           
% Update History ----------------------------------------------------------
%
% Aug.19, 2011
%
% 1. If the gravity constant typed is between 31 and 33, the input spectrum
% will be converted to ft^2-sec/rad.   
% 2. If the gravity constant is less than 9.5, between 10.5 and 31, and
% bigger than 33, the error message will pop up.
% 3. N_narrow = exp(TEV^2/2) will be included in RunTimeStat.dat

% Aug.22, 2011
%
% 1. TEV<4.5 limitation has been removed for the use with DLG_TF

% April.05, 2012
%
% 1. This version of DLG no longer follows the previous version of DLG.
%    Some of the input parameters are deleted or changed into new ones.
% 2. Input files should be stored in \Inputs. In addition, the output files
%    (except RunTimeStats.dat) will also be saved in \Inputs, too. 
%    The rationale for this change is those output files are still the 
%    input files of the whole DLG process.
% 3. The name of this function has been changed from DLG0.m to DLGpre.m

% Jan. 01, 2013
% Upgraded for the multidirectional DLG
% 
% 1. to make it work for both unidirectional and multidirectional DLG. 
% 2. InputSpectrum.dat should be generated outside of DLG.
%
% Input:    CaseParameters.dat
%           TransferFunction.dat
%           InputSpectrum.dat (When SpecType in CaseParameters.dat is 0)
%           WaveHeadings.dat  <- has been added.

clear all;close all;clc
set(0,'defaulttextinterpreter','latex')

tic



% Read in CaseParameters.dat

temp_filename=fullfile('Inputs','CaseParameters.dat'); 
[data, ~] = textread(temp_filename,'%f %s','headerlines',0);

NumCompIn = data(1)
N = data(2);
io = data(3);
fo = data(4);
TEV = data(6);
CaseNumber = data(7);
Heading = data(8);
Gravity = data(10);

[Char_CaseNo]=generateConsistentCharNumber(CaseNumber,3);

% No matter what happens, this should be the final omega ==================
omega= linspace(io,fo,N);
% =========================================================================

% if NumCompIn < N
%     disp({'Input Data will be interpolated according to NumComponentsIn'})
% end

%disp error message if io and fo are outside of the frequency range given 
%in InputSpectrum or transferFunction.

if TEV > 7.5
   disp('TEV = 7.0 corresponds to the exposure time of about 13848 years when the period of one cycle is 10 sec.');
   disp(' ');
   error('Error: The limit of TEV is set to 7.5.')
end	

if length(num2str(CaseNumber)) > 3
   error('Error: Use a number between 1 ~ 999 for CaseNumber.');
end	
    
if Gravity>31 && Gravity<33
   disp('You seem to be using the Britsh Units.'); 
   disp('Make it sure that the second column of InputSpectrum.dat is given in [ft^2-sec/rad].'); 
   disp(' ')
elseif Gravity>9.5 && Gravity<10.5  % default: MKS unit
   disp('You seem to be using the Metric Units.'); 
   disp('Make it sure that the second column of InputSpectrum.dat is given in [m^2-sec/rad].'); 
   disp(' ')
else
   error('Error: The unit of the gravity constant you typed in is not recognized. Make it sure you don''t have a typo.'); 
end
    
if Heading == 1000       % Secret # may be used
    temp_filename=fullfile('Inputs','WaveHeadings.dat');
    WaveHeadings=load(temp_filename);
    NumWaveHeadings=length(WaveHeadings);
    d_theta=(WaveHeadings(2)-WaveHeadings(1))*pi/180;
else
    NumWaveHeadings=1;
    WaveHeadings=Heading;
    d_theta=1;
end

ResponseVAR=0;
WaveVAR=0;

for k=1:NumWaveHeadings
        
        if NumWaveHeadings ==1    
          k=[];  % not a good way of doing the job, but it will do the work. 
        end
        
        temp_filename=fullfile('Inputs',strcat('TransferFunction',num2str(k),'.dat'));
        TransferFunction = load(temp_filename);
        
        temp_filename=fullfile('Inputs',strcat('InputSpectrum',num2str(k),'.dat'));
        SpecInRAW = load(temp_filename);
    
        Specomega=SpecInRAW(:,1)';
        SpecIn=SpecInRAW(:,2)';
        
        if Specomega(1) > omega(1) || Specomega(end) < omega(end)
            error('At least one of the input spectrum files does not cover the entire frequency range defined by MinimumOmega and MaximumOmega.')
    
        end
        
        if length(TransferFunction) ~= length(SpecInRAW)
            error('The number of wave frequency in InputSpectrum.dat is different from that of the supplied transfer function(s).');

        end
    
        if SpecInRAW(1,1) ~= TransferFunction(1,1) || SpecInRAW(end,1) ~= TransferFunction(end,1)
            error('The range of wave frequency in InputSpectrum.dat is different from that of the supplied transfer function(s).');
        end
%     
%         figure
%         subplot(211);plot(TransferFunction(:,1),TransferFunction(:,2),'-*')
%         subplot(212);plot(TransferFunction(:,1),TransferFunction(:,3),'-*')

        %% Do the interpolation if necessary...

        if length(TransferFunction) ~= NumCompIn
            error('NumcomponentsIn is different from the number of wave frequency in the supplied transfer function(s).')
        end

        Transomega=TransferFunction(:,1);
        TransFuncAmp=TransferFunction(:,2);
        TransFuncPhase=TransferFunction(:,3);  % should be in degree

        if length(Transomega) ~= NumCompIn
            error('Error: The number of wave frequency in TransferFunction.dat is different from NumComponentsIn.')
        end
        

        if omega(1) ~= TransferFunction(1,1) || omega(end) ~= TransferFunction(end,1)
            disp('The range of frequency in TransferFunction.dat is different from the range defined by ''io'' and ''fo''.');
            disp('The data in the TransferFunction.dat will be linearly interpolated according to ''io'', ''fo'', if necessary.');
            disp(' ')
        end

        if NumCompIn ~= N  
            disp('NumComponentsOut is different from NumComponentsIn.'); 
            disp('The interpolation has been carried out based on NumComponentsOut.');
            disp(' ')

            Transomega_interp = linspace(io,fo,N);

            TransFuncAmp_interp = interp1(Transomega,TransFuncAmp,Transomega_interp);
            TransFuncPhase_interp = interp1(Transomega,unwrap(TransFuncPhase*pi/180)*180/pi,Transomega_interp);

            TransFuncAmp = TransFuncAmp_interp';
            TransFuncPhase = TransFuncPhase_interp';

            SpecIn_interp = interp1(SpecInRAW(:,1), SpecInRAW(:,2),Transomega_interp);
            SpecIn = SpecIn_interp;
   
            omega = Transomega_interp;
            
        end

%         figure;hold on
%         plot(omega,SpecIn,'ro');grid on
%         xlabel('\omega [rad/sec]','FontSize',12);ylabel('S_{in}','FontSize',12)
%         legend('Input','Output')

%        figure(2);
        figure
        subplot(211);
        plot(omega,TransFuncAmp,'ro',TransferFunction(:,1),TransferFunction(:,2),'-');grid on
        ylabel('$|H(i\omega)|$','FontSize',20);legend('Modified','Original');
        
        if Heading == 1000       % Secret # may be used
            title_content=strcat('WaveHeading = ',num2str(WaveHeadings(k)));
        else
            title_content=strcat('WaveHeading = ',num2str(WaveHeadings));
        end
        
        title(title_content)
        
        subplot(212);
        plot(omega,TransFuncPhase,'ro',TransferFunction(:,1),TransferFunction(:,3),'-');grid on
        xlabel('$\omega$ [rad/sec]','FontSize',20);ylabel('$<H(i\omega)$','FontSize',20);legend('Modified','Original')
        SpecOut = (TransFuncAmp.^2).*(SpecIn');
        d_omega=omega(2)-omega(1)  % omega generated by io, fo and NumComponentsOut
        a_excitation = sqrt(2*SpecIn*d_omega*d_theta); 
        a = sqrt(2*SpecOut*d_omega*d_theta);

        figure(2)
        plot(omega,a,'ro');grid on
        xlabel('$\omega$ [rad/sec]','FontSize',20);ylabel('a','FontSize',20)
        title('Output Amplitude Spectrum','FontSize',20)

        figure(3)
        plot(SpecInRAW(:,1), SpecInRAW(:,2))
        hold on
        plot(omega,SpecIn,'ro');grid on
        
        % WaveRMS might be used to check Hsig. (i.e. 4*WaveRMS = Hsig)
        WaveVAR=WaveVAR+1/2*sum(a_excitation.^2);

        % The previous name of the variable (InputRMS) was misleading. 
        ResponseVAR=ResponseVAR+1/2*sum(a.^2);  
        rms = sqrt(ResponseVAR);

        Mphi = [TransFuncAmp TransFuncPhase];
%         if TEV/(sum(a)/ResponseRMS) >= 1.0
%             disp(' ');
%             error('Error: TEV exceeds the maximum extreme bounded by the linear model. N should be increased.');
%         elseif TEV/(sum(a)/ResponseRMS) > 0.65 && TEV/(sum(a)/ResponseRMS) < 1.0
%             disp(' ');
%             disp('Warning: N is too small compared to TEV. Consider increasing N.');
%         end

        % -------------------------------------------------------------------------
        % Write out a_excitation.dat, a.dat, TransferPhase.dat, RunTimeStats.dat and omega.dat

        % Fourier Components for Wave Time Serie
        fout1=a_excitation';
        temp_filename=fullfile('Inputs',strcat('a_excitation',num2str(k),'.dat'));
        fid1 = fopen(temp_filename,'wt');
        fprintf(fid1,'%16.15E\n',fout1);
        fclose(fid1);

        % Forier Components for Response Time Series
        fout2=a';
        temp_filename=fullfile('Inputs',strcat('a',num2str(k),'.dat'));
        fid2 = fopen(temp_filename,'wt');
        fprintf(fid2,'%16.15E\n',fout2);
        fclose(fid2);

        % Convert the phase in the transfer function to radian and keep it for later use.
        fout3=TransFuncPhase'*pi/180;  % convert it to radian
        temp_filename=fullfile('Inputs',strcat('TransferPhase',num2str(k),'.dat'));
        fid3 = fopen(temp_filename,'wt');
        fprintf(fid3,'%16.15E\n',fout3);
        fclose(fid3);
        

end

% Write out omega.dat
fout5=omega';
temp_filename=fullfile('Inputs','omega.dat');
fid5 = fopen(temp_filename,'wt');
fprintf(fid5,'%16.15E   \n',fout5);
fclose(fid5);

% Make a new folder named Outputs
[~, ~, ~]=mkdir('Outputs');

% Keep a few run time stats
temp_filename=fullfile('Outputs',strcat('RunTimeStatsCase',Char_CaseNo,'.dat'));
fid4 = fopen(temp_filename,'wt');

fprintf(fid4,strcat('TEV = ',num2str(TEV),'\n'));
fprintf(fid4,strcat('m =',num2str(round(TEVtoNumberOfSamples(TEV))),'\n'));
fprintf(fid4,strcat('N_narrow =',num2str(round(exp(0.5*TEV^2))),'\n'));
fprintf(fid4,strcat('WaveRMS = ',num2str(sqrt(WaveVAR)),'\n'));
fprintf(fid4,strcat('ResponseRMS = ',num2str(sqrt(ResponseVAR)),'\n'));
fprintf(fid4,strcat('N = ',num2str(N),'\n'));
fprintf(fid4,strcat('Xmax/ReseponseRMS = ',num2str(sum(a)/sqrt(ResponseVAR)),' \n'));
fprintf(fid4,strcat('Smin = ',num2str(-determineSmax(sqrt(ResponseVAR),TEV,0)),' \n'));
fprintf(fid4,strcat('Smax = ',num2str(determineSmax(sqrt(ResponseVAR),TEV,0)),' \n'));
fclose(fid4);

disp(' ')
disp('DLGpre has been completed.')
disp(' ')

toc



